
/*********************************************************************
*
*  Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  lldp_example.c
*
* @purpose   LLDP OpEN API Examples.
*
* @component OpEN
*
* @comments
*
* @create    19/10/2012
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_lldp.h"

#define STRNCAT(_dst, _src) \
  strncat((_dst), (_src), sizeof((_dst))-strlen((_dst))-1)

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the LLDP Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printLldpAppMenu()
{
  printf("Usage: lldp_example <test#> <interface> <arg1> <arg2> ... \n");
  printf("Test 1: Set LLDP Interface Transmit Mode: lldp_example 1 <interface> <tramsmit mode> \n");
  printf("Test 2: Get LLDP Interface Transmit Mode: lldp_example 2 <interface> \n");
  printf("Test 3: Set LLDP Interface Receive Mode: lldp_example 3 <interface> <receive mode> \n");
  printf("Test 4: Get LLDP Interface Receive Mode: lldp_example 4 <interface> \n");
  printf("Test 5: Set LLDP Interface TLV Transmit Mode: lldp_example 5 <interface> <TLV type> <transmit mode> \n");
  printf("Test 6: Get LLDP Interface TLV Transmit Mode: lldp_example 6 <interface> \n");
  printf("Test 7: Set LLDP Interface Notification Mode: lldp_example 7 <interface> <notification mode> \n");
  printf("Test 8: Get LLDP Interface Notification Mode: lldp_example 8 <interface> \n");
  printf("Test 9: Get LLDP Remote Device details: lldp_example 9 <interface> \n");
  printf("Test 10: Get LLDP Local Device details: lldp_example 10 <interface> \n");
  printf("Test 11: Get LLDP-MED interface administrative mode: lldp_example 11 <interface> \n");
  printf("Test 12: Set LLDP-MED interface administrative mode: lldp_example 12 <interface> <mode> \n");
  printf("Test 13: Get LLDP-MED interface config notification mode: lldp_example 13 <interface> \n");
  printf("Test 14: Set LLDP-MED interface config notification mode: lldp_example 14 <interface> <mode> \n");
  printf("Test 15: Get LLDP-MED interface TLVs enabled for transmision: lldp_example 15 <interface> \n");
  printf("Test 16: Set LLDP-MED interface TLVs for transmision: lldp_example 16 <interface> \n");
  printf("Test 17: Get LLDP-MED Remote Device data: lldp_example 17 <interface> \n");
  printf("Test 18: Test LLDP OpEN APIs sanity: lldp_example 18 <interface> \n");

  return;
}

/*********************************************************************
* @purpose  Sets LLDP PDU transmit mode on the specified interface.
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   ID of the interface
* @param    mode           @b{(input)}   LLDP transmit mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void lldpIntfTransmitModeSet(openapiClientHandle_t *clientHandle, uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiLldpIntfTransmitModeSet(clientHandle, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set LLDP Interface Transmit Mode. (result = %d)\n", result);
  }
  else
  {
    printf("LLDP Interface Transmit Mode set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets LLDP PDU transmit mode on the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void lldpIntfTransmitModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t lldpTxMode;
  
  if ((result = openapiLldpIntfTransmitModeGet(clientHandle, ifNum, &lldpTxMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get LLDP Interface Transmit Mode. (result = %d)\n", result);
  }
  else
  {
    printf("LLDP Interface Transmit Mode for interface %u: %u \n", ifNum, lldpTxMode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets LLDP PDU reception mode on the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
* @param    mode           @b{(input)}  LLDP receive mode
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void lldpIntfReceiveModeSet(openapiClientHandle_t *clientHandle, uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiLldpIntfReceiveModeSet(clientHandle, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set LLDP Interface Receive Mode. (result = %d)\n", result);
  }
  else
  {
    printf("LLDP Interface Receive Mode set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets LLDP PDU reception mode on the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void lldpIntfReceiveModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t lldpRxMode;
  
  if ((result = openapiLldpIntfReceiveModeGet(clientHandle, ifNum, &lldpRxMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get LLDP Interface Receive Mode. (result = %d)\n", result);
  }
  else
  {
    printf("LLDP Interface Receive Mode for interface %u: %u \n", ifNum, lldpRxMode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the LLDP optional TLVs transmit mode for the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
* @param    tlvType          @b{(input)}   LLDP TLV of type OPEN_LLDP_TLV_TYPE_t
* @param    tlvTransmitMode  @b{(input)}   TLV transmit mode
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void lldpIntfTLVTransmitModeSet(openapiClientHandle_t *clientHandle, uint32_t ifNum, 
                                OPEN_LLDP_TLV_TYPE_t tlvType, OPEN_CONTROL_t tlvTransmitMode)
{
  open_error_t result;
  
  if ((result = openapiLldpIntfTLVTransmitModeSet(clientHandle, ifNum, tlvType, tlvTransmitMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set LLDP Interface TLV Transmit Mode. (result = %d)\n", result);
  }
  else
  {
    printf("LLDP Interface TLV Transmit Mode set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the LLDP optional TLVs transmit mode for the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void lldpIntfTLVTransmitModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t lldpTLVTxMode;
  uint32_t index;
  
  for (index = 0; index < OPEN_LLDP_LAST_TLV; index++)
  {
    if ((result = openapiLldpIntfTLVTransmitModeGet(clientHandle, ifNum, index, &lldpTLVTxMode)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get LLDP Interface TLV Transmit Mode. (result = %d)\n", result);
      return;
    }
    else
    {
      printf("Interface: %u, TLV type: %u, Transmit Mode: %u \n", ifNum, index, lldpTLVTxMode);
    }
  }
  return;
}

/*********************************************************************
* @purpose  Sets the LLDP Notification mode for the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
* @param    mode           @b{(input)}  LLDP Notification mode
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void lldpIntfNotificationModeSet(openapiClientHandle_t *clientHandle, uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiLldpIntfNotificationModeSet(clientHandle, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set LLDP Interface Notification Mode. (result = %d)\n", result);
  }
  else
  {
    printf("LLDP Interface Notification Mode set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the LLDP Notification mode for the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void lldpIntfNotificationModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t lldpNotMode;
  
  if ((result = openapiLldpIntfNotificationModeGet(clientHandle, ifNum, &lldpNotMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get LLDP Interface Notification Mode. (result = %d)\n", result);
  }
  else
  {
    printf("LLDP Interface Notification Mode for interface %u: %u \n", ifNum, lldpNotMode);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the LLDP Remote Device details for the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void lldpIntfRemoteDeviceDetailsGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_LLDP_CHASSIS_ID_SUBTYPE_t chassisIdSubtype;
  OPEN_LLDP_PORT_ID_SUBTYPE_t portIdSubtype;
  open_buffdesc chassisId, portId;
  open_buffdesc sysName, sysDesc, portDesc;
  open_buffdesc sysCapSupp, sysCapEnabled;
  char *str = NULL;
  open_buffdesc mgmtAddress;
  OPEN_LLDP_IANA_ADDR_FAMILY_t addressFamily = 0;
  uint32_t remIndex = 0;
  uint32_t timeStamp = 0;
  uint32_t ttl = 0;
  uint8_t addrLen;
  uint32_t strLen = 0;

  if ((result = openapiLldpMgmtStrLenGet(clientHandle, &strLen)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n", result);
    return;
  }

  if ((str = (char*)malloc(strLen)) == NULL) 
  {
    printf("Could not allocate memory.\n");
    return;
  }

  printf("LLDP Remote Device details for interface %u\n", ifNum);
  printf("============================================\n\n");

  while(openapiLldpIntfRemoteDeviceNextGet(clientHandle, ifNum, 
                                           &remIndex, &timeStamp) == OPEN_E_NONE)
  {
    memset(str, 0, strLen);
    chassisId.pstart = str;
    chassisId.size = strLen;

    if ((result = openapiLldpIntfRemoteDeviceChassisIdGet(clientHandle, ifNum, remIndex, timeStamp, 
                                                          &chassisIdSubtype, &chassisId)) != OPEN_E_NONE)
    {
      printf("Bad return code. (result = %d)\n\n", result);
    }
    else
    {
      printf("Chassis Id Subtype: %u\n", chassisIdSubtype);
      printf("Chassis Id: %s\n\n", str);
    }

    memset(str, 0, strLen);
    portId.pstart = str;
    portId.size = strLen;

    if ((result = openapiLldpIntfRemoteDevicePortIdGet(clientHandle, ifNum, remIndex, timeStamp, 
                                                       &portIdSubtype, &portId)) != OPEN_E_NONE)
    {
      printf("Bad return code. (result = %d)\n\n", result);
    }
    else
    {
      printf("Port Id Subtype: %u\n", portIdSubtype);
      printf("Port Id: %s\n\n", str);
    }

    memset(str, 0, strLen);
    sysName.pstart = str;
    sysName.size = strLen;

    if ((result = openapiLldpIntfRemoteDeviceSystemNameGet(clientHandle, ifNum, remIndex, 
                                                           timeStamp, &sysName)) != OPEN_E_NONE)
    {
      printf("Bad return code. (result = %d)\n\n", result);
    }
    else
    {
      printf("System name: %s\n\n", str);
    }

    memset(str, 0, strLen);
    sysDesc.pstart = str;
    sysDesc.size = strLen;

    if ((result = openapiLldpIntfRemoteDeviceSystemDescGet(clientHandle, ifNum, remIndex, 
                                                           timeStamp, &sysDesc)) != OPEN_E_NONE)
    {
      printf("Bad return code. (result = %d)\n\n", result);
    }
    else
    {
      printf("System Description: %s\n\n", str);
    }

    memset(str, 0, strLen);
    portDesc.pstart = str;
    portDesc.size = strLen;

    if ((result = openapiLldpIntfRemoteDevicePortDescGet(clientHandle, ifNum, remIndex, 
                                                         timeStamp, &portDesc)) != OPEN_E_NONE)
    {
      printf("Bad return code. (result = %d)\n\n", result);
    }
    else
    {
      printf("Port Description: %s\n\n", str);
    }

    memset(str, 0, strLen);
    sysCapSupp.pstart = str;
    sysCapSupp.size = strLen;

    if ((result = openapiLldpIntfRemoteDeviceSysCapSuppGet(clientHandle, ifNum, remIndex, 
                                                           timeStamp, &sysCapSupp)) != OPEN_E_NONE)
    {
      printf("Bad return code. (result = %d)\n\n", result);
    }
    else
    {
      printf("System Capabilities Supported: %s\n\n", str);
    }

    memset(str, 0, strLen);
    sysCapEnabled.pstart = str;
    sysCapEnabled.size = strLen;

    if ((result = openapiLldpIntfRemoteDeviceSysCapEnabledGet(clientHandle, ifNum, remIndex, 
                                                              timeStamp, &sysCapEnabled)) != OPEN_E_NONE)
    {
      printf("Bad return code. (result = %d)\n\n", result);
    }
    else
    {
      printf("System Capabilities Enabled: %s\n\n", str);
    }

    memset(str, 0, strLen);
    mgmtAddress.pstart = str;
    mgmtAddress.size = strLen;
    addrLen = 0;

    while (openapiLldpIntfRemoteDeviceMgmtAddressNextGet(clientHandle, ifNum, remIndex, timeStamp,
                                                         &addressFamily, &mgmtAddress, &addrLen) == OPEN_E_NONE)
    {
      printf("Management Address Type: %u\n", addressFamily);
      printf("Management Address: %s\n\n", str);
    }

    if ((result = openapiLldpIntfRemoteDeviceTTLGet(clientHandle, ifNum, remIndex, 
                                                    timeStamp, &ttl)) != OPEN_E_NONE)
    {
      printf("Bad return code. (result = %d)\n\n", result);
    }
    else
    {
      printf("LLDP Neighbor TTL: %u\n\n", ttl);
    }
  }

  free(str);
  return;
}

/*********************************************************************
* @purpose  Gets the LLDP Remote Device for the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void lldpIntfRemoteDevicesGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  uint32_t currIfNum, nextIfNum;

  if (ifNum == 0)
  {
    currIfNum = ifNum;
    while (openapiLldpIntfNextGet(clientHandle, currIfNum, &nextIfNum) == OPEN_E_NONE)
    {
      lldpIntfRemoteDeviceDetailsGet(clientHandle, nextIfNum);
      currIfNum = nextIfNum;
    }
  }
  else
  {
    lldpIntfRemoteDeviceDetailsGet(clientHandle, ifNum);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the LLDP Local Device details for the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void lldpIntfLocalDeviceGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_LLDP_CHASSIS_ID_SUBTYPE_t chassisIdSubtype;
  OPEN_LLDP_PORT_ID_SUBTYPE_t portIdSubtype;
  open_buffdesc chassisId, portId;
  open_buffdesc sysName, sysDesc, portDesc;
  open_buffdesc sysCapSupp, sysCapEnabled;
  open_buffdesc mgmtAddress;
  OPEN_LLDP_IANA_ADDR_FAMILY_t addressFamily;
  char *str = NULL;
  uint32_t strLen = 0;

  if ((result = openapiLldpMgmtStrLenGet(clientHandle, &strLen)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n", result);
    return;
  }

  if ((str = (char*)malloc(strLen)) == NULL) 
  {
    printf("Could not allocate memory.\n");
    return;
  }

  printf("LLDP Local Device details for interface %u\n", ifNum);
  printf("===========================================\n\n");

  memset(str, 0, strLen);
  chassisId.pstart = str;
  chassisId.size = strLen;
    
  if ((result = openapiLldpIntfLocalDeviceChassisIdGet(clientHandle, ifNum, &chassisIdSubtype, 
                                                       &chassisId)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n\n", result);
  }
  else
  {
    printf("Chassis Id Subtype: %u\n", chassisIdSubtype);
    printf("Chassis Id: %s\n\n", str);
  }

  memset(str, 0, strLen);
  portId.pstart = str;
  portId.size = strLen;

  if ((result = openapiLldpIntfLocalDevicePortIdGet(clientHandle, ifNum, &portIdSubtype, 
                                                    &portId)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n\n", result);
  }
  else
  {
    printf("Port Id Subtype: %u\n", portIdSubtype);
    printf("Port Id: %s\n\n", str);
  }

  memset(str, 0, strLen);
  sysName.pstart = str;
  sysName.size = strLen;

  if ((result = openapiLldpIntfLocalDeviceSystemNameGet(clientHandle, ifNum, &sysName)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n\n", result);
  }
  else
  {
    printf("System name: %s\n\n", str);
  }

  memset(str, 0, strLen);
  sysDesc.pstart = str;
  sysDesc.size = strLen;

  if ((result = openapiLldpIntfLocalDeviceSystemDescGet(clientHandle, ifNum, &sysDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n\n", result);
  }
  else
  {
    printf("System Description: %s\n\n", str);
  }

  memset(str, 0, strLen);
  portDesc.pstart = str;
  portDesc.size = strLen;

  if ((result = openapiLldpIntfLocalDevicePortDescGet(clientHandle, ifNum, &portDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n\n", result);
  }
  else
  {
    printf("Port Description: %s\n\n", str);
  }

  memset(str, 0, strLen);
  sysCapSupp.pstart = str;
  sysCapSupp.size = strLen;

  if ((result = openapiLldpIntfLocalDeviceSysCapSuppGet(clientHandle, ifNum, &sysCapSupp)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n\n", result);
  }
  else
  {
    printf("System Capabilities Supported: %s\n\n", str);
  }

  memset(str, 0, strLen);
  sysCapEnabled.pstart = str;
  sysCapEnabled.size = strLen;

  if ((result = openapiLldpIntfLocalDeviceSysCapEnabledGet(clientHandle, ifNum, &sysCapEnabled)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n\n", result);
  }
  else
  {
    printf("System Capabilities Enabled: %s\n\n", str);
  }

  memset(str, 0, strLen);
  mgmtAddress.pstart = str;
  mgmtAddress.size = strLen;

  if ((result = openapiLldpIntfLocalDeviceMgmtAddressGet(clientHandle, ifNum, 
                                                         &addressFamily, &mgmtAddress)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n\n", result);
  }
  else
  {
    printf("Management Address Type: %u\n", addressFamily);
    printf("Management Address: %s\n\n", str);
  }

  free(str);
  return;
}

/*********************************************************************
* @purpose  Get the interface LLDP-MED administrative mode.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedIntfModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  if ((result = openapiLldpMedIntfModeGet(clientHandle, ifNum, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get LLDP-MED interface mode. (result = %d)\n", result);
  }
  else
  {
    printf("LLDP-MED interface mode for interface %u: %u \n", ifNum, mode);
  }

  return;
}

/*********************************************************************
* @purpose  Set LLDP-MED administrative mode on LLDP valid interfaces.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  an internal interface number or 
*                          @ref OPEN_ALL_INTERFACES for all interfaces
* @param    mode           @b{(input)}  LLDP mode
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of the switch
* @notes    Administrative mode is set on all LLDP valid interfaces, if 
*           @ref OPEN_ALL_INTERFACES is passed in for the 'ifNum' parameter.

*
* @end
*********************************************************************/
void lldpMedIntfModeSet(openapiClientHandle_t *clientHandle, uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiLldpMedIntfModeSet(clientHandle, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set LLDP-MED mode on interface. (result = %d)\n", result);
  }
  else
  {
    printf("LLDP-MED mode set to %d on interface. (result = %d)\n", mode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the interface LLDP-MED configuration notification mode.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedIntfConfigNotificationModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_BOOL_t mode;

  if ((result = openapiLldpMedIntfConfigNotificationModeGet(clientHandle, ifNum, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get LLDP-MED interface configuration notification mode. (result = %d)\n", result);
  }
  else
  {
    printf("LLDP-MED interface configuration notification mode for interface %u: %u \n", ifNum, mode);
  }

  return;
}

/*********************************************************************
* @purpose  Set LLDP-MED configuration notification mode on LLDP valid interfaces.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  an internal interface number or 
*                          @ref OPEN_ALL_INTERFACES for all interfaces
* @param    mode           @b{(input)}  notification mode
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of the switch
* @notes    Administrative mode is set on all LLDP valid interfaces, if 
*           @ref OPEN_ALL_INTERFACES is passed in for the 'ifNum' parameter.

*
* @end
*********************************************************************/
void lldpMedIntfConfigNotificationModeSet(openapiClientHandle_t *clientHandle, uint32_t ifNum, OPEN_BOOL_t mode)
{
  open_error_t result;

  if ((result = openapiLldpMedIntfConfigNotificationModeSet(clientHandle, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set LLDP-MED configuration notification mode on interface. (result = %d)\n", result);
  }
  else
  {
    printf("LLDP-MED configuration notification mode set to %d on interface. (result = %d)\n", mode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the interface LLDP-MED TLVs enabled for transmision.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedInterfaceTransmitTlvGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_LLDP_MED_CAPABILITIES_t cap;
  char str[256];

  memset(&cap, 0, sizeof(cap));
  if ((result = openapiLldpMedIntfTransmitTlvGet(clientHandle, ifNum, &cap)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get LLDP-MED interface TLVs enabled for transmision. (result = %d)\n", result);
  }
  else
  {
    memset(&str, 0, sizeof(str));
    if (cap.bitmap[1] & OPEN_LLDP_MED_CAP_CAPABILITIES_BITMASK)
    {
      STRNCAT(str, "0-CAPABILITIES ");
    }
    if (cap.bitmap[1] & OPEN_LLDP_MED_CAP_NETWORKPOLICY_BITMASK)
    {
      STRNCAT(str, "1-NETWORKPOLICY ");
    }
    if (cap.bitmap[1] & OPEN_LLDP_MED_CAP_LOCATION_BITMASK)
    {
      STRNCAT(str, "2-LOCATION ");
    }
    if (cap.bitmap[1] & OPEN_LLDP_MED_CAP_EXT_PSE_BITMASK)
    {
      STRNCAT(str, "3-EXTENDEDPSE ");
    }
    if (cap.bitmap[1] & OPEN_LLDP_MED_CAP_EXT_PD_BITMASK)
    {
      STRNCAT(str, "4-EXTENDEDPD ");
    }
    if (cap.bitmap[1] & OPEN_LLDP_MED_CAP_INVENTORY_BITMASK)
    {
      STRNCAT(str, "5-INVENTORY");
    }
    str[sizeof(str) - 1] = '\0';

    printf("LLDP-MED interface TLVs enabled for transmision for interface %u: %s. \n", ifNum, str);
  }

  return;
}

/*********************************************************************
* @purpose  Set LLDP-MED TLVs for transmision on LLDP valid interfaces.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  an internal interface number or 
*                          @ref OPEN_ALL_INTERFACES for all interfaces
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of the switch
* @notes    Administrative mode is set on all LLDP valid interfaces, if 
*           @ref OPEN_ALL_INTERFACES is passed in for the 'ifNum' parameter. 
*
* @end
*********************************************************************/
void lldpMedInterfaceTransmitTlvSet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_LLDP_MED_CAPABILITIES_t current_cap;
  OPEN_LLDP_MED_CAPABILITIES_t test_cap;

  memset(&current_cap, 0, sizeof(current_cap));
  /* not all platforms support the same set of capabilities, test setting to no capabilities is safe */
  memset(&test_cap, 0, sizeof(test_cap));

  if ((result = openapiLldpMedIntfTransmitTlvGet(clientHandle, ifNum, &current_cap)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get current LLDP-MED TLVs for transmision on interface. (result = %d)\n", result);
  }
  else
  {
    if ((result = openapiLldpMedIntfTransmitTlvSet(clientHandle, ifNum, &test_cap)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to set LLDP-MED TLVs for transmision on interface. (result = %d)\n", result);
    }
    else
    {
      if ((result = openapiLldpMedIntfTransmitTlvSet(clientHandle, ifNum, &current_cap)) != OPEN_E_NONE)
      {
        printf("Bad return code trying to set LLDP-MED TLVs for transmision on interface. (result = %d)\n", result);
      }
      else
      {
        printf("Setting LLDP-MED TLVs for transmision on interface successful.\n");
      }
    }
  }

  return;
}

/*********************************************************************
* @purpose  Get the remote device information for the given interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface number
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedRemoteDeviceInfoGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  uint32_t remIndex = 0;
  uint32_t timeStamp = 0;
  open_buffdesc modelName;
  char *str = NULL;
  uint32_t strLen = 0;
  OPEN_LLDP_MED_CAPABILITIES_t supportedCapabilities;
  OPEN_LLDP_MED_CAPABILITIES_t enabledCapabilities;
  uint32_t deviceClass;

  if ((result = openapiLldpMedRemoteModelNameSizeGet(clientHandle, &strLen)) != OPEN_E_NONE)
  {
    printf("Bad return code from openapiLldpMedRemoteModelNameSizeGet(). (result = %d)\n", result);
    return;
  }

  if ((str = malloc(strLen)) == NULL)
  {
    printf("Could not allocate memory.\n");
    return;
  }

  printf("LLDP-MED Remote Device details for interface %u\n", ifNum);
  printf("=================================================\n\n");

  while(openapiLldpIntfRemoteDeviceNextGet(clientHandle, ifNum, &remIndex, &timeStamp) == OPEN_E_NONE)
  {
    if ((result = openapiLldpMedRemoteCapabilitiesGet(clientHandle, ifNum, remIndex, timeStamp,
                                                      &supportedCapabilities, &enabledCapabilities)) != OPEN_E_NONE)
    {
      printf("Bad return code from openapiLldpMedRemoteCapabilitiesGet(). (result = %d)\n\n", result);
    }
    else
    {
      printf("Supported capabilities bitmap: 0x%02x%02x\n", supportedCapabilities.bitmap[0], supportedCapabilities.bitmap[1]);
      printf("Enabled capabilities bitmap:   0x%02x%02x\n", enabledCapabilities.bitmap[0], enabledCapabilities.bitmap[1]);
    }

    if ((result = openapiLldpMedRemoteDeviceClassGet(clientHandle, ifNum, remIndex, timeStamp,
                                                     &deviceClass)) != OPEN_E_NONE)
    {
      printf("Bad return code from openapiLldpMedRemoteDeviceClassGet(). (result = %d)\n\n", result);
    }
    else
    {
      printf("Device class: %u\n", deviceClass);
    }

    memset(str, 0, strLen);
    modelName.pstart = str;
    modelName.size = strLen;

    if ((result = openapiLldpMedRemoteModelNameGet(clientHandle, ifNum, remIndex, timeStamp,
                                                   &modelName)) != OPEN_E_NONE)
    {
      printf("Bad return code from openapiLldpMedRemoteModelNameGet(). (result = %d)\n\n", result);
    }
    else
    {
      printf("Model name: %s\n\n", str);
    }
  }

  free(str);
  return;
}

/*********************************************************************
* @purpose  This function sanity checks all LLDP OpEN APIs. All LLDP OpEN APIs are 
*           called with possible NULL and invalid parameters to check the API robustness.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void lldpOpENAPIsTestSanity(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t controlMode;
  uint32_t invalidControlMode = 10;
  uint32_t lldpMgmtStrLen, strLen, ttl;
  uint32_t invalidTLVType = 10;
  uint32_t nextIfNum, remIndex, timeStamp;
  OPEN_LLDP_CHASSIS_ID_SUBTYPE_t chassisIdSubtype;
  open_buffdesc buffDesc;
  char *str;
  OPEN_LLDP_PORT_ID_SUBTYPE_t portIdSubtype;
  OPEN_LLDP_IANA_ADDR_FAMILY_t  addressFamily;
  uint8_t addrLen;
  OPEN_BOOL_t boolMode;
  OPEN_LLDP_MED_CAPABILITIES_t cap;

  printf("Testing LLDP OpEN APIs sanity:\n\n");

  if ((result = openapiLldpMgmtStrLenGet(clientHandle, &lldpMgmtStrLen)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n", result);
    return;
  }

  if ((str = (char*)malloc(lldpMgmtStrLen)) == NULL) 
  {
    printf("Could not allocate memory.\n");
    return;
  }

  memset(str, 0, lldpMgmtStrLen);

  /* openapiLldpMgmtStrLenGet() */
  printf("Testing openapiLldpMgmtStrLenGet():\n");

  result = openapiLldpMgmtStrLenGet(NULL, &strLen);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpMgmtStrLenGet(clientHandle, NULL);
  printf("NULL String Length:(result = %d)\n", result);

  printf("openapiLldpMgmtStrLenGet() sanity successful\n\n");

  /* openapiLldpIntfTransmitModeSet() */
  printf("Testing openapiLldpIntfTransmitModeSet():\n");

  result = openapiLldpIntfTransmitModeSet(NULL, ifNum, OPEN_ENABLE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfTransmitModeSet(clientHandle, ifNum, invalidControlMode);
  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiLldpIntfTransmitModeSet() sanity successful\n\n");

  /* openapiLldpIntfTransmitModeGet() */
  printf("Testing openapiLldpIntfTransmitModeGet():\n");

  result = openapiLldpIntfTransmitModeGet(NULL, ifNum, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfTransmitModeGet(clientHandle, ifNum, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiLldpIntfTransmitModeGet() sanity successful\n\n");

  /* openapiLldpIntfReceiveModeSet() */
  printf("Testing openapiLldpIntfReceiveModeSet():\n");

  result = openapiLldpIntfReceiveModeSet(NULL, ifNum, OPEN_ENABLE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfReceiveModeSet(clientHandle, ifNum, invalidControlMode);
  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiLldpIntfReceiveModeSet() sanity successful\n\n");

  /* openapiLldpIntfReceiveModeGet() */
  printf("Testing openapiLldpIntfReceiveModeGet():\n");

  result = openapiLldpIntfReceiveModeGet(NULL, ifNum, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfReceiveModeGet(clientHandle, ifNum, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiLldpIntfReceiveModeGet() sanity successful\n\n");

  /* openapiLldpIntfTLVTransmitModeSet() */
  printf("Testing openapiLldpIntfTLVTransmitModeSet():\n");

  result = openapiLldpIntfTLVTransmitModeSet(NULL, ifNum, OPEN_LLDP_SYS_DESC_TLV, OPEN_ENABLE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfTLVTransmitModeSet(clientHandle, ifNum, invalidTLVType, OPEN_ENABLE);
  printf("Invalid TLV type:(result = %d)\n", result);

  result = openapiLldpIntfTLVTransmitModeSet(clientHandle, ifNum, OPEN_LLDP_SYS_DESC_TLV, invalidControlMode);
  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiLldpIntfTLVTransmitModeSet() sanity successful\n\n");

  /* openapiLldpIntfTLVTransmitModeGet() */
  printf("Testing openapiLldpIntfTLVTransmitModeGet():\n");

  result = openapiLldpIntfTLVTransmitModeGet(NULL, ifNum, OPEN_LLDP_SYS_DESC_TLV, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfTLVTransmitModeGet(clientHandle, ifNum, OPEN_LLDP_SYS_DESC_TLV, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiLldpIntfTLVTransmitModeGet() sanity successful\n\n");

  /* openapiLldpIntfNotificationModeSet() */
  printf("Testing openapiLldpIntfNotificationModeSet():\n");

  result = openapiLldpIntfNotificationModeSet(NULL, ifNum, OPEN_ENABLE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfNotificationModeSet(clientHandle, ifNum, invalidControlMode);
  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiLldpIntfNotificationModeSet() sanity successful\n\n");

  /* openapiLldpIntfNotificationModeGet() */
  printf("Testing openapiLldpIntfNotificationModeGet():\n");

  result = openapiLldpIntfNotificationModeGet(NULL, ifNum, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfNotificationModeGet(clientHandle, ifNum, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiLldpIntfNotificationModeGet() sanity successful\n\n");

  /* openapiLldpIntfNextGet() */
  printf("Testing openapiLldpIntfNextGet():\n");

  result = openapiLldpIntfNextGet(NULL, ifNum, &nextIfNum);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfNextGet(clientHandle, ifNum, NULL);
  printf("NULL Next Interface Number:(result = %d)\n", result);

  printf("openapiLldpIntfNextGet() sanity successful\n\n");

  /* openapiLldpIntfRemoteDeviceNextGet() */
  printf("Testing openapiLldpIntfRemoteDeviceNextGet():\n");

  result = openapiLldpIntfRemoteDeviceNextGet(NULL, ifNum, &remIndex, &timeStamp);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDeviceNextGet(clientHandle, ifNum, NULL, &timeStamp);
  printf("NULL Remote Index:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDeviceNextGet(clientHandle, ifNum, &remIndex, NULL);
  printf("NULL Time Stamp:(result = %d)\n", result);

  printf("openapiLldpIntfRemoteDeviceNextGet() sanity successful\n\n");

  /* openapiLldpIntfRemoteDeviceChassisIdGet() */
  printf("Testing openapiLldpIntfRemoteDeviceChassisIdGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfRemoteDeviceChassisIdGet(NULL, ifNum, remIndex, timeStamp,
                                                   &chassisIdSubtype, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDeviceChassisIdGet(clientHandle, ifNum, remIndex, timeStamp,
                                                   NULL, &buffDesc);
  printf("NULL Chassis Id Subtype:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDeviceChassisIdGet(clientHandle, ifNum, remIndex, timeStamp,
                                                   &chassisIdSubtype, NULL);
  printf("NULL Chassis Id:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfRemoteDeviceChassisIdGet(clientHandle, ifNum, remIndex, timeStamp,
                                                   &chassisIdSubtype, &buffDesc);
  printf("NULL Chassis Id buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfRemoteDeviceChassisIdGet(clientHandle, ifNum, remIndex, timeStamp,
                                                   &chassisIdSubtype, &buffDesc);
  printf("NULL Chassis Id buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfRemoteDeviceChassisIdGet() sanity successful\n\n");

  /* openapiLldpIntfRemoteDevicePortIdGet() */
  printf("Testing openapiLldpIntfRemoteDevicePortIdGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfRemoteDevicePortIdGet(NULL, ifNum, remIndex, timeStamp,
                                                &portIdSubtype, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDevicePortIdGet(clientHandle, ifNum, remIndex, timeStamp,
                                                NULL, &buffDesc);
  printf("NULL Port Id Subtype:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDevicePortIdGet(clientHandle, ifNum, remIndex, timeStamp,
                                                &portIdSubtype, NULL);
  printf("NULL Port Id:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfRemoteDevicePortIdGet(clientHandle, ifNum, remIndex, timeStamp,
                                                &portIdSubtype, &buffDesc);
  printf("NULL Port Id buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfRemoteDevicePortIdGet(clientHandle, ifNum, remIndex, timeStamp,
                                                &portIdSubtype, &buffDesc);
  printf("NULL Port Id buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfRemoteDevicePortIdGet() sanity successful\n\n");

  /* openapiLldpIntfRemoteDeviceSystemNameGet() */
  printf("Testing openapiLldpIntfRemoteDeviceSystemNameGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfRemoteDeviceSystemNameGet(NULL, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDeviceSystemNameGet(clientHandle, ifNum, remIndex, timeStamp, NULL);
  printf("NULL System Name:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfRemoteDeviceSystemNameGet(clientHandle, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL System Name buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfRemoteDeviceSystemNameGet(clientHandle, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL System Name buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfRemoteDevicePortIdGet() sanity successful\n\n");

  /* openapiLldpIntfRemoteDeviceSystemDescGet() */
  printf("Testing openapiLldpIntfRemoteDeviceSystemDescGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfRemoteDeviceSystemDescGet(NULL, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDeviceSystemDescGet(clientHandle, ifNum, remIndex, timeStamp, NULL);
  printf("NULL System Description:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfRemoteDeviceSystemDescGet(clientHandle, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL System Description buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfRemoteDeviceSystemDescGet(clientHandle, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL System Description buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfRemoteDeviceSystemDescGet() sanity successful\n\n");

  /* openapiLldpIntfRemoteDevicePortDescGet() */
  printf("Testing openapiLldpIntfRemoteDevicePortDescGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfRemoteDevicePortDescGet(NULL, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDevicePortDescGet(clientHandle, ifNum, remIndex, timeStamp, NULL);
  printf("NULL Port Description:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfRemoteDevicePortDescGet(clientHandle, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL Port Description buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfRemoteDevicePortDescGet(clientHandle, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL Port Description buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfRemoteDeviceSystemDescGet() sanity successful\n\n");

  /* openapiLldpIntfRemoteDeviceSysCapSuppGet() */
  printf("Testing openapiLldpIntfRemoteDeviceSysCapSuppGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfRemoteDeviceSysCapSuppGet(NULL, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDeviceSysCapSuppGet(clientHandle, ifNum, remIndex, timeStamp, NULL);
  printf("NULL Supported Sys Capabilities:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfRemoteDeviceSysCapSuppGet(clientHandle, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL Supported Sys Capabilities buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfRemoteDeviceSysCapSuppGet(clientHandle, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL Supported Sys Capabilities buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfRemoteDeviceSysCapSuppGet() sanity successful\n\n");

  /* openapiLldpIntfRemoteDeviceSysCapEnabledGet() */
  printf("Testing openapiLldpIntfRemoteDeviceSysCapEnabledGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfRemoteDeviceSysCapEnabledGet(NULL, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDeviceSysCapEnabledGet(clientHandle, ifNum, remIndex, timeStamp, NULL);
  printf("NULL Enabled Sys Capabilities:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfRemoteDeviceSysCapEnabledGet(clientHandle, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL Enabled Sys Capabilities buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfRemoteDeviceSysCapEnabledGet(clientHandle, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL Enabled Sys Capabilities buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfRemoteDeviceSysCapEnabledGet() sanity successful\n\n");

  /* openapiLldpIntfRemoteDeviceMgmtAddressNextGet() */
  printf("Testing openapiLldpIntfRemoteDeviceMgmtAddressNextGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfRemoteDeviceMgmtAddressNextGet(NULL, ifNum, remIndex, timeStamp, 
                                                         &addressFamily, &buffDesc, &addrLen);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDeviceMgmtAddressNextGet(clientHandle, ifNum, remIndex, timeStamp, 
                                                         NULL, &buffDesc, &addrLen);
  printf("NULL Address Family:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDeviceMgmtAddressNextGet(clientHandle, ifNum, remIndex, timeStamp, 
                                                         &addressFamily, &buffDesc, NULL);
  printf("NULL Address Length:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfRemoteDeviceMgmtAddressNextGet(clientHandle, ifNum, remIndex, timeStamp, 
                                                         &addressFamily, &buffDesc, &addrLen);
  printf("NULL Address buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfRemoteDeviceMgmtAddressNextGet(clientHandle, ifNum, remIndex, timeStamp, 
                                                         &addressFamily, &buffDesc, &addrLen);
  printf("NULL Address buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfRemoteDeviceMgmtAddressNextGet() sanity successful\n\n");

  /* openapiLldpIntfRemoteDeviceTTLGet() */
  printf("Testing openapiLldpIntfRemoteDeviceTTLGet():\n");

  result = openapiLldpIntfRemoteDeviceTTLGet(NULL, ifNum, remIndex, timeStamp, &ttl);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDeviceTTLGet(clientHandle, ifNum, remIndex, timeStamp, NULL);
  printf("NULL TTL:(result = %d)\n", result);

  printf("openapiLldpIntfRemoteDeviceTTLGet() sanity successful\n\n");

  /* openapiLldpIntfLocalDeviceChassisIdGet() */
  printf("Testing openapiLldpIntfLocalDeviceChassisIdGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfLocalDeviceChassisIdGet(NULL, ifNum, &chassisIdSubtype, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfLocalDeviceChassisIdGet(clientHandle, ifNum, NULL, &buffDesc);
  printf("NULL Chassis Id Subtype:(result = %d)\n", result);

  result = openapiLldpIntfLocalDeviceChassisIdGet(clientHandle, ifNum, &chassisIdSubtype, NULL);
  printf("NULL Chassis Id:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfLocalDeviceChassisIdGet(clientHandle, ifNum, &chassisIdSubtype, &buffDesc);
  printf("NULL Chassis Id buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfLocalDeviceChassisIdGet(clientHandle, ifNum, &chassisIdSubtype, &buffDesc);
  printf("NULL Chassis Id buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfLocalDeviceChassisIdGet() sanity successful\n\n");

  /* openapiLldpIntfLocalDevicePortIdGet() */
  printf("Testing openapiLldpIntfLocalDevicePortIdGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfLocalDevicePortIdGet(NULL, ifNum, &portIdSubtype, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfLocalDevicePortIdGet(clientHandle, ifNum, NULL, &buffDesc);
  printf("NULL Port Id Subtype:(result = %d)\n", result);

  result = openapiLldpIntfLocalDevicePortIdGet(clientHandle, ifNum, &portIdSubtype, NULL);
  printf("NULL Port Id:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfLocalDevicePortIdGet(clientHandle, ifNum, &portIdSubtype, &buffDesc);
  printf("NULL Port Id buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfLocalDevicePortIdGet(clientHandle, ifNum, &portIdSubtype, &buffDesc);
  printf("NULL Port Id buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfLocalDevicePortIdGet() sanity successful\n\n");

  /* openapiLldpIntfLocalDeviceSystemNameGet() */
  printf("Testing openapiLldpIntfLocalDeviceSystemNameGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfLocalDeviceSystemNameGet(NULL, ifNum, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfLocalDeviceSystemNameGet(clientHandle, ifNum, NULL);
  printf("NULL System Name:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfLocalDeviceSystemNameGet(clientHandle, ifNum, &buffDesc);
  printf("NULL System Name buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfLocalDeviceSystemNameGet(clientHandle, ifNum, &buffDesc);
  printf("NULL System Name buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfLocalDeviceSystemNameGet() sanity successful\n\n");

  /* openapiLldpIntfLocalDeviceSystemDescGet() */
  printf("Testing openapiLldpIntfLocalDeviceSystemDescGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfLocalDeviceSystemDescGet(NULL, ifNum, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfLocalDeviceSystemDescGet(clientHandle, ifNum, NULL);
  printf("NULL System Description:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfLocalDeviceSystemDescGet(clientHandle, ifNum, &buffDesc);
  printf("NULL System Description buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfLocalDeviceSystemDescGet(clientHandle, ifNum, &buffDesc);
  printf("NULL System Description buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfLocalDeviceSystemDescGet() sanity successful\n\n");

  /* openapiLldpIntfLocalDevicePortDescGet() */
  printf("Testing openapiLldpIntfLocalDevicePortDescGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfLocalDevicePortDescGet(NULL, ifNum, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfLocalDevicePortDescGet(clientHandle, ifNum, NULL);
  printf("NULL Port Description:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfLocalDevicePortDescGet(clientHandle, ifNum, &buffDesc);
  printf("NULL Port Description buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfLocalDevicePortDescGet(clientHandle, ifNum, &buffDesc);
  printf("NULL Port Description buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfLocalDevicePortDescGet() sanity successful\n\n");

  /* openapiLldpIntfLocalDeviceSysCapSuppGet() */
  printf("Testing openapiLldpIntfLocalDeviceSysCapSuppGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfLocalDeviceSysCapSuppGet(NULL, ifNum, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfLocalDeviceSysCapSuppGet(clientHandle, ifNum, NULL);
  printf("NULL Supported Sys Capabilities:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfLocalDeviceSysCapSuppGet(clientHandle, ifNum, &buffDesc);
  printf("NULL Supported Sys Capabilities buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfLocalDeviceSysCapSuppGet(clientHandle, ifNum, &buffDesc);
  printf("NULL Supported Sys Capabilities buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfLocalDeviceSysCapSuppGet() sanity successful\n\n");

  /* openapiLldpIntfLocalDeviceSysCapEnabledGet() */
  printf("Testing openapiLldpIntfLocalDeviceSysCapEnabledGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfLocalDeviceSysCapEnabledGet(NULL, ifNum, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfLocalDeviceSysCapEnabledGet(clientHandle, ifNum, NULL);
  printf("NULL Enabled Sys Capabilities:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfLocalDeviceSysCapEnabledGet(clientHandle, ifNum, &buffDesc);
  printf("NULL Enabled Sys Capabilities buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfLocalDeviceSysCapEnabledGet(clientHandle, ifNum, &buffDesc);
  printf("NULL Enabled Sys Capabilities buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfLocalDeviceSysCapEnabledGet() sanity successful\n\n");

  /* openapiLldpIntfLocalDeviceMgmtAddressGet() */
  printf("Testing openapiLldpIntfLocalDeviceMgmtAddressGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfLocalDeviceMgmtAddressGet(NULL, ifNum, &addressFamily, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfLocalDeviceMgmtAddressGet(clientHandle, ifNum, NULL, &buffDesc);
  printf("NULL Address Family:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfLocalDeviceMgmtAddressGet(clientHandle, ifNum, &addressFamily, &buffDesc);
  printf("NULL Address buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfLocalDeviceMgmtAddressGet(clientHandle, ifNum, &addressFamily, &buffDesc);
  printf("NULL Address buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfLocalDeviceMgmtAddressGet() sanity successful\n\n");

  /* openapiLldpMedIntfModeGet() */
  printf("Testing openapiLldpMedIntfModeGet():\n");

  result = openapiLldpMedIntfModeGet(NULL, ifNum, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpMedIntfModeGet(clientHandle, ifNum, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiLldpMedIntfModeGet() sanity successful\n\n");

  /* openapiLldpMedIntfModeSet() on an LLDP-MED valid interface */
  printf("Testing openapiLldpMedIntfModeSet() on an LLDP-MED valid interface:\n");

  result = openapiLldpMedIntfModeSet(NULL, ifNum, OPEN_ENABLE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpMedIntfModeSet(clientHandle, ifNum, invalidControlMode);
  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiLldpMedIntfModeSet() sanity successful on an LLDP-MED valid interface\n\n");

  /* openapiLldpMedIntfConfigNotificationModeGet() */
  printf("Testing openapiLldpMedIntfConfigNotificationModeGet():\n");

  result = openapiLldpMedIntfConfigNotificationModeGet(NULL, ifNum, &boolMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpMedIntfConfigNotificationModeGet(clientHandle, ifNum, NULL);
  printf("NULL Bool Mode:(result = %d)\n", result);

  printf("openapiLldpMedIntfConfigNotificationModeGet() sanity successful\n\n");

  /* openapiLldpMedIntfConfigNotificationModeSet() on an LLDP-MED valid interface */
  printf("Testing openapiLldpMedIntfConfigNotificationModeSet() on an LLDP-MED valid interface:\n");

  result = openapiLldpMedIntfConfigNotificationModeSet(NULL, ifNum, OPEN_TRUE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpMedIntfConfigNotificationModeSet(clientHandle, ifNum, invalidControlMode);
  printf("Invalid Bool Mode:(result = %d)\n", result);

  printf("openapiLldpMedIntfConfigNotificationModeSet() sanity successful on an LLDP-MED valid interface\n\n");

  /* openapiLldpMedIntfTransmitTlvGet() */
  printf("Testing openapiLldpMedIntfTransmitTlvGet():\n");

  result = openapiLldpMedIntfTransmitTlvGet(NULL, ifNum, &cap);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpMedIntfTransmitTlvGet(clientHandle, ifNum, NULL);
  printf("NULL TLV:(result = %d)\n", result);

  printf("openapiLldpMedIntfTransmitTlvGet() sanity successful\n\n");

  /* openapiLldpMedIntfTransmitTlvSet() on an LLDP-MED valid interface */
  printf("Testing openapiLldpMedIntfTransmitTlvSet() on an LLDP-MED valid interface:\n");

  result = openapiLldpMedIntfTransmitTlvSet(NULL, ifNum, &cap);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpMedIntfTransmitTlvSet(clientHandle, ifNum, NULL);
  printf("NULL TLV:(result = %d)\n", result);

  printf("openapiLldpMedIntfTransmitTlvSet() sanity successful on an LLDP-MED valid interface\n\n");

  free(str);
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for LLDP.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t ifNum, testNum, arg1, arg2;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];

  if (argc < 3)
  {
    printLldpAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);
  ifNum = atoi(argv[2]);
  
  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("lldp_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting LLDP API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");
  /* exercise various OPEN API LLDP functions */

  switch (testNum)
  {
    case 1:
      if (argc != 4)
      {
        printLldpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      lldpIntfTransmitModeSet(&clientHandle, ifNum, arg1);
      break;

    case 2:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpIntfTransmitModeGet(&clientHandle, ifNum);
      break;

    case 3:
      if (argc != 4)
      {
        printLldpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      lldpIntfReceiveModeSet(&clientHandle, ifNum, arg1);
      break;

    case 4:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpIntfReceiveModeGet(&clientHandle, ifNum);
      break;
    
    case 5:
      if (argc != 5)
      {
        printLldpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      arg2 = atoi(argv[4]);
      lldpIntfTLVTransmitModeSet(&clientHandle, ifNum, arg1, arg2);
      break;

    case 6:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpIntfTLVTransmitModeGet(&clientHandle, ifNum);
      break;
  
    case 7:
      if (argc != 4)
      {
        printLldpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      lldpIntfNotificationModeSet(&clientHandle, ifNum, arg1);
      break;
  
    case 8:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpIntfNotificationModeGet(&clientHandle, ifNum);
      break;
  
    case 9:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpIntfRemoteDevicesGet(&clientHandle, ifNum);
      break;

    case 10:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpIntfLocalDeviceGet(&clientHandle, ifNum);
      break;

    case 11:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedIntfModeGet(&clientHandle, ifNum);
      break;

    case 12:
      if (argc != 4)
      {
        printLldpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      lldpMedIntfModeSet(&clientHandle, ifNum, arg1);
      break;

    case 13:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedIntfConfigNotificationModeGet(&clientHandle, ifNum);
      break;

    case 14:
      if (argc != 4)
      {
        printLldpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      lldpMedIntfConfigNotificationModeSet(&clientHandle, ifNum, arg1);
      break;

    case 15:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedInterfaceTransmitTlvGet(&clientHandle, ifNum);
      break;
    
    case 16:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedInterfaceTransmitTlvSet(&clientHandle, ifNum);
      break;

    case 17:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedRemoteDeviceInfoGet(&clientHandle, ifNum);
      break;

    case 18:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpOpENAPIsTestSanity(&clientHandle, ifNum);
      break;

    default:
      printLldpAppMenu();
      break;
  }
                                          
  /* Log goodbye message with OPEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping LLDP API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
