/*********************************************************************
*
* Copyright 2016-2017 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  bfd_show.c
*
* @purpose Display all known BFD sessions.
*
* @component OPEN
*
* @comments
*
* @create    09/09/2015
*
* @end
*
**********************************************************************/
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <signal.h>
#include <stdlib.h>
#include <pthread.h>
#include <fcntl.h>
#include <unistd.h>
#include <netdb.h>
#include <sys/ioctl.h>
#include <linux/if.h>
#include <linux/if_packet.h>
#include <sys/errno.h>
#include <ctype.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_routing.h"
#include "openapi_mpls.h"


openapiClientHandle_t clientHandle;

/*********************************************************************
* @purpose  Show list of current BFD sessions
*
* @param    client_handle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void rtrBfdSessionShow(openapiClientHandle_t *client_handle)
{
  uint32_t sessId;
  uint32_t count = 0;
  open_error_t err;
  openBfdSessionInfo_t info;
  openBfdSessionStats_t stats;
  uint32_t ip_addr;

  char *pStrInfo_bfd_header_1 = "SessionId:%u  State:%s  Diag:%d Local Disc:%u  Remote Disc:%u\r\n";
  char *pStrInfo_bfd_header_2 = "     Destination IP:%s \r\n";
  char *pStrInfo_bfd_header_2_1 = "     Next Hop IP:%s \r\n";
  char *pStrInfo_bfd_header_3 = "     Source IP:%s \r\n";
  char *pStrInfo_bfd_header_3_1 = "     MPLS Label Stack:%d %d %d\r\n";
  char *pStrInfo_bfd_header_4 = "     Interface:%u  VrfId:%u  Uptime:%u secs\r\n";
  char *pStrInfo_bfd_header_5 = "     Min Transmit Interval:%u milliseconds\r\n";
  char *pStrInfo_bfd_header_6 = "     Min Receive Interval:%u milliseconds\r\n";
  char *pStrInfo_bfd_header_7 = "     Detection interval multiplier:%u \r\n";
  char *pStrInfo_bfd_header_8 = "     In Packets:%u  Out Packets:%u  Dropped Packets:%u\r\n";
  char *pStrInfo_bfd_header_9 = "     Echo Mode:%u  Echo In Packets:%u  Echo Out Packets:%u \r\n";
  char  strSrcIp[80] = {0};
  char  strDstIp[80] = {0};
  char  strNhopIp[80] = {0};
  char  strState[80] = {0};

  /* Get the BFD session info */

  for (sessId=1; sessId < 4096; sessId++)
  {
    memset(&info, 0x0, sizeof(info));
    memset(&stats, 0x0, sizeof(stats));
    err = openapiBfdSessionInfoGet(client_handle, sessId, &info);
    if (err == OPEN_E_NONE)
    {
      count++;
      openapiBfdSessionStatsGet(client_handle, sessId, &stats);

      if (info.key.srcIpAddr.family == OPEN_AF_INET)
      {
        ip_addr = htonl (info.key.srcIpAddr.addr.ipv4);
        inet_ntop(AF_INET, &ip_addr, strSrcIp, sizeof(strSrcIp));
      }
      else if (info.key.srcIpAddr.family == OPEN_AF_INET6)
      {
        inet_ntop(AF_INET6, &info.key.srcIpAddr.addr.ipv6.u.addr8, strSrcIp, sizeof(strSrcIp));
      }

      if (info.key.dstIpAddr.family == OPEN_AF_INET)
      {
        ip_addr = htonl (info.key.dstIpAddr.addr.ipv4);
        inet_ntop(AF_INET, &ip_addr, strDstIp, sizeof(strDstIp));
      }
      else if (info.key.dstIpAddr.family == OPEN_AF_INET6)
      {
        inet_ntop(AF_INET6, &info.key.dstIpAddr.addr.ipv6.u.addr8, strDstIp, sizeof(strDstIp));
      }

      if (info.nextHopIpAddr.family == OPEN_AF_INET)
      {
        ip_addr = htonl (info.nextHopIpAddr.addr.ipv4);
        inet_ntop(AF_INET, &ip_addr, strNhopIp, sizeof(strNhopIp));
      }
      else if (info.nextHopIpAddr.family == OPEN_AF_INET6)
      {
        inet_ntop(AF_INET6, &info.nextHopIpAddr.addr.ipv6.u.addr8, strNhopIp, sizeof(strNhopIp));
      }

      snprintf(strState, sizeof(strState), "%s", 
               (info.state == OPEN_BFD_SESSION_STATE_UP) ? "UP" : "DOWN");

      printf(pStrInfo_bfd_header_1, sessId, strState, info.diag, info.localDiscr, info.remoteDiscr);
      printf(pStrInfo_bfd_header_2, strDstIp);
      printf(pStrInfo_bfd_header_2_1, strNhopIp);
      printf(pStrInfo_bfd_header_3, strSrcIp);
      printf(pStrInfo_bfd_header_3_1, info.key.mpls_label.label[0], info.key.mpls_label.label[1], info.key.mpls_label.label[2]);
      printf(pStrInfo_bfd_header_4, info.key.intIfNum, info.key.vrfId, info.upTime );
      printf(pStrInfo_bfd_header_5, info.localMinTx);
      printf(pStrInfo_bfd_header_6, info.localMinRx);
      printf(pStrInfo_bfd_header_7, info.localDetectMult);
      printf(pStrInfo_bfd_header_8, stats.inPkts, stats.outPkts, stats.dropPkts );
      printf(pStrInfo_bfd_header_9, info.echoEnable, stats.echoInPkts, stats.echoOutPkts);
    }
  }

  if (count == 0)
  {
    printf("No BFD sessions found.\r\n");
  }
}

int main(int argc, char **argv)
{
  int rc;


  /* Register with OpEN */
  if ((rc = openapiClientRegister("bfd_show", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", rc);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  rtrBfdSessionShow (&clientHandle);

  return 0;
}


