/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  mpls_router.c
*
* @purpose Performance test for adding MPLS-tagged routes via RPPI
*
* @component OPEN
*
* @comments
*
* @create    07/23/2012
*
* @end
*
**********************************************************************/
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <signal.h>
#include <stdlib.h>
#include <pthread.h>
#include <fcntl.h>
#include <unistd.h>
#include <netdb.h>
#include <sys/ioctl.h>
#include <linux/if.h>
#include <linux/if_packet.h>
#include <sys/errno.h>
#include <ctype.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_routing.h"
#include "openapi_mpls.h"


openapiClientHandle_t clientHandle;

char *protocolName = "OLY";
char *protocolCode = "O";
char *routeTypeNameStr = "OLY MPLS";
char *routeTypeCodeStr = "OM";

uint32_t protoId;
uint32_t routeType;


/*********************************************************************
* @purpose  Utility to get the current time, as number of milliseconds 
*           since the system booted.
*
* @param    void
*
* @returns  Time in milliseconds
*
* @notes
*
* @end
*********************************************************************/
uint32_t currentTimeGet(void)
{
  struct timespec tp;

  (void) clock_gettime(CLOCK_MONOTONIC, &tp);
  return ((1000 * tp.tv_sec) + (tp.tv_nsec / 1000000));
}

/*********************************************************************
* @purpose  Run the router performance test.
*
* @param    void
*
* @returns  None
*
* @notes
*
* @end
*********************************************************************/
void routerPerfTest (void)
{
  open_inet_addr_t ip_prefix;
  open_inet_addr_t ip_nhop;
  openRoute_t  route;
  openNextHop_t next_hop;
  open_buffdesc  nhop_buff;

  uint32_t total_routes;
  uint32_t start_time;
  uint32_t end_time;
  uint32_t delta_time;
  uint32_t tor, rt_per_tor;

  memset (&route, 0, sizeof (route));
  memset (&ip_prefix, 0, sizeof (ip_prefix));
  memset (&ip_nhop, 0, sizeof (ip_nhop));

  /* Create 3000 routes to 1000 ToR switches via 22.23.0.5
  */ 
  total_routes = 0;
  printf ("Adding routes to 61.0.0.0/24 via 22.23.0.5\n");
  (void) openapiSystemUpTimeMsecGet (&clientHandle, &start_time);
  for (tor = 0; tor < 1000; tor++)
  {
    for (rt_per_tor = 0; rt_per_tor < 3; rt_per_tor++)
    {
     ip_prefix.family = OPEN_AF_INET;
     ip_prefix.addr.ipv4 = 0x3d000000 + (tor << 16) + (rt_per_tor << 8);

     ip_nhop.family = OPEN_AF_INET;
     ip_nhop.addr.ipv4 = 0x16170005;

     memcpy (&route.destPfx, &ip_prefix, sizeof (ip_prefix));
     route.pfxLen = 24;
     route.routeType = routeType;
     route.protoId = protoId;
     route.pref = 19;
     route.metric = 1;
     route.numNextHops = 1;

     next_hop.ifNum = 76; /* Port-based routing interface 0/76 */
     memcpy (&next_hop.nhAddr, &ip_nhop, sizeof (ip_nhop));
     next_hop.label[0] = 1000 + tor;
     next_hop.label[1] = 0;
     next_hop.label[2] = 0;

     nhop_buff.pstart = &next_hop;
     nhop_buff.size = sizeof (next_hop);

     (void) openapiRouteAddVr (&clientHandle, 0, &route, &nhop_buff);
     total_routes++;
    }
  }
  (void) openapiSystemUpTimeMsecGet (&clientHandle, &end_time);
  delta_time = end_time - start_time;
  printf ("Added %d routes in %d ms\n", total_routes, delta_time);

  sleep (3);
  total_routes = 0;
  printf ("Modifying routes to 61.0.0.0/24 via 55.0.0.5\n");
  (void) openapiSystemUpTimeMsecGet (&clientHandle, &start_time);
  for (tor = 0; tor < 1000; tor++)
  {
    for (rt_per_tor = 0; rt_per_tor < 3; rt_per_tor++)
    {
     ip_prefix.family = OPEN_AF_INET;
     ip_prefix.addr.ipv4 = 0x3d000000 + (tor << 16) + (rt_per_tor << 8);

     ip_nhop.family = OPEN_AF_INET;
     ip_nhop.addr.ipv4 = 0x37000005;

     memcpy (&route.destPfx, &ip_prefix, sizeof (ip_prefix));
     route.pfxLen = 24;
     route.routeType = routeType;
     route.protoId = protoId;
     route.pref = 19;
     route.metric = 1;
     route.numNextHops = 1;

     next_hop.ifNum = 1; /* Port-based routing interface 0/1 */
     memcpy (&next_hop.nhAddr, &ip_nhop, sizeof (ip_nhop));
     next_hop.label[0] = 1000 + tor;
     next_hop.label[1] = 0;
     next_hop.label[2] = 0;

     nhop_buff.pstart = &next_hop;
     nhop_buff.size = sizeof (next_hop);

     (void) openapiRouteModVr (&clientHandle, 0, &route, &nhop_buff);
     total_routes++;
    }
  }
  (void) openapiSystemUpTimeMsecGet (&clientHandle, &end_time);
  delta_time = end_time - start_time;
  printf ("Modified %d routes in %d ms\n", total_routes, delta_time);


  sleep (3);
  total_routes = 0;
  printf ("Deleting routes to 61.0.0.0/24 via 55.0.0.5\n");
  (void) openapiSystemUpTimeMsecGet (&clientHandle, &start_time);
  for (tor = 0; tor < 1000; tor++)
  {
    for (rt_per_tor = 0; rt_per_tor < 3; rt_per_tor++)
    {
     ip_prefix.family = OPEN_AF_INET;
     ip_prefix.addr.ipv4 = 0x3d000000 + (tor << 16) + (rt_per_tor << 8);

     ip_nhop.family = OPEN_AF_INET;
     ip_nhop.addr.ipv4 = 0x37000005;

     memcpy (&route.destPfx, &ip_prefix, sizeof (ip_prefix));
     route.pfxLen = 24;
     route.routeType = routeType;
     route.protoId = protoId;
     route.pref = 19;
     route.metric = 1;
     route.numNextHops = 1;

     next_hop.ifNum = 1; /* Port-based routing interface 0/1 */
     memcpy (&next_hop.nhAddr, &ip_nhop, sizeof (ip_nhop));
     next_hop.label[0] = 1000 + tor;
     next_hop.label[1] = 0;
     next_hop.label[2] = 0;

     nhop_buff.pstart = &next_hop;
     nhop_buff.size = sizeof (next_hop);

     (void) openapiRouteDelVr (&clientHandle, 0, &route, &nhop_buff);
     total_routes++;
    }
  }
  (void) openapiSystemUpTimeMsecGet (&clientHandle, &end_time);
  delta_time = end_time - start_time;
  printf ("Deleted %d routes in %d ms\n", total_routes, delta_time);
}

/*********************************************************************
* @purpose  Register the routing protocol.
*
* @param    void
*
* @returns  None
*
* @notes
*
* @end
*********************************************************************/
void routeTypesRegister(void)
{
  char pCode[2];
  char rtCode[3];

  open_buffdesc  protoName;
  open_buffdesc  protoCode;
  open_buffdesc  routeTypeName;
  open_buffdesc  routeTypeCode;

  protoName.size = strlen(protocolName) + 1;
  protoName.pstart = malloc(protoName.size);
  if (protoName.pstart == NULL)
  {
    printf("Out of memory");
    exit(1);
  }
  strcpy(protoName.pstart, protocolName);

  strcpy(pCode, protocolCode);
  protoCode.size = strlen(pCode) + 1;
  protoCode.pstart = pCode;

  if (openapiRoutingProtocolRegister(&clientHandle, 0, OPEN_AF_INET, 
					&protoName, &protoCode, &protoId) != OPEN_E_NONE)
  {
    printf("\nRegistered protocol ID %u", protoId);
    free(protoName.pstart);
    exit (1);
  }
  else
  {
    free(protoName.pstart);
  }

  routeTypeName.size = strlen(routeTypeNameStr) + 1;

  routeTypeName.pstart = malloc(routeTypeName.size);
  if (routeTypeName.pstart == NULL)
  {
    printf("Out of memory");
    exit(1);
  }
  strcpy(routeTypeName.pstart, routeTypeNameStr);

  strcpy(rtCode, routeTypeCodeStr);
  routeTypeCode.size = strlen(rtCode) + 1;
  routeTypeCode.pstart = rtCode;

  if (openapiRouteTypeRegister(&clientHandle, OPEN_AF_INET, protoId, &routeTypeName,
                              &routeTypeCode, &routeType) != OPEN_E_NONE)
  {
    free(routeTypeName.pstart);
    exit (1);
  }
  else
  {
    free(routeTypeName.pstart);
  }

  printf ("ProtoId=%d routeType=%d\n", protoId, routeType);

}

int main(int argc, char **argv)
{
  int rc;

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((rc = openapiClientRegister("mpls_router", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", rc);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  routeTypesRegister ();

  routerPerfTest ();

  return 0;
}


