
/*********************************************************************
*
*  Copyright 2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  ping_example.c
*
* @purpose   Ping OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    03/29/2018
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_ping.h"

/*******************************************************************
*
* @brief  This function prints the Ping Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printPingAppMenu()
{
  printf("Usage: ping_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: Ping host or IPv4/IPv6 Address: ping_example 1 <ip-address/hostname/ipv6-address> \n");
  printf("Test 2: Ping host or IPv4/IPv6 Address with count: ping_example 2 <ip-address/hostname/ipv6-address> count <count> \n");
  printf("Test 3: Ping host or IPv4/IPv6 Address with interval: ping_example 3 <ip-address/hostname/ipv6-address> interval <interval> \n");
  printf("Test 4: Ping host or IPv4/IPv6 Address with size: ping_example 4 <ip-address/hostname/ipv6-address> size <size> \n");
  printf("Test 5: Ping host or IPv4/IPv6 Address with source interface: ping_example 5 <ip-address/hostname/ipv6-address> source <srcIfType> \n");
  printf("Test 6: Ping host or IPv4/IPv6 Address with count, size, interval and source interface: ping_example 6 <ip-address/hostname/ipv6-address>  count <count> interval <interval> size <size> source <srcIfType> \n");
  printf("Test 7: ping_example OpEN APIs sanity: ping_example 7 \n");

  return;
}

/*********************************************************************
* @purpose  Ping an IP address or hostname or IPv6 address.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    pingAddr         @b{(input)}   ping IP address or hostname or 
*                                          ipv6 address
*
* @returns  none
*
*
* @end
*********************************************************************/
void pingAddress(openapiClientHandle_t *clientHandle, char *pingAddr)
{
  open_error_t result;
  open_buffdesc pingIpAddr;
  open_buffdesc pingOutput;
  char str[OPEN_PING_ADDRESS_MAX_LEN];
  char tmpBuf[OPEN_PING_STRING_SIZE];

  memset(str, 0, sizeof(str));
  strncpy(str, pingAddr, sizeof(str));
  pingIpAddr.pstart = str;
  pingIpAddr.size = strlen(str)+1;

  memset(tmpBuf, 0, sizeof(tmpBuf));
  pingOutput.pstart = tmpBuf;
  pingOutput.size = sizeof(tmpBuf);

  printf("\nPinging %s with %d bytes of data:\n", pingAddr, OPEN_PING_DEFAULT_PROBE_SIZE);

  result = openapiPingAddress(clientHandle, &pingIpAddr, &pingOutput);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to ping address %s. (result = %d)\n", pingAddr, result);
    printf("Ping to address %s failed\n", pingAddr);
  }
  else
  {
    printf("\n----%s PING statistics----\n", pingAddr);
    printf("\n %s\n", (char *)pingOutput.pstart);
  }

  return;
}

/*********************************************************************
* @purpose  Ping an IP address or hostname or ipv6 address with count.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    pingAddr         @b{(input)}   ping IP address or hostname or
*                                          ipv6 address
* @param    pingCount        @b{(input)}   number of ping packets to send
*
* @returns  none
*
*
* @end
*********************************************************************/
void pingAddressWithCount(openapiClientHandle_t *clientHandle, char *pingAddr,
                          uint32_t pingCount)
{
  open_error_t result;
  open_buffdesc pingIpAddr;
  open_buffdesc pingOutput;
  char str[OPEN_PING_ADDRESS_MAX_LEN];
  char tmpBuf[OPEN_PING_STRING_SIZE];

  memset(str, 0, sizeof(str));
  strncpy(str, pingAddr, sizeof(str));
  pingIpAddr.pstart = str;
  pingIpAddr.size = strlen(str)+1;

  memset(tmpBuf, 0, sizeof(tmpBuf));
  pingOutput.pstart = tmpBuf;
  pingOutput.size = sizeof(tmpBuf);

  printf("\nPinging %s  for %d time(s) with %d bytes of data:\n", pingAddr, pingCount, 
                                                           OPEN_PING_DEFAULT_PROBE_SIZE);

  result = openapiPingAddressWithCount(clientHandle, &pingIpAddr, pingCount, &pingOutput);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to ping address %s. (result = %d)\n", pingAddr, result);
    printf("Ping to address %s failed\n", pingAddr);
  }
  else
  {
    printf("\n----%s PING statistics----\n", pingAddr);
    printf("\n %s\n", (char *)pingOutput.pstart);
  }

  return;
}

/*********************************************************************
* @purpose  Ping an IP address or hostname or ipv6 address with interval.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    pingAddr         @b{(input)}   ping IP address or hostname or
*                                          ipv6 address
* @param    pingInterval     @b{(input)}   interval between ping packets in 
*                                          seconds
*
* @returns  none
*
*
* @end
*********************************************************************/
void pingAddressWithInterval(openapiClientHandle_t *clientHandle, 
                             char *pingAddr,
                             uint32_t pingInterval)
{
  open_error_t result;
  open_buffdesc pingIpAddr;
  open_buffdesc pingOutput;
  char str[OPEN_PING_ADDRESS_MAX_LEN];
  char tmpBuf[OPEN_PING_STRING_SIZE];

  memset(str, 0, sizeof(str));
  strncpy(str, pingAddr, sizeof(str));
  pingIpAddr.pstart = str;
  pingIpAddr.size = strlen(str)+1;

  memset(tmpBuf, 0, sizeof(tmpBuf));
  pingOutput.pstart = tmpBuf;
  pingOutput.size = sizeof(tmpBuf);

  printf("\nPinging %s with %d bytes of data with interval of %d: seconds\n", pingAddr, OPEN_PING_DEFAULT_PROBE_SIZE,
                                                                      pingInterval);

  result = openapiPingAddressWithInterval(clientHandle, &pingIpAddr, pingInterval, &pingOutput);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to ping address %s. (result = %d)\n", pingAddr, result);
    printf("Ping to address %s failed\n", pingAddr);
  }
  else
  {
    printf("\n----%s PING statistics----\n", pingAddr);
    printf("\n %s\n", (char *)pingOutput.pstart);
  }

  return;
}

/*********************************************************************
* @purpose  Ping an IP address or hostname or ipv6 address with size.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    pingAddr         @b{(input)}   ping IP address or hostname or
*                                          ipv6 address
* @param    pingSize         @b{(input)}   size of ping packet 
*
* @returns  none
*
*
* @end
*********************************************************************/
void pingAddressWithSize(openapiClientHandle_t *clientHandle, 
                         char *pingAddr,
                         uint32_t pingSize)
{
  open_error_t result;
  open_buffdesc pingIpAddr;
  open_buffdesc pingOutput;
  char str[OPEN_PING_ADDRESS_MAX_LEN];
  char tmpBuf[OPEN_PING_STRING_SIZE];

  memset(str, 0, sizeof(str));
  strncpy(str, pingAddr, sizeof(str));
  pingIpAddr.pstart = str;
  pingIpAddr.size = strlen(str)+1;

  memset(tmpBuf, 0, sizeof(tmpBuf));
  pingOutput.pstart = tmpBuf;
  pingOutput.size = sizeof(tmpBuf);

  printf("\nPinging %s with %d bytes of data:\n", pingAddr, pingSize);

  result = openapiPingAddressWithPDUSize(clientHandle, &pingIpAddr, pingSize, 
                                         &pingOutput);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to ping address %s. (result = %d)\n", pingAddr, result);
    printf("Ping to address %s failed\n", pingAddr);
  }
  else
  {
    printf("\n----%s PING statistics----\n", pingAddr);
    printf("\n %s\n", (char *)pingOutput.pstart);
  }

  return;
}

/*********************************************************************
* @purpose  Ping an IP address or hostname or iv6 address with source interface.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    pingAddr         @b{(input)}   ping IP address or hostname
*                                          ipv6 address
* @param    pingSrcIntf      @b{(input)}   source interface
*
* @returns  none
*
*
* @end
*********************************************************************/
void pingAddressWithSrcIntf(openapiClientHandle_t *clientHandle, 
                            char *pingAddr,
                            OPEN_PING_SOURCE_INTF_TYPE_t pingSrcIntf)
{
  open_error_t result;
  open_buffdesc pingIpAddr;
  open_buffdesc pingOutput;
  char str[OPEN_PING_ADDRESS_MAX_LEN];
  char tmpBuf[OPEN_PING_STRING_SIZE];
  char sourceInterfaceStr[OPEN_PING_STRING_SIZE];

  memset(str, 0, sizeof(str));
  strncpy(str, pingAddr, sizeof(str));
  pingIpAddr.pstart = str;
  pingIpAddr.size = strlen(str)+1;

  memset(tmpBuf, 0, sizeof(tmpBuf));
  pingOutput.pstart = tmpBuf;
  pingOutput.size = sizeof(tmpBuf);

  memset(sourceInterfaceStr, 0, sizeof(sourceInterfaceStr));
  if(OPEN_PING_SOURCE_INTF_SERVICE_PORT == pingSrcIntf)
  {
    strncpy(sourceInterfaceStr, "service port", sizeof(sourceInterfaceStr));
  }
  else if(OPEN_PING_SOURCE_INTF_NETWORK_PORT == pingSrcIntf)
  {
    strncpy(sourceInterfaceStr, "network port", sizeof(sourceInterfaceStr));
  }

  printf("\nPinging %s with %d bytes of data via %s:\n", pingAddr, OPEN_PING_DEFAULT_PROBE_SIZE, 
                                                         sourceInterfaceStr);

  result = openapiPingAddressWithSourceInterface(clientHandle, &pingIpAddr, 
                                                 pingSrcIntf, &pingOutput);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to ping address %s. (result = %d)\n", pingAddr, result);
    printf("Ping to address %s failed\n", pingAddr);
  }
  else
  {
    printf("\n----%s PING statistics----\n", pingAddr);
    printf("\n %s\n", (char *)pingOutput.pstart);
  }

  return;
}
 
/*********************************************************************
* @purpose  Ping an IP address or hostname or ipv6 address explicitly 
*           specifying various parameters
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    pingAddr         @b{(input)}   ping IP address or hostname or
*                                          ipv6 address
* @param    pingCount        @b{(input)}   number of ping packets to send
* @param    pingInterval     @b{(input)}   interval between ping packets in 
* @param    pingSize         @b{(input)}   size of ping packet 
* @param    pingSrcIntf      @b{(input)}   source interface
*
* @returns  none
*
*
* @end
*********************************************************************/
void pingAddressExplicit(openapiClientHandle_t *clientHandle, 
                         char *pingAddr,
                         uint32_t pingCount,
                         uint32_t pingInterval,
                         uint32_t pingSize,
                         OPEN_PING_SOURCE_INTF_TYPE_t pingSrcIntf)
{
  open_error_t result;
  open_buffdesc pingIpAddr;
  open_buffdesc pingOutput;
  char str[OPEN_PING_ADDRESS_MAX_LEN];
  char tmpBuf[OPEN_PING_STRING_SIZE];
  char sourceInterfaceStr[OPEN_PING_STRING_SIZE];

  memset(str, 0, sizeof(str));
  strncpy(str, pingAddr, sizeof(str));
  pingIpAddr.pstart = str;
  pingIpAddr.size = strlen(str)+1;

  memset(tmpBuf, 0, sizeof(tmpBuf));
  pingOutput.pstart = tmpBuf;
  pingOutput.size = sizeof(tmpBuf);

  memset(sourceInterfaceStr, 0, sizeof(sourceInterfaceStr));
  if(OPEN_PING_SOURCE_INTF_SERVICE_PORT == pingSrcIntf)
  {
    strncpy(sourceInterfaceStr, "service port", sizeof(sourceInterfaceStr));
  }
  else if(OPEN_PING_SOURCE_INTF_NETWORK_PORT == pingSrcIntf)
  {
    strncpy(sourceInterfaceStr, "network port", sizeof(sourceInterfaceStr));
  }

  printf("\nPinging %s for %d time(s) with %d bytes of data with interval of %d seconds via %s:\n", 
          pingAddr, pingCount, pingSize, pingInterval, sourceInterfaceStr);

  result = openapiPingAddressExplicit(clientHandle, &pingIpAddr, pingCount, 
                                      pingInterval, pingSize, pingSrcIntf, 
                                      &pingOutput);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to ping address %s. (result = %d)\n", pingAddr, result);
    printf("Ping to address %s failed\n", pingAddr);
  }
  else
  {
    printf("\n----%s PING statistics----\n", pingAddr);
    printf("\n %s\n", (char *)pingOutput.pstart);
  }

  return;
}
 
/*********************************************************************
* @purpose  Sanity test Ping information
*
* @param    clientHandle   @b{(input)} client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void pingSanityTest(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc pingIpAddr;
  open_buffdesc pingOutput;
  char buf[OPEN_PING_ADDRESS_MAX_LEN];
  char tmpBuf[OPEN_PING_STRING_SIZE];
  uint32_t count = 3;
  uint32_t interval = 3;
  uint32_t size = 0;
  OPEN_PING_SOURCE_INTF_TYPE_t srcIntf = OPEN_PING_SOURCE_INTF_NETWORK_PORT;

  memset(buf, 0, sizeof(buf));
  pingIpAddr.pstart = buf;
  pingIpAddr.size = sizeof(buf);

  memset(tmpBuf, 0, sizeof(tmpBuf));
  pingOutput.pstart = tmpBuf;
  pingOutput.size = sizeof(tmpBuf);

  printf("Testing Ping OpEN APIs sanity:\n");

  /* openapiPingAddress () */
  printf("\nTesting openapiPingAddress(): \n");

  result = openapiPingAddress(NULL, &pingIpAddr, &pingOutput);
  printf("NULL Client Handle. (result = %d)\n", result);
  
  result = openapiPingAddress(clientHandle, NULL, &pingOutput);
  printf("NULL buff descriptor to ping address. (result = %d)\n", result);
  
  result = openapiPingAddress(clientHandle, &pingIpAddr, NULL);
  printf("Invalid ping output. (result = %d)\n", result);
  
  printf("openapiPingAddress() sanity successful. \n");

  /* openapiPingAddressWithCount() */
  printf("\nTesting openapiPingAddressWithCount(): \n");

  result = openapiPingAddressWithCount(NULL, &pingIpAddr, count, &pingOutput);
  printf("NULL Client Handle. (result = %d)\n", result);
  
  result = openapiPingAddressWithCount(clientHandle, NULL, count, &pingOutput);
  printf("NULL buff descriptor to ping address. (result = %d)\n", result);
  
  result = openapiPingAddressWithCount(clientHandle, &pingIpAddr, 0, &pingOutput);
  printf("Invalid ping packet count. (result = %d)\n", result);
  
  result = openapiPingAddressWithCount(clientHandle, &pingIpAddr, count, NULL);
  printf("Invalid ping output. (result = %d)\n", result);
  
  printf("openapiPingAddressWithCount() sanity successful. \n");

  /* openapiPingAddressWithInterval() */
  printf("\nTesting openapiPingAddressWithInterval(): \n");

  result = openapiPingAddressWithInterval(NULL, &pingIpAddr, interval, &pingOutput);
  printf("NULL Client Handle. (result = %d)\n", result);
  
  result = openapiPingAddressWithInterval(clientHandle, NULL, interval, &pingOutput);
  printf("NULL buff descriptor to ping address. (result = %d)\n", result);
  
  result = openapiPingAddressWithInterval(clientHandle, &pingIpAddr, 0, &pingOutput);
  printf("Invalid interval between ping packets in seconds. (result = %d)\n", result);
  
  result = openapiPingAddressWithInterval(clientHandle, &pingIpAddr, interval, NULL);
  printf("Invalid ping output. (result = %d)\n", result);
  
  printf("openapiPingAddressWithInterval() sanity successful. \n");

  /* openapiPingAddressWithPDUSize() */
  printf("\nTesting openapiPingAddressWithPDUSize(): \n");

  result = openapiPingAddressWithPDUSize(NULL, &pingIpAddr, size, &pingOutput);
  printf("NULL Client Handle. (result = %d)\n", result);
  
  result = openapiPingAddressWithPDUSize(clientHandle, NULL, size, &pingOutput);
  printf("NULL buff descriptor to ping address. (result = %d)\n", result);
  
  result = openapiPingAddressWithPDUSize(clientHandle, &pingIpAddr, 13001, &pingOutput);
  printf("Invalid ping packet size. (result = %d)\n", result);
  
  result = openapiPingAddressWithPDUSize(clientHandle, &pingIpAddr, size, NULL);
  printf("Invalid ping output. (result = %d)\n", result);
  
  printf("openapiPingAddressWithPDUSize() sanity successful. \n");

  /* openapiPingAddressWithSourceInterface() */
  printf("\nTesting openapiPingAddressWithSourceInterface(): \n");

  result = openapiPingAddressWithSourceInterface(NULL, &pingIpAddr, srcIntf, &pingOutput);
  printf("NULL Client Handle. (result = %d)\n", result);
  
  result = openapiPingAddressWithSourceInterface(clientHandle, NULL, srcIntf, &pingOutput);
  printf("NULL buff descriptor to ping address. (result = %d)\n", result);
  
  result = openapiPingAddressWithSourceInterface(clientHandle, &pingIpAddr, 10, &pingOutput);
  printf("Invalid ping source interface. (result = %d)\n", result);
  
  result = openapiPingAddressWithSourceInterface(clientHandle, &pingIpAddr, srcIntf, NULL);
  printf("Invalid ping output. (result = %d)\n", result);
  
  printf("openapiPingAddressWithSourceInterface() sanity successful. \n");

  /* openapiPingAddressExplicit() */
  printf("\nTesting openapiPingAddressExplicit(): \n");

  result = openapiPingAddressExplicit(NULL, &pingIpAddr, count,
                                           interval, size, srcIntf, &pingOutput);
  printf("NULL Client Handle. (result = %d)\n", result);
  
  result = openapiPingAddressExplicit(clientHandle, NULL, count,
                                           interval, size, srcIntf, &pingOutput);
  printf("NULL buff descriptor to ping address. (result = %d)\n", result);
  
  result = openapiPingAddressExplicit(clientHandle, &pingIpAddr, 0,
                                           interval, size, srcIntf, &pingOutput);
  printf("Invalid ping packet count. (result = %d)\n", result);
  
  result = openapiPingAddressExplicit(clientHandle, &pingIpAddr, count,
                                           0, size, srcIntf, &pingOutput);
  printf("Invalid interval between ping packets in seconds. (result = %d)\n", result);
  
  result = openapiPingAddressExplicit(clientHandle, &pingIpAddr, count,
                                           interval, 13001, srcIntf, &pingOutput);
  printf("Invalid ping packet size. (result = %d)\n", result);
  
  result = openapiPingAddressExplicit(clientHandle, &pingIpAddr, count,
                                           interval, size, 10, &pingOutput);
  printf("Invalid ping source interface. (result = %d)\n", result);
  
  result = openapiPingAddressExplicit(clientHandle, &pingIpAddr, count, 
                                           interval, size, srcIntf, NULL);
  printf("Invalid ping output. (result = %d)\n", result);
  
  printf("openapiPingAddressExplicit() sanity successful. \n");

  return;
}
/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  uint32_t arg1, arg2, arg3, arg4;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  
  if (argc < 2)
  {
    printPingAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("ping_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Ping API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS Version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 3)
      {
        printPingAppMenu();
        exit(1);
      }
      pingAddress(&clientHandle, argv[2]);
      break;

    case 2:
      if (argc != 5)
      {
        printPingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[4]);
      pingAddressWithCount(&clientHandle, argv[2], arg1);
      break;

    case 3:
      if (argc != 5)
      {
        printPingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[4]);
      pingAddressWithInterval(&clientHandle, argv[2], arg1);
      break;

    case 4:
      if (argc != 5)
      {
        printPingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[4]);
      pingAddressWithSize(&clientHandle, argv[2], arg1);
      break;

    case 5:
      if (argc != 5)
      {
        printPingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[4]);
      pingAddressWithSrcIntf(&clientHandle, argv[2], arg1);
      break;

    case 6:
      if (argc != 11)
      {
        printPingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[4]);
      arg2 = atoi(argv[6]);
      arg3 = atoi(argv[8]);
      arg4 = atoi(argv[10]);
      pingAddressExplicit(&clientHandle, argv[2], arg1, arg2, arg3, arg4);
      break;

    case 7:
      if (argc != 2)
      {
        printPingAppMenu();
        exit(1);
      } 
      pingSanityTest(&clientHandle); 
      break;

    default:
      printPingAppMenu();
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Ping API example application");

  return 0;
}
