=begin
*
* Copyright 2016-2018 Broadcom.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0*
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
=end

require "OpEN"
require "OpENUtil"

Puppet::Type.type(:netdev_lag).provide(:icos) do
  confine :exists => "/etc/brl-release"
  @doc = "NOS netdev LAG interface"

  mk_resource_methods

  def exists?
    init_resource
    @property_hash[:ensure] == :present
  end

  def create
    begin
      printf "netdev_stdlib_icos-lag create\n"

      open = OpENUtil.new()
      open.connect("netdev_stdlib_icos-lag")

      # Get the internal interface number
      lag_intf_num_p  = OpEN.new_uint32_tp()
      buffdesc        = OpEN::Open_buffdesc.new
      buffdesc.size   = resource[:name].length + 1
      buffdesc.pstart = open.getCharBuffer(buffdesc.size, resource[:name])
      ret = OpEN::openapiIfNumGet(open.client, buffdesc, lag_intf_num_p)

      if ret == OpEN::OPEN_E_NONE
        lag_intf_num = OpEN.uint32_tp_value(lag_intf_num_p)
      
        # Post a info message that the LAG interface can't be created.
        # Only its properties can be set.
        info('LAG interface can not be created. Only its properties can be set.')

        # Set the LAG interface lacp mode
        if resource[:lacp] == :active || resource[:lacp] == :passive
          static_mode = OpEN::OPEN_DISABLE
        else
          static_mode = OpEN::OPEN_ENABLE
        end

        ret = OpEN::openapiLagStaticModeSet(open.client, lag_intf_num, static_mode)
        if ret != OpEN::OPEN_E_NONE
          printf "Failed to set LAG interface %s lacp mode, return value = %d\n", resource[:name], ret
          warning('Failed to set LAG interface lacp mode.')
        end

        # Set the minimum uplinks for the LAG interface
        ret = OpEN::openapiLagMinUplinksSet(open.client, lag_intf_num, resource[:minimum_links].to_i)
        if ret != OpEN::OPEN_E_NONE
          printf "Failed to set LAG interface %s minimum uplinks %d, return value = %d\n", resource[:name], resource[:minimum_links].to_i, ret
          warning('Failed to set LAG interface minimum uplinks.')
        end

        # Set the dynamic LAG lacp mode
        if resource[:lacp] == :active
          lacp_mode = OpEN::OPEN_ENABLE
        else
          lacp_mode = OpEN::OPEN_DISABLE
        end

        intf_num_p      = OpEN.new_uint32_tp()
        # Add the physical interfaces to the LAG
        resource[:links].each do |interface|
          # Get the internal interface number
          buffdesc.size   = interface.to_s.length + 1
          buffdesc.pstart = open.getCharBuffer(buffdesc.size, interface.to_s)
          ret = OpEN::openapiIfNumGet(open.client, buffdesc, intf_num_p)
          if ret == OpEN::OPEN_E_NONE
            intf_num = OpEN.uint32_tp_value(intf_num_p)
            ret1 = OpEN::openapiLagPortAdd(open.client, lag_intf_num, intf_num)
            if static_mode != OpEN::OPEN_ENABLE
              ret2 = OpEN::openapiDot3adAggPortLacpModeSet(open.client, intf_num, lacp_mode)
            else
              ret2 = OpEN::OPEN_E_NONE
            end
            if ret1 != OpEN::OPEN_E_NONE || ret2 != OpEN::OPEN_E_NONE
              printf "Failed to add interface %s to the LAG %s, return value = %d, %d\n", interface, resource[:name], ret1, ret2
              warning('Failed to add interface to LAG interface.')
            end
          end
        end

      else
        printf "LAG Interface %s can not be created, return value = %d\n", resource[:name], ret
        fail('Failed to create LAG interface.')
      end

      open.terminate()

      @property_hash[:ensure] = :present

    rescue Puppet::ExecutionFailure => e
      printf "Failed to create LAG interface\n"

    ensure
      if lag_intf_num_p
        OpEN.delete_uint32_tp(lag_intf_num_p)
      end
      if intf_num_p
        OpEN.delete_uint32_tp(intf_num_p)
      end

    end
  end

  def destroy
    begin
      printf "netdev_stdlib_icos-lag destroy\n"

      open = OpENUtil.new()
      open.connect("netdev_stdlib_icos-lag")

      # Get the internal interface number
      lag_intf_num_p = OpEN.new_uint32_tp()
      buffdesc        = OpEN::Open_buffdesc.new
      buffdesc.size   = resource[:name].length + 1
      buffdesc.pstart = open.getCharBuffer(buffdesc.size, resource[:name])
      ret = OpEN::openapiIfNumGet(open.client, buffdesc, lag_intf_num_p)

      if ret == OpEN::OPEN_E_NONE
        lag_intf_num = OpEN.uint32_tp_value(lag_intf_num_p)

        # Post a info message that the LAG interface can't be deleted.
        # Only its properties can be reset.
        info('LAG interface can not be deleted. Only its properties can be reset.')

        # Reset the LAG interface lacp mode disabled i.e., static LAG
        static_mode = OpEN::OPEN_ENABLE
        ret = OpEN::openapiLagStaticModeSet(open.client, lag_intf_num, static_mode)
        if ret != OpEN::OPEN_E_NONE
          printf "Failed to reset LAG interface %s lacp mode, return value = %d\n", resource[:name], ret
          warning('Failed to reset LAG interface lacp mode.')
        end

        intf_num_p      = OpEN.new_uint32_tp()
        lacp_mode       = OpEN::OPEN_ENABLE
        # Remove the physical interfaces from the LAG
        resource[:links].each do |interface|
          # Get the internal interface number
          buffdesc.size   = interface.to_s.length + 1
          buffdesc.pstart = open.getCharBuffer(buffdesc.size, interface.to_s)
          ret = OpEN::openapiIfNumGet(open.client, buffdesc, intf_num_p)
          if ret == OpEN::OPEN_E_NONE
            intf_num = OpEN.uint32_tp_value(intf_num_p)
            ret1 = OpEN::openapiLagPortDelete(open.client, lag_intf_num, intf_num)
            ret2 = OpEN::openapiDot3adAggPortLacpModeSet(open.client, intf_num, lacp_mode)
            if ret1 != OpEN::OPEN_E_NONE || ret2 != OpEN::OPEN_E_NONE
              printf "Failed to remove interface %s from the LAG %s, return value = %d\n", interface, resource[:name], ret
              warning('Failed to remove interface from LAG interface.')
            end
          end
        end
      else
        fail('Failed to delete LAG interface.')
      end

      open.terminate()

      @property_hash.clear

    rescue Puppet::ExecutionFailure => e
      printf "Failed to destroy LAG interface\n"

    ensure
      if lag_intf_num_p
        OpEN.delete_uint32_tp(lag_intf_num_p)
      end
      if intf_num_p
        OpEN.delete_uint32_tp(intf_num_p)
      end

    end
  end

  def self.instances
    begin
      printf "netdev_stdlib_icos-lag self.instances\n"

      open = OpENUtil.new()
      open.connect("netdev_stdlib_icos-lag")

      # Get the maximum supported members in any LAG interface
      lag_max_member_count_p = OpEN.new_uint32_tp()
      OpEN::openapiLagMaxMemberCountGet(open.client, lag_max_member_count_p)
      lag_max_member_count = OpEN.uint32_tp_value(lag_max_member_count_p)

      lag_interfaces = []

      # Get the internal interface number
      lag_intf_num   = 0
      lag_intf_num_p = OpEN.new_uint32_tp()
      intf_type = OpEN::OPEN_INTF_TYPE_LAG

      buffdesc           = OpEN::Open_buffdesc.new
      static_mode_p      = OpEN.new_OPEN_CONTROL_tp()
      lacp_mode_p        = OpEN.new_OPEN_CONTROL_tp()
      lag_min_up_links_p = OpEN.new_uint32_tp()
      lag_member_count_p = OpEN.new_uint32_tp()
      lag_buffdesc       = OpEN::Open_buffdesc.new
      lag_mem_buffdesc   = OpEN::Open_buffdesc.new

      ret = OpEN::openapiIfNextGet(open.client, intf_type, lag_intf_num, lag_intf_num_p)
      while ret == OpEN::OPEN_E_NONE
        lag_intf_num = OpEN.uint32_tp_value(lag_intf_num_p)
        lag_members  = []

        # Get the interface name
        buffdesc.size   = 256 
        name = open.getCharBuffer(buffdesc.size, nil)
        buffdesc.pstart = name 
        OpEN::openapiIfNameGet(open.client, lag_intf_num, buffdesc)
        lag_intf_name = name.cast()

        # Get the LAG interface mode
        OpEN::openapiLagStaticModeGet(open.client, lag_intf_num, static_mode_p)
        static_mode = OpEN.OPEN_CONTROL_tp_value(static_mode_p)
        if static_mode == OpEN::OPEN_ENABLE
          lacp = :disabled
        end

        # Get the LAG minimum up links
        OpEN::openapiLagMinUplinksGet(open.client, lag_intf_num, lag_min_up_links_p)
        lag_min_up_links = OpEN.uint32_tp_value(lag_min_up_links_p)

        lag_buffdesc.size   = lag_max_member_count * 4
        tmpIntBuf           = open.getIntBuffer(lag_max_member_count) 
        lag_buffdesc.pstart = tmpIntBuf 

        OpEN.uint32_tp_assign(lag_member_count_p, lag_max_member_count)
        lag_member_count = OpEN.uint32_tp_value(lag_member_count_p)
        ret = OpEN::openapiLagMembersGet(open.client, lag_intf_num, lag_member_count_p, lag_buffdesc)
        lag_member_count = OpEN.uint32_tp_value(lag_member_count_p)

        for x in 0..(lag_member_count - 1)
          intf_num = tmpIntBuf[x]

          if static_mode == OpEN::OPEN_DISABLE
            ret = OpEN::openapiDot3adAggPortLacpModeGet(open.client, intf_num, lacp_mode_p)
            if ret == OpEN::OPEN_E_NONE
              lacp_mode = OpEN.OPEN_CONTROL_tp_value(lacp_mode_p)
              if lacp_mode == OpEN::OPEN_ENABLE
                lacp = :active
              else
                lacp = :passive
              end
            end
          end

          # Get the LAG member interface name
          lag_mem_buffdesc.size   = 256 
          tmpCharBuf              = open.getCharBuffer(lag_mem_buffdesc.size, nil)
          lag_mem_buffdesc.pstart = tmpCharBuf 
          OpEN::openapiIfNameGet(open.client, intf_num, lag_mem_buffdesc)
          intf_name = tmpCharBuf.cast() 

          lag_members << intf_name
        end

        lag_interfaces << new(:name => lag_intf_name, :lacp => lacp, :minimum_links => lag_min_up_links, :links => lag_members, :ensure => :present)

        ret = OpEN::openapiIfNextGet(open.client, intf_type, lag_intf_num, lag_intf_num_p)
      end

      open.terminate()

      OpEN.delete_uint32_tp(lag_max_member_count_p)
      OpEN.delete_uint32_tp(lag_intf_num_p)
      OpEN.delete_OPEN_CONTROL_tp(static_mode_p)
      OpEN.delete_OPEN_CONTROL_tp(lacp_mode_p)
      OpEN.delete_uint32_tp(lag_min_up_links_p)
      OpEN.delete_uint32_tp(lag_member_count_p)

      lag_interfaces
    end
  end

  def self.prefetch(resources)
    begin
      printf "netdev_stdlib_icos-lag self.prefetch\n"

      lag_interfaces = instances
      found          = nil
      resources.each do |name, params|
        if params[:ensure] == :present # This is to support re-configuration
          if provider = lag_interfaces.find { |lag_interface| (lag_interface.name == name) and (lag_interface.lacp == params[:lacp]) and (lag_interface.minimum_links == params[:minimum_links]) }
            params[:links].each do |interface|
              found = provider.links.find { |my_interface| my_interface == interface.to_s }
              break if found == nil
            end

            if (found != nil)
              resources[name].provider = provider
            end
          end
        elsif params[:ensure] == :absent
          if provider = lag_interfaces.find { |lag_interface| lag_interface.name == name }
            resources[name].provider = provider
          end
        end
      end
    end
  end

  def init_resource
    resource[:links] = resource[:links].to_a || []
  end

end
