/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  arptable_example.c
*
* @purpose   Routing ARP Status APIs Example.
*
* @component OPEN
*
* @comments
*
* @create    12/05/2016
*
* @end
*
**********************************************************************/
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <stdlib.h>
#include <ctype.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <getopt.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_routing_arptable.h"
#include "openapi_routing_config.h"
#include "openapi_fdb.h"

/*********************************************************************
* @purpose  Emulate cli "show arp brief"
*
* @param    client_handle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/

void showArpBrief(openapiClientHandle_t *client_handle)
{
  uint32_t val, val2, val3;
  OPEN_CONTROL_t mode;

  if (openapiArpAgeTimeGet(client_handle, &val) == OPEN_E_NONE)
  {
    printf("Age Time (seconds)............................. %d\n", val);
  }
  else
  {
    printf("Failed to get age time\n");
  } 
  if (openapiArpResponseTimeGet(client_handle, &val) == OPEN_E_NONE)
  {
    printf("Response Time (seconds)........................ %d\n", val);
  }
  else
  {
    printf("Failed to get response time\n");
  } 
  if (openapiArpNumRetriesGet(client_handle, &val) == OPEN_E_NONE)
  {
    printf("Retries........................................ %d\n", val);
  }
  else
  {
    printf("Failed to get retry count\n");
  } 
  if (openapiArpCacheSizeGet(client_handle, &val) == OPEN_E_NONE)
  {
    printf("Cache Size..................................... %d\n", val);
  }
  else
  {
    printf("Failed to get cache size\n");
  } 
  if (openapiArpDynamicRenewModeGet(client_handle, &mode) == OPEN_E_NONE)
  {
    printf("Dynamic Renew Mode ............................ %s\n", mode == OPEN_ENABLE ? "Enable" : "Disable");
  }
  else
  {
    printf("Failed to get dynamic renew mode\n");
  } 
  if (openapiArpTotalEntryCountInfoGet(client_handle, &val, &val2) == OPEN_E_NONE)
  {
    printf("Total Entry Count Current / Peak .............. %d / %d\n", val, val2);
  }
  else
  {
    printf("Failed to get entry count info\n");
  } 
  if (openapiArpStaticEntryCountInfoGet(client_handle, &val, &val2, &val3) == OPEN_E_NONE)
  {
    printf("Static Entry Count Configured / Active / Max .. %d / %d / %d\n", val, val2, val3);
  }
  else
  {
    printf("Failed to get static entry count info\n");
  } 
}

/*********************************************************************
* @purpose  Emulate cli "show arp" and "show arp vrf vrf_name"
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    client_handle    @b{(input)}   vrf name, or OPEN_DEFAULT_VRF_NAME
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/

void showArpVrf(openapiClientHandle_t *client_handle, const char *name)
{
  open_error_t rc;
  uint32_t intf = 0;
  open_inet_addr_t ipAddr;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1];
  char ipAddrStr[24];
  open_buffdesc vrfName;
  OPEN_ARP_ENTRY_t entry;
  int len;
  int headerShown = 0;
  uint32_t ip;
  open_buffdesc ifName;
  char ifNameStr[OPEN_INTF_NAME_MAX_LENGTH];

  if ((len = strlen(name)) > OPEN_VRF_MAX_NAME_LEN)
  {
    printf("VRF name %s is too long\n", name);
    return;
  }

  strncpy(vrfNameStr, name, OPEN_VRF_MAX_NAME_LEN);
  vrfName.pstart = vrfNameStr;
  vrfName.size = sizeof(vrfNameStr) - 1;

  ipAddr.family = OPEN_AF_INET;
  ipAddr.addr.ipv4 = 0;

  ifName.pstart = ifNameStr;
  ifName.size = OPEN_INTF_NAME_MAX_LENGTH;

  rc = OPEN_E_NONE;

  while (rc == OPEN_E_NONE)
  {
    rc = openapiArpEntryVrfNextGet(client_handle, &vrfName, &intf, &ipAddr);
    if (rc == OPEN_E_NONE)
    {
      rc = openapiArpEntryInfoGet(client_handle, &vrfName, intf, ipAddr,
                                  &entry);
      if (rc != OPEN_E_NONE) 
      {
        printf("openapiArpEntryInfoGet failed\n");
        break;
      }
      ip = htonl(ipAddr.addr.ipv4);
      if (inet_ntop(AF_INET, (void*)&ip, ipAddrStr, sizeof(ipAddrStr)) == 0)
      {
        printf("Bad return code trying to convert IP.\n");
        return;
      }

      if (!headerShown)
      {
        printf("IP Address        MAC Address        Interface        Type        Age\n");
        printf("----------------  -----------------  --------------   --------    -----------\n");
        headerShown = 1;
      }
     
      rc = openapiIntfNameGet(client_handle, intf, &ifName); 
      printf("%-16s  %02X:%02X:%02X:%02X:%02X:%02X  %-15s  ", ipAddrStr,
              entry.macAddr[0], entry.macAddr[1], entry.macAddr[2], entry.macAddr[3], entry.macAddr[4], entry.macAddr[5], ifNameStr);
      if ((entry.type & OPEN_ARP_LOCAL) != 0)
      {
        printf("%-8s    n/a\n", "Local");
      }
      else if ((entry.type & OPEN_ARP_GATEWAY) != 0)
      {
        printf("%-8s    %d\n", "Gateway", entry.age);
      }
      else if ((entry.type & OPEN_ARP_STATIC) != 0)
      {
        printf("%-8s    n/a\n", "Static");
      }
      else if ((entry.type & OPEN_ARP_KERNEL) != 0)
      {
        printf("%-8s    n/a\n", "Kernel");
      }
      else
      {
        printf("%-8s    %d\n", "Dynamic", entry.age);
      }
    }
  }
}

void print_usage(const char *name)
{
  printf("%s [-v vrfname]\n", name);
  exit(0);
}

int main(int argc, char *argv[])
{
  openapiClientHandle_t clientHandle;
  char vrfName[OPEN_VRF_MAX_NAME_LEN + 1];
  int option = 0;
  int rc;

  vrfName[0] = '\0';

  while ((option = getopt(argc, argv,"v:")) != -1) {
    switch (option) {
      case 'v' : 
        strncpy(vrfName, optarg, sizeof(vrfName));
        break;
      default: 
        print_usage(argv[0]); 
        break;
    }
  }

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((rc = openapiClientRegister("routing_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", rc);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  showArpBrief(&clientHandle);

  if (strlen(vrfName))
  {
    /* show a specific arp VRF entry (specify with -v argument) */

    showArpVrf(&clientHandle, vrfName);
  }
  else 
  {
    /* show all arp VRF entries */

    showArpVrf(&clientHandle, "");
  }
  
  return 0;
}


