/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  routing_example.c
*
* @purpose   Routing Configuration and Status APIs Example.
*
* @component OPEN
*
* @comments
*
* @create    07/23/2012
*
* @end
*
**********************************************************************/
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <stdlib.h>
#include <ctype.h>
#include <arpa/inet.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_routing.h"


/*********************************************************************
* @purpose  Convert an IP address in a open_inet_addr_t structure to
*           a string for display
*
* @param    addr  @b{(input)}  address to be converted
* @param    buffer @b{(output)} storage to contain
*
* @returns  pointer to buffer if conversion successful
*           NULL if error
*
* @notes
*
* @end
*********************************************************************/
static const char *ipAddressFormat(open_inet_addr_t *addr, char *buffer)
{
  uint32_t  ip4_addr;
  uint8_t   addr8[16];
  int af;

  switch (addr->family)
  {
  case OPEN_AF_INET:
    af = AF_INET;
    ip4_addr = htonl(addr->addr.ipv4);
    memcpy(addr8, &ip4_addr, sizeof(ip4_addr));
    break;
  case OPEN_AF_INET6:
    af = AF_INET6;
    memcpy(addr8, addr->addr.ipv6.u.addr8, sizeof(addr8));
    break;
  default:
    return(NULL);
  }

  return(inet_ntop(af, addr8, buffer, INET6_ADDRSTRLEN));
}

/*******************************************************************
*
* @brief  This function prints the Routing Configuration Example 
*         Application Menu.
*
* @param    none
*
* @returns  none
*
* @end
*********************************************************************/
void printRoutingAppMenu()
{
  printf("Usage: routing_example API <test number> [<arg1>][<arg2>] ... \n");
  printf("Test API 1: Add IP address on an interface: routing_example API 1 <interface> "
         "<address family> <address type> <IP Address> <Pfx length> <arg1>\n");
  printf("Test API 2: Delete IP address on an interface: routing_example API 2 <interface> "
         "<address family> <address type> <IP Address> <Pfx length> <arg1>\n");
  printf("Test API 3: Enable/Disable IP Routing Globally: routing_example API 3 <address family> <mode>\n");
  printf("Test API 4: Add a test static ARP entry {192.168.1.1 00:11:22:33:44:55} : routing_example API 4\n");
  printf("Test API 5: Delete the test static ARP entry {192.168.1.1 00:11:22:33:44:55}: routing_example API 5\n");
  printf("Test API 6: Display all ARP entries: routing_example API 6\n");
  printf("Test API 7: Enable/Disable IP Redirects mode: routing_example API 7 <mode>\n");
  printf("Test API 8: Enable/Disable IP ICMP Echo Reply mode: routing_example API 8 <mode>\n");
  printf("Test API 9: Enable/Disable IP Helper mode: routing_example API 9 <mode>\n");
  printf("Test API 10: Add IP Helper address: routing_example API 10 <IP Address> <UDP Port>\n");
  printf("Test API 11: Delete IP Helper address: routing_example API 11 <IP Address> <UDP Port>\n");
  printf("Test API 12: Configure Interface MTU: routing_example API 12 <interface> <address family> <MTU>\n");
  printf("Test API 13: Enable/Disable IP Net Directed BCast mode: routing_example API 13 <interface> <mode>\n");
  printf("Test API 14: Enable/Disable IP Proxy ARP mode: routing_example API 14 <interface> <mode>\n");
  printf("Test API 15: Enable/Disable IP Redirects mode: routing_example API 15 <interface> <mode>\n");
  printf("Test API 16: Enable/Disable IP Destination Unreachable mode: routing_example API 16 <interface> <mode>\n");
  printf("Test API 17: Enable/Disable IP Interface Routing mode: routing_example API 17 <interface> <mode>\n");
  printf("Test API 18: Add Interface IP Helper address: routing_example API 18 <interface> <IP Address> <UDP Port>\n");
  printf("Test API 19: Delete Interface IP Helper address: routing_example API 19 <interface> <IP Address> <UDP Port>\n");
  printf("Test API 20: Add Interface IP Helper Discard entry: routing_example API 20 <interface> <UDP Port>\n");
  printf("Test API 21: Delete Interface IP Helper Discard entry: routing_example API 21 <interface> <UDP Port>\n");
  printf("Test API 22: Add an IP route: routing_example API 22 <Address Family> <IP Address> <Pfx length> "
         "<Next Hop> <rtePref> <interface> <mpls-label1> <mpls-label2> <mpls-label3>\n");
  printf("Test API 23: Delete an IP route: routing_example API 23 <Address Family> <IP Address> <Pfx length> "
         "<Next Hop> <interface> <mpls-label1> <mpls-label2> <mpls-label3>\n");
  printf("Test API 24: Create a VLAN Routing Interface: routing_example API 24 <VLAN Id> \n");
  printf("Test API 25: Delete a VLAN Routing Interface: routing_example API 25 <VLAN Id> \n");
  printf("Test API 26: Test Routing Configuration OpEN APIs sanity: routing_example API 26 \n");
  printf("Test API 27: Create a BFD session: routing_example API 27 <Address family> <Intf> <Dest IP> <Src IP> \n");
  printf("Test API 28: Delete a BFD session: routing_example API 28 <Address family> <Intf> <Dest IP> <Src IP> \n");
  printf("Test API 29: Show BFD neighbors: routing_example API 29\n");
  printf("Test API 30: Find a BFD session: routing_example API 30 <Address family> <Dest IP>\n");
  printf("Test API 31: Iterate the configred VRFs\n");
  printf("Test API 32: Show the ifnum for a VLAN Routing Interface: routing_example API 32 <VLAN Id>\n");

  return;
}

/*********************************************************************
* @purpose  Add an IP address on a given router interface for a given address family.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   Address Family (IPv4 or IPv6)
* @param    intf             @b{(input)}   A routing interface ID
* @param    addrType         @b{(input)}   Address Type
* @param    IPStr            @b{(input)}   IP address and prefix length. This argument is not
*                                          applicable when addrType of DHCP is used.  
* @param    pfxLen           @b{(input)}   Prefix Length
* @param    extArg           @b{(input)}   If used for Address Family IPv4:
*                                          Enable the DHCP client to specify the unique client 
*                                          Id Option 61.
*                                          If used for Address Family IPv6:
*                                          Enable use of eui-64 Interface Identifier.
*
* @returns  Void
*
* @notes Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void rtrIntfIpAddrAdd(openapiClientHandle_t *client_handle,
                      OPEN_AF_t af,uint32_t intf, 
                      OPEN_INTF_IP_ADDR_TYPE_t addrType,
                      char *IPStr, uint32_t pfxLen,
                      OPEN_CONTROL_t extArg)
{
  open_error_t result;
  open_inet_pfx_t ipAddr;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, IPStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(&ipAddr, 0, sizeof(ipAddr));

  if ((addrType == OPEN_IP_ADDR_TYPE_STATIC_PRIMARY) || (addrType == OPEN_IP_ADDR_TYPE_STATIC_SECONDARY))
  {
    if (af == OPEN_AF_INET)
    {
      if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &(ipAddr.ipAddr))) != OPEN_E_NONE)
      {
        printf("Bad return code trying to convert IP address. (result = %d)\n", result);
        return;
      }
      printf("IP = %u, PfxLen = %u\n", ipAddr.ipAddr.addr.ipv4, pfxLen);

      ipAddr.pfxLen = pfxLen;
      ipAddr.ipAddr.family = OPEN_AF_INET;
    }
    else /* (af == OPEN_AF_INET6) */
    {
      if (inet_pton(AF_INET6, IPStr, (void*)&(ipAddr.ipAddr.addr.ipv6)) < 0)
      {
        printf("Bad return code trying to convert IP.\n");
      }

      ipAddr.pfxLen = pfxLen;
      ipAddr.ipAddr.family = OPEN_AF_INET6;
    }
  }
  
  if ((result = openapiRtrIntfIpAddrAdd(client_handle, af, intf, addrType, &ipAddr, extArg)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to add Interface IP. (result = %d)\n", result);
  }
  else
  {
    printf("Interface IP added successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Deletes an IP address on a given router interface for a given address family.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   Address Family (IPv4 or IPv6)
* @param    intf             @b{(input)}   A routing interface ID
* @param    addrType         @b{(input)}   Address Type
* @param    IPStr            @b{(input)}   IP address and prefix length. This argument is not
*                                          applicable when addrType of DHCP is used.  
* @param    pfxLen           @b{(input)}   Prefix Length
* @param    extArg           @b{(input)}   If used for Address Family IPv4:
*                                          Enable the DHCP client to specify the unique client 
*                                          Id Option 61.
*                                          If used for Address Family IPv6:
*                                          Enable use of eui-64 Interface Identifier.
*
* @returns  none
*
* @notes Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void rtrIntfIpAddrDel(openapiClientHandle_t *client_handle,
                      OPEN_AF_t af,uint32_t intf, 
                      OPEN_INTF_IP_ADDR_TYPE_t addrType,
                      char *IPStr, uint32_t pfxLen,
                      OPEN_CONTROL_t extArg)
{
  open_error_t result;
  open_inet_pfx_t ipAddr;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, IPStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(&ipAddr, 0, sizeof(ipAddr));

  if ((addrType == OPEN_IP_ADDR_TYPE_STATIC_PRIMARY) || (addrType == OPEN_IP_ADDR_TYPE_STATIC_SECONDARY))
  {
    if (af == OPEN_AF_INET)
    {
      if (strcmp(IPStr, "all") != 0)
      {
        if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &(ipAddr.ipAddr))) != OPEN_E_NONE)
        {
          printf("Bad return code trying to convert IP address. (result = %d)\n", result);
          return;
        }
        printf("IP = %u, PfxLen = %u\n", ipAddr.ipAddr.addr.ipv4, pfxLen);

        ipAddr.pfxLen = pfxLen;
        ipAddr.ipAddr.family = OPEN_AF_INET;
      }
    }
    else /* (af == OPEN_AF_INET6) */
    {
      if (strcmp(IPStr, "all") != 0)
      {
        if (inet_pton(AF_INET6, IPStr, (void*)&(ipAddr.ipAddr.addr.ipv6)) < 0)
        {
          printf("Bad return code trying to convert IP.\n");
          return;
        }

        ipAddr.pfxLen = pfxLen;
        ipAddr.ipAddr.family = OPEN_AF_INET6;
      }
    }
  }

  if (strcmp(IPStr, "all") == 0)
  {
    result = openapiRtrIntfIpAddrDel(client_handle, af, intf, addrType, 0, extArg);
  }
  else
  {
    result = openapiRtrIntfIpAddrDel(client_handle, af, intf, addrType, &ipAddr, extArg);
  }

  
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete Interface IP. (result = %d)\n", result);
  }
  else
  {
    printf("Interface IP deleted successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Enables/Disables IP Routing Admin mode.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    af              @b{(input)}   Address Family (IPv4 or IPv6)
* @param    routingMode     @b{(input)}   Routing mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void rtrAdminModeSet(openapiClientHandle_t *client_handle, 
                     OPEN_AF_t af, OPEN_CONTROL_t routingMode)
{
  open_error_t result;
  
  if ((result = openapiRtrAdminModeSet(client_handle, af, routingMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure Global Routing Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Global Routing Mode configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Add a test static ARP entry {192.168.1.1 00:11:22:33:44:55}.
*
* @param    client_handle   @b{(input)} client handle from registration API
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void arpEntryAdd(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  char IPStr[] = "192.168.1.1";
  char mac_address[] = {0x00, 0x11, 0x22, 0x33, 0x44, 0x55};
  open_buffdesc mac_addr;

  memset(&ipAddr, 0, sizeof(ipAddr));

  if (inet_pton(AF_INET, IPStr, (void*)&(ipAddr.addr.ipv4)) < 0)
  {
    printf("Bad return code trying to convert IP.\n");
    return;
  }

  ipAddr.addr.ipv4 = ntohl(ipAddr.addr.ipv4);
  ipAddr.family = OPEN_AF_INET;

  mac_addr.pstart = mac_address;
  mac_addr.size = 6;
  
  if ((result = openapiArpEntryAdd(client_handle, ipAddr, &mac_addr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to add ARP entry. (result = %d)\n", result);
  }
  else
  {
    printf("ARP entry added successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Delete a test static ARP entry {192.168.1.1 00:11:22:33:44:55}.
*
* @param    client_handle   @b{(input)} client handle from registration API
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void arpEntryDel(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  char IPStr[] = "192.168.1.1";

  memset(&ipAddr, 0, sizeof(ipAddr));

  if (inet_pton(AF_INET, IPStr, (void*)&(ipAddr.addr.ipv4)) < 0)
  {
    printf("Bad return code trying to convert IP.\n");
    return;
  }

  ipAddr.addr.ipv4 = ntohl(ipAddr.addr.ipv4);
  ipAddr.family = OPEN_AF_INET;

  if ((result = openapiArpEntryDel(client_handle, ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete ARP entry. (result = %d)\n", result);
  }
  else
  {
    printf("ARP entry deleted successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Display all static ARP entries.
*
* @param    client_handle   @b{(input)} client handle from registration API
*
* @returns  none
*
* @notes  
*
* @end
*********************************************************************/
void arpEntryShow(openapiClientHandle_t *client_handle)
{
  uint32_t intf = 0;
  open_inet_addr_t ipAddr;
  char ipAddrStr[24];
  char macStr[18];
  open_buffdesc mac_addr;

  memset(&ipAddr, 0, sizeof(ipAddr));

  memset(macStr, 0, 18);
  mac_addr.pstart = macStr;
  mac_addr.size = 18;

  while(openapiArpEntryNextGet(client_handle, &intf, &ipAddr, &mac_addr) == OPEN_E_NONE)
  {
    /* Note: openapiArpEntryNextGet violates OpEN convention by returning
       the ipv4 address in network byte order. Thus, we call inet_ntop
       without need for conversion from host byte order to network byte
       order using htonl. */

    if (inet_ntop(AF_INET, (void*)&(ipAddr.addr.ipv4), ipAddrStr, sizeof(ipAddrStr)) == 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }
    printf("IP: %s, MAC: %s\n", ipAddrStr, macStr);
  }

  return;
}

/*********************************************************************
* @purpose  Enables/Disables generation of IP Redirects messages.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    mode            @b{(input)}   Redirects mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void ipRedirectsModeSet(openapiClientHandle_t *client_handle, 
                        OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiIpRedirectsModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure IP Redirects Mode. (result = %d)\n", result);
  }
  else
  {
    printf("IP Redirects Mode configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Enables/Disables IP ICMP Echo Reply mode.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    mode            @b{(input)}   Redirects mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void ipICMPEchoReplyModeSet(openapiClientHandle_t *client_handle, 
                            OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiIpICMPEchoReplyModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure IP ICMP Echo Reply Mode. (result = %d)\n", result);
  }
  else
  {
    printf("IP ICMP Echo Reply Mode configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Enables/Disables IP Helper mode.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    mode            @b{(input)}   IP Helper mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void ipHelperModeSet(openapiClientHandle_t *client_handle, 
                     OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiIpHelperModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure IP Helper Mode. (result = %d)\n", result);
  }
  else
  {
    printf("IP Helper Mode configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Adds IP Helper Address and UDP port number.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    IPStr           @b{(input)}   Server IP address.
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void ipHelperAddressAdd(openapiClientHandle_t *client_handle, 
                        char *IPStr, uint32_t udpPort)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, IPStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  /* Validate IP address */
  if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert IP address. (result = %d)\n", result);
    return;
  }
              
  ipAddr.family = OPEN_AF_INET;
 
  if ((result = openapiIpHelperAddressAdd(client_handle, ipAddr, udpPort)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to add IP Helper Address. (result = %d)\n", result);
  }
  else
  {
    printf("IP Helper Address configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Deletes IP Helper Address and UDP port number.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    IPStr           @b{(input)}   Server IP address.
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void ipHelperAddressDel(openapiClientHandle_t *client_handle, 
                        char *IPStr, uint32_t udpPort)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, IPStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(&ipAddr, 0, sizeof(ipAddr));

  if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert IP address. (result = %d)\n", result);
    return;
  }

  ipAddr.family = OPEN_AF_INET;
 
  if ((result = openapiIpHelperAddressDel(client_handle, ipAddr, udpPort)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete IP Helper Address. (result = %d)\n", result);
  }
  else
  {
    printf("IP Helper Address deleted successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Configures Router Interface MTU.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    af              @b{(input)}   Address Family (IPv4 or IPv6)
* @param    intf            @b{(input)}   Router Interface
* @param    mtu             @b{(input)}   MTU
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void rtrIntfMTUSet(openapiClientHandle_t *client_handle, 
                   OPEN_AF_t af,uint32_t intf, uint32_t mtu)
{
  open_error_t result;
  
  if ((result = openapiRtrIntfMTUSet(client_handle, af, intf, mtu)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure IP interface MTU. (result = %d)\n", result);
  }
  else
  {
    printf("IP interface MTU configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Enables/Disables forwarding of Network-directed broadcast on a Router Interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    mode            @b{(input)}   IP Network-directed broadcast mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void rtrIntfIpNetDirBroadcastModeSet(openapiClientHandle_t *client_handle, 
                                     uint32_t intf, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiRtrIntfIpNetDirBroadcastModeSet(client_handle, intf, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure IP interface Net Directed BCast Mode. (result = %d)\n", result);
  }
  else
  {
    printf("IP interface Net Directed BCast Mode configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Enables/Disables IP Proxy Arp mode on a Router Interface.
*
* @param    client_handle         @b{(input)}   client handle from registration API
* @param    intf                  @b{(input)}   Router Interface
* @param    mode                  @b{(input)}   IP Proxy Arp mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void rtrIntfIpProxyArpModeSet(openapiClientHandle_t *client_handle, 
                              uint32_t intf, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiRtrIntfIpProxyArpModeSet(client_handle, intf, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure IP interface Proxy ARP Mode. (result = %d)\n", result);
  }
  else
  {
    printf("IP interface Proxy ARP Mode configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Enables/Disables generation of IP Redirects messages on a Router interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    mode            @b{(input)}   Redirects mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void rtrIntfIpRefirectsModeSet(openapiClientHandle_t *client_handle, 
                               uint32_t intf, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiRtrIntfIpRedirectsModeSet(client_handle, intf, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure IP interface Redirects Mode. (result = %d)\n", result);
  }
  else
  {
    printf("IP interface Redirects Mode configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Enables/Disables generation of IP Destination Unreachable messages
*           on a Router interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    mode            @b{(input)}   Unreachable mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void rtrIntfIpDestUnreachableModeSet(openapiClientHandle_t *client_handle, 
                                     uint32_t intf, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiRtrIntfIpDestUnreachableModeSet(client_handle, intf, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure IP interface Destination Unreachable Mode. (result = %d)\n", result);
  }
  else
  {
    printf("IP interface Unreachable Mode configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Enables/Disables IP Routing mode on a Router interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    mode            @b{(input)}   Routing mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void rtrIntfRtrAdminModeSet(openapiClientHandle_t *client_handle, 
                            uint32_t intf, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiRtrIntfRtrAdminModeSet(client_handle, intf, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure IP interface Routing Mode. (result = %d)\n", result);
  }
  else
  {
    printf("IP interface Routing Mode configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Adds IP Helper Address and UDP port number on an interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    IPStr           @b{(input)}   Server IP address.
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void rtrIntfIpHelperAddressAdd(openapiClientHandle_t *client_handle, 
                               uint32_t intf, char *IPStr, uint32_t udpPort)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, IPStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(&ipAddr, 0, sizeof(ipAddr));

  if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert IP address. (result = %d)\n", result);
    return;
  }

  ipAddr.family = OPEN_AF_INET;
 
  if ((result = openapiRtrIntfIpHelperAddressAdd(client_handle, intf, ipAddr, udpPort)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to add IP Helper Address. (result = %d)\n", result);
  }
  else
  {
    printf("IP Helper Address configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Deletes IP Helper Address and UDP port number on an interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    IPStr           @b{(input)}   Server IP address.
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void rtrIntfIpHelperAddressDel(openapiClientHandle_t *client_handle, 
                               uint32_t intf, char *IPStr, uint32_t udpPort)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, IPStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(&ipAddr, 0, sizeof(ipAddr));

  if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert IP address. (result = %d)\n", result);
    return;
  }

  ipAddr.family = OPEN_AF_INET;
 
 if ((result = openapiRtrIntfIpHelperAddressDel(client_handle, intf, ipAddr, udpPort)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete IP Helper Address. (result = %d)\n", result);
  }
  else
  {
    printf("IP Helper Address deleted successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Adds IP Helper Discard entry in an interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void rtrIntfIpHelperDiscardAdd(openapiClientHandle_t *client_handle, 
                               uint32_t intf, uint32_t udpPort)
{
  open_error_t result;
 
  if ((result = openapiRtrIntfIpHelperDiscardAdd(client_handle, intf, udpPort)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to add IP Helper Discard. (result = %d)\n", result);
  }
  else
  {
    printf("IP Helper Discard configured successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Deletes IP Helper Discard entry in an interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void rtrIntfIpHelperDiscardDel(openapiClientHandle_t *client_handle, 
                               uint32_t intf, uint32_t udpPort)
{
  open_error_t result;

  if ((result = openapiRtrIntfIpHelperDiscardDel(client_handle, intf, udpPort)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete IP Helper Discard. (result = %d)\n", result);
  }
  else
  {
    printf("IP Helper Discard deleted successfully \n");
  }
  return;
}


/*********************************************************************
* @purpose  Add an IP Route for a given address family.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   Address Family (IPv4 or IPv6)
* @param    IPStr            @b{(input)}   IP address and prefix length. 
* @param    pfxLen           @b{(input)}   Prefix Length
* @param    NextHopStr       @b{(input)}   Next Hop
* @param    rtPref           @b{(input)}   Route preference
* @param    intf             @b{(input)}   Router Interface
* @param    mpls             @b{(input)}   List of MPLS labels

*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void rtrIpRouteAdd(openapiClientHandle_t *client_handle,
                   OPEN_AF_t af, char *IPStr, uint32_t pfxLen,
                   char *NextHopStr, uint32_t rtPref,
                   uint32_t intf, OPEN_MPLS_LABELS_t *mpls)
{
  open_error_t result;
  open_inet_pfx_t ipAddr;
  open_inet_addr_t nextHop;
  open_inet_addr_t *nHptr;

  open_buffdesc ipBuffdesc, nextHopBuffdesc;
  char str[40], str1[40];

  memset(str, 0, sizeof(str));
  strncpy(str, IPStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(str1, 0, sizeof(str1));
  strncpy(str1, NextHopStr, sizeof(str1) - 1);
  nextHopBuffdesc.pstart = str1;
  nextHopBuffdesc.size = strlen(str1) + 1;

  memset(&ipAddr, 0, sizeof(ipAddr));

  if (af == OPEN_AF_INET)
  {
    if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &(ipAddr.ipAddr))) != OPEN_E_NONE)
    {
      printf("Bad return code trying to convert IP address. (result = %d)\n", result);
      return;
    }
    printf("IP = %u, PfxLen = %u\n", ipAddr.ipAddr.addr.ipv4, pfxLen);

    ipAddr.pfxLen = pfxLen;
    ipAddr.ipAddr.family = OPEN_AF_INET;

    if (strcmp(NextHopStr, "null0") == 0)
    {
      nHptr = (void*)0;
    }
    else
    {
      if ((result = openapiInetAddrGet(client_handle, &nextHopBuffdesc, &nextHop)) != OPEN_E_NONE)
      {
        printf("Bad return code trying to convert IP address. (result = %d)\n", result);
        return;
      }
      nextHop.family = OPEN_AF_INET;

      nHptr = &nextHop;
    }

    printf("Next Hop = %u\n", nextHop.addr.ipv4);
  }
  else /* (af == OPEN_AF_INET6) */
  {
    if (inet_pton(AF_INET6, IPStr, (void*)&(ipAddr.ipAddr.addr.ipv6)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
    }

    ipAddr.pfxLen = pfxLen;
    ipAddr.ipAddr.family = OPEN_AF_INET6;

    if (strcmp(NextHopStr, "null0") == 0)
    {
      nHptr = (void*)0;
    }
    else
    {
      memset(&nextHop, 0, sizeof(nextHop));
      
      if (inet_pton(AF_INET6, NextHopStr, (void*)&(nextHop.addr.ipv6)) < 0)
      {
        printf("Bad return code trying to convert Next Hop.\n");
      }
      nextHop.family = OPEN_AF_INET6;
      
      nHptr = &nextHop;
    }
  }

  result = openapiIpRouteAdd(client_handle, af, &ipAddr, nHptr, rtPref, intf, mpls);  

  if ((result == OPEN_E_NONE) || (result == OPEN_E_NOT_FOUND))
  {
    printf("IP route added successfully \n");
  }
  else
  {
    printf("Bad return code trying to add IP route. (result = %d)\n", result);
  }
  return;
}

/*********************************************************************
* @purpose  Delete an IP Route for a given address family.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   Address Family (IPv4 or IPv6)
* @param    IPStr            @b{(input)}   IP address and prefix length. 
* @param    pfxLen           @b{(input)}   Prefix Length
* @param    NextHopStr       @b{(input)}   Next Hop
* @param    intf             @b{(input)}   Router Interface

*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void rtrIpRouteDel(openapiClientHandle_t *client_handle,
                   OPEN_AF_t af, char *IPStr, uint32_t pfxLen,
                   char *NextHopStr, uint32_t intf, OPEN_MPLS_LABELS_t *mpls)
{
  open_error_t result;
  open_inet_pfx_t ipAddr;
  open_inet_addr_t nextHop;
  open_inet_addr_t *nHptr;

  open_buffdesc ipBuffdesc, nextHopBuffdesc;
  char str[40], str1[40];

  memset(str, 0, sizeof(str));
  strncpy(str, IPStr, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;

  memset(str1, 0, sizeof(str1));
  strncpy(str1, NextHopStr, sizeof(str1) - 1);
  nextHopBuffdesc.pstart = str1;
  nextHopBuffdesc.size = strlen(str1) + 1;

  memset(&ipAddr, 0, sizeof(ipAddr));

  if (af == OPEN_AF_INET)
  {
    if ((result = openapiInetAddrGet(client_handle, &ipBuffdesc, &(ipAddr.ipAddr))) != OPEN_E_NONE)
    {
      printf("Bad return code trying to convert IP address. (result = %d)\n", result);
      return;
    }
    printf("IP = %u, PfxLen = %u\n", ipAddr.ipAddr.addr.ipv4, pfxLen);

    ipAddr.pfxLen = pfxLen;
    ipAddr.ipAddr.family = OPEN_AF_INET;

    if (strcmp(NextHopStr, "null0") == 0)
    {
      nHptr = (void*)0;
    }
    else
    {
      if ((result = openapiInetAddrGet(client_handle, &nextHopBuffdesc, &nextHop)) != OPEN_E_NONE)
      {
        printf("Bad return code trying to convert IP address. (result = %d)\n", result);
        return;
      }
      nextHop.family = OPEN_AF_INET;

      nHptr = &nextHop;
    }
  
      printf("Next Hop = %u\n", nextHop.addr.ipv4);
  }
  else /* (af == OPEN_AF_INET6) */
  {
    if (inet_pton(AF_INET6, IPStr, (void*)&(ipAddr.ipAddr.addr.ipv6)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
    }

    ipAddr.pfxLen = pfxLen;
    ipAddr.ipAddr.family = OPEN_AF_INET6;

    if (strcmp(NextHopStr, "null0") == 0)
    {
      nHptr = (void*)0;
    }
    else
    {
      memset(&nextHop, 0, sizeof(nextHop));
      
      if (inet_pton(AF_INET6, NextHopStr, (void*)&(nextHop.addr.ipv6)) < 0)
      {
        printf("Bad return code trying to convert Next Hop.\n");
      }
      nextHop.family = OPEN_AF_INET6;
      
      nHptr = &nextHop;
    }
  }
  
  if ((result = openapiIpRouteDel(client_handle, af, &ipAddr, nHptr, intf, mpls)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete IP route. (result = %d)\n", result);
  }
  else
  {
    printf("IP route deleted successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Create a VLAN Routing Interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    vlanId           @b{(input)}   VLAN ID
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void rtrVlanIntfCreate(openapiClientHandle_t *client_handle,
                       uint32_t vlanId)
{
  open_error_t result;
  
  if ((result = openapiRtrVlanIntfCreate(client_handle, vlanId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to create VLAN Routing Interface %u. (result = %d)\n", vlanId, result);
  }
  else
  {
    printf("VLAN Routing Interface created successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Delete a VLAN Routing Interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    vlanId           @b{(input)}   VLAN ID
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void rtrVlanIntfDelete(openapiClientHandle_t *client_handle,
                       uint32_t vlanId)
{
  open_error_t result;
  
  if ((result = openapiRtrVlanIntfDelete(client_handle, vlanId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete VLAN Routing Interface %u. (result = %d)\n",  vlanId, result);
  }
  else
  {
    printf("VLAN Routing Interface deleted successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Display interface number associated with a VLAN Routing interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    vlanId           @b{(input)}   VLAN ID
*
* @returns  none
* 
* @end
*********************************************************************/
void rtrVlanIfNumShow(openapiClientHandle_t *client_handle,
                      uint32_t vlanId)
{
  open_error_t result;
  uint32_t ifNum;
  open_buffdesc ifName;
  char ifNameStr[OPEN_INTF_NAME_MAX_LENGTH];

  ifName.pstart = ifNameStr;
  ifName.size = sizeof(ifNameStr);
  
  if ((result = openapiRtrVlanIntfIfNumGet(client_handle, vlanId, &ifNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Interface Number for VLAN %u. (result = %d)\n",  vlanId, result);
  }
  else
  {
    printf("VLAN Routing Interface Number for VLAN %u is %d\n", vlanId, ifNum);

    result = openapiIntfNameGet(client_handle, ifNum, &ifName);
    if (result == OPEN_E_NONE)
    {
      printf("Logical interface name for interface %u is %s\n", ifNum, ifNameStr);
    }
    else
    {
      printf("Bad return code trying to get interface name for ifNum %u. (result = %d)\n",  ifNum, result);
    }
  }
  return;
}

/*********************************************************************
* @purpose  Create a BFD session
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   Address Family (IPv4=1 or IPv6=2)
* @param    ifNum            @b{(input)}   Router Interface
* @param    dstIpStr         @b{(input)}   Dest IP address.
* @param    srcIpStr         @b{(input)}   Source IP address.
*
* @returns  none
*
* @notes  
*
* @end
*********************************************************************/
void rtrBfdSessionCreate(openapiClientHandle_t *client_handle, OPEN_AF_t af,
                         uint32_t ifNum, char *dstIpStr , char *srcIpStr)
{
  uint32_t sessId;
  open_error_t rc;
  openBfdEndpoint_t ep;

  memset(&ep, 0x0, sizeof(ep));
  /* Setup the BFD session parameters */
  ep.compId = 0;
  ep.vrfId = 0;
  ep.intIfNum = ifNum;
  ep.dstIpAddr.family = af;
  ep.srcIpAddr.family = af;
  ep.type = OPEN_BFD_TUNNEL_TYPE_UDP; /* Need BFD session using IP-UDP encap */

  if (af == OPEN_AF_INET)
  {
    if (inet_pton(AF_INET, dstIpStr, (void*)&(ep.dstIpAddr.addr.ipv4)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }
    if (inet_pton(AF_INET, srcIpStr, (void*)&(ep.srcIpAddr.addr.ipv4)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }
    ep.dstIpAddr.addr.ipv4 = ntohl(ep.dstIpAddr.addr.ipv4);
    ep.srcIpAddr.addr.ipv4 = ntohl(ep.srcIpAddr.addr.ipv4);
  }
  else if (af == OPEN_AF_INET6)
  {
    if (inet_pton(AF_INET6, dstIpStr, (void*)&(ep.dstIpAddr.addr.ipv6)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }

    if (inet_pton(AF_INET6, srcIpStr, (void*)&(ep.srcIpAddr.addr.ipv6)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }
  }

  rc = openapiBfdModeSet(OPEN_BFD_ENABLE);
  if (rc != OPEN_E_NONE)
  {
    printf("Failed to enable BFD %d\n", rc);
  }

  rc = openapiBfdSessionCreate(client_handle, &ep, &sessId);
 
  if (rc == OPEN_E_NONE)
  {
    printf("Successfully created BFD session %d\n", sessId);
  }
  else
  {
    printf("Failed to create BFD session %d \n", rc);
  }
}

/*********************************************************************
* @purpose  Find a BFD session
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   Address Family (IPv4=1 or IPv6=2)
* @param    dstIpStr         @b{(input)}   Dest IP address.
*
* @returns  none
*
* @notes  
*
* @end
*********************************************************************/
void rtrBfdSessionFind(openapiClientHandle_t *client_handle, OPEN_AF_t af, char *dstIpStr)
{
  uint32_t sessIdOut;
  open_error_t rc;
  open_inet_addr_t ipAddr;

  memset(&ipAddr, 0x0, sizeof(ipAddr));
  ipAddr.family = af;

  if (af == OPEN_AF_INET)
  {
    if (inet_pton(AF_INET, dstIpStr, (void*)&(ipAddr.addr.ipv4)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }
    ipAddr.addr.ipv4 = ntohl(ipAddr.addr.ipv4);
  }
  else if (af == OPEN_AF_INET6)
  {
    if (inet_pton(AF_INET6, dstIpStr, (void*)&(ipAddr.addr.ipv6)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }
  }

  /* look up the session based on destination */

  rc = openapiBfdSessionFind(client_handle, ipAddr, &sessIdOut);
  if (rc != OPEN_E_NONE)
  {
    printf("Failed to find BFD session rc=%d\n", rc);
  }
  else
  {
    printf("Successfully found BFD session %d\n", sessIdOut);
  }
}

/*********************************************************************
* @purpose  Iterate the VRF names
*
* @param    client_handle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void iterateVRFs(openapiClientHandle_t *client_handle)
{
  char vrfName[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  char vrfNameNext[OPEN_VRF_MAX_NAME_LEN + 1];
  open_error_t ret;
  int count = 0;

  open_buffdesc vrfNameBuf;
  open_buffdesc vrfNameNextBuf;

  vrfNameBuf.pstart = vrfName;
  vrfNameBuf.size = 1;

  vrfNameNextBuf.pstart = vrfNameNext;
  vrfNameNextBuf.size = OPEN_VRF_MAX_NAME_LEN + 1;

  printf("List of VRFs\n");
  printf("************\n\n");
  while (1) 
  {
    ret = openapiVrfNameNextGet(client_handle, &vrfNameBuf, &vrfNameNextBuf);
    if (ret != OPEN_E_NONE) 
    {
      if (ret == OPEN_E_PARAM) 
      {
        printf("%s: invalid argument passed to openapiVrfNameNextGet\n",
               __FUNCTION__);
      }
      break;
    }
    printf("%d: %s\n", ++count, vrfNameNext);
    strncpy(vrfName, vrfNameNext, sizeof(vrfName));
    vrfNameBuf.size = strlen(vrfName) + 1;
    vrfNameNextBuf.size = OPEN_VRF_MAX_NAME_LEN + 1;
  }
  printf("\n%d VRFs found\n", count);
}

/*********************************************************************
* @purpose  Delete a BFD session
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   Address Family (IPv4 or IPv6)
* @param    ifNum            @b{(input)}   Router Interface
* @param    dstIpStr         @b{(input)}   Dest IP address.
* @param    srcIpStr         @b{(input)}   Source IP address.
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void rtrBfdSessionDelete(openapiClientHandle_t *client_handle, OPEN_AF_t af,
                         uint32_t ifNum, char *dstIpStr , char *srcIpStr)
{
  uint32_t sessId;
  open_error_t rc;
  openBfdEndpoint_t ep;

  memset(&ep, 0x0, sizeof(ep));
  /* Setup the BFD session parameters */
  ep.compId = 0;
  ep.vrfId = 0;
  ep.intIfNum = ifNum;
  ep.dstIpAddr.family = af;
  ep.srcIpAddr.family = af;
  ep.type = OPEN_BFD_TUNNEL_TYPE_UDP; /* Need BFD session using IP-UDP encap */

  sessId = OPEN_BFD_SESSION_ID_INVALID;
  if (af == OPEN_AF_INET)
  {
    if (inet_pton(AF_INET, dstIpStr, (void*)&(ep.dstIpAddr.addr.ipv4)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }
    if (inet_pton(AF_INET, srcIpStr, (void*)&(ep.srcIpAddr.addr.ipv4)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }
    ep.dstIpAddr.addr.ipv4 = ntohl(ep.dstIpAddr.addr.ipv4);
    ep.srcIpAddr.addr.ipv4 = ntohl(ep.srcIpAddr.addr.ipv4);
  }
  else if (af == OPEN_AF_INET6)
  {
    if (inet_pton(AF_INET6, dstIpStr, (void*)&(ep.dstIpAddr.addr.ipv6)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }

    if (inet_pton(AF_INET6, srcIpStr, (void*)&(ep.srcIpAddr.addr.ipv6)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }
  }

  rc = openapiBfdSessionDelete(client_handle, &ep, sessId);

  if (rc == OPEN_E_NONE)
  {
    printf("Successfully deleted BFD session\n");
  }
  else
  {
    printf("Failed to delete BFD session %d\n", rc);
  }
}

/*********************************************************************
* @purpose  Show list of current BFD sessions
*
* @param    client_handle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void rtrBfdSessionShow(openapiClientHandle_t *client_handle)
{
  uint32_t sessId;
  uint32_t count = 0;
  open_error_t err;
  openBfdSessionInfo_t info;
  openBfdSessionStats_t stats;

  char *pStrInfo_bfd_header_1 = "SessionId:%u  State:%s  Local Disc:%u  Remote Disc:%u\r\n";
  char *pStrInfo_bfd_header_2 = "     Destination IP:%s \r\n";
  char *pStrInfo_bfd_header_3 = "     Source IP:%s \r\n";
  char *pStrInfo_bfd_header_4 = "     Interface:%u  VrfId:%u  Uptime:%u secs\r\n";
  char *pStrInfo_bfd_header_5 = "     Min Transmit Interval:%u milliseconds\r\n";
  char *pStrInfo_bfd_header_6 = "     Min Receive Interval:%u milliseconds\r\n";
  char *pStrInfo_bfd_header_7 = "     Detection interval multiplier:%u \r\n";
  char *pStrInfo_bfd_header_8 = "     In Packets:%u  Out Packets:%u  Dropped Packets:%u\r\n";
  char *pStrInfo_bfd_header_9 = "     Echo In Packets:%u  Echo Out Packets:%u \r\n";
  char  strSrcIp[80] = {0};
  char  strDstIp[80] = {0};
  char  strState[80] = {0};

  /* Get the BFD session info */

  for (sessId=1; sessId < 96; sessId++)
  {
    memset(&info, 0x0, sizeof(info));
    memset(&stats, 0x0, sizeof(stats));
    err = openapiBfdSessionInfoGet(client_handle, sessId, &info);
    if (err == OPEN_E_NONE)
    {
      count++;
      openapiBfdSessionStatsGet(client_handle, sessId, &stats);

      ipAddressFormat(&info.key.srcIpAddr, strSrcIp);
      ipAddressFormat(&info.key.dstIpAddr, strDstIp);

      snprintf(strState, sizeof(strState), "%s", 
               (info.state == OPEN_BFD_SESSION_STATE_UP) ? "UP" : "DOWN");

      printf(pStrInfo_bfd_header_1, sessId, strState, info.localDiscr, info.remoteDiscr);
      printf(pStrInfo_bfd_header_2, strDstIp);
      printf(pStrInfo_bfd_header_3, strSrcIp);
      printf(pStrInfo_bfd_header_4, info.key.intIfNum, info.key.vrfId, info.upTime );
      printf(pStrInfo_bfd_header_5, info.localMinTx);
      printf(pStrInfo_bfd_header_6, info.localMinRx);
      printf(pStrInfo_bfd_header_7, info.localDetectMult);
      printf(pStrInfo_bfd_header_8, stats.inPkts, stats.outPkts, stats.dropPkts );
      printf(pStrInfo_bfd_header_9, stats.echoInPkts, stats.echoOutPkts);
    }
  }

  if (count == 0)
  {
    printf("No BFD sessions found.\r\n");
  }
}

/*********************************************************************
* @purpose  This function sanity checks all Routing Config OpEN APIs. All 
*           Routing Config OpEN APIs are called with possible NULL and 
*           invalid parameters to check the API robustness.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void routingOpENAPIsTestSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t invalidControlMode = 10;
  uint32_t invalidAddressFamily = 10;
  uint32_t invalidIntfAddressType = 10;
  uint32_t intf = 1, udpPort = 10, mtu = 1500;
  open_buffdesc buffDesc;
  char str[32];
  open_inet_pfx_t ipAddrPfx;
  open_inet_addr_t ipAddr;

  printf("Testing Routing Configuration OpEN APIs sanity:\n\n");

  /* openapiRtrAdminModeSet() */
  printf("Testing openapiRtrAdminModeSet():\n");

  result = openapiRtrAdminModeSet(NULL, OPEN_AF_INET, OPEN_ENABLE);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiRtrAdminModeSet(clientHandle, OPEN_AF_NONE, OPEN_ENABLE);

  printf("Address family NONE :(result = %d)\n", result);

  result = openapiRtrAdminModeSet(clientHandle, invalidAddressFamily, OPEN_ENABLE);

  printf("Invalid Address family :(result = %d)\n", result);

  result = openapiRtrAdminModeSet(clientHandle, OPEN_AF_NONE, invalidControlMode);

  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiRtrAdminModeSet() sanity successful\n\n");

  /* openapiRtrIntfIpAddrAdd() */
  printf("Testing openapiRtrIntfIpAddrAdd():\n");

  memset(&ipAddr, 0, sizeof(open_inet_pfx_t));

  result = openapiRtrIntfIpAddrAdd(NULL, OPEN_AF_INET, intf,
                                   OPEN_IP_ADDR_TYPE_STATIC_PRIMARY, 
                                   &ipAddrPfx, OPEN_ENABLE);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiRtrIntfIpAddrAdd(clientHandle, OPEN_AF_NONE, intf,
                                   OPEN_IP_ADDR_TYPE_STATIC_PRIMARY, 
                                   &ipAddrPfx, OPEN_ENABLE);

  printf("Address family NONE :(result = %d)\n", result);

  result = openapiRtrIntfIpAddrAdd(clientHandle, invalidAddressFamily, intf,
                                   OPEN_IP_ADDR_TYPE_STATIC_PRIMARY, 
                                   &ipAddrPfx, OPEN_ENABLE);

  printf("Invalid Address family :(result = %d)\n", result);

  result = openapiRtrIntfIpAddrAdd(clientHandle, OPEN_AF_INET, intf,
                                   invalidIntfAddressType, 
                                   &ipAddrPfx, OPEN_ENABLE);

  printf("Invalid Address Type:(result = %d)\n", result);

  result = openapiRtrIntfIpAddrAdd(clientHandle, OPEN_AF_INET, intf,
                                   OPEN_IP_ADDR_TYPE_STATIC_PRIMARY, 
                                   NULL, OPEN_ENABLE);

  printf("NULL IP Address:(result = %d)\n", result);

  result = openapiRtrIntfIpAddrAdd(clientHandle, OPEN_AF_INET, intf,
                                   OPEN_IP_ADDR_TYPE_STATIC_PRIMARY, 
                                   &ipAddrPfx, invalidControlMode);

  printf("NULL Ext Mode:(result = %d)\n", result);

  printf("openapiRtrIntfIpAddrAdd() sanity successful\n\n");


  /* openapiRtrIntfIpAddrDel() */
  printf("Testing openapiRtrIntfIpAddrDel():\n");

  memset(&ipAddr, 0, sizeof(open_inet_pfx_t));

  result = openapiRtrIntfIpAddrDel(NULL, OPEN_AF_INET, intf,
                                   OPEN_IP_ADDR_TYPE_STATIC_PRIMARY, 
                                   &ipAddrPfx, OPEN_ENABLE);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiRtrIntfIpAddrDel(clientHandle, OPEN_AF_NONE, intf,
                                   OPEN_IP_ADDR_TYPE_STATIC_PRIMARY, 
                                   &ipAddrPfx, OPEN_ENABLE);

  printf("Address family NONE :(result = %d)\n", result);

  result = openapiRtrIntfIpAddrDel(clientHandle, invalidAddressFamily, intf,
                                   OPEN_IP_ADDR_TYPE_STATIC_PRIMARY, 
                                   &ipAddrPfx, OPEN_ENABLE);

  printf("Invalid Address family :(result = %d)\n", result);

  result = openapiRtrIntfIpAddrDel(clientHandle, OPEN_AF_INET, intf,
                                   invalidIntfAddressType, 
                                   &ipAddrPfx, OPEN_ENABLE);

  printf("Invalid Address Type:(result = %d)\n", result);

  result = openapiRtrIntfIpAddrDel(clientHandle, OPEN_AF_INET, intf,
                                   OPEN_IP_ADDR_TYPE_STATIC_PRIMARY, 
                                   NULL, OPEN_ENABLE);

  printf("NULL IP Address:(result = %d)\n", result);

  result = openapiRtrIntfIpAddrDel(clientHandle, OPEN_AF_INET, intf,
                                   OPEN_IP_ADDR_TYPE_STATIC_PRIMARY, 
                                   &ipAddrPfx, invalidControlMode);

  printf("NULL Ext Mode:(result = %d)\n", result);

  printf("openapiRtrIntfIpAddrDel() sanity successful\n\n");


  /* openapiArpEntryAdd() */
  printf("Testing openapiArpEntryAdd():\n");

  result = openapiArpEntryAdd(NULL, ipAddr, &buffDesc);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiArpEntryAdd(clientHandle, ipAddr, NULL);
  printf("NULL Mac Address:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  buffDesc.size = 6;

  result = openapiArpEntryAdd(clientHandle, ipAddr, &buffDesc);
  printf("NULL Mac Address buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiArpEntryAdd(clientHandle, ipAddr, &buffDesc);
  printf("NULL Mac Address buffer length:(result = %d)\n", result);

  printf("openapiArpEntryAdd() sanity successful\n\n");

  /* openapiArpEntryDel() */
  printf("Testing openapiArpEntryDel():\n");

  result = openapiArpEntryDel(NULL, ipAddr);

  printf("NULL Client Handle:(result = %d)\n", result);

  printf("openapiArpEntryDel() sanity successful\n\n");


  /* openapiArpEntryNextGet() */
  printf("Testing openapiArpEntryNextGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = 18;

  result = openapiArpEntryNextGet(NULL, &intf, &ipAddr, &buffDesc);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiArpEntryNextGet(clientHandle, NULL, &ipAddr, &buffDesc);

  printf("NULL Intf:(result = %d)\n", result);

  result = openapiArpEntryNextGet(clientHandle, &intf, NULL, &buffDesc);

  printf("NULL ipAddr:(result = %d)\n", result);

  result = openapiArpEntryNextGet(clientHandle, &intf, &ipAddr, NULL);

  printf("NULL Mac Address:(result = %d)\n", result);


  buffDesc.pstart = NULL;
  buffDesc.size = 18;

  result = openapiArpEntryNextGet(clientHandle, &intf, &ipAddr, &buffDesc);
  printf("NULL Mac Address buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiArpEntryNextGet(clientHandle, &intf, &ipAddr, &buffDesc);
  printf("NULL Mac Address buffer length:(result = %d)\n", result);

  printf("openapiArpEntryNextGet() sanity successful\n\n");


  /* openapiIpRedirectsModeSet() */
  printf("Testing openapiIpRedirectsModeSet():\n");

  result = openapiIpRedirectsModeSet(NULL, OPEN_ENABLE);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiIpRedirectsModeSet(clientHandle, invalidControlMode);

  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiIpRedirectsModeSet() sanity successful\n\n");

  /* openapiIpICMPEchoReplyModeSet() */
  printf("Testing openapiIpICMPEchoReplyModeSet():\n");

  result = openapiIpICMPEchoReplyModeSet(NULL, OPEN_ENABLE);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiIpICMPEchoReplyModeSet(clientHandle, invalidControlMode);

  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiIpICMPEchoReplyModeSet() sanity successful\n\n");

  /* openapiIpHelperModeSet() */
  printf("Testing openapiIpHelperModeSet():\n");

  result = openapiIpHelperModeSet(NULL, OPEN_ENABLE);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiIpHelperModeSet(clientHandle, invalidControlMode);

  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiIpHelperModeSet() sanity successful\n\n");

  /* openapiIpHelperAddressAdd() */
  printf("Testing openapiIpHelperAddressAdd():\n");

  result = openapiIpHelperAddressAdd(NULL, ipAddr, udpPort);

  printf("NULL Client Handle:(result = %d)\n", result);

  printf("openapiIpHelperAddressAdd() sanity successful\n\n");

  /* openapiIpHelperAddressDel() */
  printf("Testing openapiIpHelperAddressDel():\n");

  result = openapiIpHelperAddressDel(NULL, ipAddr, udpPort);

  printf("NULL Client Handle:(result = %d)\n", result);

  printf("openapiIpHelperAddressDel() sanity successful\n\n");

  /* openapiRtrIntfIpHelperAddressAdd() */
  printf("Testing openapiRtrIntfIpHelperAddressAdd():\n");

  result = openapiRtrIntfIpHelperAddressAdd(NULL, intf, ipAddr, udpPort);

  printf("NULL Client Handle:(result = %d)\n", result);

  printf("openapiRtrIntfIpHelperAddressAdd() sanity successful\n\n");

  /* openapiRtrIntfIpHelperAddressDel() */
  printf("Testing openapiRtrIntfIpHelperAddressDel():\n");

  result = openapiRtrIntfIpHelperAddressDel(NULL, intf, ipAddr, udpPort);

  printf("NULL Client Handle:(result = %d)\n", result);

  printf("openapiRtrIntfIpHelperAddressDel() sanity successful\n\n");

  /* openapiRtrIntfIpHelperDiscardAdd() */
  printf("Testing openapiRtrIntfIpHelperDiscardAdd():\n");

  result = openapiRtrIntfIpHelperDiscardAdd(NULL, intf, udpPort);

  printf("NULL Client Handle:(result = %d)\n", result);

  printf("openapiRtrIntfIpHelperDiscardAdd() sanity successful\n\n");

  /* openapiRtrIntfIpHelperDiscardDel() */
  printf("Testing openapiRtrIntfIpHelperDiscardDel():\n");

  result = openapiRtrIntfIpHelperDiscardDel(NULL, intf, udpPort);

  printf("NULL Client Handle:(result = %d)\n", result);

  printf("openapiRtrIntfIpHelperDiscardDel() sanity successful\n\n");

  /* openapiRtrIntfMTUSet() */
  printf("Testing openapiRtrIntfMTUSet():\n");

  result = openapiRtrIntfMTUSet(NULL, OPEN_AF_INET, intf, mtu);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiRtrIntfMTUSet(clientHandle, invalidAddressFamily, intf, mtu);

  printf("Invalid Address family :(result = %d)\n", result);

  printf("openapiRtrIntfMTUSet() sanity successful\n\n");

  /* openapiRtrIntfIpNetDirBroadcastModeSet() */
  printf("Testing openapiRtrIntfIpNetDirBroadcastModeSet():\n");

  result = openapiRtrIntfIpNetDirBroadcastModeSet(NULL, intf, OPEN_ENABLE);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiRtrIntfIpNetDirBroadcastModeSet(clientHandle, intf, invalidControlMode);

  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiRtrIntfIpNetDirBroadcastModeSet() sanity successful\n\n");

  /* openapiRtrIntfIpProxyArpModeSet() */
  printf("Testing openapiRtrIntfIpProxyArpModeSet():\n");

  result = openapiRtrIntfIpProxyArpModeSet(NULL, intf, OPEN_ENABLE);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiRtrIntfIpProxyArpModeSet(clientHandle, intf, invalidControlMode);

  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiRtrIntfIpProxyArpModeSet() sanity successful\n\n");

  /* openapiRtrIntfIpRedirectsModeSet() */
  printf("Testing openapiRtrIntfIpRedirectsModeSet():\n");

  result = openapiRtrIntfIpRedirectsModeSet(NULL, intf, OPEN_ENABLE);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiRtrIntfIpRedirectsModeSet(clientHandle, intf, invalidControlMode);

  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiRtrIntfIpRedirectsModeSet() sanity successful\n\n");

  /* openapiRtrIntfIpDestUnreachableModeSet() */
  printf("Testing openapiRtrIntfIpDestUnreachableModeSet():\n");

  result = openapiRtrIntfIpDestUnreachableModeSet(NULL, intf, OPEN_ENABLE);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiRtrIntfIpDestUnreachableModeSet(clientHandle, intf, invalidControlMode);

  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiRtrIntfIpDestUnreachableModeSet() sanity successful\n\n");

  /* openapiRtrIntfRtrAdminModeSet() */
  printf("Testing openapiRtrIntfRtrAdminModeSet():\n");

  result = openapiRtrIntfRtrAdminModeSet(NULL, intf, OPEN_ENABLE);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiRtrIntfRtrAdminModeSet(clientHandle, intf, invalidControlMode);

  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiRtrIntfRtrAdminModeSet() sanity successful\n\n");

  /* openapiIpRouteAdd() */
  printf("Testing openapiIpRouteAdd():\n");

  result = openapiIpRouteAdd(NULL, OPEN_AF_INET, &ipAddrPfx, &ipAddr, 0, intf, NULL);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiIpRouteAdd(clientHandle, invalidAddressFamily, &ipAddrPfx, &ipAddr, 0, intf, NULL);

  printf("Invalid Address Family:(result = %d)\n", result);

  result = openapiIpRouteAdd(clientHandle, OPEN_AF_INET, NULL, &ipAddr, 0, intf, NULL);

  printf("NULL IP Address Pfx:(result = %d)\n", result);

  result = openapiIpRouteAdd(clientHandle, OPEN_AF_INET, &ipAddrPfx, NULL, 0, intf, NULL);

  printf("NULL Next Hop:(result = %d)\n", result);

  printf("openapiIpRouteAdd() sanity successful\n\n");

  /* openapiIpRouteDel() */
  printf("Testing openapiIpRouteDel():\n");

  result = openapiIpRouteDel(NULL, OPEN_AF_INET, &ipAddrPfx, &ipAddr, intf, NULL);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiIpRouteDel(clientHandle, invalidAddressFamily, &ipAddrPfx, &ipAddr, intf, NULL);

  printf("Invalid Address Family:(result = %d)\n", result);

  result = openapiIpRouteDel(clientHandle, OPEN_AF_INET, NULL, &ipAddr, intf, NULL);

  printf("NULL IP Address Pfx:(result = %d)\n", result);

  result = openapiIpRouteDel(clientHandle, OPEN_AF_INET, &ipAddrPfx, NULL, intf, NULL);

  printf("NULL Next Hop:(result = %d)\n", result);

  printf("openapiIpRouteDel() sanity successful\n\n");

  /* openapiRtrVlanIntfCreate */
  result = openapiRtrVlanIntfCreate(NULL, 2);

  printf("NULL Client Handle:(result = %d)\n", result);

  printf("openapiRtrVlanIntfCreate() sanity successful\n\n");

  /* openapiRtrVlanIntfIfNumGet */
  result = openapiRtrVlanIntfIfNumGet(NULL, 2, &intf);

  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiRtrVlanIntfIfNumGet(clientHandle, 2, NULL);

  printf("NULL ifnum: (result = %d)\n", result);

  result = openapiRtrVlanIntfIfNumGet(clientHandle, 3, &intf);

  printf("Invalid VLAN ifnum: (result = %d)\n", result);

  printf("openapiRtrVlanIntfIfNumGet() sanity successful\n\n");

  /* openapiRtrVlanIntfDelete */
  result = openapiRtrVlanIntfDelete(NULL, 2);

  printf("NULL Client Handle:(result = %d)\n", result);

  printf("openapiRtrVlanIntfDelete() sanity successful\n\n");

  return;
}


/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for Routing Configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  int rc;
  uint32_t ifNum, testNum, addrFamily;
  uint32_t addrType, pfxLen, extArg, mode;
  uint32_t udpPort, mtu, pref, vlanId;
  OPEN_MPLS_LABELS_t mpls;

  if (argc < 2 )
  {
    printRoutingAppMenu();
    exit(1);
  }

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((rc = openapiClientRegister("routing_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", rc);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  if (strcmp("API", argv[1]) == 0)
  {
    if (argc < 3 )
    {
      printRoutingAppMenu();
      exit(1);
    }

    testNum = atoi(argv[2]);
   
    switch (testNum)
    {
      case 1:
        if (argc != 9)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        addrFamily = atoi(argv[4]);
        addrType = atoi(argv[5]);
        pfxLen = atoi(argv[7]);
        extArg = atoi(argv[8]);
        
        rtrIntfIpAddrAdd(&clientHandle, addrFamily, ifNum, addrType, argv[6], pfxLen, extArg);
        break;

      case 2:
        if (argc != 9)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        addrFamily = atoi(argv[4]);
        addrType = atoi(argv[5]);
        pfxLen = atoi(argv[7]);
        extArg = atoi(argv[8]);
        
        rtrIntfIpAddrDel(&clientHandle, addrFamily, ifNum, addrType, argv[6], pfxLen, extArg);
        break;
        
      case 3:
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        addrFamily = atoi(argv[3]);
        mode = atoi(argv[4]);
        
        rtrAdminModeSet(&clientHandle, addrFamily, mode);
        break;
          
      case 4:
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }
        
        arpEntryAdd(&clientHandle);
        break;
              
      case 5:
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }
        
        arpEntryDel(&clientHandle);
        break;
      
      case 6:
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }
        
        arpEntryShow(&clientHandle);
        break;
              
      case 7:
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        mode = atoi(argv[3]);
        
        ipRedirectsModeSet(&clientHandle, mode);
        break;
          
      case 8:
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        mode = atoi(argv[3]);
        
        ipICMPEchoReplyModeSet(&clientHandle, mode);
        break;
          
      case 9:
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        mode = atoi(argv[3]);
        
        ipHelperModeSet(&clientHandle, mode);
        break;
          
      case 10:
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        udpPort = atoi(argv[4]);
        
        ipHelperAddressAdd(&clientHandle, argv[3], udpPort);
        break;
          
      case 11:
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        udpPort = atoi(argv[4]);
        
        ipHelperAddressDel(&clientHandle, argv[3], udpPort);
        break;
          
      case 12:
        if (argc != 6)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        addrFamily = atoi(argv[4]);
        mtu = atoi(argv[5]);
        
        rtrIntfMTUSet(&clientHandle, addrFamily, ifNum, mtu);
        break;
          
      case 13:
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        mode = atoi(argv[4]);
        
        rtrIntfIpNetDirBroadcastModeSet(&clientHandle, ifNum, mode);
        break;
          
      case 14:
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        mode = atoi(argv[4]);
        
        rtrIntfIpProxyArpModeSet(&clientHandle, ifNum, mode);
        break;
          
      case 15:
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        mode = atoi(argv[4]);
        
        rtrIntfIpRefirectsModeSet(&clientHandle, ifNum, mode);
        break;
          
      case 16:
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        mode = atoi(argv[4]);
        
        rtrIntfIpDestUnreachableModeSet(&clientHandle, ifNum, mode);
        break;
          
      case 17:
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        mode = atoi(argv[4]);
        
        rtrIntfRtrAdminModeSet(&clientHandle, ifNum, mode);
        break;
        
      case 18:
        if (argc != 6)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        udpPort = atoi(argv[5]);
        
        rtrIntfIpHelperAddressAdd(&clientHandle, ifNum, argv[4], udpPort);
        break;
          
      case 19:
        if (argc != 6)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        udpPort = atoi(argv[5]);
        
        rtrIntfIpHelperAddressDel(&clientHandle, ifNum, argv[4], udpPort);
        break;
          
      case 20:
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        udpPort = atoi(argv[4]);
        
        rtrIntfIpHelperDiscardAdd(&clientHandle, ifNum, udpPort);
        break;
          
      case 21:
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        ifNum = atoi(argv[3]);
        udpPort = atoi(argv[4]);
        
        rtrIntfIpHelperDiscardDel(&clientHandle, ifNum, udpPort);
        break;
          
      case 22:
        if ((argc < 9) || (argc > 12))
        {
          printRoutingAppMenu();
          exit(1);
        }
        addrFamily = atoi(argv[3]);
        pfxLen = atoi(argv[5]);
        pref = atoi(argv[7]);
        ifNum = atoi(argv[8]);

        memset(&mpls, 0, sizeof(mpls));  
        /* Read MPLS labels */
        if (argc >= 10)
        {
          mpls.label[0] = atoi(argv[9]);
        }
        if (argc >= 11)
        {     
          mpls.label[1] = atoi(argv[10]);
        }     
        if (argc == 12)
        {
          mpls.label[2] = atoi(argv[11]);
        }

        rtrIpRouteAdd(&clientHandle, addrFamily, argv[4], 
                      pfxLen, argv[6], pref, ifNum, &mpls);
                               
        break;

      case 23:
        if ((argc < 8) || (argc > 11))
        {
          printRoutingAppMenu();
          exit(1);
        }
        addrFamily = atoi(argv[3]);
        pfxLen = atoi(argv[5]);
        ifNum = atoi(argv[7]);
      
        memset(&mpls, 0, sizeof(mpls));  
        /* Read MPLS labels */
        if (argc >= 9)
        {
          mpls.label[0] = atoi(argv[8]);
        }
        if (argc >= 10)
        {     
          mpls.label[1] = atoi(argv[9]);
        }     
        if (argc == 11)
        {
          mpls.label[2] = atoi(argv[10]);
        }

        rtrIpRouteDel(&clientHandle, addrFamily, argv[4], 
                      pfxLen, argv[6], ifNum, &mpls);
                               
        break;

      case 24:
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        vlanId = atoi(argv[3]);
        
        rtrVlanIntfCreate(&clientHandle, vlanId);
        break;

      case 25:
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        vlanId = atoi(argv[3]);
        
        rtrVlanIntfDelete(&clientHandle, vlanId);
        break;

      case 26:
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }
        routingOpENAPIsTestSanity(&clientHandle);
        break;
      
      case 27:
      case 28:
        if (argc != 7)
        {
          printRoutingAppMenu();
          exit(1);
        }
        addrFamily = atoi(argv[3]);
        ifNum = atoi(argv[4]);
        if (testNum == 27)
          rtrBfdSessionCreate(&clientHandle, addrFamily, ifNum, argv[5], argv[6]);
        else
          rtrBfdSessionDelete(&clientHandle, addrFamily, ifNum, argv[5], argv[6]);

        break;

      case 29:
        if (argc != 3)
        {
          printRoutingAppMenu();
          exit(1);
        }
        rtrBfdSessionShow(&clientHandle);
        break;
 
      case 30:
        if (argc != 5)
        {
          printRoutingAppMenu();
          exit(1);
        }
        addrFamily = atoi(argv[3]);
        rtrBfdSessionFind(&clientHandle, addrFamily, argv[4]);
 
        break;

      case 31:
        iterateVRFs(&clientHandle);
        break;

      case 32:
        if (argc != 4)
        {
          printRoutingAppMenu();
          exit(1);
        }
        vlanId = atoi(argv[3]);
        
        rtrVlanIfNumShow(&clientHandle, vlanId);
        break;

      default:
        printRoutingAppMenu();
        break;
    }
  }
  else
  {
    printRoutingAppMenu();
  }

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}


