#!/mnt/fastpath/usr/bin/ruby

require "OpEN"
require "OpENUtil"

$example_name = "suppfile_example"

#
# Copyright 2016 Broadcom.
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#


#
# Ruby 1.8.7
#

$open = OpENUtil.new()

#
# Define a reasonable number of lines to display by default
# (a typical support file can contain upwards of 20K lines!)
#
$display_default = 100

class SuppFileExample
  #Support File OpEN API Example.

  def initialize(client)
    @client = client
    @support_filename = nil
  end

  def support_filename_get()
    return @support_filename
  end

  def support_file_create()
    #Execute 'show tech-support file' command internally to generate file.

    # Get the maximum length of a file name string
    max_filename_len_p = OpEN.new_uint32_tp()
    OpEN.openapiSystemFileNameMaxLengthGet(@client, max_filename_len_p)
    max_filename_len = OpEN.uint32_tp_value(max_filename_len_p)
    OpEN.delete_uint32_tp(max_filename_len_p)

    # Allocate a buffer and descriptor to receive output filename
    # (add extra byte for end-of-string termination character)
    bufsize = max_filename_len + 1
    outfile_buf = $open.getCharBuffer(bufsize)
    outfile_desc = OpEN::Open_buffdesc.new
    outfile_desc.pstart = outfile_buf
    outfile_desc.size = bufsize

    filename = nil
    ret = OpEN.openapiSupportFileGenerate(@client, outfile_desc)
    if (ret == OpEN::OPEN_E_NONE) and (outfile_buf.cast().length > 0)
      filename = outfile_buf.cast()
      puts "Tech support file created successfully.\n"
    else
      if ret == OpEN::OPEN_E_UNAVAIL
        puts "Feature not supported on this platform.\n"
      elsif ret == OpEN::OPEN_E_PARAM
        puts "Error: Invalid parameter specified.\n"
      else
        puts "Error: Failed to create tech support file.\n"
      end
    end

    @support_filename = filename
    return filename
  end

  def support_file_display(maxout=$display_default)
    #Display file contents line-by-line. Optional maxout arg controls output:
    #      0  : displays entire file
    #      >0 : displays specified number of file lines

    if @support_filename != nil
      filename = @support_filename
      if maxout >= 0
        for line in open(filename)
          # Do not add any extra newlines
          print line
          if maxout > 1
            maxout -= 1
          end
          if maxout == 1
            break
          end
        end
      end
    end
  end
end

def main()
  #Demonstrate OpEN usage for generating Tech Support file.

  # display_max parameter is optional
  numargs = ARGV.length
  if (numargs > 1) or ((numargs == 1) and (ARGV[0] == '?'))
    puts "#{ $example_name }.py [display_max]\n"
    puts "  display_max: 0=all, >0=number of lines (default #{ $display_default })\n"
    exit
  end

  begin
    if ARGV[0] == nil
      raise ArgumentError.new()
    end
    display_max = Integer(ARGV[0]) rescue -1
    if display_max < 0
      raise RangeError.new()
    end
  rescue RangeError
    puts "The specified input parameter value must be 0 or greater.\n"
    exit
  rescue
    # display default number of lines if optional parm not specified
    display_max = $display_default
  end

  ret = $open.connect($example_name)
  if ret == OpEN::OPEN_E_NONE
    puts "\n"
    $open.getNetworkOSVersion()
    puts "\n"
    $open.getAPIVersion()
    client = $open.client()
    xmp = SuppFileExample.new(client)
    puts "\nCreating Tech Support file (this may take a while)...\n"
    xmp.support_file_create()
    if display_max >= 0
      if display_max == 0
        lines_msg = "all lines"
      elsif display_max == 1
        lines_msg = "first line"
      else
        lines_msg = "first #{ display_max } lines"
      end
      puts "\nDisplaying #{ lines_msg } of file #{ xmp.support_filename_get() }:\n\n"
    end
    xmp.support_file_display(display_max)
    $open.terminate()
  else
    puts "Unable to connect.\n"
  end
end

if __FILE__ == $0 then main() end

