/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  sntp_example.c
*
* @purpose   SNTP OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    03/22/2013
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_sntp.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the SNTP Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printSntpAppMenu()
{
  printf("Usage: sntp_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: Add SNTP server: sntp_example 1 <sntp-server> \n");
  printf("Test 2: Set SNTP server priority: sntp_example 2 <sntp-server> <priority> \n");
  printf("Test 3: Set SNTP server port: sntp_example 3 <sntp-server> <port> \n");
  printf("Test 4: Set SNTP server version: sntp_example 4 <sntp-server> <version> \n");
  printf("Test 5: Delete SNTP server: sntp_example 5 <sntp-server> \n");
  printf("Test 6: Get SNTP server info: sntp_example 6 \n");
  printf("Test 7: Set SNTP client mode: sntp_example 7 <sntp-client mode> \n");
  printf("Test 8: Get SNTP client info: sntp_example 8 \n");
  printf("Test 9: sntp_example OpEN APIs sanity: sntp_example 9 \n");

  return;
}

/*********************************************************************
* @purpose  Adds an SNTP server.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    sntpServerAddr   @b{(input)}   SNTP server address
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void sntpServerAdd(openapiClientHandle_t *clientHandle, char *sntpServerAddr)
{
  open_error_t result;
  open_buffdesc sntpServer;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, sntpServerAddr, (sizeof(str) - 1));
  sntpServer.pstart = str;
  sntpServer.size = strlen(str)+1;

  if ((result = openapiSntpServerAdd(clientHandle, &sntpServer)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to add SNTP server. (result = %d)\n", result);
  }
  else
  {
    printf("SNTP server is added successfully. (result = %d)\n", result); 
  }
  
  return; 
}

/*********************************************************************
* @purpose  Set priority to an given SNTP server.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    sntpServerAddr   @b{(input)}   SNTP server address
* @param    priority         @b{(input)}   SNTP server priority
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void sntpServerPrioritySet(openapiClientHandle_t *clientHandle, char *sntpServerAddr, uint16_t priority)
{
  open_error_t result;
  open_buffdesc sntpServer;
  uint32_t maxPriority;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, sntpServerAddr, (sizeof(str) - 1));
  sntpServer.pstart = str;
  sntpServer.size = strlen(str)+1;

  result = openapiSntpServerPriorityMaxGet(clientHandle, &maxPriority);

  if (result == OPEN_E_NONE)
  {
    if (priority > maxPriority)
    {
      printf("Invalid SNTP server priority. Maximum priority is %d.\n", maxPriority);
      result = OPEN_E_PARAM;
    }
  }

  if (result == OPEN_E_NONE)
  {
    if ((result = openapiSntpServerPrioritySet(clientHandle, &sntpServer, priority)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to set SNTP server priority. (result = %d)\n", result);
    }
    else
    {
      printf("SNTP server priority is set successfully. (result = %d)\n", result);
    }
  }

  return;
}
/*********************************************************************
* @purpose  Set port to an given SNTP server.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    sntpServerAddr   @b{(input)}   SNTP server address
* @param    port             @b{(input)}   SNTP server port
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void sntpServerPortSet(openapiClientHandle_t *clientHandle, char *sntpServerAddr, uint16_t port)
{
  open_error_t result;
  open_buffdesc sntpServer;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, sntpServerAddr, (sizeof(str) - 1));
  sntpServer.pstart = str;
  sntpServer.size = strlen(str)+1;

  if ((result = openapiSntpServerPortSet(clientHandle, &sntpServer, port)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set SNTP server port. (result = %d)\n", result);
  }
  else
  {
    printf("SNTP server port is set successfully. (result = %d)\n", result);
  }
  
  return;
}

/*********************************************************************
* @purpose  Set version to an given SNTP server.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    sntpServerAddr   @b{(input)}   SNTP server address
* @param    version          @b{(input)}   SNTP server version
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void sntpServerVersionSet(openapiClientHandle_t *clientHandle, char *sntpServerAddr, uint16_t version)
{
  open_error_t result;
  open_buffdesc sntpServer;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, sntpServerAddr, (sizeof(str) - 1));
  sntpServer.pstart = str;
  sntpServer.size = strlen(str)+1;

  if ((result = openapiSntpServerVersionSet(clientHandle, &sntpServer, version)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set SNTP server version. (result = %d)\n", result);
  }
  else
  {
    printf("SNTP server version is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Delete the SNTP server.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    sntpServerAddr   @b{(input)}   SNTP server address
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void sntpServerDelete(openapiClientHandle_t *clientHandle, char *sntpServerAddr)
{
  open_error_t result;
  open_buffdesc sntpServer;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, sntpServerAddr, (sizeof(str) - 1));
  sntpServer.pstart = str;
  sntpServer.size = strlen(str)+1;

  if ((result = openapiSntpServerDelete(clientHandle, &sntpServer)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete SNTP server. (result = %d)\n", result);
  }
  else
  {
    printf("SNTP server is deleted set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the SNTP servers details.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void sntpServerInfoGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t maxSntpServers = 0;
  uint32_t sntpServerMaxLen = 0;
  uint32_t stratum = 0;
  uint32_t sntpServerRefIdStrLen = 0;
  uint16_t priority, port, version;
  OPEN_SNTP_SERVER_MODE_t sntpServerMode;
  uint32_t currSntpServers = 0;
  open_buffdesc sntpServer, activeSntpServer, referenceId;
  OPEN_SNTP_ADDRESS_TYPE_t addressType;
  char *str, *str1, *str2;

  if ((result = openapiSntpServersMaxSupportedGet(clientHandle, &maxSntpServers)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get maximum number of SNTP servers supported. (result = %d)\n", result);
    return;
  }
  else
  {
    printf("Maximum no. of SNTP servers: %d\n", maxSntpServers);
  }

  if ((result = openapiSntpServerCurrentEntriesGet(clientHandle, &currSntpServers)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get current SNTP servers supported. (result = %d)\n", result);
    return;
  } 
  else
  {
    printf("Current SNTP servers: %d\n", currSntpServers);
  }

  if ((result = openapiSntpServerMaxAddrLenGet(clientHandle, &sntpServerMaxLen)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get maximum length of SNTP server. (result = %d)\n", result);
    return;
  }

  if ((result = openapiSntpServerReferenceStringLengthGet(clientHandle, &sntpServerRefIdStrLen)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get maximum length of SNTP server reference ID. (result = %d)\n", result);
    return;
  }

  if ((str = (char *)malloc(sntpServerMaxLen)) == NULL)
  {
    printf("Could not allocate memory.\n");
    return;
  }

  if ((str1 = (char *)malloc(sntpServerMaxLen)) == NULL)
  {
    printf("Could not allocate memory.\n");
    free(str);
    return;
  }

  if ((str2 = (char *)malloc(sntpServerRefIdStrLen)) == NULL)
  {
    printf("Could not allocate memory.\n");
    free(str);
    return;
  }

  memset(str, 0, sntpServerMaxLen);
  sntpServer.pstart = str;
  sntpServer.size = sntpServerMaxLen;

  memset(str1, 0, sntpServerMaxLen);
  activeSntpServer.pstart = str1;
  activeSntpServer.size = sntpServerMaxLen; 

  memset(str2, 0, sntpServerRefIdStrLen);
  referenceId.pstart = str2;
  referenceId.size = sntpServerRefIdStrLen;

  printf("\n");

  printf("SNTP active server details: \n");
  printf("---------------------------\n");

  printf("SNTP server Address: ");
  if ((result = openapiActiveSntpServerIpAddressGet(clientHandle, &activeSntpServer)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get active SNTP server address. (result = %d)\n", result);
  }
  else
  {
    printf("%s \n", (char *) activeSntpServer.pstart);
  }

  printf("SNTP server Type: ");
  if ((result = openapiActiveSntpServerAddressTypeGet(clientHandle, &addressType)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get active SNTP server type. (result = %d)\n", result);
  }
  else
  {
    switch(addressType)
    {
      case OPEN_SNTP_ADDRESS_UNKNOWN:
        printf("Unknown \n");
        break;
      case OPEN_SNTP_ADDRESS_IPV4:
        printf("IPV4 \n");
        break;
      case OPEN_SNTP_ADDRESS_DNS:
        printf("DNS \n");
        break;
      case OPEN_SNTP_ADDRESS_IPV6:
        printf("IPV6 \n");
        break;
      default:
        break;
    }
  }

  printf("Server stratum: ");
  if ((result = openapiActiveSntpServerStratumGet(clientHandle, &stratum)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get active SNTP server stratum. (result = %d)\n", result);
  }
  else
  {
    printf("%d \n", stratum);
  }

  printf("Server Reference ID: ");
  if ((result = openapiActiveSntpServerReferenceIdGet(clientHandle, &referenceId)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get active SNTP server reference ID. (result = %d)\n", result);
  }
  else
  {
    printf("%s \n", (char *) referenceId.pstart);
  }

  printf("Server Mode: ");
  if ((result = openapiActiveSntpServerModeGet(clientHandle, &sntpServerMode)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get active SNTP server Mode. (result = %d)\n", result);
  }
  else
  {
    switch(sntpServerMode)
    {
      case OPEN_SNTP_SERVER_MODE_RESERVED:
        printf("Reserved \n");
        break;
      case OPEN_SNTP_SERVER_MODE_SYMMETRIC_ACTIVE:
        printf("Symmetric Active \n");
        break;
      case OPEN_SNTP_SERVER_MODE_SYMMETRIC_PASSIVE:
        printf("Symmetric Passive \n");
        break;
      case OPEN_SNTP_SERVER_MODE_CLIENT:
        printf("Client \n");
        break;
      case OPEN_SNTP_SERVER_MODE_SERVER:
        printf("Server \n");
        break;
      case OPEN_SNTP_SERVER_MODE_BROADCAST:
        printf("Broadcast \n");
        break;
      case OPEN_SNTP_SERVER_MODE_RESERVED_CONTROL:
        printf("Reserved Control \n");
        break;
      case OPEN_SNTP_SERVER_MODE_RESERVED_PRIVATE:
        printf("Reserved Private \n");
        break;
      default:
      break;
    }
  }

  printf("\n");
  printf("SNTP servers details: \n");
  printf("---------------------------\n");

  if ((result = openapiSntpServerFirstGet(clientHandle, &sntpServer)) != OPEN_E_NONE)
  {
    printf("  Bad return code while getting first SNTP server. (result = %d)\n", result);
    free(str);
    return;
  }
  do
  {
    printf("SNTP server: %s \n", (char *)sntpServer.pstart);

    printf("Address Type: ");
    if ((result = openapiSntpServerAddressTypeGet(clientHandle, &sntpServer, &addressType)) != OPEN_E_NONE)
    {
      printf("Address type get failed. (result = %d) \n", result);
    }
    else
    {
      switch(addressType)
      {
        case OPEN_SNTP_ADDRESS_UNKNOWN:
          printf("Unknown \n");
          break;
        case OPEN_SNTP_ADDRESS_IPV4:
          printf("IPV4 \n");
          break;
        case OPEN_SNTP_ADDRESS_DNS:
          printf("DNS \n");
          break;
        case OPEN_SNTP_ADDRESS_IPV6:
          printf("IPV6 \n");
          break;
        default:
          break;
      }
    }

    printf("Priority: ");
    if ((result = openapiSntpServerPriorityGet(clientHandle, &sntpServer, &priority)) != OPEN_E_NONE)
    {
      printf("Priority get failed. (result = %d) \n", result);
    }
    else
    {
      printf("%d \n", priority);
    }
  
    printf("Port: ");
    if ((result = openapiSntpServerPortGet(clientHandle, &sntpServer, &port)) != OPEN_E_NONE)
    {
      printf("Port get failed. (result = %d) \n", result);
    }
    else
    {
      printf("%d \n", port);
    }

    printf("Version: ");
    if ((result = openapiSntpServerVersionGet(clientHandle, &sntpServer, &version)) != OPEN_E_NONE)
    {
      printf("Version get failed. (result = %d) \n", result);
    }
    else
    {
      printf("%d \n", version);
    }
    sntpServer.size = sntpServerMaxLen;
    printf("\n");
  }while(openapiSntpServerNextGet(clientHandle, &sntpServer, &sntpServer) == OPEN_E_NONE);

  free(str);
  return; 
}

/*********************************************************************
* @purpose  Set the SNTP client mode.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    mode             @b{(input)}   SNTP client mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void sntpClientModeSet(openapiClientHandle_t *clientHandle, OPEN_SNTP_CLIENT_MODE_t mode)
{
  open_error_t result;

  if ((result = openapiSntpClientModeSet(clientHandle, mode)) != OPEN_E_NONE) 
  {
    printf("Bad return code trying to set SNTP client mode. (result = %d)\n", result);
  }
  else
  {
    printf("SNTP client mode is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the SNTP client details.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void sntpClientInfoGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_SNTP_CLIENT_MODE_t sntpClientMode;
  OPEN_SNTP_SUPPORTED_MODE_t sntpSuppClientMode;

  printf("Getting SNTP client information. \n"); 
  
  if ((result = openapiSntpClientModeGet(clientHandle, &sntpClientMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get SNTP client mode. (result = %d)\n", result);
  }
  else
  {
    printf("SNTP client mode: ");
    switch(sntpClientMode)
    {
      case OPEN_SNTP_CLIENT_DISABLED:
        printf("Disabled \n"); 
        break;
      case OPEN_SNTP_CLIENT_UNICAST:
        printf("Unicast \n");
        break;
      case OPEN_SNTP_CLIENT_BROADCAST:
        printf("Broadcast \n");
        break;
      default:
        break;
    }
  }

  if ((result = openapiSntpClientSupportedModeGet(clientHandle, &sntpSuppClientMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get SNTP supported client mode. (result = %d)\n", result);
  }
  else
  {
    printf("SNTP supported client mode: ");
    switch(sntpSuppClientMode)
    {
      case OPEN_SNTP_SUPPORTED_DISABLED:
        printf("Disabled \n");
        break;
      case OPEN_SNTP_SUPPORTED_UNICAST:
        printf("Unicast \n");
        break;
      case OPEN_SNTP_SUPPORTED_BROADCAST:
        printf("Broadcast \n");
        break;
      case OPEN_SNTP_SUPPORTED_UNICAST_AND_BROADCAST:
        printf("Unicast Broadcast \n");
        break;
      default:
      break;
    }
  }
  return;
}

/*********************************************************************
* @purpose  Perform SNTP OpEN APIs Sanity
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void sntpOpENAPISanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc sntpServer;
  uint32_t sntpServerMaxLen, maxSntpServers;  
  uint16_t priority, port, version;
  uint32_t stratum = 0;
  uint32_t sntpServerRefIdStrLen = 0;
  OPEN_SNTP_SERVER_MODE_t sntpServerMode;
  uint32_t currSntpServers = 0;
  open_buffdesc referenceId;
  OPEN_SNTP_ADDRESS_TYPE_t addressType;
  OPEN_SNTP_SUPPORTED_MODE_t sntpClientSuppMode = OPEN_SNTP_SUPPORTED_DISABLED;
  OPEN_SNTP_CLIENT_MODE_t sntpClientMode = OPEN_SNTP_CLIENT_DISABLED;
  char *str;

  printf("Testing SNTP OpEN APIs sanity:\n"); 

  if ((result = openapiSntpServerMaxAddrLenGet(clientHandle, &sntpServerMaxLen)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get maximum length of SNTP server. (result = %d)\n", result);
    return;
  }

  if ((str = (char *)malloc(sntpServerMaxLen)) == NULL)
  {
    printf("Could not allocate memory.\n");
    return;
  }

  memset(&referenceId,0,sizeof(referenceId));
  memset(str, 0, sntpServerMaxLen);
  sntpServer.pstart = str;
  sntpServer.size = sntpServerMaxLen;

  /* openapiSntpServerMaxAddrLenGet () */
  printf("\nTesting openapiSntpServerMaxAddrLenGet(): \n");
  result = openapiSntpServerMaxAddrLenGet(NULL, &sntpServerMaxLen);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSntpServerMaxAddrLenGet(clientHandle, NULL);
  printf("NULL parameter to SNTP server length. (result = %d)\n", result);

  printf("openapiSntpServerMaxAddrLenGet() sanity successful. \n");

  /* openapiSntpServerAdd() */
  printf("\nTesting openapiSntpServerAdd(): \n");
  result = openapiSntpServerAdd(NULL, &sntpServer);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSntpServerAdd(clientHandle, NULL);
  printf("NULL buff descriptor to SNTP server. (result = %d)\n", result);

  printf("openapiSntpServerAdd() sanity successful. \n");

  /* openapiSntpServerPrioritySet() */
  printf("\nTesting openapiSntpServerPrioritySet(): \n");

  result = openapiSntpServerPrioritySet(NULL, &sntpServer, priority);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSntpServerPrioritySet(clientHandle, NULL, priority);
  printf("NULL buff descriptor to SNTP server. (result = %d)\n", result);

  result = openapiSntpServerPrioritySet(clientHandle, &sntpServer, 0);
  printf("Invalid SNTP server priority. (result = %d)\n", result);

  result = openapiSntpServerPrioritySet(clientHandle, &sntpServer, 5);
  printf("Invalid SNTP server priority. (result = %d)\n", result);

  printf("openapiSntpServerPrioritySet() sanity successful. \n");

  /* openapiSntpServerPortSet() */
  printf("\nTesting openapiSntpServerPrioritySet(): \n");

  result = openapiSntpServerPrioritySet(NULL, &sntpServer, port);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSntpServerPrioritySet(clientHandle, NULL, port);
  printf("NULL buff descriptor to SNTP server. (result = %d)\n", result);

  result = openapiSntpServerPrioritySet(clientHandle, &sntpServer, 0);
  printf("Invalid SNTP server port. (result = %d)\n", result);

  result = openapiSntpServerPrioritySet(clientHandle, &sntpServer, 7000);
  printf("Invalid SNTP server port. (result = %d)\n", result);

  printf("openapiSntpServerPortSet() sanity successful. \n");

  /* openapiSntpServerVersionSet() */
  printf("\nTesting openapiSntpServerVersionSet(): \n");

  result = openapiSntpServerVersionSet(NULL, &sntpServer, port);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSntpServerVersionSet(clientHandle, NULL, port);
  printf("NULL buff descriptor to SNTP server. (result = %d)\n", result);

  result = openapiSntpServerVersionSet(clientHandle, &sntpServer, 0);
  printf("Invalid SNTP server version. (result = %d)\n", result);

  result = openapiSntpServerVersionSet(clientHandle, &sntpServer, 6);
  printf("Invalid SNTP server version. (result = %d)\n", result);

  printf("openapiSntpServerVersionSet() sanity successful. \n");

  /* openapiSntpServersMaxSupportedGet() */
  printf("\nTesting openapiSntpServersMaxSupportedGet(): \n");
  result = openapiSntpServersMaxSupportedGet(NULL, &maxSntpServers);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSntpServersMaxSupportedGet(clientHandle, NULL);
  printf("NULL parameter to maximum no. of SNTP servers. (result = %d)\n", result);

  printf("openapiSntpServersMaxSupportedGet() sanity successful. \n");

  /* openapiSntpServerDelete() */
  printf("\nTesting openapiSntpServerDelete(): \n");

  result = openapiSntpServerDelete(NULL, &sntpServer);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSntpServerDelete(clientHandle, NULL);
  printf("NULL buff descriptor to SNTP server. (result = %d)\n", result);

  printf("openapiSntpServerDelete() sanity successful. \n");
  

  /* openapiSntpServerFirstGet() */
  printf("\nTesting openapiSntpServerFirstGet(): \n");

  result = openapiSntpServerFirstGet(NULL, &sntpServer);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSntpServerFirstGet(clientHandle, NULL);
  printf("NULL buff descriptor to SNTP server. (result = %d)\n", result);

  printf("openapiSntpServerFirstGet() sanity successful. \n");

  /* openapiSntpServerNextGet() */
  printf("\nTesting openapiSntpServerNextGet(): \n");

  result = openapiSntpServerNextGet(NULL, &sntpServer, &sntpServer);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSntpServerNextGet(clientHandle, NULL, &sntpServer);
  printf("NULL buff descriptor to previous SNTP server. (result = %d)\n", result);

  result = openapiSntpServerNextGet(clientHandle, &sntpServer, NULL);
  printf("NULL buff descriptor to next SNTP server. (result = %d)\n", result);

  printf("openapiSntpServerNextGet() sanity successful. \n");

  /* openapiSntpServerAddressTypeGet() */
  printf("\nTesting openapiSntpServerAddressTypeGet(): \n");

  result = openapiSntpServerAddressTypeGet(NULL, &sntpServer, &addressType);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSntpServerAddressTypeGet(clientHandle, NULL, &addressType);
  printf("NULL buff descriptor to SNTP server. (result = %d)\n", result);

  result = openapiSntpServerAddressTypeGet(clientHandle, &sntpServer, NULL);
  printf("NULL address type. (result = %d)\n", result);

  printf("openapiSntpServerAddressTypeGet() sanity successful. \n");

  /* openapiSntpServerPriorityGet() */
  printf("\nTesting openapiSntpServerPriorityGet(): \n");

  result = openapiSntpServerPriorityGet(NULL, &sntpServer, &priority);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSntpServerPriorityGet(clientHandle, NULL, &priority);
  printf("NULL buff descriptor to SNTP server. (result = %d)\n", result);

  result = openapiSntpServerPriorityGet(clientHandle, &sntpServer, NULL);
  printf("NULL parameter to SNTP server priority. (result = %d)\n", result);

  printf("openapiSntpServerPriorityGet() sanity successful. \n");

  /* openapiSntpServerPortGet() */
  printf("\nTesting openapiSntpServerPortGet(): \n");

  result = openapiSntpServerPortGet(NULL, &sntpServer, &port);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSntpServerPortGet(clientHandle, NULL, &port);
  printf("NULL buff descriptor to SNTP server. (result = %d)\n", result);

  result = openapiSntpServerPortGet(clientHandle, &sntpServer, NULL);
  printf("NULL parameter to SNTP server port. (result = %d)\n", result);

  printf("openapiSntpServerPortGet() sanity successful. \n");

  /* openapiSntpServerVersionGet() */
  printf("\nTesting openapiSntpServerVersionGet(): \n");

  result = openapiSntpServerVersionGet(NULL, &sntpServer, &priority);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSntpServerVersionGet(clientHandle, NULL, &version);
  printf("NULL buff descriptor to SNTP server. (result = %d)\n", result);

  result = openapiSntpServerVersionGet(clientHandle, &sntpServer, NULL);
  printf("NULL parameter to SNTP server version. (result = %d)\n", result);

  printf("openapiSntpServerVersionGet() sanity successful. \n");

  /* openapiSntpClientModeSet() */
  printf("\nTesting openapiSntpClientModeSet(): \n");

  result = openapiSntpClientModeSet(NULL, sntpClientMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSntpClientModeSet(clientHandle, 10);
  printf("Invalid SNTP client mode. (result = %d)\n", result);

  printf("openapiSntpClientModeSet() sanity successful. \n");

  /* openapiSntpClientModeGet() */
  printf("\nTesting openapiSntpClientModeGet(): \n");

  result = openapiSntpClientModeGet(NULL, &sntpClientMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSntpClientModeGet(clientHandle, NULL);
  printf("NULL parameter to SNTP client mode. (result = %d)\n", result);

  printf("openapiSntpClientModeGet() sanity successful. \n");

  /* openapiSntpClientSupportedModeGet() */
  printf("\nTesting openapiSntpClientSupportedModeGet(): \n");

  result = openapiSntpClientSupportedModeGet(NULL, &sntpClientSuppMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSntpClientSupportedModeGet(clientHandle, NULL);
  printf("NULL parameter to SNTP client supported mode. (result = %d)\n", result);

  printf("openapiSntpClientSupportedModeGet() sanity successful. \n");

  /* openapiActiveSntpServerAddressTypeGet() */
  printf("\nTesting openapiActiveSntpServerAddressTypeGet(): \n");
  result = openapiActiveSntpServerAddressTypeGet(NULL, &addressType);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiActiveSntpServerAddressTypeGet(clientHandle, NULL);
  printf("NULL address type. (result = %d)\n", result);

  printf("openapiActiveSntpServerAddressTypeGet() sanity successful. \n");


  /* openapiActiveSntpServerIpAddressGet() */
  printf("\nTesting openapiActiveSntpServerIpAddressGet(): \n");

  result = openapiActiveSntpServerIpAddressGet(NULL, &sntpServer);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiActiveSntpServerIpAddressGet(clientHandle, NULL);
  printf("NULL buff descriptor to SNTP server. (result = %d)\n", result); 

  printf("openapiActiveSntpServerIpAddressGet() sanity successful. \n");

  /* openapiActiveSntpServerStratumGet() */
  printf("\nTesting openapiActiveSntpServerStratumGet(): \n");

  result = openapiActiveSntpServerStratumGet(NULL, &stratum);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiActiveSntpServerStratumGet(clientHandle, NULL);
  printf("NULL SNTP server stratum. (result = %d)\n", result);

  printf("openapiActiveSntpServerStratumGet() sanity successful. \n");

  /* openapiActiveSntpServerReferenceIdGet() */
  printf("\nTesting openapiActiveSntpServerReferenceIdGet(): \n");

  result = openapiActiveSntpServerReferenceIdGet(NULL, &referenceId);
  printf("NULL Client Handle. (result = %d)\n", result);
 
  result = openapiActiveSntpServerReferenceIdGet(clientHandle, NULL);
  printf("NULL SNTP reference ID. (result = %d)\n", result);

  printf("openapiActiveSntpServerReferenceIdGet() sanity successful. \n");

  /* openapiSntpServerReferenceStringLengthGet() */
  printf("\nTesting openapiSntpServerReferenceStringLengthGet(): \n");

  result = openapiSntpServerReferenceStringLengthGet(NULL, &sntpServerRefIdStrLen);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSntpServerReferenceStringLengthGet(clientHandle, NULL);
  printf("NULL parameter to SNTP server reference ID. (result = %d)\n", result);

  printf("openapiSntpServerReferenceStringLengthGet() sanity successful. \n");

  /* openapiActiveSntpServerModeGet() */
  printf("\nTesting openapiActiveSntpServerModeGet(): \n");

  result = openapiActiveSntpServerModeGet(NULL, &sntpServerMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiActiveSntpServerModeGet(clientHandle, NULL);
  printf("NULL SNTP server mode. (result = %d)\n", result);

  printf("openapiActiveSntpServerModeGet() sanity successful. \n");

  /* openapiSntpServerCurrentEntriesGet() */
  printf("\nTesting openapiSntpServerCurrentEntriesGet(): \n");

  result = openapiSntpServerCurrentEntriesGet(NULL, &currSntpServers);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSntpServerCurrentEntriesGet(clientHandle, NULL);
  printf("NULL parameter to current SNTP servers. (result = %d)\n", result);
  
  printf("openapiSntpServerCurrentEntriesGet() sanity successful. \n");
   
  return;
}
/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum, arg1;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  
  if (argc < 2)
  {
    printSntpAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("sntp_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting SNTP API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 3)
      {
        printSntpAppMenu();
        exit(1);
      }
      sntpServerAdd(&clientHandle, argv[2]);
      break;
    case 2:
      if (argc != 4)
      {
        printSntpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      sntpServerPrioritySet(&clientHandle, argv[2], arg1);
      break;
    case 3:
      if (argc != 4)
      {
        printSntpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      sntpServerPortSet(&clientHandle, argv[2], arg1);
      break;
    case 4:
      if (argc != 4)
      {
        printSntpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      sntpServerVersionSet(&clientHandle, argv[2], arg1);
      break;
    case 5:
      if (argc != 3)
      {
        printSntpAppMenu();
        exit(1);
      }
      sntpServerDelete(&clientHandle, argv[2]);
      break;
    case 6:
      if (argc != 2)
      {
        printSntpAppMenu();
        exit(1);
      }
      sntpServerInfoGet(&clientHandle);
      break;
    case 7:
      if (argc != 3)
      {
        printSntpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      sntpClientModeSet(&clientHandle, arg1);
      break;
    case 8:
      if (argc != 2)
      {
        printSntpAppMenu();
        exit(1);
      }
      sntpClientInfoGet(&clientHandle);
      break;
    case 9:
      if (argc != 2)
      {
        printSntpAppMenu();
        exit(1);
      }
      sntpOpENAPISanity(&clientHandle);
      break;
    default:
      printSntpAppMenu();
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping SNTP API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
