/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  dot1s_example.c
*
* @purpose   Spanning Tree APIs Example.
*
* @component OPEN
*
* @comments
*
* @create    19/10/2012
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_dot1s.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the Spanning Tree Configuration Example 
*         Application Menu.
*
* @param    none
*
* @returns  none
*
*********************************************************************/
void printDot1sAppMenu()
{
  printf("Usage: dot1s_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: Set Spanning Tree Operational Mode: dot1s_example 1 <mode> \n");
  printf("Test 2: Get Spanning Tree Operational Mode: dot1s_example 2 \n");
  printf("Test 3: Set Spanning Tree Interface BPDU Filter Mode: dot1s_example 3 <interface> <mode> \n");
  printf("Test 4: Get Spanning Tree Interface BPDU Filter Mode: dot1s_example 4 <interface> \n");
  printf("Test 5: Set Spanning Tree Interface BPDU Flood Mode: dot1s_example 5 <interface> <mode> \n");
  printf("Test 6: Get Spanning Tree Interface BPDU Flood Mode: dot1s_example 6 <interface> \n");
  printf("Test 7: Set Spanning Tree BPDU Guard Mode: dot1s_example 7 <mode> \n");
  printf("Test 8: Get Spanning Tree BPDU Guard Mode: dot1s_example 8 \n");
  printf("Test 9: Set Spanning Tree Configuration Name: dot1s_example 9 <name> \n");
  printf("Test 10: Get Spanning Tree Configuration Name: dot1s_example 10 \n");
  printf("Test 11: Set Spanning Tree Configuration Revision: dot1s_example 11 <revision> \n");
  printf("Test 12: Get Spanning Tree Configuration Revision: dot1s_example 12 \n");
  printf("Test 13: Set Spanning Tree Configuration Force Version: dot1s_example 13 <version> \n");
  printf("Test 14: Get Spanning Tree Configuration Force Version: dot1s_example 14 \n");
  printf("Test 15: Set Spanning Tree Interface Guard Mode: dot1s_example 15 <interface> <mode> \n");
  printf("Test 16: Get Spanning Tree Interface Guard Mode: dot1s_example 16 <interface> \n");
  printf("Test 17: Create Spanning Tree Instance: dot1s_example 17 <instance id> \n");
  printf("Test 18: Delete Spanning Tree Instance: dot1s_example 18 <instance id> \n");
  printf("Test 19: Get All Spanning Tree Instances: dot1s_example 19 \n");
  printf("Test 20: Set Spanning Tree MST Instance Priority: dot1s_example 20 <instance id> <priority> \n");
  printf("Test 21: Get Spanning Tree MST Instance Priority: dot1s_example 21 <instance id> \n");
  printf("Test 22: Add Spanning Tree MST Instance - VLAN Association: dot1s_example 22 <instance id> <vlanid> \n");
  printf("Test 23: Delete Spanning Tree MST Instance - VLAN Association: dot1s_example 23 <instance id> <vlanid> \n");
  printf("Test 24: Get All VLANs associated with the Spanning Tree Instance: dot1s_example 24 <instance id> \n");
  printf("Test 25: Get All Forwarding Database Ids associated with the Spanning Tree Instance: dot1s_example 25 <instance id> \n");
  printf("Test 26: Set Spanning Tree Interface Mode: dot1s_example 26 <interface> <mode> \n");
  printf("Test 27: Get Spanning Tree Interface Mode: dot1s_example 27 <interface> \n");
  printf("Test 28: Get Spanning Tree Interface Forwarding State: dot1s_example 28 <interface> <instance id> \n");
  printf("Test 29: Get Spanning Tree Interface Mode: dot1s_example 29 <interface> <instance id> \n");
  printf("Test 30: Set Spanning Tree Interface Edge Port Mode: dot1s_example 30 <interface> <mode> \n");
  printf("Test 31: Get Spanning Tree Interface Edge Port Mode: dot1s_example 31 <interface> \n");
  printf("Test 32: Get Spanning Tree Interface Port Loop Inconsistent State: dot1s_example 32 <interface> <instance id> \n");
  printf("Test 33: Get the Spanning Tree MST Bridge Identifier for a specific instance: dot1s_example 33 <instance id>\n");
  printf("Test 34: Get the Spanning Tree MST Time since the last Topology Change for a specific instance: dot1s_example 34 <instance id>\n");
  printf("Test 35: Get the Spanning Tree MST count of Topology Changes in a specific instance: dot1s_example 35 <instance id>\n");
  printf("Test 36: Get if a topology change is in progress for any port assigned to a specific Spanning Tree MST instance: dot1s_example 36 <instance id>\n");
  printf("Test 37: Get the Spanning Tree MST Designated Root Identifier in a specific instance: dot1s_example 37 <instance id>\n");
  printf("Test 38: Get the Spanning Tree MST Designated Root Path Cost parameter in a specific instance: dot1s_example 38 <instance id>\n");
  printf("Test 39: Get the Spanning Tree Port Root Port ID in a specific instance: dot1s_example 39 <instance id>\n");
  printf("Test 40: Get the max age for the CIST: dot1s_example 40\n");
  printf("Test 41: Get Max Hop count for the device: dot1s_example 41\n");
  printf("Test 42: Get the Tx Hold count value for the device: dot1s_example 42\n");
  printf("Test 43: Get the CIST Bridge Forward Delay: dot1s_example 43\n");
  printf("Test 44: Get the CIST Hello Time for a specific port: dot1s_example 44 <interface>\n");
  printf("Test 45: Get the CIST Bridge Hold Time: dot1s_example 45\n");
  printf("Test 46: Get the CIST Regional Root Identifier: dot1s_example 46\n");
  printf("Test 47: Get the CIST Regional Root Path Cost: dot1s_example 47\n");
  printf("Test 48: Test Spanning Tree OpEN APIs sanity: dot1s_example 48 <interface> \n");

  return;
}

/*********************************************************************
* @purpose  Enables/Disables Spanning Tree Operational Mode
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    mode            @b{(input)}   spanning tree operational mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiDot1SModeSet(clientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree Operational Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Operational Mode set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree Operational Mode.
*
* @param    clientHandle   @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t mode;
  
  if ((result = openapiDot1SModeGet(clientHandle, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Operational Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Operational Mode: %u \n", mode);
  }
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree BPDU Filter mode for an interface
*
* @param    clientHandle           @b{(input)}   client handle from registration API
* @param    ifNum                   @b{(input)}   ID of the interface
* @param    mode                    @b{(input)}   spanning tree BPDU filter mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SIntfBPDUFilterModeSet(openapiClientHandle_t *clientHandle, uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiDot1SIntfBPDUFilterModeSet(clientHandle, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree Interface BPDU Filter Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface BPDU Filter Mode set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree BPDU Filter mode for an interface
*
* @param    clientHandle           @b{(input)}   client handle from registration API
* @param    ifNum                   @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SIntfBPDUFilterModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t mode;
  
  if ((result = openapiDot1SIntfBPDUFilterModeGet(clientHandle, ifNum, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Interface BPDU Filter Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface BPDU Filter Mode for interface %u: %u \n", ifNum, mode);
  }
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree BPDU Flood mode for an interface
*
* @param    clientHandle         @b{(input)}   client handle from registration API
* @param    ifNum                 @b{(input)}   ID of the interface
* @param    mode                  @b{(input)}   spanning tree BPDU Flood mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SIntfBPDUFloodModeSet(openapiClientHandle_t *clientHandle, uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiDot1SIntfBPDUFloodModeSet(clientHandle, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree Interface BPDU Flood Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface BPDU Flood Mode set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree BPDU Flood mode for an interface
*
* @param    clientHandle          @b{(input)}   client handle from registration API
* @param    ifNum                  @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SIntfBPDUFloodModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t mode;
  
  if ((result = openapiDot1SIntfBPDUFloodModeGet(clientHandle, ifNum, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Interface BPDU Flood Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface BPDU Flood Mode for interface %u: %u \n", ifNum, mode);
  }
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree BPDU Guard Mode
*
* @param    clientHandle             @b{(input)}   client handle from registration API
* @param    mode                      @b{(input)}   spanning tree BPDU guard
*
* @returns  mode 
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SBPDUGuardModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiDot1SBPDUGuardModeSet(clientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree BPDU Guard Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree BPDU Guard Mode set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the Spanning Tree BPDU Guard Mode
*
* @param    clientHandle             @b{(input)}   client handle from registration API
*
* @returns  mode 
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SBPDUGuardModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t mode;
  
  if ((result = openapiDot1SBPDUGuardModeGet(clientHandle, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree BPDU Guard Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree BPDU Guard Mode: %u \n", mode);
  }
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree Configuration Name
*
* @param    clientHandle         @b{(input)}   client handle from registration API
* @param    name                  @b{(input)}   spanning tree configuration name
*                                               Populate dot1sConfigName:pstart and size with a char buffer
*                                               containing the config name
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SConfigNameSet(openapiClientHandle_t *clientHandle, char *name)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, name, (sizeof(str) - 1));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  if ((result = openapiDot1SConfigNameSet(clientHandle, &buffDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree Configuration Name. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Configuration Name set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the Spanning Tree Configuration Name
*
* @param    clientHandle         @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SConfigNameGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char *str;
  uint32_t dot1SMgmtStrLen;

  if ((result = openapiDot1sMgmtStrLenGet(clientHandle, OPEN_DOT1S_MGMT_STR_CONFIG_NAME, &dot1SMgmtStrLen)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n", result);
    return;
  }

  if ((str = (char*)malloc(dot1SMgmtStrLen)) == NULL) 
  {
    printf("Could not allocate memory.\n");
    return;
  }

  memset(str, 0, dot1SMgmtStrLen);
  buffDesc.pstart = str;
  buffDesc.size = dot1SMgmtStrLen;

  if ((result = openapiDot1SConfigNameGet(clientHandle, &buffDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Configuration Name. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Configuration Name: %s \n", str);
  }

  free(str);
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree Configuration Revision
*
* @param    clientHandle         @b{(input)}   client handle from registration API
* @param    revision              @b{(input)}   spanning tree configuration revision
*
* @returns  none 
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SConfigRevisionSet(openapiClientHandle_t *clientHandle, uint32_t revision)
{
  open_error_t result;
  
  if ((result = openapiDot1SConfigRevisionSet(clientHandle, revision)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree Configuration Revision. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Configuration Revision set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the Spanning Tree Configuration Revision
*
* @param    clientHandle         @b{(input)}   client handle from registration API
*
* @returns  none 
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SConfigRevisionGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t revision;
  
  if ((result = openapiDot1SConfigRevisionGet(clientHandle, &revision)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Configuration Revision. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Configuration Revision: %u \n", revision);
  }
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree Force Revision
*
* @param    clientHandle             @b{(input)}   client handle from registration API
* @param    forceVersion              @b{(input)}   spanning tree force version
*
* @returns  none 
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SForceVersionSet(openapiClientHandle_t *clientHandle, uint32_t forceVersion)
{
  open_error_t result;
  
  if ((result = openapiDot1SForceVersionSet(clientHandle, forceVersion)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree Force Version. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Force Version set successfully \n");
  }

  return;
}

/*********************************************************************
* @purpose  Gets the Spanning Tree Force Revision
*
* @param    clientHandle             @b{(input)}   client handle from registration API
*
* @returns  none 
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SForceVersionGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t forceVersion;
  
  if ((result = openapiDot1SForceVersionGet(clientHandle, &forceVersion)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Force Version. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Force Version: %u \n", forceVersion);
  }
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree Guard mode
*
* @param    clientHandle         @b{(input)}   client handle from registration API
* @param    ifNum                 @b{(input)}   ID of the interface
* @param    mode                  @b{(input)}   spanning tree Guard mode
*
* @returns  none 
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SIntfGuardModeSet(openapiClientHandle_t *clientHandle, uint32_t ifNum, uint32_t mode)
{
  open_error_t result;
  
  if ((result = openapiDot1SIntfGuardModeSet(clientHandle, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree Interface Guard Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface Guard Mode set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree Guard mode
*
* @param    clientHandle          @b{(input)}   client handle from registration API
* @param    ifNum                  @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SIntfGuardModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_DOT1S_PORT_GUARD_t mode;
  
  if ((result = openapiDot1SIntfGuardModeGet(clientHandle, ifNum, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Interface Guard Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface Guard Mode for interface %u: %u \n", ifNum, mode);
  }
  return;
}

/*********************************************************************
* @purpose  Creates/Deletes Spanning Tree MST Instance
*
* @param    clientHandle             @b{(input)}   client handle from registration API
* @param    dot1sMSTInstanceId        @b{(input)}   spanning tree MST instance Id
* @param    action                    @b{(input)}   create/delete instance
*
* @returns  none 
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SMSTInstanceSet(openapiClientHandle_t *clientHandle, uint32_t dot1sMSTInstanceId, OPEN_ACTION_t action)
{
  open_error_t result;
  
  if ((result = openapiDot1SMSTInstanceSet(clientHandle, dot1sMSTInstanceId, action)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to configure Spanning Tree MST Instance. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Instance set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets all Spanning Tree MST Instances in the switch
*
* @param    clientHandle             @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SMSTInstancesGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t currInstanceId, nextInstanceId;
  
  if ((result = openapiDot1SMSTInstanceFirstGet(clientHandle, &currInstanceId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the first Spanning Tree MST Instance. (result = %d)\n", result);
    return;
  }
  else
  {
    printf("Spanning Tree MST Instances:\n");
    printf("%u", currInstanceId);
  }

  while(openapiDot1SMSTInstanceNextGet(clientHandle, currInstanceId, &nextInstanceId) == OPEN_E_NONE) 
  {
    printf(", %u", nextInstanceId);
    currInstanceId = nextInstanceId;
  }
  printf("\n");
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree MST Instance priority
*
* @param    clientHandle              @b{(input)}   client handle from registration API
* @param    dot1sMSTInstance          @b{(input)}   spanning tree MST instance
* @param    priority                  @b{(input)}   instance priority
*
* @returns  none  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SMSTInstancePrioritySet(openapiClientHandle_t *clientHandle, uint32_t dot1sMSTInstance, uint32_t priority)
{
  open_error_t result;
  
  if ((result = openapiDot1SMSTInstancePrioritySet(clientHandle, dot1sMSTInstance, priority)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree Instance Priority. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Instance Priority set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree MST Instance priority
*
* @param    clientHandle              @b{(input)}   client handle from registration API
* @param    dot1sMSTInstance          @b{(input)}   spanning tree MST instance
*
* @returns  none  
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SMSTInstancePriorityGet(openapiClientHandle_t *clientHandle, uint32_t dot1sMSTInstance)
{
  open_error_t result;
  uint16_t priority;
  
  if ((result = openapiDot1SMSTInstancePriorityGet(clientHandle, dot1sMSTInstance, &priority)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Instance Priority. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Instance Priority for instance %u: %u \n", dot1sMSTInstance, priority);
  }
  return;
}

/*********************************************************************
* @purpose  Adds/Deletes Spanning Tree MST Instance-VLAN association
*
* @param    clientHandle              @b{(input)}   client handle from registration API
* @param    dot1sMSTInstanceId        @b{(input)}   spanning tree MST instance
* @param    vlanId                    @b{(input)}   VLAN
* @param    action                    @b{(input)}   create/delete association
*
* @returns  none  
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SMSTInstanceVLANSet(openapiClientHandle_t *clientHandle, uint32_t dot1sMSTInstanceId, 
                             uint32_t vlanId, OPEN_ACTION_t action)
{
  open_error_t result;
  uint32_t vlanMax;

  result = openapiDot1VlanMaxGet(clientHandle, &vlanMax);

  if (result == OPEN_E_NONE)
  {
    if ((vlanId > vlanMax))
    {
      printf("Invalid VLAN id. The maximum id is %d.\n", vlanMax);
      result = OPEN_E_PARAM;
    }
  }

  if (result == OPEN_E_NONE)
  {
    if ((result = openapiDot1SMSTInstanceVLANSet(clientHandle, dot1sMSTInstanceId, vlanId, action)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to configure Spanning Tree MST Instance - VLAN association. (result = %d)\n", result);
    }
    else
    {
      printf("Spanning Tree MST Instance - VLAN association set successfully \n");
    }
  }

  return;
}

/*********************************************************************
* @purpose  Gets all VLAN Ids for the Spanning Tree MST Instance
*
* @param    clientHandle              @b{(input)}   client handle from registration API
* @param    dot1sMSTInstanceId        @b{(input)}   spanning tree MST instance
*
* @returns  none
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SMSTInstanceVLANsGet(openapiClientHandle_t *clientHandle, uint32_t dot1sMSTInstanceId)
{
  open_error_t result;
  uint32_t currVLANId, nextVLANId;
  
  if ((result = openapiDot1SMSTInstanceFirstVLANGet(clientHandle, dot1sMSTInstanceId, &currVLANId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the first Spanning Tree MST VLAN. (result = %d)\n", result);
    return;
  }
  else
  {
    printf("Spanning Tree MST VLANs:\n");
    printf("%u", currVLANId);
  }

  while(openapiDot1SMSTInstanceNextVLANGet(clientHandle, dot1sMSTInstanceId, currVLANId, &nextVLANId) == OPEN_E_NONE) 
  {
    printf(", %u", nextVLANId);
    currVLANId = nextVLANId;
  }
  printf("\n");
  return;
}

/*********************************************************************
* @purpose  Gets all Forwarding Database Ids for the Spanning Tree MST Instance
*
* @param    clientHandle              @b{(input)}   client handle from registration API
* @param    dot1sMSTInstanceId        @b{(input)}   spanning tree MST instance
*
* @returns  none  
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SMSTInstanceFIdsGet(openapiClientHandle_t *clientHandle, uint32_t dot1sMSTInstanceId)
{
  open_error_t result;
  uint32_t currFId, nextFId;
  
  if ((result = openapiDot1SMSTInstanceFirstFIdGet(clientHandle, dot1sMSTInstanceId, &currFId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the first Spanning Tree MST Forwarding Database Id. (result = %d)\n", result);
    return;
  }
  else
  {
    printf("Spanning Tree MST FIds:\n");
    printf("%u", currFId);
  }

  while(openapiDot1SMSTInstanceNextFIdGet(clientHandle, dot1sMSTInstanceId, currFId, &nextFId) == OPEN_E_NONE) 
  {
    printf(", %u", nextFId);
    currFId = nextFId;
  }

  printf("\n");
  return;
}

/*********************************************************************
* @purpose  Enables/Disables Spanning Tree Admin Mode on an interface
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    ifNum           @b{(input)}   ID of the interface
* @param    mode            @b{(input)}   spanning tree mode
*
* @returns  none 
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SIntfModeSet(openapiClientHandle_t *clientHandle, uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiDot1SIntfModeSet(clientHandle, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree Interface Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface Mode set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree Admin Mode on an interface
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    ifNum           @b{(input)}   ID of the interface
*
* @returns  none  
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SIntfModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t mode;
  
  if ((result = openapiDot1SIntfModeGet(clientHandle, ifNum, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface Mode for interface %u: %u \n", ifNum, mode);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree Port Forwarding State
*
* @param    clientHandle          @b{(input)}   client handle from registration API
* @param    ifNum                  @b{(input)}   ID of the interface
* @param    dot1sInstance          @b{(input)}   spanning tree MST instance
*
* @returns  none 
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SIntfPortFwdStateGet(openapiClientHandle_t *clientHandle, uint32_t ifNum, uint32_t dot1sInstance)
{
  open_error_t result;
  OPEN_DOT1S_PORT_FWD_STATE_t dot1sPortFwdState;
  
  if ((result = openapiDot1SIntfPortFwdStateGet(clientHandle, ifNum, dot1sInstance, &dot1sPortFwdState)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Port Forwarding State. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface Port Forwarding State for interface %u: %u \n", ifNum, dot1sPortFwdState);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree Port Role
*
* @param    clientHandle          @b{(input)}   client handle from registration API
* @param    ifNum                  @b{(input)}   ID of the interface
* @param    dot1sInstance          @b{(input)}   spanning tree MST instance
*
* @returns  none  
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SIntfPortRoleGet(openapiClientHandle_t *clientHandle, uint32_t ifNum, uint32_t dot1sInstance)
{
  open_error_t result;
  OPEN_DOT1S_PORT_ROLE_t dot1sPortRole;
  
  if ((result = openapiDot1SIntfPortRoleGet(clientHandle, ifNum, dot1sInstance, &dot1sPortRole)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Port Role. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Port Role for interface %u: %u \n", ifNum, dot1sPortRole);
  }
  return;
}

/*********************************************************************
* @purpose  Configures Spanning Tree Edge Port mode for an interface
*
* @param    clientHandle           @b{(input)}   client handle from registration API
* @param    ifNum                  @b{(input)}   ID of the interface
* @param    mode                   @b{(input)}   spanning tree edge port mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void dot1SIntfEdgePortModeSet(openapiClientHandle_t *clientHandle, uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiDot1SIntfEdgePortModeSet(clientHandle, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set Spanning Tree Interface Edge Port Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface Edge Port Mode set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree Edge Port mode for an interface
*
* @param    clientHandle           @b{(input)}   client handle from registration API
* @param    ifNum                  @b{(input)}   ID of the interface
*
* @returns  none
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SIntfEdgePortModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t mode;
  
  if ((result = openapiDot1SIntfEdgePortModeGet(clientHandle, ifNum, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Interface Edge Port Mode. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface Edge Port Mode for interface %u: %u \n", ifNum, mode);
  }
  return;
}

/*********************************************************************
* @purpose  Gets Spanning Tree Port Loop Inconsistent State
*
* @param    clientHandle           @b{(input)}   client handle from registration API
* @param    ifNum                  @b{(input)}   ID of the interface
* @param    dot1sInstance          @b{(input)}   spanning tree MST instance
*
* @returns  none 
* 
* @notes  
* 
* @end
*********************************************************************/
void dot1SIntfPortLoopInconsistentStateGet(openapiClientHandle_t *clientHandle, uint32_t ifNum, uint32_t dot1sInstance)
{
  open_error_t result;
  OPEN_BOOL_t loopInconsistentState;
  
  if ((result = openapiDot1SIntfPortLoopInconsistentStateGet(clientHandle, ifNum, dot1sInstance, &loopInconsistentState)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Spanning Tree Port Loop Inconsistent State. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree Interface Port Loop Inconsistent State for interface %u: %u \n", ifNum, loopInconsistentState);
  }
  return;
}

/*********************************************************************
* @purpose  Convert a bridge ID to a string of hex digits.
*
* @param    id               @b{(input)}  instance of OPEN_DOT1S_BRIDGEID_t
* @param    idStr            @b{(output)} resulting string
* @param    size             @b{(input)}  size of idStr
*
* @returns  pointer to idStr
*
* @end
*********************************************************************/
static char * idToStr(OPEN_DOT1S_BRIDGEID_t *id, char *idStr, int size)
{
  unsigned char *p = (unsigned char *) id;
  snprintf(idStr, size, "%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X",
           p[0], p[1], p[2], p[3], p[4], p[5], p[6], p[7]);
  return idStr;
}

/*********************************************************************
* @purpose  Gets the Spanning Tree MST Bridge Identifier for a specific instance.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mstId               @b{(input)}  the MSTID for the desired MSTI
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sMSTInstanceBridgeIdentifierGet(openapiClientHandle_t *client_handle, uint32_t mstId)
{
  open_error_t result;
  OPEN_DOT1S_BRIDGEID_t id;
  char idStr[25];

  if ((result = openapiDot1sMSTInstanceBridgeIdentifierGet(client_handle, mstId, &id)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST Bridge Identifier for a specific instance. (result = %d)\n", result);
  }
  else
  {
    printf("MST Instance Bridge Identifier for mstId %d: %s\n", mstId, idToStr(&id, idStr, sizeof(idStr)));
  }
  return;
}

/*********************************************************************
* @purpose  Gets the Spanning Tree MST Time since the last Topology Change for a specific instance.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mstId               @b{(input)}  the MSTID for the desired MSTI
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sMSTInstanceTimeSinceTopologyChangeGet(openapiClientHandle_t *client_handle, uint32_t mstId)
{
  open_error_t result;
  uint32_t tVal;

  if ((result = openapiDot1sMSTInstanceTimeSinceTopologyChangeGet(client_handle, mstId, &tVal)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST Time since the last Topology Change for a specific instance. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Time since the last Topology Change for mstId %d: %u seconds\n", mstId, tVal);
  }
  return;
}

/*********************************************************************
* @purpose  Get the Spanning Tree MST count of Topology Changes in a specific instance.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mstId               @b{(input)}  the MSTID for the desired MSTI
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sMSTInstanceTopologyChangeCountGet(openapiClientHandle_t *client_handle, uint32_t mstId)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiDot1sMSTInstanceTopologyChangeCountGet(client_handle, mstId, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST count of Topology Changes in a specific instance. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST count of Topology Changes for mstId %d: %u\n", mstId, count);
  }
  return;
}

/*********************************************************************
* @purpose  Get the Spanning Tree MST Topology Change in progress state.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mstId               @b{(input)}  the MSTID for the desired MSTI
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sMSTInstanceTopologyChangeStateGet(openapiClientHandle_t *client_handle, uint32_t mstId)
{
  open_error_t result;
  OPEN_BOOL_t state;

  if ((result = openapiDot1sMSTInstanceTopologyChangeStateGet(client_handle, mstId, &state)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST Topology Change state in a specific instance. (result = %d)\n", result);
  }
  else
  {
    printf("A topology change is %s for some port in the MST instance %d.\n", (state == OPEN_TRUE ? "in progress":"not in progress"), mstId);
  }
  return;
}

/*********************************************************************
* @purpose  Get the Spanning Tree MST Designated Root Identifier in a specific instance.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mstId               @b{(input)}  the MSTID for the desired MSTI
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sMSTInstanceDesignatedRootIDGet(openapiClientHandle_t *client_handle,uint32_t mstId)
{
  open_error_t result;
  OPEN_DOT1S_BRIDGEID_t id;
  char idStr[25];

  if ((result = openapiDot1sMSTInstanceDesignatedRootIDGet(client_handle, mstId, &id)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST Designated Root Identifier in a specific instance. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Designated Root Identifier for mstId %d: %s\n", mstId, idToStr(&id, idStr, sizeof(idStr)));
  }
  return;
}

/*********************************************************************
* @purpose  Get the Spanning Tree MST Designated Root Path Cost parameter in a specific instance
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mstId               @b{(input)}  the MSTID for the desired MSTI
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sMSTInstanceRootPathCostGet(openapiClientHandle_t *client_handle, uint32_t mstId)
{
  open_error_t result;
  uint32_t cost;

  if ((result = openapiDot1sMSTInstanceRootPathCostGet(client_handle, mstId, &cost)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST Designated Root Path Cost parameter in a specific instance. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Designated Root path cost for mstId %d: %u\n", mstId, cost);
  }
  return;
}

/*********************************************************************
* @purpose  Get the Spanning Tree MST Root Port ID in a specific instance.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mstId               @b{(input)}  the MSTID for the desired MSTI
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sMSTInstanceRootPortIDGet(openapiClientHandle_t *client_handle, uint32_t mstId)
{
  open_error_t result;
  uint32_t port;

  if ((result = openapiDot1sMSTInstanceRootPortIDGet(client_handle, mstId, &port)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST Root Port ID in a specific instance. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST root port ID for mstId %d: %u\n", mstId, port);
  }
  return;
}

/*********************************************************************
* @purpose  Get the max age for the CIST.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sCISTMaxAgeGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  uint32_t age;

  if ((result = openapiDot1sCISTMaxAgeGet(client_handle, &age)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the max age for the CIST. (result = %d)\n", result);
  }
  else
  {
    printf("Max age for the CIST: %u\n", age);
  }
  return;
}

/*********************************************************************
* @purpose  Get the Max Hop count for the bridge.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sBridgeMaxHopGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiDot1sBridgeMaxHopGet(client_handle, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Max Hop count for the bridge. (result = %d)\n", result);
  }
  else
  {
    printf("Max hop count for the bridge: %u\n", count);
  }
  return;
}

/*********************************************************************
* @purpose  Get the Spanning Tree MST Tx Hold count parameter for the bridge.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sBridgeTxHoldCountGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  uint32_t count;

  if ((result = openapiDot1sBridgeTxHoldCountGet(client_handle, &count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST Tx Hold count for the bridge. (result = %d)\n", result);
  }
  else
  {
    printf("Tx hold count for the bridge: %u\n", count);
  }
  return;
}

/*********************************************************************
* @purpose  Get the Spanning Tree MST Bridge Forward Delay for the CIST.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sCISTRootFwdDelayGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  uint32_t delay;

  if ((result = openapiDot1sCISTRootFwdDelayGet(client_handle, &delay)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST Bridge Forward Delay for the CIST. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Bridge Forward Delay for the CIST: %u\n", delay);
  }
  return;
}


/*********************************************************************
* @purpose  Get the Spanning Tree MST Hello Time of a specific port for the CIST.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ifNum               @b{(input)}  interface number
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sCISTHelloTimeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result;
  uint32_t helloTime;

  if ((result = openapiDot1sCISTHelloTimeGet(client_handle, ifNum, &helloTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST Hello Time of a specific port for the CIST. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Hello Time of port %u for the CIST: %u\n", ifNum, helloTime);
  }
  return;
}

/*********************************************************************
* @purpose  Get the Spanning Tree MST Bridge Hold Time for the CIST.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sCISTBridgeHoldTimeGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  uint32_t holdTime;

  if ((result = openapiDot1sCISTBridgeHoldTimeGet(client_handle, &holdTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST Bridge Hold Time for the CIST. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST Bridge Hold Time for the CIST: %u\n", holdTime);
  }
  return;
}

/*********************************************************************
* @purpose  Get the Spanning Tree MST CIST Regional Root Identifier.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sCISTRegionalRootIDGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  OPEN_DOT1S_BRIDGEID_t rootId;
  char idStr[25];

  if ((result = openapiDot1sCISTRegionalRootIDGet(client_handle, &rootId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST CIST Regional Root Identifier. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST CIST Regional Root Identifier: %s\n", idToStr(&rootId, idStr, sizeof(idStr)));
  }
  return;
}

/*********************************************************************
* @purpose  Get the Spanning Tree MST CIST Regional Root Path Cost.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void dot1sCISTRegionalRootPathCostGet(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  uint32_t cost;

  if ((result = openapiDot1sCISTRegionalRootPathCostGet(client_handle, &cost)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Spanning Tree MST CIST Regional Root Path Cost. (result = %d)\n", result);
  }
  else
  {
    printf("Spanning Tree MST CIST Regional Root Path Cost: %u\n", cost);
  }
  return;
}

/*********************************************************************
* @purpose  This function sanity checks all DOT1S OpEN APIs. All 
*           DOT1S OpEN APIs are called with possible NULL and 
*           invalid parameters to check the API robustness.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}   ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void dot1SOpENAPIsTestSanity(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t controlMode;
  uint32_t invalidControlMode = 10;
  uint32_t dot1SMgmtStrLen, strLen, dot1sConfigRev, instanceId, nextInstanceId;
  uint32_t vlan, nextVlan, FId, nextFId;
  uint16_t priority;
  OPEN_DOT1S_FORCE_VERSION_t dot1sForceVersion;
  OPEN_DOT1S_PORT_GUARD_t dot1sGuardMode;
  open_buffdesc buffDesc;
  char *str;
  OPEN_DOT1S_PORT_FWD_STATE_t dot1sPortFwdState;
  OPEN_DOT1S_PORT_ROLE_t dot1sPortRole;
  OPEN_BOOL_t loopInconsistentState;
  uint32_t var1;
  uint32_t mstid = 0;
  uint32_t badMstid = 9999;
  OPEN_BOOL_t var2;
  OPEN_DOT1S_BRIDGEID_t id;

  printf("Testing Spanning Tree OpEN APIs sanity:\n\n");

  if ((result = openapiDot1sMgmtStrLenGet(clientHandle, OPEN_DOT1S_MGMT_STR_CONFIG_NAME, &dot1SMgmtStrLen)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n", result);
    return;
  }

  if ((str = (char*)malloc(dot1SMgmtStrLen)) == NULL) 
  {
    printf("Could not allocate memory.\n");
    return;
  }

  memset(str, 0, dot1SMgmtStrLen);

  /* openapiDot1sMgmtStrLenGet() */
  printf("Testing openapiDot1sMgmtStrLenGet():\n");

  result = openapiDot1sMgmtStrLenGet(NULL, OPEN_DOT1S_MGMT_STR_CONFIG_NAME, &strLen);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1sMgmtStrLenGet(clientHandle, 1, &strLen);
  printf("Invalid string type:(result = %d)\n", result);

  result = openapiDot1sMgmtStrLenGet(clientHandle, OPEN_DOT1S_MGMT_STR_CONFIG_NAME, NULL);
  printf("NULL String Length:(result = %d)\n", result);

  printf("openapiDot1sMgmtStrLenGet() sanity successful\n\n");

  /* openapiDot1SModeSet() */
  printf("Testing openapiDot1SModeSet():\n");

  result = openapiDot1SModeSet(NULL, OPEN_ENABLE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SModeSet(clientHandle, invalidControlMode);
  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiDot1SModeSet() sanity successful\n\n");

  /* openapiDot1SModeGet() */
  printf("Testing openapiDot1SModeGet():\n");

  result = openapiDot1SModeGet(NULL, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SModeGet(clientHandle, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiDot1SModeGet() sanity successful\n\n");

  /* openapiDot1SIntfBPDUFilterModeSet() */
  printf("Testing openapiDot1SIntfBPDUFilterModeSet():\n");

  result = openapiDot1SIntfBPDUFilterModeSet(NULL, ifNum, OPEN_ENABLE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfBPDUFilterModeSet(clientHandle, ifNum, invalidControlMode);
  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiDot1SIntfBPDUFilterModeSet() sanity successful\n\n");

  /* openapiDot1SIntfBPDUFilterModeGet() */
  printf("Testing openapiDot1SIntfBPDUFilterModeGet():\n");

  result = openapiDot1SIntfBPDUFilterModeGet(NULL, ifNum, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfBPDUFilterModeGet(clientHandle, ifNum, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiDot1SIntfBPDUFloodModeSet() sanity successful\n\n");

  /* openapiDot1SIntfBPDUFloodModeSet() */
  printf("Testing openapiDot1SIntfBPDUFloodModeSet():\n");

  result = openapiDot1SIntfBPDUFloodModeSet(NULL, ifNum, OPEN_ENABLE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfBPDUFloodModeSet(clientHandle, ifNum, invalidControlMode);
  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiDot1SIntfBPDUFloodModeSet() sanity successful\n\n");

  /* openapiDot1SIntfBPDUFloodModeGet() */
  printf("Testing openapiDot1SIntfBPDUFloodModeGet():\n");

  result = openapiDot1SIntfBPDUFloodModeGet(NULL, ifNum, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfBPDUFloodModeGet(clientHandle, ifNum, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiDot1SIntfBPDUFloodModeGet() sanity successful\n\n");

  /* openapiDot1SBPDUGuardModeSet() */
  printf("Testing openapiDot1SBPDUGuardModeSet():\n");

  result = openapiDot1SBPDUGuardModeSet(NULL, OPEN_ENABLE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SBPDUGuardModeSet(clientHandle, invalidControlMode);
  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiDot1SBPDUGuardModeSet() sanity successful\n\n");

  /* openapiDot1SBPDUGuardModeGet() */
  printf("Testing openapiDot1SBPDUGuardModeGet():\n");

  result = openapiDot1SBPDUGuardModeGet(NULL, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SBPDUGuardModeGet(clientHandle, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiDot1SBPDUGuardModeGet() sanity successful\n\n");

  /* openapiDot1SConfigNameSet() */
  printf("Testing openapiDot1SConfigNameSet():\n");

  buffDesc.pstart = str;
  buffDesc.size = dot1SMgmtStrLen;

  result = openapiDot1SConfigNameSet(NULL, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SConfigNameSet(clientHandle, NULL);
  printf("NULL Config Name:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiDot1SConfigNameSet(clientHandle, &buffDesc);
  printf("NULL Config Name buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = dot1SMgmtStrLen + 10;

  result = openapiDot1SConfigNameSet(clientHandle, &buffDesc);
  printf("NULL Config Name buffer length:(result = %d)\n", result);

  printf("openapiDot1SConfigNameSet() sanity successful\n\n");

  /* openapiDot1SConfigNameGet() */
  printf("Testing openapiDot1SConfigNameGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = dot1SMgmtStrLen;

  result = openapiDot1SConfigNameGet(NULL, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SConfigNameGet(clientHandle, NULL);
  printf("NULL Config Name:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiDot1SConfigNameGet(clientHandle, &buffDesc);
  printf("NULL Config Name buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiDot1SConfigNameGet(clientHandle, &buffDesc);
  printf("NULL Config Name buffer length:(result = %d)\n", result);

  printf("openapiDot1SConfigNameGet() sanity successful\n\n");

  /* openapiDot1SConfigRevisionSet() */
  printf("Testing openapiDot1SConfigRevisionSet():\n");

  result = openapiDot1SConfigRevisionSet(NULL, dot1sConfigRev);
  printf("NULL Client Handle:(result = %d)\n", result);

  printf("openapiDot1SConfigRevisionSet() sanity successful\n\n");

  /* openapiDot1SConfigRevisionGet() */
  printf("Testing openapiDot1SConfigRevisionGet():\n");

  result = openapiDot1SConfigRevisionGet(NULL, &dot1sConfigRev);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SConfigRevisionGet(clientHandle, NULL);
  printf("NULL Config Revision:(result = %d)\n", result);

  printf("openapiDot1SConfigRevisionGet() sanity successful\n\n");

  /* openapiDot1SForceVersionSet() */
  printf("Testing openapiDot1SForceVersionSet():\n");

  result = openapiDot1SForceVersionSet(NULL, OPEN_DOT1S_FORCE_VERSION_DOT1D);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SForceVersionSet(clientHandle, invalidControlMode);
  printf("Invalid Force Version:(result = %d)\n", result);

  printf("openapiDot1SForceVersionSet() sanity successful\n\n");

  /* openapiDot1SForceVersionGet() */
  printf("Testing openapiDot1SForceVersionGet():\n");

  result = openapiDot1SForceVersionGet(NULL, &dot1sForceVersion);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SForceVersionGet(clientHandle, NULL);
  printf("NULL Force Version:(result = %d)\n", result);

  printf("openapiDot1SForceVersionGet() sanity successful\n\n");

  /* openapiDot1SIntfGuardModeSet() */
  printf("Testing openapiDot1SIntfGuardModeSet():\n");

  result = openapiDot1SIntfGuardModeSet(NULL, ifNum, OPEN_DOT1S_PORT_GUARD_LOOP);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfGuardModeSet(clientHandle, ifNum, invalidControlMode);
  printf("Invalid Guard Mode:(result = %d)\n", result);

  printf("openapiDot1SIntfGuardModeSet() sanity successful\n\n");

  /* openapiDot1SIntfGuardModeGet() */
  printf("Testing openapiDot1SIntfGuardModeGet():\n");

  result = openapiDot1SIntfGuardModeGet(NULL, ifNum, &dot1sGuardMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfGuardModeGet(clientHandle, ifNum, NULL);
  printf("NULL Guard Mode:(result = %d)\n", result);

  printf("openapiDot1SIntfGuardModeGet() sanity successful\n\n");

  /* openapiDot1SMSTInstanceSet() */
  printf("Testing openapiDot1SMSTInstanceSet():\n");

  result = openapiDot1SMSTInstanceSet(NULL, 10, OPEN_ACTION_ADD);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMSTInstanceSet(clientHandle, 10, invalidControlMode);
  printf("Invalid Action:(result = %d)\n", result);

  printf("openapiDot1SMSTInstanceSet() sanity successful\n\n");

  /* openapiDot1SMSTInstanceFirstGet() */
  printf("Testing openapiDot1SMSTInstanceFirstGet():\n");

  result = openapiDot1SMSTInstanceFirstGet(NULL, &instanceId);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMSTInstanceFirstGet(clientHandle, NULL);
  printf("NULL Instance:(result = %d)\n", result);

  printf("openapiDot1SMSTInstanceFirstGet() sanity successful\n\n");

  /* openapiDot1SMSTInstanceNextGet() */
  printf("Testing openapiDot1SMSTInstanceNextGet():\n");

  result = openapiDot1SMSTInstanceNextGet(NULL, instanceId, &nextInstanceId);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMSTInstanceNextGet(clientHandle, instanceId, NULL);
  printf("NULL Next Instance:(result = %d)\n", result);

  printf("openapiDot1SMSTInstanceNextGet() sanity successful\n\n");

  /* openapiDot1SMSTInstancePrioritySet() */
  printf("Testing openapiDot1SMSTInstancePrioritySet():\n");

  result = openapiDot1SMSTInstancePrioritySet(NULL, 10, 10);
  printf("NULL Client Handle:(result = %d)\n", result);

  printf("openapiDot1SMSTInstancePrioritySet() sanity successful\n\n");

  /* openapiDot1SMSTInstancePriorityGet() */
  printf("Testing openapiDot1SMSTInstancePriorityGet():\n");

  result = openapiDot1SMSTInstancePriorityGet(NULL, 10, &priority);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMSTInstancePriorityGet(clientHandle, 10, NULL);
  printf("NULL Priority:(result = %d)\n", result);

  printf("openapiDot1SMSTInstancePriorityGet() sanity successful\n\n");

  /* openapiDot1SMSTInstanceVLANSet() */
  printf("Testing openapiDot1SMSTInstanceVLANSet():\n");

  result = openapiDot1SMSTInstanceVLANSet(NULL, 10, 10, OPEN_ACTION_ADD);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMSTInstanceVLANSet(clientHandle, 10, 10, invalidControlMode);
  printf("NULL Action:(result = %d)\n", result);

  printf("openapiDot1SMSTInstanceVLANSet() sanity successful\n\n");

  /* openapiDot1SMSTInstanceFirstVLANGet() */
  printf("Testing openapiDot1SMSTInstanceFirstVLANGet():\n");

  result = openapiDot1SMSTInstanceFirstVLANGet(NULL, 10, &vlan);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMSTInstanceFirstVLANGet(clientHandle, 10, NULL);
  printf("NULL VLAN:(result = %d)\n", result);

  printf("openapiDot1SMSTInstanceFirstVLANGet() sanity successful\n\n");

  /* openapiDot1SMSTInstanceNextVLANGet() */
  printf("Testing openapiDot1SMSTInstanceNextVLANGet():\n");

  result = openapiDot1SMSTInstanceNextVLANGet(NULL, 10, 10, &nextVlan);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMSTInstanceNextVLANGet(clientHandle, 10, 10, NULL);
  printf("NULL Next Vlan:(result = %d)\n", result);

  printf("openapiDot1SMSTInstanceNextVLANGet() sanity successful\n\n");

  /* openapiDot1SMSTInstanceFirstFIdGet() */
  printf("Testing openapiDot1SMSTInstanceFirstFIdGet():\n");

  result = openapiDot1SMSTInstanceFirstFIdGet(NULL, 10, &FId);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMSTInstanceFirstFIdGet(clientHandle, 10, NULL);
  printf("NULL FId:(result = %d)\n", result);

  printf("openapiDot1SMSTInstanceFirstFIdGet() sanity successful\n\n");

  /* openapiDot1SMSTInstanceNextFIdGet() */
  printf("Testing openapiDot1SMSTInstanceNextFIdGet():\n");

  result = openapiDot1SMSTInstanceNextFIdGet(NULL, 10, 10, &nextFId);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SMSTInstanceNextFIdGet(clientHandle, 10, 10, NULL);
  printf("NULL Next FId:(result = %d)\n", result);

  printf("openapiDot1SMSTInstanceNextFIdGet() sanity successful\n\n");

  /* openapiDot1SIntfModeSet() */
  printf("Testing openapiDot1SIntfModeSet():\n");

  result = openapiDot1SIntfModeSet(NULL, ifNum, OPEN_ENABLE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfModeSet(clientHandle, ifNum, invalidControlMode);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiDot1SIntfModeSet() sanity successful\n\n");

  /* openapiDot1SIntfModeGet() */
  printf("Testing openapiDot1SIntfModeGet():\n");

  result = openapiDot1SIntfModeGet(NULL, ifNum, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfModeGet(clientHandle, ifNum, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiDot1SIntfModeGet() sanity successful\n\n");

  /* openapiDot1SIntfModeGet() */
  printf("Testing openapiDot1SIntfModeGet():\n");

  result = openapiDot1SIntfModeGet(NULL, ifNum, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfModeGet(clientHandle, ifNum, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiDot1SIntfModeGet() sanity successful\n\n");

  /* openapiDot1SIntfPortFwdStateGet() */
  printf("Testing openapiDot1SIntfPortFwdStateGet():\n");

  result = openapiDot1SIntfPortFwdStateGet(NULL, ifNum, 10, &dot1sPortFwdState);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfPortFwdStateGet(clientHandle, ifNum, 10, NULL);
  printf("NULL FwdState:(result = %d)\n", result);

  printf("openapiDot1SIntfPortFwdStateGet() sanity successful\n\n");

  /* openapiDot1SIntfPortRoleGet() */
  printf("Testing openapiDot1SIntfPortRoleGet():\n");

  result = openapiDot1SIntfPortRoleGet(NULL, ifNum, 10, &dot1sPortRole);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfPortRoleGet(clientHandle, ifNum, 10, NULL);
  printf("NULL PortRole:(result = %d)\n", result);

  printf("openapiDot1SIntfPortRoleGet() sanity successful\n\n");

  /* openapiDot1SIntfEdgePortModeSet() */
  printf("Testing openapiDot1SIntfEdgePortModeSet():\n");

  result = openapiDot1SIntfEdgePortModeSet(NULL, ifNum, OPEN_ENABLE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfEdgePortModeSet(clientHandle, ifNum, invalidControlMode);
  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiDot1SIntfEdgePortModeSet() sanity successful\n\n");

  /* openapiDot1SIntfEdgePortModeGet() */
  printf("Testing openapiDot1SIntfEdgePortModeGet():\n");

  result = openapiDot1SIntfEdgePortModeGet(NULL, ifNum, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfEdgePortModeGet(clientHandle, ifNum, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiDot1SIntfEdgePortModeGet() sanity successful\n\n");

  /* openapiDot1SIntfPortLoopInconsistentStateGet() */
  printf("Testing openapiDot1SIntfPortLoopInconsistentStateGet():\n");

  result = openapiDot1SIntfPortLoopInconsistentStateGet(NULL, ifNum, 10, &loopInconsistentState);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiDot1SIntfPortLoopInconsistentStateGet(clientHandle, ifNum, 10, NULL);
  printf("NULL Loop Inconsistent State:(result = %d)\n", result);

  printf("openapiDot1SIntfPortLoopInconsistentStateGet() sanity successful\n\n");

  free(str);

  mstid = 0;
  printf("Testing openapiDot1sMSTInstanceBridgeIdentifierGet():\n");
  result = openapiDot1sMSTInstanceBridgeIdentifierGet(NULL, mstid, &id);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceBridgeIdentifierGet():\n");
  result = openapiDot1sMSTInstanceBridgeIdentifierGet(clientHandle, badMstid, &id);
  printf("Invalid MST ID:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceBridgeIdentifierGet():\n");
  result = openapiDot1sMSTInstanceBridgeIdentifierGet(clientHandle, mstid, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("openapiDot1sMSTInstanceBridgeIdentifierGet() sanity successful\n\n");

  printf("Testing openapiDot1sMSTInstanceTimeSinceTopologyChangeGet():\n");
  result = openapiDot1sMSTInstanceTimeSinceTopologyChangeGet(NULL, mstid, &var1);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceTimeSinceTopologyChangeGet():\n");
  result = openapiDot1sMSTInstanceTimeSinceTopologyChangeGet(clientHandle, badMstid, &var1);
  printf("Invalid MST ID:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceTimeSinceTopologyChangeGet():\n");
  result = openapiDot1sMSTInstanceTimeSinceTopologyChangeGet(clientHandle, mstid, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("openapiDot1sMSTInstanceTimeSinceTopologyChangeGet() sanity successful\n\n");

  printf("Testing openapiDot1sMSTInstanceTopologyChangeCountGet():\n");
  result = openapiDot1sMSTInstanceTopologyChangeCountGet(NULL, mstid, &var1);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceTopologyChangeCountGet():\n");
  result = openapiDot1sMSTInstanceTopologyChangeCountGet(clientHandle, badMstid, &var1);
  printf("Invalid MST ID:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceTopologyChangeCountGet():\n");
  result = openapiDot1sMSTInstanceTopologyChangeCountGet(clientHandle, mstid, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("openapiDot1sMSTInstanceTopologyChangeCountGet() sanity successful\n\n");

  printf("Testing openapiDot1sMSTInstanceTopologyChangeStateGet():\n");
  result = openapiDot1sMSTInstanceTopologyChangeStateGet(NULL, mstid, &var2);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceTopologyChangeStateGet():\n");
  result = openapiDot1sMSTInstanceTopologyChangeStateGet(clientHandle, badMstid, &var2);
  printf("Invalid MST ID:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceTopologyChangeStateGet():\n");
  result = openapiDot1sMSTInstanceTopologyChangeStateGet(clientHandle, mstid, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("openapiDot1sMSTInstanceTopologyChangeStateGet() sanity successful\n\n");
  printf("Testing openapiDot1sMSTInstanceDesignatedRootIDGet():\n");

  result = openapiDot1sMSTInstanceDesignatedRootIDGet(NULL, mstid, &id);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceDesignatedRootIDGet():\n");
  result = openapiDot1sMSTInstanceDesignatedRootIDGet(clientHandle, badMstid, &id);
  printf("Invalid MST ID:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceDesignatedRootIDGet():\n");
  result = openapiDot1sMSTInstanceDesignatedRootIDGet(clientHandle, mstid, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("openapiDot1sMSTInstanceDesignatedRootIDGet() sanity successful\n\n");

  printf("Testing openapiDot1sMSTInstanceRootPathCostGet():\n");
  result = openapiDot1sMSTInstanceRootPathCostGet(NULL, mstid, &var1);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceRootPathCostGet():\n");
  result = openapiDot1sMSTInstanceRootPathCostGet(clientHandle, badMstid, &var1);
  printf("Invalid MST ID:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceRootPathCostGet():\n");
  result = openapiDot1sMSTInstanceRootPathCostGet(clientHandle, mstid, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("openapiDot1sMSTInstanceRootPathCostGet() sanity successful\n\n");

  printf("Testing openapiDot1sMSTInstanceRootPortIDGet():\n");

  result = openapiDot1sMSTInstanceRootPortIDGet(NULL, mstid, &var1);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceRootPortIDGet():\n");
  result = openapiDot1sMSTInstanceRootPortIDGet(clientHandle, badMstid, &var1);
  printf("Invalid MST ID:(result = %d)\n", result);

  printf("Testing openapiDot1sMSTInstanceRootPortIDGet():\n");
  result = openapiDot1sMSTInstanceRootPortIDGet(clientHandle, mstid, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("openapiDot1sMSTInstanceRootPortIDGet() sanity successful\n\n");

  printf("Testing openapiDot1sCISTMaxAgeGet():\n");
  result = openapiDot1sCISTMaxAgeGet(NULL, &var1);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sCISTMaxAgeGet():\n");
  result = openapiDot1sCISTMaxAgeGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("openapiDot1sCISTMaxAgeGet() sanity successful\n\n");

  printf("Testing openapiDot1sBridgeMaxHopGet():\n");
  result = openapiDot1sBridgeMaxHopGet(NULL, &var1);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sBridgeMaxHopGet():\n");
  result = openapiDot1sBridgeMaxHopGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("openapiDot1sBridgeMaxHopGet() sanity successful\n\n");

  printf("Testing openapiDot1sBridgeTxHoldCountGet():\n");
  result = openapiDot1sBridgeTxHoldCountGet(NULL, &var1);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sBridgeTxHoldCountGet():\n");
  result = openapiDot1sBridgeTxHoldCountGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("openapiDot1sBridgeTxHoldCountGet() sanity successful\n\n");

  printf("Testing openapiDot1sCISTRootFwdDelayGet():\n");
  result = openapiDot1sCISTRootFwdDelayGet(NULL, &var1);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sCISTRootFwdDelayGet():\n");
  result = openapiDot1sCISTRootFwdDelayGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("openapiDot1sCISTRootFwdDelayGet() sanity successful\n\n");

  printf("Testing openapiDot1sCISTHelloTimeGet():\n");
  result = openapiDot1sCISTHelloTimeGet(NULL, 1, &var1);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sCISTHelloTimeGet():\n");
  result = openapiDot1sCISTHelloTimeGet(clientHandle, 9999, &var1);
  printf("Likely invalid interface number:(result = %d)\n", result);

  printf("Testing openapiDot1sCISTHelloTimeGet():\n");
  result = openapiDot1sCISTHelloTimeGet(clientHandle, 1, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("openapiDot1sCISTHelloTimeGet() sanity successful\n\n");

  printf("Testing openapiDot1sCISTBridgeHoldTimeGet():\n");
  result = openapiDot1sCISTBridgeHoldTimeGet(NULL, &var1);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sCISTBridgeHoldTimeGet():\n");
  result = openapiDot1sCISTBridgeHoldTimeGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("openapiDot1sCISTBridgeHoldTimeGet() sanity successful\n\n");

  printf("Testing openapiDot1sCISTRegionalRootIDGet():\n");
  result = openapiDot1sCISTRegionalRootIDGet(NULL, &id);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sCISTRegionalRootIDGet():\n");
  result = openapiDot1sCISTRegionalRootIDGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("openapiDot1sCISTRegionalRootIDGet() sanity successful\n\n");

  printf("Testing openapiDot1sCISTRegionalRootPathCostGet():\n");
  result = openapiDot1sCISTRegionalRootPathCostGet(NULL, &var1);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiDot1sCISTRegionalRootPathCostGet():\n");
  result = openapiDot1sCISTRegionalRootPathCostGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("openapiDot1sCISTRegionalRootPathCostGet() sanity successful\n\n");

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for Spanning Tree Configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum, arg1, arg2;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];

  if (argc < 2)
  {
    printDot1sAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);
  
  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("dot1s_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Spanning Tree API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");
  /* exercise various OPEN API Spanning Tree functions */

  switch (testNum)
  {
    case 1:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SModeSet(&clientHandle, arg1);
      break;

    case 2:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1SModeGet(&clientHandle);
      break;

    case 3:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SIntfBPDUFilterModeSet(&clientHandle, arg1, arg2);
      break;

    case 4:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SIntfBPDUFilterModeGet(&clientHandle, arg1);
      break;
    
    case 5:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SIntfBPDUFloodModeSet(&clientHandle, arg1, arg2);
      break;

    case 6:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SIntfBPDUFloodModeGet(&clientHandle, arg1);
      break;
  
    case 7:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SBPDUGuardModeSet(&clientHandle, arg1);
      break;
  
    case 8:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1SBPDUGuardModeGet(&clientHandle);
      break;
  
    case 9:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1SConfigNameSet(&clientHandle, argv[2]);
      break;

    case 10:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1SConfigNameGet(&clientHandle);
      break;

    case 11:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SConfigRevisionSet(&clientHandle, arg1);
      break;

    case 12:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1SConfigRevisionGet(&clientHandle);
      break;

    case 13:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SForceVersionSet(&clientHandle, arg1);
      break;

    case 14:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1SForceVersionGet(&clientHandle);
      break;

    case 15:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SIntfGuardModeSet(&clientHandle, arg1, arg2);
      break;

    case 16:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SIntfGuardModeGet(&clientHandle, arg1);
      break;

    case 17:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SMSTInstanceSet(&clientHandle, arg1, OPEN_ACTION_ADD);
      break;

    case 18:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SMSTInstanceSet(&clientHandle, arg1, OPEN_ACTION_DELETE);
      break;

    case 19:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1SMSTInstancesGet(&clientHandle);
      break;

    case 20:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SMSTInstancePrioritySet(&clientHandle, arg1, arg2);
      break;

    case 21:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SMSTInstancePriorityGet(&clientHandle, arg1);
      break;

    case 22:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SMSTInstanceVLANSet(&clientHandle, arg1, arg2, OPEN_ACTION_ADD);
      break;

    case 23:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SMSTInstanceVLANSet(&clientHandle, arg1, arg2, OPEN_ACTION_DELETE);
      break;

    case 24:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SMSTInstanceVLANsGet(&clientHandle, arg1);
      break;

    case 25:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SMSTInstanceFIdsGet(&clientHandle, arg1);
      break;

    case 26:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SIntfModeSet(&clientHandle, arg1, arg2);
      break;

    case 27:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SIntfModeGet(&clientHandle, arg1);
      break;

    case 28:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SIntfPortFwdStateGet(&clientHandle, arg1, arg2);
      break;

    case 29:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SIntfPortRoleGet(&clientHandle, arg1, arg2);
      break;

    case 30:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SIntfEdgePortModeSet(&clientHandle, arg1, arg2);
      break;

    case 31:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SIntfEdgePortModeGet(&clientHandle, arg1);
      break;

    case 32:
      if (argc != 4)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      dot1SIntfPortLoopInconsistentStateGet(&clientHandle, arg1, arg2);
      break;

    case 33:
      if (argc != 3)
      { 
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1sMSTInstanceBridgeIdentifierGet(&clientHandle, arg1);      
      break;
    case 34:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1sMSTInstanceTimeSinceTopologyChangeGet(&clientHandle, arg1);
      break;
    case 35:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1sMSTInstanceTopologyChangeCountGet(&clientHandle, arg1);
      break;
    case 36:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1sMSTInstanceTopologyChangeStateGet(&clientHandle, arg1);    
      break;
    case 37:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1sMSTInstanceDesignatedRootIDGet(&clientHandle, arg1);
      break;
    case 38:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1sMSTInstanceRootPathCostGet(&clientHandle, arg1);
      break;
    case 39:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1sMSTInstanceRootPortIDGet(&clientHandle, arg1);
      break;
    case 40:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sCISTMaxAgeGet(&clientHandle);
      break;
    case 41:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sBridgeMaxHopGet(&clientHandle);
      break;
    case 42:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sBridgeTxHoldCountGet(&clientHandle);
      break;
    case 43:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sCISTRootFwdDelayGet(&clientHandle);
      break;
    case 44:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1sCISTHelloTimeGet(&clientHandle, arg1);
      break;
    case 45:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sCISTBridgeHoldTimeGet(&clientHandle);
      break;
    case 46:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sCISTRegionalRootIDGet(&clientHandle);
      break;
    case 47:
      if (argc != 2)
      {
        printDot1sAppMenu();
        exit(1);
      }
      dot1sCISTRegionalRootPathCostGet(&clientHandle);
      break;
    
    case 48:
      if (argc != 3)
      {
        printDot1sAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      dot1SOpENAPIsTestSanity(&clientHandle, arg1);
      break;

    default:
      printDot1sAppMenu();
      break;
  }
                                          
  /* Log goodbye message with OPEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Spanning Tree API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
