/*********************************************************************
*
*  Copyright 2016-2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  image_example.c
*
* @purpose   Example application to demonstrate various code image configuration APIs
*
* @component OpEN
*
* @comments
*
* @create    11/17/2015
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "openapi_common.h"
#include "proc_util.h"

#define OPENAPI_IMAGE_NAME_MAX 40           
#define OPENAPI_DEFAULT_IMAGE_DESCR_MAX 256 

#define OPEN_BUFF_EX_SIZE (256+1)
char *exampleName;

/*********************************************************************
* @purpose  Test code image API to retrieve image labels.
* 
* @param    clientHandle     @b{(input)}  Client handle from registration API
* @param    activeLabel      @b{(output)} Active label name
* @param    backupLabel      @b{(output)} Backup label name
*
* @returns none
* 
* @end
*********************************************************************/
void imageLabelsGet (openapiClientHandle_t *clientHandle,
                     char *activeLabel,
                     char *backupLabel)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc aBuffdesc;
  open_buffdesc bBuffdesc;

  aBuffdesc.pstart = activeLabel;
  aBuffdesc.size = OPENAPI_IMAGE_NAME_MAX;

  bBuffdesc.pstart = backupLabel;
  bBuffdesc.size = OPENAPI_IMAGE_NAME_MAX;

  result = openapiImageLabelsGet(clientHandle, &aBuffdesc, &bBuffdesc);

  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to get code image labels. (result = %d)\n", result);
  }
  else
  {
    printf("imageLabelsGet: activeLabel=%s backupLabel=%s\n",
            (char *)aBuffdesc.pstart, (char *)bBuffdesc.pstart);
  }
}

/*********************************************************************
* @purpose  Test code image API to retrieve image names.
* 
* @param    clientHandle     @b{(input)}  Client handle from registration API
* @param    unit             @b{(input)}  Unit number
* @param    activeImage      @b{(output)} Active image name
* @param    backupImage      @b{(output)} Backup image name
* @param    activatedImage   @b{(output)} Activated image name
* 
* @returns none
* 
* @end
*********************************************************************/
void imageNamesGet (openapiClientHandle_t *clientHandle,
                    uint32_t unit,
                    char *activeImage,
                    char *backupImage,
                    char *activatedImage)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc aBuffdesc;
  open_buffdesc bBuffdesc;
  open_buffdesc cBuffdesc;

  aBuffdesc.pstart = activeImage;
  aBuffdesc.size = OPENAPI_IMAGE_NAME_MAX;

  bBuffdesc.pstart = backupImage;
  bBuffdesc.size = OPENAPI_IMAGE_NAME_MAX;

  cBuffdesc.pstart = activatedImage;
  cBuffdesc.size = OPENAPI_IMAGE_NAME_MAX;

  result = openapiImageNamesGet(clientHandle, unit, &aBuffdesc, &bBuffdesc, &cBuffdesc);

  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to get code image names. (result = %d)\n", result);
  }
  else
  {
    printf("imageNamesGet: activeImage=%s backupImage=%s activatedImage=%s\n",
            (char *)aBuffdesc.pstart, (char *)bBuffdesc.pstart, (char *)cBuffdesc.pstart);
  }
}

/*********************************************************************
* @purpose  Test code image API to retrieve the image description.
* 
* @param    clientHandle     @b{(input)}  Client handle from registration API
* @param    imageLabel       @b{(input)}  Image label name
* @param    imageDescr       @b{(output)} Associated text
* 
* @returns none
* 
* @end
*********************************************************************/
void imageDescrGet (openapiClientHandle_t *clientHandle,
                    char *imageLabel,
                    char *imageDescr)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc aBuffdesc;
  open_buffdesc bBuffdesc;
  char labelStr[OPENAPI_IMAGE_NAME_MAX];

  strcpy(labelStr, imageLabel);
  aBuffdesc.pstart = labelStr;
  aBuffdesc.size = strlen(labelStr)+1;

  bBuffdesc.pstart = imageDescr;
  bBuffdesc.size = OPENAPI_DEFAULT_IMAGE_DESCR_MAX;

  result = openapiImageDescrGet(clientHandle, &aBuffdesc, &bBuffdesc);

  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to get code image description. (result = %d)\n", result);
  }
  else
  {
    printf("imageDescrGet: imageLabel=%s imageDescr=%s\n", imageLabel, (char *)bBuffdesc.pstart);
  }
}

/*********************************************************************
* @purpose  Set the text description for a given image.
*
* @param    client_handle    @b{(input)} Client handle from registration API
* @param    imageLabel       @b{(input)} Image label name
* @param    imageDescr       @b{(input)} Associated text
*
* @returns none
* 
* @end
*********************************************************************/
void imageDescrSet (openapiClientHandle_t *clientHandle,
                    char *imageLabel,
                    char *imageDescr)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc aBuffdesc;
  open_buffdesc bBuffdesc;
  char labelStr[OPENAPI_IMAGE_NAME_MAX];
  char descrStr[OPENAPI_DEFAULT_IMAGE_DESCR_MAX];

  strcpy(labelStr, imageLabel);
  aBuffdesc.pstart = labelStr;
  aBuffdesc.size = strlen(labelStr)+1;

  strcpy(descrStr, imageDescr);
  bBuffdesc.pstart = descrStr;
  bBuffdesc.size = strlen(descrStr)+1;

  result = openapiImageDescrSet(clientHandle, &aBuffdesc, &bBuffdesc);

  if (result != OPEN_E_NONE)
  {
    printf("imageDescrSet: Bad return code trying to set code image description. (result = %d)\n", result);
  }
  else
  {
    printf("imageDescrSet: imageLabel=%s imageDescr=%s\n", imageLabel, imageDescr);
  }
}

/*********************************************************************
* @purpose  Determines if the image label refers to a valid image name.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    imageLabel       @b{(input)}  Image label name
* @param    -                             Minimum value 1  
* @param    -                             Maximum value @ref OPENAPI_IMAGE_NAME_MAX 
* @param    isValid          @b{(output)} Returns @ref OPEN_TRUE if image label
*                                         is valid, else @ref OPEN_FALSE if not
*
* @returns none
*
* @end
*********************************************************************/
void imageLabelIsValid (openapiClientHandle_t *clientHandle,
                        char *imageLabel,
                        bool *isValid)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc buffdesc;
  OPEN_BOOL_t flag = OPEN_FALSE;
  char labelStr[OPENAPI_IMAGE_NAME_MAX];

  strcpy(labelStr, imageLabel);
  buffdesc.pstart = labelStr;
  buffdesc.size = strlen(labelStr)+1;

  result = openapiImageLabelIsValid(clientHandle, &buffdesc, &flag);

  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to validate code image. (result = %d)\n", result);
  }
  else
  {
    printf("imageLabelIsValid: imageLabel=%s isValid=%s.\n", imageLabel, (flag==OPEN_TRUE)?"true":"false");
    *isValid = (flag==OPEN_TRUE)?true:false;
  }
}

/*********************************************************************
* @purpose  Sets the specified image as the active image for the subsequent re-boots.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    unit             @b{(input)}  Unit number
* @param    -                             Minimum value 1  
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    imageLabel       @b{(input)}  Image label name
* @param    -                             Minimum value 1  
* @param    -                             Maximum value @ref OPENAPI_IMAGE_NAME_MAX 
* @param    updateBootCode   @b{(input)}  @ref OPEN_TRUE to update boot code, else @ref OPEN_FALSE
* 
* @returns none
*
* @end
*********************************************************************/
void imageActivate(openapiClientHandle_t *clientHandle,
                   uint32_t unit,
                   char *imageLabel,
                   OPEN_BOOL_t updateBootCode)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc buffdesc;
  char labelStr[OPENAPI_IMAGE_NAME_MAX];

  strcpy(labelStr, imageLabel);
  buffdesc.pstart = labelStr;
  buffdesc.size = strlen(labelStr)+1;

  result = openapiImageActivate(clientHandle, unit, &buffdesc, updateBootCode);

  if (result != OPEN_E_NONE)
  {
    printf("imageActivate: Bad return code trying to set active image. (result = %d)\n", result);
  }
  else
  {
    printf("imageActivate: imageLabel=%s unit=%d.\n", imageLabel, unit);
  }
}

/*********************************************************************
* @purpose  Deletes the specified image.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    unit             @b{(input)}  Unit number
* @param    -                             Minimum value 1  
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    imageLabel       @b{(input)}  Image label name
* @param    -                             Minimum value 1  
* @param    -                             Maximum value @ref OPENAPI_IMAGE_NAME_MAX 
* 
* @returns none
*
* @end
*********************************************************************/
void imageDelete(openapiClientHandle_t *clientHandle,
                 uint32_t unit,
                 char *imageLabel)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc buffdesc;
  char labelStr[OPENAPI_IMAGE_NAME_MAX];

  strcpy(labelStr, imageLabel);
  buffdesc.pstart = labelStr;
  buffdesc.size = strlen(labelStr)+1;

  result = openapiImageDelete(clientHandle, unit, &buffdesc);

  if (result != OPEN_E_NONE)
  {
    printf("imageDelete: Bad return code trying to delete image. (result = %d)\n", result);
  }
  else
  {
    printf("imageDelete: imageLabel=%s unit=%d.\n", imageLabel, unit);
  }
}

/*********************************************************************
* @purpose  Updates the bootcode from the currently activated image.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    unit             @b{(input)}  Unit number
* @param    -                             Minimum value 1  
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* 
* @returns none
*
* @end
*********************************************************************/
void bootcodeUpdate(openapiClientHandle_t *clientHandle,
                    uint32_t unit)
{
  open_error_t result = OPEN_E_NONE;

  result = openapiBootCodeUpdate(clientHandle, unit);

  if (result != OPEN_E_NONE)
  {
    printf("bootcodeUpdate: Bad return code trying to update bootcode. (result = %d)\n", result);
  }
  else
  {
    printf("bootcodeUpdate: unit=%d.\n", unit);
  }
}

/*********************************************************************
* @purpose  Validate and upgrade a locally available image
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    imagePath        @b{(input)}  Full path of image to upgrade.
* @param    imageLabel       @b{(input)}  Label of image to upgrade.
* 
* @returns none
*
* @end
*********************************************************************/
void imageOpENAPIUpgradeTest(openapiClientHandle_t *clientHandle,
                             char *imagePath,
                             char *imageLabel)
{
  open_error_t  result = OPEN_E_NONE;
  open_buffdesc imageLocation;
  open_buffdesc imageName;
  open_buffdesc activeBuffdesc;
  open_buffdesc backupBuffdesc;
  char          activeLabel[OPENAPI_IMAGE_NAME_MAX];
  char          backupLabel[OPENAPI_IMAGE_NAME_MAX];
  
  printf("Testing imageOpENAPIUpgradeTest()\n");

  memset(activeLabel, 0, sizeof(activeLabel));  	
  memset(backupLabel, 0, sizeof(backupLabel));

  activeBuffdesc.pstart = activeLabel;
  activeBuffdesc.size   = sizeof(activeLabel); /* openapiImageLabelsGet() allows max OPENAPI_IMAGE_NAME_MAX */ 

  backupBuffdesc.pstart = backupLabel;
  backupBuffdesc.size   = sizeof(backupLabel);

  result = openapiImageLabelsGet(clientHandle, &activeBuffdesc, &backupBuffdesc);

  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to get code image labels. (result = %d)\n", result);
    return;
  }

  imageLocation.pstart = imagePath;
  imageLocation.size   = strlen(imagePath) + 1;

  if (0 == strcmp(imageLabel, "active"))
  {
    imageName.pstart = activeLabel;
    imageName.size   = strlen(activeLabel) + 1;  
  }
  else
  {
    imageName.pstart = backupLabel;
    imageName.size   = strlen(backupLabel) + 1;
  }
   
  result = openapiImageUpgrade(clientHandle, &imageLocation, &imageName);

  if (result != OPEN_E_NONE)
  {
    printf("imageUpgrade: Bad return code trying to upgrade image [%s]. (result = %d)\n", imagePath, result);
  }
  else
  {
    printf("imageUpgrade: Successfully upgraded with image [%s]\n", imagePath);
  }
}

/*******************************************************************
*
* @brief  This function prints the Image Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printImageAppMenu()
{
  printf("\nUsage:  image_example <test#> <arg1> <arg2> ...\n\n");
  printf("Test  1: get image labels: %s 1\n", exampleName);
  printf("Test  2: get image names: %s 2\n", exampleName);
  printf("Test  3: get image descriptions: %s 3\n", exampleName);
  printf("Test  4: get image description: %s 4 image\n", exampleName);
  printf("Test  5: set image description: %s 5 image description\n", exampleName);
  printf("Test  6: delete backup image: %s 6 image\n", exampleName);
  printf("Test  7: check that that argument (e.g., 'active') is a valid image label for this platform: %s 7 image\n", exampleName);
  printf("Test  8: update boot code: %s 8\n", exampleName);
  printf("Test  9: activate backup image and update boot code: %s 9\n", exampleName);
  printf("Test  10: activate backup image but do not update boot code: %s 10\n", exampleName);
  printf("Test  11: copy image to switch: %s 11 label path\n", exampleName);
  printf("Test  12: run negative tests: %s 12\n", exampleName);
  return;
}

/*********************************************************************
* @purpose  Sanity of Image OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
*
* @returns  none
*
* @note
*
* @end
*********************************************************************/
void imageOpENAPISanity(openapiClientHandle_t *clientHandle)
{
  char     activeLabel[OPENAPI_IMAGE_NAME_MAX];
  char     activeImage[OPENAPI_IMAGE_NAME_MAX];
  char     activeDescr[OPENAPI_DEFAULT_IMAGE_DESCR_MAX];
  char     backupLabel[OPENAPI_IMAGE_NAME_MAX];
  char     backupImage[OPENAPI_IMAGE_NAME_MAX];
  char     activatedImage[OPENAPI_IMAGE_NAME_MAX];
  uint32_t unit = 1;
  OPEN_BOOL_t flag = OPEN_FALSE;

  open_error_t result;
  open_buffdesc aBuffdesc;
  open_buffdesc bBuffdesc;
  open_buffdesc cBuffdesc;
  open_buffdesc buffdesc;

  aBuffdesc.pstart = activeImage;
  aBuffdesc.size = OPENAPI_IMAGE_NAME_MAX;

  bBuffdesc.pstart = backupImage;
  bBuffdesc.size = OPENAPI_IMAGE_NAME_MAX;

  cBuffdesc.pstart = activatedImage;
  cBuffdesc.size = OPENAPI_IMAGE_NAME_MAX;

  result = openapiImageNamesGet(NULL, unit, &aBuffdesc, &bBuffdesc, &cBuffdesc);

  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  result = openapiImageNamesGet(clientHandle, unit, NULL, &bBuffdesc, &cBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  result = openapiImageNamesGet(clientHandle, unit, &aBuffdesc, NULL, &cBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  result = openapiImageNamesGet(clientHandle, unit, &aBuffdesc, &bBuffdesc, NULL);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  aBuffdesc.pstart = NULL;
  result = openapiImageNamesGet(clientHandle, unit, &aBuffdesc, &bBuffdesc, &cBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  aBuffdesc.pstart = activeImage;
  aBuffdesc.size = OPENAPI_IMAGE_NAME_MAX + 1;
  result = openapiImageNamesGet(clientHandle, unit, &aBuffdesc, &bBuffdesc, &cBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  aBuffdesc.size = OPENAPI_IMAGE_NAME_MAX;
  bBuffdesc.pstart = NULL;
  bBuffdesc.size = OPENAPI_IMAGE_NAME_MAX;
  result = openapiImageNamesGet(clientHandle, unit, &aBuffdesc, &bBuffdesc, &cBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  bBuffdesc.pstart = backupImage;
  bBuffdesc.size = OPENAPI_IMAGE_NAME_MAX + 1;
  result = openapiImageNamesGet(clientHandle, unit, &aBuffdesc, &bBuffdesc, &cBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  bBuffdesc.size = OPENAPI_IMAGE_NAME_MAX;
  cBuffdesc.pstart = NULL;
  cBuffdesc.size = OPENAPI_IMAGE_NAME_MAX;
  result = openapiImageNamesGet(clientHandle, unit, &aBuffdesc, &bBuffdesc, &cBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  cBuffdesc.pstart = activatedImage;
  cBuffdesc.size = OPENAPI_IMAGE_NAME_MAX + 1;
  result = openapiImageNamesGet(clientHandle, unit, &aBuffdesc, &bBuffdesc, &cBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  aBuffdesc.pstart = NULL;
  aBuffdesc.size = OPENAPI_IMAGE_NAME_MAX;

  bBuffdesc.pstart = backupLabel;
  bBuffdesc.size = OPENAPI_IMAGE_NAME_MAX;

  result = openapiImageLabelsGet(NULL, &aBuffdesc, &bBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  aBuffdesc.pstart = activeLabel;
  aBuffdesc.size = OPENAPI_IMAGE_NAME_MAX + 1;

  bBuffdesc.pstart = backupLabel;
  bBuffdesc.size = OPENAPI_IMAGE_NAME_MAX;

  result = openapiImageLabelsGet(clientHandle, &aBuffdesc, &bBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  aBuffdesc.pstart = activeLabel;
  aBuffdesc.size = OPENAPI_IMAGE_NAME_MAX;

  bBuffdesc.pstart = NULL;
  bBuffdesc.size = OPENAPI_IMAGE_NAME_MAX;

  result = openapiImageLabelsGet(clientHandle, &aBuffdesc, &bBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  aBuffdesc.pstart = activeLabel;
  aBuffdesc.size = OPENAPI_IMAGE_NAME_MAX;

  bBuffdesc.pstart = backupLabel;
  bBuffdesc.size = OPENAPI_IMAGE_NAME_MAX + 1;

  result = openapiImageLabelsGet(clientHandle, &aBuffdesc, &bBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  result = openapiImageLabelIsValid(NULL, &buffdesc, &flag);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  result = openapiImageLabelIsValid(clientHandle, NULL, &flag);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  result = openapiImageLabelIsValid(clientHandle, &buffdesc, NULL);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  buffdesc.pstart = NULL;
  buffdesc.size = OPENAPI_IMAGE_NAME_MAX;
  result = openapiImageLabelIsValid(clientHandle, &buffdesc, &flag);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  result = openapiImageLabelIsValid(clientHandle, &buffdesc, &flag);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  buffdesc.pstart = activeLabel;
  buffdesc.size = OPENAPI_IMAGE_NAME_MAX + 1;
  result = openapiImageLabelIsValid(clientHandle, &buffdesc, &flag);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  strcpy(activeLabel, "active");
  aBuffdesc.pstart = activeLabel;
  aBuffdesc.size = strlen(activeLabel)+1;

  bBuffdesc.pstart = activeDescr;
  bBuffdesc.size = OPENAPI_DEFAULT_IMAGE_DESCR_MAX;

  result = openapiImageDescrGet(NULL, &aBuffdesc, &bBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  result = openapiImageDescrGet(clientHandle, NULL, &bBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  result = openapiImageDescrGet(clientHandle, &aBuffdesc, NULL);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  aBuffdesc.pstart = NULL;
  aBuffdesc.size = strlen(activeLabel)+1;

  bBuffdesc.pstart = activeDescr;
  bBuffdesc.size = OPENAPI_DEFAULT_IMAGE_DESCR_MAX;

  result = openapiImageDescrGet(clientHandle, &aBuffdesc, &bBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  aBuffdesc.pstart = activeLabel;
  aBuffdesc.size = OPENAPI_IMAGE_NAME_MAX + 1;

  bBuffdesc.pstart = activeDescr;
  bBuffdesc.size = OPENAPI_DEFAULT_IMAGE_DESCR_MAX;

  result = openapiImageDescrGet(clientHandle, &aBuffdesc, &bBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  aBuffdesc.pstart = activeLabel;
  aBuffdesc.size = strlen(activeLabel)+1;

  bBuffdesc.pstart = NULL;
  bBuffdesc.size = OPENAPI_DEFAULT_IMAGE_DESCR_MAX;

  result = openapiImageDescrGet(clientHandle, &aBuffdesc, &bBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  aBuffdesc.pstart = activeLabel;
  aBuffdesc.size = strlen(activeLabel)+1;

  bBuffdesc.pstart = activeDescr;
  bBuffdesc.size = OPENAPI_DEFAULT_IMAGE_DESCR_MAX + 1;

  result = openapiImageDescrGet(clientHandle, &aBuffdesc, &bBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  aBuffdesc.pstart = activeLabel;
  aBuffdesc.size = strlen(activeLabel)+1;

  bBuffdesc.pstart = activeDescr;
  bBuffdesc.size = OPENAPI_DEFAULT_IMAGE_DESCR_MAX;

  result = openapiImageDescrSet(NULL, &aBuffdesc, &bBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  aBuffdesc.pstart = activeLabel;
  aBuffdesc.size = strlen(activeLabel)+1;

  bBuffdesc.pstart = activeDescr;
  bBuffdesc.size = OPENAPI_DEFAULT_IMAGE_DESCR_MAX;

  result = openapiImageDescrSet(clientHandle, NULL, &bBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  aBuffdesc.pstart = activeLabel;
  aBuffdesc.size = strlen(activeLabel)+1;

  bBuffdesc.pstart = activeDescr;
  bBuffdesc.size = OPENAPI_DEFAULT_IMAGE_DESCR_MAX;

  result = openapiImageDescrSet(clientHandle, &aBuffdesc, NULL);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  aBuffdesc.pstart = NULL;
  aBuffdesc.size = strlen(activeLabel)+1;

  bBuffdesc.pstart = activeDescr;
  bBuffdesc.size = OPENAPI_DEFAULT_IMAGE_DESCR_MAX;

  result = openapiImageDescrSet(clientHandle, &aBuffdesc, &bBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  aBuffdesc.pstart = activeLabel;
  aBuffdesc.size = OPENAPI_IMAGE_NAME_MAX + 1;

  bBuffdesc.pstart = activeDescr;
  bBuffdesc.size = OPENAPI_DEFAULT_IMAGE_DESCR_MAX;

  result = openapiImageDescrSet(clientHandle, &aBuffdesc, &bBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  aBuffdesc.pstart = activeLabel;
  aBuffdesc.size = 0;

  bBuffdesc.pstart = activeDescr;
  bBuffdesc.size = OPENAPI_DEFAULT_IMAGE_DESCR_MAX;

  result = openapiImageDescrSet(clientHandle, &aBuffdesc, &bBuffdesc);
  if (result != OPEN_E_FAIL) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_FAIL got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  aBuffdesc.pstart = activeLabel;
  aBuffdesc.size = strlen(activeLabel)+1;

  bBuffdesc.pstart = NULL;
  bBuffdesc.size = OPENAPI_DEFAULT_IMAGE_DESCR_MAX;

  result = openapiImageDescrSet(clientHandle, &aBuffdesc, &bBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }


  aBuffdesc.pstart = activeLabel;
  aBuffdesc.size = strlen(activeLabel)+1;

  bBuffdesc.pstart = activeDescr;
  bBuffdesc.size = OPENAPI_DEFAULT_IMAGE_DESCR_MAX + 1; 

  result = openapiImageDescrSet(clientHandle, &aBuffdesc, &bBuffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  buffdesc.pstart = activeLabel;
  buffdesc.size = strlen(activeLabel)+1;

  result = openapiImageActivate(NULL, unit, &buffdesc, OPEN_TRUE);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  buffdesc.pstart = NULL;
  buffdesc.size = strlen(activeLabel)+1;

  result = openapiImageActivate(NULL, unit, &buffdesc, OPEN_TRUE);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  buffdesc.pstart = activeLabel;
  buffdesc.size = strlen(activeLabel)+1;

  result = openapiImageActivate(NULL, unit, &buffdesc, OPEN_TRUE);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  result = openapiImageActivate(clientHandle, unit, NULL, OPEN_TRUE);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  /* note, this is a bad parameter (17) but implementation returns OPEN_E_FAIL */
  result = openapiImageActivate(clientHandle, unit, &buffdesc, 17);
  if (result != OPEN_E_FAIL) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_FAIL got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  result = openapiImageDelete(NULL, unit, &buffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  /* Note, this is a bad param, but impl returns OPEN_E_FAIL */

  result = openapiImageDelete(clientHandle, -1, &buffdesc);
  if (result != OPEN_E_FAIL) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_FAIL got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  result = openapiImageDelete(clientHandle, unit, NULL);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  buffdesc.pstart = NULL;
  buffdesc.size = OPENAPI_DEFAULT_IMAGE_DESCR_MAX;
  result = openapiImageDelete(clientHandle, unit, &buffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  buffdesc.pstart = activeLabel;
  buffdesc.size = OPENAPI_DEFAULT_IMAGE_DESCR_MAX + 1;
  result = openapiImageDelete(clientHandle, unit, &buffdesc);
  if (result != OPEN_E_PARAM) 
  {
    fprintf(stderr, "%s:%d FAIL expect OPEN_E_PARAM got %d\n", 
                    __FUNCTION__, __LINE__, result);
    exit(1);
  }

  fprintf(stderr, "All sanity tests were successful.");

  return;  
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for code image configuration.
*
* @returns  0: Success
* @returns  1: Usage error
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_buffdesc         switch_os_revision;
  char                  switch_os_revision_string[100];
  open_error_t          result;  
  uint32_t              testNum;
  int                   unit = 1;
  bool                  isValid;
  char                  activeLabel[OPENAPI_IMAGE_NAME_MAX];
  char                  activeImage[OPENAPI_IMAGE_NAME_MAX];
  char                  activeDescr[OPENAPI_DEFAULT_IMAGE_DESCR_MAX];
  char                  backupLabel[OPENAPI_IMAGE_NAME_MAX];
  char                  backupImage[OPENAPI_IMAGE_NAME_MAX];
  char                  backupDescr[OPENAPI_DEFAULT_IMAGE_DESCR_MAX];
  char                  activatedImage[OPENAPI_IMAGE_NAME_MAX];

  exampleName = argv[0];
  if (argc < 2)
  {
    printImageAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("image_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Code Image API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
  {
    printf("Network OS version = %s\n", switch_os_revision_string);
  }
  else
  {
    printf("Network OS version retrieve error\n");
  }

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 2)
      {
        printImageAppMenu();
        exit(1);
      }

      imageLabelsGet(&clientHandle, activeLabel, backupLabel);
      break;
	  
    case 2:
      if (argc != 2)
      {
        printImageAppMenu();
        exit(1);
      }

      imageNamesGet(&clientHandle, unit, activeImage, backupImage, activatedImage);
      break;
	  
    case 3:
      if (argc != 2)
      {
        printImageAppMenu();
        exit(1);
      }
	  
      imageLabelsGet(&clientHandle, activeLabel, backupLabel);
      imageDescrGet(&clientHandle, activeLabel, activeDescr);
      imageDescrGet(&clientHandle, backupLabel, backupDescr);
      break;
	  
    case 4:
      if (argc != 3)
      {
        printImageAppMenu();
        exit(1);
      }

      imageLabelsGet(&clientHandle, activeLabel, backupLabel);
      if ((0 != strcmp(argv[2], activeLabel)) && (0 != strcmp(argv[2], backupLabel)))
      {
        printImageAppMenu();
        exit(1);
      }
	  
      imageDescrGet(&clientHandle, argv[2], activeDescr);
      break;
	  
    case 5:
      if (argc != 4)
      {
        printImageAppMenu();
        exit(1);
      }

      imageLabelsGet(&clientHandle, activeLabel, backupLabel);
      if ((0 != strcmp(argv[2], activeLabel)) && (0 != strcmp(argv[2], backupLabel)))
      {
        printImageAppMenu();
        exit(1);
      }

      imageDescrSet (&clientHandle, argv[2], argv[3]);
      break;
	  
    case 6:
      if (argc != 3)
      {
        printImageAppMenu();
        exit(1);
      }

      imageLabelsGet(&clientHandle, activeLabel, backupLabel);
      if (strcmp(backupLabel, argv[2]) == 0) 
      {
        imageDelete(&clientHandle, unit, argv[2]);    
      } 
      else 
      {
        printf("Unable to delete a non-backup image '%s'. Try again with '%s'\n",
               argv[2], backupLabel);
      }
      break;
	  
    case 7:
      if (argc != 3)
      {
        printImageAppMenu();
        exit(1);
      }

      imageLabelIsValid(&clientHandle, argv[2], &isValid);
      break;
	  
    case 8:
      if (argc != 2)
      {
        printImageAppMenu();
        exit(1);
      }

      bootcodeUpdate(&clientHandle, unit);   
      break;
	  
    case 9:
      if (argc != 2)
      {
        printImageAppMenu();
        exit(1);
      }

      imageLabelsGet(&clientHandle, activeLabel, backupLabel);
      imageActivate(&clientHandle, unit, backupLabel, OPEN_TRUE);  
      break;
	  
    case 10:
      if (argc != 2)
      {
        printImageAppMenu();
        exit(1);
      }

      imageLabelsGet(&clientHandle, activeLabel, backupLabel);
      imageActivate(&clientHandle, unit, backupLabel, OPEN_FALSE);  
      break;
	  
    case 11:
      if (argc != 4)
      {
        printImageAppMenu();
        exit(1);
      }

      if ((0 != strcmp(argv[2], "active")) && (0 != strcmp(argv[2], "backup")))
      {
        printImageAppMenu();
        exit(1);
      }
	  
      imageOpENAPIUpgradeTest(&clientHandle, argv[3], argv[2]);
      break;
	  
    case 12:
      if (argc != 2)
      {
        printImageAppMenu();
        exit(1);
      }

      imageOpENAPISanity(&clientHandle);
      break;
	  
    default:
      printImageAppMenu();
      break;
  }
    
  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Code Image API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
