/*********************************************************************
*
*  Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  system_example.c
*
* @purpose   System Configuration OpEN APIs Example 
*
* @component OpEN
*
* @comments
*
* @create    04/22/2013
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_system.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the System Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printSystemAppMenu()
{
  printf("Usage:  system_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: Set system prompt: system_example 1 <system-prompt> \n");
  printf("Test 2: Set system host name: system_example 2 <system host-name>\n");
  printf("Test 3: Get system host name: system_example 3 \n");
  printf("Test 4: Set terminal length of an access-line: system_example 4 <access-line> <terminal length> \n");
  printf("Test 5: Set terminal length of an access-line to default: system_example 5 <access-line> \n");
  printf("Test 6: Get terminal length of an access-line: system_example 6 <access-line> \n");
  printf("Test 7: Set serial timeout value: system_example 7 <timeout> \n");
  printf("Test 8: Set serial timeout value to default: system_example 8 \n");
  printf("Test 9: Get serial timeout value: system_example 9 \n");
  printf("Test 10: Set network port IPv4 parameters: system_example 10 <ipv4-address> <subnet-mask> <gateway> \n");
  printf("Test 11: Get network port IPv4 parameters: system_example 11 \n");
  printf("Test 12: Delete network port IPv4 parameters: system_example 12 \n");
  printf("Test 13: Set network port IPv6 parameters: system_example 13 <ipv6-address> <prefix-length> \n");
  printf("Test 14: Set network port IPv6 gateway: system_example 14 <gateway> \n");
  printf("Test 15: Get network port IPv6 parameters: system_example 15 \n");
  printf("Test 16: Delete network port IPv6 prefix: system_example 16 <ipv6-address> <prefix-length> \n");
  printf("Test 17: Set service port IPv4 parameters: system_example 17 <ipv4-address> <subnet-mask> <gateway> \n");
  printf("Test 18: Get service port IPv4 parameters: system_example 18 \n");
  printf("Test 19: Delete service port IPv4 parameters: system_example 19 \n");
  printf("Test 20: Set service port IPv6 parameters: system_example 20 <ipv6-address> <prefix-length> \n");
  printf("Test 21: Set service port IPv6 gateway: system_example 21 <gateway> \n");
  printf("Test 22: Get service port IPv6 parameters: system_example 22 \n");
  printf("Test 23: Delete service port IPv6 parameters: system_example 23 <ipv6-address> <prefix-length>\n");
  printf("Test 24: Set network port IPv4 protocol mode: system_example 24 <mode> \n");
  printf("Test 25: Get network port Ipv4 protocol mode: system_example 25 \n");
  printf("Test 26: Set service port Ipv4 protocol mode: system_example 26 <mode> \n");
  printf("Test 27: Get service port Ipv4 protocol mode: system_example 27 \n");
  printf("Test 28: Set telnet server admin mode: system_example 28 <mode> \n");
  printf("Test 29: Get telnet server admin mode: system_example 29 \n");
  printf("Test 30: Set network port IPv6 administrative mode: system_example 30 <mode> \n");
  printf("Test 31: Get network port IPv6 administrative mode: system_example 31 \n");
  printf("Test 32: Set network port IPv6 auto config mode: system_example 32 <mode>\n");
  printf("Test 33: Get network port IPv6 auto config mode: system_example 33 \n");
  printf("Test 34: Set network port IPv6 protocol mode: system_example 34 <mode>\n");
  printf("Test 35: Get network port IPv6 protocol mode: system_example 35 \n");
  printf("Test 36: Set service port IPv6 administrative mode: system_example 36 <mode> \n");
  printf("Test 37: Get service port IPv6 administrative mode: system_example 37 \n");
  printf("Test 38: Set service port IPv6 auto config mode: system_example 38 <mode>\n");
  printf("Test 39: Get service port IPv6 auto config mode: system_example 39 \n");
  printf("Test 40: Set service port IPv6 protocol mode: system_example 40 <mode>\n");
  printf("Test 41: Get service port IPv6 protocol mode: system_example 41 \n");
  printf("Test 42: Get System Burned-in MAC address : system_example 42 \n");
  printf("Test 43: Get System MAC address : system_example 43 \n");
  printf("Test 44: System Save configuration : system_example 44 \n");
  printf("Test 45: System Reload : system_example 45 \n");
  printf("Test 48: Set network management VLAN: system_example 48 <vlan> \n");
  printf("Test 49: Get network management VLAN: system_example 49 \n");
  printf("Test 50: Get system maintenance level: system_example 50 \n");
  printf("Test 51: Set resource tracking flag: system_example 51 <mode> \n");
  printf("Test 52: Get system memory information: system_example 52 \n");
  printf("Test 53: System OpEN APIs Sanity: system_example 53 \n");

  return;
}

/*********************************************************************
* @purpose  Set system prompt
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    systemPrompt     @b{(input)}   System prompt
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void systemPromptSet(openapiClientHandle_t *clientHandle, char *systemPrompt)
{
  open_error_t result;
  open_buffdesc buffdesc;
  char str[256];

  memset(str, 0, sizeof(str));
  strncpy(str, systemPrompt, (sizeof(str) - 1));
  buffdesc.pstart = str;
  buffdesc.size = strlen(str)+1;

  if ((result = openapiSystemPromptSet(clientHandle, &buffdesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set system prompt. (result = %d)\n", result);
  }
  else
  {
    printf("System prompt set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Set system hostname
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    hostName         @b{(input)}   System hostname
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void systemHostNameSet(openapiClientHandle_t *clientHandle, char *hostName)
{
  open_error_t result;
  open_buffdesc buffdesc;
  char str[256];

  memset(str, 0, sizeof(str));
  strncpy(str, hostName, (sizeof(str) - 1));
  buffdesc.pstart = str;
  buffdesc.size = strlen(str)+1;

  if ((result = openapiSystemHostnameSet(clientHandle, &buffdesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set system hostname. (result = %d)\n", result);
  }
  else
  {
    printf("System hostname is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get system hostname
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    hostName         @b{(output)}  System hostname
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void systemHostNameGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc buffdesc;
  uint32_t hostNameSize = 0;
  char *str;

  if ((result = openapiSystemHostnameSizeGet(clientHandle, &hostNameSize) != OPEN_E_NONE))
  {
    printf("Bad return code trying to get hostname maximum size. (result = %d)\n", result);
  }

  if ((str = (char *)malloc(hostNameSize)) == NULL)
  {
    printf("Could nor allocate memory.\n"); 
    return;
  }

  buffdesc.pstart = str;
  buffdesc.size = hostNameSize ;

  if ((result = openapiSystemHostnameGet(clientHandle, &buffdesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get system hostname. (result = %d)\n", result);
  }
  else
  {
    printf("System hostname: %s. (result = %d)\n", str, result);
  }

  free(str);
  return;
}

/*********************************************************************
* @purpose  Get system hardware version
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    hardwareVersion  @b{(output)}  System Hardware Version
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void systemHardwareVersionGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc buffdesc;
  char hardwareVersion[OPENAPI_HW_VERSION_STR_SIZE];
  
  buffdesc.pstart = hardwareVersion;
  buffdesc.size = sizeof(hardwareVersion);

  if ((result = openapiHardwareVersionGet(clientHandle, &buffdesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get system hardware version. (result = %d)\n", result);
  }
  else
  {
    printf("System hardware version: %s. (result = %d)\n", hardwareVersion, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set the terminal length of an access line.
* 
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    accessLine       @b{(input)}   Access Line
* @param    termLength       @b{(input)}   Terminal length of the access line
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void terminalLengthSet(openapiClientHandle_t *clientHandle, OPEN_ACCESS_LINE_t accessLine, uint32_t termLength)
{
  open_error_t result;

  if ((result = openapiLineTerminalLengthSet(clientHandle, accessLine, termLength)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set terminal length for an accessline. (result = %d)\n", result);
  }
  else
  {
    printf("Terminal length of access line %d is set successfully. (result = %d)\n", accessLine, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set the terminal length of an access line to default.
* 
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    accessLine       @b{(input)}   Access Line
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void terminalLengthDefaultSet(openapiClientHandle_t *clientHandle, OPEN_ACCESS_LINE_t accessLine)
{
  open_error_t result;

  if ((result = openapiLineTerminalLengthDefaultSet(clientHandle, accessLine)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set terminal length to default for an accessline. (result = %d)\n", result);
  }
  else
  {
    printf("Terminal length of access line %d is set to default successfully. (result = %d)\n", accessLine, result);
  }
  
  return;
} 

/*********************************************************************
* @purpose  Get the terminal length of an access line.
* 
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    accessLine       @b{(input)}   Access Line
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void terminalLengthGet(openapiClientHandle_t *clientHandle, OPEN_ACCESS_LINE_t accessLine)
{
  open_error_t result;
  uint32_t termLength;

  if ((result = openapiLineTerminalLengthGet(clientHandle, accessLine, &termLength)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get terminal length for an accessline. (result = %d)\n", result);
  }
  else
  {
    printf("Terminal length of access line %d: %d. (result = %d)\n", accessLine, termLength, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set serial timeout value.
* 
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    serialTimeout    @b{(input)}   Serial timeout
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void serialTimeoutSet(openapiClientHandle_t *clientHandle, uint32_t serialTimeout)
{
  open_error_t result;

  if ((result = openapiSerialTimeoutSet(clientHandle, serialTimeout)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set serial timeout. (result = %d)\n", result);
  }
  else
  {
    printf("Serial timeout set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Set serial timeout to default.
* 
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void serialTimeoutDefaultSet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  if ((result = openapiSerialTimeoutDefaultSet(clientHandle)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set serial timeout to default value. (result = %d)\n", result);
  }
  else
  {
    printf("Serial timeout default value is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get serial timeout value.
* 
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
*   
* @notes  
*
*@end
*********************************************************************/
void serialTimeoutGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t serialTimeout;

  if ((result = openapiSerialTimeoutGet(clientHandle, &serialTimeout)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get serial timeout. (result = %d)\n", result);
  }
  else
  {
    printf("Serial timeout: %d. (result = %d)\n", serialTimeout, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set network management VLAN.
* 
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    mgmtVlan         @b{(input)}   Network management VLAN ID
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch.
* 
* @end
*********************************************************************/
void networkMgmtVlanSet(openapiClientHandle_t *clientHandle, uint32_t mgmtVlan)
{
  open_error_t result;

  if ((result = openapiNetworkMgmtVlanSet(clientHandle, mgmtVlan)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set network management VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("Network management VLAN set to %d. (result = %d)\n", mgmtVlan, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get network management VLAN.
* 
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
*   
* @notes  
*
*@end
*********************************************************************/
void networkMgmtVlanGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t mgmtVlan;

  if ((result = openapiNetworkMgmtVlanGet(clientHandle, &mgmtVlan)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get network management VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("Network management VLAN: %d. (result = %d)\n", mgmtVlan, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set network port IPv4 address, subnet mask and gateway.
*   
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    ipStr            @b{(input)}   IPv4 address
* @param    netmaskStr       @b{(input)}   Subnet mask
* @param    gatewayStr       @b{(input)}   Gateway address
* 
* @returns  Calling this API will change the running configuration of the switch
*   
* @notes  
* 
*@end
*********************************************************************/
void networkPortIPv4ParametersSet(openapiClientHandle_t *clientHandle, 
                                char *ipStr, char *netmaskStr, char *gatewayStr)
{
  open_error_t result;
  open_inet_addr_t ipAddr, netmask, gateway;
  open_buffdesc ipBuffdesc, maskBuffdesc, gatewayBuffdesc;
  char str[40], str1[40], str2[40];

  memset(str, 0, sizeof(str));
  strncpy(str, ipStr, (sizeof(str) - 1));
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = sizeof(str);

  memset(str1, 0, sizeof(str1));
  strncpy(str1, netmaskStr, (sizeof(str1) - 1));
  maskBuffdesc.pstart = str1;
  maskBuffdesc.size = sizeof(str1);

  memset(str2, 0, sizeof(str2));
  strncpy(str2, gatewayStr, (sizeof(str2) - 1));
  gatewayBuffdesc.pstart = str2;
  gatewayBuffdesc.size = sizeof(str2);
  
  /* Validate IP address */
  if ((result = openapiInetAddrGet(clientHandle, &ipBuffdesc, &ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert IP address. (result = %d)\n", result);
    return;
  }

  /* Validate network mask */
  if ((result = openapiInetAddrGet(clientHandle, &maskBuffdesc, &netmask)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert network mask. (result = %d)\n", result);
    return;
  }
  /* Validate gateway */
  if ((result = openapiInetAddrGet(clientHandle, &gatewayBuffdesc, &gateway)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert gateway IP address. (result = %d)\n", result);
    return;
  }

  if ((result = openapiNetworkPortIPv4ParametersSet(clientHandle, &ipAddr, &netmask, &gateway)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set network port parameters. (result = %d)\n", result);
  }
  else
  {
    printf("Network port parameter are set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Delete network port IPv4 address, subnet mask and gateway.
*   
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch.
* 
*@end
*********************************************************************/
void networkPortIPv4ParametersDelete(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  if ((result = openapiNetworkPortIPv4ParametersDelete(clientHandle)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete network port parameters.. (result = %d)\n", result);
  }
  else
  {
    printf("Network port parameters are deleted successfully. (result = %d)\n", result);
  }
  return;

}

/*********************************************************************
* @purpose  Get network port IPv4 address, subnet mask and gateway.
*   
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
*   
* @none
* 
*@end
*********************************************************************/
void networkPortIPv4ParametersGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_inet_addr_t ipAddr, netmask, gateway;
  struct in_addr ipAddrStr;
  OPEN_LINK_STATE_t linkState;

  /* Get the IP address  and netmask*/
  if ((result = openapiNetworkPortIPv4AddrGet(clientHandle, &ipAddr, &netmask)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get nework port IP address and subnet mask. (result = %d)\n", result);
  }
  else
  {
    ipAddrStr.s_addr = htonl(ipAddr.addr.ipv4);
    printf("Network IP address: %s\n", inet_ntoa(ipAddrStr));

    ipAddrStr.s_addr = htonl(netmask.addr.ipv4);
    printf("Subnet Mask: %s\n", inet_ntoa(ipAddrStr));
  }

  /* Get the gateway */
  if ((result = openapiNetworkPortGatewayIPv4AddrGet(clientHandle, &gateway)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get nework port gateway. (result = %d)\n", result);
  }
  else
  {
    ipAddrStr.s_addr = htonl(gateway.addr.ipv4);
    printf("Gateway: %s\n", inet_ntoa(ipAddrStr));
  }

  /* Get Interface status */
  if (openapiNetworkPortLinkStateGet(clientHandle, &linkState) == OPEN_E_NONE)
  {
    printf("Interface Status: %s\n", ((linkState == OPEN_LINK_UP) ? "Up" : "Down"));
  }
  else
  {
    printf("Bad return code trying to get interface link state. \n");
  }

  return; 
}

/*********************************************************************
* @purpose  Set network port IPv6 address and prefix length.
*   
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    ipv6AddrStr      @b{(input)}   IPv6 address
* @param    prefixLen        @b{(input)}   Prefix Length
* 
* @returns none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
*@end
*********************************************************************/
void networkPortIPv6PrefixSet(openapiClientHandle_t *clientHandle, char *ipv6AddrStr, uint32_t prefixLen)
{
  open_error_t result;
  open_inet_pfx_t ipAddr;
  open_buffdesc ipBuffDesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, ipv6AddrStr, (sizeof(str) - 1));
  ipBuffDesc.pstart = str;
  ipBuffDesc.size = sizeof(str);

  /* Validate IP address */
  if ((result = openapiInetAddrGet(clientHandle, &ipBuffDesc, &(ipAddr.ipAddr))) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert IP address. (result = %d)\n", result);
    return;
  }

  ipAddr.pfxLen = prefixLen;

  if ((result = openapiNetworkPortIPv6PrefixSet(clientHandle, &ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set network port IPv6 prefix and prefix length. (result = %d)\n", result);
  }
  else
  {
    printf("Network port IPv6 prefix and prefix length set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Set network port IPv6 gateway.
*   
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    gatewayStr       @b{(input)}   IPv6 gateway address
* 
* @returns none
*   
* @notes Calling this API will change the running configuration of the switch 
* @notes Pass IPv6 address containing all zeroes to unconfigure
* 
*@end
*********************************************************************/
void networkPortIPv6GatewaySet(openapiClientHandle_t *clientHandle, char *gatewayStr)
{
  open_error_t result;
  open_inet_addr_t gateway;
  open_buffdesc gatewayBuffDesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, gatewayStr, (sizeof(str) - 1));
  gatewayBuffDesc.pstart = str;
  gatewayBuffDesc.size = sizeof(str);

  /* Validate gateway IPv6 address */
  if ((result = openapiInetAddrGet(clientHandle, &gatewayBuffDesc, &gateway)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert IP address. (result = %d)\n", result);
    return;
  }

  if ((result = openapiNetworkPortIPv6GatewaySet(clientHandle, &gateway)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set network port IPv6 gateway. (result = %d)\n", result);
  }
  else
  {
    printf("Network port IPv6 gateway is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get network port IPv6 address, prefix length and gateway.
*   
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns none
*   
* @notes
* 
*@end
*********************************************************************/
void networkPortIPv6ParametersGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_inet_pfx_t ipAddr;
  open_inet_addr_t gateway;
  char ipAddrStr[40];
  char gatewayStr[40];

  memset(&gateway, 0, sizeof(open_inet_addr_t));
  if ((result = openapiNetworkPortIPv6GatewayGet(clientHandle, &gateway)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get network port IPv6 gateway address. (result = %d)\n", result);
  }
  else
  {
    memset(gatewayStr, 0, sizeof(gatewayStr));
    if (inet_ntop(AF_INET6, (void*)&(gateway.addr.ipv6), gatewayStr, sizeof(gatewayStr)) == NULL)
    {
      printf("Bad return code trying to convert IPv6 gateway address.\n");
      return;
    }
    printf("Gateway: %s\n", gatewayStr);
  } 

  printf("\n");
  memset(&ipAddr, 0, sizeof(open_inet_pfx_t));
  if ((result = openapiNetworkPortIPv6AddressesNextGet(clientHandle, &ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get first network port IPv6 address. (result = %d)\n", result);
    return;
  }

  do
  {
    printf("IPv6 Address: ");
    if (inet_ntop(AF_INET6, (void*)&(ipAddr.ipAddr.addr.ipv6), ipAddrStr, sizeof(ipAddrStr)) == NULL)
    {
      printf("Bad return code trying to convert IPv6 address. (result = %d)\n", result);
      return;
    }
    printf("%s\n", ipAddrStr);

    printf("Prefix Length: %d\n", ipAddr.pfxLen);
    printf("\n");
  }while(openapiNetworkPortIPv6AddressesNextGet(clientHandle, &ipAddr) == OPEN_E_NONE);

 return; 
}

/*********************************************************************
* @purpose  Delete network port IPv6 prefix and prefix length.
*   
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    ipv6AddrStr      @b{(input)}   IPv6 address
* @param    prefixLen        @b{(input)}   IPv6 prefix length
* 
* @returns none
*   
* @notes Calling this API will change the running configuration of the switch 
* 
*@end
*********************************************************************/
void networkPortIPv6PrefixDelete(openapiClientHandle_t *clientHandle, char *ipv6AddrStr, uint32_t prefixLen)
{
  open_error_t result; 
  open_inet_pfx_t ipAddr;
  char str[40];
  open_buffdesc ipBuffDesc;

  memset(str, 0, sizeof(str));
  strncpy(str, ipv6AddrStr, (sizeof(str) - 1));
  ipBuffDesc.pstart = str;
  ipBuffDesc.size = sizeof(str);

  if (strncmp(ipv6AddrStr, "all", strlen(ipv6AddrStr)+1) == 0)
  {
    memset(&ipAddr, 0, sizeof(open_inet_pfx_t));
  }
  else 
  {
    if ((result = openapiInetAddrGet(clientHandle, &ipBuffDesc, &(ipAddr.ipAddr))) != OPEN_E_NONE)
    {
      printf("Bad return code trying to convert IP address. (result = %d)\n", result);
      return;
    }
  }

  ipAddr.pfxLen = prefixLen; 
  if ((result = openapiNetworkPortIPv6PrefixDelete(clientHandle, &ipAddr)) != OPEN_E_NONE) 
  {
    printf("Bad return code trying to delete network Ipv6 prefix. (result = %d) \n", result);
  }
  else
  {
    printf("Network IPv6 prefix deleted successfully. (result = %d) \n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Set network protocol mode.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    protoMode        @b{(input)}   Network protocol mode
* 
* @returns  none
*   
* @notes   Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void networkProtocolModeSet(openapiClientHandle_t *clientHandle, OPEN_PROTOCOL_MODE_t protoMode)
{
  open_error_t result;

  if ((result = openapiNetworkPortIPv4ProtocolModeSet(clientHandle, protoMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set network protocol mode. (result = %d)\n", result);
  }
  else
  {
    printf("Network protocol mode set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get network protocol mode.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void networkProtocolModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_PROTOCOL_MODE_t protoMode;

  if ((result = openapiNetworkPortIPv4ProtocolModeGet(clientHandle, &protoMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get network protocol mode. (result = %d)\n", result);
  }
  else
  {
    printf("Network protocol mode: %d. (result = %d)\n", protoMode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set service port protocol mode.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    protoMode        @b{(input)}   Network protocol mode
* 
* @returns  none
*   
* @notes   Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void serviceProtocolModeSet(openapiClientHandle_t *clientHandle, OPEN_PROTOCOL_MODE_t protoMode)
{
  open_error_t result;

  if ((result = openapiServicePortIPv4ProtocolModeSet(clientHandle, protoMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set network protocol mode. (result = %d)\n", result);
  }
  else
  {
    printf("Service protocol mode set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get service port protocol mode.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void servicePortIPv4ProtocolModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_PROTOCOL_MODE_t protoMode;

  if ((result = openapiServicePortIPv4ProtocolModeGet(clientHandle, &protoMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get network protocol mode. (result = %d)\n", result);
  }
  else
  {
    printf("Service protocol mode: %d. (result = %d)\n", protoMode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set telnet server administrative mode.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    mode             @b{(input)}   Telnet server admin mode
* 
* @returns  none
* 
* @notes   Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void telnetServerAdminModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiTelnetServerAdminModeSet(clientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set telnet server admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("Telnet server admin mode is set successfully. (result = %d)\n", result);
  }

  return;  
}

/*********************************************************************
* @purpose  Get telnet server administrative mode.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void telnetServerAdminModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  if ((result = openapiTelnetServerAdminModeGet(clientHandle, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get telnet server admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("Telnet server admin mode: %d. (result = %d)\n", mode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set network port IPv6 admin mode.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    adminMode        @b{(input)}   Network port IPv6 admin mode
* 
* @returns  none
* 
* @notes   Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void networkPortIpv6AdminModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t adminMode)
{
  open_error_t result;

  if ((result = openapiNetworkPortIPv6AdminModeSet(clientHandle, adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set network port IPv6 admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("Network port IPv6 admin mode is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get network port IPv6 admin mode.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void networkPortIpv6AdminModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t adminMode;

  if ((result = openapiNetworkPortIPv6AdminModeGet(clientHandle, &adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get network port IPv6 admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("Network port IPv6 admin mode: %d. (result = %d)\n", adminMode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set network port auto config mode.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    autoConfigMode   @b{(input)}   Network port IPv6 auto config mode
* 
* @returns  none
* 
* @notes   Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void networkPortIpv6AutoConfigModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t autoConfigMode)
{
  open_error_t result;

  if ((result = openapiNetworkPortIPv6AutoConfigModeSet(clientHandle, autoConfigMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set network port IPv6 auto config mode. (result = %d)\n", result);
  }
  else
  {
    printf("Network port IPv6 auto config mode set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get network port auto config mode.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void networkPortIpv6AutoConfigModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t autoConfigMode;

  if ((result = openapiNetworkPortIPv6AutoConfigModeGet(clientHandle, &autoConfigMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set network port IPv6 auto config mode. (result = %d)\n", result);
  }
  else
  {
    printf("Network port IPv6 auto config mode: %d. (result = %d)\n", autoConfigMode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set network port IPv6 protocol mode.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    protoMode        @b{(input)}   Network port IPv6 protocol mode
* 
* @returns  none
* 
* @notes   Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void networkPortIpv6ProtocolModeSet(openapiClientHandle_t *clientHandle, OPEN_PROTOCOL_MODE_t protoMode)
{
  open_error_t result;

  if ((result = openapiNetworkPortIPv6ProtocolModeSet(clientHandle, protoMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set network port IPv6 protocol mode. (result = %d)\n", result);
  }
  else
  {
    printf("Network port IPv6 protocol mode is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get network port IPv6 protocol mode.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void networkPortIpv6ProtocolModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_PROTOCOL_MODE_t protoMode;

  if ((result = openapiNetworkPortIPv6ProtocolModeGet(clientHandle, &protoMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get network port IPv6 protocol mode. (result = %d)\n", result);
  }
  else
  {
    printf("Network port IPv6 protocol mode: %d. (result = %d)\n", protoMode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set service port IPv6 admin mode.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    adminMode        @b{(input)}   Service port IPv6 admin mode
* 
* @returns  none
* 
* @notes   Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void servicePortIpv6AdminModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t adminMode) 
{
  open_error_t result;

  if ((result = openapiServicePortIPv6AdminModeSet(clientHandle, adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set service port IPv6 admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("Service port IPv6 admin mode is set success. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get service port IPv6 admin mode.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void servicePortIpv6AdminModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t adminMode;

  if ((result = openapiServicePortIPv6AdminModeGet(clientHandle, &adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get service port IPv6 admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("Service port IPv6 admin mode: %d. (result = %d)\n", adminMode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set service port IPv6 auto config mode.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    autoConfigMode   @b{(input)}   Service port IPv6 auto config mode
* 
* @returns  none
* 
* @notes   Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void servicePortIpv6AutoConfigModeSet(openapiClientHandle_t *clientHandle, 
                                      OPEN_CONTROL_t autoConfigMode)
{
  open_error_t result;

  if ((result = openapiServicePortIPv6AutoConfigModeSet(clientHandle, autoConfigMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set service port IPv6 auto config. (result = %d)\n", result);
  }
  else
  {
    printf("Service port IPv6 auto config mode is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get service port IPv6 auto config mode.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void servicePortIpv6AutoConfigModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t autoConfigMode;

  if ((result = openapiServicePortIPv6AutoConfigModeGet(clientHandle, &autoConfigMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get service port IPv6 auto config. (result = %d)\n", result);
  }
  else
  {
    printf("Service port IPv6 auto config mode: %d. (result = %d)\n", autoConfigMode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set service port IPv6 protocol mode.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    protoMode        @b{(input)}   Service port IPv6 protocol mode
* 
* @returns  none
* 
* @notes   Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void servicePortIpv6ProtocolModeSet(openapiClientHandle_t *clientHandle, OPEN_PROTOCOL_MODE_t protoMode)
{
  open_error_t result;

  if ((result = openapiServicePortIPv6ProtocolModeSet(clientHandle, protoMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set service port protocol mode. (result = %d)\n", result); 
  }
  else
  {
    printf("Service port IPv6 protocol mode is set successfully. (result = %d)\n", result);
  }

  return; 
}

/*********************************************************************
* @purpose  Get service port IPv6 protocol mode.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void servicePortIpv6ProtocolModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_PROTOCOL_MODE_t protoMode;
  
  if ((result = openapiServicePortIPv6ProtocolModeGet(clientHandle, &protoMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get service port protocol mode. (result = %d)\n", result);
  }
  else
  {
    printf("Service port IPv6 protocol mode: %d. (result = %d)\n", protoMode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set service port IP address, subnet mask and gateway.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    ipStr            @b{(input)}   Service port IP address.
* @param    netmaskStr       @b{(input)}   Subnet mask
* @param    gatewayStr       @b{(input)}   Gateway
* 
* @returns  none
* 
* @notes   Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void servicePortIPv4ParametersSet(openapiClientHandle_t *clientHandle, char *ipStr, char *netmaskStr, char *gatewayStr)
{
  open_error_t result;
  open_inet_addr_t ipAddr, netmask, gateway;
  open_buffdesc ipBuffdesc, maskBuffdesc, gatewayBuffdesc;
  char str[40], str1[40], str2[40];

  memset(str, 0, sizeof(str));
  strncpy(str, ipStr, (sizeof(str) - 1));
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = sizeof(str);

  memset(str1, 0, sizeof(str1));
  strncpy(str1, netmaskStr, (sizeof(str1) - 1));
  maskBuffdesc.pstart = str1;
  maskBuffdesc.size = sizeof(str1);

  memset(str2, 0, sizeof(str2));
  strncpy(str2, gatewayStr, (sizeof(str2) - 1));
  gatewayBuffdesc.pstart = str2;
  gatewayBuffdesc.size = sizeof(str2);

  /* Validate IP address */
  if ((result = openapiInetAddrGet(clientHandle, &ipBuffdesc, &ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert IP address. (result = %d)\n", result);
    return;
  }

  /* Validate network mask */
  if ((result = openapiInetAddrGet(clientHandle, &maskBuffdesc, &netmask)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert service port network mask. (result = %d)\n", result);
    return;
  }
  /* Validate gateway */
  if ((result = openapiInetAddrGet(clientHandle, &gatewayBuffdesc, &gateway)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert gateway IP address. (result = %d)\n", result);
    return;
  }

  if ((result = openapiServicePortIPv4ParametersSet(clientHandle, &ipAddr, &netmask, &gateway)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set service port parameters.. (result = %d)\n", result);
  }
  else
  {
    printf("Service port parameters are set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Delete service port IPv4 address, subnet mask and gateway.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @notes   Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void servicePortIPv4ParametersDelete(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  if ((result = openapiServicePortIPv4ParametersDelete(clientHandle)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete service port parameters.. (result = %d)\n", result);
  }
  else
  {
    printf("Service port parameters are deleted successfully. (result = %d)\n", result);
  }
  return;

}

/*********************************************************************
* @purpose  Get service port IP address, subnet mask and gateway
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void servicePortIPv4ParametersGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_inet_addr_t ipAddr, netmask, gateway;
  struct in_addr ipAddrStr;

  /* Get the IP address and netmask*/
  if ((result = openapiServicePortIPv4AddrGet(clientHandle, &ipAddr, &netmask)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get service port IP address and subnet mask. (result = %d)\n", result);
  }
  else
  {
    ipAddrStr.s_addr = htonl(ipAddr.addr.ipv4);
    printf("Service IP address: %s\n", inet_ntoa(ipAddrStr));

    ipAddrStr.s_addr = htonl(netmask.addr.ipv4);
    printf("Subnet Mask: %s\n", inet_ntoa(ipAddrStr));
  }

  /* Get the gateway */
  if ((result = openapiServicePortGatewayIPv4AddrGet(clientHandle, &gateway)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get service port gateway. (result = %d)\n", result);
  }
  else
  {
    ipAddrStr.s_addr = htonl(gateway.addr.ipv4);
    printf("Gateway: %s\n", inet_ntoa(ipAddrStr));
  }

  return;
}

/*********************************************************************
* @purpose  Set service port IPv6 address and prefix length.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    ipv6Str          @b{(input)}   Service port IPv6 address.
* @param    prefixLen        @b{(input)}   Prefix length
* 
* @returns  none
* 
* @notes   Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void servicePortIPv6PrefixSet(openapiClientHandle_t *clientHandle, char *ipv6Str, uint32_t prefixLen)
{
  open_error_t result;
  open_inet_pfx_t ipAddr;
  char str[40];
  open_buffdesc ipBuffdesc;

  memset(str, 0, sizeof(str));
  strncpy(str, ipv6Str, (sizeof(str) - 1));
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = sizeof(str);
 

  /* Validate IP address */
  if ((result = openapiInetAddrGet(clientHandle, &ipBuffdesc, &(ipAddr.ipAddr))) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert IP address. (result = %d)\n", result);
    return;
  }

  ipAddr.pfxLen = prefixLen;

  if ((result = openapiServicePortIPv6PrefixSet(clientHandle, &ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set service port IPv6 prefix and prefix length. (result = %d)\n", result);
  }
  else
  {
    printf("Service port IPv6 prefix and prefix length set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Set service port IPv6 gateway address.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    gatewayStr       @b{(input)}   IPv6 gateway address
* 
* @returns  none
* 
* @notes   Calling this API will change the running configuration of the switch
* @notes   Pass IPv6 address containing all zeroes to unconfigure
* 
* @end
*********************************************************************/
void servicePortIPv6GatewaySet(openapiClientHandle_t *clientHandle, char *gatewayStr)
{
  open_error_t result;
  open_inet_addr_t gateway;
  char str[40];
  open_buffdesc gatewayBuffdesc;


  memset(str, 0, sizeof(str));
  strncpy(str, gatewayStr, (sizeof(str) - 1));
  gatewayBuffdesc.pstart = str;
  gatewayBuffdesc.size = sizeof(str);

  /* Validate gateway IPv6 address */
  if ((result = openapiInetAddrGet(clientHandle, &gatewayBuffdesc, &gateway)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to convert IP address. (result = %d)\n", result);
    return;
  }


  if ((result = openapiServicePortIPv6GatewaySet(clientHandle, &gateway)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set service port IPv6 gateway. (result = %d)\n", result);
  }
  else
  {
    printf("Service port IPv6 gateway is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get service port IPv6 address(es), prefix length and gateway
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void servicePortIPv6ParametersGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_inet_pfx_t ipAddr;
  open_inet_addr_t gateway;
  char ipAddrStr[40];
  char gatewayStr[40];

  memset(&gateway, 0, sizeof(open_inet_addr_t));
  if ((result = openapiServicePortIPv6GatewayGet(clientHandle, &gateway)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get service port IPv6 gateway address. (result = %d)\n", result);
  }
  else
  {
    if (inet_ntop(AF_INET6, (void*)&(gateway.addr.ipv6), gatewayStr, sizeof(gatewayStr)) == NULL)
    {
      printf("Bad return code trying to convert IPv6 gateway address.\n");
      return;
    }
    printf("Gateway: %s\n", gatewayStr);
  }
  printf("\n");

  memset(&ipAddr, 0, sizeof(open_inet_pfx_t));
  if ((result = openapiServicePortIPv6AddressesNextGet(clientHandle, &ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get first service port IPv6 address. (result = %d)\n", result);
    return;
  }

  do
  {
    printf("IPv6 Address: ");
    if (inet_ntop(AF_INET6, (void*)&(ipAddr.ipAddr.addr.ipv6), ipAddrStr, sizeof(ipAddrStr)) == NULL)
    {
      printf("Bad return code trying to convert IPv6 address. (result = %d)\n", result);
      return;
    }
    printf("%s\n", ipAddrStr);

    printf("Prefix Length: %d\n", ipAddr.pfxLen);
    printf("\n");
  }while(openapiServicePortIPv6AddressesNextGet(clientHandle, &ipAddr) == OPEN_E_NONE);

 return;
}

/*********************************************************************
* @purpose  Delete service port Ipv6 prefix and prefix length.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    ipv6AddrStr      @b{(input)}   Service port IPv6 address
* @param    prefixLen        @b{(input)}   Prefix length
* 
* @returns  none
* 
* @notes   Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void servicePortIPv6PrefixDelete(openapiClientHandle_t *clientHandle, char *ipv6AddrStr, uint32_t prefixLen)
{
  open_error_t result;
  open_inet_pfx_t ipAddr;
  char str[40];
  open_buffdesc ipBuffdesc;

  memset(str, 0, sizeof(str));
  strncpy(str, ipv6AddrStr, (sizeof(str) - 1));
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = sizeof(str);

  if (strncmp(ipv6AddrStr, "all", strlen(ipv6AddrStr)+1) == 0)
  {
    memset(&ipAddr, 0, sizeof(open_inet_pfx_t));
  }
  else
  {
    if ((result = openapiInetAddrGet(clientHandle, &ipBuffdesc, &(ipAddr.ipAddr))) != OPEN_E_NONE)
    {
      printf("Bad return code trying to convert IP address. (result = %d)\n", result);
      return;
    }
  }

  ipAddr.pfxLen = prefixLen;
  if ((result = openapiServicePortIPv6PrefixDelete(clientHandle, &ipAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete service port Ipv6 prefix. (result = %d) \n", result);
  }
  else
  {
    printf("Service port IPv6 prefix deleted successfully. result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get system burned-in MAC address.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void systemBurnedInMacGet (openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc buffdesc;
  char *str = NULL;

  if ((str = (char *)malloc(OPEN_MAC_ADDR_LEN)) == NULL)
  {
    printf("Could not allocate memory.\n");
    return;
  }

  buffdesc.pstart = str;
  buffdesc.size = OPEN_MAC_ADDR_LEN;

  result = openapiSystemBurnedInMacAddrGet(clientHandle, &buffdesc);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get system burned-in MAC address. (result = %d)\n", result);
  }
  else
  {
    printf("System Burned-in MAC addresss (formatted) %02X-%02X-%02X-%02X-%02X-%02X size %d. (result = %d)\n",
         str[0]&0xFF,str[1]&0xFF,str[2]&0xFF,str[3]&0xFF,str[4]&0xFF,str[5]&0xFF, buffdesc.size, result);
  }

  free(str);

  return;
}

/*********************************************************************
* @purpose  Get system MAC address.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void systemMacGet (openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc buffdesc;
  char *str = NULL;

  if ((str = (char *)malloc(OPEN_MAC_ADDR_LEN)) == NULL)
  {
    printf("Could not allocate memory.\n");
    return;
  }

  buffdesc.pstart = str;
  buffdesc.size = OPEN_MAC_ADDR_LEN;

  result = openapiSystemMacAddrGet(clientHandle, &buffdesc);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get system MAC address. (result = %d)\n", result);
  }
  else
  {
    printf("System MAC addresss (formatted) %02X-%02X-%02X-%02X-%02X-%02X size %d. (result = %d)\n",
         str[0]&0xFF,str[1]&0xFF,str[2]&0xFF,str[3]&0xFF,str[4]&0xFF,str[5]&0xFF, buffdesc.size, result);
  }

  free(str);

  return;
}

/*********************************************************************
* @purpose  Save the system configuration.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void systemSaveConfig (openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  result = openapiSaveConfigurationSet(clientHandle);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to save the system configuration. (result = %d)\n", result);
  }
  else
  {
    printf("System Configuration saved successfully! (result = %d)\n", result);
    printf("If using devshell, please exit to allow startup configuration file generation prior to system reload.\n");
  }

  return;
}

/*********************************************************************
* @purpose  Reset the system.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void systemReload (openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  result = openapiSystemReload(clientHandle);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to reset the system. (result = %d)\n", result);
  }
  else
  {
    printf("System reset initiated successfully! (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Sanity of System OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note
* 
* @end
*********************************************************************/
void systemOpENAPISanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc buffdesc; 
  uint32_t hostNameSize = 0;
  char *str;
  open_inet_addr_t ipAddr, netmask, gateway;
  open_inet_pfx_t ipAddrPrefix;
  uint32_t termLength = 0, serialTimeout = 0;  
  OPEN_CONTROL_t adminMode = OPEN_DISABLE, autoConfigMode = OPEN_DISABLE;
  OPEN_PROTOCOL_MODE_t protoMode = OPEN_PROTOCOL_MODE_NONE;
  uint32_t memTotal = 0, memAlloc = 0, memFree = 0;

  if ((result = openapiSystemHostnameSizeGet(clientHandle, &hostNameSize) != OPEN_E_NONE))
  { 
    printf("Bad return code trying to get hostname maximum size. (result = %d)\n", result);
    return;
  }
  
  if ((str = (char *)malloc(hostNameSize)) == NULL)
  { 
    printf("Could nor allocate memory.\n");
    return;
  }
  
  buffdesc.pstart = str;
  buffdesc.size = sizeof(str);

  /* openapiSystemPromptSet() */
  printf("Testing openapiSystemPromptSet(): \n");

  result = openapiSystemPromptSet(NULL, &buffdesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSystemPromptSet(clientHandle, NULL);
  printf("NULL buff descriptor to system prompt. (result = %d)\n", result);

  printf("openapiSystemPromptSet() sanity successful.\n");

  /* openapiSystemPromptMaxSizeGet() */
  printf("Testing openapiSystemPromptMaxSizeGet(): \n");

  result = openapiSystemPromptMaxSizeGet(NULL, &hostNameSize);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSystemPromptMaxSizeGet(clientHandle, NULL);
  printf("NULL system prompt maximum size. (result = %d)\n", result);

  printf("openapiSystemPromptMaxSizeGet() sanity successful.\n");

  /* openapiSystemHostnameSet() */
  printf("Testing openapiSystemHostnameSet(): \n");

  result = openapiSystemHostnameSet(NULL, &buffdesc);  
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSystemHostnameSet(clientHandle, NULL);  
  printf("NULL buff descriptor to system hostname. (result = %d)\n", result);

  printf("openapiSystemHostnameSet() sanity successful.\n");

  /* openapiSystemHostnameGet() */
  printf("Testing openapiSystemHostnameGet(): \n");

  result = openapiSystemHostnameGet(NULL, &buffdesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSystemHostnameGet(clientHandle, NULL);
  printf("NULL buff descriptor to system hostname. (result = %d)\n", result);

  printf("openapiSystemHostnameGet() sanity successful.\n");

  /* openapiSystemHostnameSizeGet() */
  printf("Testing openapiSystemHostnameSizeGet(): \n");

  result = openapiSystemHostnameSizeGet(NULL, &hostNameSize);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSystemHostnameSizeGet(clientHandle, NULL);
  printf("NULL system hostname size. (result = %d)\n", result);

  printf("openapiSystemHostnameSizeGet() sanity successful.\n");

  /* openapiLineTerminalLengthSet() */
  printf("Testing openapiLineTerminalLengthSet(): \n");

  result = openapiLineTerminalLengthSet(NULL, OPEN_ACCESS_LINE_CONSOLE, termLength);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiLineTerminalLengthSet(clientHandle, OPEN_ACCESS_LINE_UNKNOWN, termLength);
  printf("Invalid access line. (result = %d)\n", result);

  result = openapiLineTerminalLengthSet(clientHandle, OPEN_ACCESS_LINE_CONSOLE, 100);
  printf("Invalid terminal length (result = %d)\n", result);

  printf("openapiLineTerminalLengthSet() sanity successful.\n");

  /* openapiLineTerminalLengthDefaultSet() */
  printf("Testing openapiLineTerminalLengthDefaultSet(): \n");

  result = openapiLineTerminalLengthDefaultSet(NULL, OPEN_ACCESS_LINE_TELNET);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiLineTerminalLengthDefaultSet(clientHandle, OPEN_ACCESS_LINE_UNKNOWN);
  printf("Invalid access line. (result = %d)\n", result);

  printf("openapiLineTerminalLengthDefaultSet() sanity successful.\n");

  /* openapiLineTerminalLengthGet() */
  printf("Testing openapiLineTerminalLengthGet(): \n");

  result = openapiLineTerminalLengthGet(NULL, OPEN_ACCESS_LINE_CONSOLE, &termLength);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiLineTerminalLengthGet(clientHandle, OPEN_ACCESS_LINE_UNKNOWN, &termLength);
  printf("Invalid access line. (result = %d)\n", result);

  result = openapiLineTerminalLengthGet(clientHandle, OPEN_ACCESS_LINE_CONSOLE, NULL);
  printf("NULL terminal length. (result = %d)\n", result);

  printf("openapiLineTerminalLengthGet() sanity successful.\n");

  /* openapiSerialTimeoutSet() */
  printf("Testing openapiSerialTimeoutSet(): \n");

  result = openapiSerialTimeoutSet(NULL, serialTimeout);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSerialTimeoutSet(clientHandle, 200);
  printf("Invalid serial timeout. (result = %d)\n", result);

  printf("openapiSerialTimeoutSet() sanity successful.\n");

  /* openapiSerialTimeoutDefaultSet() */
  printf("Testing openapiSerialTimeoutDefaultSet(): \n");

  result = openapiSerialTimeoutDefaultSet(NULL);
  printf("NULL Client Handle. (result = %d)\n", result);

  printf("openapiSerialTimeoutDefaultSet() sanity successful.\n");

  /* openapiSerialTimeoutGet() */
  printf("Testing openapiSerialTimeoutGet(): \n");

  result = openapiSerialTimeoutGet(NULL, &serialTimeout);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSerialTimeoutGet(clientHandle, NULL);
  printf("NULL serial timeout. (result = %d)\n", result);

  printf("openapiSerialTimeoutGet() sanity successful.\n");

  /* openapiNetworkPortIPv4ParametersSet() */
  printf("Testing openapiNetworkPortIPv4ParametersSet(): \n");

  result = openapiNetworkPortIPv4ParametersSet(NULL, &ipAddr, &netmask, &gateway);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNetworkPortIPv4ParametersSet(clientHandle, NULL, &netmask, &gateway);
  printf("NULL IP address. (result = %d)\n", result);

  result = openapiNetworkPortIPv4ParametersSet(clientHandle, &ipAddr, NULL, &gateway);
  printf("NULL subnet mask. (result = %d)\n", result);


  result = openapiNetworkPortIPv4ParametersSet(clientHandle, &ipAddr, &netmask, NULL);
  printf("NULL gateway. (result = %d)\n", result);

  printf("openapiNetworkPortIPv4ParametersSet() sanity successful.\n");

  /* openapiNetworkPortIPv4ParametersDelete() */
  printf("Testing openapiNetworkPortIPv4ParametersDelete(): \n");

  result = openapiNetworkPortIPv4ParametersDelete(NULL);
  printf("NULL Client Handle. (result = %d)\n", result);

  printf("openapiNetworkPortIPv4ParametersDelete() sanity successful.\n");

  /* openapiNetworkPortIPv4AddrGet() */
  printf("Testing openapiNetworkPortIPv4AddrGet(): \n");

  result = openapiNetworkPortIPv4AddrGet(NULL, &ipAddr, &netmask);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNetworkPortIPv4AddrGet(clientHandle, NULL, &netmask);
  printf("NULL IP address. (result = %d)\n", result);

  result = openapiNetworkPortIPv4AddrGet(clientHandle, &ipAddr, NULL);
  printf("NULL subnet mask. (result = %d)\n", result);

  printf("openapiNetworkPortIPv4AddrGet() sanity successful.\n");

  /* openapiNetworkPortGatewayIPv4AddrGet() */
  printf("Testing openapiNetworkPortGatewayIPv4AddrGet(): \n");

  result = openapiNetworkPortGatewayIPv4AddrGet(NULL, &gateway);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNetworkPortGatewayIPv4AddrGet(clientHandle, NULL);
  printf("NULL gateway. (result = %d)\n", result);

  printf("openapiNetworkPortGatewayIPv4AddrGet() sanity successful.\n");

  /* openapiNetworkPortIPv6AdminModeSet() */
  printf("Testing openapiNetworkPortIPv6AdminModeSet(): \n");

  result = openapiNetworkPortIPv6AdminModeSet(NULL, adminMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNetworkPortIPv6AdminModeSet(clientHandle, 5);
  printf("Invalid network port Ipv6 admin mode. (result = %d)\n", result);

  printf("openapiNetworkPortIPv6AdminModeSet() sanity successful.\n");

  /* openapiNetworkPortIPv6AdminModeGet() */
  printf("Testing openapiNetworkPortIPv6AdminModeGet(): \n");

  result = openapiNetworkPortIPv6AdminModeGet(NULL, &adminMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNetworkPortIPv6AdminModeGet(clientHandle, NULL);
  printf("NULL network port Ipv6 admin mode. (result = %d)\n", result);

  printf("openapiNetworkPortIPv6AdminModeGet() sanity successful.\n");

  /* openapiNetworkPortIPv6AutoConfigModeSet() */
  printf("Testing openapiNetworkPortIPv6AutoConfigModeSet(): \n");

  result = openapiNetworkPortIPv6AutoConfigModeSet(NULL, autoConfigMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNetworkPortIPv6AutoConfigModeSet(clientHandle, 5);
  printf("Invalid network port Ipv6 auto config mode. (result = %d)\n", result);

  printf("openapiNetworkPortIPv6AutoConfigModeSet() sanity successful.\n");

  /* openapiNetworkPortIPv6AutoConfigModeGet() */
  printf("Testing openapiNetworkPortIPv6AutoConfigModeGet(): \n");

  result = openapiNetworkPortIPv6AutoConfigModeGet(NULL, &autoConfigMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNetworkPortIPv6AutoConfigModeGet(clientHandle, NULL);
  printf("NULL network port Ipv6 auto config mode. (result = %d)\n", result);

  printf("openapiNetworkPortIPv6AutoConfigModeGet() sanity successful.\n");

  /* openapiNetworkPortIPv6ProtocolModeSet() */
  printf("Testing openapiNetworkPortIPv6ProtocolModeSet(): \n");

  result = openapiNetworkPortIPv6ProtocolModeSet(NULL, protoMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNetworkPortIPv6ProtocolModeSet(clientHandle, OPEN_PROTOCOL_MODE_INVALID);
  printf("Invalid network port Ipv6 protocol mode. (result = %d)\n", result);

  printf("openapiNetworkPortIPv6ProtocolModeSet() sanity successful.\n"); 

  /* openapiNetworkPortIPv6ProtocolModeGet() */
  printf("Testing openapiNetworkPortIPv6ProtocolModeGet(): \n");

  /* openapiNetworkPortIPv6PrefixSet() */
  printf("Testing openapiNetworkPortIPv6PrefixSet(): \n");

  result = openapiNetworkPortIPv6PrefixSet(NULL, &ipAddrPrefix);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNetworkPortIPv6PrefixSet(clientHandle, NULL);
  printf("NULL NULL network port Ipv6 address. (result = %d)\n", result);

  printf("openapiNetworkPortIPv6PrefixSet() sanity successful.\n"); 

  /* openapiNetworkPortIPv6ProtocolModeGet() */
  printf("Testing openapiNetworkPortIPv6ProtocolModeGet(): \n");

  result = openapiNetworkPortIPv6ProtocolModeGet(NULL, &protoMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNetworkPortIPv6ProtocolModeGet(clientHandle, NULL);
  printf("NULL network port Ipv6 protocol mode. (result = %d)\n", result);

  printf("openapiNetworkPortIPv6ProtocolModeGet() sanity successful.\n");

  /* openapiNetworkPortIPv6GatewaySet() */
  printf("Testing openapiNetworkPortIPv6GatewaySet(): \n");

  result = openapiNetworkPortIPv6GatewaySet(NULL, &(ipAddrPrefix.ipAddr));
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNetworkPortIPv6GatewaySet(clientHandle, NULL);
  printf("NULL network port IPv6 gateway. (result = %d)\n", result);

  printf("openapiNetworkPortIPv6GatewaySet() sanity successful.\n");

  /* openapiNetworkPortIPv6AddressesNextGet() */
  printf("Testing openapiNetworkPortIPv6AddressesNextGet(): \n");

  result = openapiNetworkPortIPv6AddressesNextGet(NULL, &ipAddrPrefix);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNetworkPortIPv6AddressesNextGet(clientHandle, NULL);
  printf("NULL IPv6 prefix. (result = %d)\n", result);

  printf("openapiNetworkPortIPv6AddressesNextGet() sanity successful.\n");

  /* openapiNetworkPortIPv6GatewayGet() */
  printf("Testing openapiNetworkPortIPv6GatewayGet(): \n");

  result = openapiNetworkPortIPv6GatewayGet(NULL, &(ipAddrPrefix.ipAddr));
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNetworkPortIPv6GatewayGet(clientHandle, NULL);
  printf("NULL network port IPv6 gateway. (result = %d)\n", result);

  printf("openapiNetworkPortIPv6GatewayGet() sanity successful.\n");

  /* openapiNetworkPortIPv6PrefixDelete() */
  printf("Testing openapiNetworkPortIPv6PrefixDelete(): \n");

  result = openapiNetworkPortIPv6PrefixDelete(NULL, &ipAddrPrefix);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNetworkPortIPv6PrefixDelete(clientHandle, NULL);
  printf("NULL network port ipv6 prefix. (result = %d)\n", result);

  printf("openapiNetworkPortIPv6PrefixDelete() sanity successful.\n");

  /* openapiServicePortIPv4ParametersSet() */
  printf("Testing openapiServicePortIPv4ParametersSet(): \n");

  result = openapiServicePortIPv4ParametersSet(NULL, &ipAddr, &netmask, &gateway);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiServicePortIPv4ParametersSet(clientHandle, NULL, &netmask, &gateway);
  printf("NULL IP address. (result = %d)\n", result);

  result = openapiServicePortIPv4ParametersSet(clientHandle, &ipAddr, NULL, &gateway);
  printf("NULL subnet mask. (result = %d)\n", result);

  result = openapiServicePortIPv4ParametersSet(clientHandle, &ipAddr, &netmask, NULL);
  printf("NULL gateway. (result = %d)\n", result);

  printf("openapiServicePortIPv4ParametersSet() sanity successful.\n");

  /* openapiServicePortIPv4ParametersDelete() */
  printf("Testing openapiServicePortIPv4ParametersDelete(): \n");

  result = openapiServicePortIPv4ParametersDelete(NULL);
  printf("NULL Client Handle. (result = %d)\n", result);

  printf("openapiServicePortIPv4ParametersDelete() sanity successful.\n");

  /* openapiServicePortIPv4AddrGet() */
  printf("Testing openapiServicePortIPv4AddrGet(): \n");

  result = openapiServicePortIPv4AddrGet(NULL, &ipAddr, &netmask);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiServicePortIPv4AddrGet(clientHandle, NULL, &netmask);
  printf("NULL IP address. (result = %d)\n", result);

  result = openapiServicePortIPv4AddrGet(clientHandle, &ipAddr, NULL);
  printf("NULL subnet mask. (result = %d)\n", result);

  printf("openapiServicePortIPv4AddrGet() sanity successful.\n");

  /* openapiServicePortGatewayIPv4AddrGet() */
  printf("Testing openapiServicePortGatewayIPv4AddrGet(): \n");

  result = openapiServicePortGatewayIPv4AddrGet(NULL, &gateway);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiServicePortGatewayIPv4AddrGet(clientHandle, NULL);
  printf("NULL gateway. (result = %d)\n", result);

  printf("openapiServicePortGatewayIPv4AddrGet() sanity successful.\n");

  /* openapiServicePortIPv6AdminModeSet() */
  printf("Testing openapiServicePortIPv6AdminModeSet(): \n");

  result = openapiServicePortIPv6AdminModeSet(NULL, adminMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiServicePortIPv6AdminModeSet(clientHandle, 4);
  printf("Invalid service port ipv6 admin mode. (result = %d)\n", result);

  printf("openapiServicePortIPv6AdminModeSet() sanity successful.\n");

  /* openapiServicePortIPv6AdminModeGet() */
  printf("Testing openapiServicePortIPv6AdminModeGet(): \n");

  result = openapiServicePortIPv6AdminModeGet(NULL, &adminMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiServicePortIPv6AdminModeGet(clientHandle, NULL);
  printf("NULL service port ipv6 admin mode. (result = %d)\n", result);

  printf("openapiServicePortIPv6AdminModeGet() sanity successful.\n");

  /* openapiServicePortIPv6AutoConfigModeSet() */
  printf("Testing openapiServicePortIPv6AutoConfigModeSet(): \n");

  result = openapiServicePortIPv6AutoConfigModeSet(NULL, autoConfigMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiServicePortIPv6AutoConfigModeSet(clientHandle, 3);
  printf("Invalid service port ipv6 auto config mode. (result = %d)\n", result);

  printf("openapiServicePortIPv6AutoConfigModeSet() sanity successful.\n");

  /* openapiServicePortIPv6AutoConfigModeGet() */
  printf("Testing openapiServicePortIPv6AutoConfigModeGet(): \n");

  result = openapiServicePortIPv6AutoConfigModeGet(NULL, &autoConfigMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiServicePortIPv6AutoConfigModeGet(clientHandle, NULL);
  printf("NULL service port ipv6 auto config mode. (result = %d)\n", result);

  printf("openapiServicePortIPv6AutoConfigModeGet() sanity successful.\n");

  /* openapiServicePortIPv6ProtocolModeSet() */
  printf("Testing openapiServicePortIPv6ProtocolModeSet(): \n");

  result = openapiServicePortIPv6ProtocolModeSet(NULL, protoMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiServicePortIPv6ProtocolModeSet(clientHandle, OPEN_PROTOCOL_MODE_INVALID);
  printf("Invalid service port ipv6 protocol mode. (result = %d)\n", result);

  printf("openapiServicePortIPv6ProtocolModeSet() sanity successful.\n");

  /* openapiServicePortIPv6ProtocolModeGet() */
  printf("Testing openapiServicePortIPv6ProtocolModeGet(): \n");

  result = openapiServicePortIPv6ProtocolModeGet(NULL, &protoMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiServicePortIPv6ProtocolModeGet(clientHandle, NULL);
  printf("NULL service port ipv6 protocol mode. (result = %d)\n", result);

  printf("openapiServicePortIPv6ProtocolModeGet() sanity successful.\n");

  /* openapiServicePortIPv6PrefixSet() */
  printf("Testing openapiServicePortIPv6PrefixSet(): \n");

  result = openapiServicePortIPv6PrefixSet(NULL, &ipAddrPrefix);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiServicePortIPv6PrefixSet(clientHandle, NULL);
  printf("NULL service port ipv6 prefix. (result = %d)\n", result);

  printf("openapiServicePortIPv6PrefixSet() sanity successful.\n");

  /* openapiServicePortIPv6GatewaySet() */
  printf("Testing openapiServicePortIPv6GatewaySet(): \n");

  result = openapiServicePortIPv6GatewaySet(NULL, &(ipAddrPrefix.ipAddr));
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiServicePortIPv6GatewaySet(clientHandle, NULL);
  printf("NULL service port ipv6 gateway. (result = %d)\n", result);

  printf("openapiServicePortIPv6GatewaySet() sanity successful.\n");

  /* openapiServicePortIPv6PrefixDelete() */
  printf("Testing openapiServicePortIPv6PrefixDelete(): \n");

  result = openapiServicePortIPv6PrefixDelete(clientHandle, &ipAddrPrefix);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiServicePortIPv6PrefixDelete(clientHandle, NULL);
  printf("NULL service port ipv6 prefix. (result = %d)\n", result);

  printf("openapiServicePortIPv6PrefixDelete() sanity successful.\n");

  /* openapiServicePortIPv6AddressesNextGet() */
  printf("Testing openapiServicePortIPv6AddressesNextGet(): \n");

  result = openapiServicePortIPv6AddressesNextGet(NULL, &ipAddrPrefix);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiServicePortIPv6AddressesNextGet(clientHandle, NULL);
  printf("NULL service port ipv6 prefix. (result = %d)\n", result);

  printf("openapiServicePortIPv6AddressesNextGet() sanity successful.\n");

  /* openapiServicePortIPv6GatewayGet() */
  printf("Testing openapiServicePortIPv6GatewayGet(): \n");

  result = openapiServicePortIPv6GatewayGet(NULL, &gateway);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiServicePortIPv6GatewayGet(clientHandle, NULL);
  printf("NULL Gateway Ip address. (result = %d)\n", result);

  printf("openapiServicePortIPv6GatewayGet() sanity successful.\n");

  /* openapiNetworkPortIPv4ProtocolModeSet() */
  printf("Testing openapiNetworkPortIPv4ProtocolModeSet(): \n");

  result = openapiNetworkPortIPv4ProtocolModeSet(NULL, protoMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNetworkPortIPv4ProtocolModeSet(clientHandle, OPEN_PROTOCOL_MODE_INVALID);
  printf("Invalid network port protocol mode. (result = %d)\n", result);

  printf("openapiNetworkPortIPv4ProtocolModeSet() sanity successful.\n");

  /* openapiNetworkPortIPv4ProtocolModeGet() */
  printf("Testing openapiNetworkPortIPv4ProtocolModeGet(): \n");

  result = openapiNetworkPortIPv4ProtocolModeGet(NULL, &protoMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNetworkPortIPv4ProtocolModeGet(clientHandle, NULL);
  printf("NULL network port protocol mode. (result = %d)\n", result);

  printf("openapiNetworkPortIPv4ProtocolModeGet() sanity successful.\n");

  /* openapiServicePortIPv4ProtocolModeSet() */
  printf("Testing openapiServicePortIPv4ProtocolModeSet(): \n");

  result = openapiServicePortIPv4ProtocolModeSet(NULL, protoMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiServicePortIPv4ProtocolModeSet(clientHandle, OPEN_PROTOCOL_MODE_INVALID);
  printf("Invalid service port protocol mode. (result = %d)\n", result);

  printf("openapiServicePortIPv4ProtocolModeSet() sanity successful.\n");

  /* openapiServicePortIPv4ProtocolModeGet() */
  printf("Testing openapiServicePortIPv4ProtocolModeGet(): \n");

  result = openapiServicePortIPv4ProtocolModeGet(NULL, &protoMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiServicePortIPv4ProtocolModeGet(clientHandle, NULL);
  printf("NULL service port protocol mode. (result = %d)\n", result);

  printf("openapiServicePortIPv4ProtocolModeGet() sanity successful.\n");

  /* openapiTelnetServerAdminModeSet() */
  printf("Testing openapiTelnetServerAdminModeSet(): \n");

  result = openapiTelnetServerAdminModeSet(NULL, adminMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiTelnetServerAdminModeSet(clientHandle, 4);
  printf("Invalid telnet server admin mode. (result = %d)\n", result);

  printf("openapiTelnetServerAdminModeSet() sanity successful.\n");

  /* openapiTelnetServerAdminModeGet() */
  printf("Testing openapiTelnetServerAdminModeGet(): \n");

  result = openapiTelnetServerAdminModeGet(NULL, &adminMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiTelnetServerAdminModeGet(clientHandle, NULL);
  printf("NULL telnet server admin mode. (result = %d)\n", result);

  printf("openapiTelnetServerAdminModeGet sanity successful.\n");

  /* openapiSystemBurnedInMacAddrGet() */
  printf("Testing openapiSystemBurnedInMacAddrGet(): \n");

  result = openapiSystemBurnedInMacAddrGet(NULL, &buffdesc);
  printf(" %s -- NULL Client Handle. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  buffdesc.size = 4;
  result = openapiSystemBurnedInMacAddrGet(clientHandle, &buffdesc);
  printf(" %s -- Buffer descriptor too small. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  buffdesc.size = sizeof(str);
  buffdesc.pstart = str;
  result = openapiSystemBurnedInMacAddrGet(clientHandle, &buffdesc);
  printf(" %s -- System Burned-in MAC addresss %02X-%02X-%02X-%02X-%02X-%02X size %d. (result = %d)\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         str[0]&0xFF,str[1]&0xFF,str[2]&0xFF,str[3]&0xFF,str[4]&0xFF,str[5]&0xFF, 
         buffdesc.size, result);

  /* openapiSystemMacAddrGet() */
  printf("Testing openapiSystemMacAddrGet(): \n");

  result = openapiSystemMacAddrGet(NULL, &buffdesc);
  printf(" %s -- NULL Client Handle. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  buffdesc.size = 4;
  result = openapiSystemMacAddrGet(clientHandle, &buffdesc);
  printf(" %s -- Buffer descriptor too small. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  buffdesc.size = sizeof(str);
  buffdesc.pstart = str;
  result = openapiSystemMacAddrGet(clientHandle, &buffdesc);
  printf(" %s -- System MAC addresss %02X-%02X-%02X-%02X-%02X-%02X size %d. (result = %d)\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         str[0]&0xFF,str[1]&0xFF,str[2]&0xFF,str[3]&0xFF,str[4]&0xFF,str[5]&0xFF, 
         buffdesc.size, result);

  free(str);

  /* openapiSystemMemoryInfoGet() */
  printf("Testing openapiSystemMemoryInfoGet(): \n");

  result = openapiSystemMemoryInfoGet(NULL, &memTotal, &memAlloc, &memFree);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSystemMemoryInfoGet(clientHandle, NULL, &memAlloc, &memFree);
  printf("NULL total memory ptr. (result = %d)\n", result);

  result = openapiSystemMemoryInfoGet(clientHandle, &memTotal, NULL, &memFree);
  printf("NULL allocated memory ptr. (result = %d)\n", result);

  result = openapiSystemMemoryInfoGet(clientHandle, &memTotal, &memAlloc, NULL);
  printf("NULL free memory ptr. (result = %d)\n", result);

  printf("openapiSystemMemoryInfoGet sanity successful.\n");

  return;
}

/**************************************************************************
* @purpose  Sanity of OpEN API to get system maintenance level
*
* @param    clientHandle  @b{(input)}   Client handle from registration API
*
* @returns  none
*
* @note
*
* @end
**************************************************************************/
void systemMaintenanceLevelGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc switch_mLevel;
  char switch_mLevel_string[OPEN_SYSTEM_PARAM_SIZE+1];

  switch_mLevel.pstart = switch_mLevel_string;
  switch_mLevel.size = sizeof(switch_mLevel_string);
  result = openapiMaintenanceLevelGet(clientHandle, &switch_mLevel);
  if (OPEN_E_NONE == result)
    printf("Maintenance level = %s\n", switch_mLevel_string);
  else
    printf("Maintenance level retrieve error. (result = %d)\n", result);

  return;
}

/*********************************************************************
* @purpose  Set client resource tracking.
* 
* @param    clientHandle     @b{(input)}   Client handle from registration API
* @param    resTrackingFlag  @b{(input)}   Resource tracking flag.
* 
* @returns  none
*   
* @notes  When tracking is enabled, all the HW resources allocated by
*         a given OpEN API client will be tracked and those resources
*         will be freed when OpEN API client ceases to exist.
* 
* @end
*********************************************************************/
void clientResourceTrackingSet(openapiClientHandle_t *clientHandle, uint32_t resTrackingFlag)
{
  open_error_t result;

  if ((result = openapiClientResourceTrackingSet(clientHandle, resTrackingFlag)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set client resource tracking. (result = %d)\n", result);
  }
  else
  {
    printf("Client resource tracking set to %d. (result = %d)\n", resTrackingFlag, result);
  }

  return;
}

/**************************************************************************
* @purpose  Get system memory information
*
* @param    clientHandle  @b{(input)}   Client handle from registration API
*
* @returns  none
*
* @note
*
* @end
**************************************************************************/
void systemMemoryInfoGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t memTotal = 0, memAlloc = 0, memFree = 0;

  result = openapiSystemMemoryInfoGet(clientHandle, &memTotal, &memAlloc, &memFree);
  if (OPEN_E_NONE == result)
  {
    printf("System memory info (KB):\n");
    printf("  allocated = %10u\n", memAlloc);
    printf("  free      = %10u\n", memFree);
    printf("  total     = %10u\n", memTotal);
  }
  else
  {
    printf("System memory info get error. (result = %d)\n", result);
  }

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  open_buffdesc switch_hw_version;
  open_revision_data_t openApiVersion;
  char switch_os_revision_string[OPENAPI_OS_REVISION_STR_SIZE];
  char switch_hw_version_string[OPENAPI_HW_VERSION_STR_SIZE];
  uint32_t arg1, arg2;

  if (argc < 2)
  {
    printSystemAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("system_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting System API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  switch_hw_version.pstart = switch_hw_version_string;
  switch_hw_version.size = sizeof(switch_hw_version_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  if (openapiApiVersionGet(&clientHandle, &openApiVersion) == OPEN_E_NONE)
    printf("Open API Version = %d.%d.%d.%d\n", 
           openApiVersion.release,
           openApiVersion.version,
           openApiVersion.maint_level,
           openApiVersion.build_num);
  else
    printf("Open API Version retrieve error\n");

  if (openapiHardwareVersionGet(&clientHandle, &switch_hw_version) == OPEN_E_NONE)
    printf("Network Processing Device = %s\n", switch_hw_version_string);
  else
    printf("Network processing device retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 3)
      {
        printSystemAppMenu();
        exit(1);
      }
      systemPromptSet(&clientHandle, argv[2]);
      break;
    case 2:
      if (argc != 3)
      {
        printSystemAppMenu();
        exit(1);
      }
      systemHostNameSet(&clientHandle, argv[2]);
      break;
    case 3:
      if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      systemHostNameGet(&clientHandle);
      break;
    case 4:
      if (argc != 4)
      {
        printSystemAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      terminalLengthSet(&clientHandle, arg1, arg2);
      break;
    case 5:
      if (argc != 3)
      {
        printSystemAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      terminalLengthDefaultSet(&clientHandle, arg1);
      break;
    case 6:
      if (argc != 3)
      {
        printSystemAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      terminalLengthGet(&clientHandle, arg1); 
      break;
    case 7:
      if (argc != 3)
      {
        printSystemAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      serialTimeoutSet(&clientHandle, arg1);
      break;
    case 8:
      if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      serialTimeoutDefaultSet(&clientHandle); 
      break;
    case 9:
      if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      serialTimeoutGet(&clientHandle);
      break;
    case 10:
      if (argc != 5)
      {
        printSystemAppMenu();
        exit(1);
      }
      networkPortIPv4ParametersSet(&clientHandle, argv[2], argv[3], argv[4]);
      break;
    case 11:
      if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      networkPortIPv4ParametersGet(&clientHandle);
      break;
      case 12:
      if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      networkPortIPv4ParametersDelete(&clientHandle);
      break;
    case 13:
      if (argc != 4)
      {
        printSystemAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      networkPortIPv6PrefixSet(&clientHandle, argv[2], arg1);
      break; 
    case 14:
      if (argc != 3)
      {
        printSystemAppMenu();
        exit(1);
      }
      networkPortIPv6GatewaySet(&clientHandle, argv[2]);
      break;
    case 15:
      if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      networkPortIPv6ParametersGet(&clientHandle);
      break;
    case 16:
      if (argc != 4)
      {
        printSystemAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      networkPortIPv6PrefixDelete(&clientHandle, argv[2], arg1);
      break;
    case 17:
      if (argc != 5)
      {
        printSystemAppMenu();
        exit(1);
      }
      servicePortIPv4ParametersSet(&clientHandle, argv[2], argv[3], argv[4]);
      break;
    case 18:
      if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      servicePortIPv4ParametersGet(&clientHandle);
      break;
    case 19:
      if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      servicePortIPv4ParametersDelete(&clientHandle);
      break;
    case 20:
      if (argc != 4)
      {
        printSystemAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      servicePortIPv6PrefixSet(&clientHandle, argv[2], arg1);
      break;
    case 21:
      if (argc != 3)
      {
        printSystemAppMenu();
        exit(1);
      }
      servicePortIPv6GatewaySet(&clientHandle, argv[2]);
      break;
    case 22:
      if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      servicePortIPv6ParametersGet(&clientHandle);
      break;
    case 23:
      if (argc != 4)
      {
        printSystemAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      servicePortIPv6PrefixDelete(&clientHandle, argv[2], arg1);
      break;
    case 24:
      if (argc != 3)
      {
        printSystemAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      networkProtocolModeSet(&clientHandle, arg1);
      break;
    case 25:
      if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      networkProtocolModeGet(&clientHandle);
      break;
    case 26:
      if (argc != 3)
      {
        printSystemAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      serviceProtocolModeSet(&clientHandle, arg1);
      break;
    case 27:
      if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      servicePortIPv4ProtocolModeGet(&clientHandle);
      break;
    case 28:
      if (argc != 3)
      {
        printSystemAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      telnetServerAdminModeSet(&clientHandle, arg1);
      break;
    case 29:
      if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      telnetServerAdminModeGet(&clientHandle);
      break;
    case 30:
      if (argc != 3)
      {
        printSystemAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      networkPortIpv6AdminModeSet(&clientHandle, arg1);
      break;
    case 31:
      if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      networkPortIpv6AdminModeGet(&clientHandle);
      break; 
    case 32:
      if (argc != 3)
      {
        printSystemAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      networkPortIpv6AutoConfigModeSet(&clientHandle, arg1);
      break;    
    case 33:
      if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      networkPortIpv6AutoConfigModeGet(&clientHandle);
      break;
    case 34:
      if (argc != 3)
      {
        printSystemAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      networkPortIpv6ProtocolModeSet(&clientHandle, arg1);
      break;
    case 35:
      if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      networkPortIpv6ProtocolModeGet(&clientHandle);
      break;
    case 36:
      if (argc != 3)
      {
        printSystemAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      servicePortIpv6AdminModeSet(&clientHandle, arg1);
      break;
    case 37:
      if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      servicePortIpv6AdminModeGet(&clientHandle);
      break; 
    case 38:
      if (argc != 3)
      {
        printSystemAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      servicePortIpv6AutoConfigModeSet(&clientHandle, arg1);
      break;
    case 39:
      if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      servicePortIpv6AutoConfigModeGet(&clientHandle);
      break;
    case 40:
      if (argc != 3)
      {
        printSystemAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      servicePortIpv6ProtocolModeSet(&clientHandle, arg1);
      break;
    case 41:
      if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      servicePortIpv6ProtocolModeGet(&clientHandle);
      break;
     case 42:
       if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      systemBurnedInMacGet(&clientHandle);
      break;
     case 43:
       if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      systemMacGet(&clientHandle);
      break;
     case 44:
       if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      systemSaveConfig(&clientHandle);
      break;
     case 45:
       if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      systemReload(&clientHandle);
      break;
    case 48:
      if (argc != 3)
      {
        printSystemAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      networkMgmtVlanSet(&clientHandle, arg1);
      break;
    case 49:
      if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      networkMgmtVlanGet(&clientHandle);
      break;
    case 50:
      if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      systemMaintenanceLevelGet(&clientHandle);
      break;
    case 51:
      if (argc != 3)
      {
        printSystemAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      clientResourceTrackingSet(&clientHandle, arg1);
      break;
    case 52:
      if (argc != 2)
      {
        printSystemAppMenu();
        exit(1);
      }
      systemMemoryInfoGet(&clientHandle);
      break;
    case 53:
      if (argc != 2)
     {
       printSystemAppMenu();
       exit(1);
     }
     systemOpENAPISanity(&clientHandle);
     break;
    default:
      printSystemAppMenu();
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping System API example application");
        
  (void) openapiClientTearDown(&clientHandle);          
  return 0;
}
