/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  user_accounts_example.c
*
* @purpose   User Accounts OpEN APIs Example.
*
* @component OpEN
*
* @comments
*
* @create    13/03/2013
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_user_manager.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the User Accounts Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printUserAccountsAppMenu()
{
  printf("Usage: user_accounts <test#> <arg1> <arg2> ... \n");
  printf("Test 1: Set user name and password: user_accounts_example 1 <username> <password> \n");
  printf("Test 2: Set access level to the user: user_accounts_example 2 <username> <level> \n");
  printf("Test 3: Get the information of users configured:  user_accounts_example 3 \n");
  printf("Test 4: Delete the user account: user_accounts_example 4 <username> \n");
  printf("Test 5: Set the enable password: user_accounts_example 5 <password> \n");
  printf("Test 6: Remove enable password: user_accounts_example 6 \n");
  printf("Test 7: Authenticate login user: user_accounts_example 7 <username> <password> \n");
  printf("Test 8: user_accounts_example OpEN APIs sanity: user_accounts_example 8 \n");
  printf("Test 9: Search the next valid ID of SSH session: user_accounts_example 9 <session type> <start session ID> \n");
  printf("Test 10: Closes the session by ID: user_accounts_example 10 <session ID> \n");

  return;
}

/*********************************************************************
* @purpose  Configure Username and password
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    userName        @b{(input)}   User name to configure.
* @param    password        @b{(input)}   Password for the user.
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void userNamePasswordSet(openapiClientHandle_t *clientHandle, char *userName, char *password)
{
  open_error_t result;
  open_buffdesc userBuffDesc;
  open_buffdesc passwordBuffDesc;
  uint32_t userNameMax;
  uint32_t passwordMin;
  uint32_t passwordMax;
  char *userNameStr;
  char *passwordStr;

  if ((result = openapiUserNameSizeGet(clientHandle, &userNameMax)) != OPEN_E_NONE)
  {
    printf("Bad return code while getting the username max. (result = %d)\n", result);
    return;
  }

  if ((result = openapiPasswordSizeGet(clientHandle, &passwordMin, &passwordMax)) != OPEN_E_NONE)
  {
    printf("Bad return code while getting the password max. (result = %d)\n", result);
    return;
  }

  if ((userNameStr = (char*)malloc(userNameMax)) == NULL)
  {
    printf("Could not allocate memory for user name.\n");
    return;
  }

  if ((passwordStr = (char*)malloc(passwordMax)) == NULL)
  {
    printf("Could not allocate memory for password.\n");
    free(userNameStr);
    return;
  }

  memset(userNameStr, 0, userNameMax);
  strncpy(userNameStr, userName, userNameMax-1);
  userBuffDesc.pstart = userNameStr;
  userBuffDesc.size = userNameMax-1;

  memset(passwordStr, 0, passwordMax);
  strncpy(passwordStr, password, passwordMax-1);
  passwordBuffDesc.pstart = passwordStr;
  passwordBuffDesc.size = passwordMax-1;

  if ((result = openapiUserSet(clientHandle, &userBuffDesc, &passwordBuffDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set username and password. (result = %d)\n", result);
  }
  else
  {
    printf("Username and password are configured successfully\n");
  }

  free(userNameStr);
  free(passwordStr);

  return;
}

/*********************************************************************
* @purpose  Set privilege level to a user.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    userName        @b{(input)}   User name to configure.
* @param    level           @b{(input)}   User privilege level
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void userLevelSet(openapiClientHandle_t *clientHandle, char *userName, OPEN_LOGIN_ACCESS_LEVELS_t level)
{
  open_error_t result;
  open_buffdesc userBuffDesc;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, userName, (sizeof(str) - 1));
  userBuffDesc.pstart = str;
  userBuffDesc.size = strlen(str)+1;

  if ((result = openapiUserLevelSet(clientHandle, &userBuffDesc, level)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set access level to user. (result = %d)\n", result);
  }
  else
  {
    printf("Access level is set successfully to the user\n");
  }

  return;
}

/*********************************************************************
* @purpose  Get the user information
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void userInfoGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc userBuffDesc;
  uint32_t userNameSize;
  uint32_t maxUsers;
  uint32_t level;
  char *str;


  printf("Getting maximum no. of users supported\n");
  if ((result = openapiMaxUsersSupportedGet(clientHandle, &maxUsers)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get maximum number of users supported. (result = %d)\n", result);
  }
  else
  {
    printf("  Maximum no. of users: %d\n", maxUsers);
  }

  if ((result = openapiUserNameSizeGet(clientHandle, &userNameSize)) != OPEN_E_NONE)
  {
    printf("Bad return code while getting the username size. (result = %d)\n", result);
    return;
  }

  if ((str = (char*)malloc(userNameSize)) == NULL)
  {
    printf("Could not allocate memory.\n");
    return;
  }

  memset(str, 0, userNameSize);
  userBuffDesc.pstart = str;
  userBuffDesc.size = userNameSize;

  printf("Printing the user name and level:\n");
  if ((result = openapiUserFirstGet(clientHandle, &userBuffDesc)) != OPEN_E_NONE)
  {
    if (result == OPEN_E_NOT_FOUND)
    {
      printf("  No users found. (result = %d)\n", result);
    }
    else
    {
      printf("  Bad return code while getting first username. (result = %d)\n", result);
    }
    free(str);
    return;
  }
  do
  {
    printf("  Username: %s", str);
    if (openapiUserLevelGet(clientHandle, &userBuffDesc, &level) == OPEN_E_NONE)
    {
      printf("  Level: %d\n", level); 
    }
    userBuffDesc.size = userNameSize;
  }while((result = openapiUserNextGet(clientHandle, &userBuffDesc, &userBuffDesc)) == OPEN_E_NONE);

  free(str);
  return;
}

/*********************************************************************
* @purpose  Delete user.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    userName        @b{(input)}   User name to configure.
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void userDelete(openapiClientHandle_t *clientHandle, char *userName)
{
  open_error_t result;
  open_buffdesc userBuffDesc;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, userName, (sizeof(str) - 1));
  userBuffDesc.pstart = str;
  userBuffDesc.size = strlen(str)+1;

  if ((result = openapiUserDelete(clientHandle, &userBuffDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete username. (result = %d)\n", result);
  }
  else
  {
    printf("User deleted successfully\n");
  }

  return;
}

/*********************************************************************
* @purpose  Set enable password.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    password        @b{(input)}   Enable password
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void enablePasswordSet(openapiClientHandle_t *clientHandle, char *password)
{
  open_error_t result;
  open_buffdesc pswdBufDesc;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, password, (sizeof(str) - 1));
  pswdBufDesc.pstart = str;
  pswdBufDesc.size = strlen(str)+1;

  if ((result = openapiEnablePasswordSet(clientHandle, &pswdBufDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set enable password. (result = %d)\n", result);
  }
  else
  {
    printf("Enable password set successfully. \n");
  }

  return;
}

/*********************************************************************
* @purpose  Remove enable password.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void enablePasswordRemove(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  if ((result = openapiEnablePasswordRemove(clientHandle)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to remove enable password. (result = %d)\n", result);
  }
  else
  {
    printf("Enable password removed successfully. \n");
  }

  return;
}

/*********************************************************************
* @purpose  Authenticate username and password
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    userName        @b{(input)}   Username to validate
* @param    password        @b{(input)}   Password to Validate
*
* @returns  none
* 
* @notes 
* 
* @end
*********************************************************************/
void authenticateUser(openapiClientHandle_t *clientHandle, char *userName, char *password)
{
  open_error_t result;
  OPEN_LOGIN_ACCESS_LEVELS_t accessLevel;
  open_buffdesc userBuffDesc;
  open_buffdesc pswdBuffDesc;
  char str1[100];
  char str2[100];

  memset(str1, 0, sizeof(str1));
  strncpy(str1, userName, (sizeof(str1) - 1));
  userBuffDesc.pstart = str1;
  userBuffDesc.size = strlen(str1)+1;

  memset(str2, 0, sizeof(str2));
  strncpy(str2, password, (sizeof(str2) - 1));
  pswdBuffDesc.pstart = str2;
  pswdBuffDesc.size = strlen(str2)+1;

  if ((result = openapiAuthenticateUser(clientHandle, &userBuffDesc, &pswdBuffDesc,
                                        &accessLevel)) != OPEN_E_NONE)  
  {
    printf("Bad return code trying to authenticate user. (result = %d)\n", result);
  }
  else
  {
    printf("User authenticated successfully. (result = %d)\n", result); 
    printf("Privilege level of the user: %d \n", accessLevel);
  }

  return;
}

/*********************************************************************
* @purpose  User accounts OpEN API sanity.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void userAccountsOpENAPIsTestSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t userNameSize = 0, passwordMinSize = 0, passwordMaxSize = 0;
  uint32_t maxUsers = 0;
  uint32_t level = 0;
  uint32_t minPswdLen = 0, maxPswdLen = 0;
  char *username, *password;
  open_buffdesc userBuffDesc;
  open_buffdesc pswdBuffDesc; 

  printf("Testing User Accounts OpEN APIs sanity:\n");

  if ((result = openapiUserNameSizeGet(clientHandle, &userNameSize)) != OPEN_E_NONE)
  {
    printf("Bad return code while getting the username size. (result = %d)\n", result);
    return;
  }

  if ((result = openapiPasswordSizeGet(clientHandle, &passwordMinSize, &passwordMaxSize)) != OPEN_E_NONE)
  {
    printf("Bad return code while getting the password size. (result = %d)\n", result);
    return;
  }

  if ((username = (char *)malloc(userNameSize)) == NULL)
  {
    printf("Could not allocate memory\n");
    return;
  }
  memset(username, 0, sizeof(userNameSize));
  userBuffDesc.pstart = username;
  userBuffDesc.size = userNameSize; 

  if ((password = (char *)malloc(passwordMaxSize)) == NULL)
  {
    printf("Could not allocate memory\n");
    free(username);
    return;
  }
  memset(password, 0, passwordMaxSize);
  pswdBuffDesc.pstart = password;
  pswdBuffDesc.size = passwordMaxSize;
  

  /* openapiUserNameSizeGet() */
  printf("\nTesting openapiUserNameSizeGet(): \n");
  result = openapiUserNameSizeGet(NULL, &userNameSize);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiUserNameSizeGet(clientHandle, NULL);
  printf("NULL username size. (result = %d)\n", result);

  printf("openapiUserNameSizeGet() sanity successful.\n");

  /* openapiPasswordSizeGet() */
  printf("\nTesting openapiPasswordSizeGet(): \n");

  result = openapiPasswordSizeGet(NULL, &passwordMinSize, &passwordMaxSize);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPasswordSizeGet(clientHandle, NULL, &passwordMaxSize);
  printf("NULL minimum password length. (result = %d)\n", result);

  result = openapiPasswordSizeGet(clientHandle, &passwordMinSize, NULL);
  printf("NULL maximum password length. (result = %d)\n", result);

  printf("openapiPasswordSizeGet() sanity successful.\n");
   
  /* openapiUserSet() */
  printf("\nTesting openapiUserSet(): \n");

  result = openapiUserSet(NULL, &userBuffDesc, &pswdBuffDesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiUserSet(clientHandle, NULL, &pswdBuffDesc);
  printf("NULL buff descriptor to user name. (result = %d)\n", result);

  result = openapiUserSet(clientHandle, &userBuffDesc, NULL);\
  printf("NULL buff descriptor to password. (result = %d)\n", result);

  printf("openapiUserSet() sanity successful.\n");

  /* openapiUserFirstGet() */
  printf("\nTesting openapiUserFirstGet() \n");
  result = openapiUserFirstGet(NULL, &userBuffDesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiUserFirstGet(clientHandle, NULL);
  printf("NULL buff descriptor to user name. (result = %d)\n", result);

  printf("openapiUserFirstGet() sanity successful.\n");


  /* openapiUserNextGet() */
  printf("\nTesting openapiUserNextGet(): \n");

  result = openapiUserNextGet(NULL, &userBuffDesc, &userBuffDesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiUserNextGet(clientHandle, NULL, &userBuffDesc);
  printf("NULL buff descriptor to previous user name. (result = %d)\n", result);

  result = openapiUserNextGet(clientHandle, &userBuffDesc, NULL);
  printf("NULL buff descriptor to next user name. (result = %d)\n", result);

  printf("openapiUserNextGet() sanity successful. \n");
  

  /* openapiMaxUsersSupportedGet() */
  printf("\nTesting openapiMaxUsersSupportedGet(): \n");
  result = openapiMaxUsersSupportedGet(NULL, &maxUsers);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiMaxUsersSupportedGet(clientHandle, NULL);
  printf("NULL parameter to maximum users. (result = %d)\n", result);

  printf("openapiMaxUsersSupportedGet() sanity successful.\n");

  /* openapiUserLevelSet() */
  printf("\nTesting openapiUserLevelSet(): \n");
  result = openapiUserLevelSet(NULL, &userBuffDesc, level);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiUserLevelSet(clientHandle, NULL, level);
  printf("NULL buff descriptor to user name. (result = %d)\n", result);

  result = openapiUserLevelSet(clientHandle, &userBuffDesc, 20);
  printf("Invalid user level. (result = %d)\n", result);

  printf("openapiUserLevelSet() sanity successful. \n");

  /* openapiUserLevelGet() */ 
  printf("\nTesting openapiUserLevelGet(): \n");

  result = openapiUserLevelGet(NULL, &userBuffDesc, &level);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiUserLevelGet(clientHandle, NULL, &level);
  printf("NULL buff descriptor to user name. (result = %d)\n", result);

  result = openapiUserLevelGet(clientHandle, &userBuffDesc, NULL);
  printf("NULL user level. (result = %d)\n", result);

  printf("openapiUserLevelGet() sanity successful. \n");

  /* openapiUserNameValidate() */
  printf("\nTesting openapiUserNameValidate(): \n");

  result = openapiUserNameValidate(NULL, &userBuffDesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiUserNameValidate(clientHandle, NULL);
  printf("NULL buff descriptor to user name. (result = %d)\n", result);

  printf("openapiUserNameValidate() sanity successful. \n");

  /* openapiPasswordSizeGet() */
  printf("\nTesting openapiPasswordSizeGet(): \n");

  result = openapiPasswordSizeGet(NULL, &minPswdLen, &maxPswdLen);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiPasswordSizeGet(clientHandle, NULL, &maxPswdLen);
  printf("NULL parameter to passowrd minimum length. (result = %d)\n", result);

  result = openapiPasswordSizeGet(clientHandle, &minPswdLen, NULL);
  printf("NULL parameter to passowrd maximum length. (result = %d)\n", result);

  printf("openapiPasswordSizeGet() sanity successful. \n");

  /* openapiUserDelete() */
  printf("\nTesting openapiUserDelete(): \n");

  result = openapiUserDelete(NULL, &userBuffDesc);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiUserDelete(clientHandle, NULL);
  printf("NULL buff descriptor to user name. (result = %d)\n", result);

  printf("openapiUserDelete() sanity successful. \n");

  /* openapiAuthenticateUser() */
  printf("\nTesting openapiAuthenticateUser(): \n");
  result = openapiAuthenticateUser(NULL, &userBuffDesc, &pswdBuffDesc, &level);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAuthenticateUser(clientHandle, NULL, &pswdBuffDesc, &level);
  printf("NULL buff descriptor to user name. (result = %d)\n", result);

  result = openapiAuthenticateUser(clientHandle, &userBuffDesc, NULL, &level);
  printf("NULL buff descriptor to password. (result = %d)\n", result);

  result = openapiAuthenticateUser(clientHandle, &userBuffDesc, &pswdBuffDesc, NULL);
  printf("NULL parameter to access level. (result = %d)\n", result);

  printf("openapiAuthenticateUser() sanity successful. \n");
 

  free(username);
  free(password);
  return;
}

/*********************************************************************
* @purpose  Search the next valid ID of session
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void sessionIDNextValidGetTest(openapiClientHandle_t *clientHandle, uint32_t sessionType, uint32_t sessionID)
{
  open_error_t result;
  OPEN_LOGIN_TYPE_t sessionLoginType = (OPEN_LOGIN_TYPE_t)sessionType;
  uint32_t SID = sessionID;
  char buf[256];

  switch (sessionLoginType)
  {
    case OPEN_LOGIN_TYPE_SERIAL:
      sprintf(buf, "%s", "SERIAL");
      break;
    case OPEN_LOGIN_TYPE_TELNET:
      sprintf(buf, "%s", "TELNET");
      break;
    case OPEN_LOGIN_TYPE_SSH:
      sprintf(buf, "%s", "SSH");
      break;
    case OPEN_LOGIN_TYPE_HTTP:
      sprintf(buf, "%s", "HTTP");
      break;
    case OPEN_LOGIN_TYPE_HTTPS:
      sprintf(buf, "%s", "HTTPS");
      break;
    default:
      sprintf(buf, "%s", "UNKNOWN");
      break;
  }

  do
  {
    result = openapiSessionIDNextValidGet(clientHandle, sessionLoginType, &SID);
    switch (result)
    {
      case OPEN_E_NONE:
        printf("The next ID of %s session is returned successfully. (ID = %u, result = %d)\n", buf, SID, result);
        break;
      case OPEN_E_NOT_FOUND:
        printf("The valid next ID of %s session wasn't found. (result = %d)\n", buf, result);
        break;
      case OPEN_E_PARAM:
        printf("Invalid parameter is passed. (result = %d)\n", result);
        break;
      default:
        break;
    }
  } while (result == OPEN_E_NONE);
}

/*********************************************************************
* @purpose  Closes the session by ID
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void sessionLogoutTest(openapiClientHandle_t *clientHandle, uint32_t sessionID)
{
  open_error_t result;

  result = openapiSessionLogout(clientHandle, sessionID);
  switch (result)
  {
    case OPEN_E_NONE:
      printf("The logout of session ID:%u was successful. (result = %d)\n", sessionID, result);
      break;
    case OPEN_E_FAIL:
      printf("The logout of session ID:%u failed. (result = %d)\n", sessionID, result);
      break;
    case OPEN_E_PARAM:
      printf("Invalid parameter is passed. (result = %d)\n", result);
      break;
    default:
      break;
  }
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum, arg1, arg2;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];

  if (argc < 2)
  {
    printUserAccountsAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);
  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("user_accounts_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting User Accounts API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
     if (argc != 4)
     {
       printUserAccountsAppMenu();
       exit(1);
     }
     userNamePasswordSet(&clientHandle, argv[2], argv[3]); 
     break;

    case 2:
      if (argc != 4)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      userLevelSet(&clientHandle, argv[2], arg1);
      break;

    case 3:
      if (argc != 2)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      userInfoGet(&clientHandle);
      break;

    case 4:
      if (argc != 3)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      userDelete(&clientHandle, argv[2]);
      break;

    case 5:
      if (argc != 3)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      enablePasswordSet(&clientHandle, argv[2]);  
      break;

    case 6:
      if (argc != 2)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      enablePasswordRemove(&clientHandle);
      break;

    case 7:
      if (argc != 4)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      authenticateUser(&clientHandle, argv[2], argv[3]);
      break;

    case 8:
      if (argc != 2)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      userAccountsOpENAPIsTestSanity(&clientHandle);
      break;

    case 9:
      if (argc != 4)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      sessionIDNextValidGetTest(&clientHandle, arg1, arg2);
      break;

    case 10:
      if (argc != 3)
      {
        printUserAccountsAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      sessionLogoutTest(&clientHandle, arg1);
      break;

    default:
       printUserAccountsAppMenu();
       break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping User Accounts API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}

