/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  vlan_example.c
*
* @purpose   VLAN APIs Example.
*
* @component OPEN
*
* @comments
*
* @create    09/04/2012
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_vlan.h"

#define VLAN_ID_MIN 1
#define VLAN_ID_MAX 1024
#define IF_PRIORITY 3

/* 
   Some OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/
#define RETRY_COUNT 10
#define RETRY_INTERVAL_MSEC 100000

/*******************************************************************
*
* @brief  This function demonstrates the usage of OpEN APIs that
*         create and verify a created VLAN. If a VLAN is created
*         successfully it also configures and then gets the 
*         name configured for the VLAN.
*
* @param    clientHandle  @b{(input)} client handle from registration API
* @param    vlanId        @b{(input)} VLAN Id to be created.
*
* @returns  OPEN_E_NONE        The VLAN was successfully created, name configured and verified.
* @returns  OPEN_E_FAIL        The VLAN was not successfully created or verified or 
*                              VLAN name could not be configured or verified.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
*********************************************************************/
open_error_t vlanCreateAndVerify(openapiClientHandle_t *clientHandle, uint32_t vlanId)
{
  open_error_t result;
  uint32_t i;
  uint32_t vlanNameLen;
  open_buffdesc vlanNameDesc;
  char *vlanName = NULL;

  if ((result = openapiVlanCreate(clientHandle, vlanId)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to create VLAN %u. (result = %d)\n", vlanId, result);
  }
  else
  {
    for (i=0; i < RETRY_COUNT; i++)
    {
      if ((result = openapiVlanCreatedCheck(clientHandle, vlanId)) == OPEN_E_NONE)
      {
        /* got what we expected, move on */
        break;
      }
      else
      {
        usleep(RETRY_INTERVAL_MSEC);
      }
    }

    if (result == OPEN_E_NONE)
    {
      if ((result = openapiVlanNameLenGet(clientHandle, &vlanNameLen)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to get VLAN Name Length. (result = %d)\n", result);
      }
      else
      {
        vlanName = (char*)malloc(vlanNameLen);
      
        if (vlanName == NULL)
        {
          printf("  Could not allocate memory for VLAN name.\n");
          result = OPEN_E_FAIL;
        }
        else
        {
          memset(vlanName, 0, vlanNameLen);
          sprintf(vlanName, "vlanName%u", vlanId);
          vlanNameDesc.pstart = vlanName;
          vlanNameDesc.size = strlen(vlanName) + 1;
        
          if ((result = openapiVlanNameSet(clientHandle, vlanId, &vlanNameDesc)) != OPEN_E_NONE)
          {
            printf("  Bad return code trying to set VLAN %u name. (result = %d)\n", vlanId, result);
          }
          else
          {
            memset(vlanName, 0, vlanNameLen);
            vlanNameDesc.size = vlanNameLen;
            if ((result = openapiVlanNameGet(clientHandle, vlanId, &vlanNameDesc)) != OPEN_E_NONE)
            {
              printf("  Bad return code trying to get VLAN %u name. (result = %d)\n", vlanId, result);
            }
            else
            {
              printf("  VLAN %u name = %s\n", vlanId, vlanName);
            }
          }
        }
      }
    }
    else
    {
      printf("  Could not verify that VLAN %u is created. (result = %d)\n", vlanId, result);
    }
  }

  if (vlanName)
  {
    free(vlanName);
  }
  
  return(result);
}

/*******************************************************************
*
* @brief  This function demonstrates the usage of OpEN APIs that
*         delete and verify a created VLAN.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    vlanId         @b{(input)} VLAN Id to be deleted.
*
* @returns  OPEN_E_NONE        The VLAN was successfully deleted and verified.
* @returns  OPEN_E_FAIL        The VLAN was not successfully deleted or verified.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
*********************************************************************/
open_error_t vlanDeleteAndVerify(openapiClientHandle_t *clientHandle, uint32_t vlanId)
{
  open_error_t result;
  uint32_t i;

  if ((result = openapiVlanDelete(clientHandle, vlanId)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to create VLAN %u. (result = %d)\n", vlanId, result);
  }
  else
  {
    for (i=0; i < RETRY_COUNT; i++)
    {
      if ((result = openapiVlanCreatedCheck(clientHandle, vlanId)) != OPEN_E_NONE)
      {
        /* got what we expected, move on */
        break;
      }
      else
      {
        usleep(RETRY_INTERVAL_MSEC);
      }
    }
  }

  /* testing for a negative, adjust return code */
  if (result == OPEN_E_FAIL)
  {
    result = OPEN_E_NONE;
  }

  return(result);
}

/*******************************************************************
*
* @brief  This function demonstrates the usage of OpEN APIs that
*         configure and verify the Port Acceptable Frame Type for an interface.
*
* @param    clientHandle  @b{(input)} client handle from registration API
* @param    ifNum         @b{(input)} switch interface.
* @param    type          @b{(input)} OPEN_DOT1Q_ACCEPT_FRAME_TYPE_t
*
* @returns  OPEN_E_NONE        The type was successfully configured and verified.
* @returns  OPEN_E_FAIL        The type was not successfully configured or verified.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
*********************************************************************/
open_error_t interfaceAcceptFrameTypeSetAndVerify(openapiClientHandle_t *clientHandle,
                                                  uint32_t ifNum,
                                                  OPEN_DOT1Q_ACCEPT_FRAME_TYPE_t type)
{
  open_error_t result;
  uint32_t newType;
  uint32_t i;

  if ((result = openapiVlanIfAcceptFrameTypeSet(clientHandle, ifNum, type)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to set Accept Frame Type. (result = %d)\n", result);
  }
  else
  {
    for (i=0; i < RETRY_COUNT; i++)
    {
      if ((result = openapiVlanIfAcceptFrameTypeGet(clientHandle, ifNum, &newType)) != OPEN_E_NONE)
      {
        /* error from API, move on */
        printf("  Bad return code trying to get Accept Frame Type. (result = %d)\n", result);
        break;
      }
      else
      {
        if (newType != type)
        {
          usleep(RETRY_INTERVAL_MSEC);
        }
        else
        {
          /* got what we want, move on */
          break;
        }
      }
    }

    if (newType != type)
    {
      result = OPEN_E_FAIL;
    }
  }

  return(result);
}

/*******************************************************************
*
* @brief  This function demonstrates the usage of OpEN APIs that
*         configure and verify the Port VLAN ID for an interface.
*
* @param    clientHandle  @b{(input)} client handle from registration API
* @param    ifNum          @b{(input)} switch interface.
* @param    vlanId         @b{(input)} PVID.
*
* @returns  OPEN_E_NONE        The PVID was successfully configured and verified.
* @returns  OPEN_E_FAIL        The PVID was not successfully configured or verified.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Setting PVID not valid for the specified interface.
* @returns  OPEN_E_NOT_FOUND   The VLAN specified is not created.
*
*********************************************************************/
open_error_t interfacePvidSetAndVerify(openapiClientHandle_t *clientHandle, uint32_t ifNum, uint32_t vlanId)
{
  open_error_t result;
  uint32_t newVlanId;
  uint32_t i;

  if ((result = openapiVlanIfPvidSet(clientHandle, ifNum, vlanId)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to set PVID. (result = %d)\n", result);
  }
  else
  {
    for (i=0; i < RETRY_COUNT; i++)
    {
      if ((result = openapiVlanIfPvidGet(clientHandle, ifNum, &newVlanId)) != OPEN_E_NONE)
      {
        /* error from API, move on */
        printf("  Bad return code trying to get PVID. (result = %d)\n", result);
        break;
      }
      else
      {
        if (newVlanId != vlanId)
        {
          usleep(RETRY_INTERVAL_MSEC);
        }
        else
        {
          /* got what we want, move on */
          break;
        }
      }
    }

    if (newVlanId != vlanId)
    {
      result = OPEN_E_FAIL;
    }
  }

  return(result);
}

/*******************************************************************
*
* @brief  This function demonstrates the usage of OpEN APIs that
*         configure and verify the VLAN participation for an interface.
*
* @param    clientHandle  @b{(input)} client handle from registration API
* @param    ifNum          @b{(input)} switch interface.
* @param    vlanId         @b{(input)} PVID.
* @param    partStatus     @b{(input)} Port VLAN Participation Mode.
*
* @returns  OPEN_E_NONE        The participation mode was successfully configured and verified.
* @returns  OPEN_E_FAIL        The participation mode was not successfully configured or verified.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Setting participation mode not valid for the specified interface.
* @returns  OPEN_E_NOT_FOUND   The VLAN specified is not created.
*
*********************************************************************/
open_error_t interfaceVlanParticipationSetAndVerify(openapiClientHandle_t *clientHandle, uint32_t ifNum, 
                                                    uint32_t vlanId, OPEN_VLAN_PARTICIPATION_MODE_t partStatus)
{
  open_error_t result;
  OPEN_VLAN_PARTICIPATION_MODE_t newPartStatus;
  uint32_t i;

  if ((result = openapiVlanIfParticipationSet(clientHandle, vlanId, ifNum, partStatus)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to set VLAN participation configuration. (result = %d, ifNum = %u, partStatus = %u)\n", result, ifNum, partStatus);
  }
  else
  {
    for (i=0; i < RETRY_COUNT; i++)
    {
      if ((result = openapiVlanIfParticipationGet(clientHandle, vlanId, ifNum, &newPartStatus)) != OPEN_E_NONE)
      {
        /* error from API, move on */
        printf("  Bad return code trying to get VLAN participation configuration. (result = %d, ifNum = %u, partStatus = %u)\n", result, ifNum, partStatus);
        break;
      }
      else
      {
        if (newPartStatus != partStatus)
        {
          usleep(RETRY_INTERVAL_MSEC);
        }
        else
        {
          /* got what we want, move on */
          break;
        }
      }
    }

    if (newPartStatus != partStatus)
    {
      result = OPEN_E_FAIL;
    }
  }
  return(result);
}

/*******************************************************************
*
* @brief  This function demonstrates the usage of OpEN APIs that
*         configure and verify the VLAN tagging for an interface.
*
* @param    clientHandle  @b{(input)} client handle from registration API
* @param    ifNum          @b{(input)} switch interface.
* @param    vlanId         @b{(input)} PVID.
* @param    taggingMode    @b{(input)} Port VLAN Tagging Mode.
*
* @returns  OPEN_E_NONE        The VLAN tagging was successfully configured and verified.
* @returns  OPEN_E_FAIL        The VLAN tagging was not successfully configured or verified.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Setting VLAN tagging not valid for the specified interface.
* @returns  OPEN_E_NOT_FOUND   The VLAN specified is not created.
*
*********************************************************************/
open_error_t interfaceVlanTaggingSetAndVerify(openapiClientHandle_t *clientHandle, uint32_t ifNum, 
                                              uint32_t vlanId, OPEN_CONTROL_t taggingMode)
{
  open_error_t result;
  OPEN_CONTROL_t newTaggingMode;
  uint32_t i;

  if ((result = openapiVlanIfTaggingSet(clientHandle, ifNum, vlanId, taggingMode)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to set VLAN tagging configuration. (result = %d, ifNum = %u, taggingMode = %u)\n", result, ifNum, taggingMode);
  }
  else
  {
    for (i=0; i < RETRY_COUNT; i++)
    {
      if ((result = openapiVlanIfTaggingGet(clientHandle, ifNum, vlanId, &newTaggingMode)) != OPEN_E_NONE)
      {
        /* error from API, move on */
        printf("  Bad return code trying to get VLAN tagging configuration. (result = %d, ifNum = %u, taggingMode = %u)\n", result, ifNum, taggingMode);
        break;
      }
      else
      {
        if (newTaggingMode != taggingMode)
        {
          usleep(RETRY_INTERVAL_MSEC);
        }
        else
        {
          /* got what we want, move on */
          break;
        }
      }
    }

    if (newTaggingMode != taggingMode)
    {
      result = OPEN_E_FAIL;
    }
  }
  return(result);
}

/*******************************************************************
*
* @brief  This function demonstrates the usage of OpEN APIs that
*         configure and verify the user priority for an interface.
*
* @param    clientHandle  @b{(input)} client handle from registration API
* @param    ifNum         @b{(input)} switch interface.
* @param    priority      @b{(input)} priority
*
* @returns  OPEN_E_NONE        The default priority was successfully configured and verified.
* @returns  OPEN_E_FAIL        The default priority was not successfully configured or verified.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
*********************************************************************/
open_error_t interfacePrioritySetAndVerify(openapiClientHandle_t *clientHandle, uint32_t ifNum,
                                           uint32_t priority)
{
  open_error_t result;
  uint32_t i;
  uint32_t numTrafficClasses;
  uint32_t newPriority;

  if ((result = openapiVlanIfNumTrafficClassesGet(clientHandle, ifNum, &numTrafficClasses)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to the number of traffic classes supported. (result = %d, ifNum = %u)\n", result, ifNum);
    return OPEN_E_FAIL;
  }

  if (priority > numTrafficClasses)
  {
    printf("  Parameter value for priority out of accepted range. (priority = %u, numTrafficClasses = %u)\n", priority, numTrafficClasses);
    return OPEN_E_PARAM;
  }

  if ((result = openapiVlanIfDefaultUserPrioritySet(clientHandle, ifNum, priority)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to set user priority. (result = %d, ifNum = %u, priority = %u)\n", result, ifNum, priority);
  }
  else
  {
    for (i=0; i < RETRY_COUNT; i++)
    {
      if ((result = openapiVlanIfDefaultUserPriorityGet(clientHandle, ifNum, &newPriority)) != OPEN_E_NONE)
      {
        /* error from API, move on */
        printf("  Bad return code trying to get user priority. (result = %d, ifNum = %u)\n", result, ifNum);
        break;
      }
      else
      {
        if (newPriority != priority)
        {
          usleep(RETRY_INTERVAL_MSEC);
        }
        else
        {
          /* got what we want, move on */
          break;
        }
      }
    }

    if (newPriority != priority)
    {
      result = OPEN_E_FAIL;
    }
  }
  return(result);
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for VLAN.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  uint32_t ifNum, i;
  open_error_t result;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  uint32_t partStatus;
  uint32_t vlanId, nextVlanId;
  uint32_t vlanFlag = 0;

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("vlan_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting VLAN API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");
  /* exercise various OPEN API VLAN functions */
  do
  {
    /* find a VLAN ID that is not being used by the user configuration */
    for (vlanId = VLAN_ID_MIN; vlanId < VLAN_ID_MAX; vlanId++)
    {
      if (openapiVlanCreatedCheck(&clientHandle, vlanId) != OPEN_E_NONE)
      {
        /* found a VLAN that is not already configured */
        break;
      }
    }
    if (vlanId >= VLAN_ID_MAX)
    {
      printf("  Could not find an unused VLAN to configure.\n");
      break;
    }

    printf("Using VLAN ID %u.\n", vlanId);

    printf("\n");
    /* create and delete VLAN */
    printf("Testing VLAN create/delete/check APIs.\n");
    printf("Attempting to create VLAN %u.\n", vlanId);
    if (vlanCreateAndVerify(&clientHandle, vlanId) != OPEN_E_NONE)
    {
      printf("  Failure creating VLAN %u.\n", vlanId);
      break;
    }
    printf("  Successfully created VLAN %u.\n", vlanId);

    printf("Attempting to delete VLAN %u.\n", vlanId);
    if (vlanDeleteAndVerify(&clientHandle, vlanId) != OPEN_E_NONE)
    {
      printf("  Failure deleting VLAN %u.\n", vlanId);
      break;
    }
    printf("  Successfully deleted VLAN %u.\n", vlanId);

    /* program the PVID on interfaces */
    printf("\n");
    printf("Testing PVID set/get APIs.\n");
    printf("Attempting to create VLAN %u.\n", vlanId);
    if (vlanCreateAndVerify(&clientHandle, vlanId) != OPEN_E_NONE)
    {
      printf("  Failure creating VLAN %u.\n", vlanId);
      break;
    }
    printf("  Successfully created VLAN %u.\n", vlanId);

    printf("Attempting to set the first physical interface to participate in VLAN %u.\n", vlanId);
    if ((result = openapiIfFirstGet(&clientHandle, OPEN_INTF_TYPE_PHY, &ifNum)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get first physical interface. (result = %d)\n", result);
      break;
    }
    printf("Using physical interface with ifNum = %u.\n", ifNum);

    /* program the Frame Type */
    printf("\n");
    printf("Attempting to set Acceptable Frame Type for ifNum %u to type %d.\n", ifNum, OPEN_DOT1Q_ADMIN_ONLY_VLAN_UNTAGGED);
    if (interfaceAcceptFrameTypeSetAndVerify(&clientHandle, ifNum, OPEN_DOT1Q_ADMIN_ONLY_VLAN_UNTAGGED) != OPEN_E_NONE)
    {
      printf("  Failure setting Acceptable Frame Type for ifNum %u.\n", ifNum);
      break;
    }
    printf("  Successfully set Acceptable Frame Type for ifNum %u.\n", ifNum);

    /* program the PV ID */
    printf("\n");
    printf("Attempting to set PVID for interface to %u.\n", vlanId);
    if (interfacePvidSetAndVerify(&clientHandle, ifNum, vlanId) != OPEN_E_NONE)
    {
      printf("  Failure setting PVID for ifNum %u to VLAN %u.\n", ifNum, vlanId);
      break;
    }
    printf("  Successfully set PVID for ifNum %u to VLAN %u.\n", ifNum, vlanId);

    /* program the participation mode on interfaces */
    printf("\n");
    printf("Attempting to set participation mode for interface %u on VLAN %u to INCLUDE.\n", ifNum, vlanId);
    if (interfaceVlanParticipationSetAndVerify(&clientHandle, ifNum, vlanId, OPEN_VLAN_PARTICIPATION_MODE_INCLUDE) != OPEN_E_NONE)
    {
      printf("  Failure setting VLAN participation mode for ifNum %u on VLAN %u.\n", ifNum, vlanId);
      break;
    }
    printf("  Successfully set VLAN participation mode for ifNum %u on VLAN %u.\n", ifNum, vlanId);

    printf("Attempting to get the VLAN participation status.\n");
    for (i=0; i < RETRY_COUNT; i++)
    {
      if ((result = openapiVlanIfParticipationStatusGet(&clientHandle, vlanId, ifNum, &partStatus)) != OPEN_E_NONE)
      {
        /* error from API, move on */
        break;
      }
      else
      {
        if (partStatus != OPEN_ENABLE)
        {
          usleep(RETRY_INTERVAL_MSEC);
        }
        else
        {
          /* got what we want, move on */
          break;
        }
      }
    }

    if (result != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get participation status. (result = %d)\n", result);
      break;
    }
    else
    {
      printf("  Successfully retrieved participation status for ifNum = %u. - ", ifNum);
      printf("This is %sas expected.\n", (partStatus == OPEN_ENABLE) ? "":"NOT ");
    }

    /* program the tagging mode on interfaces */
    printf("\n");
    printf("Attempting to set tagging mode for interface %u on VLAN %u to ENABLE.\n", ifNum, vlanId);
    if (interfaceVlanTaggingSetAndVerify(&clientHandle, ifNum, vlanId, OPEN_ENABLE) != OPEN_E_NONE)
    {
      printf("  Failure setting VLAN tagging mode for ifNum %u on VLAN %u.\n", ifNum, vlanId);
      break;
    }
    printf("  Successfully set VLAN tagging mode for ifNum %u on VLAN %u.\n", ifNum, vlanId);

    printf("Attempting to set tagging mode for interface %u on VLAN %u to DISABLE.\n", ifNum, vlanId);
    if (interfaceVlanTaggingSetAndVerify(&clientHandle, ifNum, vlanId, OPEN_DISABLE) != OPEN_E_NONE)
    {
      printf("  Failure setting VLAN tagging mode for ifNum %u on VLAN %u.\n", ifNum, vlanId);
      break;
    }
    printf("  Successfully set VLAN tagging mode for ifNum %u on VLAN %u.\n", ifNum, vlanId);

    /* clean up */
    printf("\n");
    printf("Attempting to delete VLAN %u.\n", vlanId);
    if (vlanDeleteAndVerify(&clientHandle, vlanId) != OPEN_E_NONE)
    {
      printf("  Failure deleting VLAN %u.\n", vlanId);
      break;
    }
    printf("  Successfully deleted VLAN %u.\n", vlanId);

    /* program the default user priority on interfaces */
    printf("\n");
    printf("Attempting to set default priority for interface %u to %u.\n", ifNum, IF_PRIORITY);
    if (interfacePrioritySetAndVerify(&clientHandle, ifNum, IF_PRIORITY) != OPEN_E_NONE)
    {
      printf("  Failure setting default priority for interface %u.\n", ifNum);
      break;
    }
    printf("  Successfully set default priority for interface %u.\n", ifNum);

  } while (0);

  /* Testing the next vlan get */
  /*Checking for the vlans ither than default vlan */
  printf("\n");
  printf("Testing vlan get next API.\n");
  printf("Attempting to check if there are any vlans.\n");
  for (vlanId = VLAN_ID_MIN+1; vlanId < VLAN_ID_MAX; vlanId++)
  {
    if (openapiVlanCreatedCheck(&clientHandle, vlanId) == OPEN_E_NONE)
    {
      /* found a VLAN that is already configured */
      vlanFlag = 1;
      break;
    }
  }

  if (!vlanFlag)
  {
    printf("  Creating vlans as there are no vlans other than default vlan.\n");
    for (vlanId = VLAN_ID_MIN+1; vlanId <= 10; vlanId++)
    {
      if ((result = vlanCreateAndVerify(&clientHandle, vlanId)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to create VLAN %u. (result = %d)\n", vlanId, result);
        break;
      }
    }
  }

  result = openapiVlanNextGet(&clientHandle, 0, &nextVlanId);

  if (result == OPEN_E_NONE)
  {
    printf("  Vlans present: ");
  }
  else
  {
    printf("  Failed to get next vlan.(result = %d)\n", result);
  }

  while (result == OPEN_E_NONE)
  {
    printf(" %u", nextVlanId);
    result = openapiVlanNextGet(&clientHandle, nextVlanId, &nextVlanId);
  }
  printf("\n\n");

  if (!vlanFlag)
  {
    printf("Attempting to delete created vlans.\n");
    nextVlanId = VLAN_ID_MIN;
    while (openapiVlanNextGet(&clientHandle, nextVlanId, &nextVlanId) == OPEN_E_NONE)
    {
      if ((result = openapiVlanDelete(&clientHandle, nextVlanId)) != OPEN_E_NONE)
      {
        printf(" Bad return code trying to delete VLAN %u. (result = %d)\n", nextVlanId, result);
        break;
      }
    }
    if (result == OPEN_E_NONE)
    {
      printf("  Successfully deleted the created vlans.\n");
    }
  }

  /* Log goodbye message with OPEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping VLAN API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
