/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_app_ctrl.h
*
* @purpose   Application Controls
*
* @component OpEN
*
* @create    11/23/2015
*
* @end
*                                    
**********************************************************************/
#ifndef OPENAPI_APP_CTRL_H_INCLUDED
#define OPENAPI_APP_CTRL_H_INCLUDED

#include "openapi_common.h"

/* max size for application status and version strings (including terminating NULL) */
#define OPEN_APP_CTRL_NAME_SIZE         17
#define OPEN_APP_CTRL_STATUS_SIZE       512
#define OPEN_APP_CTRL_VERSION_SIZE      20

/*********************************************************************
* @purpose  Creates a new Application Process Table entry for the
*           given file name.
*
* @param    client_handle   @b{(input)}   Client handle from
*                                         registration API
* @param    *appName         @b((input))  Application name
* @param    -                             Maximum length is (@ref OPENAPI_APP_NAME_SIZE - 1)
*
* @returns  OPEN_E_NONE     If successful.
* @returns  OPEN_E_PARAM    Invalid parameter is supplied.
* @returns  OPEN_E_ERROR    If failure.
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiAppCtrlEntryCreate(openapiClientHandle_t *client_handle,
                                       open_buffdesc         *appName);

/*********************************************************************
* @purpose  Deletes an Application Process Table entry for the
*           given file name.
*
* @param    client_handle   @b{(input)}   Client handle from
*                                         registration API
* @param    *appName         @b((input))  Application name
* @param    -                             Maximum length is (@ref OPENAPI_APP_NAME_SIZE - 1)
*
* @returns  OPEN_E_NONE     If successful.
* @returns  OPEN_E_PARAM    Invalid parameter is supplied.
* @returns  OPEN_E_ERROR    If failure.
*
* @notes    Calling this function will stop the applications process,
*           if it is running, before the entry is deleted.
*
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiAppCtrlEntryDelete(openapiClientHandle_t *client_handle,
                                       open_buffdesc         *appName);

/*********************************************************************
* @purpose  Gets start on boot, auto-restart, CPU sharing and memory
*           limit settings for the process entry.
*
* @param    client_handle   @b{(input)}   Client handle from
*                                         registration API
* @param    *appName         @b((input))  Application name
* @param    -                             Maximum length is (@ref OPENAPI_APP_NAME_SIZE - 1)
* @param    startOnBoot     @b{(output)}  Returns @ref OPEN_ENABLE if start on boot
*                                         enabled, else @ref OPEN_DISABLE if not
* @param    autoRestart     @b{(output)}  Returns @ref OPEN_ENABLE if auto-restart
*                                         enabled, else @ref OPEN_DISABLE if not
* @param    cpuShare        @b{(output)}  Pointer to CPU share percentage
* @param    maxMemory       @b{(output)}  Pointer to maximum memory in megabytes
*                                         the application is allowed to consume
*
* @returns  OPEN_E_NONE     If successful.
* @returns  OPEN_E_PARAM    Invalid parameter is supplied.
* @returns  OPEN_E_ERROR    If failure.
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiAppCtrlAttrGet(openapiClientHandle_t *client_handle,
                                   open_buffdesc         *appName,
                                   OPEN_CONTROL_t        *startOnBoot,
                                   OPEN_CONTROL_t        *autoRestart,
                                   uint32_t              *cpuShare,
                                   uint32_t              *maxMemory);

/*********************************************************************
* @purpose  Sets start on boot, auto-restart, CPU sharing and memory
*           limit settings for the process entry.
*
* @param    client_handle   @b{(input)}   Client handle from
*                                         registration API
* @param    *appName         @b((input))  Application name
* @param    -                             Maximum length is (@ref OPENAPI_APP_NAME_SIZE - 1)
* @param    startOnBoot     @b{(input)}   Start on boot
* @param    autoRestart     @b{(input)}   Auto restart
* @param    cpuShare        @b{(input)}   CPU share percentage
*                                         Maximum value @ref OPEN_MAX_CPU_SHARE
* @param    maxMemory       @b{(input)}   Maximum memory in megabytes the application
*                                         is allowed to consume
*                                         Maximum value @ref OPEN_MAX_MEMORY_LIMIT
*
* @returns  OPEN_E_NONE     If successful.
* @returns  OPEN_E_PARAM    Invalid parameter is supplied.
* @returns  OPEN_E_ERROR    If failure.
*
* @notes    When changing only some of these parameters, first call 
*           @ref openapiAppCtrlAttrGet to retrieve all of the values, then modify 
*           the ones being changed prior to setting the values.
* @notes    Alternate approach is to use NULL pointer for any parameter whose value is not 
*           being set.
*
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiAppCtrlAttrSet(openapiClientHandle_t *client_handle,
                                   open_buffdesc         *appName,
                                   OPEN_CONTROL_t        *startOnBoot,
                                   OPEN_CONTROL_t        *autoRestart,
                                   uint32_t              *cpuShare,
                                   uint32_t              *maxMemory);

/*********************************************************************
* @purpose  Starts the process with Process Manager using the parameters
*           configured in the process entry.
*
* @param    client_handle   @b{(input)}   Client handle from
*                                         registration API
* @param    *appName         @b((input))  Application name
* @param    -                             Maximum length is (@ref OPENAPI_APP_NAME_SIZE - 1)
*
* @returns  OPEN_E_NONE     If successful.
* @returns  OPEN_E_PARAM    Invalid parameter is supplied.
* @returns  OPEN_E_ERROR    If failure.
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiAppCtrlStart(openapiClientHandle_t *client_handle,
                                 open_buffdesc         *appName);

/*********************************************************************
* @purpose  Stops the process with Process Manager.
*
* @param    client_handle   @b{(input)}   Client handle from
*                                         registration API
* @param    *appName         @b((input))  Application name
* @param    -                             Maximum length is (@ref OPENAPI_APP_NAME_SIZE - 1)
*
* @returns  OPEN_E_NONE     If successful.
* @returns  OPEN_E_PARAM    Invalid parameter is supplied.
* @returns  OPEN_E_ERROR    If failure.
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiAppCtrlStop(openapiClientHandle_t *client_handle,
                                open_buffdesc         *appName);

/*********************************************************************
* @purpose  Get application operational status.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    *appName         @b((input))  Application name
* @param    -                             Maximum length is (@ref OPENAPI_APP_NAME_SIZE - 1)
* @param    *status          @b((output)) Application status buffer
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_APP_STATUS_SIZE bytes
*
* @returns  OPEN_E_NONE      if successful.
* @returns  OPEN_E_FAIL      if get operation has failed.
* @returns  OPEN_E_PARAM     if invalid parameter is supplied.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiAppCtrlStatusGet(openapiClientHandle_t *client_handle,
                                     open_buffdesc         *appName,
                                     open_buffdesc         *status);

/*********************************************************************
* @purpose  Set application operational status.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    *appName         @b((input))  Application name
* @param    -                             Maximum length is (@ref OPENAPI_APP_NAME_SIZE - 1)
* @param    *status          @b((input))  Application status buffer
*
* @returns  OPEN_E_NONE      if application status is set successfully.
* @returns  OPEN_E_FAIL      if application status contains non printable 
*                            characters.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   A user application uses this API to set a free-form string indicating
*          its current operational status. The caller is responsible for any
*          formatting of the displayed string (tabs, newlines, etc.).
* @notes   Calling this API does not change the running configuration of the switch.
*          The application status is set during run-time operation and does not
*          persist following a switch reset.
* @notes   At most, @ref OPENAPI_APP_STATUS_SIZE characters of the status string are 
*          stored and is truncated to fit (with null termination).
* 
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiAppCtrlStatusSet(openapiClientHandle_t *client_handle,
                                     open_buffdesc         *appName,
                                     open_buffdesc         *status);

/*********************************************************************
* @purpose  Get application version identifier.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    *appName         @b((input))  Application name
* @param    -                             Maximum length is (@ref OPENAPI_APP_NAME_SIZE - 1)
* @param    *version         @b((output)) Application version buffer
*           -                             Buffer allocated size must be at least
*           -                             @ref OPENAPI_APP_VERSION_SIZE bytes
*
* @returns  OPEN_E_NONE      if successful.
* @returns  OPEN_E_FAIL      if get operation has failed.
* @returns  OPEN_E_PARAM     if invalid parameter is supplied.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiAppCtrlVersionGet(openapiClientHandle_t *client_handle,
                                      open_buffdesc         *appName,
                                      open_buffdesc         *version);

/*********************************************************************
* @purpose  Set application version identifier.
*
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    *appName         @b((input))  Application name
* @param    -                             Maximum length is (@ref OPENAPI_APP_NAME_SIZE - 1)
* @param    *version         @b((input))  Application version buffer
*
* @returns  OPEN_E_NONE      if application version is set successfully.
* @returns  OPEN_E_FAIL      if application version contains non printable 
*                            characters.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   A user application uses this API to set a simple, free-form string identifying
*          its version number. The string typically consists of printable alphanumeric
*          characters with separator characters (space, dot, dash, slash, etc.), but
*          must not contain control characters (tab, newline, etc.). The caller is
*          responsible for providing a suitable string for display.
* @notes   Calling this API does not change the running configuration of the switch.
*          The application version is set during run-time operation and does not
*          persist following a switch reset.
* @notes   Maximum size for version string is @ref OPENAPI_APP_VERSION_SIZE characters
*          (including null termination).
* 
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiAppCtrlVersionSet(openapiClientHandle_t *client_handle,
                                      open_buffdesc         *appName,
                                      open_buffdesc         *version);

#endif /* OPENAPI_APP_CTRL_H_INCLUDED */

