/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_bgp_config.h
*
* @purpose   Border Gateway Protocol Configuration and Status 
*
* @component OPEN
*
* @create    06/25/2013
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_BGP_CONFIG_H_INCLUDED
#define OPENAPI_BGP_CONFIG_H_INCLUDED

#include <stdbool.h>
#include "openapi_common.h"

#define OPEN_BGP_NBR_MAX_PFX_NOLIMIT 0xFFFFFFFFUL /**< For neighbor maximum-prefixes command, indicates no limit. */
#define OPEN_BGP_ERR_STR_LEN 64                   /**< String length of last Received/Sent error notification message from peer */
#define OPEN_BGP_AFI_MAX 3                        /**< Maximum of BGP address family */
#define OPEN_BGP_MAX_DP_REASON_STR 16             /**< Maximum length of a reason (Why decision process was triggered) string */
#define OPEN_BGP_MAX_STATS_ENTRIES 20             /**< Number of decision process runs to keep in history */
#define OPEN_BGP_MAX_PASSWORD 25                  /**< Maximum length of neighbor password */
#define OPEN_BGP_MAX_TEMPLATE_NAME 32             /**< Maximum length of template name */
#define OPEN_BGP_PEER_DESC_LENGTH 80              /**< Maximum length of peer description */
#define OPEN_BGP_PEER_ADDR_STR_LENGTH 128         /**< Maximum length of peer address string */

/** Outbound update group ID that indicates a group is inactive or a group ID is unassigned */
#define OPEN_BGP_UPD_GROUP_NONE 0xFFFF

/** OSPF redistribution metric bits for match option */
typedef enum
{
  OPEN_OSPF_METRIC_TYPE_INTERNAL  = 1,      /**< Intra-Area */
  OPEN_OSPF_METRIC_TYPE_EXT1      = 2,      /**< External Type 1 */
  OPEN_OSPF_METRIC_TYPE_EXT2      = 4,      /**< External Type 2 */
  OPEN_OSPF_METRIC_TYPE_NSSA_EXT1 = 8,      /**< NSSA Type 1 */
  OPEN_OSPF_METRIC_TYPE_NSSA_EXT2 = 16      /**< NSSA Type 2 */
} OPEN_OSPF_METRIC_TYPES_t;

//! A peer can inherit parameter values from a peer template. When we query the
/// value of a configuration parameter for a peer, sometimes we want the value
/// configured on the peer itself, even if that is just the default value. In
/// other contexts, we want to get the actual value in use for the peer, whether
/// that is configured on the peer or inherited from a template. This enumeration
/// is used to indicate which value we want when we get a configuration parameter
/// for a peer.
typedef enum 
{
  OPEN_BGP_GET_FINAL = 0, /**< peer or inherited value from template */
  OPEN_BGP_GET_PEER  = 1  /**< peer value */
} OPEN_BGP_PARAM_TYPE;

/** Start or stop a BGP peer */
typedef enum
{
  OPEN_BGP_STOP  = 1, /**< Shutdown */
  OPEN_BGP_START = 2  /**< Startup */
} OPEN_BGP_PEER_STATE_t;

typedef enum
{
  OPEN_BGP_PEER_STATE_ILG           = 0, /**< ILG */
  OPEN_BGP_PEER_STATE_IDLE          = 1, /**< idle */
  OPEN_BGP_PEER_STATE_CONNECT       = 2, /**< connect */
  OPEN_BGP_PEER_STATE_ACTIVE        = 3, /**< active */
  OPEN_BGP_PEER_STATE_OPENSENT      = 4, /**< opensent */
  OPEN_BGP_PEER_STATE_OPENCONFIRM   = 5, /**< openconfirm */
  OPEN_BGP_PEER_STATE_ESTABLISHED   = 6  /**< established */
} OPEN_BGP4_PEER_STATE_t;

typedef enum
{
  OPEN_BGP_PEER_TYPE_DYNAMIC        = 0, /**< Dynamic */
  OPEN_BGP_PEER_TYPE_STATIC         = 1, /**< Static */
  OPEN_BGP_PEER_TYPE_UNSPECIFIED    = 2, /**< Unspecified or IPv6 */
} OPEN_BGP4_PEER_TYPE_t;

typedef enum
{
  OPEN_MULTI_PROTOCOL_CAP = 0x1,   /**< Multi-protocol */
  OPEN_ROUTE_REFLECT_CAP  = 0x8,   /**< Route Reflect */
  OPEN_COMMUNITIES_CAP    = 0x10,  /**< Communities */
  OPEN_CONFEDERATION_CAP  = 0x20,  /**< Confederation */
  OPEN_ROUTE_REFRESH_CAP  = 0x40   /**< Route Refresh */
} OPEN_E_OPTCAPTYPES_t;

/** Counters for BGP path attribute errors for received paths */
typedef struct open_bgpPeerRxAttrErrors_s
{
  uint16_t duplicateAttr;            /**< Path with duplicate attribute */
  uint16_t wellKwnOptConflict;       /**< Path with well-known/optional conflict */
  uint16_t transFlag;                /**< Transitive flag not set on transitive attr */
  uint16_t mandNonTransOrPart;       /**< Mandatory attribute non-transitive or partial */
  uint16_t optNonTransPart;          /**< Optional attribute non-transitive and partial */
  uint16_t attrTooLong;              /**< Path attribute too long */
  uint16_t attrLenError;             /**< Path attribute length error */
  uint16_t originInvalid;            /**< Invalid ORIGIN code */
  uint16_t wrongFirstAs;             /**< Unexpected first ASN in AS path */
  uint16_t invalidAsPathType;        /**< Invalid AS path segment type */
  uint16_t invalidNextHop;           /**< Invalid BGP NEXT HOP */
  uint16_t badNextHop;               /**< Bad BGP NEXT HOP */
  uint16_t invalidAggregator;        /**< Invalid AGGREGATOR attribute */
  uint16_t unknownWellKnown;         /**< Unrecognized well-known path attribute */
  uint16_t missingMandatory;         /**< Missing mandatory path attribute */
  uint16_t missingLocPref;           /**< Missing LOCAL PREF attribute */
  uint16_t invalidNetField;          /**< Invalid prefix in Update NLRI */
  uint16_t rxAsPathTooLong;          /**< Received AS Path Too Long */
  uint16_t origIdFromExtPeer;        /**< Received Originator ID from external peer */
  uint16_t clusterListFromExtPeer;   /**< Received Cluster List from external peer */

} open_bgpPeerRxAttrErrors_t;

/** Structure used to report BGP peer status */
typedef struct open_bgpPeerStatus_s
{
  /** Peer Router Id */
  uint32_t peerRouterId;

  /** Peer status */
  OPEN_BGP4_PEER_STATE_t peerState;

  /** Peer admin status */
  uint32_t peerAdminStatus;

  /** Negotiated Peer Version */ 
  uint32_t peerNegotiatedVersion;

  /** IP address/port of local end of TCP connection to peer */
  open_inet_addr_t localAddr;
  uint16_t localTcpPort; 

  /** Remote TCP port numbers for TCP connection to peer */
  uint16_t remoteTcpPort;

  /** Remote AS */
  uint32_t remoteAS;

  /** Bitmask indexed by e_OptCapTypes */
  uint16_t capabilities;

  /** Hold and keepalive times negotiated with peer */
  uint16_t negHoldTime;
  uint16_t negKeepaliveTime;

  /** Error code and subcode in last NOTIFICATION message sent to or received from peer.
   * If lastErrorInbound is L7_TRUE, then NOTIFICATION was received. Otherwise, last
   * NOTIFICATION was sent. lastErrorTime is the number of seconds since the NOTIFICATION
   * flew. */
  unsigned char lastErrorCode;
  unsigned char lastErrorSubcode;
  OPEN_BOOL_t lastErrorInbound;
  uint32_t lastErrorTime;

  /** String representations of the previous two. Pointer to static strings. */
  unsigned char lastErrorString[OPEN_BGP_ERR_STR_LEN];
  unsigned char lastSuberrorString[OPEN_BGP_ERR_STR_LEN];

  /** Receive path attribute error counters */
  open_bgpPeerRxAttrErrors_t attrErr;

  /** Number of transitions to the ESTABLISHED state since BGP counters last cleared */
  uint32_t estTransitions;

  /** How long, in seconds, since the adjacency transitioned into or out of the 
   * ESTABLISHED state */
  uint32_t estTime;

  /** Time intervals required by MIB */
  uint32_t retryIntervalConf;     /**< duplicate of one in config structure */
  uint32_t holdTimeConf;          /**< duplicate of configured hold time */
  uint32_t keepAliveTimeConf;     /**< duplicate of configured keep alive time */
  uint32_t minASOriginationConf;  /**< Not currently supported */
  uint32_t minRouteAdvIntConf;    /**< duplicate of configured advertisementInterval */

  /** How long, in seconds, since last UPDATE message received from peer */
  uint32_t lastUpdate;

  /** ID of outbound update group this peer is in */
  uint16_t updateGroupId[OPEN_BGP_AFI_MAX];

  /** Transmit message stats */
  uint32_t txOpen;
  uint32_t txUpdate;
  uint32_t txKeepalive;
  uint32_t txNotif;
  uint32_t txRefresh;
  uint32_t txTotal;

  /** Receive message stats */
  uint32_t rxOpen;
  uint32_t rxUpdate;
  uint32_t rxKeepalive;
  uint32_t rxNotif;
  uint32_t rxRefresh;
  uint32_t rxTotal;

  /** Number of prefixes received from this peer */
  uint32_t inPrefix[OPEN_BGP_AFI_MAX];

  /** Number of prefixes this peer has withdrawn */
  uint32_t inWithdraws[OPEN_BGP_AFI_MAX];

  /** Largest and smallest number of NLRI included in any one UPDATE 
   * message received from this peer */
  uint32_t inMaxNlriPerUpdate[OPEN_BGP_AFI_MAX];
  uint32_t inMinNlriPerUpdate[OPEN_BGP_AFI_MAX];

  /** Current number of prefixes this peer is advertising to us */
  uint32_t inPfxCurrent[OPEN_BGP_AFI_MAX];

  /** Similar on the send side */
  uint32_t outPrefix[OPEN_BGP_AFI_MAX];
  uint32_t outWithdraws[OPEN_BGP_AFI_MAX];
  uint32_t outMaxNlriPerUpdate[OPEN_BGP_AFI_MAX];
  uint32_t outMinNlriPerUpdate[OPEN_BGP_AFI_MAX];
  uint32_t outPfxCurrent[OPEN_BGP_AFI_MAX];

  /** Number of prefixes accepted from this peer, whose paths
   * do not have an AS path loop and whose BGP NEXT HOP are resolved */
  uint32_t inPfxAccepted[OPEN_BGP_AFI_MAX];
 
  /** Number of prefixes rejected from this peer, whose paths
   * have failed the inbound policy checks */
  uint32_t inPfxRejected[OPEN_BGP_AFI_MAX];

  /** Receive queue related statistics */
  uint32_t updateQueueLen; /**< current queue length */
  uint32_t updateQueueHigh;
  uint32_t updateQueueLimit;/**< maximum length accomodated */
  uint32_t updateQueueDrops;

  /** AFI data received on this peer */
  OPEN_BOOL_t  remoteAfiEnabled[OPEN_BGP_AFI_MAX];

  /** Total number of times the neighbors flaps */
  uint32_t flapCount;

  /** RFC 5549 supported */
  OPEN_BOOL_t   rfc5549Support;
  OPEN_BOOL_t   ipv4Active;
  OPEN_BOOL_t   ipv6Active;

  /** Peer type for IPv4 */
  OPEN_BGP4_PEER_TYPE_t peerType;

} open_bgpPeerStatus_t;

/** Structure used to report BGP decision process history */
typedef struct open_bgpDecProcHist_s
{
  /** Number of seconds since decision process ran */
  uint32_t decProcTime;

  /** Decision process phase that ran */
  uint16_t decProcPhase;

  /** Outbound update group ID. Only set when decProcPhase is 3. */
  uint16_t updateGroupId;

  /** generation ID of BGP routing table when decision process was run. */
  uint32_t genId;

  /** reason for decision process triggered */
  unsigned char reason[OPEN_BGP_MAX_DP_REASON_STR];

  /** Only set if decProcPhase is 1. Identifies the peer whose paths are
   * reprocessed. */
  open_inet_addr_t peerAddr;

  /** How long the decision process phase took. milliseconds. */
  uint16_t duration;

  /** Number of adds, mods, dels during decision process phase. */
  uint16_t adds;
  uint16_t mods;
  uint16_t dels;

} open_bgpDecProcHist_t;

/*********************************************************************
* @purpose  Get the (configured) maximum number of BGP IPv4 routes
*           for this platform.
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    maxRoutes           @b{(output)} maximum number of routes
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpIpv4RouteMax(openapiClientHandle_t *client_handle,
                                    uint32_t *maxRoutes);
 
/*********************************************************************
* @purpose  Get the (configured) maximum number of BGP IPv6 routes
*           for this platform.
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    maxRoutes           @b{(output)} maximum number of routes
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpIpv6RouteMax(openapiClientHandle_t *client_handle,
                                    uint32_t *maxRoutes);

/*********************************************************************
* @purpose  Get the configured hold time for all peers
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    holdTime            @b{(output)} hold time (secs)
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpGlobalHoldTimeConfiguredGet(openapiClientHandle_t *client_handle,
                                                   uint32_t *holdTime);

/********************************************************************* 
* @purpose  Set the configured hold time for all peers
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    holdTime            @b{(input)}  hold time (secs) 
* @param    -                   Minimum value @ref OPENAPI_BGP4_HOLD_TIME_MIN  
* @param    -                   Maximum value @ref OPENAPI_BGP4_HOLD_TIME_MAX 
*  
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpGlobalHoldTimeConfiguredSet(openapiClientHandle_t *client_handle,
                                                   uint32_t holdTime);

/*********************************************************************
* @purpose  Get the configured keep alive time for all peers
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    keepAlive           @b{(output)} keep alive time (secs)
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpGlobalKeepAliveConfiguredGet(openapiClientHandle_t *client_handle,
                                                    uint32_t *keepAlive);

/********************************************************************* 
* @purpose  Set the configured keep alive time for all peers
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    keepAlive           @b{(input)}  keep alive time (secs)
* @param    -                   Minimum value @ref OPENAPI_BGP4_KEEP_ALIVE_TIME_MIN  
* @param    -                   Maximum value @ref OPENAPI_BGP4_KEEP_ALIVE_TIME_MAX 
*  
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpGlobalKeepAliveConfiguredSet(openapiClientHandle_t *client_handle,
                                                    uint32_t keepAlive);

/*********************************************************************
* @purpose  Get the Local Autonomous System (AS) number
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    localAS             @b{(output)} local AS number
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpLocalASGet(openapiClientHandle_t *client_handle,
                                  uint32_t *localAS);

/*********************************************************************
* @purpose  Set the Local Autonomous System (AS) number
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    localAS             @b{(input)}  local AS number
* @param    -                   Minimum value 0
* @param    -                   Maximum value signed integer (65535)
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes    Setting AS to 0 destroys the existing BGP instance.
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpLocalASSet(openapiClientHandle_t *client_handle,
                                  uint32_t localAS);

/*********************************************************************
* @purpose  Get the BGP router ID
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    routerId            @b{(output)} router id
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpLocalIdGet(openapiClientHandle_t *client_handle,
                                  uint32_t *routerId);

/*********************************************************************
* @purpose  Set the BGP router ID
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    routerId            @b{(input)}  router ipv4 address for BGP
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpLocalIdSet(openapiClientHandle_t *client_handle,
                                  uint32_t routerId);

/*********************************************************************
* @purpose  Get the value of local preference attribute
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    localPref           @b{(output)} local preference
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpLocalPrefGet(openapiClientHandle_t *client_handle,
                                    uint32_t *localPref);

/*********************************************************************
* @purpose  Set the value of local preference attribute
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    localPref           @b{(input)}  local preference
* @param    -                   Minimum value 0
* @param    -                   Maximum value unsigned integer (4294967295)
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpLocalPrefSet(openapiClientHandle_t *client_handle,
                                    uint32_t localPref);

/*********************************************************************
* @purpose  Get the log neighbor changes parameter
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    log                 @b{(output)} if true, then logging is allowed
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpLogNeighborChangesGet(openapiClientHandle_t *client_handle,
                                             bool *log);

/*********************************************************************
* @purpose  Enable/Disable log neighbor changes
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    log                 @b{(input)}  if true, then logging is allowed
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpLogNeighborChangesSet(openapiClientHandle_t *client_handle,
                                             bool log);

/*********************************************************************
* @purpose  Get the value of maximum Paths attribute
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    af                  @b{(input)}  address family
* @param    isIbgp              @b{(input)}  if true, then selects ibgp else ebgp 
* @param    maxPaths            @b{(output)} storage space for maximum path value
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpMaxPathsGet(openapiClientHandle_t *client_handle,
                                   OPEN_AF_t af, 
                                   bool isIbgp, 
                                   uint32_t *maxPaths);

/*********************************************************************
* @purpose  Set the value of maximum Paths attribute
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    af                  @b{(input)}  address family
* @param    isIbgp              @b{(input)}  if true, then selects ibgp else ebgp 
* @param    maxPaths            @b{(input)}  storage space for maximum path value
* @param    -                   Minimum value @ref OPENAPI_BGP4_MAXPATH_MIN  
* @param    -                   Maximum value openapiBgpMaxPathsGet()
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes    The maxPaths value is the maximum number of paths supported for
*           the given platform. The maximum paths is essentially the maximum
*           number of ECMP next hops supported for the active template. This
*           value can be obtained by using the OpEN API openapiBgpMaxPathsGet()
*           function.
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpMaxPathsSet(openapiClientHandle_t *client_handle,
                                   OPEN_AF_t af, 
                                   bool isIbgp, 
                                   uint32_t maxPaths);

/********************************************************************* 
* @purpose  Add or Delete a logically originated network
* 
* @param    client_handle       @b{(input)}  client handle from registration API  
* @param    normalMode          @b{(input)}  true to add, false to remove
* @param    useRm               @b{(input)}  true if a route-map is specified
* @param    addr                @b{(input)}  ip address
* @param    pfxLen              @b{(input)}  prefix length
* @param    rmName              @b{(input)}  route map name 
* @param    -                   Maximum length @ref OPENAPI_ROUTEMAP_NAME_SIZE_MAX 
*  
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpNetworkAddDelete(openapiClientHandle_t *client_handle,
                                        bool normalMode,
                                        bool useRm,
                                        open_inet_addr_t addr,
                                        uint32_t pfxLen,
                                        open_buffdesc *rmName);

/********************************************************************* 
* @purpose  Set route redistribution options for routes learned from
*           a specific source.
* 
* @param    client_handle       @b{(input)} client handle from registration API
* @param    add                 @b{(input)}  if true, customer is setting or adding options
* @param    protoName           @b{(input)}  redistribution route name 
* @param    -                   Maximum length openapiRouteProtoNameLenMax()
* @param    matchSet            @b{(input)}  if true, set OSPF match
* @param    matchBits           @b{(input)}  match options for OSPF
* @param    metricSet           @b{(input)}  if true, turn metric options on
* @param    metricValueSet      @b{(input)}  if true, use new metric value
* @param    metricValue         @b{(input)}  new metric value 
* @param    -                   Minimum value 0
* @param    -                   Maximum value unsigned integer (4294967295)
* @param    routeMapSet         @b{(input)}  if true, select optional route map
* @param    af                  @b{(input)}  address family
* @param    rmName              @b{(input)}  route map tag
* @param    -                   Maximum length @ref OPENAPI_ROUTEMAP_NAME_SIZE_MAX
* 
* @returns  OPEN_E_NONE         set successful.
* @returns  OPEN_E_FAIL         set failed.
* @returns  OPEN_E_PARAM        error in parameter passed.  
* @returns  OPEN_E_FULL         if route-map couldn't be created 
* @returns  OPEN_E_NOT_FOUND    if the route-map name doesn't match (no form)
* @returns  OPEN_E_UNAVAIL      if the peer address requires a non-zero scope ID
* 
* @notes    The match arguments are ignored if protocol is not REDIST_RT_OSPF.
* @notes    If disabling redistribution, don't clear a distribute list, if set,
*           since a distribute list is configured with a separate call.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpRedistributionSet(openapiClientHandle_t *client_handle,
                                         bool add,
                                         open_buffdesc *protoName,
                                         bool matchSet,
                                         uint32_t matchBits,
                                         bool metricSet,
                                         bool metricValueSet,
                                         uint32_t metricValue,
                                         bool routeMapSet,
                                         OPEN_AF_t af,
                                         open_buffdesc *rmName);

/********************************************************************/
/*********************** NEIGHBOR / PEER API's **********************/
/********************************************************************/

/*********************************************************************
* @purpose  Determine whether a given peer is configured to exchange routes
*           for a given address family. 
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pType               @b{(input)}  Whether to get the value configured directly
*                                            on the peer or the active value, which may be
*                                            inherited from a peer template
* @param    remoteAddr          @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  If remoteAddr has link scope, internal
*                                            interface number where address is valid
* @param    af                  @b{(input)}  address family
* @param    activate            @b{(output)} true if address family is active
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_NOT_FOUND    Set if the neighbor is not found
* 
* @notes    Available for ipv6 peers only.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerActivateGet(openapiClientHandle_t *client_handle,
                                       OPEN_BGP_PARAM_TYPE pType,
                                       open_inet_addr_t remoteAddr, 
                                       uint32_t scopeId, 
                                       OPEN_AF_t af, 
                                       bool *activate);

/*********************************************************************
* @purpose  Configure whether a given peer advertises and accepts routes
*           for a given address family.
* 
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    remoteAddr          @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  If remoteAddr has link scope, internal
*                                            interface number where address is valid
* @param    af                  @b{(input)}  address family
* @param    activate            @b{(input)}  true if address family is active
* @param    defValue            @b{(input)}  true if interval is set to a non-default value
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_NOT_FOUND    Set if the neighbor is not found
* @returns  OPEN_E_UNAVAIL      Set if there is a failure applying the configuration change
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Available for ipv6 peers only.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerActivateSet(openapiClientHandle_t *client_handle,
                                       open_inet_addr_t remoteAddr, 
                                       uint32_t scopeId, 
                                       OPEN_AF_t af, 
                                       bool activate,
                                       bool defValue);

/********************************************************************* 
* @purpose  Get the administrative status (start/stop) of a given BGP peer.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pType               @b{(input)}  Whether to get the value configured
*                                            directly on the peer or the active
*                                            value, which may be inherited from
*                                            a peer template
* @param    remoteAddr          @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  If remoteAddr has link scope, internal
*                                            interface number where address is valid
* @param    status              @b{(output)} admin status of the peer (start/stop)
*  
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_NOT_FOUND    Set if the neighbor is not found
* @returns  OPEN_E_INTERNAL     Catch for lower level failures
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerAdminStatusGet(openapiClientHandle_t *client_handle,                                      
                                          OPEN_BGP_PARAM_TYPE pType,
                                          open_inet_addr_t remoteAddr,
                                          uint32_t scopeId, 
                                          OPEN_BGP_PEER_STATE_t *status);

/*********************************************************************
* @purpose  Get the administrative status (start/stop) of a given BGP peer.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    vrf                 @b{(input)}  VRF name ("" for default) 
* @param    pType               @b{(input)}  Whether to get the value configured
*                                            directly on the peer or the active
*                                            value, which may be inherited from
*                                            a peer template
* @param    remoteAddr          @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  If remoteAddr has link scope, internal
*                                            interface number where address is valid
* @param    status              @b{(output)} admin status of the peer (start/stop)
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_NOT_FOUND    Set if the neighbor is not found
* @returns  OPEN_E_INTERNAL     Catch for lower level failures
*
* @notes    Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.16
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerAdminStatusVrfGet(openapiClientHandle_t *client_handle,
                                             open_buffdesc *vrf,
                                             OPEN_BGP_PARAM_TYPE pType,
                                             open_inet_addr_t remoteAddr,
                                             uint32_t scopeId,
                                             OPEN_BGP_PEER_STATE_t *status);


/********************************************************************* 
* @purpose  Set the administrative status (start/stop) of a given BGP peer.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    remoteAddr          @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  If remoteAddr has link scope, internal
*                                            interface number where address is valid
* @param    status              @b{(input)}  admin status of the peer (start/stop)
* @param    defValue            @b{(input)}  enable if status is set to a non-default value
*  
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_NOT_FOUND    Set if the neighbor is not found
* @returns  OPEN_E_INTERNAL     Catch for lower level failures
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerAdminStatusSet(openapiClientHandle_t *client_handle,
                                          open_inet_addr_t remoteAddr, 
                                          uint32_t scopeId, 
                                          OPEN_BGP_PEER_STATE_t status, 
                                          bool defValue);

/*********************************************************************
* @purpose  Set the administrative status (start/stop) of a given BGP peer.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    vrf                 @b{(input)}  VRF name ("" for default)
* @param    remoteAddr          @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  If remoteAddr has link scope, internal
*                                            interface number where address is valid
* @param    status              @b{(input)}  admin status of the peer (start/stop)
* @param    defValue            @b{(input)}  enable if status is set to a non-default value
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_NOT_FOUND    Set if the neighbor is not found
* @returns  OPEN_E_INTERNAL     Catch for lower level failures
*
* @notes    Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.16
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerAdminStatusVrfSet(openapiClientHandle_t *client_handle,
                                             open_buffdesc *vrf,
                                             open_inet_addr_t remoteAddr,
                                             uint32_t scopeId,
                                             OPEN_BGP_PEER_STATE_t status,
                                             bool defValue);


/*********************************************************************
* @purpose  Get the advertisement interval for a given BGP peer
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pType               @b{(input)}  Whether to get the value configured
*                                            directly on the peer or the active
*                                            value, which may be inherited from
*                                            a peer template
* @param    remoteAddr          @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  If remoteAddr has link scope, internal
*                                            interface number where address is valid
* @param    af                  @b{(input)}  address family
* @param    interval            @b{(output)} advertisement interval (secs)
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes  
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerAdvertisementIntervalGet(openapiClientHandle_t *client_handle,
                                                    OPEN_BGP_PARAM_TYPE pType,
                                                    open_inet_addr_t remoteAddr,
                                                    uint32_t scopeId, 
                                                    OPEN_AF_t af, 
                                                    uint32_t *interval);

/*********************************************************************
* @purpose  Set the advertisement interval for a given BGP peer
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    remoteAddr          @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  If remoteAddr has link scope, internal
*                                            interface number where address is valid
* @param    af                  @b{(input)}  address family
* @param    interval            @b{(input)}  advertisement interval (secs)
* @param    -                   Minimum value @ref OPENAPI_BGP4_ADV_INTERVAL_MIN
* @param    -                   Maximum value @ref OPENAPI_BGP4_ADV_INTERVAL_MAX
* @param    defValue            @b{(input)}  true if interval is set to a non-default value
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_NOT_FOUND    Set if the neighbor is not found
* @returns  OPEN_E_UNAVAIL      Set if there is a failure applying the configuration change.
*                               Could possibly be due to an invalid route for the specified
*                               transport. For example, ipv4 routes not being supported
*                               over an ipv6 transport. 
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerAdvertisementIntervalSet(openapiClientHandle_t *client_handle,
                                                    open_inet_addr_t remoteAddr, 
                                                    uint32_t scopeId, 
                                                    OPEN_AF_t af, 
                                                    uint32_t interval, 
                                                    bool defValue);

/*********************************************************************
* @purpose  Get the configured hold time for a given BGP peer
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pType               @b{(input)}  Whether to get the value configured
*                                            directly on the peer or the active
*                                            value, which may be inherited from
*                                            a peer template
* @param    remoteAddr          @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  If remoteAddr has link scope, internal
*                                            interface number where address is valid
* @param    holdTime            @b{(output)} hold time (secs)
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerHoldTimeConfiguredGet(openapiClientHandle_t *client_handle,
                                                 OPEN_BGP_PARAM_TYPE pType,
                                                 open_inet_addr_t remoteAddr, 
                                                 uint32_t scopeId,
                                                 uint32_t *holdTime);

/********************************************************************* 
* @purpose  Set the configured hold time for a given BGP peer
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    remoteAddr          @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  If remoteAddr has link scope, internal
*                                            interface number where address is valid
* @param    holdTime            @b{(input)}  hold time (secs) 
* @param    -                   Minimum value @ref OPENAPI_BGP4_HOLD_TIME_MIN
* @param    -                   Maximum value @ref OPENAPI_BGP4_HOLD_TIME_MAX
* @param    defValue            @b{(input)}  true if interval is set to a non-default value
*  
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_NOT_FOUND    Set if the neighbor is not found
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerHoldTimeConfiguredSet(openapiClientHandle_t *client_handle,
                                                 open_inet_addr_t remoteAddr, 
                                                 uint32_t scopeId, 
                                                 uint32_t holdTime, 
                                                 bool defValue);

/*********************************************************************
* @purpose  Get the configured keep alive time for a given BGP peer
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pType               @b{(input)}  Whether to get the value configured
*                                            directly on the peer or the active
*                                            value, which may be inherited from
*                                            a peer template
* @param    remoteAddr          @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  If remoteAddr has link scope, internal
*                                            interface number where address is valid
* @param    keepAlive           @b{(output)} keep alive time (secs)
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerKeepAliveConfiguredGet(openapiClientHandle_t *client_handle,
                                                  OPEN_BGP_PARAM_TYPE pType,
                                                  open_inet_addr_t remoteAddr, 
                                                  uint32_t scopeId,
                                                  uint32_t *keepAlive);

/********************************************************************* 
* @purpose  Set the configured keep alive time for a given BGP peer
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    remoteAddr          @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  If remoteAddr has link scope, internal
*                                            interface number where address is valid
* @param    keepAlive           @b{(input)}  keep alive time (secs) 
* @param    -                   Minimum value @ref OPENAPI_BGP4_KEEP_ALIVE_TIME_MIN
* @param    -                   Maximum value @ref OPENAPI_BGP4_KEEP_ALIVE_TIME_MAX
* @param    defValue            @b{(input)}  true if interval is set to a non-default value
*  
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_NOT_FOUND    Set if the neighbor is not found
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerKeepAliveConfiguredSet(openapiClientHandle_t *client_handle,
                                                  open_inet_addr_t remoteAddr, 
                                                  uint32_t scopeId, 
                                                  uint32_t keepAlive, 
                                                  bool defValue);

/*********************************************************************
* @purpose  Determine whether BGP sets the BGP NEXT_HOP to a local address
*           when forwarding external routes to a given internal peer. 
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pType               @b{(input)}  Whether to get the value configured directly
*                                            on the peer or the active value, which may be
*                                            inherited from a peer template
* @param    remoteAddr          @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  If remoteAddr has link scope, internal
*                                            interface number where address is valid
* @param    af                  @b{(input)}  address family
* @param    enable              @b{(output)} enabled if BGP uses its own address as 
*                                            the BGP NEXT HOP in advertisements to this peer
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerNextHopSelfModeGet(openapiClientHandle_t *client_handle,
                                              OPEN_BGP_PARAM_TYPE pType,
                                              open_inet_addr_t remoteAddr,
                                              uint32_t scopeId, 
                                              OPEN_AF_t af, 
                                              OPEN_CONTROL_t *enable);

/*********************************************************************
* @purpose  Configure BGP to use a local address when advertising routes to
*           a given internal peer.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    remoteAddr          @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  If remoteAddr has link scope, internal
*                                            interface number where address is valid
* @param    af                  @b{(input)}  address family
* @param    enable              @b{(input)}  enabled if BGP uses its own address as 
*                                            the BGP NEXT HOP in advertisements to this peer
* @param    defValue            @b{(input)}  true if enable is set to a non-default value
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_NOT_FOUND    Set if the neighbor is not found
* 
* @notes    Calling this API will change the running configuration of the switch.
* @notes    External peers cannot be configured for next-hop-self.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerNextHopSelfModeSet(openapiClientHandle_t *client_handle,
                                              open_inet_addr_t remoteAddr,
                                              uint32_t scopeId,
                                              OPEN_AF_t af, 
                                              OPEN_CONTROL_t enable, 
                                              bool defValue);

/*********************************************************************
* @purpose  Get the prefix limit configured for a given peer.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pType               @b{(input)}  Whether to get the value configured directly
*                                            on the peer or the active value, which may be
*                                            inherited from a peer template
* @param    remoteAddr          @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  If remoteAddr has link scope, internal
*                                            interface number where address is valid
* @param    af                  @b{(input)}  address family
* @param    pfxLimit            @b{(output)} maximum number of prefixes
* @param    threshold           @b{(output)} threshold percentage for warning generation
* @param    warningOnly         @b{(output)} true if new prefixes are allowed, but give
*                                            warning message when limit exceeded.
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes    The peerAddr and scopeId fields specify which peer to examine. 
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerPfxLimitGet(openapiClientHandle_t *client_handle,
                                       OPEN_BGP_PARAM_TYPE pType, 
                                       open_inet_addr_t remoteAddr,
                                       uint32_t scopeId,
                                       OPEN_AF_t af,
                                       uint32_t *pfxLimit,
                                       uint32_t *threshold,
                                       bool *warningOnly);

/*********************************************************************
* @purpose  Set the prefix limit configured for a given peer.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    remoteAddr          @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  If remoteAddr has link scope, internal
*                                            interface number where address is valid
* @param    af                  @b{(input)}  address family
* @param    pfxLimit            @b{(output)} maximum number of prefixes (see notes)
* @param    threshold           @b{(input)}  threshold percentage for warning generation
* @param    warningOnly         @b{(input)}  true if new prefixes are allowed, but give
*                                            warning message when limit exceeded.
* @param    defValue            @b{(input)}  true if source is set to a non-default value
* 
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_NOT_FOUND    Set if the neighbor is not found
* @returns  OPEN_E_UNAVAIL      Set if there is a failure applying the configuration change.
*                               Could possibly be due to an invalid route for the specified
*                               transport. For example, ipv4 routes not being supported
*                               over an ipv6 transport. 
* 
* @notes    The peerAddr and scopeId fields specify which peer to configure.
* @notes    pfxLimit - the maximum routes are platform specific.
* @notes    pfxLimit - use openapiBgpIpv4RouteMax() to determine Ipv4 route max.
* @notes    pfxLimit - use openapiBgpIpv6RouteMax() to determine Ipv6 route max.
* @notes    pfxLimit - use @ref OPEN_BGP_NBR_MAX_PFX_NOLIMIT for unlimited (platform max).
* @notes    Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerPfxLimitSet(openapiClientHandle_t *client_handle,
                                       open_inet_addr_t remoteAddr,
                                       uint32_t scopeId,
                                       OPEN_AF_t af,
                                       uint32_t pfxLimit,
                                       uint32_t threshold,
                                       bool warningOnly,
                                       bool defValue);

/*********************************************************************
* @purpose  Return a remote Autonomous System (AS) number that this peer belongs to. 
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    remoteAddr          @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  If remoteAddr has link scope, internal
*                                            interface number where address is valid
* @param    remoteAS            @b{(output)} AS number of peer
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerRemoteASGet(openapiClientHandle_t *client_handle,
                                       open_inet_addr_t remoteAddr, 
                                       uint32_t scopeId,
                                       uint32_t *remoteAS);

/*********************************************************************
* @purpose  Return a remote Autonomous System (AS) number that this peer belongs to.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    vrf                 @b{(input)}  VRF name ("" for default VRF)
* @param    remoteAddr          @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  If remoteAddr has link scope, internal
*                                            interface number where address is valid
* @param    remoteAS            @b{(output)} AS number of peer
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.16
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerRemoteASVrfGet(openapiClientHandle_t *client_handle,
                                          open_buffdesc *vrf,
                                          open_inet_addr_t remoteAddr,
                                          uint32_t scopeId,
                                          uint32_t *remoteAS);

/*********************************************************************
* @purpose  Create a peer with a given IP address and Autonomous System (AS) number. 
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    remoteAddr          @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  If remoteAddr has link scope, internal
*                                            interface number where address is valid
* @param    remoteAS            @b{(input)}  AS number of peer
* @param    -                   Minimum value 0
* @param    -                   Maximum value signed integer (65535)
*
* @returns  OPEN_E_NONE         set successful.
* @returns  OPEN_E_FAIL         set failed.
* @returns  OPEN_E_PARAM        error in parameter passed.
* @returns  OPEN_E_FULL         exceeded the maximum number of peers
* @returns  OPEN_E_UNAVAIL      if a peer is already configured at this IP address
*                               but with a different AS number.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerRemoteASSet(openapiClientHandle_t *client_handle,
                                       open_inet_addr_t remoteAddr, 
                                       uint32_t scopeId,
                                       uint32_t remoteAS);

/*********************************************************************
* @purpose  Create a peer with a given IP address and Autonomous System (AS) number.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    vrf                 @b{(input)}  VRF name ("" for default VRF)
* @param    remoteAddr          @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  If remoteAddr has link scope, internal
*                                            interface number where address is valid
* @param    remoteAS            @b{(input)}  AS number of peer
* @param    -                   Minimum value 0
* @param    -                   Maximum value signed integer (65535)
*
* @returns  OPEN_E_NONE         set successful.
* @returns  OPEN_E_FAIL         set failed.
* @returns  OPEN_E_PARAM        error in parameter passed.
* @returns  OPEN_E_FULL         exceeded the maximum number of peers
* @returns  OPEN_E_UNAVAIL      if a peer is already configured at this IP address
*                               but with a different AS number.
*
* @notes    Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.16
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerRemoteASVrfSet(openapiClientHandle_t *client_handle,
                                          open_buffdesc *vrf,
                                          open_inet_addr_t remoteAddr,
                                          uint32_t scopeId,
                                          uint32_t remoteAS);


/********************************************************************* 
* @purpose  Get the interface whose IP address that BGP uses as the source
*           IP address in packets sent to a given peer
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pType               @b{(input)}  Whether to get the value configured
*                                            directly on the peer or the active
*                                            value, which may be inherited from
*                                            a peer template
* @param    remoteAddr          @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  If remoteAddr has link scope, internal
*                                            interface number where address is valid
* @param    source              @b{(output)} internal interface number
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerUpdateSourceGet(openapiClientHandle_t *client_handle,
                                           OPEN_BGP_PARAM_TYPE pType,
                                           open_inet_addr_t remoteAddr, 
                                           uint32_t scopeId, 
                                           uint32_t *source);

/********************************************************************* 
* @purpose  Set the interface whose IP address that BGP uses as the source
*           IP address in packets sent to a given peer
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    remoteAddr          @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  If remoteAddr has link scope, internal
*                                            interface number where address is valid
* @param    source              @b{(input)}  internal interface number
* @param    defValue            @b{(input)}  true if source is set to a non-default value
*  
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_NOT_FOUND    Set if the neighbor is not found
* 
* @notes    Calling this API will change the running configuration of the switch. 
* @notes    Source can also be a loopback interface. 
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerUpdateSourceSet(openapiClientHandle_t *client_handle,
                                           open_inet_addr_t remoteAddr, 
                                           uint32_t scopeId, 
                                           uint32_t source, 
                                           bool defValue);

/********************************************************************* 
* @purpose  Gets the Bgp Admin Mode
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    adminMode           @b{(output)} status of admin mode 
*  
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_FAIL         Set failed if the api failed to get admin mode.
* 
* @notes    Get Bgp admin mode @ref OPEN_ENABLE if the admin mode is enabled
*           @ref OPEN_DISABLE if the admin mode is disabled
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpAdminModeGet (openapiClientHandle_t * client_handle,
                                     OPEN_CONTROL_t * adminMode);

/********************************************************************* 
* @purpose  Sets the Bgp Admin mode
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    adminMode           @b{(input)}  status of admin mode
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_FAIL         Set failed if the api failed to set admin mode.
* 
* @notes    Calling this API will change the running configuration of the switch.
*           Sets the Bgp Admin mode @ref OPEN_ENABLE to enable and 
*           @ref OPEN_DISABLE to disable admin mode
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpAdminModeSet (openapiClientHandle_t * client_handle,
                                     OPEN_CONTROL_t adminMode);

/********************************************************************* 
* @purpose  Get BGP global status of number of network entries
*           and number of AS paths
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    af                  @b{(input)}   address family
* @param    numNetworks         @b{(output)}  Number of distinct prefixes 
*                                             in the local RIB
* @param    numAsPaths          @b{(output)}  Number of AS paths in the local RIB
*  
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_FAIL         Failed to get BGP global status 
* @returns  OPEN_E_INTERNAL     Catch for lower level failures 
* 
* @notes    None
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpMapGlobalStatusInfoGet (openapiClientHandle_t * client_handle, 
                                               OPEN_AF_t af,
                                               uint32_t * numNetworks,
                                               uint32_t * numAsPaths);

/********************************************************************* 
* @purpose  Given the IP address of one peer, get the IP address of the next peer
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    af                  @b{(input)}  address family
* @param    prevRemoteAddr      @b{(input)}  IP address of previous peer. Pass NULL to get
*                                            the first peer.
* @param    prevScopeId         @b{(input)}  scope ID of previous peer
* @param    remoteAddr          @b{(output)} IP address of next peer 
* @param    scopeId             @b{(output)} (optional) scope ID of next peer address
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_FAIL         Failed to get next peer address
* @returns  OPEN_E_INTERNAL     Lower-level error occurred
* 
* @notes    None
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpMapPeerRemoteAddrGetNext (openapiClientHandle_t * client_handle, 
                                                 OPEN_AF_t af,
                                                 open_inet_addr_t prevRemoteAddr,
                                                 uint32_t prevScopeId,
                                                 open_inet_addr_t * remoteAddr,
                                                 uint32_t * scopeId);

/********************************************************************* 
* @purpose  Given the IP address of one peer, and a VRF name, get the IP address of the next peer
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    af                  @b{(input)}  address family
* @param    prevRemoteAddr      @b{(input)}  IP address of previous peer. Pass NULL to get
*                                            the first peer.
* @param    prevScopeId         @b{(input)}  scope ID of previous peer
* @param    vrf                 @b{(input)}  VRF name. Use "" for default.
* @param    remoteAddr          @b{(output)} IP address of next peer 
* @param    scopeId             @b{(output)} (optional) scope ID of next peer address
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_FAIL         Failed to get next peer address
* @returns  OPEN_E_INTERNAL     Lower-level error occurred
* 
* @notes    The vrf parameter must be "" for IPv6 instances since VRF names
*           are not supported for IPv6.
* 
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiBgpMapPeerRemoteAddrVrfGetNext (openapiClientHandle_t * client_handle,
                                                    OPEN_AF_t af,
                                                    open_inet_addr_t prevRemoteAddr,
                                                    uint32_t prevScopeId,
                                                    open_buffdesc *vrf,
                                                    open_inet_addr_t * remoteAddr,
                                                    uint32_t * scopeId);

/********************************************************************* 
* @purpose  Get the status of a peer for a default VRF instance with a given
*           IP address
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    remoteAddr          @b{(input)}   the IP address of the peer
* @param    scopeId             @b{(input)}   scope ID of the peer's IP address
* @param    af                  @b{(input)}   address family of interest (only affects
*                                             which minRouteAdvInterval is reported)
* @param    status              @b{(output)}  the status record to fill
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_FAIL         Failed to get status of given peer
* @returns  OPEN_E_NOT_FOUND    No peer with the given IP address
* @returns  OPEN_E_INTERNAL     Lower-level error occurred
* 
* @notes    None
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpMapPeerStatusGet (openapiClientHandle_t * client_handle,
                                         open_inet_addr_t remoteAddr,
                                         uint32_t scopeId,
                                         OPEN_AF_t af,
                                         open_bgpPeerStatus_t * status);

/*********************************************************************
* @purpose  Get the status of a peer for the given VRF instance and IP address
*
* @param    client_handle       @b{(input)}   Client handle from registration API
* @param    vrf                 @b{(input)}   VRF name. Use "" for default.
*                                             Maximum length @ref L7_VRF_MAX_NAME_LEN
* @param    remoteAddr          @b{(input)}   The IP address of the peer
* @param    scopeId             @b{(input)}   Scope ID of the peer's IP address
* @param    af                  @b{(input)}   Address family of interest (only affects
*                                             which minRouteAdvInterval is reported)
* @param    status              @b{(output)}  The status record to fill
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_FAIL         Failed to get status of given peer
* @returns  OPEN_E_NOT_FOUND    No peer with the given IP address
* @returns  OPEN_E_INTERNAL     Lower-level error occurred
*
* @notes    None
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiBgpMapPeerStatusVrfGet (openapiClientHandle_t * client_handle,
                                            open_buffdesc *vrf,
                                            open_inet_addr_t remoteAddr,
                                            uint32_t scopeId,
                                            OPEN_AF_t af,
                                            open_bgpPeerStatus_t * status);

/*********************************************************************
* @purpose  Gets the Template Name string length. This length is used to
*           allocate and initialize the open_buffdesc type argument buffers
*           used by various BGP OpEN APIs.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    maxLen          @b{(output)}  string length 
*                                         maximum length @ref OPEN_BGP_MAX_TEMPLATE_NAME
*
* @returns  OPEN_E_NONE        String length Get successful.
* @returns  OPEN_E_PARAM       Error in parameter passed. 
* 
* @notes    None
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpTemplateNameMaxLengthGet (openapiClientHandle_t * client_handle,
                                                 uint32_t * maxLen);

/*********************************************************************
* @purpose  Set the template name and inuse flag
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    bufdName            @b{(input)}  template name
* @param    -                   Maximum length @ref OPEN_BGP_MAX_TEMPLATE_NAME
*
* @returns  OPEN_E_NONE         set successful
* @returns  OPEN_E_FAIL         set failed
* @returns  OPEN_E_PARAM        error in parameter passed
* @returns  OPEN_E_FULL         exceeded the maximum number of templates
* @returns  OPEN_E_INTERNAL     catch for lower level failures
*
* @notes    Calling this API will change the running configuration of the switch
*           Find the first unused slot and set the name and the inUse
*           flag. If we find that the template name already exists
*           then simply make sure the inUse flag is set.
*
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpTemplateSet (openapiClientHandle_t * client_handle,
                                    open_buffdesc * bufdName);

/*********************************************************************
* @purpose  Clear the template name and inuse flag
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    bufdName            @b{(input)}  template name
* @param    -                   Maximum length @ref OPEN_BGP_MAX_TEMPLATE_NAME
*
* @returns  OPEN_E_NONE         set successful
* @returns  OPEN_E_FAIL         set failed
* @returns  OPEN_E_PARAM        error in parameter passed
* @returns  OPEN_E_NOT_FOUND    set if the template is not found
* @returns  OPEN_E_UNAVAIL      set if there is a failure applying the configuration change
* @returns  OPEN_E_INTERNAL     catch for lower level failures
*
* @notes    Calling this API will change the running configuration of the switch
*           Search for the given template name and if found, reset all
*           the fields and reset the inUse flag.
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpTemplateDelete (openapiClientHandle_t * client_handle,
                                       open_buffdesc * bufdName);

/*********************************************************************
* @purpose  Set the address-family of given template
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    bufdName            @b{(input)}  template name
* @param    -                   Maximum length @ref OPEN_BGP_MAX_TEMPLATE_NAME
* @param    af                  @b{(input)}  address family
*
* @returns  OPEN_E_NONE         set successful
* @returns  OPEN_E_FAIL         set failed
* @returns  OPEN_E_PARAM        error in parameter passed
* @returns  OPEN_E_INTERNAL     catch for lower level failures
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpTemplateAFSet (openapiClientHandle_t * client_handle,
                                      open_buffdesc * bufdName, OPEN_AF_t af);

/*********************************************************************
* @purpose  Clear the address-family of given template
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    bufdName            @b{(input)}  template name
* @param    -                   Maximum length @ref OPEN_BGP_MAX_TEMPLATE_NAME
* @param    af                  @b{(input)}  address family
*
* @returns  OPEN_E_NONE         set successful
* @returns  OPEN_E_FAIL         set failed
* @returns  OPEN_E_PARAM        error in parameter passed
* @returns  OPEN_E_INTERNAL     catch for lower level failures
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpTemplateAFClear (openapiClientHandle_t * client_handle,
                                        open_buffdesc * bufdName, OPEN_AF_t af);

/*********************************************************************
* @purpose  Clear the address-family of given template
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    bufdName            @b{(input)}  template name
* @param    -                   Maximum length @ref OPEN_BGP_MAX_TEMPLATE_NAME
* @param    af                  @b{(input)}  address family
*
* @returns  OPEN_E_NONE         set successful
* @returns  OPEN_E_FAIL         set failed
* @returns  OPEN_E_PARAM        error in parameter passed
* @returns  OPEN_E_INTERNAL     catch for lower level failures
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpTemplateSendCommunitySet (openapiClientHandle_t *client_handle,
                                                 open_buffdesc * bufdName,
                                                 OPEN_AF_t af,
                                                 OPEN_CONTROL_t enable);

/*********************************************************************
* @purpose  Gets the description name maximum length
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    maxLen          @b{(output)}  string length 
*                                         maximum length @ref OPEN_BGP_PEER_DESC_LENGTH
*
* @returns  OPEN_E_NONE        String length Get successful.
* @returns  OPEN_E_PARAM       Error in parameter passed. 
* 
* @notes    None
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerDescriptionNameMaxLengthGet (openapiClientHandle_t * client_handle,
                                                        uint32_t * maxLen);

/********************************************************************* 
* @purpose  Set the description field for the specified peer
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    remoteAddr          @b{(input)}  IP address of peer
* @param    scopeId             @b{(input)}  address scope
* @param    bufdName            @b{(input)}  the description of the peer
* @param    defValue            @b{(input)}  true if interval is set to a non-default value
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_NOT_FOUND    if peer not found
* @returns  OPEN_E_INTERNAL     catch for lower level failures 
* 
* @notes    None
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerDescriptionSet (openapiClientHandle_t * client_handle,
                                           open_inet_addr_t remoteAddr,
                                           uint32_t scopeId,
                                           open_buffdesc * bufdName,
                                           bool defValue);

/*********************************************************************
* @purpose  Set the description field for the specified peer
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    vrf                 @b{(input)}  VRF name ("" for default)
* @param    remoteAddr          @b{(input)}  IP address of peer
* @param    scopeId             @b{(input)}  address scope
* @param    bufdName            @b{(input)}  the description of the peer
* @param    defValue            @b{(input)}  true if interval is set to a non-default value
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_NOT_FOUND    if peer not found
* @returns  OPEN_E_INTERNAL     catch for lower level failures
*
* @notes    None
*
* @supported-in-version OpEN API Version: 1.16
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerDescriptionVrfSet (openapiClientHandle_t * client_handle,
                                              open_buffdesc *vrf,
                                              open_inet_addr_t remoteAddr,
                                              uint32_t scopeId,
                                              open_buffdesc * bufdName,
                                              bool defValue);

/********************************************************************* 
* @purpose  Get the description for a given peer
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    remoteAddr          @b{(input)}  IP address of peer
* @param    scopeId             @b{(input)}  address scope
* @param    bufdName            @b{(output)} the description of the peer
*
* @returns  OPEN_E_NONE         Set successful
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_NOT_FOUND    Set if the neighbor is not found
* @returns  OPEN_E_INTERNAL     Catch for lower level failures       
* 
* @notes    None
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerDescriptionGet (openapiClientHandle_t * client_handle,
                                           open_inet_addr_t remoteAddr,
                                           uint32_t scopeId,
                                           open_buffdesc * bufdName);

/*********************************************************************
* @purpose  Get the description for a given peer
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    vrf                 @b{(input)}  VRF name ("" for default)
* @param    remoteAddr          @b{(input)}  IP address of peer
* @param    scopeId             @b{(input)}  address scope
* @param    bufdName            @b{(output)} the description of the peer
*
* @returns  OPEN_E_NONE         Set successful
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_NOT_FOUND    Set if the neighbor is not found
* @returns  OPEN_E_INTERNAL     Catch for lower level failures
*
* @notes    None
*
* @supported-in-version OpEN API Version: 1.16
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerDescriptionVrfGet (openapiClientHandle_t * client_handle,
                                              open_buffdesc *vrf,
                                              open_inet_addr_t remoteAddr,
                                              uint32_t scopeId,
                                              open_buffdesc * bufdName);


/********************************************************************* 
* @purpose  Configure a given peer to inherit configuration parameters from
*           a named peer template.
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    remoteAddr          @b{(input)}  IP address of peer
* @param    scopeId             @b{(input)}  address scope
* @param    bufdName            @b{(input)}  template name 
*           -                                maximum length @ref OPEN_BGP_MAX_TEMPLATE_NAME
* @param    defValue            @b{(input)}  true if interval is set to a non-default value
*
* @returns  OPEN_E_NONE         Set successful
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_NOT_FOUND    Set if the neighbor is not found
* @returns  OPEN_E_INTERNAL     Catch for lower level failures       
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerInheritSet (openapiClientHandle_t * client_handle,
                                       open_inet_addr_t remoteAddr,
                                       uint32_t scopeId,
                                       open_buffdesc * bufdName, bool defValue);

/********************************************************************* 
* @purpose  Get the template name for a given peer
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    remoteAddr          @b{(input)}  IP address of peer
* @param    scopeId             @b{(input)}  address scope
* @param    bufdName            @b{(input)}  the template name 
*           -                                maximum length @ref OPEN_BGP_MAX_TEMPLATE_NAME
*
* @returns  OPEN_E_NONE         Set successful
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_NOT_FOUND    Set if the neighbor is not found
* @returns  OPEN_E_INTERNAL     Catch for lower level failures       
* 
* @notes    None
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerInheritGet (openapiClientHandle_t * client_handle,
                                       open_inet_addr_t remoteAddr,
                                       uint32_t scopeId,
                                       open_buffdesc * bufdName);

/********************************************************************* 
* @purpose  Get the connection retry interval for a given BGP peer
* 
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pType               @b{(input)}   Whether to get the value configured
*                                             directly on the peer or the active
*                                             value, which may be inherited from
*                                             a peer template
* @param    remoteAddr          @b{(input)}   IP address of peer
* @param    scopeId             @b{(input)}   address scope
* @param    connRetryInterval   @b{(output)}  connection retry interval in seconds
*
* @returns  OPEN_E_NONE         Set successful
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_NOT_FOUND    Set if the neighbor is not found
* @returns  OPEN_E_INTERNAL     Catch for lower level failures       
* 
* @notes    None
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpMapPeerConnRetryIntervalGet (openapiClientHandle_t * client_handle,
                                                    OPEN_BGP_PARAM_TYPE pType,
                                                    open_inet_addr_t remoteAddr,
                                                    uint32_t scopeId,
                                                    uint32_t * connRetryInterval);

/*********************************************************************
* @purpose  Get maximum length of password
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    maxLen          @b{(output)}  string length 
*                                         maximum length @ref OPEN_BGP_MAX_PASSWORD
*
* @returns  OPEN_E_NONE        String length Get successful.
* @returns  OPEN_E_PARAM       Error in parameter passed. 
* 
* @notes    None
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpMapPeerPasswordMaxLengthGet (openapiClientHandle_t * client_handle,
                                                    uint32_t * maxLen);

/********************************************************************* 
* @purpose  Get the TCP MD5 password for a given peer
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    pType               @b{(input)}   Whether to get the value configured
*                                             directly on the peer or the active
*                                             value, which may be inherited from
*                                             a peer template
* @param    remoteAddr          @b{(input)}   IP address of peer
* @param    scopeId             @b{(input)}   address scope
* @param    bufdName            @b{(output)}  password 
*                                             Must be at least OPEN_BGP_MAX_PASSWORD + 1.
*
* @returns  OPEN_E_NONE         Set successful
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_INTERNAL     Catch for lower level failures 
* 
* @notes    None
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpMapPeerPasswordGet (openapiClientHandle_t * client_handle,
                                           OPEN_BGP_PARAM_TYPE pType,
                                           open_inet_addr_t remoteAddr,
                                           uint32_t scopeId,
                                           open_buffdesc * bufdName);

/********************************************************************* 
* @purpose  Set the TCP MD5 password for a given peer
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    remoteAddr          @b{(input)}   IP address of peer
* @param    scopeId             @b{(input)}   address scope
* @param    bufdName            @b{(output)}  TCP MD5 password to set
*           -                                 maximum length @ref OPEN_BGP_MAX_PASSWORD
* @param    defValue            @b{(input)}   true if interval is set to a non-default value
*
* @returns  OPEN_E_NONE         Set successful
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_NOT_FOUND    Set if the neighbor is not found
* @returns  OPEN_E_INTERNAL     Catch for lower level failures 
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpMapPeerPasswordSet (openapiClientHandle_t * client_handle,
                                           open_inet_addr_t remoteAddr,
                                           uint32_t scopeId,
                                           open_buffdesc * bufdName,
                                           bool defValue);

/********************************************************************* 
* @purpose  Get Decision Process History information of BGP for 
*           a given index and address family
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    af                  @b{(input)}   address family
* @param    index               @b{(input)}   index of process history information 
* @param    pHist               @b{(output)}  pointer to the history structure 
*                                             @ref open_bgpDecProcHist_t
*
* @returns  OPEN_E_NONE         Set successful
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_NOT_FOUND    Set if the neighbor is not found
* @returns  OPEN_E_INTERNAL     Catch for lower level failures 
* 
* @notes    None 
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpMapDecProcHistoryByIndexGet (openapiClientHandle_t * client_handle, 
                                                    OPEN_AF_t af,
                                                    uint32_t index,
                                                    open_bgpDecProcHist_t * pHist);

/********************************************************************* 
* @purpose  Get Decision Process History count of BGP for a given address family
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    af                  @b{(input)}   address family
* @param    numEntries          @b{(output)}  process history count
*
* @returns  OPEN_E_NONE         Set successful
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_NOT_FOUND    Set if the neighbor is not found
* @returns  OPEN_E_INTERNAL     Catch for lower level failures 
* 
* @notes    None 
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpMapDecProcHistoryCountGet (openapiClientHandle_t * client_handle,
                                                  OPEN_AF_t af,
                                                  uint32_t * numEntries);

/********************************************************************* 
* @purpose  Get flap count of given neighbor
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    remoteAddr          @b{(input)}   IP address of peer
* @param    scopeId             @b{(input)}   address scope
* @param    af                  @b{(input)}   address family
* @param    flapCount           @b{(output)}  number of times the given peer 
*                                             state changed from established to idle
*
* @returns  OPEN_E_NONE         Set successful
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_INTERNAL     Catch for lower level failures 
* 
* @notes    Calling this API will return the flap count and clears the 
*           flap count of the neighbor
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpMapPeerFlapCountGet (openapiClientHandle_t * client_handle,
                                            open_inet_addr_t remoteAddr,
                                            uint32_t scopeId, OPEN_AF_t af,
                                            uint32_t * flapCount);

/*********************************************************************
* @purpose  Get the IP address of autodetected peer on the specifed interface.
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    scopeId             @b{(input)}   scope ID of the peer's IP address 
* @param    nbrPeerAddr         @b{(output)}  IP address of detected peer
*
* @returns  OPEN_E_NONE         Get successful
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_FAIL         Get failed
* @returns  OPEN_E_ERROR        if the multiple peers are autodetected
* @returns  OPEN_E_UNAVAIL      RFC 5549 feature is not supported on this platform
* @returns  OPEN_E_INTERNAL     Catch for lower level failures 
* 
* @notes    None
*
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerAutodetectedIpGet (openapiClientHandle_t * client_handle, 
                                              uint32_t scopeId,
                                              open_inet_addr_t * nbrPeerAddr);

/*********************************************************************
* @purpose  Utility to concatenate an address scope to a peer address.
*
* @param    client_handle       @b{(input)}          client handle from registration API
* @param    bufdName            @b{(input/output)}   String representation of peer's IP address
*           -                                        Buffer must be at least 
*           -                                        @ref OPEN_BGP_PEER_ADDR_STR_LENGTH long
* @param    scopeId             @b{(input)}          address scope
*
* @returns  OPEN_E_NONE   Get successful
* @returns  OPEN_E_PARAM  Error in parameter passed

*
* @notes    prints as <peer addr> interface <if-name>.
* 
*           This is only used for BGP peer addresses, the interface
*           cannot be a loopback interface. 
*
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerAddrScopeCat (openapiClientHandle_t * client_handle,
                                         open_buffdesc * bufdName,
                                         uint32_t scopeId);

/*********************************************************************
* @purpose  Gets the BGP peer address name string length. 
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    maxLen          @b{(output)}  string length 
*                                         maximum length @ref OPEN_BGP_PEER_ADDR_STR_LENGTH
*
* @returns  OPEN_E_NONE     String length Get successful.
* @returns  OPEN_E_PARAM    Error in parameter passed. 
* 
* @notes    None
* 
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpPeerAddressStringMaxLengthGet (openapiClientHandle_t * client_handle,
                                                      uint32_t * maxLen);

/*********************************************************************
* @purpose  Get the BFD fast fallover setting to the peer.
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    peerAddr            @b{(input)}   IP address of peer
* @param    scopeId             @b{(input)}   address scope
* @param    enableBfd           @b{(output)}  @ref OPEN_TRUE is BFD session is desired
*
* @returns  OPEN_E_NONE         Get successful
* @returns  OPEN_E_NOT_FOUND    if the neighbor is not found
* @returns  OPEN_E_FAIL         any other failure
* @returns  OPEN_E_UNAVAIL      BFD feature is not supported on this platform
* @returns  OPEN_E_PARAM        Error in parameter passed
*
* @notes    The function supports auto detected IPv6 remote peer on the specifed interface.
*           To call the function for the peer, it is necessary to set remoteAddr
*           parameter to 0 IPv6 address inside it and pass the scopeId as a valid interface.
*
* @supported-in-version OpEN API Version: 1.6
*
* @end
*********************************************************************/
open_error_t openapiBgpMapPeerFalloverBfdGet (openapiClientHandle_t * client_handle,
                                              open_inet_addr_t peerAddr,
                                              uint32_t scopeId,
                                              OPEN_BOOL_t * enableBfd);

#endif /* OPENAPI_BGP_CONFIG_H_INCLUDED */

