/********************************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
********************************************************************************
*
* @filename  openapi_mfdb.h
*
* @purpose   Provide interface to Switch MFDB
*
* @component OpEN
*
* @create    06/18/2015
*
* @end
*
********************************************************************************/
#ifndef OPENAPI_MFDB_H_INCLUDED
#define OPENAPI_MFDB_H_INCLUDED

#include "openapi_common.h"

#define OPEN_MATCH_EXACT     1
#define OPEN_MATCH_GETNEXT   2
#define OPEN_MATCH_FREE      3

#define OPEN_MFDB_INTF_LIST_LENGTH   4096
#define OPEN_MFDB_VIDMAC_LENGTH      24
#define OPEN_MFDB_MAC_LENGTH         18
#define OPENAPI_MFDB_DESCR_LEN       16

/** OpEN uses this enumeration to define type of MFDB protocol */
typedef enum
{
  OPEN_MFDB_PROTOCOL_STATIC = 1,  /** MFDB potocol type is static */
  OPEN_MFDB_PROTOCOL_GMRP   = 2,  /** MFDB potocol type is GMRP */
  OPEN_MFDB_PROTOCOL_IGMP   = 3,  /** MFDB potocol type is IGMP */
  OPEN_MFDB_PROTOCOL_MLD    = 4,  /** MFDB potocol type is MLD */
  OPEN_MFDB_PROTOCOL_MVR    = 5,  /** MFDB potocol type is MVR */
  OPEN_MFDB_PROTOCOL_MMRP   = 6,  /** MFDB potocol type is MMRP */
  OPEN_MFDB_PROTOCOL_MSRP   = 7,  /** MFDB potocol type is MSRP */
  OPEN_MFDB_TOTAL_NUM_PROTOCOLS   /** total number of protocols */
                                  /** (for error checking)      */
} OPEN_MFDB_PROTOCOL_TYPE_t;

/** OpEN uses this enumeration to define type of MFDB entry */
typedef enum
{
  OPEN_MFDB_TYPE_STATIC  = 0,     /** MFDB entry type is static */
  OPEN_MFDB_TYPE_DYNAMIC = 1      /** MFDB entry type is dynamic */
} OPEN_MFDB_ENTRY_TYPE_t;

/*****************************************************************************************************
* @purpose  Checks if a VLANID-MAC entry, that uses a specific protocol, exists in the MFDB table.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    vidMac              @b{(input)}   pointer to the VID-MAC combo.
* @param    -                                 (2-byte VLAN ID + 6-byte MAC address)
* @param    -                                 buffer size must be @ref OPEN_MFDB_VIDMAC_LENGTH
* @param    protocol            @b{(input)}   the protocol type (see OPEN_MFDB_PROTOCOL_TYPE_t)
* 
* @returns  OPEN_E_NONE         VLANID-MAC entry using that protocol exists in the MFDB table.
* @returns  OPEN_E_FAIL         VLANID-MAC entry using that protocol doesn't exist in the MFDB table.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*****************************************************************************************************/
open_error_t openapiMfdbEntryProtocolGetExact(openapiClientHandle_t * client_handle,
                                              open_buffdesc * vidMac,
                                              OPEN_MFDB_PROTOCOL_TYPE_t protocol);

/*****************************************************************************************************
* @purpose  Returns the first VLANID-MAC entry.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    vidMac              @b{(output)}  pointer to the VID-MAC combo.
* @param    -                                 (2-byte VLAN ID + 6-byte MAC address)
* @param    -                                 buffer size must be @ref OPEN_MFDB_VIDMAC_LENGTH
* @param    prot                @b{(output)}  pointer to protocol type
*                                             (see OPEN_MFDB_PROTOCOL_TYPE_t)
* 
* @returns  OPEN_E_NONE         First VLANID-MAC entry returned.
* @returns  OPEN_E_FAIL         Failed to get first VLANID-MAC entry.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    The entry is indexed by VLAN ID, MAC address, and the protocol type.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*****************************************************************************************************/
open_error_t openapiMfdbEntryGetFirst(openapiClientHandle_t * client_handle,
                                      open_buffdesc * vidMac,
                                      OPEN_MFDB_PROTOCOL_TYPE_t * prot);

/*****************************************************************************************************
* @purpose  Returns the next VLANID-MAC entry.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    vidMac              @b{(input)}   pointer to the VID-MAC combo.
* @param    -                                 (2-byte VLAN ID + 6-byte MAC address)
* @param    -                                 buffer size must be @ref OPEN_MFDB_VIDMAC_LENGTH
* @param    nextProt            @b{(output)}  pointer to protocol type
*                                             (see OPEN_MFDB_PROTOCOL_TYPE_t)
* 
* @returns  OPEN_E_NONE         Next VLANID-MAC entry returned.
* @returns  OPEN_E_FAIL         Failed to get next VLANID-MAC entry.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    The entry is indexed by VLAN ID, MAC address, and the protocol type.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*****************************************************************************************************/
open_error_t openapiMfdbEntryGetNext(openapiClientHandle_t * client_handle,
                                     open_buffdesc * vidMac,
                                     OPEN_MFDB_PROTOCOL_TYPE_t * nextProt);

/*****************************************************************************************************
* @purpose  Returns the MFDB Entry data.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    vidMac              @b{(input)}   pointer to the VID-MAC combo.
* @param    -                                 (2-byte VLAN ID + 6-byte MAC address)
* @param    -                                 buffer size must be @ref OPEN_MFDB_VIDMAC_LENGTH
* @param    nextProt            @b{(input)}   protocol type
*                                             (see OPEN_MFDB_PROTOCOL_TYPE_t)
* @param    mac                 @b{(output)}  pointer to MAC address
* @param    -                                 buffer size must be @ref OPEN_MFDB_MAC_LENGTH
* @param    vlanId              @b{(output)}  pointer to VLAN ID
* @param    compId              @b{(output)}  pointer to component ID
* @param    type                @b{(output)}  pointer to type
* @param    desc                @b{(output)}  pointer to description
* @param    -                                 minimum buffer size 1
* @param    -                                 maximum buffer size @ref OPENAPI_MFDB_COMPONENT_DESCR_STRING_LEN
* @param    frwdIntMask         @b{(output)}  forward interface mask
* @param    -                                 minimum buffer size 1
* @param    -                                 maximum buffer size @ref OPEN_MFDB_INTF_LIST_LENGTH
* @param    fltIntMask          @b{(output)}  filter interface mask
* @param    -                                 minimum buffer size 1
* @param    -                                 maximum buffer size @ref OPEN_MFDB_INTF_LIST_LENGTH
* @param    absPortMask         @b{(output)}  absolute port mask
* @param    -                                 minimum buffer size 1
* @param    -                                 maximum buffer size @ref OPEN_MFDB_INTF_LIST_LENGTH
* 
* @returns  OPEN_E_NONE         MFDB Entry data returned.
* @returns  OPEN_E_FAIL         Failed to get MFDB Entry data.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*****************************************************************************************************/
open_error_t openapiMfdbEntryDataGet(openapiClientHandle_t * client_handle,
                                     open_buffdesc * vidMac,
                                     OPEN_MFDB_PROTOCOL_TYPE_t nextProt,
                                     open_buffdesc * mac,
                                     uint32_t * vlanId,
                                     OPEN_MFDB_PROTOCOL_TYPE_t * compId,
                                     OPEN_MFDB_ENTRY_TYPE_t * type,
                                     open_buffdesc * desc,
                                     open_buffdesc * frwdIntMask,
                                     open_buffdesc * fltIntMask,
                                     open_buffdesc * absPortMask);

/*****************************************************************************************************
* @purpose  Returns the next VLANID-MAC entry of the same user.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    vidMac              @b{(input)}   pointer to the VID-MAC combo.
* @param    -                                 (2-byte VLAN ID + 6-byte MAC address)
* @param    -                                 buffer size must be @ref OPEN_MFDB_VIDMAC_LENGTH
* @param    compId              @b{(input)}   component id, start
* @param    nextCompId          @b{(output)}  pointer to next component's info
* @param    type                @b{(output)}  type
* @param    desc                @b{(output)}  description
* @param    -                                 minimum buffer size 1
* @param    -                                 maximum buffer size @ref OPENAPI_MFDB_COMPONENT_DESCR_STRING_LEN
* @param    frwdPortMask        @b{(output)}  forward ports mask
* @param    -                                 minimum buffer size 1
* @param    -                                 maximum buffer size @ref OPEN_MFDB_INTF_LIST_LENGTHGTH
* @param    fltPortMask         @b{(output)}  filtering ports mask
* @param    -                                 minimum buffer size 1
* @param    -                                 maximum buffer size @ref OPEN_MFDB_INTF_LIST_LENGTH
* 
* @returns  OPEN_E_NONE         Next VLANID-MAC entry of the same user returned.
* @returns  OPEN_E_FAIL         Failed to get next VLANID-MAC entry of the same user
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    The entry is indexed by VLAN ID, MAC address, and component id.
* @notes    To get the first entry, the reference entry must be 0 (zero).
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*****************************************************************************************************/
open_error_t openapiMfdbComponentEntryGetNext(openapiClientHandle_t * client_handle,
                                              open_buffdesc * vidMac,
                                              OPEN_MFDB_PROTOCOL_TYPE_t compId,
                                              OPEN_MFDB_PROTOCOL_TYPE_t * nextCompId,
                                              OPEN_MFDB_ENTRY_TYPE_t * type,
                                              open_buffdesc * desc,
                                              open_buffdesc * frwdPortMask,
                                              open_buffdesc * fltPortMask);

/*****************************************************************************************************
* @purpose  Checks if a VLANID-MAC entry exists in the MFDB table.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    vidMac              @b{(input)}   pointer to the VID-MAC combo.
* @param    -                                 (2-byte VLAN ID + 6-byte MAC address)
* @param    -                                 buffer size must be @ref OPEN_MFDB_VIDMAC_LENGTH
* 
* @returns  OPEN_E_NONE         VLANID-MAC entry exists in the MFDB table.
* @returns  OPEN_E_FAIL         VLANID-MAC entry doesn't exist in the MFDB table.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*****************************************************************************************************/
open_error_t openapiMfdbEntryFwdIntfGetExact(openapiClientHandle_t * client_handle,
                                             open_buffdesc * vidMac);

/*****************************************************************************************************
* @purpose  Returns the forwarding interfaces of the next VLANID-MAC entry.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    vidMac              @b{(input)}   pointer to the VID-MAC combo.
* @param    -                                 (2-byte VLAN ID + 6-byte MAC address)
* @param    -                                 buffer size must be @ref OPEN_MFDB_VIDMAC_LENGTH
* @param    matchType           @b{(input)}   OPEN_MATCH_EXACT or OPEN_MATCH_GETNEXT
* @param    absMask             @b{(input)}   pointer to mask
* @param    -                                 minimum buffer size 1
* @param    -                                 maximum buffer size @ref OPEN_MFDB_INTF_LIST_LENGTH
* 
* @returns  OPEN_E_NONE         Forwarding interfaces of the next VLANID-MAC entry returned.
* @returns  OPEN_E_FAIL         Failed to get forwarding interfaces of the next VLANID-MAC entry.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    The entry is indexed by VLANID-MAC address combination.
* @notes    To get the first entry's forwarding interfaces, the reference entry's VLANID-MAC address must be 0 (zero).
* @notes    To get the forwarding interfaces of a given VLANID-MAC entry, the 'matchType' must be OPEN_MATCH_EXACT.
* @notes    To get the forwarding interfaces of all VLANID-MAC entries, the 'matchType' must be OPEN_MATCH_GETNEXT.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*****************************************************************************************************/
open_error_t openapiMfdbEntryFwdIntfGetNext(openapiClientHandle_t * client_handle,
                                            open_buffdesc * vidMac,
                                            uint32_t matchType,
                                            open_buffdesc * absMask);

/*****************************************************************************************************
* @purpose  Returns the forwarding interfaces of the next learned VLANID-MAC entry.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    vidMac              @b{(input)}   pointer to the VID-MAC combo.
* @param    -                                 (2-byte VLAN ID + 6-byte MAC address)
* @param    -                                 buffer size must be @ref OPEN_MFDB_VIDMAC_LENGTH
* @param    matchType           @b{(input)}   OPEN_MATCH_EXACT or OPEN_MATCH_GETNEXT
* @param    absMask             @b{(output)}  pointer to mask
* @param    -                                 minimum buffer size 1
* @param    -                                 maximum buffer size @ref OPEN_MFDB_INTF_LIST_LENGTH
* 
* @returns  OPEN_E_NONE         Forwarding interfaces of the next learned VLANID-MAC entry returned.
* @returns  OPEN_E_FAIL         Failed to get forwarding interfaces of the next learned VLANID-MAC entry.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    The entry is indexed by VLANID-MAC address combination.
* @notes    To get the first entry's forwarding interfaces, the reference entry's VLANID-MAC address must be 0 (zero).
* @notes    To get the forwarding interfaces of a given VLANID-MAC entry, the 'matchType' must be OPEN_MATCH_EXACT.
* @notes    To get the forwarding interfaces of all VLANID-MAC entries, the 'matchType' must be OPEN_MATCH_GETNEXT.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*****************************************************************************************************/
open_error_t openapiMfdbEntryLearntIntfGetNext(openapiClientHandle_t * client_handle,
                                               open_buffdesc * vidMac,
                                               uint32_t matchType,
                                               open_buffdesc * absMask);

/*****************************************************************************************************
* @purpose  Returns the maximum number of entries that the MFDB table can hold.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    maxEntries          @b{(output)}  pointer to the number of total entries
* 
* @returns  OPEN_E_NONE         Maximum number of entries returned.
* @returns  OPEN_E_FAIL         Failed to get maximum number of entries.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*****************************************************************************************************/
open_error_t openapiMfdbMaxTableEntriesGet(openapiClientHandle_t * client_handle,
                                           uint32_t * maxEntries);

/*****************************************************************************************************
* @purpose  Returns the most number of entries ever in the MFDB table.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    mostEntries         @b{(output)}  pointer to the number of most entries
* 
* @returns  OPEN_E_NONE         Most number of entries returned.
* @returns  OPEN_E_FAIL         Failed to get most number of entries.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*****************************************************************************************************/
open_error_t openapiMfdbMostEntriesGet(openapiClientHandle_t * client_handle,
                                       uint32_t * mostEntries);

/*****************************************************************************************************
* @purpose  Returns the number of current entries in the MFDB table.
* 
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    currEntries         @b{(output)}  pointer to the number of current entries
* 
* @returns  OPEN_E_NONE         Number of current entries returned.
* @returns  OPEN_E_FAIL         Failed to get number of current entries.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*****************************************************************************************************/
open_error_t openapiMfdbCurrEntriesGet(openapiClientHandle_t * client_handle,
                                       uint32_t * currEntries);
#endif /* OPENAPI_MFDB_H_INCLUDED */
