#
# Copyright 2017 Broadcom.
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#

import OpEN_py as OpEN
from OpENUtil import *
import socket
import struct
import getopt, sys

open = OpENUtil()

def int_to_ip(addr):
  #Convert ipv4 integer to string

  return socket.inet_ntoa(struct.pack("!I", addr))

def openIPtoStrGet(client, ipAddr):
  ipStr = ""
  max_len_p = OpEN.new_uint32_tp()
  result = OpEN.openapiIpAddressMaxStringLength(client, max_len_p)
  if result == OpEN.OPEN_E_NONE:
    max_len = OpEN.uint32_tp_value(max_len_p)
    buff_string = open.getCharBuffer(max_len)
    buff        = OpEN.open_buffdesc() 
    buff.pstart = buff_string
    buff.size   = max_len
    result = OpEN.openapiOpenIPtoStringGet(client, ipAddr, buff)
    if result == OpEN.OPEN_E_NONE and len(buff_string.cast())>0:
      ipStr = buff_string.cast()
    OpEN.delete_uint32_tp(max_len_p)
  return ipStr

def capFlagsToString(cap):
  str = ""
  if cap == 0:
    str = "None"
  else:
    str += "" if cap & OpEN.OPEN_MULTI_PROTOCOL_CAP == 0 else "MP "
    str += "" if cap & OpEN.OPEN_ROUTE_REFLECT_CAP  == 0 else "RR "
    str += "" if cap & OpEN.OPEN_COMMUNITIES_CAP    == 0 else "CM "
    str += "" if cap & OpEN.OPEN_CONFEDERATION_CAP  == 0 else "CF "
    str += "" if cap & OpEN.OPEN_ROUTE_REFRESH_CAP  == 0 else "RF"
  return str

def printPrefixStats(status, addFamily):
   print("                           Inbound       Outbound")
   tmp1_p = OpEN.new_uint32_tp()
   tmp2_p = OpEN.new_uint32_tp()
   tmp1 = OpEN.uint32_tArray_getitem(status.inPrefix, addFamily)
   tmp2 = OpEN.uint32_tArray_getitem(status.outPrefix, addFamily)
   print("Prefixes Advertised {0:13}{1:15}".format(tmp1, tmp2))
   OpEN.delete_uint32_tp(tmp1_p)
   OpEN.delete_uint32_tp(tmp2_p)

   tmp1_p = OpEN.new_uint32_tp()
   tmp2_p = OpEN.new_uint32_tp()
   tmp1 = OpEN.uint32_tArray_getitem(status.inWithdraws, addFamily)
   tmp2 = OpEN.uint32_tArray_getitem(status.outWithdraws, addFamily)
   print("Prefixes Withdrawn  {0:13}{1:15}".format(tmp1, tmp2))
   OpEN.delete_uint32_tp(tmp1_p)
   OpEN.delete_uint32_tp(tmp2_p)

   tmp1_p = OpEN.new_uint32_tp()
   tmp2_p = OpEN.new_uint32_tp()
   tmp1 = OpEN.uint32_tArray_getitem(status.inPfxCurrent, addFamily)
   tmp2 = OpEN.uint32_tArray_getitem(status.outPfxCurrent, addFamily)
   print("Prefixes Current    {0:13}{1:15}".format(tmp1, tmp2))
   OpEN.delete_uint32_tp(tmp1_p)
   OpEN.delete_uint32_tp(tmp2_p)

   tmp1_p = OpEN.new_uint32_tp()
   tmp1 = OpEN.uint32_tArray_getitem(status.inPfxAccepted, addFamily)
   print("Prefixes Accepted   {0:13}{1:>15}".format(tmp1,"N/A"))
   OpEN.delete_uint32_tp(tmp1_p)

   tmp1_p = OpEN.new_uint32_tp()
   tmp1 = OpEN.uint32_tArray_getitem(status.inPfxRejected, addFamily)
   print("Prefixes Rejected   {0:13}{1:>15}".format(tmp1, "N/A"))
   OpEN.delete_uint32_tp(tmp1_p)

   tmp1_p = OpEN.new_uint32_tp()
   tmp2_p = OpEN.new_uint32_tp()
   tmp1 = OpEN.uint32_tArray_getitem(status.inMaxNlriPerUpdate, addFamily)
   tmp2 = OpEN.uint32_tArray_getitem(status.outMaxNlriPerUpdate, addFamily)
   print("Max NLRI per Update {0:13}{1:15}".format(tmp1, tmp2))
   OpEN.delete_uint32_tp(tmp1_p)
   OpEN.delete_uint32_tp(tmp2_p)

   tmp1_p = OpEN.new_uint32_tp()
   tmp2_p = OpEN.new_uint32_tp()
   tmp1 = OpEN.uint32_tArray_getitem(status.inMinNlriPerUpdate, addFamily)
   tmp2 = OpEN.uint32_tArray_getitem(status.outMinNlriPerUpdate, addFamily)
   print("Min NLRI per Update {0:13}{1:15}".format(tmp1, tmp2))
   OpEN.delete_uint32_tp(tmp1_p)
   OpEN.delete_uint32_tp(tmp2_p)

def statusToStr(peerStatus):
    ret = "ILG"

    if peerStatus == OpEN.OPEN_BGP_PEER_STATE_ILG:
      ret = "ILG"
    elif peerStatus == OpEN.OPEN_BGP_PEER_STATE_IDLE:
      ret = "IDLE"
    elif peerStatus == OpEN.OPEN_BGP_PEER_STATE_CONNECT:
      ret = "CONNECT"
    elif peerStatus == OpEN.OPEN_BGP_PEER_STATE_ACTIVE:
      ret = "ACTIVE"
    elif peerStatus == OpEN.OPEN_BGP_PEER_STATE_OPENSENT:
      ret = "OPENSENT"
    elif peerStatus == OpEN.OPEN_BGP_PEER_STATE_OPENCONFIRM:
      ret = "OPENCONFIRM"
    elif peerStatus == OpEN.OPEN_BGP_PEER_STATE_ESTABLISHED:
      ret = "ESTABLISHED"
    return ret

def performAction(client, addFamily):
    # Demonstrate OpEN usage for BGP APIs

    # Fetch IPv4 Routing
    v4rtrAdminMode_p = OpEN.new_OPEN_CONTROL_tp()
    v4rtrAdminMode_rc = OpEN.openapiRtrAdminModeGet(client, OpEN.OPEN_AF_INET, v4rtrAdminMode_p)
    if v4rtrAdminMode_rc == OpEN.OPEN_E_NONE:
       v4rtrAdminMode = OpEN.OPEN_CONTROL_tp_value(v4rtrAdminMode_p)
    OpEN.delete_OPEN_CONTROL_tp(v4rtrAdminMode_p)

    # Fetch IPv6 Routing
    v6rtrAdminMode_p = OpEN.new_OPEN_CONTROL_tp()
    v6rtrAdminMode_rc = OpEN.openapiRtrAdminModeGet(client, OpEN.OPEN_AF_INET6, v6rtrAdminMode_p)
    if v6rtrAdminMode_rc == OpEN.OPEN_E_NONE:
       v6rtrAdminMode = OpEN.OPEN_CONTROL_tp_value(v6rtrAdminMode_p)
    OpEN.delete_OPEN_CONTROL_tp(v6rtrAdminMode_p)

    # Fetch BGP Admin Mode
    bgpAdminMode_p = OpEN.new_OPEN_CONTROL_tp()
    bgpAdminMode_rc = OpEN.openapiBgpAdminModeGet(client, bgpAdminMode_p)
    if bgpAdminMode_rc == OpEN.OPEN_E_NONE:
       bgpAdminMode = OpEN.OPEN_CONTROL_tp_value(bgpAdminMode_p)
    OpEN.delete_OPEN_CONTROL_tp(bgpAdminMode_p)

    # Fetch BGP Router ID
    bgpLocalId_p = OpEN.new_uint32_tp()
    bgpLocalId = 0
    bgpLocalId_rc = OpEN.openapiBgpLocalIdGet(client, bgpLocalId_p)
    if bgpLocalId_rc == OpEN.OPEN_E_NONE:
       bgpLocalId = OpEN.uint32_tp_value(bgpLocalId_p)
    OpEN.delete_uint32_tp(bgpLocalId_p)

    bgp_valid_status = v4rtrAdminMode_rc | v6rtrAdminMode_rc | bgpAdminMode_rc | bgpLocalId_rc
    if not bgp_valid_status == OpEN.OPEN_E_NONE:
       print("\rWarning:  Unknown error - BGP data is unavailable at this time.")
    else:
      routingEnabled = OpEN.OPEN_FALSE
      if v4rtrAdminMode == OpEN.OPEN_ENABLE or v6rtrAdminMode == OpEN.OPEN_ENABLE:
        routingEnabled = OpEN.OPEN_TRUE

      valid = True
      if bgpAdminMode == OpEN.OPEN_DISABLE or not routingEnabled or bgpLocalId == 0:
         valid = False
         print("\rWarning:  BGP is not running. Further data is unavailable.")

      i = 0
      if routingEnabled == OpEN.OPEN_FALSE:
         i += 2
         print("\r   %d) IP routing is not enabled.".format(i))
      if bgpAdminMode == OpEN.OPEN_DISABLE:
         i += 2
         print("\r   %d) BGP is administratively disabled.".format(i))
      if bgpLocalId == 0:
         i += 2
         print("\r   %d) No BGP router ID is configured.".format(i));

    if not valid:
       print("\n")
       sys.exit(1)

    prevRemoteAddr = OpEN.open_inet_addr_t()
    prevRemoteAddr.family = addFamily
    remoteAddr = OpEN.open_inet_addr_t()
    remoteAddr.family = addFamily
    prevScopeId = 0
    scopeId_p = OpEN.new_uint32_tp()
    while True:
      print("\n\n")
      rc = OpEN.openapiBgpMapPeerRemoteAddrGetNext(client, addFamily, prevRemoteAddr,
                                                   prevScopeId, remoteAddr, scopeId_p)
      if OpEN.OPEN_E_NONE == rc:
         status_p = OpEN.new_open_bgpPeerStatus_tp()
         scopeId = OpEN.uint32_tp_value(scopeId_p)
         rc1 = OpEN.openapiBgpMapPeerStatusGet(client, remoteAddr, scopeId, addFamily, status_p)
         if rc1 == OpEN.OPEN_E_NONE:
           remoteAS_p = OpEN.new_uint32_tp()
           result = OpEN.openapiBgpPeerRemoteASGet(client, remoteAddr, scopeId, remoteAS_p)
           if result == OpEN.OPEN_E_NONE:
             remoteAS = OpEN.uint32_tp_value(remoteAS_p)
           else:
             remoteAS = 0
           OpEN.delete_uint32_tp(remoteAS_p)
           status = OpEN.open_bgpPeerStatus_tp_value(status_p)

           max_desc_len_p = OpEN.new_uint32_tp()
           result = OpEN.openapiBgpPeerDescriptionNameMaxLengthGet(client, max_desc_len_p)
           if result == OpEN.OPEN_E_NONE:
              max_desc_len = OpEN.uint32_tp_value(max_desc_len_p) + 1
              desc_buff_string = open.getCharBuffer(max_desc_len)
              desc_buff        = OpEN.open_buffdesc()
              desc_buff.pstart = desc_buff_string
              desc_buff.size   = max_desc_len
              result = OpEN.openapiBgpPeerDescriptionGet(client, remoteAddr, scopeId, desc_buff)
              if result == OpEN.OPEN_E_NONE and len(desc_buff_string.cast())>0:
                print("Description ....................................... %s".format(desc_buff_string.cast()))
              OpEN.delete_uint32_tp(max_desc_len_p)

           peer_addr_max_len_p = OpEN.new_uint32_tp()
           peer_addr_max_len = 0
           result = OpEN.openapiBgpPeerAddressStringMaxLengthGet(client, peer_addr_max_len_p)
           if result == OpEN.OPEN_E_NONE:
              peer_addr_max_len = OpEN.uint32_tp_value(peer_addr_max_len_p)
           OpEN.delete_uint32_tp(peer_addr_max_len_p)

           buff        = OpEN.open_buffdesc()
           buff.size   = peer_addr_max_len

           tmp_p = OpEN.new_OPEN_BOOL_tp()
           ipStr = ""
           rc = OpEN.openapiIsInetAddrZero(client, remoteAddr, tmp_p)
           if OpEN.OPEN_E_NONE == rc:
              isZero = OpEN.OPEN_BOOL_tp_value(tmp_p)
              if not isZero == OpEN.OPEN_TRUE:
                 nbrPeerAddr = OpEN.open_inet_addr_t()
                 nbrPeerAddr.family = addFamily
                 nbrRc = OpEN.openapiBgpPeerAutodetectedIpGet(client, scopeId, nbrPeerAddr)
                 if OpEN.OPEN_E_NONE == nbrRc:
                    ipStr = openIPtoStrGet(client, nbrPeerAddr) + " (autodetected)"
                    print("Remote Address .................................... {0}".format(ipStr))
                    print("Autodetect status ................................. Peer is detected")
                 elif OpEN.OPEN_E_ERROR == nbrRc:
                    ipStr = "Unresolved"
                    print("Remote Address .................................... {0}".format(ipStr))
                    print("Autodetect status ................................. Multiple peers are detected")
                 else:
                    ipStr = "Unresolved"
                    print("Remote Address .................................... {0}".format(ipStr))
                    print("Autodetect status ................................. Peer is not detected")
              else:
                 ipStr = openIPtoStrGet(client, remoteAddr)
                 print("Remote Address .................................... {0}".format(ipStr))
           OpEN.delete_OPEN_BOOL_tp(tmp_p)

           if scopeId:
             max_desc_len_p = OpEN.new_uint32_tp()
             result = OpEN.openapiIntfNameSizeGet(client, max_desc_len_p)
             if result == OpEN.OPEN_E_NONE:
                max_desc_len = OpEN.uint32_tp_value(max_desc_len_p) + 1
                desc_buff_string = open.getCharBuffer(max_desc_len)
                desc_buff        = OpEN.open_buffdesc()
                desc_buff.pstart = desc_buff_string
                desc_buff.size   = max_desc_len
                result = OpEN.openapiRtrIntfNameGet(client, scopeId, desc_buff)
                if result == OpEN.OPEN_E_NONE and len(desc_buff_string.cast())>0:
                  print("Interface ......................................... {0}".format(desc_buff_string.cast()))
                OpEN.delete_uint32_tp(max_desc_len_p)
           print("Remote AS ......................................... {0}".format(remoteAS))
           print("Peer ID ........................................... {0}".format(int_to_ip(status.peerRouterId)))

           tmp_p = OpEN.new_OPEN_BGP_PEER_STATE_tp()
           result = OpEN.openapiBgpPeerAdminStatusGet(client, OpEN.OPEN_BGP_GET_FINAL, remoteAddr, scopeId, tmp_p)
           if result == OpEN.OPEN_E_NONE:
              tmp = OpEN.OPEN_BGP_PEER_STATE_tp_value(tmp_p)
              if tmp == OpEN.OPEN_BGP_START:
                admin_status = "START"
              elif tmp == OpEN.OPEN_BGP_STOP:
                admin_status = "STOP"
              else:
                admin_status = "STOP"
           OpEN.delete_OPEN_BGP_PEER_STATE_tp(tmp_p)

           print("Peer Admin Status ................................. {0}".format(admin_status))
           print("Peer State ........................................ {0}".format(statusToStr(status.peerState)))

           ipStr = ""
           tmp_p = OpEN.new_OPEN_BOOL_tp()
           rc = OpEN.openapiIsInetAddrZero(client, status.localAddr, tmp_p)
           if OpEN.OPEN_E_NONE == rc:
              isZero = OpEN.OPEN_BOOL_tp_value(tmp_p)
              if isZero == OpEN.OPEN_TRUE:
                 ipStr = openIPtoStrGet(client, status.localAddr)
           else:
              ipStr = "None"
           OpEN.delete_OPEN_BOOL_tp(tmp_p)

           print("Local Interface Address ........................... {0}".format(ipStr))
           print("Local Port ........................................ {0}".format(status.localTcpPort))
           print("Remote Port ....................................... {0}".format(status.remoteTcpPort))

           tmp_p = OpEN.new_uint32_tp()
           result = OpEN.openapiBgpMapPeerConnRetryIntervalGet(client, OpEN.OPEN_BGP_GET_FINAL,
                                                               remoteAddr, scopeId, tmp_p)
           if result == OpEN.OPEN_E_NONE:
              tmp = OpEN.uint32_tp_value(tmp_p)
              print("Connection Retry Interval ......................... {0} sec".format(tmp))
           OpEN.delete_uint32_tp(tmp_p)

           print("Neighbor Capabilities ............................. {0}".format(capFlagsToString(status.capabilities)))

           unicastSupport4 = ""
           tmpb_p = OpEN.new_bool_tp()
           ipv4Active = False
           result = OpEN.openapiBgpPeerActivateGet(client, OpEN.OPEN_BGP_GET_FINAL, remoteAddr,
                                                   scopeId, OpEN.OPEN_AF_INET, tmpb_p)
           if result == OpEN.OPEN_E_NONE:
              activate = OpEN.bool_tp_value(tmpb_p)
              ipv4Active = activate
              if activate == True:
                 if OpEN.OPEN_BOOL_tArray_getitem(status.remoteAfiEnabled, OpEN.OPEN_AF_INET) == OpEN.OPEN_TRUE:
                    unicastSupport4 = "Both"
                 else:
                    unicastSupport4 = "Sent"
              else:
                 if not OpEN.OPEN_BOOL_tArray_getitem(status.remoteAfiEnabled, OpEN.OPEN_AF_INET) == OpEN.OPEN_FALSE:
                   unicastSupport4 = "Received"
                 else:
                   unicastSupport4 = "None"
           else:
              unicastSupport4 = "Error"
           OpEN.delete_bool_tp(tmpb_p)
           print("IPv4 Unicast Support .............................. {0}".format(unicastSupport4))

           unicastSupport6 = ""
           tmpb_p = OpEN.new_bool_tp()
           ipv6Active = False
           result = OpEN.openapiBgpPeerActivateGet(client, OpEN.OPEN_BGP_GET_FINAL, remoteAddr,
                                                   scopeId, OpEN.OPEN_AF_INET6, tmpb_p)
           if result == OpEN.OPEN_E_NONE:
              activate = OpEN.bool_tp_value(tmpb_p)
              ipv6Active = activate
              if activate == True:
                 if not OpEN.OPEN_BOOL_tArray_getitem(status.remoteAfiEnabled, OpEN.OPEN_AF_INET6) == OpEN.OPEN_FALSE:
                    unicastSupport6 = "Both"
                 else:
                    unicastSupport6 = "Sent"
              else:
                 if not OpEN.OPEN_BOOL_tArray_getitem(status.remoteAfiEnabled, OpEN.OPEN_AF_INET6) == OpEN.OPEN_FALSE:
                   unicastSupport6 = "Received"
                 else:
                   unicastSupport6 = "None"
           else:
              unicastSupport6 = "Error"
           OpEN.delete_bool_tp(tmpb_p)
           print("IPv6 Unicast Support .............................. {0}".format(unicastSupport6))

           max_desc_len_p = OpEN.new_uint32_tp()
           desc = "None"
           result = OpEN.openapiBgpTemplateNameMaxLengthGet(client, max_desc_len_p)
           if result == OpEN.OPEN_E_NONE:
              max_desc_len = OpEN.uint32_tp_value(max_desc_len_p) + 1
              desc_buff_string = open.getCharBuffer(max_desc_len)
              desc_buff        = OpEN.open_buffdesc()
              desc_buff.pstart = desc_buff_string
              desc_buff.size   = max_desc_len
              result = OpEN.openapiBgpPeerInheritGet(client, remoteAddr, scopeId, desc_buff)
              if result == OpEN.OPEN_E_NONE and len(desc_buff_string.cast())>0:
                desc = desc_buff_string.cast()
              OpEN.delete_uint32_tp(max_desc_len_p)
           print("Template Name ..................................... {0}".format(desc))

           tmp_p = OpEN.new_uint32_tp()
           result = OpEN.openapiBgpPeerUpdateSourceGet(client, OpEN.OPEN_BGP_GET_FINAL,
                                                       remoteAddr, scopeId, tmp_p)
           if result == OpEN.OPEN_E_NONE:
             tmp = OpEN.uint32_tp_value(tmp_p)
             if tmp == 0:
                str = "None"
             else:
               max_len_p = OpEN.new_uint32_tp()
               result = OpEN.openapiIntfNameMaxLengthGet(client, max_len_p)
               if result == OpEN.OPEN_E_NONE:
                  max_len = OpEN.uint32_tp_value(max_len_p)
                  buff_string = open.getCharBuffer(max_len)
                  buff        = OpEN.open_buffdesc()
                  buff.pstart = buff_string
                  buff.size   = max_len
                  result = OpEN.openapiIntfNameGet(client, tmp, buff)
                  if result == OpEN.OPEN_E_NONE:
                    str = buff_string.cast()
               OpEN.delete_uint32_tp(max_len_p)
             print("Update Source ..................................... {0}".format(str))
           OpEN.delete_uint32_tp(tmp_p)

           tmp_p = OpEN.new_uint32_tp()
           time = 0
           result = OpEN.openapiBgpPeerHoldTimeConfiguredGet(client, OpEN.OPEN_BGP_GET_FINAL,
                                                             remoteAddr, scopeId, tmp_p)
           if result == OpEN.OPEN_E_NONE:
             time = OpEN.uint32_tp_value(tmp_p)
           OpEN.delete_uint32_tp(tmp_p)
           timeStr = None if time == 4294967295 else "{0} sec".format(time)
           print("Configured Hold Time .............................. {0}".format(timeStr))

           tmp_p = OpEN.new_uint32_tp()
           time = 0
           result = OpEN.openapiBgpPeerKeepAliveConfiguredGet(client, OpEN.OPEN_BGP_GET_FINAL, remoteAddr, scopeId, tmp_p)
           if result == OpEN.OPEN_E_NONE:
             time = OpEN.uint32_tp_value(tmp_p)
           OpEN.delete_uint32_tp(tmp_p)
           timeStr = None if time == 4294967295 else "{0} sec".format(time)
           print("Configured Keep Alive Time ........................ {0}".format(timeStr))

           if  status.peerState >= OpEN.OPEN_BGP_PEER_STATE_OPENCONFIRM:
             print("Negotiated Hold Time .............................. {0} sec".format(status.negHoldTime))
             print("Negotiated Keep Alive Time ........................ {0} sec".format(status.negKeepaliveTime))

           max_desc_len_p = OpEN.new_uint32_tp()
           passwd = "None"
           result = OpEN.openapiBgpMapPeerPasswordMaxLengthGet(client, max_desc_len_p)
           if result == OpEN.OPEN_E_NONE:
              max_desc_len = OpEN.uint32_tp_value(max_desc_len_p) + 1
              desc_buff_string = open.getCharBuffer(max_desc_len)
              desc_buff        = OpEN.open_buffdesc()
              desc_buff.pstart = desc_buff_string
              desc_buff.size   = max_desc_len
              result = OpEN.openapiBgpMapPeerPasswordGet(client, OpEN.OPEN_BGP_GET_FINAL, remoteAddr, scopeId, desc_buff)
              if result == OpEN.OPEN_E_NONE and len(desc_buff_string.cast())>0:
                passwd = desc_buff_string.cast()
              OpEN.delete_uint32_tp(max_desc_len_p)
           print("MD5 Password ...................................... {0}".format(passwd))

           if not status.lastErrorCode == 0:
              if not status.lastErrorInbound == 0:
                txRx = "Received"
              else:
                txRx = "Sent"
           else:
              txRx = ""    

           str = OpEN.getStringFromUcharArray(status.lastErrorString)
           #txRx = "{0}".format(txRx) if len(txRx) > 0 else ""
           print("Last Error ........................................ {0} {1}".format(str, txRx))

           str = OpEN.getStringFromUcharArray(status.lastSuberrorString)
           print("Last SubError ..................................... {0}".format(str))

           timeSinceLastErr = "Never"
           if not status.lastErrorCode == 0:
             let = status.lastErrorTime
             mm, ss = divmod(let, 60)
             hh, mm = divmod(mm, 60)
             dd, hh = divmod(hh, 24)
             timeSinceLastErr = "{0} days {1} hrs {2} mins {3} secs".format(dd, hh, mm, ss)
           print("Time Since Last Error ............................. {0}".format(timeSinceLastErr))

           if not status.attrErr.duplicateAttr == 0:
             print("Path with duplicate attribute ..................... {0}".format(status.attrErr.duplicateAttr))

           if not status.attrErr.wellKwnOptConflict == 0:
             print("Path with well-known/optional conflict ............ {0}".format(status.attrErr.wellKwnOptConflict))

           if not status.attrErr.transFlag == 0:
             print("Transitive flag not set on transitive attr ........ {0}".format(status.attrErr.transFlag))

           if not status.attrErr.mandNonTransOrPart == 0:
             print("Mandatory attribute non-transitive or partial ..... {0}".format(status.attrErr.mandNonTransOrPart))

           if not status.attrErr.optNonTransPart == 0:
             print("Optional attribute non-transitive and partial ..... {0}".format(status.attrErr.optNonTransPart))

           if not status.attrErr.attrTooLong == 0:
             print("Path attribute too long ........................... {0}".format(status.attrErr.attrTooLong))

           if not status.attrErr.attrLenError == 0:
             print("Path attribute length error ....................... {0}".format(status.attrErr.attrLenError))

           if not status.attrErr.originInvalid == 0:
             print("Invalid ORIGIN code ............................... {0}".format(status.attrErr.originInvalid))

           if not status.attrErr.wrongFirstAs == 0:
             print("Unexpected first ASN in AS path ................... {0}".format(status.attrErr.wrongFirstAs))

           if not status.attrErr.invalidAsPathType == 0:
             print("Invalid AS path segment type ...................... {0}".format(status.attrErr.invalidAsPathType))

           if not status.attrErr.invalidNextHop == 0:
             print("Invalid BGP NEXT HOP .............................. {0}".format(status.attrErr.invalidNextHop))

           if not status.attrErr.badNextHop == 0:
             print("Bad BGP NEXT HOP .................................. {0}".format(status.attrErr.badNextHop))

           if not status.attrErr.invalidAggregator == 0:
             print("Invalid AGGREGATOR attribute ...................... {0}".format(status.attrErr.invalidAggregator))

           if not status.attrErr.unknownWellKnown == 0:
             print("Unrecognized well-known path attribute............. {0}".format(status.attrErr.unknownWellKnown))

           if not status.attrErr.missingMandatory == 0:
             print("Missing mandatory path attribute .................. {0}".format(status.attrErr.missingMandatory))

           if not status.attrErr.missingLocPref == 0:
             print("Missing LOCAL PREF attribute ...................... {0}".format(status.attrErr.missingLocPref))

           if not status.attrErr.invalidNetField == 0:
             print("Invalid prefix in UPDATE NLRI ..................... {0}".format(status.attrErr.invalidNetField))

           if not status.attrErr.rxAsPathTooLong == 0:
             print("Received AS Path Too Long ......................... {0}".format(status.attrErr.rxAsPathTooLong))

           if not status.attrErr.origIdFromExtPeer == 0:
             print("Received ORIGINATOR_ID from external peer ......... {0}".format(status.attrErr.origIdFromExtPeer))

           if not status.attrErr.clusterListFromExtPeer == 0:
             print("Received CLUSTER_LIST from external peer .......... {0}".format(status.attrErr.clusterListFromExtPeer))

           print("Established Transitions ........................... {0}".format(status.estTransitions))
           print("Flap Count ........................................ {0}".format(status.flapCount))

           eet = status.estTime
           mm, ss = divmod(eet, 60)
           hh, mm = divmod(mm, 60)
           dd, hh = divmod(hh, 24)
           estTime = "{0} days {1} hrs {2} mins {3} secs".format(dd, hh, mm, ss)
           print("Established Time .................................. {0}".format(estTime))

           timeSinceLastUpdate = "No UPDATE received"
           if status.rxUpdate > 0:
             lastUpdate = status.lastUpdate
             mm, ss = divmod(lastUpdate, 60)
             hh, mm = divmod(mm, 60)
             dd, hh = divmod(hh, 24)
             timeSinceLastUpdate = "{0} days {1} hrs {2} mins {3} secs".format(dd, hh, mm, ss)
           print("Time Since Last Update ............................ {0}".format(timeSinceLastUpdate))

           if addFamily == OpEN.OPEN_AF_INET:
             v4outBoundUpdGrp = ""
             tmp_p = OpEN.new_uint16_tp()
             if remoteAddr.family == OpEN.OPEN_AF_INET:
                tmp = OpEN.uint16_tArray_getitem(status.updateGroupId, OpEN.OPEN_AF_INET)
                if tmp == OpEN.OPEN_BGP_UPD_GROUP_NONE:
                  v4outBoundUpdGrp = "None"
                else:
                  v4outBoundUpdGrp = "{0}".format(tmp) 
             print("IPv4 Outbound Update Group ........................ {0}".format(v4outBoundUpdGrp))
             OpEN.delete_uint16_tp(tmp_p)

           v6outBoundUpdGrp = "None"
           tmp_p = OpEN.new_uint16_tp()
           if remoteAddr.family == OpEN.OPEN_AF_INET6:
              tmp = OpEN.uint16_tArray_getitem(status.updateGroupId, OpEN.OPEN_AF_INET6)
              if tmp == OpEN.OPEN_BGP_UPD_GROUP_NONE:
                v6outBoundUpdGrp = "None"
              else:
                v6outBoundUpdGrp = "{0}".format(tmp)
           print("IPv6 Outbound Update Group ........................ {0}".format(v6outBoundUpdGrp))
           OpEN.delete_uint16_tp(tmp_p)

           tmp_p = OpEN.new_OPEN_BOOL_tp()
           bfdStr = ""
           rc = OpEN.openapiBgpMapPeerFalloverBfdGet(client, remoteAddr, scopeId, tmp_p)
           if OpEN.OPEN_E_NONE == rc:
              isZero = OpEN.OPEN_BOOL_tp_value(tmp_p)
              bfdStr = "Yes" if isZero == OpEN.OPEN_TRUE else "No"
           OpEN.delete_OPEN_BOOL_tp(tmp_p)
           print("BFD Enabled to Detect Fast Fallover ............... {0}".format(bfdStr))

           print("\n             Open    Update    Keepalive    Notification    Refresh    Total\n")
           print("Msgs Sent{0:8} {1:9} {2:12} {3:15} {4:10} {5:8}".format(status.txOpen, status.txUpdate, status.txKeepalive, status.txNotif, status.txRefresh, (status.txOpen + status.txUpdate + status.txKeepalive + status.txNotif + status.txRefresh))) 
           print("Msgs Rcvd{0:8} {1:9} {2:12} {3:15} {4:10} {5:8}\n".format(status.rxOpen, status.rxUpdate, status.rxKeepalive, status.rxNotif, status.rxRefresh, (status.rxOpen + status.rxUpdate + status.rxKeepalive + status.rxNotif + status.rxRefresh)))
           print("Received UPDATE Queue Size: {0} bytes. High: {1}  Limit: {2} Drops: {3}".format(status.updateQueueLen,status.updateQueueHigh,status.updateQueueLimit, status.updateQueueDrops))

           if (remoteAddr.family == OpEN.OPEN_AF_INET):
              print("\nIPv4 Prefix Statistics:\n")
              printPrefixStats(status, addFamily)

           if ipv6Active:
              print("\nIPv6 Prefix Statistics:\n")
              printPrefixStats(status, addFamily)

         OpEN.delete_open_bgpPeerStatus_tp(status_p)
      else:
         break
      prevRemoteAddr = remoteAddr
      prevScopeId = scopeId
      scopeId = OpEN.uint32_tp_value(scopeId_p)
    OpEN.delete_uint32_tp(scopeId_p)

def usage(name):
    print("bgp_neighbors [-4 | -6 | -h]\n")
    print("-4: display BGP IPv4 peer information")
    print("-6: display BGP IPv6 peer information")
    print("-h: display this message, and exit")

def main():
    do4 = False
    do6 = False

    try:
        opts, args = getopt.getopt(sys.argv[1:], "46h")
    except getopt.GetoptError as err:
        print str(err)  
        usage(sys.argv[0])
        sys.exit(2)
    output = None
    verbose = False
    for o, a in opts:
        if o == "-4":
          do4 = True
        elif o == "-6":
          do6 = True
        elif o in ("-h"):
            usage(sys.argv[0])
            sys.exit()
        else:
            assert False, "unhandled option"

    open = OpENUtil()
    ret = open.connect("bgp_neighbors")
    if ret == OpEN.OPEN_E_NONE :
      open.getNetworkOSVersion()
      open.getAPIVersion()
      client = open.get_client()

      if do4:
        performAction(client, OpEN.OPEN_AF_INET)
      if do6:
        performAction(client, OpEN.OPEN_AF_INET6)

if __name__ == "__main__":
    main()

