#
# Chef Cookbook   : netdev_icos
# File            : provider/l2_interface.rb
#
# Copyright 2016 Broadcom.
#
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#
require "OpEN"
require "OpENUtil"

def whyrun_supported?
  true
end

action :create do
  intf_exists, intf_num = get_intf_num('netdev-icos-l2_interface', new_resource.name, OpEN::OPEN_INTF_TYPE_PHY)
  if !intf_exists && intf_num != -1
    converge_by("create l2interface #{@new_resource.name}") do
      msg = "l2 interface can not be created. Only its properties can be set"
      log_msg("info", msg)
      modify_l2interface(intf_num)
    end
  elsif intf_exists && intf_num != -1 
    converge_by("edit l2interface #{@new_resource.name}") do
      modify_l2interface(intf_num)
    end
  else
    msg = "Failed to create l2interface #{@new_resource.name}, not a valid l2interface"
    log_msg("info", msg)
  end
end

action :delete do
  if @current_resource.exists
    converge_by("reset l2interface #{@current_resource.name}") do
      reset_l2interface
    end
  else
    msg = "l2interface doesn't exist or not a valid l2inteface, nothing to delete/reset"
    log_msg("info", msg)
  end
end

def reset_l2interface
  intf_exists, intf_num = get_intf_num('netdev-icos-l2_interface', new_resource.name, OpEN::OPEN_INTF_TYPE_PHY)

  if intf_exists && intf_num != -1
    msg = "l2 interface '#{@new_resource.name}' can not be deleted. Only its properties can be reset"
    log_msg("info", msg)

    open = OpENUtil.new()
    conn_ret = open.connect("netdev-icos-l2_interface")
    if conn_ret == OpEN::OPEN_E_NONE
      client = open.client
      tagging = OpEN::OPEN_DISABLE
      participation_mode = OpEN::OPEN_VLAN_PARTICIPATION_MODE_AUTO

      # Get the vlan
      vlan   = 0
      vlan_p = OpEN.new_uint32_tp()

      ret = OpEN::openapiVlanNextGet(client, vlan, vlan_p)
      while ret == OpEN::OPEN_E_NONE
        vlan = OpEN.uint32_tp_value(vlan_p)
        # Reset the vlan participation and tagging
        if @current_resource.tagged_vlans
          @current_resource.tagged_vlans.each do |name|
            if vlan.to_s == name.to_s
              ret1 = OpEN::openapiVlanIfTaggingSet(client, intf_num, vlan, tagging)
              ret2 = OpEN::openapiVlanIfParticipationSet(client, vlan, intf_num, participation_mode)
              if ret1 != OpEN::OPEN_E_NONE || ret2 != OpEN::OPEN_E_NONE
                msg = "Failed to reset L2 interface '#{@new_resource.name}' tagging, returned value = '#{ret1}', '#{ret2}'"
                log_msg("info", msg)
              end
            end
          end
        end

        # Reset the vlan participation and pvid
        if vlan.to_s == @current_resource.untagged_vlan
          ret1 = OpEN::openapiVlanIfParticipationSet(client, vlan, intf_num, participation_mode)
          ret2 = OpEN::openapiVlanIfPvidSet(client, intf_num, 1)
          if ret1 != OpEN::OPEN_E_NONE || ret2 != OpEN::OPEN_E_NONE
            msg = "Failed to reset L2 interface '#{@new_resource.name}' untagged vlan, returned value = '#{ret1}', '#{ret2}'"
            log_msg("info", msg)
          end
        end

        ret = OpEN::openapiVlanNextGet(client, vlan, vlan_p)
      end # Loop through all the vlans

      # Set the interface description
      buffdesc        = OpEN::Open_buffdesc.new
      buffdesc.size   = @current_resource.description.length + 1
      buffdesc.pstart = open.getCharBuffer(buffdesc.size, "")
      ret = OpEN::openapiIfDescrSet(client, intf_num, buffdesc)
      if ret != OpEN::OPEN_E_NONE
        msg = "Failed to set L2 interface '#{new_resource.name}' description, returned value = '#{ret}'"
        log_msg("info", msg)
      end

      #reset acceptframe
      accept_frame_type = OpEN::OPEN_DOT1Q_ADMIT_ALL
      ret = OpEN::openapiVlanIfAcceptFrameTypeSet(client, intf_num, accept_frame_type)
      if ret != OpEN::OPEN_E_NONE
        msg = "Failed to reset L2 interface '#{new_resource.name}' accept frame type, returned value = '#{ret}'"
        log_msg("info", msg)
      end
      OpEN.delete_uint32_tp(vlan_p)
    else
      msg = "Failed to connect to 'netdev-icos-l2_interface'"
      log_msg("info", msg)
    end
    open.terminate()

  else
    msg = "Failed to reset l2interface '#{@new_resource.name}', not a valid l2interface"
    log_msg("info", msg)
  end
end

def load_current_resource
  msg = "Loading current resource '#{@new_resource.name}'"
  log_msg("info", msg)
  interface = {}

  @current_resource = Chef::Resource::NetdevL2Interface.new(@new_resource.name)
  @current_resource.exists = false
  
  intf_exists, intf_num = get_intf_num('netdev-icos-l2_interface', new_resource.name, OpEN::OPEN_INTF_TYPE_PHY)
  if intf_exists && intf_num != -1

    open = OpENUtil.new()
    conn_ret = open.connect("netdev-icos-l2_interface")
    if conn_ret == OpEN::OPEN_E_NONE
      client = open.client
      # Get the maximum length of the vlan name
      name_len_p = OpEN.new_uint32_tp()
      OpEN::openapiVlanNameLenGet(client, name_len_p)
      name_len = OpEN.uint32_tp_value(name_len_p)

      # Get the internal interface number
      intf_num_p    = OpEN.new_uint32_tp()
      intf_type     = OpEN::OPEN_INTF_TYPE_PHY

      buffdesc             = OpEN::Open_buffdesc.new
      vlan                 = 0
      vlan_p               = OpEN.new_uint32_tp()
      participation_mode_p = OpEN.new_OPEN_VLAN_PARTICIPATION_MODE_tp()
      tagging_p            = OpEN.new_OPEN_CONTROL_tp()
      descr_num_p          = OpEN.new_uint32_tp()

      tagged_vlans = []
      vlan_tagging = 'disable'
      # Get the vlan
      ret = OpEN::openapiVlanNextGet(client, vlan, vlan_p)
      while ret == OpEN::OPEN_E_NONE
        vlan = OpEN.uint32_tp_value(vlan_p)
  
        # Get the interface partipation in that vlan
        OpEN::openapiVlanIfParticipationGet(client, vlan, intf_num, participation_mode_p)
        participation_mode = OpEN.OPEN_VLAN_PARTICIPATION_MODE_tp_value(participation_mode_p)

        # Get the vlan tagging mode
        OpEN::openapiVlanIfTaggingGet(client, intf_num, vlan, tagging_p)
        tagging = OpEN.OPEN_CONTROL_tp_value(tagging_p)

        # Get the tagged vlan names
        if participation_mode == OpEN::OPEN_VLAN_PARTICIPATION_MODE_INCLUDE
          tagged_vlans << vlan if vlan != 1
          if tagging == OpEN::OPEN_ENABLE
            vlan_tagging = 'enable'
          end
        end
        ret = OpEN::openapiVlanNextGet(client, vlan, vlan_p)
      end

      # Get the pvid of the interface
      OpEN::openapiVlanIfPvidGet(client, intf_num, vlan_p)
      vlan = OpEN.uint32_tp_value(vlan_p)

      # Get the interface partipation in that vlan
      OpEN::openapiVlanIfParticipationGet(client, vlan, intf_num, participation_mode_p)
      participation_mode = OpEN.OPEN_VLAN_PARTICIPATION_MODE_tp_value(participation_mode_p)

      if vlan != 0 && participation_mode == OpEN::OPEN_VLAN_PARTICIPATION_MODE_INCLUDE
        untagged_vlan = vlan if vlan != 1 
      end

      # Get the interface description
      OpEN::openapiIfDescrSizeGet(client, intf_num, descr_num_p)
      descr_num = OpEN.uint32_tp_value(descr_num_p)

      buffdesc.size   = descr_num
      desc = open.getCharBuffer(buffdesc.size, nil)
      buffdesc.pstart = desc
      OpEN::openapiIfDescrGet(client, intf_num, buffdesc)
      description = desc.cast()

      interface['description'] = description
      interface['vlan_tagging'] = vlan_tagging
      interface['tagged_vlans'] = tagged_vlans
      interface['untagged_vlan'] = untagged_vlan.to_s

      @current_resource.name(new_resource.name)
      @current_resource.description(interface['description'])
      @current_resource.untagged_vlan(interface['untagged_vlan'])
      @current_resource.tagged_vlans(interface['tagged_vlans'])
      @current_resource.vlan_tagging(interface['vlan_tagging'])
      @current_resource.exists = intf_exists

      OpEN.delete_uint32_tp(name_len_p)
      OpEN.delete_uint32_tp(intf_num_p)
      OpEN.delete_uint32_tp(vlan_p)
      OpEN.delete_OPEN_VLAN_PARTICIPATION_MODE_tp(participation_mode_p)
      OpEN.delete_OPEN_CONTROL_tp(tagging_p)
      OpEN.delete_uint32_tp(descr_num_p)
    else
      msg = "Failed to connect to 'netdev-icos-l2_interface'"
      log_msg("info", msg)
    end
    open.terminate()

  else
    msg = "l2 interface #{@new_resource.name} doesn't exist, or not a valid l2interface"
    log_msg("info", msg)
  end

end

def modify_l2interface(intf_num)
  open = OpENUtil.new()
  conn_ret = open.connect("netdev-icos-l2_interface")
  if conn_ret == OpEN::OPEN_E_NONE
    client = open.client
    if new_resource.vlan_tagging == 'enable'
      tagging = OpEN::OPEN_ENABLE
    else
      tagging = OpEN::OPEN_DISABLE
    end

    participation_mode = OpEN::OPEN_VLAN_PARTICIPATION_MODE_INCLUDE

    # Get the maximum vlan name length
    name_len_p = OpEN.new_uint32_tp()
    OpEN::openapiVlanNameLenGet(client, name_len_p)
    name_len = OpEN.uint32_tp_value(name_len_p)

    # Get the vlan
    vlan   = 0
    vlan_p = OpEN.new_uint32_tp()

    ret = OpEN::openapiVlanNextGet(client, vlan, vlan_p)
    while ret == OpEN::OPEN_E_NONE
      vlan = OpEN.uint32_tp_value(vlan_p)

      # Set the vlan participation and tagging
      if new_resource.tagged_vlans
        new_resource.tagged_vlans.each do |name|
          if vlan.to_s == name.to_s
            ret1 = OpEN::openapiVlanIfParticipationSet(client, vlan, intf_num, participation_mode)
            ret2 = OpEN::openapiVlanIfTaggingSet(client, intf_num, vlan, tagging)
            if ret1 != OpEN::OPEN_E_NONE || ret2 != OpEN::OPEN_E_NONE
              msg = "Failed to set L2 interface '#{new_resource.name}' tagging, returned value = '#{ret1}', '#{ret2}'"
              log_msg("info", msg)
            end
          end
        end
      end

      # Set the vlan participation and pvid
      if vlan.to_s == new_resource.untagged_vlan
        ret1 = OpEN::openapiVlanIfParticipationSet(client, vlan, intf_num, participation_mode)
        ret2 = OpEN::openapiVlanIfPvidSet(client, intf_num, vlan)
        ret3 = OpEN::OPEN_E_NONE
        if new_resource.vlan_tagging == 'disable'
          # Set accept frame type to tagged only
          accept_frame_type = OpEN::OPEN_DOT1Q_ADMIN_ONLY_VLAN_TAGGED
          ret3 = OpEN::openapiVlanIfAcceptFrameTypeSet(client, intf_num, accept_frame_type)
        end
        if ret1 != OpEN::OPEN_E_NONE || ret2 != OpEN::OPEN_E_NONE || ret3 != OpEN::OPEN_E_NONE
          msg = "Failed to set L2 interface '#{new_resource.name}' untagged vlan, returned value = '#{ret1}', '#{ret2}', '#{ret3}'"
          log_msg("info", msg)
        end
      end

      ret = OpEN::openapiVlanNextGet(client, vlan, vlan_p)
    end # Loop through all the vlans

    # Set the interface description
    if new_resource.description
      buffdesc        = OpEN::Open_buffdesc.new
      buffdesc.size   = new_resource.description.length + 1
      buffdesc.pstart = open.getCharBuffer(buffdesc.size, new_resource.description)
      ret = OpEN::openapiIfDescrSet(client, intf_num, buffdesc)
      if ret != OpEN::OPEN_E_NONE
        msg = "Failed to set L2 interface '#{new_resource.name}' description, returned value = '#{ret}'"
        log_msg("info", msg)
      end
    end

    OpEN.delete_uint32_tp(name_len_p)
    OpEN.delete_uint32_tp(vlan_p)
  else
    msg = "Failed to connect to 'netdev-icos-l2_interface'"
    log_msg("info", msg)
  end
  open.terminate()

end

