/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  hash_example.c
*
* @purpose   Hash Prediction Example.
*
* @component OPEN
*
* @comments
*
* @create    10/17/2014
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>
#include <stdio.h>
#include <ctype.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_if.h"
#include "openapi_routing.h"
#include "openapi_hash.h"

#define ASCII_0_VALUE 48
#define ASCII_9_VALUE 57
#define ASCII_A_VALUE 65
#define ASCII_F_VALUE 70

/*******************************************************************
*
* @brief  This function prints the Hash Prediction Example 
*         Application Menu.
*
* @param    none
*
* @returns  none
*
* @end
*********************************************************************/
void printHashAppMenu()
{
  printf("Usage: hash_example lag <lag_id> <in_port_usp> <src_mac> <dst_mac> <vlan-id> <ethertype>"
         "<ip_family> <src_ip> <dst-ip> <protocol> <src-l4-port> <dst-l4-port>\n");
  printf("Usage: hash_example ecmp <ecmp_member_prefix> <ecmp_member_prefix_length> <in_port_usp>"
         "<src_mac> <dst_mac> <vlan-id> <ethertype> <ip_family> <src_ip> <dst-ip> <protocol> <src-l4-port>"
         "<dst-l4-port>\n");
  printf("Usage: MAC addrese is entered in form of aabbccddeeff\n");
  printf("Usage: enter vlan_id as 0 for untagged VLAN\n");
  printf("Usage: enter ip_family 1 as IPv4 and 2 for IPv6\n");
  printf("Usage: enter ethertype in form of XXXX of 0xXXXX Ethernet type\n");
  printf("Usage: enter src-l4-port and dst-l4-port as integers\n");
  printf("Usage: the examples are only available for non-stacking system\n");
  return;
}

static open_error_t getMacAddrFromInput (char *input_mac, unsigned char *mac)
{
  int i;
  unsigned char c;
  unsigned char temp_mac [12];

  for ( i = 0; i < 12; i++ )
  {
    c = input_mac [i];
    if ( (c >= '0') && (c <= '9') )
      temp_mac[i] = c - '0';
    else if ( (c >= 'a') && (c <= 'f') )
      temp_mac[i] = c - 'a' + 10;
    else if ( (c >= 'A') && (c <= 'F') )
      temp_mac[i] = c - 'A' + 10;
    else return(1);   /* Invalid character */
  };

  for ( i = 0; i < 12; i+= 2 )
  {
    mac[i/2] = (temp_mac[i] * 16) + temp_mac[i+1];
  };
  return(0);
};

static open_error_t getEthertypeFromInput (char *input_ethertype, unsigned int *ethertype)
{
    unsigned int result = 0;
    char *c = input_ethertype;
    char thisChar;

    while((thisChar = *c) != '\0')
    {
        unsigned int add;
        thisChar = toupper(thisChar);

        result <<= 4;

        if( thisChar >= ASCII_0_VALUE &&  thisChar <= ASCII_9_VALUE )
            add = thisChar - ASCII_0_VALUE;
        else if( thisChar >= ASCII_A_VALUE && thisChar <= ASCII_F_VALUE)
            add = thisChar - ASCII_A_VALUE + 10;
        else
        {
            printf("Unrecognized hex character \"%c\"\n", thisChar);
            return(1);
        }

        result += add;
        ++c;
    }

    *ethertype = result;
    return 0;  
}

/*******************************************************************
*
* @brief  This is the main function that will demonstrate 
*         ACL OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main (int argc, char **argv)
{
   openapiClientHandle_t clientHandle;
   open_hashPktInfo_t pktInfo;
   open_USP_t destUSP;
   uint32_t intfNum;
   open_error_t result;
   open_buffdesc buffDesc;
   OPEN_MPLS_LABELS_t mpls_label;
   uint32_t i;

   if (argc < 7 )
   {
      printHashAppMenu();
      exit(1);
   }

   l7proc_crashlog_register ();

   /* Register with OpEN */
   if ((result = openapiClientRegister ("hash_example", &clientHandle)) != OPEN_E_NONE)
   {
       printf ("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
       exit (2);
   }

   /* RPC call can fail until server starts. Keep trying */
   while (openapiConnectivityCheck (&clientHandle) != OPEN_E_NONE)
   {
       sleep (1);
   }

   L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0, "Starting HASH API example application");

   memset(&pktInfo, 0, sizeof(pktInfo));

   if (strcmp("lag", argv[1]) == 0) 
   {
      pktInfo.aggrType = OPEN_AGGR_TYPE_LAG;
      pktInfo.egressId.aggrId.lagId = atoi(argv[2]);

      buffDesc.pstart = argv[3];
      buffDesc.size = strlen(argv[3]);
      if (openapiIfNumGet(&clientHandle, &buffDesc, &intfNum) != OPEN_E_NONE)
      {
         printf ("Wrong incoming port %s\n", argv[3]);
         exit(1);
      }
      else 
      {
         if (openapiHashIntfUnitSlotPortGet(&clientHandle, intfNum, &(pktInfo.inUsp)) != OPEN_E_NONE)
         {
            printf ("Wrong intfNum %d\n", intfNum);
            exit(1);
         }
      }

      if (getMacAddrFromInput(argv[4], pktInfo.src_mac))
      {
         printf ("Wrong src mac %s\n", argv[4]);
         exit(1);
      }

      if (getMacAddrFromInput(argv[5], pktInfo.dst_mac))
      {
         printf ("Wrong dst mac %s\n", argv[5]);
         exit(1);
      }

      pktInfo.vid = atoi(argv[6]);

      if (getEthertypeFromInput(argv[7], &(pktInfo.ethertype)))
      {
         printf ("Wrong Ethertype %s\n", argv[7]);
         exit(1);
      }

      if (argv[8] != NULL) 
      {
         if (atoi(argv[8]) == 1) 
         {
            if ((argv[9] != NULL) && (inet_pton(AF_INET, argv[9], (void*)&(pktInfo.srcIp))) < 0)
            {
               printf ("Wrong IPv4 src address %s\n", argv[9]);
               exit(1);
            }
            pktInfo.srcIp = ntohl(pktInfo.srcIp);

            if ((argv[10] != NULL) && (inet_pton(AF_INET, argv[10], (void*)&(pktInfo.destIp))) < 0)
            {
               printf ("Wrong IPv4 dest address %s\n", argv[10]);
               exit(1);
            }
            pktInfo.destIp = ntohl(pktInfo.destIp);
         }
         else if (atoi(argv[8]) == 2) 
         {
            if ((argv[9] != NULL) && (inet_pton(AF_INET6, argv[9], (void*)&(pktInfo.srcIp6))) < 0)
            {
               printf ("Wrong IPv6 src address %s\n", argv[9]);
               exit(1);
            }

            if ((argv[10] != NULL) && (inet_pton(AF_INET6, argv[10], (void*)&(pktInfo.destIp6))) < 0)
            {
               printf ("Wrong IPv6 src address %s\n", argv[10]);
               exit(1);
            }
         }


         if (argv[11] != NULL) {
            pktInfo.protocol = atoi(argv[11]);
         }
         else
         {
            printf ("No protocol\n");
            exit(1);
         }

         if (argv[12] != NULL) {
            pktInfo.srcL4Port = atoi(argv[12]);
         }
         else
         {
            printf ("No L4 src port\n");
            exit(1);
         }

         if (argv[13] != NULL) {
            pktInfo.dstL4Port = atoi(argv[13]);
         }
         else
         {
            printf ("No L4 dest port\n");
            exit(1);
         }
      }
   }
   else if (strcmp("ecmp", argv[1]) == 0) {
      pktInfo.aggrType = OPEN_AGGR_TYPE_ECMP;
      pktInfo.egressId.aggrId.ecmpId.pfxLen = atoi(argv[3]);

      buffDesc.pstart = argv[4];
      buffDesc.size = strlen(argv[4]);
      if (openapiIfNumGet(&clientHandle, &buffDesc, &intfNum) != OPEN_E_NONE)
      {
         printf ("Wrong incoming port %s\n", argv[4]);
         exit(1);
      }
      else
      {
         if (openapiHashIntfUnitSlotPortGet(&clientHandle, intfNum, &(pktInfo.inUsp)) != OPEN_E_NONE)
         {
            printf ("Wrong intfNum %d\n", intfNum);
            exit(1);
         }
      }

      if (getMacAddrFromInput(argv[5], pktInfo.src_mac))
      {
         printf ("Wrong src mac %s\n", argv[5]);
         exit(1);
      }

      if (getMacAddrFromInput(argv[6], pktInfo.dst_mac))
      {
         printf ("Wrong dst mac %s\n", argv[6]);
         exit(1);
      }

      pktInfo.vid = atoi(argv[7]);

      if (getEthertypeFromInput(argv[8], &(pktInfo.ethertype)))
      {
         printf ("Wrong Ethertype %s\n", argv[8]);
         exit(1);
      }

      if (argv[9] != NULL) 
      {
      	if (atoi(argv[9]) == 1) {
           pktInfo.egressId.aggrId.ecmpId.ipAddr.family = OPEN_AF_INET;

           if (inet_pton(AF_INET, argv[2], (void*)&(pktInfo.egressId.aggrId.ecmpId.ipAddr.addr.ipv4)) < 0)
           {
              printf ("Wrong IPv4 egress member address %s\n", argv[2]);
              exit(1);
           }

           if (inet_pton(AF_INET, argv[10], (void*)&(pktInfo.srcIp)) < 0)
           {
              printf ("Wrong IPv4 src address %s\n", argv[10]);
              exit(1);
           }
           pktInfo.srcIp = ntohl(pktInfo.srcIp);

           if (inet_pton(AF_INET, argv[11], (void*)&(pktInfo.destIp)) < 0)
           {
              printf ("Wrong IPv4 dest address %s\n", argv[11]);
              exit(1);
           }
           pktInfo.destIp = ntohl(pktInfo.destIp);
        }
        else if (atoi(argv[9]) == 2) 
        {
           pktInfo.egressId.aggrId.ecmpId.ipAddr.family = OPEN_AF_INET6;

           if (inet_pton(AF_INET6, argv[2], (void*)&(pktInfo.egressId.aggrId.ecmpId.ipAddr.addr.ipv6)) < 0)
           {
              printf ("Wrong IPv6 egress member address %s\n", argv[2]);
              exit(1);
           }

           if (inet_pton(AF_INET6, argv[10], (void*)&(pktInfo.srcIp6)) < 0)
           {
              printf ("Wrong IPv6 src address %s\n", argv[10]);
              exit(1);
           }

           if (inet_pton(AF_INET6, argv[11], (void*)&(pktInfo.destIp6)) < 0)
           {
              printf ("Wrong IPv6 src address %s\n", argv[11]);
              exit(1);
           }
        }

        if (argv[12] != NULL) {
           pktInfo.protocol = atoi(argv[12]);
        }
        else
        {
           printf ("No protocol\n");
           exit(1);
        }

        if (argv[13] != NULL) {
           pktInfo.srcL4Port = atoi(argv[13]);
        }
        else
        {
           printf ("No L4 src port\n");
           exit(1);
        }

        if (argv[14] != NULL) {
           pktInfo.dstL4Port = atoi(argv[14]);
        }
        else
        {
           printf ("No L4 dest port\n");
           exit(1);
        }
      }
   }
   else
   {
      printHashAppMenu();
      exit(1);
   }

   result = openapiHashDestGet(&pktInfo, &destUSP, &mpls_label);

   if (result == OPEN_E_NONE) {
      printf ("The destination port is %d/%d/%d\n", destUSP.unit, destUSP.slot, destUSP.port);
      if (mpls_label.label[0])
      {
          printf("MPLS Labels: { ");
          for (i = 0; i < OPEN_MAX_MPLS_IMPOSE_LABELS; i++ )
          {
              if (mpls_label.label[i] == 0)
              {
                  break;
              }
              printf("%d ", mpls_label.label[i]);
          }
          printf("}\n");
      }
   }
   else
   {
      printf ("Failed to find the destination port\n");
   }

  (void) openapiClientTearDown(&clientHandle);
   return 0;
}

