/*********************************************************************
*
* Copyright 2016-2017 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  bfd_create.c
*
* @purpose Create BFD session with specified parameters.
*
* @component OPEN
*
* @comments
*
* @create    09/09/2015
*
* @end
*
**********************************************************************/
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <signal.h>
#include <stdlib.h>
#include <pthread.h>
#include <fcntl.h>
#include <unistd.h>
#include <netdb.h>
#include <sys/ioctl.h>
#include <linux/if.h>
#include <linux/if_packet.h>
#include <sys/errno.h>
#include <ctype.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_routing.h"
#include "openapi_mpls.h"


openapiClientHandle_t clientHandle;

/*********************************************************************
* @purpose  Create a BFD session
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   Address Family (IPv4=1 or IPv6=2)
* @param    ifNum            @b{(input)}   Router Interface
* @param    dstIpStr         @b{(input)}   Dest IP address.
* @param    srcIpStr         @b{(input)}   Source IP address.
*
* @returns  none
*
* @notes  
*
* @end
*********************************************************************/
void rtrBfdSessionCreate(openapiClientHandle_t *client_handle, OPEN_AF_t af,
                         uint32_t ifNum, char *dstIpStr , char *srcIpStr,
			 const unsigned char *nhop_ip, 
			 OPEN_MPLS_LABELS_t *mpls_label)
{
  uint32_t sessId;
  open_error_t rc;
  openBfdEndpoint_t ep;

  memset(&ep, 0x0, sizeof(ep));
  /* Setup the BFD session parameters */
  ep.compId = 0;
  ep.vrfId = 0;
  ep.intIfNum = ifNum;
  ep.dstIpAddr.family = af;
  ep.srcIpAddr.family = af;
  ep.nextHopIpAddr.family = af;
  ep.type = OPEN_BFD_TUNNEL_TYPE_UDP; /* Need BFD session using IP-UDP encap */

  /* Use configuration provided in the session instead of intf/global config.
  */
  ep.override_config = 1; 
  ep.bfdEchoMode = 0;  /* Disable Echo Mode */

  /* Slow timer is 2000ms (The timer is used to send messages when session is down.
  */
  ep.bfdSlowTimer = 2000; 

  /* Use 100ms timer when session is up.
  */
  ep.bfdMinRx = 100;  
  ep.bfdMinTx = 100;

  ep.bfdMultiplier = 3;  /* Fail if missed 3 messages */

  if (af == OPEN_AF_INET)
  {
    if (inet_pton(AF_INET, dstIpStr, (void*)&(ep.dstIpAddr.addr.ipv4)) < 0)
    {
      printf("Bad return code trying to convert DIP.\n");
      return;
    }
    if (inet_pton(AF_INET, srcIpStr, (void*)&(ep.srcIpAddr.addr.ipv4)) < 0)
    {
      printf("Bad return code trying to convert SIP.\n");
      return;
    }

    if (nhop_ip) 
    {
       if (inet_pton(AF_INET, (void *) nhop_ip, (void*)&(ep.nextHopIpAddr.addr.ipv4)) < 0)
       {
         printf("Bad return code trying to convert Nhop IP.\n");
         return;
       }
    }
    /* IPv4 addresses must be in host format.
    */
    ep.dstIpAddr.addr.ipv4 = ntohl (ep.dstIpAddr.addr.ipv4);
    ep.srcIpAddr.addr.ipv4 = ntohl (ep.srcIpAddr.addr.ipv4);
    ep.nextHopIpAddr.addr.ipv4 = ntohl (ep.nextHopIpAddr.addr.ipv4);
  }
  else if (af == OPEN_AF_INET6)
  {
    if (inet_pton(AF_INET6, dstIpStr, (void*)&(ep.dstIpAddr.addr.ipv6)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }

    if (inet_pton(AF_INET6, srcIpStr, (void*)&(ep.srcIpAddr.addr.ipv6)) < 0)
    {
      printf("Bad return code trying to convert IP.\n");
      return;
    }

    if (nhop_ip) 
    {
       if (inet_pton(AF_INET6, (void *)nhop_ip, (void*)&(ep.nextHopIpAddr.addr.ipv6)) < 0)
       {
         printf("Bad return code trying to convert Nhop IP.\n");
         return;
       }
    }
  }
  memcpy (&ep.mpls_label, mpls_label, sizeof (OPEN_MPLS_LABELS_t));

  rc = openapiBfdSessionCreate(client_handle, &ep, &sessId);
 
  if (rc == OPEN_E_NONE)
  {
    printf("Successfully created BFD session %d\n", sessId);
  }
  else
  {
    printf("Failed to create BFD session %d \n", rc);
  }
}

int main(int argc, char **argv)
{
  int rc;
  uint32_t addrFamily;
  uint32_t ifNum; 
  OPEN_MPLS_LABELS_t mpls_label;
  const unsigned char *nhop_ip;

  memset (&mpls_label, 0, sizeof (mpls_label));
  nhop_ip = 0;

  if (argc < 5)
  {
    printf ("Usage: bfd_create [4|6] <intf-num> <sip> <dip> [hhop-ip] [label-1] [label-2] [label-3]\n");
    exit (1);
  }
  if (argv[1][0] == '4')
  {
    addrFamily = OPEN_AF_INET;
  } else if (argv[1][0] == '6')
  {
    addrFamily = OPEN_AF_INET6;
  } else
  {
    printf ("Address family:%s is not valid. Specify '4' for IPv4 or '6' for IPv6\n", argv[1]);
    exit (1);
  }
  ifNum = atoi (argv[2]);

  if (argc > 5)
  {
   nhop_ip = (unsigned char *) argv[5];
  }
  if (argc > 6)
  {
    mpls_label.label[0] = atoi(argv[6]);
  }
  if (argc > 7)
  {
    mpls_label.label[1] = atoi(argv[7]);
  }
  if (argc > 8)
  {
    mpls_label.label[2] = atoi(argv[8]);
  }

  /* Register with OpEN */
  if ((rc = openapiClientRegister("bfd_delete", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", rc);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  rtrBfdSessionCreate(&clientHandle, addrFamily, 
				ifNum, argv[4], argv[3],
				 nhop_ip, &mpls_label);
 
  (void) openapiClientTearDown (&clientHandle);
  return 0;
}


