/*********************************************************************
*
*  Copyright 2020 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  msrp_example.c
*
* @purpose   OpEN msrp example.
*
* @component OpEN
*
* @create    05/15/2020
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_msrp.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the msrp Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Get the global MSRP mode: %s 1 \n", name);
  printf("Test 2: set the global MSRP mode: %s 2 <mode>(0-Disable, 1-Enable)\n", name);
  printf("Test 3: Get the MSRP boundary propagate mode: %s 3 \n", name);
  printf("Test 4: Set the MSRP boundary propagate mode: %s 4 <mode>(0-Disable, 1-Enable)\n", name);
  printf("Test 5: Get the MSRP mode for the specified interface: %s 5 <intIfNum>\n", name);
  printf("Test 6: Set the MSRP mode for the given interface: %s 6 <intIfNum><mode>(0-Disable, 1-Enable)\n", name);
  printf("Test 7: Get the MSRP SR Qav priority: %s 7 <intIfNum> <trClass> (0-Class A or 1-Class B)\n", name);
  printf("Test 8: Set the MSRP SR Qav priority: %s 8 <intIfNum> <trClass>(0-Class A or 1-Class B) <priority>(0-7)\n", name);
  printf("Test 9: Get the MSRP Sr Qav remapped priority: %s 9 <intIfNum> <trClass>(0-Class A or 1-Class B)\n", name);
  printf("Test 10: Set the MSRP Sr Qav remapped priority: %s 10 <intIfNum> <trClass>(0-Class A or 1-Class B) <remap>(0-7)\n", name);
  printf("Test 11: Get the MSRP talker pruning mode: %s 11 \n", name);
  printf("Test 12: Set the MSRP talker pruning mode: %s 12 <mode>(0-Disable, 1-Enable)\n", name);
  printf("Test 13: Get the maximum number of the ports, where MSRP registrations are allowed: %s 13 \n", name);
  printf("Test 14: Set the maximum number of the ports, where MSRP registrations are allowed: %s 14 <maxFanInPorts>\n", name);
  printf("Test 15: Get the MSRP SR class PVID value on the interface: %s 15 <intIfNum>\n", name);
  printf("Test 16: Set the MSRP SR class PVID value on the interface: %s 16 <intIfNum> <pvid>(1-4094)\n", name);
  printf("Test 17: Get the delta bandwidth values for the interface: %s 17 <intIfNum>\n", name);
  printf("Test 18: Set the delta bandwidth values for the interface: %s 18 <intIfNum> <trafficClass>(0-Class A or 1-Class B) <deltaBandwidth>(0-75)\n", name);
  printf("Test 19: Get the max free bandwidth values for the interface: %s 19 <intIfNum>\n", name);
  printf("Test 20: Get the current used bandwidth values for the interface: %s 20 <intIfNum>\n", name);
  printf("Test 21: Check if a port is boundary port: %s 21 <intIfNum> <trClass>(0-Class A or 1-Class B)\n", name);
  printf("Test 22: Get the next stream index to the given index: %s 22 <currentIndex>\n", name);
  printf("Test 23: Get the MSRP Reservation Information: %s 23 <intIfNum> <streamId>\n", name);
  printf("Test 24: Get the value of specified MSRP counter: %s 24 <type>\n", name);
  printf("Test 25: Get the value of specified MSRP counter for an interface: %s 25 <intIfNum> <type> \n", name);
  printf("Test 26: Get Failed Registration Counter: %s 26 <intIfNum>\n", name);
  printf("Test 27: Get the MSRP Stream Information: %s 27 <streamId>\n", name);
  printf("Test 28: Clear MSRP traffic counters on specified interface: %s 28 <intIfNum>\n", name);
  printf("Test 29: Clear all MSRP traffic counters: %s 29 \n", name);
  printf("Test 30: Run API sanity checks: %s 30 \n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  uint64_t trClassB;
  uint32_t maxFanInPorts;
  uint32_t remap;
  uint64_t trClassA;
  OPEN_BOOL_t isBoundary;
  OPEN_MSRP_QAV_STREAMCLASS_t trClass;
  uint32_t counter;
  uint32_t deltaClassA;
  uint32_t deltaClassB;
  uint32_t priority;
  uint32_t pvid;
  uint16_t nextIndex;
  OPEN_CONTROL_t mode;
  uint32_t streamId;
  uint64_t total;
  OPEN_MSRP_STATS_t type;
  uint32_t intIfNum;
  uint16_t currentIndex;
  OPEN_MSRP_RESERVATION_INFO_t  reservationInfo;
  OPEN_MSRP_STEAM_INFO_t        streamInfo;

  char buffer[128];
  open_buffdesc buf;
  open_buffdesc zeroLenBuf;
  open_buffdesc badBufdescPointer;

  badBufdescPointer.pstart = (void *) NULL;
  badBufdescPointer.size = sizeof(buffer);
  badBufdescPointer.size = 0;
  buf.pstart = buffer;
  buf.size = sizeof(buffer);
  zeroLenBuf.pstart = buffer;
  zeroLenBuf.size = 0;

  printf("Testing msrp OpEN APIs sanity:\n");

  printf("Testing openapiMsrpModeGet():\n");
  result = openapiMsrpModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMsrpModeGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiMsrpBoundaryPropagateModeGet():\n");
  result = openapiMsrpBoundaryPropagateModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMsrpBoundaryPropagateModeGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiMsrpIntfModeGet():\n");
  intIfNum = 1;
  result = openapiMsrpIntfModeGet(NULL, intIfNum, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMsrpIntfModeGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiMsrpSrQavPriorityGet():\n");
  trClass = 0;
  result = openapiMsrpSrQavPriorityGet(NULL, intIfNum, trClass, &priority);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMsrpSrQavPriorityGet(clientHandle, intIfNum, trClass, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiMsrpSrQavPrioritySet():\n");
  result = openapiMsrpSrQavPrioritySet(NULL, intIfNum, trClass, priority);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMsrpSrQavPrioritySet(clientHandle, intIfNum, trClass, -1);
  printf("less than minimum value argument 4:(result = %d)\n", result);
  result = openapiMsrpSrQavPrioritySet(clientHandle, intIfNum, trClass, 8);
  printf("greater than maximum value argument 4:(result = %d)\n", result);

  printf("Testing openapiMsrpSrQavRePriorityGet():\n");
  result = openapiMsrpSrQavRePriorityGet(NULL, intIfNum, trClass, &remap);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMsrpSrQavRePriorityGet(clientHandle, intIfNum, trClass, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiMsrpSrQavRePrioritySet():\n");
  result = openapiMsrpSrQavRePrioritySet(NULL, intIfNum, trClass, remap);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMsrpSrQavRePrioritySet(clientHandle, intIfNum, trClass, -1);
  printf("less than minimum value argument 4:(result = %d)\n", result);
  result = openapiMsrpSrQavRePrioritySet(clientHandle, intIfNum, trClass, 8);
  printf("greater than maximum value argument 4:(result = %d)\n", result);

  printf("Testing openapiMsrpTalkerPruningModeGet():\n");
  result = openapiMsrpTalkerPruningModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMsrpTalkerPruningModeGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiMsrpMaxFanInPortsGet():\n");
  result = openapiMsrpMaxFanInPortsGet(NULL, &maxFanInPorts);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMsrpMaxFanInPortsGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiMsrpSRclassPVIDGet():\n");
  result = openapiMsrpSRclassPVIDGet(NULL, intIfNum, &pvid);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMsrpSRclassPVIDGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiMsrpQavDeltaBandwidthGet():\n");
  result = openapiMsrpQavDeltaBandwidthGet(NULL, intIfNum, &deltaClassA, &deltaClassB);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMsrpQavDeltaBandwidthGet(clientHandle, intIfNum, NULL, &deltaClassB);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiMsrpQavDeltaBandwidthGet(clientHandle, intIfNum, &deltaClassA, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiMsrpQavMaxRateGet():\n");
  result = openapiMsrpQavMaxRateGet(NULL, intIfNum, &trClassA, &trClassB, &total);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMsrpQavMaxRateGet(clientHandle, intIfNum, NULL, &trClassB, &total);
  printf("NULL argument 5:(result = %d)\n", result);
  result = openapiMsrpQavMaxRateGet(clientHandle, intIfNum, &trClassA, NULL, &total);
  printf("NULL argument 5:(result = %d)\n", result);
  result = openapiMsrpQavMaxRateGet(clientHandle, intIfNum, &trClassA, &trClassB, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiMsrpQavCurrentBandwidthGet():\n");
  result = openapiMsrpQavCurrentBandwidthGet(NULL, intIfNum, &trClassA, &trClassB, &total);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMsrpQavCurrentBandwidthGet(clientHandle, intIfNum, NULL, &trClassB, &total);
  printf("NULL argument 5:(result = %d)\n", result);
  result = openapiMsrpQavCurrentBandwidthGet(clientHandle, intIfNum, &trClassA, NULL, &total);
  printf("NULL argument 5:(result = %d)\n", result);
  result = openapiMsrpQavCurrentBandwidthGet(clientHandle, intIfNum, &trClassA, &trClassB, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiMsrpIsBoundaryPortGet():\n");
  result = openapiMsrpIsBoundaryPortGet(NULL, intIfNum, trClass, &isBoundary);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMsrpIsBoundaryPortGet(clientHandle, intIfNum, trClass, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiMsrpStreamIndexedNextGet():\n");
  currentIndex = 1;
  result = openapiMsrpStreamIndexedNextGet(NULL, currentIndex, &nextIndex);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMsrpStreamIndexedNextGet(clientHandle, currentIndex, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiMsrpReservationInformationGet():\n");
  buf.pstart = &reservationInfo;
  buf.size   = sizeof(reservationInfo);
  streamId = 1;
  result = openapiMsrpReservationInformationGet(NULL, streamId, intIfNum, &buf);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMsrpReservationInformationGet(clientHandle, streamId, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiMsrpReservationInformationGet(clientHandle, streamId, intIfNum, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiMsrpReservationInformationGet(clientHandle, streamId, intIfNum, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);

  printf("Testing openapiMsrpTrafficPduCounterGet():\n");
  type = 2;
  result = openapiMsrpTrafficPduCounterGet(NULL, type, &counter);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMsrpTrafficPduCounterGet(clientHandle, type, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiMsrpIntfTrafficPduCounterGet():\n");
  result = openapiMsrpIntfTrafficPduCounterGet(NULL, intIfNum, type, &counter);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMsrpIntfTrafficPduCounterGet(clientHandle, intIfNum, type, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiMsrpFailedRegistrationCounterGet():\n");
  result = openapiMsrpFailedRegistrationCounterGet(NULL, intIfNum, &counter);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMsrpFailedRegistrationCounterGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiMsrpStreamInformationGet():\n");
  buf.pstart = (void *)&streamInfo;
  buf.size   = sizeof(streamInfo);
  result = openapiMsrpStreamInformationGet(NULL, streamId, &buf);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMsrpStreamInformationGet(clientHandle, streamId, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiMsrpStreamInformationGet(clientHandle, streamId, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiMsrpStreamInformationGet(clientHandle, streamId, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);

  printf("Testing openapiMsrpTrafficCountersPerIfClear():\n");
  result = openapiMsrpTrafficCountersPerIfClear(NULL, intIfNum);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiMsrpTrafficCountersClear():\n");
  result = openapiMsrpTrafficCountersClear(NULL);
  printf("NULL client handle:(result = %d)\n", result);

  return;
}


/*********************************************************************
* @purpose  Get the global MSRP mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(output)}  Global admin mode.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiMsrpModeGet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the global MSRP mode. (result = %d)\n", result);
  }
  else
  {
    printf("Global Msrp mode is %d.\n",*mode);
  }
  return;
}


/*********************************************************************
* @purpose  set the global MSRP mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Global admin mode.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiMsrpModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the global MSRP mode. (result = %d)\n", result);
  }
  else
  {
    printf("Global MSRP mode is set\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the MSRP boundary propagate mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(output)}  Boundary propagate mode.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpBoundaryPropagateModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiMsrpBoundaryPropagateModeGet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the MSRP boundary propagate mode. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP boundary propogation mode is %d.\n", *mode);
  }
  return;
}


/*********************************************************************
* @purpose  Set the MSRP boundary propagate mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Boundary propagate mode.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpBoundaryPropagateModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiMsrpBoundaryPropagateModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the MSRP boundary propagate mode. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP boundary propogation mode is set.\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the MSRP mode for the specified interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.
* @param    mode       @b{(output)}  Interface MSRP admin mode.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpIntfModeGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiMsrpIntfModeGet(client_handle, intIfNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the MSRP mode for the specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP mode on interface %d is %d.\n",intIfNum, *mode);
  }
  return;
}


/*********************************************************************
* @purpose  Set the MSRP mode for the given interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.
* @param    mode       @b{(input)}  Interface MSRP admin mode.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpIntfModeSet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiMsrpIntfModeSet(client_handle, intIfNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the MSRP mode for the given interface. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP mode on interface %d is set.\n",intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Get the 802.1Qav priority.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.
* @param    trClass       @b{(input)}  EAV class.
* @param    priority       @b{(output)}  Priority.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpSrQavPriorityGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_MSRP_QAV_STREAMCLASS_t trClass, uint32_t *priority)
{
  open_error_t result;

  if ((result = openapiMsrpSrQavPriorityGet(client_handle, intIfNum, trClass, priority)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the 802.1Qav priority. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP Sr Qav priority of type %d on interface %d is %d.\n",
	   trClass, intIfNum, *priority);
  }
  return;
}


/*********************************************************************
* @purpose  Set the 802.1Qav priority.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.
* @param    trClass       @b{(input)}  EAV class.
* @param    priority       @b{(input)}  Priority.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpSrQavPrioritySet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_MSRP_QAV_STREAMCLASS_t trClass, uint32_t priority)
{
  open_error_t result;

  if ((result = openapiMsrpSrQavPrioritySet(client_handle, intIfNum, trClass, priority)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the 802.1Qav priority. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP Sr Qav priority of type %d on interface %d is set.\n",
	   trClass, intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Get the 802.1Qav remapped priority.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.
* @param    trClass       @b{(input)}  EAV class.
* @param    remap       @b{(output)}  Remapped priority.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpSrQavRePriorityGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_MSRP_QAV_STREAMCLASS_t trClass, uint32_t *remap)
{
  open_error_t result;

  if ((result = openapiMsrpSrQavRePriorityGet(client_handle, intIfNum, trClass, remap)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the 802.1Qav remapped priority. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP Sr Qav remapped priority of type %d on interface %d is %d.\n",
	   trClass, intIfNum, *remap);
  }
  return;
}


/*********************************************************************
* @purpose  Set the 802.1Qav remapped priority.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.
* @param    trClass       @b{(input)}  EAV class.
* @param    remap       @b{(input)}  Remapped priority.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpSrQavRePrioritySet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_MSRP_QAV_STREAMCLASS_t trClass, uint32_t remap)
{
  open_error_t result;

  if ((result = openapiMsrpSrQavRePrioritySet(client_handle, intIfNum, trClass, remap)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the 802.1Qav remapped priority. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP Sr Qav remapped priority of type %d on interface %d is set.\n",
	   trClass, intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Get the MSRP talker pruning mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(output)}  Talker pruning mode.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpTalkerPruningModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiMsrpTalkerPruningModeGet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the MSRP talker pruning mode. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP talker pruning mode is %d\n",*mode);
  }
  return;
}


/*********************************************************************
* @purpose  Set the MSRP talker pruning mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Talker pruning mode.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpTalkerPruningModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiMsrpTalkerPruningModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the MSRP talker pruning mode. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP talker pruning mode is set.\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the maximum number of the ports, where MSRP registrations are allowed.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    maxFanInPorts       @b{(output)}  Number of the ports where MSRP registrations are allowed.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpMaxFanInPortsGet(openapiClientHandle_t *client_handle, uint32_t *maxFanInPorts)
{
  open_error_t result;

  if ((result = openapiMsrpMaxFanInPortsGet(client_handle, maxFanInPorts)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the maximum number of the ports, where MSRP registrations are allowed. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP max fan in ports is %d\n",*maxFanInPorts);
  }
  return;
}


/*********************************************************************
* @purpose  Set the maximum number of the ports, where MSRP registrations are allowed.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    maxFanInPorts       @b{(input)}  Number of the ports where MSRP registrations are allowed.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpMaxFanInPortsSet(openapiClientHandle_t *client_handle, uint32_t maxFanInPorts)
{
  open_error_t result;

  if ((result = openapiMsrpMaxFanInPortsSet(client_handle, maxFanInPorts)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the maximum number of the ports, where MSRP registrations are allowed. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP maximum fan in ports is set.\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the MSRP SR class PVID value on the interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.
* @param    pvid       @b{(output)}  SRclassPVID.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpSRclassPVIDGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *pvid)
{
  open_error_t result;

  if ((result = openapiMsrpSRclassPVIDGet(client_handle, intIfNum, pvid)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the MSRP SR class PVID value on the interface. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP SRclass PVID on interface %d is %d.\n", intIfNum, *pvid);
  }
  return;
}


/*********************************************************************
* @purpose  Set the MSRP SR class PVID value on the interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.
* @param    pvid       @b{(input)}  SRclassPVID.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpSRclassPVIDSet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t pvid)
{
  open_error_t result;

  if ((result = openapiMsrpSRclassPVIDSet(client_handle, intIfNum, pvid)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the MSRP SR class PVID value on the interface. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP SRclass PVID on interface %d is set.\n", intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Get the delta bandwidth values for the interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.
* @param    deltaClassA       @b{(output)}  class A delta bandwidth.
* @param    deltaClassB       @b{(output)}  class B delta bandwidth.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpQavDeltaBandwidthGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *deltaClassA, uint32_t *deltaClassB)
{
  open_error_t result;

  if ((result = openapiMsrpQavDeltaBandwidthGet(client_handle, intIfNum, deltaClassA, deltaClassB)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the delta bandwidth values for the interface. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP Qav Delta bandwidth on interface %d is Class A %d, Class B %d\n",
	   intIfNum, *deltaClassA, *deltaClassB);
  }
  return;
}


/*********************************************************************
* @purpose  Get the delta bandwidth values for the interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.
* @param    trafficClass       @b{(input)}  Stream class (A or B).
* @param    deltaBandwidth       @b{(input)}  Delta bandwidth.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpQavDeltaBandwidthSet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_MSRP_QAV_STREAMCLASS_t trafficClass, uint32_t deltaBandwidth)
{
  open_error_t result;

  if ((result = openapiMsrpQavDeltaBandwidthSet(client_handle, intIfNum, trafficClass, deltaBandwidth)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the delta bandwidth values for the interface. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP Qav Delta bandwidth on interface %d is set.\n", intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Get the max free bandwidth values for the interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.
* @param    trClassA       @b{(output)}  class A free bandwidth.
* @param    trClassB       @b{(output)}  class B free bandwidth.
* @param    total       @b{(output)}  Total free bandwidth.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpQavMaxRateGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint64_t *trClassA, uint64_t *trClassB, uint64_t *total)
{
  open_error_t result;

  if ((result = openapiMsrpQavMaxRateGet(client_handle, intIfNum, trClassA, trClassB, total)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the max free bandwidth values for the interface. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP Qav Max rate on interface %d is Class A %llu, Class B %llu, Total %llu \n", intIfNum, *trClassA, *trClassB, *total);
  }
  return;
}


/*********************************************************************
* @purpose  Get the current used bandwidth values for the interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.
* @param    trClassA       @b{(output)}  class A free bandwidth.
* @param    trClassB       @b{(output)}  class B free bandwidth.
* @param    total       @b{(output)}  Total free bandwidth.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpQavCurrentBandwidthGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint64_t *trClassA, uint64_t *trClassB, uint64_t *total)
{
  open_error_t result;

  if ((result = openapiMsrpQavCurrentBandwidthGet(client_handle, intIfNum, trClassA, trClassB, total)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the current used bandwidth values for the interface. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP Qav current bandwidth on interface %d is \n"
	   "Class A %llu, Class B %llu, Total %llu\n", intIfNum, *trClassA, *trClassB, *total);
  }
  return;
}


/*********************************************************************
* @purpose  Check if a port is boundary port.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.
* @param    trClass       @b{(input)}  EAV class.
* @param    isBoundary       @b{(output)}  Flag to check if the port is bondary port.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpIsBoundaryPortGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_MSRP_QAV_STREAMCLASS_t trClass, OPEN_BOOL_t *isBoundary)
{
  open_error_t result;

  if ((result = openapiMsrpIsBoundaryPortGet(client_handle, intIfNum, trClass, isBoundary)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to check if a port is boundary port. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP interface %d boundary port status is %d.\n", 
	   intIfNum, *isBoundary);
  }
  return;
}


/*********************************************************************
* @purpose  Given the next stream index to the given index.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    currentIndex       @b{(input)}  Present index.
* @param    nextIndex       @b{(output)}  Next index.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpStreamIndexedNextGet(openapiClientHandle_t *client_handle, uint16_t currentIndex, uint16_t *nextIndex)
{
  open_error_t result;

  if ((result = openapiMsrpStreamIndexedNextGet(client_handle, currentIndex, nextIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to given the next stream index to the given index. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP next stream index(present index %d) is %d\n", 
	   currentIndex, *nextIndex);
  }
  return;
}


/*********************************************************************
* @purpose  Get the MSRP Reservation Information.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum            @b{(input)}  Interface Number.
* @param    reservationInfo     @b{(output)}  Reservation information.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpReservationInformationGet(openapiClientHandle_t *client_handle, uint32_t streamId, uint32_t intIfNum)
{
  open_error_t result;
  OPEN_MSRP_RESERVATION_INFO_t  resInfo;
  open_buffdesc                 reservationInfo;

  reservationInfo.pstart = &resInfo;
  reservationInfo.size   = sizeof(resInfo);

  if ((result = openapiMsrpReservationInformationGet(client_handle, streamId, intIfNum, &reservationInfo)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the MSRP Reservation Information. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP reservation information on interface %d is\n", intIfNum);
    printf("StreamId: \r\n%u  \n MAC addr-%02x:%02x:%02x:%02x:%02x:%02x\n"
	   "Listener type %u  talker type %u\n"
	   "Fail code %u fail interface %u\n",
	   resInfo.streamID, resInfo.streamMacAddr[0], resInfo.streamMacAddr[1], 
	   resInfo.streamMacAddr[2], resInfo.streamMacAddr[3], 
	   resInfo.streamMacAddr[4], resInfo.streamMacAddr[5],
	   resInfo.listenerDeclType, resInfo.talkerDeclType,
	   resInfo.failCode, resInfo.failIntf);
  }
  return;
}


/*********************************************************************
* @purpose  Get the value of specified MSRP counter.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    type       @b{(input)}  Type of counter.
* @param    counter       @b{(output)}  Counter value.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpTrafficPduCounterGet(openapiClientHandle_t *client_handle, OPEN_MSRP_STATS_t type, uint32_t *counter)
{
  open_error_t result;

  if ((result = openapiMsrpTrafficPduCounterGet(client_handle, type, counter)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the value of specified MSRP counter. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP PDU Counter of type %d is %d.\n", type, *counter);
  }
  return;
}


/*********************************************************************
* @purpose  Get the value of specified MSRP counter for an interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.
* @param    type       @b{(input)}  Type of counter.
* @param    counter       @b{(output)}  Counter value.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpIntfTrafficPduCounterGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_MSRP_STATS_t type, uint32_t *counter)
{
  open_error_t result;

  if ((result = openapiMsrpIntfTrafficPduCounterGet(client_handle, intIfNum, type, counter)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the value of specified MSRP counter for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP traffic PDU counter of type %d on interface %d is %d.\n",
	   type, intIfNum, *counter);
  }
  return;
}


/*********************************************************************
* @purpose  Get Failed Registration Counter.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.
* @param    counter       @b{(output)}  Counter value.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpFailedRegistrationCounterGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *counter)
{
  open_error_t result;

  if ((result = openapiMsrpFailedRegistrationCounterGet(client_handle, intIfNum, counter)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get Failed Registration Counter. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP failed registration counter on interface %d is %d\n",
	   intIfNum, *counter);
  }
  return;
}


/*********************************************************************
* @purpose  Get the MSRP Stream Information.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    streamId       @b{(input)}  Stream id.
* @param    streamInfo       @b{(output)}  Stream information.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpStreamInformationGet(openapiClientHandle_t *client_handle, uint32_t streamId)
{
  open_error_t result;
  OPEN_MSRP_STEAM_INFO_t  strInfo;
  open_buffdesc           streamInfo;

  streamInfo.pstart = &strInfo;
  streamInfo.size   = sizeof(strInfo);

  if ((result = openapiMsrpStreamInformationGet(client_handle, streamId, &streamInfo)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the MSRP Stream Information. (result = %d)\n", result);
  }
  else
  {
    printf("\r\nstreamId:%u  \nsourceMac:%02x:%02x:%02x:%02x:%02x:%02x\n"
           "Destination Mac:%02x:%02x:%02x:%02x:%02x:%02x\n"
	   "Latency %u  vlanId %u  Rank %d\n" , strInfo.streamID,
	   strInfo.sourceMacAddr[0], strInfo.sourceMacAddr[1], 
	   strInfo.sourceMacAddr[2], strInfo.sourceMacAddr[3], 
	   strInfo.sourceMacAddr[4], strInfo.sourceMacAddr[5],
	   strInfo.destMacAddr[0], strInfo.destMacAddr[1], 
	   strInfo.destMacAddr[2], strInfo.destMacAddr[3], 
	   strInfo.destMacAddr[4], strInfo.destMacAddr[5],
	   strInfo.accLatency, strInfo.vlanId, strInfo.rank);

  }
  return;
}


/*********************************************************************
* @purpose  Clear MSRP traffic counters on specified interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.

*
* @returns  none
*
* @end
*********************************************************************/
void msrpTrafficCountersPerIfClear(openapiClientHandle_t *client_handle, uint32_t intIfNum)
{
  open_error_t result;

  if ((result = openapiMsrpTrafficCountersPerIfClear(client_handle, intIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to clear MSRP traffic counters on specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("MSGRP Traffic counters cleared on interface %d\n", intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Clear all MSRP traffic counters.
*
* @param    client_handle       @b{(input)}  client handle from registration API

*
* @returns  none
*
* @end
*********************************************************************/
void msrpTrafficCountersClear(openapiClientHandle_t *client_handle)
{
  open_error_t result;

  if ((result = openapiMsrpTrafficCountersClear(client_handle)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to clear all MSRP traffic counters. (result = %d)\n", result);
  }
  else
  {
    printf("All MSGRP Traffic counters cleared \n");
  }
  return;
}



/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for msrp
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  uint64_t trClassB;
  uint32_t maxFanInPorts;
  uint32_t remap;
  uint64_t trClassA;
  OPEN_BOOL_t isBoundary;
  OPEN_MSRP_QAV_STREAMCLASS_t trClass;
  uint32_t counter;
  uint32_t deltaClassA;
  OPEN_MSRP_QAV_STREAMCLASS_t trafficClass;
  uint32_t deltaClassB;
  uint32_t priority;
  uint32_t pvid;
  uint32_t deltaBandwidth;
  uint16_t nextIndex;
  OPEN_CONTROL_t mode;
  uint32_t streamId;
  uint64_t total;
  OPEN_MSRP_STATS_t type;
  uint32_t intIfNum;
  uint32_t strId;
  uint16_t currentIndex;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("msrp example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting msrp API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 2)
      {
        msrpModeGet(&client_handle, &mode);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 3)
      {
	mode = atoi(argv[2]);
        msrpModeSet(&client_handle, mode);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 2)
      {
        msrpBoundaryPropagateModeGet(&client_handle, &mode);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 3)
      {
	mode = atoi(argv[2]);
        msrpBoundaryPropagateModeSet(&client_handle, mode);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 3)
      {
	intIfNum = atoi(argv[2]);
        msrpIntfModeGet(&client_handle, intIfNum, &mode);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 4)
      {
	intIfNum = atoi(argv[2]);
	mode = atoi(argv[3]);
        msrpIntfModeSet(&client_handle, intIfNum, mode);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 4)
      {
	intIfNum = atoi(argv[2]);
	trClass = atoi(argv[3]);
        msrpSrQavPriorityGet(&client_handle, intIfNum, trClass, &priority);
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 5)
      {
	intIfNum = atoi(argv[2]);
	trClass = atoi(argv[3]);
	priority = atoi(argv[4]);
        msrpSrQavPrioritySet(&client_handle, intIfNum, trClass, priority);
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 4)
      {
	intIfNum = atoi(argv[2]);
	trClass = atoi(argv[3]);
        msrpSrQavRePriorityGet(&client_handle, intIfNum, trClass, &remap);
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 5)
      {
	intIfNum = atoi(argv[2]);
	trClass = atoi(argv[3]);
	remap   = atoi(argv[4]);
        msrpSrQavRePrioritySet(&client_handle, intIfNum, trClass, remap);
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 2)
      {
        msrpTalkerPruningModeGet(&client_handle, &mode);
        show_help = 0;
      }
      break;
    case 12:
      if (argc == 3)
      {
	mode = atoi(argv[2]);
        msrpTalkerPruningModeSet(&client_handle, mode);
        show_help = 0;
      }
      break;
    case 13:
      if (argc == 2)
      {
        msrpMaxFanInPortsGet(&client_handle, &maxFanInPorts);
        show_help = 0;
      }
      break;
    case 14:
      if (argc == 3)
      {
	maxFanInPorts = atoi(argv[2]);
        msrpMaxFanInPortsSet(&client_handle, maxFanInPorts);
        show_help = 0;
      }
      break;
    case 15:
      if (argc == 3)
      {
	intIfNum = atoi(argv[2]);
        msrpSRclassPVIDGet(&client_handle, intIfNum, &pvid);
        show_help = 0;
      }
      break;
    case 16:
      if (argc == 4)
      {
	intIfNum = atoi(argv[2]);
	pvid     = atoi(argv[3]);
        msrpSRclassPVIDSet(&client_handle, intIfNum, pvid);
        show_help = 0;
      }
      break;
    case 17:
      if (argc == 3)
      {
	intIfNum = atoi(argv[2]);
        msrpQavDeltaBandwidthGet(&client_handle, intIfNum, &deltaClassA, &deltaClassB);
        show_help = 0;
      }
      break;
    case 18:
      if (argc == 5)
      {
	intIfNum = atoi(argv[2]);
	trafficClass = atoi(argv[3]);
        deltaBandwidth = atoi(argv[4]);
        msrpQavDeltaBandwidthSet(&client_handle, intIfNum, trafficClass, deltaBandwidth);
        show_help = 0;
      }
      break;
    case 19:
      if (argc == 3)
      {
	intIfNum = atoi(argv[2]);
        msrpQavMaxRateGet(&client_handle, intIfNum, &trClassA, &trClassB, &total);
        show_help = 0;
      }
      break;
    case 20:
      if (argc == 3)
      {
	intIfNum = atoi(argv[2]);
        msrpQavCurrentBandwidthGet(&client_handle, intIfNum, &trClassA, &trClassB, &total);
        show_help = 0;
      }
      break;
    case 21:
      if (argc == 4)
      {
	intIfNum = atoi(argv[2]);
	trClass  = atoi(argv[3]);
        msrpIsBoundaryPortGet(&client_handle, intIfNum, trClass, &isBoundary);
        show_help = 0;
      }
      break;
    case 22:
      if (argc == 3)
      {
	currentIndex = atoi(argv[2]);
        msrpStreamIndexedNextGet(&client_handle, currentIndex, &nextIndex);
        show_help = 0;
      }
      break;
    case 23:
      if (argc == 4)
      {
	intIfNum = atoi(argv[2]);
	strId    = atoi(argv[3]);
        msrpReservationInformationGet(&client_handle, strId, intIfNum);
        show_help = 0;
      }
      break;
    case 24:
      if (argc == 3)
      {
	type = atoi(argv[2]);
        msrpTrafficPduCounterGet(&client_handle, type, &counter);
        show_help = 0;
      }
      break;
    case 25:
      if (argc == 4)
      {
	intIfNum = atoi(argv[2]);
	type     = atoi(argv[3]);
        msrpIntfTrafficPduCounterGet(&client_handle, intIfNum, type, &counter);
        show_help = 0;
      }
      break;
    case 26:
      if (argc == 3)
      {
	intIfNum = atoi(argv[2]);
        msrpFailedRegistrationCounterGet(&client_handle, intIfNum, &counter);
        show_help = 0;
      }
      break;
    case 27:
      if (argc == 3)
      {
	streamId = atoi(argv[2]);
        msrpStreamInformationGet(&client_handle, streamId);
        show_help = 0;
      }
      break;
    case 28:
      if (argc == 3)
      {
	intIfNum = atoi(argv[2]);
        msrpTrafficCountersPerIfClear(&client_handle, intIfNum);
        show_help = 0;
      }
      break;
    case 29:
      if (argc == 2)
      {
        msrpTrafficCountersClear(&client_handle);
        show_help = 0;
      }
      break;
    case 30:
      if (argc == 2)
      {
        runSanity(&client_handle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping msrp API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}
