
/*********************************************************************
*
* Copyright 2017-2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  sflow_example.c
*
* @purpose   sFlow Example
*
* @component OpEN
*
* @comments
*
* @create    01/27/2017
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "openapi_common.h"
#include "openapi_sflow.h"
#include "openapi_acl.h"

#define PRINTIFBADRESULT(result, msg) \
if (result==OPEN_E_UNAVAIL) { printf("Feature not supported - %s. (%d)\n", msg, result); } \
else if (result!=OPEN_E_NONE) { printf("Error returned from %s. (%d)\n", msg, result); }

open_error_t testSflowReceiver(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t receiverCount;
  uint32_t ownerStringLen;
  char testOwnerName[] = "test_owner";
  open_buffdesc owner;
  uint32_t rcvrIndex;
  int32_t timeout;
  open_inet_addr_t ipAddr;
  uint32_t testIpAddr;
  uint32_t testPort;
  uint32_t port;
  uint32_t minDatagramSize, maxDatagramSize, testDatagramSize, datagramSize;

  receiverCount = openapiSflowRcvrCountGet(clientHandle);
  if (receiverCount == 0)
  {
    printf("Switch reports supporting %d sFlow receiver entries.\n", receiverCount);
    return OPEN_E_UNAVAIL;
  }

  ownerStringLen = openapiSflowRcvrOwnerStringLenGet(clientHandle);

  /* create an entry in the receiver table */
  owner.pstart = testOwnerName;
  owner.size = strlen(owner.pstart) + 1;
  rcvrIndex = receiverCount;
  /* set entry to never time out */
  timeout = OPEN_SFLOW_RCVR_NO_TIMEOUT;
  result = openapiSflowRcvrOwnerSet(clientHandle, rcvrIndex, &owner, timeout);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowRcvrOwnerSet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }

  owner.pstart = calloc(1, ownerStringLen);
  if (!owner.pstart)
  {
    printf("Failed allocating owner string buffer.\n");
    return OPEN_E_ERROR;
  }
  owner.size = ownerStringLen;
  result = openapiSflowRcvrOwnerGet(clientHandle, rcvrIndex, &owner);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowRcvrOwnerGet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }
  if (strncmp(owner.pstart, testOwnerName, owner.size) != 0)
  {
    printf("Owner string retrieved does not match what was set.(set = %s, retrieved = %s)\n",
           testOwnerName, (char *)owner.pstart);
    return OPEN_E_ERROR;
  }

  result = openapiSflowRcvrTimeRemainingGet(clientHandle, rcvrIndex, &timeout);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowRcvrTimeRemainingGet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }
  if (timeout != OPEN_SFLOW_RCVR_NO_TIMEOUT)
  {
    printf("Unexpected value returned for receiver entry time remaining. (Expected: %d (OPEN_SFLOW_RCVR_NO_TIMEOUT_, Retrieved: %d)\n",
           OPEN_SFLOW_RCVR_NO_TIMEOUT, timeout);
    return OPEN_E_ERROR;
  }

  testIpAddr = 0x01020304;  /* IPv4 address: 1.2.3.4 */
  ipAddr.family = OPEN_AF_INET;
  ipAddr.addr.ipv4 = testIpAddr;
  result = openapiSflowRcvrAddressSet(clientHandle, rcvrIndex, &ipAddr);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowRcvrAddressSet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }

  memset(&ipAddr, 0, sizeof(ipAddr));
  result = openapiSflowRcvrAddressGet(clientHandle, rcvrIndex, &ipAddr);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowRcvrAddressGet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }
  if (ipAddr.family != OPEN_AF_INET)
  {
    printf("Unexpected value for address family returned for receiver IPv4 address. (Expected: %d (OPEN_AF_INET), Retrieved: %d)\n",
           OPEN_AF_INET, ipAddr.family);
    return OPEN_E_ERROR;
  }
  if (ipAddr.addr.ipv4 != testIpAddr)
  {
    printf("Unexpected value returned for receiver IPv4 address. (Expected: 0x%x, Retrieved: 0x%x)\n",
           testIpAddr, ipAddr.addr.ipv4);
    return OPEN_E_ERROR;
  }

  testPort = 654;
  result = openapiSflowRcvrPortSet(clientHandle, rcvrIndex, testPort);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowRcvrPortSet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }

  port = 0;
  result = openapiSflowRcvrPortGet(clientHandle, rcvrIndex, &port);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowRcvrPortSet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }
  if (port != testPort)
  {
    printf("Unexpected value returned for receiver port. (Expected: 0x%x, Retrieved: 0x%x)\n",
           testPort, port);
    return OPEN_E_ERROR;
  }

  result = openapiSflowRcvrMaxDatagramSizeRangeGet(clientHandle, &minDatagramSize, &maxDatagramSize);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowRcvrMaxDatagramSizeRangeGet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }

  /* derive a test value halfway between min and max datagram size limits */
  testDatagramSize = (minDatagramSize + ((maxDatagramSize - minDatagramSize) / 2));
  result = openapiSflowRcvrMaxDatagramSizeSet(clientHandle, rcvrIndex, testDatagramSize);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowRcvrMaxDatagramSizeRangeSet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }

  datagramSize = 0;
  result = openapiSflowRcvrMaxDatagramSizeGet(clientHandle, rcvrIndex, &datagramSize);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowRcvrMaxDatagramSizeGet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }
  if (datagramSize != testDatagramSize)
  {
    printf("Unexpected value returned for datagram size. (Expected: %d, Retrieved: %d)\n",
           testDatagramSize, datagramSize);
    return OPEN_E_ERROR;
  }

  /* remove the test receiver entry */
  timeout = 0;
  result = openapiSflowRcvrOwnerSet(clientHandle, rcvrIndex, &owner, timeout);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowRcvrOwnerSet() while attempting to remove entry. (%d)\n", result);
    return OPEN_E_ERROR;
  }

  return OPEN_E_NONE;
}

open_error_t testSflowSamplerPoller(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t ifNum;
  open_buffdesc owner;
  char testOwnerName[] = "TestOwner";
  uint32_t rcvrIndex, testRcvrIndex;
  uint32_t instanceMin, instanceMax, instance;
  uint32_t samplingRateMin, samplingRateMax, samplingRate, testSamplingRate;
  uint32_t headerSizeMin, headerSizeMax, headerSize, testHeaderSize;
  int32_t timeout;
  open_buffdesc aclDesc;
  char aclName[] = "TestIpAcl";
  uint32_t aclId, aclRule;
  uint32_t minInterval, maxInterval, interval, testInterval;

  /* retrieve the first physical interface for test */
  result = openapiIfFirstGet(clientHandle, OPEN_INTF_TYPE_PHY, &ifNum);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiIfFirstGet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }

  /* create an entry in the receiver table */
  owner.pstart = testOwnerName;
  owner.size = strlen(owner.pstart) + 1;
  rcvrIndex = 1;
  /* set entry to never time out */
  timeout = OPEN_SFLOW_RCVR_NO_TIMEOUT;
  result = openapiSflowRcvrOwnerSet(clientHandle, rcvrIndex, &owner, timeout);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowRcvrOwnerSet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }

  result = openapiSflowInstanceRangeGet(clientHandle, &instanceMin, &instanceMax);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowSamplerInstanceRangeGet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }

  instance = instanceMin;

  result = openapiSflowFsRcvrIndexSet(clientHandle, ifNum, instance, rcvrIndex);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowFsRcvrIndexSet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }

  result = openapiSflowFsRcvrIndexGet(clientHandle, ifNum, instance, &testRcvrIndex);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowFsRcvrIndexGet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }

  if (rcvrIndex != testRcvrIndex)
  {
    printf("Value returned from openapiSflowFsRcvrIndexGet() does not match what was set. "
           "expected = %d, returned = %d\n", rcvrIndex, testRcvrIndex);
    return OPEN_E_ERROR;
  }

  result = openapiSflowFsPacketSamplingRateRangeGet(clientHandle, &samplingRateMin, &samplingRateMax);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowSamplerInstanceRangeGet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }

  samplingRate = samplingRateMin;

  result = openapiSflowFsPacketSamplingRateSet(clientHandle, ifNum, instance, OPEN_SFLOW_SAMPLING_TYPE_INGRESS, samplingRate);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowFsPacketSamplingRateSet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }

  result = openapiSflowFsPacketSamplingRateGet(clientHandle, ifNum, instance, OPEN_SFLOW_SAMPLING_TYPE_INGRESS, &testSamplingRate);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowFsPacketSamplingRateGet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }

  if (samplingRate != testSamplingRate)
  {
    printf("Value returned from openapiSflowFsPacketSamplingRateGet() does not match what was set. "
           "expected = %d, returned = %d\n", samplingRate, testSamplingRate);
    return OPEN_E_ERROR;
  }

  result = openapiSflowFsMaximumHeaderSizeRangeGet(clientHandle, &headerSizeMin, &headerSizeMax);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowFsMaximumHeaderSizeRangeGet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }

  headerSize = headerSizeMax;

  result = openapiSflowFsMaximumHeaderSizeSet(clientHandle, ifNum, instance, headerSize);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowFsMaximumHeaderSizeSet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }

  result = openapiSflowFsMaximumHeaderSizeGet(clientHandle, ifNum, instance, &testHeaderSize);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowFsMaximumHeaderSizeGet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }

  if (headerSize != testHeaderSize)
  {
    printf("Value returned from openapiSflowFsMaximumHeaderSizeGet() does not match what was set. "
           "expected = %d, returned = %d\n", headerSize, testHeaderSize);
    return OPEN_E_ERROR;
  }

  /* test IP ACL flow sampler */
  aclDesc.pstart = aclName;
  aclDesc.size = strlen(aclDesc.pstart) + 1;

  result = openapiAclCreate(clientHandle, OPEN_ACL_TYPE_IP, &aclDesc, &aclId);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiAclCreate(). (%d)\n", result);
  }
  else
  {  
    aclRule = 0; /* Let the system determine first available rule number */
    result = openapiAclRuleActionAdd(clientHandle, aclId, &aclRule, OPEN_ACL_PERMIT);
    if (result != OPEN_E_NONE)
    {
      printf("Error returned from openapiAclRuleActionAdd(). (%d)\n", result);
    }
    else
    {
      /* Add match criteria - protocol UDP */
      {
        result = openapiAclRuleMatchProtocolAdd(clientHandle, aclId, aclRule, OPENAPI_ACL_PROTOCOL_UDP);
        if (result != OPEN_E_NONE)
        {
          printf("Error returned from openapiAclRuleMatchProtocolAdd(). (%d)\n", result);
        }
        else
        {
          result = openapiSflowSamplerIpAclSet(clientHandle, instance, ifNum, aclId, OPEN_ENABLE);
          if (result != OPEN_E_NONE)
          {
            printf("Error returned from openapiSflowSamplerIpAclSet() adding ACL. (%d)\n", result);
          }
          else
          {
            result = openapiSflowSamplerIpAclSet(clientHandle, instance, ifNum, aclId, OPEN_DISABLE);
            if (result != OPEN_E_NONE)
            {
              printf("Error returned from openapiSflowSamplerIpAclSet() removing ACL. (%d)\n", result);
            }
          }
        }
      }
    }
    openapiAclDelete(clientHandle, aclId);
  }

  result = openapiSflowCpRcvrIndexSet(clientHandle, ifNum, instance, rcvrIndex);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowCpRcvrIndexSet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }

  /* test counter poller related APIs */
  result = openapiSflowCpRcvrIndexGet(clientHandle, ifNum, instance, &testRcvrIndex);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowCpRcvrIndexGet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }

  if (rcvrIndex != testRcvrIndex)
  {
    printf("Value returned from openapiSflowCpRcvrIndexGet() does not match what was set. "
           "expected = %d, returned = %d\n", rcvrIndex, testRcvrIndex);
    return OPEN_E_ERROR;
  }

  result = openapiSflowCpIntervalRangeGet(clientHandle, &minInterval, &maxInterval);

  interval = maxInterval;

  result = openapiSflowCpIntervalSet(clientHandle, ifNum, instance, interval);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowCpIntervalSet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }

  result = openapiSflowCpIntervalGet(clientHandle, ifNum, instance, &testInterval);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowCpIntervalGet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }

  if (interval != testInterval)
  {
    printf("Value returned from openapiSflowCpIntervalGet() does not match what was set. "
           "expected = %d, returned = %d\n", interval, testInterval);
    return OPEN_E_ERROR;
  }

  /* remove the test receiver entry */
  timeout = 0;
  result = openapiSflowRcvrOwnerSet(clientHandle, rcvrIndex, &owner, timeout);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowRcvrOwnerSet() while attempting to remove entry. (%d)\n", result);
    return OPEN_E_ERROR;
  }

  return OPEN_E_NONE;
}

open_error_t testSflowRemoteAgent(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t ifNum;
  uint32_t ifNumDest;
  uint32_t instanceMin, instanceMax, instance;
  uint32_t remoteAgentIndex;
  uint32_t monitorSessionId;
  open_inet_addr_t ipAddr;

  /* retrieve the first physical interface for test */
  result = openapiIfFirstGet(clientHandle, OPEN_INTF_TYPE_PHY, &ifNum);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiIfFirstGet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }
  result = openapiIfNextGet(clientHandle, OPEN_INTF_TYPE_PHY, ifNum, &ifNumDest);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiIfNextGet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }

  remoteAgentIndex = openapiSflowFsRemoteAgentMaxIndexGet(clientHandle);
  if (remoteAgentIndex == 0)
  {
    printf("Platform supports %d remote agents.\n", remoteAgentIndex);
    return OPEN_E_UNAVAIL;
  }

  monitorSessionId = openapiSflowFsRemoteAgentMaxMonitorSessionIdGet(clientHandle);
  if (monitorSessionId == 0)
  {
    printf("Platform supports %d remote agent monitor sessions.\n", monitorSessionId);
    return OPEN_E_UNAVAIL;
  }

  result = openapiSflowInstanceRangeGet(clientHandle, &instanceMin, &instanceMax);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowSamplerInstanceRangeGet(). (%d)\n", result);
    return OPEN_E_ERROR;
  }
  instance = instanceMin;

  ipAddr.family = OPEN_AF_INET;
  ipAddr.addr.ipv4 = 0x01020304;  /* 1.2.3.4 */
  result = openapiSflowRemoteAgentAddressSet(clientHandle, remoteAgentIndex, &ipAddr);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowRemoteAgentAddressSet(). (%d)\n", result);
  }

  result = openapiSflowRemoteAgentUdpDestPortSet(clientHandle, remoteAgentIndex, 999);
  if (result != OPEN_E_NONE)
  {
    printf("Error returned from openapiSflowRemoteAgentUdpDestPortSet(). (%d)\n", result);
  }

  result = openapiSflowRemoteAgentMonitorSessionSet(clientHandle, remoteAgentIndex, monitorSessionId);
	PRINTIFBADRESULT(result, "openapiSflowRemoteAgentMonitorSessionSet()");

  result = openapiSflowRemoteAgentMonitorSessionDestIfaceSet(clientHandle, remoteAgentIndex, ifNumDest);
	PRINTIFBADRESULT(result, "openapiSflowRemoteAgentMonitorSessionDestIfaceSet()");

  result = openapiSflowFsRemoteAgentIndexSet(clientHandle, ifNum, instance, remoteAgentIndex);
	PRINTIFBADRESULT(result, "openapiSflowFsRemoteAgentIndexSet()");

  return OPEN_E_NONE;
}

int main (int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;

  /* Register with OpEN */
  if ((result = openapiClientRegister ("sflow_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit (2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck (&clientHandle) != OPEN_E_NONE)
  {
    sleep (1);
  }

  printf("\n");
  printf("Begin sFlow tests...\n");

  /* Receiver APIs */
  printf("\tTesting sFlow receiver APIs...\n");
  result = testSflowReceiver(&clientHandle);
  if (result != OPEN_E_NONE)
  {
    printf("\t\tError.\n");
  }
  else
  {
    printf("\t\tSuccess.\n");
  }

  /* Sampler APIs */
  printf("\tTesting sFlow sampler and poller APIs...\n");
  result = testSflowSamplerPoller(&clientHandle);
  if (result != OPEN_E_NONE)
  {
    printf("\t\tError.\n");
  }
  else
  {
    printf("\t\tSuccess.\n");
  }

  /* Remote Agent APIs */
  printf("\tTesting sFlow remote agent APIs...\n");
  result = testSflowRemoteAgent(&clientHandle);
  if (result != OPEN_E_NONE)
  {
    printf("\t\tError.\n");
  }
  else
  {
    printf("\t\tSuccess.\n");
  }

  printf("Complete.\n");

  return 0;
}

