/*********************************************************************
*
* Copyright 2016-2019 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  ssh_example.c
*
* @purpose   SSH OpEN APIs Example 
*
* @component OpEN
*
* @comments
*
* @create    29/04/2013
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_ssh.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the ssh
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/
/*******************************************************************
*
* @brief  This function prints the SSH Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printSSHAppMenu()
{
  printf("Usage:  ssh_example <test#> <arg1> <arg2> ... \n");
  printf("Usage:  In below, <key-type> should be set to 0 for RSA, and 1 for DSA\n");
  printf("Usage:  In below, <protocol_version> should be set to 0 for OPEN_SSH_PROTO_VERSION_1, and 1 for OPEN_SSH_PROTO_VERSION_2\n");
  printf("Usage:  In below, <mode> should be set to 0 for OPEN_DISABLE and 1 for OPEN_ENABLE\n");
  printf("Test 1: Generate SSH key: ssh_example 1 <key-type> \n");
  printf("Test 2: Get SSH key status: ssh_example 2 <key-type> \n");
  printf("Test 3: Delete SSH key: ssh_example 3 <key-type>\n");
  printf("Test 4: Set SSH admin mode: ssh_example 4 <mode>\n");
  printf("Test 5: Get SSH admin mode: ssh_example 5 \n");
  printf("Test 6: Set SSH protocol version: ssh_example 6 <protocol-version> \n");
  printf("Test 7: Get SSH protocol version: ssh_example 7 \n");
  printf("Test 8: Set SSH session idle timeout value: ssh_example 8 <timeout>\n");
  printf("Test 9: Get SSH session idle timeout value: ssh_example: ssh_example 9 \n");
  printf("Test 10: Set maximum number of allowed SSH sessions: ssh_example 10 <max-sessions>\n");
  printf("Test 11: Set maximum number of allowed SSH sessions to default: ssh_example 11 \n");
  printf("Test 12: Get maximum number of allowed SSH sessions: ssh_example 12 \n");
  printf("Test 13: Get maximum number of active SSH sessions: ssh_example 13 \n");
  printf("Test 14: SSH OpEN API Sanity: ssh_example 14 \n");
  printf("Test 15: Set SCP Server admin mode: ssh example 15 <mode>\n");
  printf("Test 16: Get SCP server admin mode: ssh_example 16 \n");
  printf("Test 17: SCP server file transfer start: ssh_example 17  <filename> \n");
  printf("Test 18: Set SCP server file transfer complete status: ssh_example 18 \n");
  printf("Test 19: Get the remote server password for a given client socket ID for outbound SSH: ssh_example 19 <clientSockID> \n");
  return;
}

/*********************************************************************
* @purpose  Generate SSH key.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    keyType          @b{(input)}   SSH key type
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void sshKeyGenerate(openapiClientHandle_t *clientHandle, OPEN_SSH_KEY_TYPE_t keyType)
{
  open_error_t result;

  if ((result = openapiSSHKeyGenerate(clientHandle, keyType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to generate SSH key. (result = %d)\n", result);
  }
  else
  {
    if (keyType == OPEN_SSH_KEY_TYPE_RSA)
    {
      printf("RSA");
    }
    else
    {
      printf("DSA");
    }
    printf(" key generated successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the status of the SSH key.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    keyType          @b{(input)}   SSH key type
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void sshKeyStatusGet(openapiClientHandle_t *clientHandle, OPEN_SSH_KEY_TYPE_t keyType)
{
  open_error_t result;
  char str[20];

  if (keyType == OPEN_SSH_KEY_TYPE_RSA)
  {
    strcpy(str, "RSA");
  }
  else if (keyType == OPEN_SSH_KEY_TYPE_DSA)
  {
    strcpy(str, "DSA");
  }
  else
  {
    memset(str, 0, sizeof(str));
  }
 
  
  result = openapiSSHKeyStatusGet(clientHandle, keyType);

  if (result == OPEN_E_NONE)
  {
    printf(" %s key is present. \n", str);
  }
  else if (result == OPEN_E_FAIL)
  {
    printf(" %s key generation is in progress. \n", str);
  }
  else if (result == OPEN_E_NOT_FOUND)
  {
    printf(" %s key is not found. \n", str);
  }
  else
  {
    printf("Bad return code trying to get SSH keys status. (resul = %d)\n", result);
  }

  
  return;
}

/*********************************************************************
* @purpose  Delete SSH key.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    keyType          @b{(input)}   SSH key type
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void sshKeyDelete(openapiClientHandle_t *clientHandle, OPEN_SSH_KEY_TYPE_t keyType)
{
  open_error_t result;

  if ((result = openapiSSHKeyDelete(clientHandle, keyType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete SSH key. (result = %d)\n", result);
  }
  else
  {
    if (keyType == OPEN_SSH_KEY_TYPE_RSA)
    {
      printf("RSA");
    }
    else
    {
      printf("DSA");
    }
    printf(" key deleted successfully. (result = %d)\n", result);
  }

  return;
} 

/*********************************************************************
* @purpose  Set SSH admin mode.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    mode             @b{(input)}   SSH admin mode
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void sshAdminModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiSSHAdminModeSet(clientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set SSH admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("SSH admin mode set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get SSH admin mode.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void sshAdminModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  if ((result = openapiSSHAdminModeGet(clientHandle, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set SSH admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("SSH admin mode: %d (result = %d)\n", mode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set SSH protocol version.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    protoVersion     @b{(input)}   SSH protocol version
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void sshProtocolVersionSet(openapiClientHandle_t *clientHandle, OPEN_SSH_PROTO_VERSION_t protoVersion)
{
  open_error_t result;

  if ((result = openapiSSHProtocolVersionSet(clientHandle, protoVersion)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set SSH protocol version. (result = %d)\n", result);
  }
  else
  {
    printf("SSH protocol version is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get SSH protocol version.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void sshProtocolVersionGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_SSH_PROTO_VERSION_t protoVersion;

  if ((result = openapiSSHProtocolVersionGet(clientHandle, &protoVersion)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get SSH protocol version. (result = %d)\n", result);
  }
  else
  {
    printf("SSH protocol version: %d (result = %d)\n", protoVersion, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set SSH session timeout value.
* 
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    sshSessionTimeout  @b{(input)}   SSH session timeout value
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void sshSessionTimeoutSet(openapiClientHandle_t *clientHandle, uint32_t sshSessionTimeout)
{
  open_error_t result;

  if ((result = openapiSSHSessionTimeoutSet(clientHandle, sshSessionTimeout)) != OPEN_E_NONE)
  { 
    printf("Bad return code trying to set SSH session timeout. (result = %d)\n", result);
  }
  else
  { 
    printf("SSH session timeout is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get SSH session timeout value
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void sshSessionTimeoutGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t sshSessionTimeout;

  if ((result = openapiSSHSessionTimeoutGet(clientHandle, &sshSessionTimeout)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get SSH session timeout. (result = %d)\n", result);
  }
  else
  {
    printf("SSH session timeout: %d. (result = %d)\n", sshSessionTimeout, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set maximum number of SSH sessions allowed.
* 
* @param    clientHandle       @b{(input)}   client handle from registration API
* @param    maxSshSessions     @b{(input)}   Maximum SSH sessions allowed
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void sshMaxSessionsSet(openapiClientHandle_t *clientHandle, uint32_t maxSshSessions)
{
  open_error_t result;

  if ((result = openapiSSHMaxSessionsSet(clientHandle, maxSshSessions)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set maximum SSH sessions allowed. (result = %d)\n", result);
  }
  else
  {
    printf("Maximum number of SSH sessions is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Set maximum number of SSH sessions allowed to default.
* 
* @param    clientHandle       @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void sshMaxSessionsDefaultSet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  if ((result = openapiSSHMaxSessionsDefaultSet(clientHandle)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set maximum SSH sessions allowed to default. (result = %d)\n", result);
  }
  else
  {
    printf("Maximum number of SSH sessions is set to default successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get maximum number of SSH sessions allowed.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void sshMaxSessionsGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t maxSshSessions;

  if ((result = openapiSSHMaxSessionsGet(clientHandle, &maxSshSessions)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get maximum SSH sessions allowed. (result = %d)\n", result);
  }
  else
  {
    printf("Maximum number of SSH sessions: %d. (result = %d)\n", maxSshSessions, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the number of active SSH sessions.
* 
* @param    clientHandle  @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void sshActiveSessionsGet(openapiClientHandle_t *clientHandle) 
{
  open_error_t result;
  uint32_t activeSshSessions;

  if ((result = openapiSSHActiveSessionsGet(clientHandle, &activeSshSessions)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get active SSH sessions. (result = %d)\n", result);
  }
  else
  {
    printf("Number of active SSH sessions: %d. (result = %d)\n", activeSshSessions, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set SCP server admin mode.
*
* @param    clientHandle  @b{(input)}   client handle from registration API
* @param    mode          @b{(input)}   SSH admin mode
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void scpServerAdminModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiScpServerAdminModeSet(clientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set SCP server admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("SCP server admin mode set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get SCP server admin mode.
*
* @param    clientHandle  @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void scpServerAdminModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  if ((result = openapiScpServerAdminModeGet(clientHandle, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set SCP server  admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("SCP server admin mode: %s (result = %d)\n", (mode == OPEN_ENABLE ? "enable" : "disable"), result);
  }

  return;
}

/*********************************************************************
* @purpose  Initialte SCP server file transfer. 
*
* @param    clientHandle  @b{(input)}  client handle from registration API
* @param    fileParams    @b{(inout)}  contains the download file name 
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void scpServerRemoteTransferStart (openapiClientHandle_t *clientHandle,
                                   open_buffdesc *fileParams)
{
  open_error_t result;

  if ((result = openapiScpRemoteTransferStart(clientHandle, getpid(), fileParams)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to start SCP server remote file transfer (result = %d)\n", result);
  }
  else
  {
    printf("SCP server start remote file  transfer : (result = %d)\n", result);
  }
  return;
}

/*********************************************************************
* @purpose  To update appication about the completion of scp remote file transfer.
*
* @param    client_handle   @b{(input)} Client handle from registration API.
* @param    status          @b{(input)} SCP remote server transfer complete status.
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void ScpServerRemoteTransferComplete (openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  if ((result = openapiScpRemoteTransferComplete(clientHandle)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set SCP transfer complete (result = %d)\n", result);
  }
  else
  {
    printf("SCP server file transfer complete : (result = %d)\n", result);
  }
  return;
}

/*********************************************************************************
* @purpose  Get the remote server password for a client socket ID for outbound SSH.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    clientSockID        @b{(input)}  client socket ID
* @param    password            @b{(output)} remote server password 
*
* @returns  none
* 
* @end
*********************************************************************/
void sshRemotePasswordGet(openapiClientHandle_t *client_handle,
                          uint32_t clientSockID,  open_buffdesc *password)
{ 
  open_error_t result = OPEN_E_NONE;

  if ((result = openapiSSHRemotePasswordGet(client_handle, clientSockID, password)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the password for a client socket ID. (result = %d)\n", result);
  }
  else
  {
    printf("Password is returned successfully : (result = %d)\n", result); 
  }
  return;
}

/*********************************************************************
* @purpose  SSH OpEN APIs Sanity
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void sshOpENAPISanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_error_t sanityResult = OPEN_E_NONE;
  OPEN_SSH_KEY_TYPE_t keyType = OPEN_SSH_KEY_TYPE_RSA;
  OPEN_CONTROL_t mode = OPEN_DISABLE;
  OPEN_SSH_PROTO_VERSION_t protoVersion = OPEN_SSH_PROTO_VERSION_2; 
  uint32_t activeSshSessions = 0;
  uint32_t maxSshSessions = 0;
  uint32_t sshSessionTimeout = 0;
  open_buffdesc fileParams;
  open_buffdesc password;
  uint32_t clientSockID = 1;


  /* openapiSSHKeyGenerate() */ 
  printf("\nTesting openapiSSHKeyGenerate(): \n");

  result = openapiSSHKeyGenerate(NULL, keyType);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSSHKeyGenerate(clientHandle, 4);
  printf("Invalid SSH key. (result = %d)\n", result);

  printf("openapiSSHKeyGenerate() sanity successful.\n");

  /* openapiSSHKeyDelete() */ 
  printf("\nTesting openapiSSHKeyDelete(): \n");

  result = openapiSSHKeyDelete(NULL, keyType);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSSHKeyDelete(clientHandle, 4);
  printf("Invalid SSH key. (result = %d)\n", result);

  printf("openapiSSHKeyDelete() sanity successful.\n");

  /* openapiSSHKeyStatusGet() */
  printf("\nTesting openapiSSHKeyStatusGet(): \n");

  openapiSSHKeyStatusGet(NULL, keyType);
  printf("NULL Client Handle. (result = %d)\n", result);
  
  openapiSSHKeyStatusGet(clientHandle, 4);
  printf("Invalid SSH key. (result = %d)\n", result);

  printf("openapiSSHKeyStatusGet() sanity successful.\n");

  /* openapiSSHAdminModeSet() */  
  printf("\nTesting openapiSSHAdminModeSet(): \n");

  result = openapiSSHAdminModeSet(NULL, mode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSSHAdminModeSet(clientHandle, 4);
  printf("Invalid SSH admin mode. (result = %d)\n", result); 

  printf("openapiSSHKeyStatusSet() sanity successful.\n");

  /* openapiSSHAdminModeGet() */
  printf("\nTesting openapiSSHAdminModeGet(): \n");

  result = openapiSSHAdminModeGet(NULL, &mode);
  printf("NULL Client Handle. (result = %d)\n", result);
  
  result = openapiSSHAdminModeGet(clientHandle, NULL);
  printf("SSH admin mode. (result = %d)\n", result);
  
  printf("openapiSSHKeyStatusGet() sanity successful.\n");

  /* openapiSSHProtocolVersionSet() */
  printf("\nTesting openapiSSHProtocolVersionSet(): \n");

  result = openapiSSHProtocolVersionSet(NULL, protoVersion); 
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSSHProtocolVersionSet(clientHandle, 5); 
  printf("Invalid SSH protocol version. (result = %d)\n", result);

  printf("openapiSSHProtocolVersionSet() sanity successful.\n");

  /* openapiSSHProtocolVersionGet() */
  printf("\nTesting openapiSSHProtocolVersionGet(): \n");

  result = openapiSSHProtocolVersionGet(NULL, &protoVersion);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSSHProtocolVersionGet(clientHandle, NULL);
  printf("SSH protocol version. (result = %d)\n", result);

  printf("openapiSSHProtocolVersionGet() sanity successful.\n");

  /* openapiSSHSessionTimeoutSet() */  
  printf("\nTesting openapiSSHSessionTimeoutSet(): \n");

  result = openapiSSHSessionTimeoutSet(NULL, sshSessionTimeout);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSSHSessionTimeoutSet(clientHandle, 200);
  printf("Invalid SSH timeout value. (result = %d)\n", result);

  printf("openapiSSHSessionTimeoutSet() sanity successful.\n");

  /* openapiSSHSessionTimeoutGet() */
  printf("\nTesting openapiSSHSessionTimeoutGet(): \n");

  result = openapiSSHSessionTimeoutGet(NULL, &sshSessionTimeout);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSSHSessionTimeoutGet(clientHandle, NULL);
  printf("NULL SSH timeout value. (result = %d)\n", result);

  printf("openapiSSHSessionTimeoutGet() sanity successful.\n");

  /* openapiSSHMaxSessionsSet() */
  printf("\nTesting openapiSSHMaxSessionsSet(): \n");

  result = openapiSSHMaxSessionsSet(NULL, maxSshSessions);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSSHMaxSessionsSet(clientHandle, 10);
  printf("Invalid maximum SSH sessions. (result = %d)\n", result);

  printf("openapiSSHMaxSessionsSet() sanity successful.\n");

  /* openapiSSHMaxSessionsDefaultSet() */
  printf("\nTesting openapiSSHMaxSessionsDefaultSet(): \n");

  result = openapiSSHMaxSessionsDefaultSet(NULL);
  printf("NULL Client Handle. (result = %d)\n", result);

  printf("openapiSSHMaxSessionsDefaultSet() sanity successful.\n");

  /* openapiSSHMaxSessionsGet() */
  printf("\nTesting openapiSSHMaxSessionsGet(): \n");

  result = openapiSSHMaxSessionsGet(NULL, &maxSshSessions);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSSHMaxSessionsGet(clientHandle, NULL);
  printf("NULL maximum SSH sessions. (result = %d)\n", result);

  printf("openapiSSHMaxSessionsGet() sanity successful.\n");

  /* openapiSSHActiveSessionsGet() */
  printf("\nTesting openapiSSHActiveSessionsGet(): \n");

  result = openapiSSHActiveSessionsGet(NULL, &activeSshSessions);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiSSHActiveSessionsGet(clientHandle, NULL);
  printf("NULL active SSH sessions. (result = %d)\n", result);

  printf("openapiSSHActiveSessionsGet() sanity successful.\n");

  /* openapiScpServerAdminModeSet() */
  printf("\nTesting openapiScpServerAdminModeSet(): \n");

  sanityResult = OPEN_E_NONE;

  /* client handle is NULL. The result should be failed */ 
  result = openapiScpServerAdminModeSet(NULL, mode);
  printf("NULL Client Handle. (result = %d)\n", result);
  if (OPEN_E_NONE == result) 
  {
    sanityResult = OPEN_E_FAIL;
  }
  /* Invalid scp admin mode. The result should be failed */ 
  result = openapiScpServerAdminModeSet(clientHandle, 4);
  printf("Invalid SCP Server admin mode. (result = %d)\n", result);
  if (OPEN_E_NONE == result) 
  {
    sanityResult = OPEN_E_FAIL;
  }

  if (OPEN_E_NONE == sanityResult)
  {
    printf("openapiScpServerAdminModeSet() sanity successful.\n");
  }
  else 
  {
    printf("openapiScpServerAdminModeSet() sanity failed.\n");
  }

  /* openapiScpServerAdminModeGet() */
  printf("\nTesting openapiScpServerAdminModeGet(): \n");

  sanityResult = OPEN_E_NONE;

  /* client handle is NULL. The result should be failed */ 
  result = openapiScpServerAdminModeGet(NULL, &mode);
  printf("NULL Client Handle. (result = %d)\n", result);
  if (OPEN_E_NONE == result) 
  {
    sanityResult = OPEN_E_FAIL;
  }
  /* scp admin mode input argument is NULL. The result should be failed */ 
  result = openapiScpServerAdminModeGet(clientHandle, NULL);
  printf("SCP sever admin mode. (result = %d)\n", result);
  if (OPEN_E_NONE == result) 
  {
    sanityResult = OPEN_E_FAIL;
  }

  if (OPEN_E_NONE == sanityResult)
  {
    printf("openapiScpServerAdminModeGet() sanity successful.\n");
  }
  else 
  {
    printf("openapiScpServerAdminModeGet() sanity failed.\n");
  }

  /* "openapiScpRemoteTransferStart() */
  printf("\nTesting openapiScpRemoteTransferStart(): \n");

  sanityResult = OPEN_E_NONE;

  /* client handle is NULL. The result should be failed */ 
  result = openapiScpRemoteTransferStart(NULL, getpid(), &fileParams);
  printf("NULL Client Handle. (result = %d)\n", result);
  if (OPEN_E_NONE == result)
  {
    sanityResult = OPEN_E_FAIL;
  }

  /* file name argument is NULL. The result should be failed */ 
  result = openapiScpRemoteTransferStart(clientHandle, getpid(), NULL);
  printf("Invalid SCP transfer file name. (result = %d)\n", result);
  if (OPEN_E_NONE == result)
  {
    sanityResult = OPEN_E_FAIL;
  }

  if (OPEN_E_NONE == sanityResult)
  {
    printf("openapiScpRemoteTransferStart() sanity successful.\n");
  }
  else
  {
    printf("openapiScpRemoteTransferStart() sanity failed.\n");
  }

  /* openapiScpRemoteTransferComplete() */
  printf("\nTesting openapiScpRemoteTransferComplete(): \n");

  sanityResult = OPEN_E_NONE;

  /* client handle is NULL. The result should be failed */ 
  result = openapiScpRemoteTransferComplete(NULL);
  printf("NULL Client Handle. (result = %d)\n", result);
  if (OPEN_E_NONE == result) 
  {
    sanityResult = OPEN_E_FAIL;
  }

  if (OPEN_E_NONE == sanityResult)
  {
    printf("openapiScpRemoteTransferComplete() sanity successful.\n");
  }
  else 
  {
    printf("openapiScpRemoteTransferComplete() sanity failed.\n");
  }

  /* openapiSSHRemotePasswordGet() */
  printf("\nTesting openapiSSHRemotePasswordGet(): \n");

  result = openapiSSHRemotePasswordGet(NULL, clientSockID, &password);
  printf("NULL client handle:(result = %d)\n", result);
  
  result = openapiSSHRemotePasswordGet(clientHandle, clientSockID, NULL);
  printf("NULL argumet 3:(result = %d)\n", result);

  return;  
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  uint32_t arg1;
  open_buffdesc fileParams;
  open_buffdesc password;
  char fileNameStr[OPENAPI_FILE_NAME_LENGTH_MAX];

  if (argc < 2)
  {
    printSSHAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("ssh_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting SSH API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 3)
      {
        printSSHAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      sshKeyGenerate(&clientHandle, arg1);
      break;
    case 2:
      if (argc != 3)
      {
        printSSHAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      sshKeyStatusGet(&clientHandle, arg1);
      break;
    case 3:
      if (argc != 3)
      {
        printSSHAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      sshKeyDelete(&clientHandle, arg1);
      break;
    case 4:
      if (argc != 3)
      {
        printSSHAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      sshAdminModeSet(&clientHandle, arg1);
      break;
    case 5:
      if (argc != 2)
      {
        printSSHAppMenu();
        exit(1);
      }
      sshAdminModeGet(&clientHandle);
      break;
    case 6:
      if (argc != 3)
      {
        printSSHAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      sshProtocolVersionSet(&clientHandle, arg1);
      break;
    case 7:
      if (argc != 2)
      {
        printSSHAppMenu();
        exit(1);
      }
      sshProtocolVersionGet(&clientHandle);
      break;
    case 8:
      if (argc != 3)
      {
        printSSHAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      sshSessionTimeoutSet(&clientHandle, arg1);
      break;
    case 9:
      if (argc != 2)
      {
        printSSHAppMenu();
        exit(1);
      }
      sshSessionTimeoutGet(&clientHandle);
      break;
    case 10:
      if (argc != 3)
      {
        printSSHAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      sshMaxSessionsSet(&clientHandle, arg1);
      break;
    case 11:
      if (argc != 2)
      {
        printSSHAppMenu();
        exit(1);
      }
      sshMaxSessionsDefaultSet(&clientHandle);
      break;
    case 12:
      if (argc != 2)
      {
        printSSHAppMenu();
        exit(1);
      }
      sshMaxSessionsGet(&clientHandle);
      break;
    case 13:
      if (argc != 2)
      {
        printSSHAppMenu();
        exit(1);
      }
      sshActiveSessionsGet(&clientHandle);
      break;
    case 14:
      if (argc != 2)
      {
        printSSHAppMenu();
        exit(1);
      }
      sshOpENAPISanity(&clientHandle);
      break;
    case 15:
      if (argc != 3)
      {
        printSSHAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      scpServerAdminModeSet(&clientHandle, arg1);
      break;
    case 16:
      if (argc != 2)
      {
        printSSHAppMenu();
        exit(1);
      }
      scpServerAdminModeGet(&clientHandle);
      break;
    case 17:
      if (argc != 3)
      {
        printSSHAppMenu();
        exit(1);
      }
      if (strlen(argv[2]) != 0)
      { 
        strncpy(fileNameStr, argv[2], strlen(argv[2]));
        fileParams.size = sizeof(fileNameStr);
        fileParams.pstart = &fileNameStr;
        scpServerRemoteTransferStart(&clientHandle, &fileParams);
      } 
      break;
    case 18:
      if (argc != 2)
      {
        printSSHAppMenu();
        exit(1);
      }
      ScpServerRemoteTransferComplete(&clientHandle);
      break;

    case 19:
      if (argc != 3)
      {
        printSSHAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      sshRemotePasswordGet(&clientHandle, arg1, &password);
      break;

    default:
      printSSHAppMenu();
      break;
  }

  
  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping SSH API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
