/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  stacking_example.c
*
* @purpose   Stack Configuration OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    02/03/2016
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_stacking.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the Stacking Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printStackingAppMenu()
{
  printf("Usage:  stacking_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: Get the next member of the current stack: stacking_example 1 <start-unit> \n");
  printf("Test 2: Get the manager's unit number: stacking_example 2 \n");
  printf("Test 3: Get the unit's status: stacking_example 3 <unit> \n");
  printf("Test 4: Get the unit's management status: stacking_example 4 <unit> \n");
  printf("Test 5: Get the configured and standby unit number and admin mode in the stack: stacking_example 5 \n");
  printf("Test 6: Get the unit's model identifier string and pre-configured model identifier string assigned by HPC: stacking_example 6 <unit> \n");
  printf("Test 7: Get the unit's version of code in flash: stacking_example 7 <unit> \n");
  printf("Test 8: Get the unit's running version of code: stacking_example 8 <unit> \n");
  printf("Test 9: Get the unit's hardware and admin management preference: stacking_example 9 <unit> \n");
  printf("Test 10: Get the unit's type and index: stacking_example 10 <unit> \n");
  printf("Test 11: Get unit description that corresponds to the index provided: stacking_example 11 <unit-type-index> \n");
  printf("Test 12: Get stack firmware synchronization status of member unit: stacking_example 12 <unit> \n");
  printf("Test 13: Get stack firmware synchronization status: stacking_example 13 \n");
  printf("Test 14: Get stack firmware synchronization last attempt status: stacking_example 14 <unit> \n");
  printf("Test 15: Get active template id of the stack unit: stacking_example 15 <unit> \n");
  printf("Test 16: Get description of specified template id: stacking_example 16 <template-id> \n");
  printf("Test 17: Get the unit's up time in seconds: stacking_example 17 <unit> \n");
  printf("Test 18: Get the unit's serial number: stacking_example 18 <unit> \n");
  printf("Test 19: Get the unit's service tag: stacking_example 19 <unit> \n");
  printf("Test 20: Get the unit's model identifier string assigned by HPC: stacking_example 20 <unit> \n");
  printf("Test 21: Verify unit number: stacking_example 21 <unit> \n");
  printf("Test 22: Get the maximum stacking unit number: stacking_example 22 \n");
  printf("Test 23: Get unit type identifier that corresponds to the index provided: stacking_example 23 <unit-type-index> \n");
  printf("Test 24: Get unit type identifier that corresponds to the next index: stacking_example 24 <unit-type-index> \n");
  printf("Test 25: Create a new unit record for unit: stacking_example 25 <unit> <unit-type-index> <synch-mode> \n");
  printf("Test 26: Remove a unit only if it is not an active member of the stack: stacking_example 26 <unit> \n");
  printf("Test 27: Transfers the management function from the current unit to another: stacking_example 27 <target-unit> \n");
  printf("Test 28: Get QOS configuration for all front-panel stacking ports stack port: stacking_example 28 \n");
  printf("Test 29: Set QOS configuration for all front-panel stacking ports stack port: stacking_example 29 <fps-qos-mode> \n");
  printf("Test 30: Set the unit as standby in the stack: stacking_example 30 <standby-unit> <standby-admin-mode> \n");
  printf("Test 31: Get stacking mode configuration for a front-panel stack port: stacking_example 31 <unit> <slot> <port> \n");
  printf("Test 32: Set stacking mode configuration for a front-panel stack port: stacking_example 32 <unit> <slot> <port> <stack-port-mode> \n");
  printf("Test 33: Verify slot number: stacking_example 33 <unit> <slot> \n");
  printf("Test 34: List available slot(s) on given unit: stacking_example 34 <unit> \n");
  printf("Test 35: Verify the slot is full: stacking_example 35 <unit> <slot> \n");
  printf("Test 36: Get the admin mode of slot: stacking_example 36 <unit> <slot> \n");
  printf("Test 37: Set the admin mode of slot: stacking_example 37 <unit> <slot> <admin mode> \n");
  printf("Test 38: Get the power of slot: stacking_example 38 <unit> <slot> \n");
  printf("Test 39: Set the power mode of slot: stacking_example 39 <unit> <slot> <power mode> \n");
  printf("Test 40: Get the card type: stacking_example 40 <unit> <slot> \n");
  printf("Test 41: Get model id and description of configured card: stacking_example 41 <unit> <slot> \n");
  printf("Test 42: Get model id and description of inserted card: stacking_example 42 <unit> <slot> \n");
  printf("Test 43: Verify the slot is pluggable: stacking_example 43 <unit> <slot> \n");
  printf("Test 44: Verify the slot is power down: stacking_example 44 <unit> <slot> \n");
  printf("Test 45: Transfer arbitrary file from mgmr to member unit: stacking_example 45 <unit/-1> <src_file> <dst_file> \n");

  return;
}

/*********************************************************************
* @purpose  Given a unit, gets the next member of the current stack
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrStackMemberGetNext(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  uint32_t nextUnit = 0;

  if ((result = openapiUnitMgrStackMemberGetNext(clientHandle, unit, &nextUnit)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get next member of the current stack. (result = %d)\n", result);
  }
  else
  {
    printf("Next member of the current stack: %d\n", nextUnit);
  }
  
  return;
}

/*********************************************************************
* @purpose  Get the manager's unit number 
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrNumberGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t unit = 0;

  if ((result = openapiUnitMgrNumberGet(clientHandle, &unit)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get manager's unit number. (result = %d)\n", result);
  }
  else
  {
    printf("Manager's unit number: %d\n", unit);
  }
  
  return;
}

/*********************************************************************
* @purpose  Given a unit, gets the unit's status 
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrUnitStatusGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  OPEN_USMDB_UNITMGR_STATUS_t status;

  if ((result = openapiUnitMgrUnitStatusGet(clientHandle, unit, &status)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the unit's status. (result = %d)\n", result);
  }
  else
  {
    switch (status)
    {
      case OPEN_USMDB_UNITMGR_UNIT_OK:
        printf("Status for unit (%d): Ok\n", unit);
        break;
      case OPEN_USMDB_UNITMGR_UNIT_UNSUPPORTED:
        printf("Status for unit (%d): Unsupported\n", unit);
        break;
      case OPEN_USMDB_UNITMGR_CODE_MISMATCH:
        printf("Status for unit (%d): Code Mismatch\n", unit);
        break;
      case OPEN_USMDB_UNITMGR_CFG_MISMATCH:
        printf("Status for unit (%d): Configuration Mismatch\n", unit);
        break;
      case OPEN_USMDB_UNITMGR_SDM_MISMATCH:
        printf("Status for unit (%d): SDM Mismatch\n", unit);
        break;
      case OPEN_USMDB_UNITMGR_UNIT_NOT_PRESENT:
        printf("Status for unit (%d): Unit Not Present\n", unit);
        break;
      case OPEN_USMDB_UNITMGR_CODE_UPDATE:
        printf("Status for unit (%d): Code Update\n", unit);
        break;
      case OPEN_USMDB_UNITMGR_STM_MISMATCH:
        printf("Status for unit (%d): STM Mismatch\n", unit);
        break;
      default:
        break;
    }
  }
  
  return;
}

/*********************************************************************
* @purpose  Given a unit, gets the unit's management status 
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrMgmtUnitStatusGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  OPEN_USMDB_UNITMGR_MGMT_FLAG_t mgmtStatus;

  if ((result = openapiUnitMgrMgmtUnitStatusGet(clientHandle, unit, &mgmtStatus)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the unit's management status. (result = %d)\n", result);
  }
  else
  {
    switch (mgmtStatus)
    {
      case OPEN_USMDB_UNITMGR_MGMT_ENABLED:
        printf("Management status for unit (%d): Enable\n", unit);
        break;
      case OPEN_USMDB_UNITMGR_MGMT_DISABLED:
        printf("Management status for unit (%d): Disable\n", unit);
        break;
      case OPEN_USMDB_UNITMGR_MGMT_UNASSIGNED:
        printf("Management status for unit (%d): Unassigned\n", unit);
        break;
      default:
        break;
    }
  }
  
  return;
}

/*********************************************************************
* @purpose  Gets the configured and standby unit number and admin mode in the stack 
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrStandbyGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t configUnit = 0;
  uint32_t standbyUnit = 0;
  OPEN_BOOL_t standbyAdminMode = OPEN_FALSE;
  
  if ((result = openapiUnitMgrStandbyGet(clientHandle, &configUnit, &standbyUnit, &standbyAdminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the unit numbers and admin mode for standby unit. (result = %d)\n", result);
  }
  else
  {
    printf("Configured Unit : %d\n", configUnit);
    printf("Standby Unit : %d\n", standbyUnit);
    if (standbyAdminMode == OPEN_TRUE)
    {
      printf("Admin mode of Standby Unit (%d) is Enable\n", standbyUnit);
    }
    else
    {
      printf("Admin mode of Standby Unit (%d) is Disable\n", standbyUnit);
    }
  }

  return;
}

/*********************************************************************
* @purpose  Given a unit, gets the unit's model identifier string
*           assigned by HPC and pre-configured model identifier string 
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrModelIdentifierGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  open_buffdesc modelId, preconfigModelId;
  char bufModelId[512] = {0};
  char bufPreconfigModelId[512] = {0};

  modelId.pstart = bufModelId;
  modelId.size = sizeof(bufModelId);
  preconfigModelId.pstart = bufPreconfigModelId;
  preconfigModelId.size = sizeof(bufPreconfigModelId);
  
  if ((result = openapiUnitMgrModelIdentifierGet(clientHandle, unit, &modelId, &preconfigModelId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the unit numbers and admin mode for standby unit. (result = %d)\n", result);
  }
  else
  {
    printf("Model identifier : %s\nPreconfigured model identifier : %s\n", (char *)modelId.pstart, (char *)preconfigModelId.pstart);
  }

  return;
}

/*********************************************************************
* @purpose  Given a unit, gets the unit's version of code in flash 
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrDetectCodeVerFlashGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  open_revision_data_t codeVersion;

  if ((result = openapiUnitMgrDetectCodeVerFlashGet(clientHandle, unit, &codeVersion)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the unit's version code in flash. (result = %d)\n", result);
  }
  else
  {
    printf("Release : %d\nVersion : %d\nMaintenance Level : %d\nBuild Parts : %d\n", codeVersion.release, codeVersion.version, codeVersion.maint_level, codeVersion.build_num);
  }
  
  return;
}

/*********************************************************************
* @purpose  Given a unit, gets the unit's running version of code 
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrDetectCodeVerRunningGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  open_revision_data_t codeVersion;

  if ((result = openapiUnitMgrDetectCodeVerRunningGet(clientHandle, unit, &codeVersion)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the unit's running version of code. (result = %d)\n", result);
  }
  else
  {
    printf("Release : %d\nVersion : %d\nMaintenance Level : %d\nBuild Parts : %d\n", codeVersion.release, codeVersion.version, codeVersion.maint_level, codeVersion.build_num);
  }
  
  return;
}

/*********************************************************************
* @purpose  Print management preference output 
*
* @param    type            @b{(input)} Type
* @param    unit            @b{(input)} Unit number
* @param    pref            @b{(input)} Preference
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void printMgmtPref(OPEN_BOOL_t type, uint32_t unit, OPEN_USMDB_UNITMGR_MGMTPREF_t pref)
{
  if (type == OPEN_TRUE)
  {
    printf("Hardware ");
  }
  else
  {
    printf("Admin ");
  }

  switch (pref)
  {
    case OPEN_USMDB_UNITMGR_MGMTPREF_DISABLED:
      printf("management preference for unit (%d): Disable\n", unit);
      break;
    case OPEN_USMDB_UNITMGR_MGMTFUNC_UNASSIGNED:
      printf("management preference for unit (%d): Unassigned\n", unit);
      break;
    default:
      printf("management preference for unit (%d): %d\n", unit, pref);
      break;
  }
}

/*********************************************************************
* @purpose  Given a unit, gets the unit's hardware and admin management preference 
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrMgmtPrefGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  OPEN_USMDB_UNITMGR_MGMTPREF_t hwPref;
  OPEN_USMDB_UNITMGR_MGMTPREF_t adminPref;
  
  if ((result = openapiUnitMgrMgmtPrefGet(clientHandle, unit, &hwPref, &adminPref)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the unit's hardware and admin management preference. (result = %d)\n", result);
  }
  else
  {
    printMgmtPref(OPEN_TRUE, unit, hwPref);
    printMgmtPref(OPEN_FALSE, unit, adminPref);
  }
  
  return;
}

/*********************************************************************
* @purpose  Given a unit, gets the unit's type and index 
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrUnitTypeGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  uint32_t unitType = 0;
  uint32_t unitTypeIndex = 0;
  
  if ((result = openapiUnitMgrUnitTypeGet(clientHandle, unit, &unitType, &unitTypeIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the unit's type and index. (result = %d)\n", result);
  }
  else
  {
    printf("Unit type : 0x%x\nUnit type index : %d\n", unitType, unitTypeIndex);
  }
  
  return;
}

/*********************************************************************
* @purpose  Get unit description that corresponds to the index provided 
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unitTypeIndex   @b{(input)} Unit type index
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrSupportedDescriptionGet(openapiClientHandle_t *clientHandle, uint32_t unitTypeIndex)
{
  open_error_t result;
  open_buffdesc unitDescr;
  char bufUnitDescr[512] = {0};

  unitDescr.pstart = bufUnitDescr;
  unitDescr.size = sizeof(bufUnitDescr);
  
  if ((result = openapiUnitMgrSupportedDescriptionGet(clientHandle, unitTypeIndex, &unitDescr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the unit description that corresponds to the index provided. (result = %d)\n", result);
  }
  else
  {
    printf("Description for unit index (%d): %s\n", unitTypeIndex, (char *)unitDescr.pstart);
  }

  return;
}

/*********************************************************************
* @purpose  Get stack firmware synchronization status of member unit 
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrStackFirmwareSyncMemberStatusGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  OPEN_UNIT_MGR_SFS_SWITCH_STATUS_t firmwareSyncStatus;
  
  if ((result = openapiUnitMgrStackFirmwareSyncMemberStatusGet(clientHandle, unit, &firmwareSyncStatus)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the stack firmware synchronization status of member unit. (result = %d)\n", result);
  }
  else
  {
    switch (firmwareSyncStatus)
    {
      case OPEN_SFS_SWITCH_STATUS_IN_PROGRESS:
        printf("Stack firmware synchronization status of member unit(%d) : In Progress\n", unit);
        break;
      case OPEN_SFS_SWITCH_STATUS_NO_ACTION:
      default:
        printf("Stack firmware synchronization status of member unit(%d) : No Action\n", unit);
        break;
    }
  }
  
  return;
}

/*********************************************************************
* @purpose  Get stack firmware synchronization status 
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrStackFirmwareSyncStatusGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_UNIT_MGR_SFS_SWITCH_STATUS_t firmwareSyncStatus;
  
  if ((result = openapiUnitMgrStackFirmwareSyncStatusGet(clientHandle, &firmwareSyncStatus)) != OPEN_E_NONE)
  {
    if (result == OPEN_E_UNAVAIL)
    {
      printf("Feature is not supported.\n");
    }
    else
    {
      printf("Bad return code trying to get the stack firmware synchronization status. (result = %d)\n", result);
    }
  }
  else
  {
    switch (firmwareSyncStatus)
    {
      case OPEN_SFS_SWITCH_STATUS_IN_PROGRESS:
        printf("Stack firmware synchronization status : In Progress\n");
        break;
      case OPEN_SFS_SWITCH_STATUS_NO_ACTION:
      default:
        printf("Stack firmware synchronization status : No Action\n");
        break;
    }
  }
  
  return;
}

/*********************************************************************
* @purpose  Get stack firmware synchronization last attempt status 
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrStackFirmwareSyncMemberLastAttemptStatusGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  uint32_t lastAttemptStatus = 0;
  
  if ((result = openapiUnitMgrStackFirmwareSyncMemberLastAttemptStatusGet(clientHandle, unit, &lastAttemptStatus)) != OPEN_E_NONE)
  {
    if (result == OPEN_E_UNAVAIL)
    {
      printf("Feature is not supported.\n");
    }
    else
    {
      printf("Bad return code trying to get the stack firmware synchronization last attempt status of member unit. (result = %d)\n", result);
    }
  }
  else
  {
    switch (lastAttemptStatus)
    {
      case OPEN_SFS_SUCCESS:
        printf("Stack firmware synchronization last attempt status of member unit(%d) : Success\n", unit);
        break;
      case OPEN_SFS_FAIL:
        printf("Stack firmware synchronization last attempt status of member unit(%d) : Failure\n", unit);
        break;
      case OPEN_SFS_MIN_BOOTCODE_VERSION_NOT_PRESENT:
        printf("Stack firmware synchronization last attempt status of member unit(%d) : Min boot code version not present\n", unit);
        break;
      case OPEN_SFS_NONE:
      default:
        printf("Stack firmware synchronization last attempt status of member unit(%d) : None\n", unit);
        break;
    }
  }
  
  return;
}

/*********************************************************************
* @purpose  Gets active template id of the stack unit 
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackStmUnitActiveTemplateIdGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  uint32_t templateId = 0;
  
  if ((result = openapiStmUnitActiveTemplateIdGet(clientHandle, unit, &templateId)) != OPEN_E_NONE)
  {
    if (result == OPEN_E_UNAVAIL)
    {
      printf("Feature is not supported.\n");
    }
    else
    {
      printf("Bad return code trying to get the active template id of the stack unit. (result = %d)\n", result);
    }
  }
  else
  {
    printf("Active template id of the stack unit(%d) : %d\n", unit, templateId);
  }
  
  return;
}

/*********************************************************************
* @purpose  Gets description of specified template id 
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    templateId      @b{(input)} Template id
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackStmTemplateDescriptionGet(openapiClientHandle_t *clientHandle, uint32_t templateId)
{
  open_error_t result;
  open_buffdesc templateDescr;
  char bufTemplateDescr[512] = {0};

  templateDescr.pstart = bufTemplateDescr;
  templateDescr.size = sizeof(bufTemplateDescr);
  
  if ((result = openapiStmTemplateDescriptionGet(clientHandle, templateId, &templateDescr)) != OPEN_E_NONE)
  {
    if (result == OPEN_E_UNAVAIL)
    {
      printf("Feature is not supported.\n");
    }
    else
    {
      printf("Bad return code trying to get the description of specified template id. (result = %d)\n", result);
    }
  }
  else
  {
    printf("Description of template id(%d): %s\n", templateId, (char *)templateDescr.pstart);
  }
  
  return;
}

/*********************************************************************
* @purpose  Given a unit, gets the unit's up time in seconds 
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrUnitUpTimeGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  uint32_t upTime = 0;
  
  if ((result = openapiUnitMgrUnitUpTimeGet(clientHandle, unit, &upTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the unit's up time in seconds. (result = %d)\n", result);
  }
  else
  {
    printf("Uptime in seconds for unit(%d) : %d\n", unit, upTime);
  }
  
  return;
}

/*********************************************************************
* @purpose  Given a unit, gets the unit's serial number 
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrSerialNumberGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  open_buffdesc serialNumber;
  char bufSerialNumber[512] = {0};

  serialNumber.pstart = bufSerialNumber;
  serialNumber.size = sizeof(bufSerialNumber);
  
  if ((result = openapiUnitMgrSerialNumberGet(clientHandle, unit, &serialNumber)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the unit's serial number. (result = %d)\n", result);
  }
  else
  {
    printf("Serial Number for unit (%d): %s\n", unit, (char *)serialNumber.pstart);
  }
  
  return;
}

/*********************************************************************
* @purpose  Given a unit, gets the unit's service tag
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrServiceTagGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  open_buffdesc serviceTag;
  char bufServiceTag[512] = {0};

  serviceTag.pstart = bufServiceTag;
  serviceTag.size = sizeof(bufServiceTag);
  
  if ((result = openapiUnitMgrServiceTagGet(clientHandle, unit, &serviceTag)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the unit's service tag. (result = %d)\n", result);
  }
  else
  {
    printf("Service Tag for unit (%d): %s\n", unit, (char *)serviceTag.pstart);
  }
  
  return;
}

/*********************************************************************
* @purpose  Given a unit, gets the unit's model identifier string
*           assigned by HPC(Hardware Platform Control)
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrPreconfigModelIdGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  open_buffdesc modelId;
  char bufModelId[512] = {0};

  modelId.pstart = bufModelId;
  modelId.size = sizeof(bufModelId);
  
  if ((result = openapiUnitMgrPreconfigModelIdGet(clientHandle, unit, &modelId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the unit's pre-configured model identifier. (result = %d)\n", result);
  }
  else
  {
    printf("Pre-configured model identifier for unit (%d): %s\n", unit, (char *)modelId.pstart);
  }
  
  return;
}

/*********************************************************************
* @purpose  Given a unit, verify that it's within the valid range
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrUnitIsValid(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  OPEN_BOOL_t isValid = OPEN_FALSE;
  
  if ((result = openapiUnitMgrUnitIsValid(clientHandle, unit, &isValid)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to verify valid unit. (result = %d)\n", result);
  }
  else
  {
    if (isValid == OPEN_TRUE)
    {
      printf("Unit (%d) is Valid\n", unit);
    }
    else
    {
      printf("Unit (%d) is Invalid\n", unit);
    }
  }
  
  return;
}

/*********************************************************************
* @purpose  Gets the maximum stacking unit number (largest possible
*           number of units in the stack).
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrUnitMaxGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t unit = 0;

  if ((result = openapiUnitMgrUnitMaxGet(clientHandle, &unit)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the maximum stacking unit number. (result = %d)\n", result);
  }
  else
  {
    printf("Maximum stacking unit number: %d\n", unit);
  }
  
  return;
}

/*********************************************************************
* @purpose  Get unit type identifier that corresponds to
*           the index provided
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unitTypeIndex   @b{(input)} Unit type index
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrSupportedUnitIdGet(openapiClientHandle_t *clientHandle, uint32_t unitTypeIndex)
{
  open_error_t result;
  uint32_t unitType = 0;

  if ((result = openapiUnitMgrSupportedUnitIdGet(clientHandle, unitTypeIndex, &unitType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get unit type identifier that corresponds to the index provided. (result = %d)\n", result);
  }
  else
  {
    printf("Unit type identifier for index (%d): 0x%x\n", unitTypeIndex, unitType);
  }
  
  return;
}

/*********************************************************************
* @purpose  Get unit type identifier that corresponds to
*           the next index
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unitTypeIndex   @b{(input)} Unit type index
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrSupportedUnitIdNextGet(openapiClientHandle_t *clientHandle, uint32_t unitTypeIndex)
{
  open_error_t result;
  uint32_t nextUnitTypeIndex = 0;
  uint32_t unitType = 0;

  if ((result = openapiUnitMgrSupportedUnitIdNextGet(clientHandle, unitTypeIndex, &nextUnitTypeIndex, &unitType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get unit type identifier that corresponds to the next index. (result = %d)\n", result);
  }
  else
  {
    printf("Unit type identifier for next index (%d): 0x%x\n", nextUnitTypeIndex, unitType);
  }
  
  return;
}

/*********************************************************************
* @purpose  Create a new unit record for unit 
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
* @param    unitTypeIndex   @b{(input)} Unit type index
* @param    synchMode       @b{(input)} Synchronous mode
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrUnitCreate(openapiClientHandle_t *clientHandle, uint32_t unit, 
                            uint32_t unitTypeIndex, OPEN_BOOL_t synchMode)
{
  open_error_t result;

  if ((result = openapiUnitMgrUnitCreate(clientHandle, unit, unitTypeIndex, synchMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to create a new unit record for unit. (result = %d)\n", result);
  }
  else
  {
    printf("New unit record created successfully for unit(%d).\n", unit);
  }
  
  return;
}

/*********************************************************************
* @purpose  Remove a unit only if it is not an active member of
*           the stack
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrUnitDelete(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;

  if ((result = openapiUnitMgrUnitDelete(clientHandle, unit)) != OPEN_E_NONE)
  {
    if (result == OPEN_E_ERROR)
    {
      printf("Failed to remove. Unit (%d) is an active member of the stack.\n", unit);
    }
    else
    {
      printf("Bad return code trying to remove a unit. (result = %d)\n", result);
    }
  }
  else
  {
    printf("Unit (%d) removed successfully.\n", unit);
  }
  
  return;
}

/*********************************************************************
* @purpose  Transfers the management function from the current unit
*           to another
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    targetUnit      @b{(input)} Target unit
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrUnitTransfer(openapiClientHandle_t *clientHandle, uint32_t targetUnit)
{
  open_error_t result;

  if ((result = openapiUnitMgrUnitTransfer(clientHandle, targetUnit)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to transfers the management function from the current unit to another. (result = %d)\n", result);
  }
  else
  {
    printf("Unit (%d) transfer successful.\n", targetUnit);
  }
  
  return;
}

/*********************************************************************
* @purpose  Get QOS configuration for all front-panel stacking ports
*           stack port
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackSpmFpsConfigQosModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t fpsQosMode = 0;

  if ((result = openapiSpmFpsConfigQosModeGet(clientHandle, &fpsQosMode)) != OPEN_E_NONE)
  {
    if (result == OPEN_E_UNAVAIL)
    {
      printf("Feature is not supported.\n");
    }
    else
    {
      printf("Bad return code trying to get QOS configuration for all front-panel stacking ports stack port. (result = %d)\n", result);
    }
  }
  else
  {
    if (fpsQosMode == OPEN_ENABLE)
    {
      printf("QOS Mode for all front-panel stacking ports stack port: Enable.\n");
    }
    else
    {
      printf("QOS Mode for all front-panel stacking ports stack port: Disable.\n");
    }
  }
  
  return;
}

/*********************************************************************
* @purpose  Set QOS configuration for all front-panel stacking ports
*           stack port
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    fpsQosMode      @b{(input)} Front panel stacking QOS mode
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackSpmFpsConfigQosModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t fpsQosMode)
{
  open_error_t result;

  if ((result = openapiSpmFpsConfigQosModeSet(clientHandle, fpsQosMode)) != OPEN_E_NONE)
  {
    if (result == OPEN_E_UNAVAIL)
    {
      printf("Feature is not supported.\n");
    }
    else
    {
      printf("Bad return code trying to Set QOS configuration for all front-panel stacking ports stack port. (result = %d)\n", result);
    }
  }
  else
  {
    printf("QOS Mode set successfully.\n");
  }
  
  return;
}

/*********************************************************************
* @purpose  Sets the unit as standby in the stack
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    standbyUnit      @b{(input)}  Standby unit
* @param    standbyAdminMode @b{(input)}  Standby admin mode
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitMgrStandbySet(openapiClientHandle_t *clientHandle, uint32_t standbyUnit, 
                            OPEN_BOOL_t standbyAdminMode)
{
  open_error_t result;

  if ((result = openapiUnitMgrStandbySet(clientHandle, standbyUnit, standbyAdminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the unit as standby in the stack. (result = %d)\n", result);
  }
  else
  {
    printf("Unit (%d) set as standby in the stack successfully.\n", standbyUnit);
  }
  
  return;
}

/*********************************************************************
* @purpose  Get stacking mode configuration for a front-panel
*           stack port
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
* @param    slot            @b{(input)} Slot
* @param    port            @b{(input)} Port
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackSpmFpsConfigStackingModeGet(openapiClientHandle_t *clientHandle, uint32_t unit, 
                                      uint32_t slot, uint32_t port)
{
  open_error_t result;
  OPEN_BOOL_t stackPortMode = OPEN_FALSE;

  if ((result = openapiSpmFpsConfigStackingModeGet(clientHandle, unit, slot, port, &stackPortMode)) != OPEN_E_NONE)
  {
    if (result == OPEN_E_UNAVAIL)
    {
      printf("Feature is not supported.\n");
    }
    else
    {
      printf("Bad return code trying to get stacking mode configuration for a front-panel stack port. (result = %d)\n", result);
    }
  }
  else
  {
    if (stackPortMode == OPEN_TRUE)
    {
      printf("Stacking mode configuration for a front-panel stack port is Enable.\n");
    }
    else
    {
      printf("Stacking mode configuration for a front-panel stack port is Disable.\n");
    }
  }
  
  return;
}

/*********************************************************************
* @purpose  Set stacking mode configuration for a front-panel
*           stack port
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
* @param    slot            @b{(input)} Slot
* @param    port            @b{(input)} Port
* @param    stackPortMode   @b{(input)} Stack port mode
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackSpmFpsConfigStackingModeSet(openapiClientHandle_t *clientHandle, uint32_t unit, 
                                      uint32_t slot, uint32_t port, OPEN_BOOL_t stackPortMode)
{
  open_error_t result;

  if ((result = openapiSpmFpsConfigStackingModeSet(clientHandle, unit, slot, port, stackPortMode)) != OPEN_E_NONE)
  {
    if (result == OPEN_E_UNAVAIL)
    {
      printf("Feature is not supported.\n");
    }
    else
    {
      printf("Bad return code trying to set stacking mode configuration for a front-panel stack port. (result = %d)\n", result);
    }
  }
  else
  {
    printf("Stacking mode configuration for a front-panel stack port set successfully.\n");
  }
  
  return;
}

/*********************************************************************
* @purpose  Given a unit and slot, verify that slot is available on unit
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
* @param    slot            @b{(input)} Slot number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackSlotGet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t slot)
{
  open_error_t result;
  OPEN_BOOL_t isValid = OPEN_FALSE;
  
  if ((result = openapiSlotGet(clientHandle, unit, slot, &isValid)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to verify slot is available. (result = %d)\n", result);
  }
  else
  {
    if (isValid == OPEN_TRUE)
    {
      printf("Slot (%d) is available on Unit (%d)\n", slot, unit);
    }
    else
    {
      printf("Slot (%d) is not available on Unit (%d)\n", slot, unit);
    }
  }
  
  return;
}


/*********************************************************************
* @purpose  Given a unit, gets the available slots on the unit.
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackUnitDbEntrySlotsGet(openapiClientHandle_t *clientHandle, uint32_t unit)
{
  open_error_t result;
  open_buffdesc slot;
  uint32_t size, i;
  uint32_t bufSlot[3] = {0};

  slot.pstart = bufSlot;
  slot.size = sizeof(bufSlot);
  
  if ((result = openapiUnitDbEntrySlotsGet(clientHandle, unit, &size, &slot)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the unit's slots. (result = %d)\n", result);
  }
  else
  {
    for(i = 0; i < size; i++)
    {
      printf("Slot (%d) is available on Unit (%d)\n", bufSlot[i], unit);
    }
  }
  
  return;
}

/*********************************************************************
* @purpose  Given a unit and slot, verify that slot is full 
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
* @param    slot            @b{(input)} Slot number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackSlotIsFullGet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t slot)
{
  open_error_t result;
  OPEN_BOOL_t isValid = OPEN_FALSE;

  if ((result = openapiSlotIsFullGet(clientHandle, unit, slot, &isValid)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to verify slot is full. (result = %d)\n", result);
  }
  else
  {
    if (isValid == OPEN_TRUE)
    {
      printf("Slot (%d) is full\n", slot);
    }
    else
    {
      printf("Slot (%d) is not full\n", slot);
    }
  }

  return;
}

/*********************************************************************
* @purpose  Given a unit and slot number, gets the slot's admin mode
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
* @param    slot            @b{(input)} Slot number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackSlotAdminModeGet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t slot)
{
  open_error_t result;
  uint32_t mode = 0;

  if ((result = openapiSlotAdminModeGet(clientHandle, unit, slot, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the slot's admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("Admin Mode for Slot(%d) : %s\n", slot, (mode == OPEN_ENABLE ? "Enabled" : "Disabled"));
  }

  return;
}

/*********************************************************************
* @purpose  Given a unit, slot number and admin mode, set the slot's admin mode
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
* @param    slot            @b{(input)} Slot number
* @param    mode            @b{(input)} Admin Mode 
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackSlotAdminModeSet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t slot, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiSlotAdminModeSet(clientHandle, unit, slot, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the slot's admin mode. (result = %d)\n", result);
  }
  else
  {
    printf("Admin Mode for Slot(%d) : %s is set\n", slot, (mode == OPEN_ENABLE ? "Enabled" : "Disabled"));
  }

  return;
}

/*********************************************************************
* @purpose  Given a unit and slot number, gets the slot's power mode
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
* @param    slot            @b{(input)} Slot number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackSlotPowerModeGet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t slot)
{
  open_error_t result;
  uint32_t mode = 0;

  if ((result = openapiSlotPowerModeGet(clientHandle, unit, slot, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the slot's power mode. (result = %d)\n", result);
  }
  else
  {
    printf("Power Mode for Slot(%d) : %s\n", slot, (mode == OPEN_ENABLE ? "Enabled" : "Disabled"));
  }

  return;
}

/*********************************************************************
* @purpose  Given a unit, slot number and power mode, set the slot's admin mode
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
* @param    slot            @b{(input)} Slot number
* @param    mode            @b{(input)} Power Mode 
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackSlotPowerModeSet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t slot, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiSlotPowerModeSet(clientHandle, unit, slot, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the slot's power mode. (result = %d)\n", result);
  }
  else
  {
    printf("Power Mode for Slot(%d) : %s is set\n", slot, (mode == OPEN_ENABLE ? "Enabled" : "Disabled"));
  }

  return;
}

/*********************************************************************
* @purpose  Given a unit and slot number, gets the card's type
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
* @param    slot            @b{(input)} Slot number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackCardTypeGet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t slot)
{
  open_error_t result;
  uint32_t configured, inserted;

  if ((result = openapiCardTypeGet(clientHandle, unit, slot, &configured, &inserted)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the slot's card type. (result = %d)\n", result);
  }
  else
  {
    printf("Configured card id for slot (%d): 0x%x (%d)\n", slot, configured, configured);
    printf("Inserted card id for slot (%d): 0x%x (%d)\n", slot, inserted, inserted);
  }

  return;
}

/*********************************************************************
* @purpose  Given a card id, gets the card's model identifier string
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    cardid          @b{(input)} Card id
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackCardSupportedConfiguredCardInfoGet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t slot)
{
  open_error_t result;
  open_buffdesc modelId;
  open_buffdesc desc;
  char bufModelId[256] = {0};
  char bufDesc[256] = {0};
  uint32_t configured, inserted;

  if ((result = openapiCardTypeGet(clientHandle, unit, slot, &configured, &inserted)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the slot's card type. (result = %d)\n", result);
    return;
  }

  modelId.pstart = bufModelId;
  modelId.size = sizeof(bufModelId);
  desc.pstart = bufDesc;
  desc.size = sizeof(bufDesc);
  
  if ((result = openapiCardSupportedCardModelIdGet(clientHandle, configured, &modelId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the card's model identifier. (result = %d)\n", result);
  }
  else
  {
    printf("Model identifier for configured card (0x%x): %s\n", configured, bufModelId);
  }
  
  if ((result = openapiCardSupportedCardDescriptionGet(clientHandle, configured, &desc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the card's description. (result = %d)\n", result);
  }
  else
  {
    printf("Description for configured card (0x%x): %s\n", configured, bufDesc);
  }

  return;
}

/*********************************************************************
* @purpose  Given a card id, gets the card's description string
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    cardid          @b{(input)} Card id
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackCardSupportedInsertedCardInfoGet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t slot)
{
  open_error_t result;
  open_buffdesc modelId;
  open_buffdesc desc;
  char bufModelId[256] = {0};
  char bufDesc[256] = {0};
  uint32_t configured, inserted;

  if ((result = openapiCardTypeGet(clientHandle, unit, slot, &configured, &inserted)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the slot's card type. (result = %d)\n", result);
    return;
  }

  modelId.pstart = bufModelId;
  modelId.size = sizeof(bufModelId);
  desc.pstart = bufDesc;
  desc.size = sizeof(bufDesc);
  
  if ((result = openapiCardSupportedCardModelIdGet(clientHandle, inserted, &modelId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the card's model identifier. (result = %d)\n", result);
  }
  else
  {
    printf("Model identifier for inserted card (0x%x): %s\n", inserted, bufModelId);
  }
  
  if ((result = openapiCardSupportedCardDescriptionGet(clientHandle, inserted, &desc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the card's description. (result = %d)\n", result);
  }
  else
  {
    printf("Description for inserted card (0x%x): %s\n", inserted, bufDesc);
  }

  return;
}

/*********************************************************************
* @purpose  Given a unit and slot, verify that slot is pluggable 
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
* @param    slot            @b{(input)} Slot number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackSlotSupportedPluggableGet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t slot)
{
  open_error_t result;
  OPEN_BOOL_t isValid = OPEN_FALSE;

  if ((result = openapiSlotSupportedPluggableGet(clientHandle, unit, slot, &isValid)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to verify slot is pluggable. (result = %d)\n", result);
  }
  else
  {
    if (isValid == OPEN_TRUE)
    {
      printf("Slot (%d) is pluggable\n", slot);
    }
    else
    {
      printf("Slot (%d) is not pluggable\n", slot);
    }
  }

  return;
}

/*********************************************************************
* @purpose  Given a unit and slot, verify that slot is power down
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
* @param    slot            @b{(input)} Slot number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackSlotSupportedPowerdownGet(openapiClientHandle_t *clientHandle, uint32_t unit, uint32_t slot)
{
  open_error_t result;
  OPEN_BOOL_t isValid = OPEN_FALSE;

  if ((result = openapiSlotSupportedPowerdownGet(clientHandle, unit, slot, &isValid)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to verify slot is power down. (result = %d)\n", result);
  }
  else
  {
    if (isValid == OPEN_TRUE)
    {
      printf("Slot (%d) is power down\n", slot);
    }
    else
    {
      printf("Slot (%d) is not power down\n", slot);
    }
  }

  return;
}

/*********************************************************************
* @purpose  Given a unit and slot, verify that slot is power down
*
* @param    client_handle   @b{(input)} Client handle from
*                                       registration API
* @param    unit            @b{(input)} Unit number
* @param    slot            @b{(input)} Slot number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void stackCopyFile(openapiClientHandle_t *clientHandle, uint32_t unit,
                   char *srcFileName, char *dstFileName)
{
  open_error_t result;
  open_buffdesc buffDescSrcFileName;
  open_buffdesc buffDescDstFileName;

  buffDescSrcFileName.pstart = srcFileName;
  buffDescSrcFileName.size = strlen(srcFileName) + 1;

  buffDescDstFileName.pstart = dstFileName;
  buffDescDstFileName.size = strlen(dstFileName) + 1;

  result = openapiCdaTransferFile(clientHandle, unit, &buffDescSrcFileName, &buffDescDstFileName);

  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to verify slot is power down. (result = %d)\n", result);
  }
  else
  {
    printf("ok\n");
  }

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  uint32_t arg1, arg2, arg3, arg4;
  
  if (argc < 2)
  {
    printStackingAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("stacking_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Stacking API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 3) 
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackUnitMgrStackMemberGetNext(&clientHandle, arg1);
      break;
    case 2:
      if (argc != 2)
      {
        printStackingAppMenu();
        exit(1);
      }
      stackUnitMgrNumberGet(&clientHandle);
      break;
    case 3:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackUnitMgrUnitStatusGet(&clientHandle, arg1);
      break;
    case 4:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackUnitMgrMgmtUnitStatusGet(&clientHandle, arg1);
      break;       
    case 5:
      if (argc != 2)
      {
        printStackingAppMenu();
        exit(1);
      }
      stackUnitMgrStandbyGet(&clientHandle);
      break;       
    case 6:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackUnitMgrModelIdentifierGet(&clientHandle, arg1);
      break;       
    case 7:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackUnitMgrDetectCodeVerFlashGet(&clientHandle, arg1);
      break;       
    case 8:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackUnitMgrDetectCodeVerRunningGet(&clientHandle, arg1);
      break;       
    case 9:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackUnitMgrMgmtPrefGet(&clientHandle, arg1);
      break;       
    case 10:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackUnitMgrUnitTypeGet(&clientHandle, arg1);
      break;       
    case 11:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackUnitMgrSupportedDescriptionGet(&clientHandle, arg1);
      break;       
    case 12:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackUnitMgrStackFirmwareSyncMemberStatusGet(&clientHandle, arg1);
      break;       
    case 13:
      if (argc != 2)
      {
        printStackingAppMenu();
        exit(1);
      }
      stackUnitMgrStackFirmwareSyncStatusGet(&clientHandle);
      break;       
    case 14:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackUnitMgrStackFirmwareSyncMemberLastAttemptStatusGet(&clientHandle, arg1);
      break;       
    case 15:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackStmUnitActiveTemplateIdGet(&clientHandle, arg1);
      break;       
    case 16:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackStmTemplateDescriptionGet(&clientHandle, arg1);
      break;       
    case 17:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackUnitMgrUnitUpTimeGet(&clientHandle, arg1);
      break;       
    case 18:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackUnitMgrSerialNumberGet(&clientHandle, arg1);
      break;       
    case 19:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackUnitMgrServiceTagGet(&clientHandle, arg1);
      break;       
    case 20:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackUnitMgrPreconfigModelIdGet(&clientHandle, arg1);
      break;       
    case 21:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackUnitMgrUnitIsValid(&clientHandle, arg1);
      break;       
    case 22:
      if (argc != 2)
      {
        printStackingAppMenu();
        exit(1);
      }
      stackUnitMgrUnitMaxGet(&clientHandle);
      break;       
    case 23:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackUnitMgrSupportedUnitIdGet(&clientHandle, arg1);
      break;       
    case 24:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackUnitMgrSupportedUnitIdNextGet(&clientHandle, arg1);
      break;       
    case 25:
      if (argc != 5)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg3 = atoi(argv[4]);
      stackUnitMgrUnitCreate(&clientHandle, arg1, arg2, (OPEN_BOOL_t)arg3);
      break;       
    case 26:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackUnitMgrUnitDelete(&clientHandle, arg1);
      break;       
    case 27:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackUnitMgrUnitTransfer(&clientHandle, arg1);
      break;       
    case 28:
      if (argc != 2)
      {
        printStackingAppMenu();
        exit(1);
      }
      stackSpmFpsConfigQosModeGet(&clientHandle);
      break;       
    case 29:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackSpmFpsConfigQosModeSet(&clientHandle, (OPEN_CONTROL_t)arg1);
      break;       
    case 30:
      if (argc != 4)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      stackUnitMgrStandbySet(&clientHandle, arg1, arg2);
      break;       
    case 31:
      if (argc != 5)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg3 = atoi(argv[4]);
      stackSpmFpsConfigStackingModeGet(&clientHandle, arg1, arg2, arg3);
      break;       
    case 32:
      if (argc != 6)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg3 = atoi(argv[4]);
      arg4 = atoi(argv[5]);
      stackSpmFpsConfigStackingModeSet(&clientHandle, arg1, arg2, arg3, (OPEN_BOOL_t)arg4);
      break;       
    case 33:
      if (argc != 4)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      stackSlotGet(&clientHandle, arg1, arg2);
      break;       
    case 34:
      if (argc != 3)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackUnitDbEntrySlotsGet(&clientHandle, arg1);
      break;       
    case 35:
      if (argc != 4)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      stackSlotIsFullGet(&clientHandle, arg1, arg2);
      break;       
    case 36:
      if (argc != 4)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      stackSlotAdminModeGet(&clientHandle, arg1, arg2);
      break;       
    case 37:
      if (argc != 5)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg3 = atoi(argv[4]);
      stackSlotAdminModeSet(&clientHandle, arg1, arg2, arg3);
      break;       
    case 38:
      if (argc != 4)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      stackSlotPowerModeGet(&clientHandle, arg1, arg2);
      break;       
    case 39:
      if (argc != 5)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      arg3 = atoi(argv[4]);
      stackSlotPowerModeSet(&clientHandle, arg1, arg2, arg3);
      break;       
    case 40:
      if (argc != 4)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      stackCardTypeGet(&clientHandle, arg1, arg2);
      break;       
    case 41:
      if (argc != 4)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      stackCardSupportedConfiguredCardInfoGet(&clientHandle, arg1, arg2);
      break;       
    case 42:
      if (argc != 4)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      stackCardSupportedInsertedCardInfoGet(&clientHandle, arg1, arg2);
      break;       
    case 43:
      if (argc != 4)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      stackSlotSupportedPluggableGet(&clientHandle, arg1, arg2);
      break;       
    case 44:
      if (argc != 4)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      stackSlotSupportedPowerdownGet(&clientHandle, arg1, arg2);
      break;       
    case 45:
      if (argc != 5)
      {
        printStackingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      stackCopyFile(&clientHandle, arg1, argv[3], argv[4]);
      break;
    default:
      printStackingAppMenu();
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Stacking API example application");
        
  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
