/*********************************************************************
*
* Copyright 2016-2020 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  transfer_example.c
*
* @purpose   Image Transfer Configuration OpEN APIs Example 
*
* @component OpEN
*
* @comments
*
* @create    2/19/2016
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <stdio.h>
#include <sys/select.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_transfer.h"

#define OPEN_BUFF_EX_SIZE (256+1)
char *exampleName;

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/**************************************************************************
*
* @purpose      create ascii displayable string from transfer mode
*
* @param        transferMode    transfer mode enum value
*
* @returns      output string
*
* @end
*
*************************************************************************/
char *transferModePrintable(OPEN_TRANSFER_MODES_t transferMode)
{
   static char *tmp;

   switch (transferMode)
   {
     case OPEN_TRANSFER_TFTP:  tmp = "TFTP";      break;
     case OPEN_TRANSFER_SFTP:  tmp = "SFTP";      break;
     case OPEN_TRANSFER_SCP:   tmp = "SCP";       break;
     default:                  tmp = "<INVALID>"; break;
   }
   return tmp;
}

/**************************************************************************
*
* @purpose      create ascii displayable string from transfer file type
*
* @param        fileType    transfer file type
*
* @returns      output string
*
* @end
*
*************************************************************************/
char *transferFiletypePrintable(OPEN_FILE_TYPES_t fileType)
{
   static char *tmp;

   switch (fileType)
   {
     case OPEN_FILE_TYPE_CODE:    tmp = "CODE";      break;
     case OPEN_FILE_TYPE_CONFIG:  tmp = "CONFIG";    break;
     case OPEN_FILE_TYPE_LICENSE: tmp = "LICENSE";    break;
     default:                     tmp = "<INVALID>"; break;
   }
   return tmp;
}

/**************************************************************************
*
* @purpose      create ascii displayable string from verification option
*
* @param        dsvConfigOption   digital signature verification option
*
* @returns      output string
*
* @end
*
*************************************************************************/
char *verificationOptionPrintable(OPEN_DIGITAL_SIGNATURE_OPTION_t dsvConfigOption)
{
  static char *tmp;

  switch (dsvConfigOption)
  {
    case OPEN_DIGITAL_SIGNATURE_VERIFY_NONE:                tmp = "None";     break;
    case OPEN_DIGITAL_SIGNATURE_VERIFY_IMAGE_ONLY:          tmp = "Image";    break;
    case OPEN_DIGITAL_SIGNATURE_VERIFY_CONFIG_SCRIPT_ONLY:  tmp = "Config";   break;
    case OPEN_DIGITAL_SIGNATURE_VERIFY_ALL:                 tmp = "All";      break;
    default:                                                tmp = "<INVALID>";break;
   }
   return tmp;
}

/**************************************************************************
*
* @purpose      create ascii displayable string from transfer mode
*
* @param        transferMode    transfer mode enum value
*
* @returns      output string
*
* @end
*
*************************************************************************/
char *resultCodePrintable(OPEN_TRANSFER_CODE_t resultCode)
{
  static char *tmp;

  switch (resultCode)
  {
  case OPEN_TRANSFER_CODE_NONE:          tmp = "None";          break;
  case OPEN_TRANSFER_CODE_STARTING:      tmp = "Starting";      break;
  case OPEN_TRANSFER_CODE_WRITING_FLASH: tmp = "Write Flash";   break;
  case OPEN_TRANSFER_CODE_FAILURE:       tmp = "Failure";       break;
  case OPEN_TRANSFER_CODE_SUCCESS:       tmp = "Success";       break;
  default:                               tmp = "Other";         break;
  }
  return tmp;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     Dumps current configured transfer parameters
* 
* @end
*********************************************************************/
void transferOpENAPIConfigDump(openapiClientHandle_t *clientHandle) 
{
  open_error_t result;
  OPEN_TRANSFER_MODES_t xfrMode;
  OPEN_FILE_TYPES_t fileType;
  open_inet_addr_t inetAddr;
  OPEN_DIGITAL_SIGNATURE_OPTION_t dsvConfigOption;
  open_buffdesc buffDesc;
  open_buffdesc buffDesc2;
  char buffer[OPEN_BUFF_EX_SIZE];
  char buffer2[OPEN_BUFF_EX_SIZE];

  result = openapiTransferModeGet(clientHandle, &xfrMode);
  if (result != OPEN_E_NONE) return;
  printf(" CONFIG -- Transfer Mode:\t\t%s\n",transferModePrintable(xfrMode));

  result = openapiTransferFileTypeGet(clientHandle, &fileType);
  if (result != OPEN_E_NONE) return;
  printf(" CONFIG -- FileType:\t\t\t%s\n", transferFiletypePrintable(fileType));

  result = openapiTransferServerAddressGet(clientHandle, &inetAddr);
  if (result != OPEN_E_NONE) return;
  if (OPEN_E_NONE==result)
  {
    if (inetAddr.family == OPEN_AF_INET)
    {
      inet_ntop(AF_INET, (void*)&(inetAddr.addr.ipv4), buffer, OPEN_BUFF_EX_SIZE);
    }
    else if (inetAddr.family == OPEN_AF_INET6)
    {
      inet_ntop(AF_INET6, (void*)&(inetAddr.addr.ipv6), buffer, OPEN_BUFF_EX_SIZE);
    }
  }
  printf(" CONFIG -- Transfer Server Address:\t%s\n", buffer);

  buffDesc.pstart = buffer;
  buffDesc.size   = sizeof(buffer);
  buffDesc2.pstart = buffer2;
  buffDesc2.size   = sizeof(buffer2);
  result = openapiTransferFileRemoteGet(clientHandle, &buffDesc2, &buffDesc);
  if (result != OPEN_E_NONE) return;
  printf(" CONFIG -- Remote File Path:\t\t'%s'\n", (char *)buffDesc2.pstart);
  printf(" CONFIG -- Remote File Name:\t\t'%s'\n", (char *)buffDesc.pstart);

  buffDesc.size   = sizeof(buffer);
  result = openapiTransferFileLocalGet(clientHandle, &buffDesc);
  if (result != OPEN_E_NONE) return;
  printf(" CONFIG -- Local File Name (label):\t'%s'\n", (char *)buffDesc.pstart);


  buffDesc.size   = sizeof(buffer);
  buffDesc2.size   = sizeof(buffer2);
  result = openapiTransferRemoteUserCredentialsGet(clientHandle, &buffDesc, &buffDesc2);
  if (result != OPEN_E_NONE) return;
  printf(" CONFIG -- User Credentials Username:\t'%s'\n", (char *)buffDesc.pstart);
  printf(" CONFIG -- User Credentials Password:\t'%s'\n", (char *)buffDesc2.pstart);

  result = openapiDigitalSignatureVerifyConfigGet(clientHandle, &dsvConfigOption);
  if (result != OPEN_E_NONE) return;
  printf(" CONFIG -- Digital Signature Verify:\t%s\n", verificationOptionPrintable(dsvConfigOption));

  return;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiInetAddrFromIPAddrHostNameStr()
* 
* @end
*********************************************************************/
void transferOpENAPISanityConvertHost(openapiClientHandle_t *clientHandle) 
{
  printf("Testing openapiInetAddrFromIPAddrHostNameStr(): Sanity test N/A. Test explicitly with '%s 1 <string>'\n", exampleName);
  return;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiImageHeaderSizeGet()
* 
* @end
*********************************************************************/
void transferOpENAPISanityHeaderSize(openapiClientHandle_t *clientHandle) 
{
  open_error_t result;
  uint32_t value;

  printf("Testing openapiImageHeaderSizeGet():\n");
  result = openapiImageHeaderSizeGet(clientHandle, &value);
  printf(" %s -- Get Image Header Size: %d.\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED", value);

  return;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiImageHeaderCheck()
* 
* @end
*********************************************************************/
void transferOpENAPISanityHeaderCheck(openapiClientHandle_t *clientHandle)
{
  printf("Testing openapiImageHeaderCheck(): Sanity test N/A. Test explicitly with '%s 3 <filename>'\n", exampleName);
  return;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiTransferModeSet()
* 
* @end
*********************************************************************/
void transferOpENAPISanityModeSet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_TRANSFER_MODES_t xfrModeSet, xfrModeTest, xfrModeCurrent;

  printf("Testing openapiTransferModeSet(): ");

  /* test Code */
  result = openapiTransferModeGet(clientHandle, &xfrModeCurrent);
  printf("Current Transfer Mode: %s (%d).\n",
         transferModePrintable(xfrModeCurrent), xfrModeCurrent);
  if (result != OPEN_E_NONE) return;

  for (xfrModeSet = OPEN_TRANSFER_NONE+1; xfrModeSet < OPEN_TRANSFER_LAST; xfrModeSet++)
  {
    result = openapiTransferModeSet(clientHandle, xfrModeSet);
    printf(" %s -- Transfer Mode Set: %s (%d).\n",
           (OPEN_E_NONE==result)?"PASSED":"FAILED",
           transferModePrintable(xfrModeSet), xfrModeSet);
    if (result != OPEN_E_NONE) return;

    result = openapiTransferModeGet(clientHandle, &xfrModeTest);
    printf(" %s -- Transfer Mode Get: %s (%d).\n",
           ((OPEN_E_NONE==result)&&(xfrModeTest==xfrModeSet))?"PASSED":"FAILED",
           transferModePrintable(xfrModeTest), xfrModeTest);
    if (result != OPEN_E_NONE) return;
  }

  result = openapiTransferModeSet(clientHandle, xfrModeCurrent);
  if (result != OPEN_E_NONE) return;
  result = openapiTransferModeGet(clientHandle, &xfrModeSet);
  printf(" Restoring Transfer Mode: %s (%d).\n",
         transferModePrintable(xfrModeSet), xfrModeSet);

  return;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiTransferModeGet()
* 
* @end
*********************************************************************/
void transferOpENAPISanityModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_TRANSFER_MODES_t xfrMode;

  printf("Testing openapiTransferModeGet():\n");

  result = openapiTransferModeGet(clientHandle, &xfrMode);
  printf(" %s -- Transfer Mode Get: %s (%d).\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         transferModePrintable(xfrMode), xfrMode);
  return;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiTransferFileTypeSet()
* 
* @end
*********************************************************************/
void transferOpENAPISanityFileTypeSet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_FILE_TYPES_t fileTypeSet, fileTypeTest, fileTypeCurrent;

  printf("Testing openapiTransferFileTypeSet(): ");

  result = openapiTransferFileTypeGet(clientHandle, &fileTypeCurrent);
  printf("Current FileType: %s (%d).\n",
         transferFiletypePrintable(fileTypeCurrent), fileTypeCurrent);
  if (result != OPEN_E_NONE) return;

  /* File type config not yet settable: OPEN_FILE_TYPE_LAST */
  for (fileTypeSet = OPEN_FILE_TYPE_NONE+1; fileTypeSet < OPEN_FILE_TYPE_CONFIG; fileTypeSet++)
  {
    result = openapiTransferFileTypeSet(clientHandle, fileTypeSet);
    printf(" %s -- Transfer FileType Set: %s (%d).\n",
           (OPEN_E_NONE==result)?"PASSED":"FAILED",
           transferFiletypePrintable(fileTypeSet), fileTypeSet);
    if (result != OPEN_E_NONE) return;

    result = openapiTransferFileTypeGet(clientHandle, &fileTypeTest);
    printf(" %s -- Transfer FileType Get: %s (%d).\n",
           ((OPEN_E_NONE==result)&&(fileTypeTest==fileTypeSet))?"PASSED":"FAILED",
           transferFiletypePrintable(fileTypeTest), fileTypeTest);
    if (result != OPEN_E_NONE) return;
  }

  result = openapiTransferFileTypeSet(clientHandle, fileTypeCurrent);
  if (result != OPEN_E_NONE) return;
  result = openapiTransferFileTypeGet(clientHandle, &fileTypeCurrent);
  printf(" Restoring FileType: %s (%d).\n",
         transferFiletypePrintable(fileTypeCurrent), fileTypeCurrent);

  return;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiTransferFileTypeGet()
* 
* @end
*********************************************************************/
void transferOpENAPISanityFileTypeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_FILE_TYPES_t fileType;

  printf("Testing openapiTransferFileTypeGet():\n");

  result = openapiTransferFileTypeGet(clientHandle, &fileType);
  printf(" %s -- Transfer File Type Get: %s (%d). (result = %d)\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         transferFiletypePrintable(fileType), fileType, result);
  return;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiTransferFileTypeGet()
* 
* @end
*********************************************************************/
void transferOpENAPISanityServerAddressGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_inet_addr_t inetAddr;
  char buffer[OPEN_BUFF_EX_SIZE];

  printf("Testing openapiTransferServerAddressGet():\n");

  result = openapiTransferServerAddressGet(clientHandle, &inetAddr);
  snprintf(buffer, sizeof(buffer), "Fail");
  if (OPEN_E_NONE==result)
  {
    if (inetAddr.family == OPEN_AF_INET)
    {
      inet_ntop(AF_INET, (void*)&(inetAddr.addr.ipv4), buffer, OPEN_BUFF_EX_SIZE);
    }
    else if (inetAddr.family == OPEN_AF_INET6)
    {
      inet_ntop(AF_INET6, (void*)&(inetAddr.addr.ipv6), buffer, OPEN_BUFF_EX_SIZE);
    }
  }
  printf(" %s -- Transfer Server Address Get: %s.\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED", buffer);

  return;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiTransferFileTypeSet()
* 
* @end
*********************************************************************/
void transferOpENAPISanityServerAddressSet(openapiClientHandle_t *clientHandle)
{
  printf("Testing openapiTransferServerAddressSet(): Sanity test N/A. Test explicitly with '%s 9 <Address/Hostname>'\n", exampleName);
  return;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiTransferFileRemoteGet()
* 
* @end
*********************************************************************/
void transferOpENAPISanityFileRemoteGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char buffer[OPEN_BUFF_EX_SIZE];
  open_buffdesc buffDesc2;
  char buffer2[OPEN_BUFF_EX_SIZE];

  printf("Testing openapiTransferFileRemoteGet():\n");

  buffDesc.pstart = buffer;
  buffDesc.size   = sizeof(buffer);
  buffDesc2.pstart = buffer2;
  buffDesc2.size   = sizeof(buffer2);
  result = openapiTransferFileRemoteGet(clientHandle, &buffDesc2, &buffDesc);
  printf(" %s -- Transfer Remote File Name Get   path:'%s'  file:'%s'.\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         (char *)buffDesc2.pstart, (char *)buffDesc.pstart);

  return;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiTransferFileRemoteSet()
* 
* @end
*********************************************************************/
void transferOpENAPISanityFileRemoteSet(openapiClientHandle_t *clientHandle)
{
  printf("Testing openapiTransferFileRemoteSet(): Sanity test N/A. Test explicitly with '%s 11 <name> <path>'\n", exampleName);
  return;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiTransferFileLocalGet()
* 
* @end
*********************************************************************/
void transferOpENAPISanityFileLocalGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char buffer[OPEN_BUFF_EX_SIZE];

  printf("Testing openapiTransferFileLocalGet():\n");

  buffDesc.pstart = buffer;
  buffDesc.size   = sizeof(buffer);
  result = openapiTransferFileLocalGet(clientHandle, &buffDesc);
  printf(" %s -- Transfer Local File Name Get: %s.\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         (char *)buffDesc.pstart);

  return;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiTransferFileLocalSet()
* 
* @end
*********************************************************************/
void transferOpENAPISanityFileLocalSet(openapiClientHandle_t *clientHandle)
{
  printf("Testing openapiTransferFileLocalSet(): Sanity test N/A. Test explicitly with '%s 13  <label>'\n", exampleName);
  return;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiTransferRemoteUserCredentialsGet()
* 
* @end
*********************************************************************/
void transferOpENAPISanityRemoteUserCredentialsGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char buffer[OPEN_BUFF_EX_SIZE];
  open_buffdesc buffDesc2;
  char buffer2[OPEN_BUFF_EX_SIZE];

  printf("Testing openapiTransferRemoteUserCredentialsGet():\n");

  buffDesc.pstart = buffer;
  buffDesc.size   = sizeof(buffer);
  buffDesc2.pstart = buffer2;
  buffDesc2.size   = sizeof(buffer2);
  result = openapiTransferRemoteUserCredentialsGet(clientHandle, &buffDesc, &buffDesc2);
  printf(" %s -- Remote User Credentials Get Username: '%s'  Password:'%s'.\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         (char *)buffDesc.pstart, (char *)buffDesc2.pstart);

  return;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiTransferRemoteUserCredentialsSet()
* 
* @end
*********************************************************************/
void transferOpENAPISanityRemoteUserCredentialsSet(openapiClientHandle_t *clientHandle)
{
  printf("Testing openapiTransferRemoteUserCredentialsSet(): Sanity test N/A. Test explicitly with '%s 15 <username> <password>'\n", exampleName);
  return;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiTransferDownStartSet()
* 
* @end
*********************************************************************/
void transferOpENAPISanityDownStartSet(openapiClientHandle_t *clientHandle) 
{
  open_error_t result;
  fd_set s_rd, s_wr, s_ex;
  char buffer[OPEN_BUFF_EX_SIZE];
  struct timeval timeout;
  int32_t retval, c = '\0';

  printf("Testing openapiTransferDownStartSet(): \n");

  transferOpENAPIConfigDump(clientHandle);

  printf(" INPUT  -- Continue File Transfer Operation (Y/N+<enter>)?\n");

  FD_ZERO(&s_rd);
  FD_ZERO(&s_wr);
  FD_ZERO(&s_ex);
  FD_SET(fileno(stdin), &s_rd);
  timeout.tv_sec  = 15;
  timeout.tv_usec = 0;
  retval = select(fileno(stdin)+1, &s_rd, &s_wr, &s_ex, &timeout);

  if (retval == -1)
  {
      perror("select()");
  }
  else if (retval)
  {
    /* FD_ISSET(0, &rfds) is true so input is available now. */
    fgets(buffer, sizeof(buffer), stdin);

    c = toupper(buffer[0]);
  }
  if ('Y' != c)
  {
    printf(" FAILED -- Transfer Aborted!\n");
    return;
  }
 
  result = openapiTransferDownStartSet(clientHandle);
  printf(" %s -- Start Download:\n", (OPEN_E_NONE==result)?"PASSED":"FAILED");

  return;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiTransferInProgressGet()
* 
* @end
*********************************************************************/
void transferOpENAPISanityInProgressGet(openapiClientHandle_t *clientHandle) 
{
  open_error_t result;
  OPEN_BOOL_t   isInProgress;

  printf("Testing openapiTransferInProgressGet():\n");
  (void)openapiEventWait(clientHandle, 2);
  result = openapiTransferInProgressGet(clientHandle, &isInProgress);
  printf(" %s -- Transfer In Progress: %s\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED", 
         OPEN_TRUE==isInProgress?"True":"False");

  return;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiTransferResultGet()
* 
* @end
*********************************************************************/
void transferOpENAPISanityResultGet(openapiClientHandle_t *clientHandle) 
{
  open_error_t result;
  open_buffdesc buffDesc;
  char buffer[OPEN_BUFF_EX_SIZE];
  OPEN_TRANSFER_STATUS_t resultStatus;

  buffDesc.pstart = buffer;
  buffDesc.size   = sizeof(buffer);
  printf("Testing openapiTransferResultGet():\n");
  result = openapiTransferResultGet(clientHandle, &resultStatus, &buffDesc);
  printf(" %s -- Transfer Result Get Status: %d - '%s'\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED", resultStatus,
         (char *)buffDesc.pstart);

  return;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiTransferDownloadResultCodeGet()
* 
* @end
*********************************************************************/
void transferOpENAPISanityDownloadResultCodeGet(openapiClientHandle_t *clientHandle) 
{
  open_error_t result;
  OPEN_TRANSFER_CODE_t resultCode;

  printf("Testing openapiTransferDownloadResultCodeGet():\n");
  result = openapiTransferDownloadResultCodeGet(clientHandle, &resultCode);
  printf(" %s -- Transfer Result Get Code: %d (%s)\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED", resultCode, 
         resultCodePrintable(resultCode));

  return;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiDigitalSignatureVerifyConfigGet()
* 
* @end
*********************************************************************/
void transferOpENAPISanityDigitalSignatureVerifyConfigGet(openapiClientHandle_t *clientHandle) 
{
  open_error_t result;
  OPEN_DIGITAL_SIGNATURE_OPTION_t dsvConfigOption;

  printf("Testing openapiDigitalSignatureVerifyConfigGet():\n");
  result = openapiDigitalSignatureVerifyConfigGet(clientHandle, &dsvConfigOption);
  printf(" %s -- Digital Sig Verify Config Get: %s(%d).\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         verificationOptionPrintable(dsvConfigOption), dsvConfigOption);
  return;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note     openapiDigitalSignatureVerifyConfigSet()
* 
* @end
*********************************************************************/
void transferOpENAPISanityDigitalSignatureVerifyConfigSet(openapiClientHandle_t *clientHandle) 
{
  open_error_t result;
  OPEN_DIGITAL_SIGNATURE_OPTION_t dsvConfigCurrent, dsvConfigSet, dsvConfigTest;

  printf("Testing openapiDigitalSignatureVerifyConfigSet():\n");

  /* test Code */
  result = openapiDigitalSignatureVerifyConfigGet(clientHandle, &dsvConfigCurrent);
  printf(" %s -- Digital Sig Verify Config Get: %s(%d).\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         verificationOptionPrintable(dsvConfigCurrent), dsvConfigCurrent);

  for (dsvConfigSet = OPEN_DIGITAL_SIGNATURE_VERIFY_NONE; dsvConfigSet < OPEN_DIGITAL_SIGNATURE_VERIFY_LAST; dsvConfigSet++)
  {
    result = openapiDigitalSignatureVerifyConfigSet(clientHandle, dsvConfigSet);
    printf(" %s -- Digital Sig Verify Config Set: %s(%d).\n",
           (OPEN_E_NONE==result)?"PASSED":"FAILED",
           verificationOptionPrintable(dsvConfigSet), dsvConfigSet);
    if (result != OPEN_E_NONE) return;

    result = openapiDigitalSignatureVerifyConfigGet(clientHandle, &dsvConfigTest);
    printf(" %s -- Digital Sig Verify Config Get: %s(%d).\n",
           (OPEN_E_NONE==result)?"PASSED":"FAILED",
           verificationOptionPrintable(dsvConfigTest), dsvConfigTest);
    if (result != OPEN_E_NONE) return;
  }

  result = openapiDigitalSignatureVerifyConfigSet(clientHandle, dsvConfigCurrent);
  if (result != OPEN_E_NONE) return;
  result = openapiDigitalSignatureVerifyConfigGet(clientHandle, &dsvConfigTest);
  printf(" Restoring Digital Sig Verify Config: %s (%d).\n",
         verificationOptionPrintable(dsvConfigTest), dsvConfigTest);

  return;
}

/*********************************************************************
* @purpose  Sanity of a Copy Config OpEN API.
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
*
* @returns  none
*
* @note     openapiCopyConfigFile()
*
* @end
*********************************************************************/
void transferOpENAPISanityCopyConfigFile(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  printf("Testing openapiCopyConfigFile():\n");
  result = openapiCopyConfigFile(clientHandle, OPEN_COPY_CONFIG_DIRECTIVE_BTOS);
  printf(" %s -- Copy Config File: \n", (OPEN_E_NONE==result)?"PASSED":"FAILED");
  return;
}

/*******************************************************************
*
* @brief  This function prints the Transfer Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printTransferAppMenu()
{
  printf("\nUsage:  transfer_example <test#> <arg1> <arg2> ... \n\n");
  printf("Test  0: Transfer OpEN APIs Sanity:             %s  0 \n", exampleName);
  printf("Test  1: Convert Host/IP String to IP Address:  %s  1 <string>\n", exampleName);
  printf("Test  2: Get Image Header Size:                 %s  2 \n", exampleName);
  printf("Test  3: Image Header Check:                    %s  3 <filename>\n", exampleName);
  printf("Test  4: Get Transfer Mode:                     %s  4 \n", exampleName);
  printf("Test  5: Set Transfer Mode:                     %s  5 <tftp|sftp|scp|#mode>\n", exampleName);
  printf("Test  6: Get Transfer File Type:                %s  6 \n", exampleName);
  printf("Test  7: Set Transfer File Type:                %s  7 <code|config|license|#type>\n", exampleName);
  printf("Test  8: Get Transfer Server Address:           %s  8 \n", exampleName);
  printf("Test  9: Set Transfer Server Address:           %s  9 <Address/Hostname>\n", exampleName);
  printf("Test 10: Get Transfer Remote File Name:         %s 10 \n", exampleName);
  printf("Test 11: Set Transfer Remote File Name:         %s 11 <name> <path>\n", exampleName);
  printf("Test 12: Get Transfer Local File Name:          %s 12 \n", exampleName);
  printf("Test 13: Set Transfer Local File Name:          %s 13 <label>\n", exampleName);
  printf("Test 14: Get User Credentials:                  %s 14 \n", exampleName);
  printf("Test 15: Set User Credentials:                  %s 15 <username> <password>\n", exampleName);
  printf("Test 16: Get Digital Signature Verification:    %s 16 \n", exampleName);
  printf("Test 17: Set Digital Signature Verification:    %s 17 <none|image|script|all|#mode>\n", exampleName);
  printf("Test 18: Start Download:                        %s 18 \n", exampleName);
  printf("Test 19: Transfer In Progress:                  %s 19 \n", exampleName);
  printf("Test 20: Transfer Result Get Status:            %s 20\n", exampleName);
  printf("Test 21: Transfer Result Get Code:              %s 21\n", exampleName);
  printf("Test 22: Dump Current Config:                   %s 22\n", exampleName);
  printf("Test 23: Copy Running Config File to Backup:    %s 23\n", exampleName);
  printf("Test 24: Set Index of License file Trasnfer :   %s 24 <index>\n", exampleName);
  printf("\n");

  return;
}

/*********************************************************************
* @purpose  Sanity of transfer OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note
* 
* @end
*********************************************************************/
void transferOpENAPISanity(openapiClientHandle_t *clientHandle)
{
  transferOpENAPISanityConvertHost(clientHandle); 
  transferOpENAPISanityHeaderSize(clientHandle);
  transferOpENAPISanityHeaderCheck(clientHandle);
  transferOpENAPISanityModeSet(clientHandle);
  transferOpENAPISanityModeGet(clientHandle);
  transferOpENAPISanityFileTypeSet(clientHandle);
  transferOpENAPISanityFileTypeGet(clientHandle);
  transferOpENAPISanityServerAddressGet(clientHandle);
  transferOpENAPISanityServerAddressSet(clientHandle); 
  transferOpENAPISanityFileRemoteGet(clientHandle);
  transferOpENAPISanityFileRemoteSet(clientHandle);
  transferOpENAPISanityFileLocalGet(clientHandle);
  transferOpENAPISanityFileLocalSet(clientHandle);
  transferOpENAPISanityRemoteUserCredentialsGet(clientHandle);
  transferOpENAPISanityRemoteUserCredentialsSet(clientHandle);
  transferOpENAPISanityDigitalSignatureVerifyConfigGet(clientHandle);
  transferOpENAPISanityDigitalSignatureVerifyConfigSet(clientHandle);
  transferOpENAPISanityDownStartSet(clientHandle);
  transferOpENAPISanityInProgressGet(clientHandle);
  transferOpENAPISanityResultGet(clientHandle);
  transferOpENAPISanityDownloadResultCodeGet(clientHandle);
  transferOpENAPISanityCopyConfigFile(clientHandle);

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc buffDesc;
  char buffer[OPEN_BUFF_EX_SIZE*2];
  open_buffdesc buffDesc2;
  char buffer2[OPEN_BUFF_EX_SIZE];
  open_revision_data_t openApiVersion;
  char switch_os_revision_string[100];
  OPEN_TRANSFER_MODES_t xfrMode;
  OPEN_FILE_TYPES_t fileType;
  uint32_t value;
  char *ipOrHost;
  open_inet_addr_t inetAddr;
  OPEN_BOOL_t   isInProgress;
  OPEN_TRANSFER_STATUS_t resultStatus;
  OPEN_TRANSFER_CODE_t resultCode;
  OPEN_DIGITAL_SIGNATURE_OPTION_t dsvConfigOption = OPEN_DIGITAL_SIGNATURE_VERIFY_LAST;
  OPEN_COPY_CONFIG_DIRECTIVE_t directive  = OPEN_COPY_CONFIG_DIRECTIVE_BTOS;
  char *fName;
  OPEN_BOOL_t isValid;
  FILE  *fp = NULL;
  uint32_t  licenseIndex = 0;

  exampleName = argv[0];
  if (argc < 2)
  {
    printTransferAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("transfer_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Transfer API example application");

  printf(" \n");
  buffDesc.pstart = switch_os_revision_string;
  buffDesc.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &buffDesc) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  if (openapiApiVersionGet(&clientHandle, &openApiVersion) == OPEN_E_NONE)
    printf("Open API Version = %d.%d.%d.%d\n", 
           openApiVersion.release,
           openApiVersion.version,
           openApiVersion.maint_level,
           openApiVersion.build_num);
  else
    printf("Open API Version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 0:
      if (argc != 2)
      {
        printTransferAppMenu();
        exit(1);
      }
      transferOpENAPISanity(&clientHandle);
      break;

    case 1:
      if (argc != 3)
      {
        printTransferAppMenu();
        exit(1);
      }
      ipOrHost = argv[2];
      buffDesc.pstart = argv[2];
      buffDesc.size   = strlen(buffDesc.pstart)+1;
      printf("Test  1: Convert Host/IP String to IP Address:  transfer_example  1 <string>\n");
      printf("\topenapiInetAddrFromIPAddrHostNameStr() %s\n", ipOrHost);
      result = openapiInetAddrFromIPAddrHostNameStr(&clientHandle, &buffDesc, &inetAddr);
      snprintf(buffer, sizeof(buffer), "Fail");
      if (OPEN_E_NONE==result)
      {
        if (inetAddr.family == OPEN_AF_INET)
        {
          inet_ntop(AF_INET, (void*)&(inetAddr.addr.ipv4), buffer, OPEN_BUFF_EX_SIZE);
        }
        else if (inetAddr.family == OPEN_AF_INET6)
        {
          inet_ntop(AF_INET6, (void*)&(inetAddr.addr.ipv6), buffer, OPEN_BUFF_EX_SIZE);
        }
      }
      printf(" %s -- Get Inet Address From String: %s => %s. (result = %d)\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED", 
             argv[2], buffer, result);
      break;

    case 2:
      if (argc != 2)
      {
        printTransferAppMenu();
        exit(1);
      }
      result = openapiImageHeaderSizeGet(&clientHandle, &value);
      printf(" %s -- Get Image Header Size: %d. (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED", value, result);
      break;

    case 3:
      result = openapiImageHeaderSizeGet(&clientHandle, &value);
      if (argc != 3)
      {
        printf("Please specify a file to sample for this test\n\n");
        printf("The file may be created from an downloadable image by extracting the top %d bytes\n", value);
        printf("using the linux command 'dd count=%d bs=1 if=image.stk of=header.bin'\n\n", value);
        printTransferAppMenu();
        exit(1);
      }
      fName = argv[2];
      if ((value<sizeof(buffer)) && (OPEN_E_NONE == result))
      {
        fp = fopen(fName, "r");
        if (NULL != fp)
        {
          buffDesc.pstart = buffer;
          buffDesc.size = fread(buffDesc.pstart, 1, value, fp);
          result = openapiImageHeaderCheck(&clientHandle, &buffDesc, &isValid);
          if (OPEN_E_UNAVAIL == result)
          {
            printf(" FAILED-- Image Header Check, is NOT available on this platform!\n\n");
          }
          else
          {
            printf(" %s -- Image Header Check - file [%s] is%sa valid image for this system (%d). (result = %d)\n\n",
                   (OPEN_E_NONE==result)?"PASSED":"FAILED", fName, 
                   isValid==OPEN_TRUE?" ":" NOT ", isValid, result);
          }
          fclose(fp);
        }
        else
        {
          printf(" FAILED-- Image Header Check, file: '%s' cannot be opened.\n\n", fName);
          printf("\nPlease specify a file to sample for this test\n\n");
          printf("The file may be created from an downloadable image by extracting the top %d bytes\n", value);
          printf("using the linux command 'dd count=%d bs=1 if=image.stk of=header.bin'\n\n", value);
        }
      }
      else
      {
        printf(" FAILED-- Image Header Check, buffer is too small: %u needs to be %d.\n\n", sizeof(buffer), value);
      }
      break;

    case 4:
      if (argc != 2)
      {
        printTransferAppMenu();
        exit(1);
      }
      result = openapiTransferModeGet(&clientHandle, &xfrMode);
      printf(" %s -- Transfer Mode Get: %s. (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED",
             transferModePrintable(xfrMode), result);
      break;

    case 5:
      if (argc != 3)
      {
        printTransferAppMenu();
        exit(1);
      }
      if (0 == strcmp(argv[2], "tftp"))
      {
        xfrMode = OPEN_TRANSFER_TFTP;
      }
      else if (0 == strcmp(argv[2], "sftp"))
      {
        xfrMode = OPEN_TRANSFER_SFTP;
      }
      else if (0 == strcmp(argv[2], "scp"))
      {
        xfrMode = OPEN_TRANSFER_SCP;
      }
      else
      {
        xfrMode = atoi(argv[2]);
      }
      result = openapiTransferModeSet(&clientHandle, xfrMode);
      printf(" %s -- Transfer Mode Set: %s. (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED",
             transferModePrintable(xfrMode), result);
      break;

    case 6:
      if (argc != 2)
      {
        printTransferAppMenu();
        exit(1);
      }
      result = openapiTransferFileTypeGet(&clientHandle, &fileType);
      printf(" %s -- Transfer Type Get: %s. (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED",
             transferFiletypePrintable(fileType), result);
      break;

    case 7:
      if (argc != 3)
      {
        printTransferAppMenu();
        exit(1);
      }
      if (0 == strcmp(argv[2], "code"))
      {
        fileType = OPEN_FILE_TYPE_CODE;
      }
      else if (0 == strcmp(argv[2], "config"))
      {
        fileType = OPEN_FILE_TYPE_CONFIG;
      }
      else if (0 == strcmp(argv[2], "license"))
      {
        fileType = OPEN_FILE_TYPE_LICENSE;
      }
      else
      {
        fileType = atoi(argv[2]);
      }
      result = openapiTransferFileTypeSet(&clientHandle, fileType);
      printf(" %s -- Transfer Type Set: %s. (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED",
             transferFiletypePrintable(fileType), result);
      break;

    case 8:
      if (argc != 2)
      {
        printTransferAppMenu();
        exit(1);
      }
      result = openapiTransferServerAddressGet(&clientHandle, &inetAddr);
      snprintf(buffer, sizeof(buffer), "Fail");
      if (OPEN_E_NONE==result)
      {
        if (inetAddr.family == OPEN_AF_INET)
        {
          inet_ntop(AF_INET, (void*)&(inetAddr.addr.ipv4), buffer, OPEN_BUFF_EX_SIZE);
        }
        else if (inetAddr.family == OPEN_AF_INET6)
        {
          inet_ntop(AF_INET6, (void*)&(inetAddr.addr.ipv6), buffer, OPEN_BUFF_EX_SIZE);
        }
      }
      printf(" %s -- Transfer Server Address Get: %s. (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED", buffer, result);
      break;


    case 9:
      if (argc != 3)
      {
        printTransferAppMenu();
        exit(1);
      }
      buffDesc.pstart = argv[2];
      buffDesc.size   = strlen(buffDesc.pstart) + 1;
      result = openapiInetAddrFromIPAddrHostNameStr(&clientHandle, &buffDesc, &inetAddr);
      if (OPEN_E_NONE==result)
      {
        result = openapiTransferServerAddressSet(&clientHandle, inetAddr);
      }
      snprintf(buffer, sizeof(buffer), "Fail");
      if (OPEN_E_NONE==result)
      {
        if (inetAddr.family == OPEN_AF_INET)
        {
          inet_ntop(AF_INET, (void*)&(inetAddr.addr.ipv4), buffer, OPEN_BUFF_EX_SIZE);
        }
        else if (inetAddr.family == OPEN_AF_INET6)
        {
          inet_ntop(AF_INET6, (void*)&(inetAddr.addr.ipv6), buffer, OPEN_BUFF_EX_SIZE);
        }
      }
      printf(" %s -- Set Server Address: %s (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED", 
             buffer, result);
      break;

    case 10:
      if (argc != 2)
      {
        printTransferAppMenu();
        exit(1);
      }
      buffDesc.pstart = buffer;
      buffDesc.size   = sizeof(buffer);
      buffDesc2.pstart = buffer2;
      buffDesc2.size   = sizeof(buffer2);
      result = openapiTransferFileRemoteGet(&clientHandle, &buffDesc2, &buffDesc);
      printf(" %s -- Transfer Remote File Name Get: [%s]%s. (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED",
             (char *)buffDesc2.pstart, (char *)buffDesc.pstart, result);
      break;

    case 11:
      if ((argc != 3) && (argc != 4))
      {
        printTransferAppMenu();
        exit(1);
      }
      if (argc == 4)
      {
        buffDesc2.pstart = argv[3];
      }
      else
      {
        strcpy(buffer2, "");
        buffDesc2.pstart = buffer2;
      }
      buffDesc2.size   = strlen(buffDesc2.pstart) + 1;
      buffDesc.pstart = argv[2];
      buffDesc.size   = strlen(buffDesc.pstart) + 1;
      result = openapiTransferFileRemoteSet(&clientHandle, &buffDesc2, &buffDesc);
      printf(" %s -- Transfer Remote File Name Set Path:'%s' Name:'%s'. (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED",
             (char *)buffDesc2.pstart, (char *)buffDesc.pstart, result);
      break;

    case 12:
      if (argc != 2)
      {
        printTransferAppMenu();
        exit(1);
      }
      buffDesc.pstart = buffer;
      buffDesc.size   = sizeof(buffer);
      result = openapiTransferFileLocalGet(&clientHandle, &buffDesc);
      printf(" %s -- Transfer Local File Name Get: %s. (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED",
             (char *)buffDesc.pstart, result);
      break;

    case 13:
      if (argc != 3)
      {
        printTransferAppMenu();
        exit(1);
      }
      buffDesc.pstart = argv[2];
      buffDesc.size   = strlen(buffDesc.pstart) + 1;
      result = openapiTransferFileLocalSet(&clientHandle, &buffDesc);
      printf(" %s -- Transfer Local File Name Set: %s. (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED",
             (char *)buffDesc.pstart, result);
      break;

    case 14:
      if (argc != 2)
      {
        printTransferAppMenu();
        exit(1);
      }
      buffDesc.pstart = buffer;
      buffDesc.size   = sizeof(buffer);
      buffDesc2.pstart = buffer2;
      buffDesc2.size   = sizeof(buffer2);
      result = openapiTransferRemoteUserCredentialsGet(&clientHandle, &buffDesc, &buffDesc2);
      printf(" %s -- Remote User Credentials Get Username: '%s'  Password:'%s'. (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED",
             (char *)buffDesc.pstart, (char *)buffDesc2.pstart, result);
      break;

    case 15:
      if ((argc != 2) && (argc != 3) && (argc != 4))
      {
        printTransferAppMenu();
        exit(1);
      }
      /* pre-set username and password to NULL */
      strcpy(buffer, "");
      buffDesc.pstart = buffer;
      strcpy(buffer2, "");
      buffDesc2.pstart = buffer2;

      if (argc == 4) /* password specified */
      {
        buffDesc2.pstart = argv[3];
      }

      if (argc > 2)  /* username specified */
      {
        buffDesc.pstart = argv[2];
      }
      buffDesc.size   = strlen(buffDesc.pstart) + 1;
      buffDesc2.size  = strlen(buffDesc2.pstart) + 1;

      result = openapiTransferRemoteUserCredentialsSet(&clientHandle, &buffDesc, &buffDesc2);
      printf(" %s -- Remote User Credentials Get Username: '%s'  Password:'%s'. (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED",
             (char *)buffDesc.pstart, (char *)buffDesc2.pstart, result);
      break;

    case 16:
      if (argc != 2)
      {
        printTransferAppMenu();
        exit(1);
      }
      result = openapiDigitalSignatureVerifyConfigGet(&clientHandle, &dsvConfigOption);
      printf(" %s -- Digital Sig Verify Config Get: %s(%d). (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED",
             verificationOptionPrintable(dsvConfigOption), dsvConfigOption,
             result);
      break;
   
    case 17:
      if (argc != 3)
      {
        printTransferAppMenu();
        exit(1);
      }
      if (0 == strcmp(argv[2], "none"))
      {
        dsvConfigOption = OPEN_DIGITAL_SIGNATURE_VERIFY_NONE;
      }
      else if (0 == strcmp(argv[2], "image"))
      {
        dsvConfigOption = OPEN_DIGITAL_SIGNATURE_VERIFY_IMAGE_ONLY;
      }
      else if (0 == strcmp(argv[2], "script"))
      {
        dsvConfigOption = OPEN_DIGITAL_SIGNATURE_VERIFY_CONFIG_SCRIPT_ONLY;
      }
      else if (0 == strcmp(argv[2], "all"))
      {
        dsvConfigOption = OPEN_DIGITAL_SIGNATURE_VERIFY_ALL;
      }
      else
      {
        dsvConfigOption = atoi(argv[2]);
      }
      result = openapiDigitalSignatureVerifyConfigSet(&clientHandle, dsvConfigOption);
      printf(" %s -- Digital Sig Verify Config Get: %s(%d). (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED",
             verificationOptionPrintable(dsvConfigOption), dsvConfigOption,
             result);
      break;

    case 18:
      if (argc != 2)
      {
        printTransferAppMenu();
        exit(1);
      }
      transferOpENAPIConfigDump(&clientHandle);
      result = openapiTransferDownStartSet(&clientHandle);
      printf(" %s -- Start Download: (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED", result);
      break;

    case 19:
      if (argc != 2)
      {
        printTransferAppMenu();
        exit(1);
      }
      result = openapiTransferInProgressGet(&clientHandle, &isInProgress);
      printf(" %s -- Transfer In Progress: %s (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED", 
             OPEN_TRUE==isInProgress?"True":"False",
             result);
      break;

    case 20:
      if (argc != 2)
      {
        printTransferAppMenu();
        exit(1);
      }
      buffDesc.pstart = buffer;
      buffDesc.size   = sizeof(buffer);
      result = openapiTransferResultGet(&clientHandle, &resultStatus, &buffDesc);
      printf(" %s -- Transfer Result Get Status: %d - '%s'. (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED", resultStatus,
             (char *)buffDesc.pstart, result);

      break;

    case 21:
      if (argc != 2)
      {
        printTransferAppMenu();
        exit(1);
      }
      result = openapiTransferDownloadResultCodeGet(&clientHandle, &resultCode);
      printf(" %s -- Transfer Result Get Code: %d (%s). (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED", resultCode, 
             resultCodePrintable(resultCode), result);
      break;

    case 22:
      transferOpENAPIConfigDump(&clientHandle);
      break;

    case 23:
      if (argc != 2)
      {
        printTransferAppMenu();
        exit(1);
      }
      result = openapiCopyConfigFile(&clientHandle, directive);
      printf(" %s -- Copy Config File directive: %d. (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED", directive, result);
      break;

    case 24:
      if (argc != 3)
      {
        printTransferAppMenu();
        exit(1);
      }
      licenseIndex = atoi(argv[2]);
      result = openapiTransferLicenseIndexSet(&clientHandle, licenseIndex);
      printf(" Set License file Index: %d. (result = %d)\n\n",
             (OPEN_E_NONE==result)?"PASSED":"FAILED", licenseIndex, result);
    break;
    default:
      printTransferAppMenu();
      break;
  }

  printf("\n");
  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Transfer API example application");
        
  (void) openapiClientTearDown(&clientHandle);        
  return 0;
}
