/*********************************************************************
*
*  Copyright 2020 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  capability_licensing_example.c
*
* @purpose   OpEN Capability_licensing example.
*
* @component OpEN
*
* @create    03/31/2020
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_capability_licensing.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the Capability_licensing Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Get the number of license files in the system: %s 1 \n", name);
  printf("Test 2: Get the first license index in the system: %s 2 \n", name);
  printf("Test 3: Get the next license index in the system: %s 3 <currentIndex>\n", name);
  printf("Test 4: Get the type of license: %s 4 <licenseIndex> \n", name);
  printf("Test 5: Get the status of license in the system: %s 5 <licenseIndex> \n", name);
  printf("Test 6: Get the parameter value of license file in the system: %s 6 <licenseIndex><paramType>\n", name);
  printf("Test 7: Get the version number of license file in the system: %s 7 <licenseIndex>\n", name);
  printf("Test 8: Get the key of license file in the system: %s 8 <licenseIndex>\n", name);
  printf("Test 9: Get the model name of license file: %s 9 <licenseIndex>\n", name);
  printf("Test 10: Get the serial number of license file: %s 10 <licenseIndex>\n", name);
  printf("Test 11: Run API sanity checks: %s 11 \n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  OPEN_CAPAB_LICENSE_STATUS_TYPE_t status;
  OPEN_CAPAB_LICENSE_TYPE_t type;
  OPEN_CAPAB_LICENSE_FILE_PARAM_TYPE_t paramType;
  uint32_t firstLicenseIndex;
  uint32_t numOfLicenses;
  uint32_t nextLicenseIndex;
  uint32_t licenseIndex;
  char buffer[128];
  open_buffdesc buf;
  open_buffdesc zeroLenBuf;
  open_buffdesc badBufdescPointer;
  open_buffdesc badBufdescLen;

  badBufdescPointer.pstart = (void *) NULL;
  badBufdescPointer.size = sizeof(buffer);
  badBufdescLen.pstart = buffer;
  badBufdescPointer.size = 0;
  buf.pstart = buffer;
  buf.size = sizeof(buffer);
  zeroLenBuf.pstart = buffer;
  zeroLenBuf.size = 0;

  printf("Testing Capability_licensing OpEN APIs sanity:\n");

  printf("Testing openapiCapabLicenseCountGet():\n");
  result = openapiCapabLicenseCountGet(NULL, &numOfLicenses);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiCapabLicenseCountGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiCapabLicenseFirstIndexGet():\n");
  result = openapiCapabLicenseFirstIndexGet(NULL, &firstLicenseIndex);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiCapabLicenseFirstIndexGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiCapabLicenseNextIndexGet():\n");
  result = openapiCapabLicenseNextIndexGet(NULL, firstLicenseIndex, &nextLicenseIndex);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiCapabLicenseNextIndexGet(clientHandle, firstLicenseIndex, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiCapabLicenseTypeGet():\n");
  licenseIndex = 1;
  result = openapiCapabLicenseTypeGet(NULL, licenseIndex, &type);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiCapabLicenseTypeGet(clientHandle, 1 - 1, &type);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseTypeGet(clientHandle, OPEN_CAPAB_LIC_MAX_LICENSE_FILES + 1, &type);
  printf("greater than maximum value argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseTypeGet(clientHandle, licenseIndex, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiCapabLicenseStatusGet():\n");
  result = openapiCapabLicenseStatusGet(NULL, licenseIndex, &status);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiCapabLicenseStatusGet(clientHandle, 1 - 1, &status);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseStatusGet(clientHandle, OPEN_CAPAB_LIC_MAX_LICENSE_FILES + 1, &status);
  printf("greater than maximum value argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseStatusGet(clientHandle, licenseIndex, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiCapabLicenseParamGet():\n");
  paramType = OPEN_CAPAB_LICENSE_FILE_PARAM_COPY;
  result = openapiCapabLicenseParamGet(NULL, licenseIndex, paramType, &buf);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiCapabLicenseParamGet(clientHandle, 1 - 1, paramType, &buf);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseParamGet(clientHandle, OPEN_CAPAB_LIC_MAX_LICENSE_FILES + 1, paramType, &buf);
  printf("greater than maximum value argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseParamGet(clientHandle, licenseIndex, OPEN_CAPAB_LICENSE_FILE_PARAM_COPY - 1, &buf);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseParamGet(clientHandle, licenseIndex, OPEN_CAPAB_LICENSE_FILE_PARAM_DESC + 1, &buf);
  printf("greater than maximum value argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseParamGet(clientHandle, licenseIndex, paramType, NULL);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiCapabLicenseParamGet(clientHandle, licenseIndex, paramType, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 4:(result = %d)\n", result);
  result = openapiCapabLicenseParamGet(clientHandle, licenseIndex, paramType, &zeroLenBuf);
  printf("buffdesc 0 length size argument 4:(result = %d)\n", result);
  badBufdescLen.size = OPEN_CAPAB_LIC_MAX_FILE_PARAM_VAL_LEN - 1;
  result = openapiCapabLicenseParamGet(clientHandle, licenseIndex, paramType, &badBufdescLen);
  printf("not equal to required value argument 4:(result = %d)\n", result);

  printf("Testing openapiCapabLicenseVersionGet():\n");
  result = openapiCapabLicenseVersionGet(NULL, licenseIndex, &buf);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiCapabLicenseVersionGet(clientHandle, 1 - 1, &buf);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseVersionGet(clientHandle, OPEN_CAPAB_LIC_MAX_LICENSE_FILES + 1, &buf);
  printf("greater than maximum value argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseVersionGet(clientHandle, licenseIndex, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseVersionGet(clientHandle, licenseIndex, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseVersionGet(clientHandle, licenseIndex, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  badBufdescLen.size = OPEN_CAPAB_LIC_MAX_VER_LEN - 1;
  result = openapiCapabLicenseVersionGet(clientHandle, licenseIndex, &badBufdescLen);
  printf("not equal to required value argument 3:(result = %d)\n", result);

  printf("Testing openapiCapabLicenseKeyGet():\n");
  result = openapiCapabLicenseKeyGet(NULL, licenseIndex, &buf);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiCapabLicenseKeyGet(clientHandle, 1 - 1, &buf);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseKeyGet(clientHandle, OPEN_CAPAB_LIC_MAX_LICENSE_FILES + 1, &buf);
  printf("greater than maximum value argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseKeyGet(clientHandle, licenseIndex, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseKeyGet(clientHandle, licenseIndex, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseKeyGet(clientHandle, licenseIndex, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  badBufdescLen.size = OPEN_CAPAB_LIC_MAX_KEY_LEN - 1;
  result = openapiCapabLicenseKeyGet(clientHandle, licenseIndex, &badBufdescLen);
  printf("not equal to required value argument 3:(result = %d)\n", result);

  printf("Testing openapiCapabLicenseModelGet():\n");
  result = openapiCapabLicenseModelGet(NULL, licenseIndex, &buf);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiCapabLicenseModelGet(clientHandle, 1 - 1, &buf);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseModelGet(clientHandle, OPEN_CAPAB_LIC_MAX_LICENSE_FILES + 1, &buf);
  printf("greater than maximum value argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseModelGet(clientHandle, licenseIndex, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseModelGet(clientHandle, licenseIndex, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseModelGet(clientHandle, licenseIndex, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  badBufdescLen.size = OPEN_CAPAB_LIC_MAX_MODEL_LEN - 1;
  result = openapiCapabLicenseModelGet(clientHandle, licenseIndex, &badBufdescLen);
  printf("not equal to required value argument 3:(result = %d)\n", result);

  printf("Testing openapiCapabLicenseSerialNumberGet():\n");
  result = openapiCapabLicenseSerialNumberGet(NULL, licenseIndex, &buf);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiCapabLicenseSerialNumberGet(clientHandle, 1 - 1, &buf);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseSerialNumberGet(clientHandle, OPEN_CAPAB_LIC_MAX_LICENSE_FILES + 1, &buf);
  printf("greater than maximum value argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseSerialNumberGet(clientHandle, licenseIndex, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseSerialNumberGet(clientHandle, licenseIndex, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiCapabLicenseSerialNumberGet(clientHandle, licenseIndex, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  badBufdescLen.size = OPEN_CAPAB_LIC_MAX_SERIAL_LEN - 1;
  result = openapiCapabLicenseSerialNumberGet(clientHandle, licenseIndex, &badBufdescLen);
  printf("not equal to required value argument 3:(result = %d)\n", result);

  return;
}


/*********************************************************************
* @purpose  Get the number of license files in the system
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    numOfLicenses       @b{(output)}  Number of license(s) present in the system.

*
* @returns  none
*
* @end
*********************************************************************/
void capabLicenseCountGet(openapiClientHandle_t *client_handle, uint32_t *numOfLicenses)
{
  open_error_t result;

  if ((result = openapiCapabLicenseCountGet(client_handle, numOfLicenses)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the number of license files in the system (result = %d)\n", result);
  }
  else
  {
    printf("Number of licenses in the system is %d. \n", *numOfLicenses);
  }
  return;
}


/*********************************************************************
* @purpose  Get the first license index in the system.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    firstLicenseIndex       @b{(output)}  First license index in the system.

*
* @returns  none
*
* @end
*********************************************************************/
void capabLicenseFirstIndexGet(openapiClientHandle_t *client_handle, uint32_t *firstLicenseIndex)
{
  open_error_t result;

  if ((result = openapiCapabLicenseFirstIndexGet(client_handle, firstLicenseIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the first license index in the system. (result = %d)\n", result);
  }
  else
  {
    printf("First license index in the system is %d. \n", *firstLicenseIndex);
  }
  return;
}

/*********************************************************************
* @purpose  Get the next license index in the system.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    nextLicenseIndex       @b{(output)}  Next valid license index in the system.

*
* @returns  none
*
* @end
*********************************************************************/
void capabLicenseNextIndexGet(openapiClientHandle_t *client_handle, uint32_t currentIndex,
                              uint32_t *nextLicenseIndex)
{
  open_error_t result;
  if ((result = openapiCapabLicenseNextIndexGet(client_handle, currentIndex, nextLicenseIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the next license index in the system. (result = %d)\n", result);
  }
  else
  {
    printf("Next license index in the system is %d. \n", *nextLicenseIndex);
  }
  return;
}


/*********************************************************************
* @purpose  Get the type of license.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    licenseIndex       @b{(input)}  License index.
* @param    type       @b{(output)}  Type of the license.

*
* @returns  none
*
* @end
*********************************************************************/
void capabLicenseTypeGet(openapiClientHandle_t *client_handle, uint32_t  licenseIndex, OPEN_CAPAB_LICENSE_TYPE_t *type)
{
  open_error_t result;

  if ((result = openapiCapabLicenseTypeGet(client_handle, licenseIndex, type)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the type of license. (result = %d)\n", result);
  }
  else
  {
    printf("License type at license index %d is %d \n", licenseIndex, *type);
  }
  return;
}


/*********************************************************************
* @purpose  Get the status of license in the system.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    licenseIndex       @b{(input)}  License index.
* @param    status       @b{(output)}  Status of the license.

*
* @returns  none
*
* @end
*********************************************************************/
void capabLicenseStatusGet(openapiClientHandle_t *client_handle, uint32_t  licenseIndex, 
                           OPEN_CAPAB_LICENSE_STATUS_TYPE_t *status)
{
  open_error_t result;

  if ((result = openapiCapabLicenseStatusGet(client_handle, licenseIndex, status)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the status of license in the system. (result = %d)\n", result);
  }
  else
  {
    printf("License status at license index %d is %d \n", licenseIndex, *status);
  }
  return;
}


/*********************************************************************
* @purpose  Get the parameter value of license file in the system.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    licenseIndex       @b{(input)}  License index.
* @param    paramType       @b{(input)}  Parameter type.
* @param    paramValue       @b{(output)}  Parameter value associated with the parameter type.

*
* @returns  none
*
* @end
*********************************************************************/
void capabLicenseParamGet(openapiClientHandle_t *client_handle, uint32_t  licenseIndex, OPEN_CAPAB_LICENSE_FILE_PARAM_TYPE_t paramType, open_buffdesc *paramValue)
{
  open_error_t result;

  if ((result = openapiCapabLicenseParamGet(client_handle, licenseIndex, paramType, paramValue)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the parameter value of license file in the system. (result = %d)\n", result);
  }
  else
  {
    printf("License param value for license file %d param type %d is %s\n", 
            licenseIndex, paramType, (char *)paramValue->pstart);
  }
  return;
}


/*********************************************************************
* @purpose  Get the version number of license file in the system.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    licenseIndex       @b{(input)}  License index.
* @param    version       @b{(output)}  Version number of the license file.

*
* @returns  none
*
* @end
*********************************************************************/
void capabLicenseVersionGet(openapiClientHandle_t *client_handle, uint32_t  licenseIndex, open_buffdesc *version)
{
  open_error_t result;

  if ((result = openapiCapabLicenseVersionGet(client_handle, licenseIndex, version)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the version number of license file in the system. (result = %d)\n", result);
  }
  else
  {
    printf("License version for license file %d is %s \n", licenseIndex, (char *)version->pstart);
  }
  return;
}


/*********************************************************************
* @purpose  Get the key of license file in the system.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    licenseIndex       @b{(input)}  License index.
* @param    version       @b{(output)}  Key of the license file.

*
* @returns  none
*
* @end
*********************************************************************/
void capabLicenseKeyGet(openapiClientHandle_t *client_handle, uint32_t  licenseIndex, 
                        open_buffdesc *key)
{
  open_error_t result;

  if ((result = openapiCapabLicenseKeyGet(client_handle, licenseIndex, key)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the key of license file in the system. (result = %d)\n", result);
  }
  else
  {
    printf("License key for license file %d is %s \n", licenseIndex, (char *)key->pstart);
  }
  return;
}


/*********************************************************************
* @purpose  Get the model name of license file.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    licenseIndex       @b{(input)}  License index.
* @param    model       @b{(output)}  Model of the license file.

*
* @returns  none
*
* @end
*********************************************************************/
void capabLicenseModelGet(openapiClientHandle_t *client_handle, uint32_t  licenseIndex, open_buffdesc *model)
{
  open_error_t result;

  if ((result = openapiCapabLicenseModelGet(client_handle, licenseIndex, model)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the model name of license file. (result = %d)\n", result);
  }
  else
  {
    printf("License model for license file %d is %s \n", licenseIndex, (char *)model->pstart);
  }
  return;
}


/*********************************************************************
* @purpose  Get the serial number of license file.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    licenseIndex       @b{(input)}  License index.
* @param    serialNum       @b{(output)}  Serial number of the license file.

*
* @returns  none
*
* @end
*********************************************************************/
void capabLicenseSerialNumberGet(openapiClientHandle_t *client_handle, uint32_t  licenseIndex, open_buffdesc *serialNum)
{
  open_error_t result;

  if ((result = openapiCapabLicenseSerialNumberGet(client_handle, licenseIndex, serialNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the serial number of license file. (result = %d)\n", result);
  }
  else
  {
    printf("Serial number of the license file %d is %s \n", licenseIndex, (char *)serialNum->pstart);
  }
  return;
}


/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for Capability_licensing
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum, numOfLicenses = 0;
  uint32_t firstLicenseIndex = 0, licenseIndex = 0, nextLicenseIndex = 0;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  OPEN_CAPAB_LICENSE_FILE_PARAM_TYPE_t   paramType = OPEN_CAPAB_LICENSE_FILE_PARAM_COPY;
  OPEN_CAPAB_LICENSE_TYPE_t   type = OPEN_CAPAB_LICENSE_TYPE_UNKNOWN;
  OPEN_CAPAB_LICENSE_STATUS_TYPE_t   status = OPEN_CAPAB_LICENSE_STATUS_FILE_NOT_PRESENT;
  unsigned char   paramValue[OPEN_CAPAB_LIC_MAX_FILE_PARAM_VAL_LEN+1];
  unsigned char   version[OPEN_CAPAB_LIC_MAX_VER_LEN+1];
  unsigned char   key[OPEN_CAPAB_LIC_MAX_KEY_LEN+1];
  unsigned char   model[OPEN_CAPAB_LIC_MAX_MODEL_LEN+1];
  unsigned char   serialNum[OPEN_CAPAB_LIC_MAX_SERIAL_LEN+1];
  open_buffdesc   buf;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);
  if (argc > 2)
  {
    licenseIndex = atoi(argv[2]);
  }

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("Capability_licensing example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Capability_licensing API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 2)
      {
        capabLicenseCountGet(&client_handle, &numOfLicenses);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 2)
      {
        capabLicenseFirstIndexGet(&client_handle, &firstLicenseIndex);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 3)
      {
        capabLicenseNextIndexGet(&client_handle, licenseIndex, &nextLicenseIndex);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 3)
      {
        capabLicenseTypeGet(&client_handle, licenseIndex, &type);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 3)
      {
        capabLicenseStatusGet(&client_handle, licenseIndex, &status);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 4)
      {
        paramType = atoi(argv[3]);
	buf.pstart  = paramValue;
	buf.size    = sizeof(paramValue);
        capabLicenseParamGet(&client_handle, licenseIndex, paramType, &buf);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 3)
      {
	buf.pstart  = version;
	buf.size    = sizeof(version);
        capabLicenseVersionGet(&client_handle, licenseIndex, &buf);
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 3)
      {
	buf.pstart  = key;
	buf.size    = sizeof(key);
        capabLicenseKeyGet(&client_handle, licenseIndex, &buf);
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 3)
      {
	buf.pstart  = model;
	buf.size    = sizeof(model);
        capabLicenseModelGet(&client_handle, licenseIndex, &buf);
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 3)
      {
	buf.pstart  = serialNum;
	buf.size    = sizeof(serialNum);
        capabLicenseSerialNumberGet(&client_handle, licenseIndex, &buf);
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 2)
      {
        runSanity(&client_handle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Capability_licensing API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}

