/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  authorization_example.c
*
* @purpose   Authorization OpEN APIs Example.
*
* @component OpEN
*
* @comments
*
* @create    15/03/2013
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_user_manager.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the Authorization Example 
*         Application Menu.
*
* @param    none
*
* @returns  none
*
*********************************************************************/
void printAuthorizationAppMenu()
{
  printf("Usage: authorization_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: Create authorization list name: authorization_example 1 <author-type> <list-name> \n");
  printf("Test 2: Add methods to authorization list: authorization_example 2 <author-type> <list-name> <method1> <method2> <method3> .... \n");
  printf("Test 3: Get authorization list names and configured methods of an access type: authorization_example 3 <authorization-type>\n");
  printf("Test 4: Set authorization list to an access line: authorization_example 4 <access-line> <author-type> <list-name> \n");
  printf("Test 5: Get authorization list name applied to access line: authorization_example 5 <access-line> <author-type> \n");
  printf("Test 6: Delete authorization list applied to the access-line: authorization_example 6 <access-line> <author-type> \n");
  printf("Test 7: Delete authorization list created: authorization_example 7 <author-type> <list-name> \n");
  printf("Test 8: Authorization OpEN APIs sanity: authorization_example 8 \n");

  return;
}

/*******************************************************************
* @purpose  Create authorization list for an authorization type.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    type           @b{(input)} Authorization type 
* @param    listName       @b{(input)} Authorization List name.
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void authorizationListCreate(openapiClientHandle_t *clientHandle, OPEN_USER_MGR_AUTHOR_TYPES_t type, char *listName)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, listName, (sizeof(str) - 1));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str)+1;

  result = openapiAuthorizationListNameValidate(clientHandle, &buffDesc);

  if (result == OPEN_E_NONE)
  {
    if ((result = openapiAuthorizationListCreate(clientHandle, type, &buffDesc)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to create authorization list name for type %d. (result = %d) \n", type, result);
    }
    else
    {
      printf("Authorization list name %s created successfully. \n", str);
    }
  }
  else
  {
    printf("Authorization list name is too long or contains invalid characters. \n");
  }
  return;
}

/*******************************************************************
* @purpose  Add methods to authorization list for an authorization type.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    type           @b{(input)} Authorization type 
* @param    listName       @b{(input)} Authorization List name.
* @param    authorMethods  @b{(input)} Authorization methods
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void authorizationMethodsAdd(openapiClientHandle_t *clientHandle, OPEN_USER_MGR_AUTHOR_TYPES_t type, char *listName, open_buffdesc *authorMethods)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, listName, (sizeof(str) - 1));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str)+1;


  if ((result = openapiAuthorizationMethodsAdd(clientHandle, type, &buffDesc, authorMethods)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to add methods to authorization list for type %d. (result = %d) \n", type, result);
  }
  else
  {
    printf("Methods added to authorization list successfully. \n");
  }
  return;

}

/*******************************************************************
* @purpose  Get Authorization list information.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    type           @b{(input)} Authorization type 
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void authorizationInfoGet(openapiClientHandle_t *clientHandle, OPEN_USER_MGR_AUTHOR_TYPES_t type)
{
   open_error_t result;
  uint32_t authorListNameSize;
  uint32_t maxAuthorMethods;
  uint32_t i = 0;
  char *str;
  char *methodStr;
  open_buffdesc listName;
  open_buffdesc methodList; 

  if ((result = openapiAuthorizationListNameSizeGet(clientHandle, &authorListNameSize)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get authorization list name size. (result = %d)", result);
    return;
  }

  if ((result = openapiAuthorizationMethodsMaxGet(clientHandle, &maxAuthorMethods)) != OPEN_E_NONE)
  {       
    printf("Bad return code trying to get maximum authorization methods supported. (result = %d) \n", result);
  }

  if ((str = (char *)malloc(authorListNameSize)) == NULL)
  {         
    printf("Could not allocate memory.\n");
    return; 
  }         
          
  if ((methodStr = (char *)malloc(maxAuthorMethods)) == NULL)
  {         
    printf("Could not allocate memory.\n");
    free(str);
    return; 
  }

  memset(str, 0, authorListNameSize);
  listName.pstart = str;
  listName.size = authorListNameSize;

  memset(methodStr, 0, maxAuthorMethods);
  methodList.pstart = methodStr;
  methodList.size = maxAuthorMethods;

  if ((result = openapiAuthorizationListFirstGet(clientHandle, type, &listName)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get first authorization list name for type: %d (result = %d) \n", type, result);
    free(methodStr);
    free(str);
    return; 
  }

  printf("Author_list_name   Methods\n");
  do
  {
    printf("%-16s   ", str);

    result = openapiAuthorizationMethodListGet(clientHandle, type, &listName, &methodList);
    if (result == OPEN_E_NONE)
    {
      for (i = 0; i < methodList.size; i++)
      {
        switch((methodStr[i])-'0')
        {
          case OPEN_AUTHOR_METHOD_TACACS:
            printf("Tacacs  ");
            break;
          case OPEN_AUTHOR_METHOD_RADIUS:
            printf("Radius  ");
            break;
          case OPEN_AUTHOR_METHOD_NONE:
            printf("None  ");
            break;
          case OPEN_AUTHOR_METHOD_LOCAL:
            printf("Local  ");
            break;
          default:
            break;
        }
      }
    }
    else
    {
      printf("Authorization Method get failed. (result = %d) \n", result);
    }
    printf("\n");
    listName.size = authorListNameSize;
    methodList.size = maxAuthorMethods;
  }while(openapiAuthorizationListNextGet(clientHandle, type, &listName, &listName) == OPEN_E_NONE);

  free(str);
  free(methodStr);
  return;
}

/*******************************************************************
* @purpose  Apply authorization list to access line for an authorization type.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    accessLine     @b{(input)} Access line
* @param    type           @b{(input)} Authorization type 
* @param    listName       @b{(input)} Authorization List name.
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void authorizationListLineApply(openapiClientHandle_t *clientHandle,  OPEN_ACCESS_LINE_t  accessLine, 
                                OPEN_USER_MGR_AUTHOR_TYPES_t type, char *listName)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, listName, (sizeof(str) - 1));
  buffDesc.pstart = str; 
  buffDesc.size = strlen(str)+1;

  if ((result = openapiAuthorizationListLineSet(clientHandle, accessLine, type, &buffDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to apply authorization list name to access line %d. (result = %d) \n", accessLine, result);
  }
  else
  {
    printf("Authorization list name applied to access line successfully \n");
  }

  return;
}

/*******************************************************************
* @purpose  Get the authorization list applied to access line for an
*           authorization type.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    accessLine     @b{(input)} Access line
* @param    type           @b{(input)} Authorization type 
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void authorizationListLineGet(openapiClientHandle_t *clientHandle, OPEN_ACCESS_LINE_t  accessLine,  OPEN_USER_MGR_AUTHOR_TYPES_t type)
{
  open_error_t result;
  char *str;
  open_buffdesc listName;
  uint32_t authorListNameSize;

  if ((result = openapiAuthorizationListNameSizeGet(clientHandle, &authorListNameSize)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get authorization list name size. (result = %d)", result);
    return;
  }
  if ((str = (char *)malloc(authorListNameSize)) == NULL)
  {
    printf("Could not allocate memory.\n");
    return;
  }

  memset(str, 0, authorListNameSize);
  listName.pstart = str;
  listName.size = authorListNameSize;

  printf(" Getting the authorization list name applied to access line %d\n", accessLine);
   if ((result = openapiAuthorizationListLineGet(clientHandle, accessLine, type, &listName)) != OPEN_E_NONE)
  {
    printf("Bad return code while getting authorization list name applied to access line. (result = %d) \n", result);
  }
  else
  {
    printf("Authorization list name: %s \n", str);
  }

  free(str);
  return;
}

/*******************************************************************
* @purpose  Delete authorization list applied to access line for an 
*           authorization type.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    accessLine     @b{(input)} Access line
* @param    type           @b{(input)} Authorization type 
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void authorizationListLineDelete(openapiClientHandle_t *clientHandle, OPEN_ACCESS_LINE_t  accessLine,  OPEN_USER_MGR_AUTHOR_TYPES_t type)
{
  open_error_t result;

  printf("Deleting authorization list applied to access line %d. \n", accessLine);

  if ((result = openapiAuthorizationListLineDelete(clientHandle, accessLine, type)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete authorization list name applied to access line. (result = %d) \n", result);
  }
  else
  {
    printf("Deleted the authorization list name from the access line %d. \n", accessLine);
  }
  return; 
}

/*******************************************************************
* @purpose  Delete authorization list of an authorization type.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    type           @b{(input)} Authorization type 
* @param    listName       @b{(input)} Authorization list name
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void authorizationListDelete(openapiClientHandle_t *clientHandle, OPEN_USER_MGR_AUTHOR_TYPES_t type, char *listName)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100];

  printf("Deleting the authorization list %s \n", listName);
  memset(str, 0, sizeof(str));
  strncpy(str, listName, (sizeof(str) - 1));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str)+1;

  if ((result = openapiAuthorizationListDelete(clientHandle, type, &buffDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete authorization list. (result = %d) \n", result);
  }
  else
  {
    printf("Authorization list %s deleted successfully. \n", (char *)buffDesc.pstart);
  }

  return;
}

/*******************************************************************
* @purpose  Authorization OpEN APIs Sanity
*
* @param    clientHandle   @b{(input)} client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void authorizationOpENAPIsSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t authorListNameSize = 0, maxAuthorMethods =0;
  OPEN_ACCESS_LINE_t  accessLine = OPEN_ACCESS_LINE_UNKNOWN;
  OPEN_USER_MGR_AUTHOR_TYPES_t type = OPEN_AUTHOR_TYPE_UNKNOWN;
  char *str, *str1;
  open_buffdesc buffDesc1;
  open_buffdesc buffDesc2;

  printf("\nTesting authorization OpEN APIs sanity\n");

  if ((result = openapiAuthorizationListNameSizeGet(clientHandle, &authorListNameSize)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get authorization name size. (result = %d) \n", result);
    return;
  }

  if ((result = openapiAuthorizationMethodsMaxGet(clientHandle, &maxAuthorMethods)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get maximum authorization methods supported. (result = %d) \n", result);
    return;
  }

  if ((str = (char *)malloc(authorListNameSize)) == NULL)
  {
    printf("Could not allocate memory.\n");
    return;
  }
  memset(str, 0, authorListNameSize);
  buffDesc1.pstart = str;
  buffDesc1.size = authorListNameSize;

  if ((str1 = (char *)malloc(maxAuthorMethods)) == NULL)
  {
    printf("Could not allocate memory.\n");
    free(str); 
    return;
  }

  memset(str1, 0, maxAuthorMethods);
  buffDesc2.pstart = str1;
  buffDesc2.size = maxAuthorMethods;

  /* openapiAuthorizationListNameSizeGet() */
  printf("\nTesting openapiAuthorizationListNameSizeGet(): \n");
  
  result = openapiAuthorizationListNameSizeGet(NULL, &authorListNameSize);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAuthorizationListNameSizeGet(clientHandle, NULL);
  printf("NULL string length. (result = %d)\n", result);

  result = openapiAuthorizationListCreate(NULL, type, &buffDesc1);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAuthorizationListCreate(clientHandle, 2, &buffDesc1);
  printf("Inavlid authorization type. (result = %d)\n", result);

  result = openapiAuthorizationListCreate(clientHandle, type, NULL);
  printf("NULL buf descriptor to authorization list name. (result = %d)\n", result);

  printf("openapiAuthorizationListCreate() sanity successful.\n");

  /* openapiAuthorizationMethodsAdd() */
  printf("\nTesting openapiAuthorizationMethodsAdd(): \n");

  result = openapiAuthorizationMethodsAdd(NULL, type, &buffDesc1, &buffDesc2);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAuthorizationMethodsAdd(clientHandle, 3, &buffDesc1, &buffDesc2);
  printf("Inavlid authorization type. (result = %d)\n", result);

  result = openapiAuthorizationMethodsAdd(clientHandle, type, NULL, &buffDesc2);
  printf("NULL buf descriptor to authorization list name. (result = %d)\n", result);

  result = openapiAuthorizationMethodsAdd(clientHandle, type, &buffDesc1, NULL);
  printf("NULL buf descriptor to authorization method list. (result = %d)\n", result);

  printf("openapiAuthorizationMethodsAdd() sanity successful.\n");

  /* openapiAuthorizationListDelete() */
  printf("\nTesting openapiAuthorizationListDelete(): \n");

  result = openapiAuthorizationListDelete(NULL, type, &buffDesc1);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAuthorizationListDelete(clientHandle, 4, &buffDesc1);
  printf("Invalid authorization type. (result = %d)\n", result);

  result = openapiAuthorizationListDelete(clientHandle, type, NULL);
  printf("NULL buf descriptor to authorization list name. (result = %d)\n", result);

  printf("openapiAuthorizationListDelete() sanity successful.\n");

  /* openapiAuthorizationListFirstGet() */
  printf("\nTesting openapiAuthorizationListFirstGet(): \n");

  result = openapiAuthorizationListFirstGet(NULL, type, &buffDesc1);
  printf("NULL Client Handle. (result = %d)\n", result);
 
  result = openapiAuthorizationListFirstGet(clientHandle, 5, &buffDesc1);
  printf("Inavlid authorization type. (result = %d)\n", result);

  result = openapiAuthorizationListFirstGet(clientHandle, type, NULL);
  printf("NULL buf descriptor to authorization list name. (result = %d)\n", result);

  printf("openapiAuthorizationListFirstGet() sanity successful.\n");

  /* openapiAuthorizationListNextGet() */
  printf("\nTesting openapiAuthorizationListNextGet(): \n");

  result = openapiAuthorizationListNextGet(NULL, type, &buffDesc1, &buffDesc1);
  printf("NULL Client Handle. (result = %d)\n", result);
  
  result = openapiAuthorizationListNextGet(clientHandle, 3, &buffDesc1, &buffDesc1);
  printf("Inavlid authorization type. (result = %d)\n", result);

  result = openapiAuthorizationListNextGet(clientHandle, type, NULL, &buffDesc1);
  printf("NULL buf descriptor to authorization list name. (result = %d)\n", result);

  result = openapiAuthorizationListNextGet(clientHandle, type, &buffDesc1, NULL);
  printf("NULL buf descriptor to authorization list name. (result = %d)\n", result);

  printf("openapiAuthorizationListNextGet() sanity successful.\n");

  /* openapiAuthorizationMethodListGet() */
  printf("\nTesting openapiAuthorizationMethodListGet(): \n");

  result = openapiAuthorizationMethodListGet(NULL, type, &buffDesc1, &buffDesc2);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAuthorizationMethodListGet(clientHandle, 5, &buffDesc1, &buffDesc2);
  printf("Inavlid authorization type. (result = %d)\n", result);

  result = openapiAuthorizationMethodListGet(clientHandle, type, NULL, &buffDesc2);
  printf("NULL buf descriptor to authorization list name. (result = %d)\n", result);

  result = openapiAuthorizationMethodListGet(clientHandle, type, &buffDesc1, NULL);
  printf("NULL buf descriptor to authorization method list. (result = %d)\n", result);

  printf("openapiAuthorizationMethodListGet() sanity successful.\n");

  /* openapiAuthorizationListLineSet() */
  printf("\nTesting openapiAuthorizationListLineSet(): \n");

  result = openapiAuthorizationListLineSet(NULL, accessLine, type, &buffDesc1);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAuthorizationListLineSet(clientHandle,6 , type, &buffDesc1);
  printf("Inavlid access line. (result = %d)\n", result);

  result = openapiAuthorizationListLineSet(clientHandle, accessLine, 5, &buffDesc1);
  printf("Inavlid authorization type. (result = %d)\n", result);

  result = openapiAuthorizationListLineSet(clientHandle, accessLine, type, NULL);
  printf("NULL buf descriptor to authorization list name. (result = %d)\n", result);

  printf("openapiAuthorizationListLineSet() sanity successful.\n");

  /* openapiAuthorizationListLineGet() */
  printf("\nTesting openapiAuthorizationListLineGet(): \n");

  result = openapiAuthorizationListLineGet(NULL, accessLine, type, &buffDesc1);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAuthorizationListLineGet(clientHandle, 6, type, &buffDesc1);
  printf("Inavlid access line. (result = %d)\n", result);

  result = openapiAuthorizationListLineGet(clientHandle, accessLine, 5, &buffDesc1);
  printf("Inavlid authorization type. (result = %d)\n", result);

  result = openapiAuthorizationListLineGet(clientHandle, accessLine, type, NULL);
  printf("NULL buf descriptor to authorization list name. (result = %d)\n", result);

  printf("openapiAuthorizationListLineGet() sanity successful.\n");

  /* openapiAuthorizationListLineDelete() */
  printf("\nTesting openapiAuthorizationListLineDelete(): \n");

  result = openapiAuthorizationListLineDelete(NULL, accessLine, type); 
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAuthorizationListLineDelete(clientHandle, 5, type); 
  printf("Inavlid access line. (result = %d)\n", result);
  
  result = openapiAuthorizationListLineDelete(clientHandle, accessLine, 6); 
  printf("Inavlid authorization type. (result = %d)\n", result);

  printf("openapiAuthorizationListLineDelete() sanity successful.\n");

  free(str);
  free(str1);
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum, arg1, arg2;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  open_buffdesc authorMethods;
  char str[100];
  uint32_t i = 0;

  if (argc < 2)
  {
    printAuthorizationAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("authorization_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Authorization API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");
  /* exercise various OPEN API Authorization functions */

  switch (testNum)
  {
    case 1:
      if (argc != 4) 
      {
        printAuthorizationAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      authorizationListCreate(&clientHandle, arg1, argv[3]);
      break;
    case 2:
      if (argc < 5)
      {
        printAuthorizationAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      memset(str, 0, sizeof(str));
      for (i = 4; i < argc; i++)
      {
        strncat(str, argv[i], strlen(argv[i]));
      }
      authorMethods.pstart = str;
      authorMethods.size = strlen(str)+1;

      authorizationMethodsAdd(&clientHandle, arg1, argv[3], &authorMethods);
      break;
    case 3:
      if (argc != 3)
      {
        printAuthorizationAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      authorizationInfoGet(&clientHandle, arg1);
      break;
    case 4:
      if (argc != 5)
      {
        printAuthorizationAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      authorizationListLineApply(&clientHandle, arg1, arg2, argv[4]);
      break;
    case 5:
      if (argc != 4)
      {
        printAuthorizationAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      authorizationListLineGet(&clientHandle, arg1, arg2);
      break;
    case 6:
      if (argc != 4)
      {
        printAuthorizationAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      authorizationListLineDelete(&clientHandle, arg1, arg2);
      break;
    case 7:
      if (argc != 4)
      {
        printAuthorizationAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      authorizationListDelete(&clientHandle, arg1, argv[3]);
      break;
    case 8:
      if (argc != 2)
      {
        printAuthorizationAppMenu();
        exit(1);
      }
      authorizationOpENAPIsSanity(&clientHandle);
      break;
    default:
      printAuthorizationAppMenu();
      break;
  }

  /* Log goodbye message with OPEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Authorization API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
