/*********************************************************************
*
* Copyright 2018-2020 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_dhcp_server.h
*
* @purpose   DHCP Server API
*
* @component OpEN
*
* @comments  This code must not include any ICOS header files.
*
* @create    7/6/2018
*
* @end
*                                    
**********************************************************************/
#ifndef OPENAPI_DHCP_SERVER_H_INCLUDED
#define OPENAPI_DHCP_SERVER_H_INCLUDED

#include "openapi_common.h"

/** Pool types */
typedef enum
{
  OPEN_DHCP_SERVER_POOL_TYPE_INACTIVE      = 0, /**< inactive */
  OPEN_DHCP_SERVER_POOL_TYPE_DYNAMIC       = 1, /**< dynamic */
  OPEN_DHCP_SERVER_POOL_TYPE_MANUAL        = 2  /**< manual */
} OPEN_DHCP_SERVER_POOL_TYPE_t;

/** DHCP server lease states */
typedef enum
{
  OPEN_DHCP_SERVER_FREE_LEASE       = 0, /**< Free */
  OPEN_DHCP_SERVER_ACTIVE_LEASE     = 1, /**< Active */
  OPEN_DHCP_SERVER_OFFERED_LEASE    = 2, /**< Offered */
  OPEN_DHCP_SERVER_EXPIRED_LEASE    = 3, /**< Expired */
  OPEN_DHCP_SERVER_ABANDONED_LEASE  = 4, /**< Abandoned */
} OPEN_DHCP_SERVER_LEASE_STATE_t;

#define OPEN_DHCP_SERVER_POOL_NAME_MAXLEN    32  /** Maximum pool name, includes terminating '\0' */
#define OPEN_DHCP_SERVER_CLIENT_ID_MAXLEN    256 /** Maximum client ID, includes terminating '\0' */
#define OPEN_DHCP_SERVER_POOL_VRF_NAME_MAXLEN 65 /** Maximum VRF name, includes terminating '\0' */

/** DHCP Pool name */
typedef struct
{
  char name[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];  /** DHCP server pool name */
} OPEN_DHCP_SERVER_POOL_NAME_t;

/** DHCP Pool's VRF-name */
typedef struct
{
  char name[OPEN_DHCP_SERVER_POOL_VRF_NAME_MAXLEN];  /** DHCP server pool's VRF name */
} OPEN_DHCP_SERVER_POOL_VRF_NAME_t;


/** DHCP client ID */
typedef struct
{
  char clientid[OPEN_DHCP_SERVER_CLIENT_ID_MAXLEN];  /** DHCP server client id  */
} OPEN_DHCP_SERVER_CLIENT_ID_t;

/** DHCP Server Statistic IDs */
typedef enum
{
  OPEN_DHCPS_STAT_ID_RESERVED          = 0,     /**< Reserved (unused).              */
  OPEN_DHCPS_STAT_ID_ACTIVE_LEASES     = 1,     /**< Active leases.                  */
  OPEN_DHCPS_STAT_ID_EXPIRED_LEASES    = 2,     /**< Expired leases.                 */
  OPEN_DHCPS_STAT_ID_MALFORMED_MSGS    = 3,     /**< Malformed messages.             */
  OPEN_DHCPS_STAT_ID_DISCOVER_DROPPED  = 4,     /**< Discover messages dropped.      */
  OPEN_DHCPS_STAT_ID_RX_DISCOVER       = 5,     /**< Discover messages received.     */
  OPEN_DHCPS_STAT_ID_RX_REQUEST        = 6,     /**< Request messages received.      */
  OPEN_DHCPS_STAT_ID_RX_DECLINE        = 7,     /**< Decline messages received.      */
  OPEN_DHCPS_STAT_ID_RX_RELEASE        = 8,     /**< Release messages received.      */
  OPEN_DHCPS_STAT_ID_RX_INFORM         = 9,     /**< Inform messages received.       */
  OPEN_DHCPS_STAT_ID_TX_OFFER          = 10,    /**< Offer messages sent.            */
  OPEN_DHCPS_STAT_ID_TX_ACK            = 11,    /**< Ack messages sent.              */
  OPEN_DHCPS_STAT_ID_TX_NACK           = 12,    /**< Nack messages sent.             */
  OPEN_DHCPS_STAT_ID_LAST                       /**< Must be last (bounds checking). */
} OPEN_DHCPS_STAT_ID_t;

/*********************************************************************
* @purpose  Get the DHCP Server Lease Data 
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    ipaddr           @b{(input)}  IP address assigned to lease
* @param    mask             @b{(output)} IP Mask
* @param    hwaddr           @b{(output)} Hardware address
* @param    clientId         @b{(output)} Client ID
* @param    poolName         @b{(output)} Pool name
* @param    remaining        @b{(output)} Remaining time in seconds
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes Hardware address is returned as a 6 byte raw hex value, and
*        must be of size @ref OPEN_MAC_ADDR_LEN
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerLeaseDataGet(openapiClientHandle_t *client_handle,
                                           open_inet_addr_t *ipaddr,
                                           uint32_t *mask,
                                           open_buffdesc *hwaddr,
                                           OPEN_DHCP_SERVER_CLIENT_ID_t *clientId,
                                           OPEN_DHCP_SERVER_POOL_NAME_t *poolName,
                                           uint32_t *remaining);

/*********************************************************************
* @purpose  Get the first DHCP Server Pool Entry 
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    poolIndex        @b{(output)} Pool index
* @param    poolName         @b{(output)} Pool name
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_NOT_FOUND if no entry found
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes None
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerPoolEntryFirstGet(openapiClientHandle_t *client_handle,
                                                uint32_t *poolIndex,
                                                OPEN_DHCP_SERVER_POOL_NAME_t *poolName);

/*********************************************************************
* @purpose  Get the next DHCP Server Pool Entry 
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    poolIndex        @b{(inout)}  Pool index
* @param    poolName         @b{(output)} Pool name
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if failure
* @returns  OPEN_E_NOT_FOUND if no entry found
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes None
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerPoolEntryNextGet(openapiClientHandle_t *client_handle,
                                               uint32_t *poolIndex,
                                               OPEN_DHCP_SERVER_POOL_NAME_t *poolName);


/*********************************************************************
* @purpose  Get the  DHCP Server Pool Type
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    poolName         @b{(input)}  Pool name
* @param    type             @b{(output)} Pool type
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if failure
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes None
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerPoolTypeGet(openapiClientHandle_t *client_handle,
                                          OPEN_DHCP_SERVER_POOL_NAME_t *poolName,
                                          OPEN_DHCP_SERVER_POOL_TYPE_t *type);


/*********************************************************************
* @purpose  Get the first DHCP Server Lease Entry
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    poolName         @b{(input)}  Pool name
* @param    leaseIndex       @b{(output)} Lease index
* @param    ipaddr           @b{(output)} Lease IP address
* @param    state            @b{(output)} State of the lease
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if failure or no entry found found.
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes None
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerLeaseEntryFirstGet(openapiClientHandle_t *client_handle,
                                                 OPEN_DHCP_SERVER_POOL_NAME_t *poolName,
                                                 uint32_t *leaseIndex,
                                                 open_inet_addr_t *ipaddr,
                                                 OPEN_DHCP_SERVER_LEASE_STATE_t *state);

/*********************************************************************
* @purpose  Get the next DHCP Server Lease Entry
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    poolName         @b{(input)}  Pool name
* @param    leaseIndex       @b{(inout)}  Lease index
* @param    ipaddr           @b{(output)} Lease IP address
* @param    state            @b{(output)} State of the lease
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if failure
* @returns  OPEN_E_NOT_FOUND if no entry found
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes None
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerLeaseEntryNextGet(openapiClientHandle_t *client_handle,
                                                OPEN_DHCP_SERVER_POOL_NAME_t *poolName,
                                                uint32_t *leaseIndex,
                                                open_inet_addr_t *ipaddr,
                                                OPEN_DHCP_SERVER_LEASE_STATE_t *state);

/*********************************************************************
* @purpose  Get the specified DHCP Server statistic
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    id               @b{(input)}  Statistic identifier
* @param    val              @b{(output)} Statistic value
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if failure
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerStatGet(openapiClientHandle_t *client_handle, 
                                      OPEN_DHCPS_STAT_ID_t id, uint32_t *val);

/*********************************************************************
* @purpose  Get the name of the VRF instance of a given DHCP server pool
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    poolName         @b{(input)}  Pool name
* @param    vrfName          @b{(output)} Name of VRF-instance of the pool
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if failure
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes None
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerPoolVrfNameGet(openapiClientHandle_t *client_handle,
                                             OPEN_DHCP_SERVER_POOL_NAME_t *poolName,
                                             OPEN_DHCP_SERVER_POOL_VRF_NAME_t *vrfName);

/*********************************************************************
* @purpose  Set the name of the VRF instance of a given DHCP server pool
*
* @param    client_handle    @b{(input)} Client handle from
*                                        registration API
* @param    poolName         @b{(input)} Pool name
* @param    vrfName          @b{(input)} Name of VRF-instance of the pool
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if failure
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes None
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerPoolVrfNameSet(openapiClientHandle_t *client_handle,
                                             OPEN_DHCP_SERVER_POOL_NAME_t *poolName,
                                             OPEN_DHCP_SERVER_POOL_VRF_NAME_t *vrfName);

/*********************************************************************
* @purpose  Set the name of the VRF instance of a given DHCP server pool
*
* @param    client_handle    @b{(input)} Client handle from
*                                        registration API
* @param    poolName         @b{(input)} Pool name
* @param    vrfName          @b{(input)} Name of VRF-instance of the pool
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if failure
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes None
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerPoolVrfNameSetToDefaultVrf(openapiClientHandle_t *client_handle,
                                                         OPEN_DHCP_SERVER_POOL_NAME_t *poolName);

/*********************************************************************
* @purpose  Get the first DHCP server excluded IP address range entry
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    pRangeIndex      @b{(input)}  Index to get first entry
* @param    vrfName          @b{(output)} Name of VRF instance in which IP
                                          addresses are configured to be excluded
* @param    startIpAddr      @b{(output)} Starting IP address of range
* @param    endIpAddr        @b{(output)} Ending IP address of range
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if failure or no entry found found.
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes None
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerExcludedAddressRangeEntryFirstGet(openapiClientHandle_t *client_handle,
                                                                uint32_t *pRangeIndex,
                                                                OPEN_DHCP_SERVER_POOL_VRF_NAME_t *vrfName,
                                                                open_inet_addr_t *startIpAddr,
                                                                open_inet_addr_t *endIpAddr);

/*********************************************************************
* @purpose  Get the next DHCP server excluded IP address range entry
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    pRangeIndex      @b{(input)}  Index to get next entry
* @param    vrfName          @b{(output)} Name of VRF instance in which IP
                                          addresses are configured to be excluded
* @param    startIpAddr      @b{(output)} Starting IP address of range
* @param    endIpAddr        @b{(output)} Ending IP address of range
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if failure
* @returns  OPEN_E_NOT_FOUND if no entry found
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes None
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerExcludedAddressRangeEntryNextGet(openapiClientHandle_t *client_handle,
                                                               uint32_t *pRangeIndex,
                                                               OPEN_DHCP_SERVER_POOL_VRF_NAME_t *vrfName,
                                                               open_inet_addr_t *startIpAddr,
                                                               open_inet_addr_t *endIpAddr);

/*********************************************************************
* @purpose  Add an excluded-address range for a given VRF instance
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    vrfName          @b{(input)}  VRF instance name
* @param    lowIpAddr        @b{(input)}  Start IPv4 address of the address range
* @param    highIpAddr       @b{(input)}  End IPv4 address of the address range
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if add operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes 
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerExcludedAddressRangeAdd(openapiClientHandle_t *client_handle,
                                                      OPEN_DHCP_SERVER_POOL_VRF_NAME_t *vrfName,
                                                      open_inet_addr_t *lowIpAddr,
                                                      open_inet_addr_t *highIpAddr);

/*********************************************************************
* @purpose  Delete an excluded-address range for a given VRF instance
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    vrfName          @b{(input)}  VRF instance name
* @param    lowIpAddr        @b{(input)}  Start IPv4 address of the address range
* @param    highIpAddr       @b{(input)}  End IPv4 address of the address range
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if delete operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes 
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerExcludedAddressRangeDelete(openapiClientHandle_t *client_handle,
                                                         OPEN_DHCP_SERVER_POOL_VRF_NAME_t *vrfName,
                                                         open_inet_addr_t *lowIpAddr,
                                                         open_inet_addr_t *highIpAddr);

/*********************************************************************
* @purpose  Get the DHCP server lease data for a given VRF name and IP address 
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    vrfName          @b{(input)}  VRF instance name
* @param    ipaddr           @b{(input)}  IP address assigned to lease
* @param    mask             @b{(output)} IP Mask
* @param    hwaddr           @b{(output)} Hardware address
* @param    clientId         @b{(output)} Client ID
* @param    poolName         @b{(output)} Pool name
* @param    remaining        @b{(output)} Remaining time in seconds
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes Hardware address is returned as a 6 byte raw hex value, and
*        must be of size @ref OPEN_MAC_ADDR_LEN
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerVrfIpAddrLeaseDataGet(openapiClientHandle_t *client_handle,
                                                    OPEN_DHCP_SERVER_POOL_VRF_NAME_t *vrfName,
                                                    open_inet_addr_t *ipaddr,
                                                    uint32_t *mask,
                                                    open_buffdesc *hwaddr,
                                                    OPEN_DHCP_SERVER_CLIENT_ID_t *clientId,
                                                    OPEN_DHCP_SERVER_POOL_NAME_t *poolName,
                                                    uint32_t *remaining);

/*********************************************************************
* @purpose  Clear lease data for VRF name and IP address 
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    vrfName          @b{(input)}  VRF instance name
* @param    ipaddr           @b{(input)}  IP address assigned to lease
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes 
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerVrfIpAddrLeaseDataClear(openapiClientHandle_t *client_handle,
                                                      OPEN_DHCP_SERVER_POOL_VRF_NAME_t *vrfName,
                                                      open_inet_addr_t *ipaddr);

/*********************************************************************
* @purpose  Clear lease data for pool name and IP address 
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    poolName         @b{(input)}  Pool name
* @param    ipaddr           @b{(input)}  IP address assigned to lease
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes 
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerPoolIpAddrLeaseDataClear(openapiClientHandle_t *client_handle,
                                                       OPEN_DHCP_SERVER_POOL_NAME_t *poolName,
                                                       open_inet_addr_t *ipaddr);

/*********************************************************************
* @purpose  Clear all lease data of a VRF instance
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    vrfName          @b{(input)}  VRF instance name
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes 
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerVrfLeaseDataClear(openapiClientHandle_t *client_handle,
                                                OPEN_DHCP_SERVER_POOL_VRF_NAME_t *vrfName);

/*********************************************************************
* @purpose  Clear all lease data of a pool 
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    poolName         @b{(input)}  Pool name
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes 
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerPoolLeaseDataClear(openapiClientHandle_t *client_handle,
                                                 OPEN_DHCP_SERVER_POOL_NAME_t *poolName);

/*********************************************************************
* @purpose  Clear all lease data across all VRFs  
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes 
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerAllLeaseDataClear(openapiClientHandle_t *client_handle);

#endif /* OPENAPI_DHCP_SERVER_H_INCLUDED */
