/**********************************************************************
*
*  Copyright 2020 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_dot1as.h
*
* @purpose   This code implements the OpEN 801AS (dot1as) API(s)
*
* @component OPEN
*
* @create    04/08/2020
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_DOT1AS_H_INCLUDED
#define OPENAPI_DOT1AS_H_INCLUDED

#include "openapi_common.h" 

/**< Clock priority range */
#define OPEN_DOT1AS_MIN_PRIORITY_VALUE 0
#define OPEN_DOT1AS_MAX_PRIORITY_VALUE 255

/**< 802.1AS propagation delay threshold valid range in nanoseconds */
#define OPEN_DOT1AS_MIN_PDELAY_THRESHOLD  0           /**< nanoseconds */
#define OPEN_DOT1AS_MAX_PDELAY_THRESHOLD  1000000000  /**< nanoseconds */

/**< Minimum and maximum limits for number of PDELAY lost responses allowed */
#define OPEN_DOT1AS_MIN_ALLOWED_RESP_LOST 0
#define OPEN_DOT1AS_MAX_ALLOWED_RESP_LOST 65535

/**< 802.1AS transmit interval limits in millisec and log base 2.
   Interval is specified in logarithm base 2 format.
* Note: Although standard supports a large range of values, only a subset
* of values are supported.
*/
#define OPEN_DOT1AS_PDU_TX_INTL_LOG2_MAX    5       /**< 127 as per standard */
#define OPEN_DOT1AS_PDU_TX_INTL_LOG2_MIN   -5       /**< -128 as per standard */

/**< 802.1AS receipt timeout limits in number of intervals */
#define OPEN_DOT1AS_MIN_PDU_RX_TIMEOUT      2
#define OPEN_DOT1AS_MAX_PDU_RX_TIMEOUT      255

/**< Clock Identity */
#define OPEN_DOT1AS_CLOCK_ID_LEN   8 /**< Length of clock identify - 8 octets */
typedef struct
{
  uint8_t id[OPEN_DOT1AS_CLOCK_ID_LEN];
} OPEN_DOT1AS_CLOCK_IDENTITY_t;

/**< 802.1AS Priority types */
typedef enum
{
  OPEN_DOT1AS_CLOCK_PRIORITY1 = 0, /**< priority 1 */
  OPEN_DOT1AS_CLOCK_PRIORITY2      /**< priority 2 */
} OPEN_DOT1AS_CLOCK_PRIORITY_t;

/**< 802.1AS Port Roles */
typedef enum
{
  OPEN_DOT1AS_ROLE_DISABLED = 0,   /**< Disabled */
  OPEN_DOT1AS_ROLE_MASTER,         /**< Master */
  OPEN_DOT1AS_ROLE_SLAVE,          /**< Slave */
  OPEN_DOT1AS_ROLE_PASSIVE,        /**< Passive */
} OPEN_DOT1AS_PORT_ROLE_t;

/**< 802.1AS transmit and receive intervals in ms */
typedef enum
{
  OPEN_DOT1AS_INITIAL_SYNC_TX_INTERVAL = 0,
  OPEN_DOT1AS_CURRENT_SYNC_TX_INTERVAL,
  OPEN_DOT1AS_INITIAL_ANNOUNCE_TX_INTERVAL,
  OPEN_DOT1AS_CURRENT_ANNOUNCE_TX_INTERVAL,
  OPEN_DOT1AS_INITIAL_PDELAY_TX_INTERVAL,
  OPEN_DOT1AS_CURRENT_PDELAY_TX_INTERVAL,
  OPEN_DOT1AS_SYNC_RX_TIMEOUT,
  OPEN_DOT1AS_ANNOUNCE_RX_TIMEOUT,
  OPEN_DOT1AS_PDELAY_RX_TIMEOUT,
  OPEN_DOT1AS_SYNC_RX_TIMEOUT_INTERVAL,
} OPEN_DOT1AS_INTF_INTERVAL_t;

/**< Interface counter types */
typedef enum
{
  OPEN_DOT1AS_SYNC_TX_COUNT = 1,
  OPEN_DOT1AS_SYNC_RX_COUNT,
  OPEN_DOT1AS_FOLLOWUP_TX_COUNT,
  OPEN_DOT1AS_FOLLOWUP_RX_COUNT,
  OPEN_DOT1AS_ANNOUNCE_TX_COUNT,
  OPEN_DOT1AS_ANNOUNCE_RX_COUNT,
  OPEN_DOT1AS_PDELAY_REQ_TX_COUNT,
  OPEN_DOT1AS_PDELAY_REQ_RX_COUNT,
  OPEN_DOT1AS_PDELAY_RESP_TX_COUNT,
  OPEN_DOT1AS_PDELAY_RESP_RX_COUNT,
  OPEN_DOT1AS_PDELAY_RESP_FOLLOWUP_TX_COUNT,
  OPEN_DOT1AS_PDELAY_RESP_FOLLOWUP_RX_COUNT,
  OPEN_DOT1AS_SIGNALING_TX_COUNT,
  OPEN_DOT1AS_SIGNALING_RX_COUNT,
  OPEN_DOT1AS_SYNC_RX_TIMEOUTS,
  OPEN_DOT1AS_SYNC_RX_DISCARDS,
  OPEN_DOT1AS_ANNOUNCE_RX_TIMEOUTS,
  OPEN_DOT1AS_ANNOUNCE_RX_DISCARDS,
  OPEN_DOT1AS_PDELAY_RX_TIMEOUTS,
  OPEN_DOT1AS_PDELAY_RX_DISCARDS,
  OPEN_DOT1AS_BAD_HEADER_COUNT,
  OPEN_DOT1AS_PTP_DISCARD_COUNT,
  OPEN_DOT1AS_PDELAY_LOST_RESP_EXCEEDED,
  OPEN_DOT1AS_COUNTER_LAST

} OPEN_DOT1AS_COUNTER_TYPE_t;

/**< DOT1AS Grand master event types */
typedef enum
{
  OPEN_DOT1AS_GM_CHANGE_EVENT,
  OPEN_DOT1AS_GM_PHASE_CHANGE_EVENT,
  OPEN_DOT1AS_GM_FREQ_CHANGE_EVENT,
} OPEN_DOT1AS_GM_EVENT_TYPE_t;

/*****************************************************************************
 * @purpose   Get the 802.1AS global admin mode.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     adminMode           @b{(output)} Global 802.1as admin mode.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     802.1AS  can be enabled at global level and interface level. 
 *            This API gets the global admin mode status
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asModeGet(openapiClientHandle_t *client_handle, 
                                  OPEN_CONTROL_t *adminMode);

/*****************************************************************************
 * @purpose   Set the 802.1AS global admin mode.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     adminMode           @b{(input)}  Global 802.1as admin mode.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     802.1AS  can be enabled at global level and interface level. 
 *            This API sets the global admin mode.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asModeSet(openapiClientHandle_t *client_handle, 
                                  OPEN_CONTROL_t  adminMode);

/*****************************************************************************
 * @purpose   Check if there is a Grand Master elected in the 802.1AS domain.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     grandmasterPresent  @b{(output)} Grand master present status.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asIsGrandmasterPresentGet(openapiClientHandle_t *client_handle, OPEN_BOOL_t *grandmasterPresent);

/*****************************************************************************
 * @purpose   Check if this bridge is capable of being a grandmaster.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     grandmasterCapable  @b{(output)} Grand master capability status.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asGrandmasterCapableGet(openapiClientHandle_t *client_handle,                                                OPEN_BOOL_t *grandmasterCapable);

/*****************************************************************************
 * @purpose   Get the Best clock identity as detected by this 801AS protocol.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     bestClockId         @b{(output)} Best clock id as detected by 801AS protocol.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asBestClkIdGet(openapiClientHandle_t *client_handle, 
                                       OPEN_DOT1AS_CLOCK_IDENTITY_t *bestClockId);

/*****************************************************************************
 * @purpose   Get the priority 1/2 value of Best clock in the 802.1AS domain.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     clockPriorityType   @b{(input)}  Type of clock prority requested, Priority 1 or Priority2.
 * @param     clockPriority       @b{(output)} Clock priority of type clockPriorityType.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asBestClkPriorityGet(openapiClientHandle_t *client_handle, 
                                             OPEN_DOT1AS_CLOCK_PRIORITY_t clockPriorityType, 
					     uint8_t *clockPriority);

/*****************************************************************************
 * @purpose   Get the steps to Best clock as detected by this protocol.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     bestClkSteps        @b{(output)} Number of steps to the best clock.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asBestClkStepsGet(openapiClientHandle_t *client_handle, 
                                          uint16_t *bestClkSteps);

/*****************************************************************************
 * @purpose   Get the clock identity of this bridge.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     bridgeClkId         @b{(output)} Clock identity of this bridge.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asBridgeClkIdGet(openapiClientHandle_t *client_handle, 
                                         OPEN_DOT1AS_CLOCK_IDENTITY_t *bridgeClkId);

/*****************************************************************************
 * @purpose   Get the priority1/2 value of the local clock.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     clockPriorityType   @b{(input)}  Type of clock prority requested, 
 *                                             Priority 1 or Priority2.
 * @param     clockPriority       @b{(output)} Clock priority of value. (0 - 255)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asBridgePriorityGet(openapiClientHandle_t *client_handle, 
                                            OPEN_DOT1AS_CLOCK_PRIORITY_t clockPriorityType, 
					    uint32_t *clockPriority);

/*****************************************************************************
 * @purpose   Set the priority1/2 value of the local clock.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     clockPriorityType   @b{(input)}  Type of clock prority to be set, 
 *                                             Priority 1 or Priority2.
 * @param     clockPriority       @b{(input)}  Clock priority value. (0-255) 
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asBridgePrioritySet(openapiClientHandle_t *client_handle, 
                                            OPEN_DOT1AS_CLOCK_PRIORITY_t clockPriorityType, 
					    uint32_t  clockPriority);

/*****************************************************************************
 * @purpose   Get the number of times the grandmaster has changed.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     gmChangeCount       @b{(output)} count of, the number of times 
 *                                             grand master has changed.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asGrandmasterChangeCountGet(openapiClientHandle_t *client_handle, uint32_t *gmChangeCount);

/*****************************************************************************
 * @purpose   Get the timestamp of the given event.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     gmEventType         @b{(input)}  Event type like GM change, 
 *                                             phase change, freq change.
 * @param     timeStamp           @b{(output)} System time when the last 
 *                                             requested event occured.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     DOT1AS events like GM change, phase change, freq change.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asEventTimestampGet(openapiClientHandle_t *client_handle, 
                                            OPEN_DOT1AS_GM_EVENT_TYPE_t gmEventType, 
					    uint32_t *timeStamp);

/*****************************************************************************
 * @purpose   Get the 802.1AS admin mode for the given interface.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intfNum             @b{(input)}  Interface number.
 * @param     adminMode           @b{(output)} Interface 802.1as admin mode.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     802.1AS  can be enabled at global level and interface level. 
 *            This API gets the interface level mode status.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asIntfModeGet(openapiClientHandle_t *client_handle, 
                                      uint32_t  intfNum, 
                                      OPEN_CONTROL_t *adminMode);

/*****************************************************************************
 * @purpose   Set the 802.1AS admin mode for the given interface.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intfNum             @b{(input)}  Interface number.
 * @param     adminMode           @b{(output)} Interface 802.1as admin mode.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     802.1AS  can be enabled at global level and interface level. 
 *            This API sets the interface level mode status.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asIntfModeSet(openapiClientHandle_t *client_handle, 
                                      uint32_t  intfNum, 
                                      OPEN_CONTROL_t  adminMode);

/*****************************************************************************
 * @purpose   Check if the given interface 802.1AS capable 
 *            (as determined by the peer delay mechanism.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intfNum             @b{(input)}  Interface number.
 * @param     isCapable           @b{(output)} 802.1as capability of interface.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asIntfIsCapableGet(openapiClientHandle_t *client_handle, 
                                           uint32_t  intfNum, 
                                           OPEN_BOOL_t *isCapable);

/*****************************************************************************
 * @purpose   This "measuring peer delay" property, which is set to TRUE if 
 *            the given interface is receiving PDELAY response messages 
 *            from other end of the link.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intfNum             @b{(input)}  Interface number.
 * @param     isMeasuringPdelay   @b{(output)} Peer delay measuring status.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asIntfIsMeasuringPdelayGet(openapiClientHandle_t *client_handle, 
                                                   uint32_t  intfNum, 
						   OPEN_BOOL_t *isMeasuringPdelay);

/*****************************************************************************
 * @purpose   Get the propagation delay on the given interface.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intfNum             @b{(input)}  Interface number.
 * @param     pDelay              @b{(output)} Propogation delay on the interface.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asIntfPdelayGet(openapiClientHandle_t *client_handle, 
                                        uint32_t  intfNum, uint32_t *pDelay);

/*****************************************************************************
 * @purpose   Get the 802.1AS role of the given interface.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intfNum             @b{(input)}  Interface number.
 * @param     intfRole            @b{(output)} Interface role in the 802.1AS protocol.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asIntfRoleGet(openapiClientHandle_t *client_handle, 
                                      uint32_t  intfNum, 
				      OPEN_DOT1AS_PORT_ROLE_t *intfRole);

/*****************************************************************************
 * @purpose   Get the propagation delay threshold on the given interface.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intfNum             @b{(input)}  Interface number.
 * @param     threshold           @b{(output)} Propogation delay threshold on the interface.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asIntfPdelayThresholdGet(openapiClientHandle_t *client_handle, uint32_t  intfNum, uint32_t *threshold);

/*****************************************************************************
 * @purpose   Set the propagation delay threshold on the given interface.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intfNum             @b{(input)}  Interface number.
 * @param     threshold           @b{(input)}  Propogation delay threshold on the interface.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asIntfPdelayThresholdSet(openapiClientHandle_t *client_handle, uint32_t  intfNum, uint32_t  threshold);

/*****************************************************************************
 * @purpose   Get the number of Pdelay_Req messages for which a valid response 
 *            is not received, on which a port is considered to not be 
 *            exchanging peer delay messages with its neighbor.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intfNum             @b{(input)}  Interface number.
 * @param     numAllowedLostResp  @b{(output)} Number of allowed lost responses.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asIntfAllowedLostRespGet(openapiClientHandle_t *client_handle, uint32_t  intfNum, uint32_t *numAllowedLostResp);

/*****************************************************************************
 * @purpose   Set the number of Pdelay_Req messages for which a valid response 
 *            is not received, on which a port is considered to not be 
 *            exchanging peer delay messages with its neighbor.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intfNum             @b{(input)}  Interface number.
 * @param     numAllowedLostResp  @b{(input)}  Number of allowed lost responses.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asIntfAllowedLostRespSet(openapiClientHandle_t *client_handle, uint32_t  intfNum, uint32_t  numAllowedLostResp);

/*****************************************************************************
 * @purpose   Get the neighbor rate ratio on the given interface 
 *            (as computed by the pdelay mechanism) (extensible version).
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intfNum             @b{(input)}  Interface number.
 * @param     rateRatio           @b{(output)} The current value of the rate ratio.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asIntfNeighborRateRatioGet(openapiClientHandle_t *client_handle, 
                                                   uint32_t  intfNum, 
						   uint32_t *rateRatio);

/*****************************************************************************
 * @purpose   Get the given interval on the given interface.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intfNum             @b{(input)}  Interface number.
 * @param     intervalType        @b{(input)}  DOT1AS interval type.
 * @param     interval            @b{(output)} The interface interval/timeout value.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     In 802.1AS protocol interfaces have transmit and receive 
 *            intervals, such as sync transmit interval, announce interval ect.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asIntfIntervalGet(openapiClientHandle_t *client_handle, uint32_t  intfNum, OPEN_DOT1AS_INTF_INTERVAL_t  intervalType, int32_t *interval);

/*****************************************************************************
 * @purpose   Set the given interval on the given interface.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intfNum             @b{(input)}  Interface number.
 * @param     intervalType        @b{(input)}  DOT1AS interval type.
 * @param     interval            @b{(input)}  The interface interval/timeout value.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asIntfIntervalSet(openapiClientHandle_t *client_handle, uint32_t  intfNum, OPEN_DOT1AS_INTF_INTERVAL_t  intervalType, int32_t  interval);

/*****************************************************************************
 * @purpose   Get the requested 802.1AS counter for the given interface.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intfNum             @b{(input)}  Interface number.
 * @param     CounterType         @b{(input)}  DOT1AS counter type.
 * @param     counterValue        @b{(output)} Counter value.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     In 802.1AS protocol interfaces have many counters associated 
 *            with them, like sync messages trasmitted/received(tx/rx), 
 *            pdelay messages (tx/rx) etc.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asIntfStatsGet(openapiClientHandle_t *client_handle, 
                                       uint32_t  intfNum, 
				       OPEN_DOT1AS_COUNTER_TYPE_t  CounterType, 
				       uint32_t *counterValue);

/*****************************************************************************
 * @purpose   Reset 802.1AS counters for the given interface.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intfNum             @b{(input)}  Interface number.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          DOT1AS feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1asIntfStatsReset(openapiClientHandle_t *client_handle, 
                                         uint32_t  intfNum);


#endif /* OPENAPI_DOT1AS_H_INCLUDED */
