/**********************************************************************
*
*  Copyright 2020 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_msrp.h
*
* @purpose   Multiple Stream Reservation Protocol (MSRP). 
*
* @component OPEN
*
* @create    05/15/2020
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_MSRP_H_INCLUDED
#define OPENAPI_MSRP_H_INCLUDED

#include "openapi_common.h" 

/**<  Maximum number of MSRP interfaces in the system. */
/**<  call openapiMaxInterfaceCountGet() to get the number of  interfaces in the system */
#define OPEN_MSRP_INTF_MAX_COUNT    1024

/**< AVB standard defines 2 SR classes */
typedef enum 
{
  OPEN_MSRP_QAV_STREAM_CLASS_A = 0,  /**< Class A */
  OPEN_MSRP_QAV_STREAM_CLASS_B,      /**< Class B */
} OPEN_MSRP_QAV_STREAMCLASS_t;

typedef struct
{
  uint32_t       streamID;                          /**< Stream id to uniquely identify a stream */
  uint8_t        streamMacAddr[OPEN_MAC_ADDR_LEN];  /**< Stream MAC address */
  OPEN_BOOL_t    listenerRegistered;
  OPEN_BOOL_t    listenerDeclared;
  OPEN_BOOL_t    talkerRegistered;
  OPEN_BOOL_t    talkerDeclared;
  uint32_t       listenerDeclType;                  /**< Listener type */ 
  uint32_t       talkerDeclType;                    /**< Talker type */
  uint32_t       accLatency;
  uint8_t        failIntf;                          /**< Failure interface */
  uint8_t        failCode;                          /**< Failure code */
  uint8_t        failMacAddr[OPEN_MAC_ADDR_LEN];
  uint32_t       age;                               /**< Stream Age */
} OPEN_MSRP_RESERVATION_INFO_t;

typedef struct
{
  uint32_t        streamID;                           /**< Stream id to uniquely identify a stream */
  uint8_t         sourceMacAddr[OPEN_MAC_ADDR_LEN];   /**< source MAC address */
  uint32_t        accLatency;                          
  uint8_t         trafficClass;                       /**< SR Traffic class */
  uint8_t         rank;                               /**< Stream Rank */
  uint32_t        maxFrameSize;
  uint32_t        maxIntervalFrames;
  uint16_t        vlanId;                            /**< VLAN tag of stream packets */
  uint8_t         destMacAddr[OPEN_MAC_ADDR_LEN];    /**< Destination MAC address for stream packets */
  uint8_t         failIntf;                          /**< Failure interface */
  uint8_t         failCode;                          /**< Failure code */
  uint8_t         failMacAddr[OPEN_MAC_ADDR_LEN];    
  uint8_t         talkerIntf;
  uint32_t        listeners[OPEN_MSRP_INTF_MAX_COUNT - 1];
} OPEN_MSRP_STEAM_INFO_t;

typedef enum
{      
  OPEN_MSRP_DECTYPE_INVALID = 0,     /**< Uninitialized */
  OPEN_MSRP_DECTYPE_ADVERTISE,       /**< Advertise */
  OPEN_MSRP_DECTYPE_FAILED,          /**< Failed */
  OPEN_MSRP_DECTYPE_ASKING_FAILED,   /**< Asking failed */
  OPEN_MSRP_DECTYPE_READY,           /**< Ready */
  OPEN_MSRP_DECTYPE_READY_FAILED,    /**< Ready failed */
  OPEN_MSRP_DECTYPE_DOMAIN,          /**< Domain attributes */
} OPEN_MSRP_DECLARATION_t;

typedef enum 
{
  OPEN_MSRP_STATS_RX = 0,        /**< Flag for all received msrp PDUs */
  OPEN_MSRP_STATS_RX_BAD_HEADER, /**< Flag for received msrp PDUs with bad header */
  OPEN_MSRP_STATS_RX_BAD_FORMAT, /**< Flag for received msrp PDUs with bad data field */
  OPEN_MSRP_STATS_TX,            /**< Flag for all transmitted msrp PDUs */
  OPEN_MSRP_STATS_TX_FAILURE,    /**< Flag for msrp PDUs which failed */
  OPEN_MSRP_STATS_MSG            /**< Flag for message queue send failed */
} OPEN_MSRP_STATS_t;


/*****************************************************************************
 * @purpose   Get the global MSRP mode.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     mode             @b{(output)} Global admin mode.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     MSRP Can be enabled/disabled at global or interface level. This API gets the global admin mode status.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *adminMode);

/*****************************************************************************
 * @purpose   set the global MSRP mode.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     mode             @b{(input)}  Global admin mode.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     MSRP Can be enabled/disabled at global or interface level. This API sets the global admin mode status.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the MSRP boundary propagate mode.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     mode             @b{(output)} Boundary propagate mode.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpBoundaryPropagateModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *adminMode);

/*****************************************************************************
 * @purpose   Set the MSRP boundary propagate mode.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     mode             @b{(input)}  Boundary propagate mode.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpBoundaryPropagateModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the MSRP mode for the specified interface.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Interface Number.
 * @param     mode             @b{(output)} Interface MSRP admin mode.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpIntfModeGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_CONTROL_t *adminMode);

/*****************************************************************************
 * @purpose   Set the MSRP mode for the given interface.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Interface Number.
 * @param     mode             @b{(input)}  Interface MSRP admin mode.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpIntfModeSet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the 802.1Qav priority.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Interface Number.
 * @param     trClass          @b{(input)}  EAV class.
 * @param     priority         @b{(output)} Priority.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpSrQavPriorityGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_MSRP_QAV_STREAMCLASS_t trClass, uint32_t *priority);

/*****************************************************************************
 * @purpose   Set the 802.1Qav priority.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Interface Number.
 * @param     trClass          @b{(input)}  EAV class.
 * @param     priority         @b{(input)}  Priority.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpSrQavPrioritySet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_MSRP_QAV_STREAMCLASS_t trClass, uint32_t priority);

/*****************************************************************************
 * @purpose   Get the 802.1Qav remapped priority.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Interface Number.
 * @param     trClass          @b{(input)}  EAV class.
 * @param     remap            @b{(output)} Remapped priority.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpSrQavRePriorityGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_MSRP_QAV_STREAMCLASS_t trClass, uint32_t *remap);

/*****************************************************************************
 * @purpose   Set the 802.1Qav remapped priority.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Interface Number.
 * @param     trClass          @b{(input)}  EAV class.
 * @param     remap            @b{(input)}  Remapped priority.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpSrQavRePrioritySet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_MSRP_QAV_STREAMCLASS_t trClass, uint32_t remap);

/*****************************************************************************
 * @purpose   Get the MSRP talker pruning mode.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     mode             @b{(output)} Talker pruning mode.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpTalkerPruningModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *adminMode);

/*****************************************************************************
 * @purpose   Set the MSRP talker pruning mode.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     mode             @b{(input)}  Talker pruning mode.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpTalkerPruningModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the maximum number of the ports, where MSRP registrations are allowed.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     maxFanInPorts    @b{(output)} Number of the ports where MSRP registrations are allowed.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpMaxFanInPortsGet(openapiClientHandle_t *client_handle, uint32_t *maxFanInPorts);

/*****************************************************************************
 * @purpose   Set the maximum number of the ports, where MSRP registrations are allowed.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     maxFanInPorts    @b{(input)}  Number of the ports where MSRP registrations are allowed.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpMaxFanInPortsSet(openapiClientHandle_t *client_handle, uint32_t maxFanInPorts);

/*****************************************************************************
 * @purpose   Get the MSRP SR class PVID value on the interface.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Interface Number.
 * @param     pvid             @b{(output)} SRclassPVID.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpSRclassPVIDGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *pvid);

/*****************************************************************************
 * @purpose   Set the MSRP SR class PVID value on the interface.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Interface Number.
 * @param     pvid             @b{(input)}  SRclassPVID.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpSRclassPVIDSet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t pvid);

/*****************************************************************************
 * @purpose   Get the delta bandwidth values for the interface.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Interface Number.
 * @param     deltaClassA      @b{(output)} class A delta bandwidth.
 * @param     deltaClassB      @b{(output)} class B delta bandwidth.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     Both delta classA, delta classB bandwidth values are read.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpQavDeltaBandwidthGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *deltaClassA, uint32_t *deltaClassB);

/*****************************************************************************
 * @purpose   Get the delta bandwidth values for the interface.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Interface Number.
 * @param     trafficClass     @b{(input)}  Stream class (A or B).
 * @param     deltaBandwidth   @b{(input)}  Delta bandwidth.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     Both delta classA, delta classB bandwidth values are read.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpQavDeltaBandwidthSet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_MSRP_QAV_STREAMCLASS_t trafficClass, uint32_t deltaBandwidth);

/*****************************************************************************
 * @purpose   Get the max free bandwidth values for the interface.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Interface Number.
 * @param     trClassA         @b{(output)} class A free bandwidth.
 * @param     trClassB         @b{(output)} class B free bandwidth.
 * @param     total            @b{(output)} Total free bandwidth.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     Total classA, classB values are read.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpQavMaxRateGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint64_t *trClassA, uint64_t *trClassB, uint64_t *total);

/*****************************************************************************
 * @purpose   Get the current used bandwidth values for the interface.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Interface Number.
 * @param     trClassA         @b{(output)} class A free bandwidth.
 * @param     trClassB         @b{(output)} class B free bandwidth.
 * @param     total            @b{(output)} Total free bandwidth.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     Total, classA, classB values are read.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpQavCurrentBandwidthGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint64_t *trClassA, uint64_t *trClassB, uint64_t *total);

/*****************************************************************************
 * @purpose   Check if a port is boundary port.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Interface Number.
 * @param     trClass          @b{(input)}  EAV class.
 * @param     isBoundary       @b{(output)} Flag to check if the port is bondary port.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpIsBoundaryPortGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_MSRP_QAV_STREAMCLASS_t trClass, OPEN_BOOL_t *isBoundary);

/*****************************************************************************
 * @purpose   Given the next stream index to the given index.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     currentIndex     @b{(input)}  Present index.
 * @param     nextIndex        @b{(output)} Next index.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     To get the first stream index, pass 0 as the given index.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpStreamIndexedNextGet(openapiClientHandle_t *client_handle, uint16_t currentIndex, uint16_t *nextIndex);

/*****************************************************************************
 * @purpose   Get the MSRP Reservation Information.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     streamId         @b{(input)}  streamId.
 * @param     intIfNum         @b{(input)}  Interface Number.
 * @param     reservationInfo  @b{(output)} Reservation information.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpReservationInformationGet(openapiClientHandle_t *client_handle, uint32_t streamId, uint32_t intIfNum, open_buffdesc *reservationInfo);

/*****************************************************************************
 * @purpose   Get the value of specified MSRP counter.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     type             @b{(input)}  Type of counter.
 * @param     counter          @b{(output)} Counter value.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpTrafficPduCounterGet(openapiClientHandle_t *client_handle, OPEN_MSRP_STATS_t type, uint32_t *counter);

/*****************************************************************************
 * @purpose   Get the value of specified MSRP counter for an interface.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Interface Number.
 * @param     type             @b{(input)}  Type of counter.
 * @param     counter          @b{(output)} Counter value.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpIntfTrafficPduCounterGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_MSRP_STATS_t type, uint32_t *counter);

/*****************************************************************************
 * @purpose   Get Failed Registration Counter.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Interface Number.
 * @param     counter          @b{(output)} Counter value.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpFailedRegistrationCounterGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *counter);

/*****************************************************************************
 * @purpose   Get the MSRP Stream Information.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     streamId         @b{(input)}  Stream id.
 * @param     streamInfo       @b{(output)} Stream information.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpStreamInformationGet(openapiClientHandle_t *client_handle, uint32_t streamId, open_buffdesc *streamInfo);

/*****************************************************************************
 * @purpose   Clear MSRP traffic counters on specified interface.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     intIfNum         @b{(input)}  Interface Number.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpTrafficCountersPerIfClear(openapiClientHandle_t *client_handle, uint32_t intIfNum);

/*****************************************************************************
 * @purpose   Clear all MSRP traffic counters.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          MSRP feature is not supported.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiMsrpTrafficCountersClear(openapiClientHandle_t *client_handle);


#endif /* OPENAPI_MSRP_H_INCLUDED */
