/*********************************************************************
*
* Copyright 2017-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_pfc.h
*
* @purpose   Priority-based Flow Control Configuration and Statistics APIs.
*
* @component OPEN
*
* @create    01/23/2017
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_PFC_H_INCLUDED
#define OPENAPI_PFC_H_INCLUDED

#include <stdio.h>
#include <stddef.h>
#include <sys/un.h>

#include "openapi_common.h"

typedef enum 
{
  OPEN_PFC_PRI_DROP_MODE = 0,           /*<< priority drop mode */
  OPEN_PFC_PRI_NODROP_MODE = 1          /*<< priority no drop mode */
} OPEN_PFC_PRI_DROP_MODE_t; 

typedef enum 
{
  OPEN_PFC_ACTION_DROP = 0,             /*<< action drop */
  OPEN_PFC_ACTION_NO_DROP = 1           /*<< action no drop */
} OPEN_PFC_ACTION_t; 

typedef enum
{
  OPEN_PFC_FEATURE_SUPPORTED = 0,       /*<< general support statement */
  OPEN_PFC_PORT_PRIORITY_FLOW_CONTROL_FEATURE_ID = 1, /*<< port priority flow control feature */
  OPEN_PFC_TX_STATS_FEATURE_ID = 2,     /*<< tx stats feature */
} OPEN_PFC_FEATURE_IDS_t;

typedef enum
{
  OPEN_PFC_MODE_DISABLE = 0,            /*<< PFC disable mode */ 
  OPEN_PFC_MODE_ENABLE = 1              /*<< PFC enable mode */
} OPEN_PFC_MODE_t;

typedef enum
{
  OPEN_PFC_STATUS_ACTIVE = 0,           /*<< PFC status active */
  OPEN_PFC_STATUS_INACTIVE = 1          /*<< PFC status inactive */
} OPEN_PFC_STATUS_t;

#define OPEN_QOS_PFC_PRIORITY_ALL         8
#define OPEN_QOS_PFC_PRIORITY_MIN         0
#define OPEN_QOS_PFC_PRIORITY_MAX         (OPEN_QOS_PFC_PRIORITY_ALL-1)

/*********************************************************************
* @purpose  Gets the count of incompatible configurations received from peer.
*
* @param    client_handle         @b{(input)}   client handle.
* @param    ifNum                 @b{(input)}   interface number.
* @param    incompatibleCfgCount  @b{(output)}  incompatible count from peer. 
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_FAIL       Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiPfcPeerIncompatibleCfgCountGet(openapiClientHandle_t *client_handle,
                                                   uint32_t ifNum,
                                                   uint32_t *incompatibleCfgCount);

/*********************************************************************
* @purpose  Get the willingness of the peer. 
*
* @param    client_handle         @b{(input)}   client handle.
* @param    ifNum                 @b{(input)}   interface number.
* @param    willing               @b{(output)}  willingness of the peer.
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_NOT_FOUND  Peer not detected.
* @returns  OPEN_E_UNAVAIL    Feature not supported.
* @returns  OPEN_E_FAIL       Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiPfcPeerWillingGet(openapiClientHandle_t *client_handle,
                                      uint32_t ifNum,
                                      OPEN_BOOL_t *willing);

/*********************************************************************
* @purpose  Get the MACSEC bypass capability of the peer.
*
* @param    client_handle         @b{(input)}   client handle.
* @param    ifNum                 @b{(input)}   interface number.
* @param    mbcStatus             @b{(output)}  MACSEC bypass capability of peer.
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_NOT_FOUND  Peer not detected. 
* @returns  OPEN_E_FAIL       Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiPfcPeerMbcStatusGet(openapiClientHandle_t *client_handle,
                                        uint32_t ifNum,
                                        OPEN_BOOL_t *mbcStatus);

/*********************************************************************
* @purpose  Get the capability of the peer. 
*
* @param    client_handle         @b{(input)}   client handle.
* @param    ifNum                 @b{(input)}   interface number.
* @param    numPfcCap             @b{(output)}  capability of the peer.
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_FAIL       Some failure occurred.
*
* @notes if peer is not detected, numPfcCap is set to 0.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiPfcPeerCapabilityGet(openapiClientHandle_t *client_handle,
                                         uint32_t ifNum,
                                         uint32_t *numPfcCap);

/*********************************************************************
* @purpose  Gets the advertised PFC Priority participation mode for
*           the interface 
*
* @param    client_handle         @b{(input)}   client handle.
* @param    ifNum                 @b{(input)}   interface number.
* @param    priority              @b{(input)}   priority.
* @param    mode                  @b{(output)}  participation mode.
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_NOT_FOUND  Interface not supported.
* @returns  OPEN_E_FAIL       Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiPfcAdvPriorityModeGet(openapiClientHandle_t *client_handle,
                                          uint32_t ifNum,
                                          uint32_t priority,
                                          OPEN_PFC_PRI_DROP_MODE_t *mode);

/*********************************************************************
* @purpose  Get the peer priority mode for a priority. 
*
* @param    client_handle         @b{(input)}   client handle.
* @param    ifNum                 @b{(input)}   interface number.
* @param    priority              @b{(input)}   priority.
* @param    mode                  @b{(output)}  peer priority mode.
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_NOT_FOUND  Priority is greater than OPEN_DOT1P_MAX_PRIORITY or peer is not detected.
* @returns  OPEN_E_FAIL       Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiPfcPeerPriorityModeGet(openapiClientHandle_t *client_handle,
                                           uint32_t ifNum,
                                           uint32_t priority,
                                           OPEN_PFC_PRI_DROP_MODE_t *mode);

/*********************************************************************
* @purpose  Enable an interface for PFC
*
* @param    client_handle         @b{(input)}   client handle.
* @param    ifNum                 @b{(input)}   interface number.
* @param    mode                  @b{(input)}   mode
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_UNAVAIL    Feature not supported.
* @returns  OPEN_E_FAIL       Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiIfPfcModeSet(openapiClientHandle_t *client_handle,
                                 uint32_t ifNum,
                                 OPEN_PFC_MODE_t mode);

/*********************************************************************
* @purpose  Set the drop mode for the priority on an interface. 
*
* @param    client_handle         @b{(input)}   client handle.
* @param    ifNum                 @b{(input)}   interface number.
* @param    priority              @b{(input)}   priority 
* @param    dropMode              @b{(input)}   drop mode
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_NOT_FOUND  Priority is less than PFC minimum priority or greater than PFC maximum priority.
* @returns  OPEN_E_UNAVAIL    Feature not supported.
* @returns  OPEN_E_FAIL       Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiIfPfcPriorityModeSet(openapiClientHandle_t *client_handle,
                                         uint32_t ifNum,
                                         uint32_t priority,
                                         OPEN_PFC_PRI_DROP_MODE_t dropMode);

/*********************************************************************
* @purpose  Returns the minimum priority that can be configured in PFC
*
* @param    client_handle         @b{(input)}   client handle.
* @param    minPrio               @b{(output)}  minimum priority.
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiPfcMinNoDropPriorityGet(openapiClientHandle_t *client_handle,
                                            uint32_t *minPrio);

/*********************************************************************
* @purpose  Returns the maximum priority that can be configured in PFC.
*
* @param    client_handle         @b{(input)}   client handle.
* @param    maxPrio               @b{(output)}  maximum priority.
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiPfcMaxNoDropPriorityGet(openapiClientHandle_t *client_handle,
                                            uint32_t *maxPrio);

/*********************************************************************
* @purpose  Clear the PFC stats on an interface. 
*
* @param    client_handle         @b{(input)}   client handle.
* @param    ifNum                 @b{(input)}   interface number.
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_UNAVAIL    Feature not supported.
* @returns  OPEN_E_FAIL       Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiIfPfcStatsClear(openapiClientHandle_t *client_handle,
                                    uint32_t ifNum);

/*********************************************************************
* @purpose  Get the active status on the interface.
*
* @param    client_handle         @b{(input)}   client handle.
* @param    ifNum                 @b{(input)}   interface number.
* @param    status                @b{(output)}  The active status of the interface.
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_FAIL       Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiIfPfcStatusGet(openapiClientHandle_t *client_handle,
                                   uint32_t ifNum,
                                   OPEN_PFC_STATUS_t *status);

/*********************************************************************
* @purpose  Get the pfc mode for the interface. 
*
* @param    client_handle         @b{(input)}   client handle.
* @param    ifNum                 @b{(input)}   interface number.
* @param    mode                  @b{(output)}  enabled for pfc or not.
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_UNAVAIL    Feature not supported.
* @returns  OPEN_E_FAIL       Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiIfPfcModeGet(openapiClientHandle_t *client_handle,
                                 uint32_t ifNum,
                                 OPEN_PFC_MODE_t *mode);

/*********************************************************************
* @purpose  Get the drop mode for the priority on an interface.
*
* @param    client_handle         @b{(input)}   client handle.
* @param    ifNum                 @b{(input)}   interface number.
* @param    priority              @b{(input)}   priority to get.
* @param    mode                  @b{(output)}  the drop mode of the interface.
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_UNAVAIL    Feature not supported.
* @returns  OPEN_E_NOT_FOUND  Priority is greater than OPEN_DOT1P_MAX_PRIORITY
* @returns  OPEN_E_FAIL       Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end  
*********************************************************************/
open_error_t openapiIfPfcPriorityModeGet(openapiClientHandle_t *client_handle,
                                         uint32_t ifNum,
                                         uint32_t priority,
                                         OPEN_PFC_PRI_DROP_MODE_t *mode);

/*********************************************************************
* @purpose  Gets the operational PFC Priority participation mode for
*           the interface.

* @param    client_handle         @b{(input)}   client handle.
* @param    ifNum                 @b{(input)}   interface number.
* @param    priority              @b{(input)}   priority to get.
* @param    mode                  @b{(output)}  the drop mode of the interface.
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_NOT_FOUND  Priority is greater than OPEN_DOT1P_MAX_PRIORITY
* @returns  OPEN_E_FAIL       Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiPfcOprPriorityModeGet(openapiClientHandle_t *client_handle,
                                          uint32_t ifNum,
                                          uint32_t priority,
                                          OPEN_PFC_PRI_DROP_MODE_t *mode);

/*********************************************************************
* @purpose  Gets the link delay allowance in bits for the interface.
*
* @param    client_handle         @b{(input)}   client handle.
* @param    ifNum                 @b{(input)}   interface number.
* @param    allowance             @b{(output)}  link delay allowance.
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_FAIL       Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiPfcOperLinkDelayAllowanceGet(openapiClientHandle_t *client_handle,
                                                 uint32_t ifNum,
                                                 uint32_t *allowance);

/*********************************************************************
* @purpose  Indicates if the peer configuration is compatible.
*
* @param    client_handle         @b{(input)}   client handle.
* @param    ifNum                 @b{(input)}   interface number.
* @param    compatible            @b{(output)}  peer configuration compatible.
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_NOT_FOUND  Peer not detected.
* @returns  OPEN_E_FAIL       Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiPfcPeerCfgCompatibleGet(openapiClientHandle_t *client_handle,
                                            uint32_t ifNum,
                                            OPEN_BOOL_t *compatible);

/*********************************************************************
* @purpose  Gets the count of compatible configurations received from peer.
*
* @param    client_handle         @b{(input)}   client handle.
* @param    ifNum                 @b{(input)}   interface number.
* @param    count                 @b{(output)}  count of compatible configurations.
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_FAIL       Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiPfcPeerCompatibleCfgCountGet(openapiClientHandle_t *client_handle,
                                                 uint32_t ifNum,
                                                 uint32_t *count);

/*********************************************************************
* @purpose  Get the PFC Rx Priority stat on an interface. 
*
* @param    client_handle       @b{(input)}   client handle.
* @param    ifNum               @b{(input)}   interface number.
* @param    priority            @b{(input)}   priority to get.
* @param    stat                @b{(output)}  the statistic.
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_UNAVAIL    Feature not supported.
* @returns  OPEN_E_NOT_FOUND  Priority is greater than OPEN_DOT1P_MAX_PRIORITY
* @returns  OPEN_E_FAIL       Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiIfPfcRxPriorityStatGet(openapiClientHandle_t *client_handle,
                                           uint32_t ifNum,
                                           uint32_t priority,
                                           uint32_t *stat);

/*********************************************************************
* @purpose  Gets the number of PFC packets transmitted per 
*           interface/priority.
*
* @param    client_handle       @b{(input)}   client handle.
* @param    ifNum               @b{(input)}   interface number.
* @param    priority            @b{(input)}   priority to get.
* @param    count               @b{(output)}  Number of pfc packets tx on this priority
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_UNAVAIL    Feature not supported.
* @returns  OPEN_E_NOT_FOUND  Priority is greater than OPEN_DOT1P_MAX_PRIORITY
* @returns  OPEN_E_FAIL       Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiPfcTxPriorityStatGet(openapiClientHandle_t *client_handle,
                                         uint32_t ifNum,
                                         uint32_t priority,
                                         uint32_t *count);

/*********************************************************************
* @purpose  Get the PFC Rx stat on an interface.
*
* @param    client_handle         @b{(input)}   client handle.
* @param    ifNum                 @b{(input)}   interface number.
* @param    count                 @b{(output)}  PFC Rx statistic.
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_UNAVAIL    Feature not supported.
* @returns  OPEN_E_FAIL       Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiIfPfcRxStatGet(openapiClientHandle_t *client_handle,
                                   uint32_t ifNum,
                                   uint32_t *count);

/*********************************************************************
* @purpose  Get the PFC Tx stat on an interface.
*
* @param    client_handle         @b{(input)}   client handle.
* @param    ifNum                 @b{(input)}   interface number.
* @param    count                 @b{(output)}  PFC Tx statistic.
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Bad parameter.
* @returns  OPEN_E_UNAVAIL    Feature not supported.
* @returns  OPEN_E_FAIL       Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiIfPfcTxStatGet(openapiClientHandle_t *client_handle,
                                   uint32_t ifNum,
                                   uint32_t *count);

#endif /* OPENAPI_PFC_H_INCLUDED */
