/*********************************************************************
*
*  Copyright 2017-2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_radius.h
*
* @purpose   RADIUS Configuration and Status
*
* @component OpEN
*
* @create    12/09/2016
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_RADIUS_H_INCLUDED
#define OPENAPI_RADIUS_H_INCLUDED

#include "openapi_common.h"

/** Designates a server as primary or secondary. */
typedef enum
{
  OPEN_RADIUS_SERVER_ENTRY_TYPE_PRIMARY = 1,
  OPEN_RADIUS_SERVER_ENTRY_TYPE_SECONDARY = 2,
} OPEN_RADIUS_SERVER_ENTRY_TYPE_t;

/*********************************************************************
* @purpose  Get the maximum size allowed for the RADIUS servers address
*           parameter. This is used as the string length including the
*           terminating NULL.
*
* @param    client_handle   @b{(input)} Client handle from registration API
*
* @returns  size of string
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
uint32_t openapiRadiusServerAddressMaxLengthGet(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Get the maximum size allowed for the RADIUS servers name
*           parameter. This is used as the string length including the
*           terminating NULL.
*
* @param    client_handle   @b{(input)} Client handle from registration API
*
* @returns  size of string
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
uint32_t openapiRadiusServerNameMaxLengthGet(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Add a RADIUS authentication server.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    serverAddress   @b{(input)} DNS name/IP address of the server
* @param    addressType     @b{(input)} indicates whether serverAddress contains a
*                                       DNS name or a IP address
* @param    serverName      @b{(input)} name identifying the server in the configuration
*
* @returns  OPEN_E_NONE     server added
* @returns  OPEN_E_FAIL     adding server failed
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error has occurred
*
* @notes  Calling this API will change the running configuration of the switch.
* @notes  Maximum size for serverAddress is determined by calling
*         openapiRadiusServerAddressMaxLengthGet().
* @notes  Maximum size for serverName is determined by calling
*         openapiRadiusServerNameMaxLengthGet().
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiRadiusAuthServerAdd(openapiClientHandle_t *client_handle, open_buffdesc *serverAddress,
                                        OPEN_IP_ADDRESS_TYPE_t addressType, open_buffdesc *serverName);

/*********************************************************************
* @purpose  Remove a RADIUS authentication server.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    serverAddress   @b{(input)} DNS name/IP address of the server
* @param    addressType     @b{(input)} indicates whether serverAddress contains a
*                                       DNS name or a IP address
*
* @returns  OPEN_E_NONE     server removed
* @returns  OPEN_E_FAIL     removing server failed
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error has occurred
*
* @notes  Calling this API will change the running configuration of the switch.
* @notes  Maximum size for serverAddress is determined by calling
*         openapiRadiusServerAddressMaxLengthGet().
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiRadiusAuthServerRemove(openapiClientHandle_t *client_handle, open_buffdesc *serverAddress,
                                           OPEN_IP_ADDRESS_TYPE_t addressType);

/*********************************************************************
* @purpose  Set the RADIUS authentication server port number.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    serverAddress   @b{(input)} DNS name/IP address of the server
* @param    addressType     @b{(input)} indicates whether serverAddress contains a
*                                       DNS name or a IP address
* @param    portNumber      @b{(input)} server's port number
*
* @returns  OPEN_E_NONE     set succeeded
* @returns  OPEN_E_FAIL     set failed
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error has occurred
*
* @notes  Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiRadiusAuthServerPortNumSet(openapiClientHandle_t *client_handle, open_buffdesc *serverAddress,
                                               OPEN_IP_ADDRESS_TYPE_t addressType, uint32_t portNumber);

/*********************************************************************
* @purpose  Get the RADIUS authentication server port number.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    serverAddress   @b{(input)}  DNS name/IP address of the server
* @param    addressType     @b{(input)}  indicates whether serverAddress contains a
*                                        DNS name or a IP address
* @param    portNumber      @b{(output)} server's port number
*
* @returns  OPEN_E_NONE     get succeeded
* @returns  OPEN_E_FAIL     get failed
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error has occurred
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiRadiusAuthServerPortNumGet(openapiClientHandle_t *client_handle, open_buffdesc *serverAddress,
                                               OPEN_IP_ADDRESS_TYPE_t addressType, uint32_t *portNumber);

/*********************************************************************
* @purpose  Set the RADIUS authentication server type.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    serverAddress   @b{(input)} DNS name/IP address of the server
* @param    addressType     @b{(input)} indicates whether serverAddress contains a
*                                       DNS name or a IP address
* @param    serverType      @b{(input)} server's type
*
* @returns  OPEN_E_NONE     set succeeded
* @returns  OPEN_E_FAIL     set failed
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error has occurred
*
* @notes  Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiRadiusAuthServerTypeSet(openapiClientHandle_t *client_handle, open_buffdesc *serverAddress,
                                            OPEN_IP_ADDRESS_TYPE_t addressType, OPEN_RADIUS_SERVER_ENTRY_TYPE_t serverType);

/*********************************************************************
* @purpose  Get the RADIUS authentication server type.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    serverAddress   @b{(input)}  DNS name/IP address of the server
* @param    addressType     @b{(input)}  indicates whether serverAddress contains a
*                                        DNS name or a IP address
* @param    serverType      @b{(output)} server's type
*
* @returns  OPEN_E_NONE     get succeeded
* @returns  OPEN_E_FAIL     get failed
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error has occurred
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiRadiusAuthServerTypeGet(openapiClientHandle_t *client_handle, open_buffdesc *serverAddress,
                                            OPEN_IP_ADDRESS_TYPE_t addressType, OPEN_RADIUS_SERVER_ENTRY_TYPE_t *serverType);

/*********************************************************************
* @purpose  Set the RADIUS authentication server attribute 4 value.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    mode            @b{(input)} set to OPEN_FALSE to unconfigure attribute 4, set
*                                       to OPEN_TRUE to set attribute 4
* @param    ipAddr          @b{(input)} the IP address to send, only IPv4 addresses supported
*
* @returns  OPEN_E_NONE     set succeeded
* @returns  OPEN_E_FAIL     set failed
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error has occurred
*
* @notes  Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiRadiusAuthServerAttribute4Set(openapiClientHandle_t *client_handle, OPEN_BOOL_t mode,
                                                  open_inet_addr_t *ipAddr);

/*********************************************************************
* @purpose  Get the RADIUS authentication server attribute 4 value.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    mode            @b{(output)} OPEN_FALSE indicates attribute 4 not configured,
*                                        OPEN_TRUE indicates attribute 4 configured
* @param    ipAddr          @b{(output)} the IP address to send, only IPv4 addresses supported
*
* @returns  OPEN_E_NONE     get succeeded
* @returns  OPEN_E_FAIL     get failed
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error has occurred
*
* @notes  Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiRadiusAuthServerAttribute4Get(openapiClientHandle_t *client_handle, OPEN_BOOL_t *mode,
                                                  open_inet_addr_t *ipAddr);

/*********************************************************************
* @purpose  Get the maximum size allowed for the RADIUS shared secret.
*           This is used as the string length including the
*           terminating NULL.
*
* @param    client_handle   @b{(input)} Client handle from registration API
*
* @returns  size of string
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
uint32_t openapiRadiusSharedSecretMaxLengthGet(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Get the size for an encrypted RADIUS shared secret.
*           This is used as the string length including the
*           terminating NULL. Encrypted shared secrets are a
*           series of hexadecimal digits represented in ASCIIZ
*           string format.
*
* @param    client_handle   @b{(input)} Client handle from registration API
*
* @returns  size of string
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
uint32_t openapiRadiusEncryptedSharedSecretSizeGet(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Set the shared secret being used between the RADIUS client and
*           the RADIUS server. The shared secret is specified in encrypted
*           form.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    serverAddress   @b{(input)} DNS name/IP address of the server
* @param    encryptedSecret @b{(input)} encrypted secret
*
* @returns  OPEN_E_NONE     set succeeded
* @returns  OPEN_E_FAIL     set failed
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error has occurred
*
* @notes  Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiRadiusAuthServerSharedSecretEncryptedSet(openapiClientHandle_t *client_handle, open_buffdesc *serverAddress,
                                                             open_buffdesc *encryptedSecret);

#endif
