/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_routing_arptable.h
*
* @purpose   Routing ARP table configuration and status.
*
* @component OPEN
*
* @create    12/02/2016
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_ROUTING_ARPTABLE_H_INCLUDED
#define OPENAPI_ROUTING_ARPTABLE_H_INCLUDED

#include <stdio.h>
#include <stddef.h>
#include <sys/un.h>

#include "openapi_common.h"
#include "openapi_routing_config.h"

typedef enum
{
  OPEN_ARP_LOCAL               = 0x01,   /**< IP address is one of switch's MAC addresses */
  OPEN_ARP_GATEWAY             = 0x02,   /**< Destination is a router */
  OPEN_ARP_RESOLVED            = 0x04,   /**< Resolved ARP entry, dest MAC address known */
  OPEN_ARP_NET_DIR_BCAST       = 0x08,   /**< Net directed broadcast addr */
  OPEN_ARP_DISCARD_WHEN_SOURCE = 0x10,   /**< Discard when seen as source */
  OPEN_ARP_STATIC              = 0x20,   /**< Static ARP entry */
  OPEN_ARP_UNNUMBERED          = 0x40,   /**< Unnumbered intf ARP entry */
  OPEN_ARP_KERNEL              = 0x80    /**< Kernel ARP entry */
} OPEN_ARP_TYPE_t;

typedef struct
{
  OPEN_ARP_TYPE_t type;    /**< ARP entry type bitmask */
  uint32_t age;            /**< ARP entry age in seconds */
  open_inet_addr_t ipAddr; /**< IP Address of the ARP table entry */
  uint16_t vrfId;          /**< VRF ID of the ARP table entry */
  unsigned char macAddr[OPEN_MAC_ADDR_LEN]; /**< MAC address of the ARP entry, if resolved */
  uint16_t vlanId;        /**< VLAN id of the ARP table entry */
  uint32_t intIfNum;      /**< Outgoing internal interface number of the ARP table entry */
  uint32_t hits;          /**< Non-zero count of the ARP table entry, if resolved */
} OPEN_ARP_ENTRY_t;

/*********************************************************************
* @purpose  API to get the ARP age time in seconds.
*
* @param    client_handle   @b{(input)}   client handle from registration API.
* @param    val             @b((output))  ARP age time in seconds
*
* @returns  OPEN_E_NONE   On success.
* @returns  OPEN_E_PARAM  Bad parameter.
* @returns  OPEN_E_FAIL   Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiArpAgeTimeGet(openapiClientHandle_t *client_handle,
                                  uint32_t *val);

/*********************************************************************
* @purpose  API to get the ARP response time in seconds.
*
* @param    client_handle   @b{(input)}   client handle from registration API.
* @param    val             @b((output))  ARP response time in seconds
*
* @returns  OPEN_E_NONE   On success.
* @returns  OPEN_E_PARAM  Bad parameter.
* @returns  OPEN_E_FAIL   Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiArpResponseTimeGet(openapiClientHandle_t *client_handle,
                                       uint32_t *val);

/*********************************************************************
* @purpose  API to get the ARP number of retries.
*
* @param    client_handle   @b{(input)}   client handle from registration API.
* @param    val             @b((output))  ARP number of retries.
*
* @returns  OPEN_E_NONE   On success.
* @returns  OPEN_E_PARAM  Bad parameter.
* @returns  OPEN_E_FAIL   Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiArpNumRetriesGet(openapiClientHandle_t *client_handle,
                                     uint32_t *val);

/*********************************************************************
* @purpose  API to get the ARP cache size.
*
* @param    client_handle   @b{(input)}   client handle from registration API.
* @param    val             @b((output))  ARP cache size.
*
* @returns  OPEN_E_NONE   On success.
* @returns  OPEN_E_PARAM  Bad parameter.
* @returns  OPEN_E_FAIL   Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiArpCacheSizeGet(openapiClientHandle_t *client_handle,
                                    uint32_t *val);

/*********************************************************************
* @purpose  API to get if ARP dynamic renew mode is enabled or not.
*
* @param    client_handle   @b{(input)}   client handle from registration API.
* @param    val             @b((output))  If OPEN_ENABLE, enabled. Otherwise, OPEN_DISABLE.
*
* @returns  OPEN_E_NONE   On success.
* @returns  OPEN_E_PARAM  Bad parameter.
* @returns  OPEN_E_FAIL   Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiArpDynamicRenewModeGet(openapiClientHandle_t *client_handle,
                                           OPEN_CONTROL_t *val);

/*********************************************************************
* @purpose  API to get the ARP total current entry count info.
*
* @param    client_handle   @b{(input)}   client handle from registration API.
* @param    current         @b((output))  total current entry count.
* @param    peak            @b((output))  total entry count peak.
*
* @returns  OPEN_E_NONE   On success.
* @returns  OPEN_E_PARAM  Bad parameter.
* @returns  OPEN_E_FAIL   Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiArpTotalEntryCountInfoGet(openapiClientHandle_t *client_handle,
                                              uint32_t *current,
                                              uint32_t *peak);

/*********************************************************************
* @purpose  API to get the ARP static entry count info.
*
* @param    client_handle   @b{(input)}   client handle from registration API.
* @param    configured      @b((output))  configured static entry count.
* @param    active          @b((output))  active static entry count. 
* @param    max             @b((output))  static entry count maximum.
*
* @returns  OPEN_E_NONE   On success.
* @returns  OPEN_E_PARAM  Bad parameter.
* @returns  OPEN_E_FAIL   Some failure occurred.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiArpStaticEntryCountInfoGet(openapiClientHandle_t *client_handle,
                                               uint32_t *configured,
                                               uint32_t *active,
                                               uint32_t *max);

/*********************************************************************
* @purpose  Get the next VRF ARP entry 
*
* @param    client_handle   @b{(input)} client handle from registration API.
* @param    vrfName         @b{(input)} VRF name (or OPEN_DEFAULT_VRF_NAME if the default VRF is desired).
* @param    intf            @b{(input/output)} Router Interface. Set to 0 to get the first entry.
* @param    ipAddr          @b{(input/output)} IP address. Set the addr.ipv4 
*                                              field to 0 to get first entry.
*
* @returns  OPEN_E_NONE      Get successful.
* @returns  OPEN_E_FAIL      Failure of some kind.
* @returns  OPEN_E_NOT_FOUND No more entries
* @returns  OPEN_E_PARAM     If an input parameter is invalid.
*
* @notes  Setting vrfName to OPEN_DEFAULT_VRF_NAME will return data for the 
*         default VRF and results in the function behaving the same as
*         openapiArpEntryNextGet()
*         Both intf and ipAddr must be set to zero to get the first entry.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiArpEntryVrfNextGet(openapiClientHandle_t *client_handle,
                                       open_buffdesc *vrfName,
                                       uint32_t *intf, 
                                       open_inet_addr_t *ipAddr);

/*********************************************************************
* @purpose  Get the ARP data for a given vrfName, ipAddr, and interface.
*
* @param    client_handle   @b{(input)} client handle from registration API.
* @param    vrfName         @b{(input)} A valid VRF name.
* @param    intf            @b{(input)} A valid interface.
* @param    ipAddr          @b{(input)} A valid IP address. 
* @param    entry           @b{(output)} ARP table entry.
*
* @returns  OPEN_E_NONE      Get successful.
* @returns  OPEN_E_FAIL      Failure of some kind.
* @returns  OPEN_E_NOT_FOUND The arguments don't specify a valid ARP entry.
* @returns  OPEN_E_PARAM     If an input parameter is invalid.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiArpEntryInfoGet(openapiClientHandle_t *client_handle,
                                    open_buffdesc *vrfName,
                                    uint32_t intf,
                                    open_inet_addr_t ipAddr,
                                    OPEN_ARP_ENTRY_t *entry);

#endif /* OPENAPI_ROUTING_ARPTABLE_H_INCLUDED */
