/*********************************************************************
*
*  Copyright 2017-2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_voip.h
*
* @purpose   Auto-VoIP Configuration
*
* @component OpEN
*
* @comments  This code must not include any ICOS header files.
*
* @create    08/02/2017
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_VOIP_H_INCLUDED
#define OPENAPI_VOIP_H_INCLUDED

#include "openapi_common.h"

typedef enum
{
  OPEN_QOS_VOIP_TYPE_PROTOCOL=1,                    
  OPEN_QOS_VOIP_TYPE_OUI,
  OPEN_QOS_VOIP_TYPE_TOTAL                          
} OPEN_QOS_VOIP_TYPE_t;

typedef enum
{
   OPEN_VOIP_PRI_REMARK = 1,
   OPEN_VOIP_PRI_TRAFFIC_CLASS,
} OPEN_VOIP_PRI_SCHEME_t; 

/*********************************************************************
* @purpose  Sets the VLAN ID for voip traffic on switch          
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    vlanId             @b{(input)}  VLAN ID 
*
* @returns  OPEN_E_NONE        Set successful.    
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipGlobalVlanIdSet(openapiClientHandle_t *client_handle,
                                        uint32_t vlanId);

/*********************************************************************
* @purpose  Resets the VLAN ID for voip traffic on switch          
*
* @param    client_handle      @b{(input)}  client handle from registration API
*
* @returns  OPEN_E_NONE        Set successful.    
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipGlobalVlanIdReset(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Gets the VLAN ID for voip traffic on switch          
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    vlanId             @b{(output)} VLAN ID 
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipGlobalVlanIdGet(openapiClientHandle_t *client_handle,
                                        uint32_t *vlanId);

/*********************************************************************
* @purpose  Get the maximum number of Auto-VoIP calls supported          
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    maxCalls           @b{(output)} maximum number of Auto-VoIP calls 
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipMaxCallsGet(openapiClientHandle_t *client_handle,
                                    uint32_t *maxCalls);

/*********************************************************************
* @purpose  Gets VoIP Protocol Mode Active Sessions 
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    sessions           @b{(output)} number of Auto-VoIP sessions 
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipNumActiveProtocolSessionsGet(openapiClientHandle_t *client_handle,
                                                     uint32_t *sessions);

/*********************************************************************
* @purpose  Gets the VOIP profile Cos Queue 
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    pType              @b{(output)} priScheme (remark or traffic-class) 
* @param    pTypeVal           @b{(output)} value
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipProfileCosQueueGet(openapiClientHandle_t *client_handle,
                                           OPEN_VOIP_PRI_SCHEME_t *pType,
                                           uint32_t *pTypeVal);

/*********************************************************************
* @purpose  Sets the VOIP profile cos queue 
*
* @param    client_handle      @b{(input)} client handle from registration API
* @param    pType              @b{(input)} priScheme (remark or traffic-class) 
* @param    pTypeVal           @b{(input)} value
*
* @returns  OPEN_E_NONE        Set successful.    
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipProfileCosQueueSet(openapiClientHandle_t *client_handle,
                                           OPEN_VOIP_PRI_SCHEME_t pType,
                                           uint32_t pTypeVal);

/*********************************************************************
* @purpose  Gets the VOIP profile mode on interface 
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface to be configured
* @param    type           @b{(input)}  Auto VoIP type (protocol/OUI based)
* @param    mode           @b{(output)} Mode value
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipIntfModeGet(openapiClientHandle_t *client_handle,
                                    uint32_t ifNum, 
                                    OPEN_QOS_VOIP_TYPE_t type,
                                    OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose  Sets the VOIP profile mode on interface 
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface to be configured
* @param    type           @b{(input)}  Auto VoIP type (protocol/OUI based)
* @param    mode           @b{(input)}  Mode value
*
* @returns  OPEN_E_NONE        Set successful.    
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipIntfModeSet(openapiClientHandle_t *client_handle,
                                    uint32_t ifNum, 
                                    OPEN_QOS_VOIP_TYPE_t type,
                                    OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose  Gets VoIP Protocol mode operational port status 
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  Interface to be configured
* @param    val            @b{(output)} Operational status (UP or DOWN)
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipIntfProtocolStatusGet(openapiClientHandle_t *client_handle,
                                              uint32_t ifNum, 
                                              OPEN_LINK_STATE_t *val);

/*********************************************************************
* @purpose  Gets VoIP OUI mode operational port status 
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  Interface to be configured
* @param    val            @b{(output)} Operational status (UP or DOWN)
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipIntfOUIStatusGet(openapiClientHandle_t *client_handle,
                                         uint32_t ifNum, 
                                         OPEN_LINK_STATE_t *val);

/*********************************************************************
* @purpose  Gets the priority for voip traffic on switch 
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    priority       @b{(output)} priority value
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipOUIPriorityGet(openapiClientHandle_t *client_handle,
                                       uint32_t *priority); 

/*********************************************************************
* @purpose  Sets the priority for voip traffic on switch 
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    priority       @b{(input)}  priority value
*
* @returns  OPEN_E_NONE        Set successful.    
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiVoipOUIPrioritySet(openapiClientHandle_t *client_handle,
                                       uint32_t priority); 

#endif /*#ifndef OPENAPI_VOIP_H_INCLUDED*/
