#!/mnt/fastpath/usr/bin/python

"""bgp_example.py: OpEN API Border Gateway Protocol (BGP) configuration example"""

import OpEN_py as OpEN
from OpENUtil import *
import socket
import struct

open_ = OpENUtil()

#
# Copyright 2016 Broadcom.
#
#  Licensed under the Apache License, Version 2.0 (the "License")
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#

#
# Python 2.6.6
#

# Maximum number of protocol names
XX_PROTO_MAX = 10

def ip_to_int(addr):
  """Convert ipv4 string to integer"""

  return struct.unpack("!I", socket.inet_aton(addr))[0]

def int_to_ip(addr):
  """Convert ipv4 integer to string"""

  return socket.inet_ntoa(struct.pack("!I", addr))

def ipv6_to_int(addr):
  """Convert ipv6 string to integer"""

  str_ = socket.inet_pton(socket.AF_INET6, addr)
  a, b = struct.unpack('!2Q', str_)
  return (a << 64) | b

def int_to_ipv6(addr):
  """Convert ipv6 integer to string"""

  a = addr >> 64
  b = addr & ((1 << 64) - 1)
  return socket.inet_ntop(socket.AF_INET6, struct.pack('!2Q', a, b))

def ip_to_open_inet(address):
  """ Convert ip address to integer and family and return as open_inet_addr_t """

  type = 'unsupported'
  addr = OpEN.open_inet_addr_t()
  addr.family = OpEN.OPEN_AF_NONE
  try:
    addr.addr.ipv4 = ip_to_int(address)
    addr.family = OpEN.OPEN_AF_INET
    result = OpEN.OPEN_E_NONE
    type = 'ipv4'
  except:
    if socket.has_ipv6:
      try:
        addr.addr.ipv6 = ipv6_to_int(address)
        addr.family = OpEN.OPEN_AF_INET6
        result = OpEN.OPEN_E_NONE
        type = 'ipv6'
      except:
        result = OpEN.OPEN_E_PARAM
    else:
      type += ' ipv6'
      result = OpEN.OPEN_E_UNAVAIL

  return (result, addr, type)

def print_sanity_results(result, test, msg, feat):
  """Print overall comparison results"""

  if result == OpEN.OPEN_E_UNAVAIL:
    print "Sanity test skipped - %s - %s." % (msg, feat)
  elif result == OpEN.OPEN_E_NONE and test == True:
    print "Sanity Success - %s - %s." % (msg, feat)
  else:
    print "Sanity Failure - %s - %s." % (msg, feat)

def print_bad_result(result, msg):
  """Print some general error messages if the result is bad"""
   
  if result == OpEN.OPEN_E_UNAVAIL:
    print "Feature not supported - %s (err %d)." % (msg, result)
  elif result != OpEN.OPEN_E_NONE:
    print "Test Failure - %s (err %d)." % (msg, result)  

def get_first_routing_protocol_name(client_handle):
  """ Get the first available routing protocol name. """

  max_proto_len_p = OpEN.new_uint32_tp()

  result = OpEN.openapiRouteProtoNameLenMax(client_handle, max_proto_len_p)
  print_bad_result(result, 'openapiRouteProtoNameLenMax')
  if result == OpEN.OPEN_E_NONE:
    max_proto_len = OpEN.uint32_tp_value(max_proto_len_p) + 1
    try:
      proto_buff_string = open_.getStringBuffer(max_proto_len) 
    except OpENBufferSizeError:
      print("get_first_routing_protocol_name: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("get_first_routing_protocol_name: getStringBuffer raised TypeError")
      return

    proto_buff        = OpEN.open_buffdesc()
    proto_buff.pstart = proto_buff_string
    proto_buff.size   = max_proto_len

    proto_id = 0
    next_proto_id_p = OpEN.new_uint32_tp()

    for idx in xrange(0, XX_PROTO_MAX) :
      result = OpEN.openapiIpRouterProtoNameNextGet(client_handle, proto_id, proto_buff, next_proto_id_p)
      # Get the first available protocol name
      if result == OpEN.OPEN_E_NONE and (len(proto_buff_string.cast())>0):
        return proto_buff_string.cast()
      proto_id = OpEN.uint32_tp_value(next_proto_id_p)
    OpEN.delete_uint32_tp(next_proto_id_p)
  
  OpEN.delete_uint32_tp(max_proto_len_p)

  return ''

def get_first_routing_interface(client_handle):
  """ Get the first available routing interface. """

  intf = 0
  intf_p = OpEN.new_uint32_tp()

  result = OpEN.openapiRtrIntfNextGet(client_handle, intf, intf_p)
  intf = OpEN.uint32_tp_value(intf_p)

  OpEN.delete_uint32_tp(intf_p)
  return result, intf

class BgpExample :
  """Simple BGP class implementing basic CRUD examples """

  def __init__(self, client) :
    self.m_client = client
        
  def test_local_as(self, asn) :
    """ Sets the local BGP autonomous system number to the given value.
    The value is then read and compared to verify the operation. """

    result = OpEN.openapiBgpLocalASSet (self.m_client, asn)
    print_bad_result(result, 'openapiBgpLocalASSet')

    tmp_p = OpEN.new_uint32_tp()

    if result == OpEN.OPEN_E_NONE:
      # Fetch for demonstration
      result = OpEN.openapiBgpLocalASGet (self.m_client, tmp_p)
      print_bad_result(result, 'openapiBgpLocalASGet')

    print_sanity_results(result, asn==OpEN.uint32_tp_value(tmp_p), 'local asn', str(asn))
    OpEN.delete_uint32_tp(tmp_p)

  def test_local_id(self, address) :
    """ Sets the BGP router id to some arbitrary address.
    The value is then read and compared to verify the operation. """

    addr_p = OpEN.new_uint32_tp()
    addr = ip_to_int(address)

    result = OpEN.openapiBgpLocalIdSet (self.m_client, addr)
    print_bad_result(result, 'openapiBgpLocalIdSet')

    if result == OpEN.OPEN_E_NONE:
      # Fetch for demonstration
      result = OpEN.openapiBgpLocalIdGet (self.m_client, addr_p)
      print_bad_result(result, 'openapiBgpLocalASGet')

    print_sanity_results(result, addr == OpEN.uint32_tp_value(addr_p), 'local id', address) 
    OpEN.delete_uint32_tp(addr_p)

  def test_local_pref(self, preference) :
    """ Sets the BGP default local preference to the given value.
    The value is then read and compared to verify the operation. """

    result = OpEN.openapiBgpLocalPrefSet (self.m_client, preference)
    print_bad_result(result, 'openapiBgpLocalPrefSet')
    tmp_p = OpEN.new_uint32_tp()

    if result == OpEN.OPEN_E_NONE:
      # Fetch for demonstration
      result = OpEN.openapiBgpLocalPrefGet (self.m_client, tmp_p)
      print_bad_result(result, 'openapiBgpLocalPrefGet')

    print_sanity_results(result, preference == OpEN.uint32_tp_value(tmp_p), 'local pref', str(preference))
    OpEN.delete_uint32_tp(tmp_p)

  def test_distance(self, external_distance, internal_distance, local_distance) :
    """ Sets the BGP distance for external, internal, and local to some arbitrary value.
    The value is then read and compared to verify the operation. """

    result = OpEN.openapiIpRouterPreferenceSet (self.m_client, OpEN.OPEN_PREF_EBGP, external_distance)
    print_bad_result(result, 'openapiIpRouterPreferenceSet EBGP')
    tmp_p = OpEN.new_uint32_tp()

    if result == OpEN.OPEN_E_NONE:
      # Fetch for demonstration
      result = OpEN.openapiIpRouterPreferenceGet (self.m_client, OpEN.OPEN_PREF_EBGP, tmp_p)
      print_bad_result(result, 'openapiIpRouterPreferenceGet EBGP')

    print_sanity_results(result, external_distance == OpEN.uint32_tp_value(tmp_p), 'external distance', str(external_distance))
    OpEN.delete_uint32_tp(tmp_p)

    result = OpEN.openapiIpRouterPreferenceSet (self.m_client, OpEN.OPEN_PREF_IBGP, internal_distance)
    print_bad_result(result, 'openapiIpRouterPreferenceSet IBGP')
    tmp_p = OpEN.new_uint32_tp()

    if result == OpEN.OPEN_E_NONE:
      # Fetch for demonstration
      result = OpEN.openapiIpRouterPreferenceGet (self.m_client, OpEN.OPEN_PREF_IBGP, tmp_p)
      print_bad_result(result, 'openapiIpRouterPreferenceGet IBGP')

    print_sanity_results(result, internal_distance == OpEN.uint32_tp_value(tmp_p), 'internal distance', str(internal_distance))
    OpEN.delete_uint32_tp(tmp_p)

    result = OpEN.openapiIpRouterPreferenceSet (self.m_client, OpEN.OPEN_PREF_LOCAL_BGP, local_distance)
    print_bad_result(result, 'openapiIpRouterPreferenceSet LOCAL')
    tmp_p = OpEN.new_uint32_tp()

    if result == OpEN.OPEN_E_NONE:
      # Fetch for demonstration
      result = OpEN.openapiIpRouterPreferenceGet (self.m_client, OpEN.OPEN_PREF_LOCAL_BGP, tmp_p)
      print_bad_result(result, 'openapiIpRouterPreferenceGet LOCAL')

    print_sanity_results(result, local_distance == OpEN.uint32_tp_value(tmp_p), 'local distance', str(local_distance))
    OpEN.delete_uint32_tp(tmp_p)

  def test_global_hold_time(self, seconds) :
    """ Set the configured hold time for a given BGP peer.
    The time is then read and compared to verify the operation. """

    result = OpEN.openapiBgpGlobalHoldTimeConfiguredSet (self.m_client, seconds)
    print_bad_result(result, 'openapiBgpGlobalHoldTimeConfiguredSet')
    tmp_p = OpEN.new_uint32_tp()

    if result == OpEN.OPEN_E_NONE:
      # Fetch for demonstration
      result = OpEN.openapiBgpGlobalHoldTimeConfiguredGet (self.m_client, tmp_p)
      print_bad_result(result, 'openapiBgpGlobalHoldTimeConfiguredGet')

    print_sanity_results(result, seconds==OpEN.uint32_tp_value(tmp_p), 'global hold time', str(seconds))
    OpEN.delete_uint32_tp(tmp_p)

  def test_global_keep_alive(self, seconds) :
    """ Set the configured keep alive time for a given BGP peer.
    The time is then read and compared to verify the operation. """

    result = OpEN.openapiBgpGlobalKeepAliveConfiguredSet (self.m_client, seconds)
    print_bad_result(result, 'openapiBgpGlobalKeepAliveConfiguredSet')
    tmp_p = OpEN.new_uint32_tp()

    if result == OpEN.OPEN_E_NONE:
      # Fetch for demonstration
      result = OpEN.openapiBgpGlobalKeepAliveConfiguredGet (self.m_client, tmp_p)
      print_bad_result(result, 'openapiBgpGlobalKeepAliveConfiguredGet')

    print_sanity_results(result, seconds==OpEN.uint32_tp_value(tmp_p), 'global keep alive time', str(seconds))
    OpEN.delete_uint32_tp(tmp_p)

  def test_log_neighbor(self, log_changes) :
    """ Sets the configuration indicating BGP to log neighbor changes.
    The value is then read and compared to verify the operation. """

    result = OpEN.openapiBgpLogNeighborChangesSet (self.m_client, log_changes)
    print_bad_result(result, 'openapiBgpLogNeighborChangesSet')
    tmp_p = OpEN.new_bool_tp()

    if result == OpEN.OPEN_E_NONE:
      # Fetch for demonstration
      result = OpEN.openapiBgpLogNeighborChangesGet (self.m_client, tmp_p)
      print_bad_result(result, 'openapiBgpLogNeighborChangesGet')

    print_sanity_results(result, log_changes==OpEN.bool_tp_value(tmp_p), 'log neighbor', str(log_changes))
    OpEN.delete_bool_tp(tmp_p)

  def test_max_paths(self, af, is_ibgp) :
    """ Set the maximum number of paths that BGP can report.
    The value is then read and compared to verify the operation. """

    tmp_p = OpEN.new_uint32_tp()
    max_p = OpEN.new_uint32_tp()

    # Get max number of paths for the active template
    result = OpEN.openapiBgpMaxPathsGet(self.m_client,  af, is_ibgp, max_p)
    print_bad_result(result, 'openapiBgpMaxPathsGet')

    if result == OpEN.OPEN_E_NONE:
      result = OpEN.openapiBgpMaxPathsSet(self.m_client, af, is_ibgp, OpEN.uint32_tp_value(max_p))
      print_bad_result(result, 'openapiBgpMaxPathsSet')

    if result == OpEN.OPEN_E_NONE:
      # Fetch for demonstration
      result = OpEN.openapiBgpMaxPathsGet(self.m_client, af, is_ibgp, tmp_p)
      print_bad_result(result, 'openapiBgpMaxPathsGet')

    max_v =  OpEN.uint32_tp_value(max_p)
    if is_ibgp:
      bgp = 'Internal (%d)' % max_v
    else:
      bgp = 'External (%d)' % max_v

    print_sanity_results(result, max_v == OpEN.uint32_tp_value(tmp_p), 'max paths', bgp)
    OpEN.delete_uint32_tp(tmp_p)
    OpEN.delete_uint32_tp(max_p)

  def test_network(self, normal_mode, use_route_map, address, prefix, route_map_name) :
    """ Configures an arbitrary network to advertise via BGP. """

    result, addr, type = ip_to_open_inet(address)
    msg = type + ' peer network'

    if addr.family != OpEN.OPEN_AF_NONE:
      try:
        name_string         = open_.getStringBuffer(len(route_map_name) + 1, route_map_name)
      except OpENBufferSizeError:
        print("test_network: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_network: getStringBuffer raised TypeError")
        return
      name_buff             = OpEN.open_buffdesc()
      name_buff.pstart      = name_string
      name_buff.size        = len(route_map_name) + 1
      result = OpEN.openapiBgpNetworkAddDelete(self.m_client,
                                               normal_mode,
                                               use_route_map,
                                               addr,
                                               prefix,
                                               name_buff)
      print_bad_result(result, 'openapiBgpNetworkAddDelete')

    print_sanity_results(result, True, msg, address)

  def test_redistribution(self, add, match_set, match_bits, metric_set,
    metric_value_set, metric_value, route_map_set, af, route_map_name) :
    """ Configures an arbitrary BGP route redistribution. """

    result = OpEN.OPEN_E_FAIL

    proto_name = get_first_routing_protocol_name(self.m_client)
    if len(proto_name)>0:
      try:
        proto_string      = open_.getStringBuffer(len(proto_name) + 1, proto_name)
      except OpENBufferSizeError:
        print("test_redistribution: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_redistribution: getStringBuffer raised TypeError")
        return
      proto_buff        = OpEN.open_buffdesc()
      proto_buff.pstart = proto_string
      proto_buff.size   = len(proto_name) + 1

      try:
        route_string      = open_.getStringBuffer(len(route_map_name) + 1, route_map_name)
      except OpENBufferSizeError:
        print("test_redistribution: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_redistribution: getStringBuffer raised TypeError")
        return

      route_buff        = OpEN.open_buffdesc()
      route_buff.pstart = route_string
      route_buff.size   = len(route_map_name) + 1

      result = OpEN.openapiBgpRedistributionSet(self.m_client,
                                                add,
                                                proto_buff,
                                                match_set,
                                                match_bits,
                                                metric_set,
                                                metric_value_set,
                                                metric_value,
                                                route_map_set,
                                                af,
                                                route_buff)
      print_bad_result(result, 'openapiBgpRedistributionSet')

    print_sanity_results(result, True, 'redistribution', route_map_name)

  def test_peer_remote_as(self, asn, address, scope_id) :
    """ Create a peer with a given IP address and remote AS number.
    The remote AS number is then read and compared to verify the operation."""

    result, addr, type = ip_to_open_inet(address)
    msg = type + ' peer remote asn %d scope %d' % (asn, scope_id)

    if addr.family != OpEN.OPEN_AF_NONE:
      result = OpEN.openapiBgpPeerRemoteASSet(self.m_client,
                                              addr, 
                                              scope_id,
                                              asn)
      print_bad_result(result, 'openapiBgpPeerRemoteASSet')

    tmp_p = OpEN.new_uint32_tp()
    if result == OpEN.OPEN_E_NONE:
      result = OpEN.openapiBgpPeerRemoteASGet(self.m_client, 
                                              addr, 
                                              scope_id, 
                                              tmp_p)
      print_bad_result(result, 'openapiBgpPeerRemoteASGet')

    print_sanity_results(result, asn == OpEN.uint32_tp_value(tmp_p), msg, address)
    OpEN.delete_uint32_tp(tmp_p)

  def test_peer_admin_status(self, address, scope_id, status, def_value) :
    """ Create a peer with a given IP address and remote AS number.
    The remote AS number is then read and compared to verify the operation."""

    result, addr, type = ip_to_open_inet(address)
    msg = type + ' peer admin status'

    if addr.family != OpEN.OPEN_AF_NONE:
      result = OpEN.openapiBgpPeerAdminStatusSet(self.m_client,
                                                 addr, 
                                                 scope_id,
                                                 status, 
                                                 def_value)
      print_bad_result(result, 'openapiBgpPeerAdminStatusSet')

    tmp_p = OpEN.new_OPEN_BGP_PEER_STATE_tp()
    if result == OpEN.OPEN_E_NONE:
      result = OpEN.openapiBgpPeerAdminStatusGet(self.m_client, 
                                              OpEN.OPEN_BGP_GET_FINAL,
                                              addr, 
                                              scope_id, 
                                              tmp_p)
      print_bad_result(result, 'openapiBgpPeerAdminStatusGet')

    print_sanity_results(result, status == OpEN.OPEN_BGP_PEER_STATE_tp_value(tmp_p), msg, status)
    OpEN.delete_OPEN_BGP_PEER_STATE_tp(tmp_p)

  def test_peer_keep_alive(self, address, scope_id, time, def_value) :
    """ Set the configured keep alive time for a given BGP peer.
    The time is then read and compared to verify the operation."""

    result, addr, type = ip_to_open_inet(address)
    msg = type + ' peer keep alive'

    if addr.family != OpEN.OPEN_AF_NONE:
      result = OpEN.openapiBgpPeerKeepAliveConfiguredSet(self.m_client,
                                                         addr, 
                                                         scope_id,
                                                         time, 
                                                         def_value)
      print_bad_result(result, 'openapiBgpPeerKeepAliveConfiguredSet')

    tmp_p = OpEN.new_uint32_tp()
    if result == OpEN.OPEN_E_NONE:
      result = OpEN.openapiBgpPeerKeepAliveConfiguredGet(self.m_client, 
                                                         OpEN.OPEN_BGP_GET_FINAL,
                                                         addr, 
                                                         scope_id, 
                                                         tmp_p)
      print_bad_result(result, 'openapiBgpPeerKeepAliveConfiguredGet')

    print_sanity_results(result, time == OpEN.uint32_tp_value(tmp_p), msg, time)
    OpEN.delete_uint32_tp(tmp_p)

  def test_peer_hold_time(self, address, scope_id, time, def_value) :
    """ Set the configured hold time for a given BGP peer.
    The time is then read and compared to verify the operation."""

    result, addr, type = ip_to_open_inet(address)
    msg = type + ' peer hold time'

    if addr.family != OpEN.OPEN_AF_NONE:
      result = OpEN.openapiBgpPeerHoldTimeConfiguredSet(self.m_client,
                                                        addr, 
                                                        scope_id,
                                                        time, 
                                                        def_value)
      print_bad_result(result, 'openapiBgpPeerHoldTimeConfiguredSet')

    tmp_p = OpEN.new_uint32_tp()
    if result == OpEN.OPEN_E_NONE:
      result = OpEN.openapiBgpPeerHoldTimeConfiguredGet(self.m_client, 
                                                        OpEN.OPEN_BGP_GET_FINAL,
                                                        addr, 
                                                        scope_id, 
                                                        tmp_p)
      print_bad_result(result, 'openapiBgpPeerHoldTimeConfiguredGet')

    print_sanity_results(result, time == OpEN.uint32_tp_value(tmp_p), msg, time)
    OpEN.delete_uint32_tp(tmp_p)

  def test_peer_activate(self, af, address, scope_id, activate, def_value) :
    """ Configure peer to advertise and accept routes for the given address family."""

    result, addr, type = ip_to_open_inet(address)
    msg = type + ' peer activate'

    if addr.family != OpEN.OPEN_AF_NONE:
      result = OpEN.openapiBgpPeerActivateSet(self.m_client,
                                              addr, 
                                              scope_id, 
                                              af,
                                              activate,
                                              def_value)
      print_bad_result(result, 'openapiBgpPeerActivateSet')

    tmp_p = OpEN.new_bool_tp()
    if result == OpEN.OPEN_E_NONE:
      result = OpEN.openapiBgpPeerActivateGet(self.m_client, 
                                              OpEN.OPEN_BGP_GET_FINAL,
                                              addr, 
                                              scope_id, 
                                              af,
                                              tmp_p)
      print_bad_result(result, 'openapiBgpPeerActivateGet')

    print_sanity_results(result, activate == OpEN.bool_tp_value(tmp_p), msg, activate)
    OpEN.delete_bool_tp(tmp_p)

  def test_peer_advertisement(self, af, address, scope_id, interval, def_value) :
    """ Set the advertisment interval for a given BGP peer.
    The interval is then read and compared to verify the operation."""

    result, addr, type = ip_to_open_inet(address)
    msg = type + ' peer advertisement interval'

    if addr.family != OpEN.OPEN_AF_NONE:
      result = OpEN.openapiBgpPeerAdvertisementIntervalSet(self.m_client,
                                                           addr, 
                                                           scope_id, 
                                                           af,
                                                           interval,
                                                           def_value)
      print_bad_result(result, 'openapiBgpPeerAdvertisementIntervalSet')

    tmp_p = OpEN.new_uint32_tp()
    if result == OpEN.OPEN_E_NONE:
      result = OpEN.openapiBgpPeerAdvertisementIntervalGet(self.m_client, 
                                              OpEN.OPEN_BGP_GET_FINAL,
                                              addr, 
                                              scope_id, 
                                              af,
                                              tmp_p)
      print_bad_result(result, 'openapiBgpPeerActivateGet')

    print_sanity_results(result, interval == OpEN.uint32_tp_value(tmp_p), msg, interval)
    OpEN.delete_uint32_tp(tmp_p)

  def test_peer_next_hop_self(self, af, address, scope_id, enable, def_value) :
    """ Configure BGP to use a local address when advertising routes to a given internal peer.
    The flag is then read and compared to verify the operation."""

    result, addr, type = ip_to_open_inet(address)
    msg = type + ' peer next hop'

    if addr.family != OpEN.OPEN_AF_NONE:
      result = OpEN.openapiBgpPeerNextHopSelfModeSet(self.m_client,
                                                     addr, 
                                                     scope_id, 
                                                     af,
                                                     enable,
                                                     def_value)
      print_bad_result(result, 'openapiBgpPeerNextHopSelfModeSet')

    tmp_p = OpEN.new_OPEN_CONTROL_tp()
    if result == OpEN.OPEN_E_NONE:
      result = OpEN.openapiBgpPeerNextHopSelfModeGet(self.m_client, 
                                                     OpEN.OPEN_BGP_GET_FINAL,
                                                     addr, 
                                                     scope_id, 
                                                    af,
                                                    tmp_p)
      print_bad_result(result, 'openapiBgpPeerNextHopSelfModeGet')

    print_sanity_results(result, enable == OpEN.OPEN_CONTROL_tp_value(tmp_p), msg, enable)
    OpEN.delete_OPEN_CONTROL_tp(tmp_p)

  def test_peer_pfx_limit(self, af, address, scope_id, threshold, warning_only, def_value) :
    """ Set the prefix limit configured for a given peer.
    In this example, the maximum prefix limit is retrieved from the switch based
    on the IP address family. This max value and the given threshold, and
    warningOnly flag is set and read back for comparison."""

    result, addr, type = ip_to_open_inet(address)
    msg = type + ' peer prefix limit'
    limit = 0

    if addr.family != OpEN.OPEN_AF_NONE:
      if addr.family == OpEN.OPEN_AF_INET:
        limit_p = OpEN.new_uint32_tp()
        OpEN.openapiBgpIpv4RouteMax(self.m_client, limit_p)
        # Demonstrate get/set the maximum number of IPv4 routes
        limit = OpEN.uint32_tp_value(limit_p)
      else:
        # Demonstrate alternate method to use the maximum number routes
        limit = OpEN.OPEN_BGP_NBR_MAX_PFX_NOLIMIT

      result = OpEN.openapiBgpPeerPfxLimitSet(self.m_client,
                                              addr, 
                                              scope_id, 
                                              af,
                                              limit,
                                              threshold,
                                              warning_only,
                                              def_value)
      print_bad_result(result, 'openapiBgpPeerPfxLimitSet')

    limit_p     = OpEN.new_uint32_tp()
    threshold_p = OpEN.new_uint32_tp()
    warning_p   = OpEN.new_bool_tp()
    if result == OpEN.OPEN_E_NONE:
      result = OpEN.openapiBgpPeerPfxLimitGet(self.m_client, 
                                              OpEN.OPEN_BGP_GET_FINAL,
                                              addr, 
                                              scope_id, 
                                              af,
                                              limit_p,
                                              threshold_p,
                                              warning_p)
      print_bad_result(result, 'openapiBgpPeerPfxLimitGet')

    test = (limit == OpEN.uint32_tp_value(limit_p) and
           threshold == OpEN.uint32_tp_value(threshold_p) and
           warning_only == OpEN.bool_tp_value(warning_p))

    print_sanity_results(result, test, msg, '%d, %d, ' % (limit, threshold) + str(warning_only))
    OpEN.delete_uint32_tp(limit_p)
    OpEN.delete_uint32_tp(threshold_p)
    OpEN.delete_bool_tp(warning_p)


  def test_peer_update_source(self, address, scope_id, source, def_value) :
    """ Set the interface whose IP address that BGP uses as the source
    IP address in packets sent to a given peer. The interface is
    then read and compared to verify the operation."""

    result, addr, type = ip_to_open_inet(address)
    msg = type + ' peer remote source scope %d source %d' % (scope_id, source)

    if addr.family != OpEN.OPEN_AF_NONE:
      result = OpEN.openapiBgpPeerUpdateSourceSet(self.m_client,
                                                     addr, 
                                                     scope_id, 
                                                     source,
                                                     def_value)
      print_bad_result(result, 'openapiBgpPeerUpdateSourceSet')

    tmp_p = OpEN.new_uint32_tp()
    if result == OpEN.OPEN_E_NONE:
      result = OpEN.openapiBgpPeerUpdateSourceGet(self.m_client, 
                                              OpEN.OPEN_BGP_GET_FINAL,
                                              addr, 
                                              scope_id, 
                                              tmp_p)
      print_bad_result(result, 'openapiBgpPeerUpdateSourceGet')

    print_sanity_results(result, source == OpEN.uint32_tp_value(tmp_p), msg, address)
    OpEN.delete_uint32_tp(tmp_p)

def main():
  """Demonstrate OpEN usage for BGP APIs"""

  print 'Begin Sanity tests...'
  ret = open_.connect("bgp_example")
  if ret == OpEN.OPEN_E_NONE:
    open_.getNetworkOSVersion()
    client = open_.get_client()
    example = BgpExample(client)
    print 'Establish BGP and global parameters...'
    example.test_local_as(1)
    example.test_local_id('10.1.1.1')
    example.test_local_pref(1234567890)
    example.test_distance(10, 20, 30)
    example.test_global_hold_time(65535)
    example.test_global_keep_alive(65535)
    example.test_log_neighbor(True)
    example.test_max_paths(OpEN.OPEN_AF_INET, False)
    example.test_max_paths(OpEN.OPEN_AF_INET, True)
    example.test_network(True, True, '20.10.0.0', 24, 'my-map1')
    example.test_network(False, True, '20.10.0.0', 24, 'my-map1')
    # Demonstrate internal, external, and NSSA external bit sets
    match_bits = 0
    match_bits |= OpEN.OPEN_OSPF_METRIC_TYPE_INTERNAL
    match_bits |= OpEN.OPEN_OSPF_METRIC_TYPE_EXT1
    match_bits |= OpEN.OPEN_OSPF_METRIC_TYPE_EXT2
    match_bits |= OpEN.OPEN_OSPF_METRIC_TYPE_NSSA_EXT1
    match_bits |= OpEN.OPEN_OSPF_METRIC_TYPE_NSSA_EXT2
    example.test_redistribution(True, True, match_bits, True, True, 4294967295, False, OpEN.OPEN_AF_INET, '')
    # Neighbor specific tests
    print 'Create a ipv4 test peer...'
    example.test_peer_remote_as(1, "20.2.2.2", 0)
    example.test_peer_admin_status("20.2.2.2", 0, OpEN.OPEN_BGP_START, True)
    example.test_peer_keep_alive("20.2.2.2", 0, 600, True)
    example.test_peer_hold_time("20.2.2.2", 0, 700, True)
    example.test_peer_activate(OpEN.OPEN_AF_INET6, "20.2.2.2", 0, True, True)
    example.test_peer_advertisement(OpEN.OPEN_AF_INET, "20.2.2.2", 0, 200, True)
    example.test_peer_next_hop_self(OpEN.OPEN_AF_INET, "20.2.2.2", 0, OpEN.OPEN_ENABLE, True)
    example.test_peer_pfx_limit(OpEN.OPEN_AF_INET, "20.2.2.2", 0, 20, True, True)
    print 'Create a ipv6 test peer...'
    example.test_peer_remote_as(1, "2222::", 0)
    example.test_peer_admin_status("2222::", 0, OpEN.OPEN_BGP_STOP, True)
    example.test_peer_keep_alive("2222::", 0, 600, True)
    example.test_peer_hold_time("2222::", 0, 700, True)
    example.test_peer_activate(OpEN.OPEN_AF_INET6, "2222::", 0, True, True)
    example.test_peer_advertisement(OpEN.OPEN_AF_INET6, "2222::", 0, 222, True)
    example.test_peer_next_hop_self(OpEN.OPEN_AF_INET6, "2222::", 0, OpEN.OPEN_ENABLE, True)
    example.test_peer_pfx_limit(OpEN.OPEN_AF_INET6, "2222::", 0, 60, True, True)
    # Attempt to retrieve a routing interface
    result, intf = get_first_routing_interface(client)
    if result == OpEN.OPEN_E_NONE:
      # Configure remote as link local address
      example.test_peer_remote_as(1, "fe80::1", intf)
      # Configure update source using same routing interface
      example.test_peer_update_source("20.2.2.2", 0, intf, True)
    else:
      print '\nSanity tests are incomplete because no routing interfaces are available!'
      print 'Please configure a routing interface and re-run test.'
    print '\nComplete'

    open_.terminate()
  else :
    print "Unable to connect"

if __name__ == '__main__': main()

