#!/mnt/fastpath/usr/bin/ruby

require "OpEN"
require "OpENUtil"
require "socket"
require "ipaddr"

#
# Copyright 2016 Broadcom.
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#


#
# Ruby 1.8.7.
#

$open = OpENUtil.new()

# Maximum number of protocol names
XX_PROTO_MAX = 10

def ip_to_open_inet(address)
  #Convert ip address to integer and family and return as open_inet_addr_t

  addr = OpEN::Open_inet_addr_t.new
  result = OpEN::OPEN_E_PARAM

  ipaddr = IPAddr.new address
  family = ipaddr.family

  if family == Socket::AF_INET
    addr.addr.ipv4 = IPAddr.new(ipaddr.to_s).to_i
    addr.family = OpEN::OPEN_AF_INET
    result = OpEN::OPEN_E_NONE
    type = 'ipv4'
  else
    begin
      addr.addr.ipv6 = IPAddr.new(ipaddr.to_s).to_i
      addr.family = OpEN::OPEN_AF_INET6
      result = OpEN::OPEN_E_NONE
    rescue
      result = OpEN::OPEN_E_UNAVAIL
    ensure
      type = 'ipv6'
    end
  end

  return result, addr, type
end

def print_sanity_results(result, test, msg, feat)
  #Print overall comparison results

  if result == OpEN::OPEN_E_UNAVAIL
    puts "Sanity test skipped."
  elsif result == OpEN::OPEN_E_NONE and test == true
    puts "Sanity Success - #{msg} - #{feat}."
  else
    puts "Sanity Failure - #{msg} - #{feat}."
  end
end

def print_bad_result(result, msg)
  #Print some general error messages if the result is bad
   
  if result == OpEN::OPEN_E_UNAVAIL
    puts "Feature not supported - #{msg} (err #{result})."
  elsif result != OpEN::OPEN_E_NONE
    puts "Test Failure - #{msg} (err #{result})."
  end
end

def get_first_routing_protocol_name(client_handle)
  #Get the first available routing protocol name.

  max_proto_len_p = OpEN.new_uint32_tp()

  result = OpEN.openapiRouteProtoNameLenMax(client_handle, max_proto_len_p)
  print_bad_result(result, 'openapiRouteProtoNameLenMax')
  if result == OpEN::OPEN_E_NONE
    max_proto_len = OpEN.uint32_tp_value(max_proto_len_p) + 1
    proto_buff_string = $open.getCharBuffer(max_proto_len) 
    proto_buff        = OpEN::Open_buffdesc.new
    proto_buff.pstart = proto_buff_string
    proto_buff.size   = max_proto_len

    proto_id = 0
    next_proto_id_p = OpEN.new_uint32_tp()

    for idx in 0..XX_PROTO_MAX
      result = OpEN.openapiIpRouterProtoNameNextGet(client_handle, proto_id, proto_buff, next_proto_id_p)
      # Get the first available protocol name
      if (result == OpEN::OPEN_E_NONE and (proto_buff_string.cast().length>0))
        OpEN.delete_uint32_tp(max_proto_len_p)
        OpEN.delete_uint32_tp(next_proto_id_p)
        return proto_buff_string.cast()
      end
      proto_id = OpEN.uint32_tp_value(next_proto_id_p)
    end
  end

  OpEN.delete_uint32_tp(max_proto_len_p)
  OpEN.delete_uint32_tp(next_proto_id_p) 
  return ''
end

def get_first_routing_interface(client)
  #Get the first available routing interface.

  intf = 0
  intf_p = OpEN.new_uint32_tp()

  result = OpEN.openapiRtrIntfNextGet(client, intf, intf_p)
  intf = OpEN.uint32_tp_value(intf_p)

  OpEN.delete_uint32_tp(intf_p)
  return result, intf
end

class BgpExample
  #Simple BGP class implementing basic CRUD examples

  def initialize(client)
    @client = client
  end

  def test_local_as(asn)
    #Sets the local BGP autonomous system number to the given value.
    #The value is then read and compared to verify the operation.

    result = OpEN.openapiBgpLocalASSet(@client, asn)
    print_bad_result(result, 'openapiBgpLocalASSet')

    if result == OpEN::OPEN_E_NONE
      # Fetch for demonstration
      tmp_p = OpEN.new_uint32_tp()
      result = OpEN.openapiBgpLocalASGet(@client, tmp_p)
      print_bad_result(result, 'openapiBgpLocalASGet')
    end

    print_sanity_results(result, (asn==OpEN.uint32_tp_value(tmp_p)), 'local asn', asn)
    OpEN.delete_uint32_tp(tmp_p)
  end

  def test_local_id(address)
    #Sets the BGP router id to some arbitrary address.
    #The value is then read and compared to verify the operation.

    addr = IPAddr.new(address).to_i

    result = OpEN.openapiBgpLocalIdSet(@client, addr)
    print_bad_result(result, 'openapiBgpLocalIdSet')

    if result == OpEN::OPEN_E_NONE
      # Fetch for demonstration
      addr_p = OpEN.new_uint32_tp()
      result = OpEN.openapiBgpLocalIdGet(@client, addr_p)
      print_bad_result(result, 'openapiBgpLocalASGet')
    end

    print_sanity_results(result, addr == OpEN.uint32_tp_value(addr_p), 'local id', address)
    OpEN.delete_uint32_tp(addr_p)
  end

  def test_local_pref(preference)
    #Sets the BGP default local preference to the given value.
    #The value is then read and compared to verify the operation.

    result = OpEN.openapiBgpLocalPrefSet(@client, preference)
    print_bad_result(result, 'openapiBgpLocalPrefSet')

    if result == OpEN::OPEN_E_NONE
      # Fetch for demonstration
      tmp_p = OpEN.new_uint32_tp()
      result = OpEN.openapiBgpLocalPrefGet(@client, tmp_p)
      print_bad_result(result, 'openapiBgpLocalPrefGet')
    end

    print_sanity_results(result, (preference == OpEN.uint32_tp_value(tmp_p)), 'local pref', preference)
    OpEN.delete_uint32_tp(tmp_p)
  end

  def test_distance(external_distance, internal_distance, local_distance)
    #Sets the BGP distance for external, internal, and local to some arbitrary value.
    #The value is then read and compared to verify the operation.

    result = OpEN.openapiIpRouterPreferenceSet(@client, OpEN::OPEN_PREF_EBGP, external_distance)
    print_bad_result(result, 'openapiIpRouterPreferenceSet EBGP')

    if result == OpEN::OPEN_E_NONE
      # Fetch for demonstration
      tmp_p = OpEN.new_uint32_tp()
      result = OpEN.openapiIpRouterPreferenceGet(@client, OpEN::OPEN_PREF_EBGP, tmp_p)
      print_bad_result(result, 'openapiIpRouterPreferenceGet EBGP')
    end

    print_sanity_results(result, (external_distance == OpEN.uint32_tp_value(tmp_p)), 'external distance', external_distance)

    result = OpEN.openapiIpRouterPreferenceSet(@client, OpEN::OPEN_PREF_IBGP, internal_distance)
    print_bad_result(result, 'openapiIpRouterPreferenceSet IBGP')

    if result == OpEN::OPEN_E_NONE
      # Fetch for demonstration
      tmp_p = OpEN.new_uint32_tp()
      result = OpEN.openapiIpRouterPreferenceGet(@client, OpEN::OPEN_PREF_IBGP, tmp_p)
      print_bad_result(result, 'openapiIpRouterPreferenceGet IBGP')
    end

    print_sanity_results(result, (internal_distance == OpEN.uint32_tp_value(tmp_p)), 'internal distance', internal_distance)

    result = OpEN.openapiIpRouterPreferenceSet(@client, OpEN::OPEN_PREF_LOCAL_BGP, local_distance)
    print_bad_result(result, 'openapiIpRouterPreferenceSet LOCAL')

    if result == OpEN::OPEN_E_NONE
      # Fetch for demonstration
      tmp_p = OpEN.new_uint32_tp()
      result = OpEN.openapiIpRouterPreferenceGet(@client, OpEN::OPEN_PREF_LOCAL_BGP, tmp_p)
      print_bad_result(result, 'openapiIpRouterPreferenceGet LOCAL')
    end

    print_sanity_results(result, (local_distance == OpEN.uint32_tp_value(tmp_p)), 'local distance', local_distance)
    OpEN.delete_uint32_tp(tmp_p)
  end

  def test_global_hold_time(seconds)
    #Set the configured hold time for a given BGP peer.
    #The time is then read and compared to verify the operation.

    result = OpEN.openapiBgpGlobalHoldTimeConfiguredSet(@client, seconds)
    print_bad_result(result, 'openapiBgpGlobalHoldTimeConfiguredSet')

    if result == OpEN::OPEN_E_NONE
      # Fetch for demonstration
      tmp_p = OpEN.new_uint32_tp()
      result = OpEN.openapiBgpGlobalHoldTimeConfiguredGet(@client, tmp_p)
      print_bad_result(result, 'openapiBgpGlobalHoldTimeConfiguredGet')
    end

    print_sanity_results(result, (seconds==OpEN.uint32_tp_value(tmp_p)), 'global hold time', seconds)
    OpEN.delete_uint32_tp(tmp_p)
  end

  def test_global_keep_alive(seconds)
    #Set the configured keep alive time for a given BGP peer.
    #The time is then read and compared to verify the operation.

    result = OpEN.openapiBgpGlobalKeepAliveConfiguredSet(@client, seconds)
    print_bad_result(result, 'openapiBgpGlobalKeepAliveConfiguredSet')

    if result == OpEN::OPEN_E_NONE
      # Fetch for demonstration
      tmp_p = OpEN.new_uint32_tp()
      result = OpEN.openapiBgpGlobalKeepAliveConfiguredGet(@client, tmp_p)
      print_bad_result(result, 'openapiBgpGlobalKeepAliveConfiguredGet')
    end

    print_sanity_results(result, (seconds==OpEN.uint32_tp_value(tmp_p)), 'global keep alive time', seconds)
    OpEN.delete_uint32_tp(tmp_p)
  end

  def test_log_neighbor(log_changes)
    #Sets the configuration indicating BGP to log neighbor changes.
    #The value is then read and compared to verify the operation.

    result = OpEN.openapiBgpLogNeighborChangesSet(@client, log_changes)
    print_bad_result(result, 'openapiBgpLogNeighborChangesSet')

    if result == OpEN::OPEN_E_NONE
      # Fetch for demonstration
      tmp_p = OpEN.new_bool_tp()
      result = OpEN.openapiBgpLogNeighborChangesGet(@client, tmp_p)
      print_bad_result(result, 'openapiBgpLogNeighborChangesGet')
    end

    print_sanity_results(result, (log_changes==OpEN.bool_tp_value(tmp_p)), 'log neighbor', log_changes)
    OpEN.delete_bool_tp(tmp_p)
  end

  def test_max_paths(af, is_ibgp)
    #Set the maximum number of paths that BGP can report.
    #The value is then read and compared to verify the operation.

    # Get max number of paths for the active template
    max_p = OpEN.new_uint32_tp()
    tmp_p = OpEN.new_uint32_tp()
    result = OpEN.openapiBgpMaxPathsGet(@client, af, is_ibgp, max_p)
    print_bad_result(result, 'openapiBgpMaxPathsGet')
    
    if result == OpEN::OPEN_E_NONE
      result = OpEN.openapiBgpMaxPathsSet(@client, af, is_ibgp, OpEN.uint32_tp_value(max_p))
      print_bad_result(result, 'openapiBgpMaxPathsSet')
    end

    if result == OpEN::OPEN_E_NONE
      # Fetch for demonstration
      result = OpEN.openapiBgpMaxPathsGet(@client, af, is_ibgp, tmp_p)
      print_bad_result(result, 'openapiBgpMaxPathsGet')
    end

    max_v =  OpEN.uint32_tp_value(max_p)
    if is_ibgp
      bgp = "Internal (%d)" % max_v
    else
      bgp = "External (%d)" % max_v
    end

    print_sanity_results(result, (max_v == OpEN.uint32_tp_value(tmp_p)), 'max paths', bgp)
    OpEN.delete_uint32_tp(max_p)
  end

  def test_network(normal_mode, use_route_map, address, prefix, route_map_name)
    #Configures an arbitrary network to advertise via BGP.

    result, addr, type = ip_to_open_inet(address)
    msg = type + " peer network"

    if addr.family != OpEN::OPEN_AF_NONE
      name_string      = $open.getCharBuffer(route_map_name.length, route_map_name)
      name_buff        = OpEN::Open_buffdesc.new
      name_buff.pstart = name_string
      name_buff.size   = route_map_name.length
      result = OpEN.openapiBgpNetworkAddDelete(@client,
                                               normal_mode,
                                               use_route_map,
                                               addr,
                                               prefix,
                                               name_buff)
      print_bad_result(result, 'openapiBgpNetworkAddDelete')
    end

    print_sanity_results(result, true, msg, address)
  end

  def test_redistribution(add, match_set, match_bits, metric_set,
    metric_value_set, metric_value, route_map_set, af, route_map_name)
    #Configures an arbitrary BGP route redistribution.

    result = OpEN::OPEN_E_FAIL

    proto_name = get_first_routing_protocol_name(@client)
    if proto_name.length>0
      proto_string      = $open.getCharBuffer(proto_name.length, proto_name)
      proto_buff        = OpEN::Open_buffdesc.new
      proto_buff.pstart = proto_string
      proto_buff.size   = proto_name.length

      route_string      = $open.getCharBuffer(route_map_name.length, route_map_name)
      route_buff        = OpEN::Open_buffdesc.new
      route_buff.pstart = route_string
      route_buff.size   = route_map_name.length

      result = OpEN.openapiBgpRedistributionSet(@client,
                                                add,
                                                proto_buff,
                                                match_set,
                                                match_bits,
                                                metric_set,
                                                metric_value_set,
                                                metric_value,
                                                route_map_set,
                                                af,
                                                route_buff)
      print_bad_result(result, 'openapiBgpRedistributionSet')
    end

    print_sanity_results(result, true, 'redistribution', route_map_name)
  end

  def test_peer_remote_as(asn, address, scope_id)
    #Create a peer with a given IP address and remote AS number.
    #The remote AS number is then read and compared to verify the operation.

    result, addr, type = ip_to_open_inet(address)
    msg = type + ' peer remote asn %d scope %d' % [asn, scope_id]

    if addr.family != OpEN::OPEN_AF_NONE
      result = OpEN.openapiBgpPeerRemoteASSet(@client,
                                              addr, 
                                              scope_id,
                                              asn)
      print_bad_result(result, 'openapiBgpPeerRemoteASSet')
    end

    tmp_p = OpEN.new_uint32_tp()
    if result == OpEN::OPEN_E_NONE
      result = OpEN.openapiBgpPeerRemoteASGet(@client,
                                              addr, 
                                              scope_id, 
                                              tmp_p)
      print_bad_result(result, 'openapiBgpPeerRemoteASGet')
    end

    print_sanity_results(result, (asn == OpEN.uint32_tp_value(tmp_p)), msg, address)
    OpEN.delete_uint32_tp(tmp_p)
  end

  def test_peer_admin_status(address, scope_id, status, def_value)
    #Create a peer with a given IP address and remote AS number.
    #The remote AS number is then read and compared to verify the operation.

    result, addr, type = ip_to_open_inet(address)
    msg = type + ' peer admin status'

    if addr.family != OpEN::OPEN_AF_NONE
      result = OpEN.openapiBgpPeerAdminStatusSet(@client,
                                                 addr, 
                                                 scope_id,
                                                 status, 
                                                 def_value)
      print_bad_result(result, 'openapiBgpPeerAdminStatusSet')
    end

    tmp_p = OpEN.new_OPEN_BGP_PEER_STATE_tp()

    if result == OpEN::OPEN_E_NONE
      result = OpEN.openapiBgpPeerAdminStatusGet(@client, 
                                                 OpEN::OPEN_BGP_GET_FINAL,
                                                 addr, 
                                                 scope_id, 
                                                 tmp_p)
      print_bad_result(result, 'openapiBgpPeerAdminStatusGet')
    end

    print_sanity_results(result, (status == OpEN.OPEN_BGP_PEER_STATE_tp_value(tmp_p)), msg, status)
    OpEN.delete_OPEN_BGP_PEER_STATE_tp(tmp_p)
  end

  def test_peer_keep_alive(address, scope_id, time, def_value)
    #Set the configured keep alive time for a given BGP peer.
    #The time is then read and compared to verify the operation.

    result, addr, type = ip_to_open_inet(address)
    msg = type + ' peer keep alive'

    if addr.family != OpEN::OPEN_AF_NONE
      result = OpEN.openapiBgpPeerKeepAliveConfiguredSet(@client,
                                                         addr, 
                                                         scope_id,
                                                         time, 
                                                         def_value)
      print_bad_result(result, 'openapiBgpPeerKeepAliveConfiguredSet')
    end

    tmp_p = OpEN.new_uint32_tp()
    if result == OpEN::OPEN_E_NONE
      result = OpEN.openapiBgpPeerKeepAliveConfiguredGet(@client, 
                                                         OpEN::OPEN_BGP_GET_FINAL,
                                                         addr, 
                                                         scope_id, 
                                                         tmp_p)
      print_bad_result(result, 'openapiBgpPeerKeepAliveConfiguredGet')
    end

    print_sanity_results(result, (time == OpEN.uint32_tp_value(tmp_p)), msg, time)
    OpEN.delete_uint32_tp(tmp_p)
  end

  def test_peer_hold_time(address, scope_id, time, def_value)
    #Set the configured hold time for a given BGP peer.
    #The time is then read and compared to verify the operation.

    result, addr, type = ip_to_open_inet(address)
    msg = type + ' peer hold time'

    if addr.family != OpEN::OPEN_AF_NONE
      result = OpEN.openapiBgpPeerHoldTimeConfiguredSet(@client,
                                                        addr, 
                                                        scope_id,
                                                        time, 
                                                        def_value)
      print_bad_result(result, 'openapiBgpPeerHoldTimeConfiguredSet')
    end

    tmp_p = OpEN.new_uint32_tp()
    if result == OpEN::OPEN_E_NONE
      result = OpEN.openapiBgpPeerHoldTimeConfiguredGet(@client, 
                                                        OpEN::OPEN_BGP_GET_FINAL,
                                                        addr, 
                                                        scope_id, 
                                                        tmp_p)
      print_bad_result(result, 'openapiBgpPeerHoldTimeConfiguredGet')
    end

    print_sanity_results(result, time == OpEN.uint32_tp_value(tmp_p), msg, time)
    OpEN.delete_uint32_tp(tmp_p)
  end

  def test_peer_activate(af, address, scope_id, activate, def_value)
    #Configure peer to advertise and accept routes for the given address family.

    result, addr, type = ip_to_open_inet(address)
    msg = type + ' peer activate'

    if addr.family != OpEN::OPEN_AF_NONE
      result = OpEN.openapiBgpPeerActivateSet(@client,
                                              addr, 
                                              scope_id, 
                                              af,
                                              activate,
                                              def_value)
      print_bad_result(result, 'openapiBgpPeerActivateSet')
    end

    tmp_p = OpEN.new_bool_tp()
    if result == OpEN::OPEN_E_NONE
      result = OpEN.openapiBgpPeerActivateGet(@client, 
                                              OpEN::OPEN_BGP_GET_FINAL,
                                              addr, 
                                              scope_id, 
                                              af,
                                              tmp_p)
      print_bad_result(result, 'openapiBgpPeerActivateGet')
    end

    print_sanity_results(result, activate == OpEN.bool_tp_value(tmp_p), msg, activate)
    OpEN.delete_bool_tp(tmp_p)
  end

  def test_peer_advertisement(af, address, scope_id, interval, def_value)
    #Set the advertisement interval for a given BGP peer.
    #The interval is then read and compared to verify the operation.

    result, addr, type = ip_to_open_inet(address)
    msg = type + ' peer advertisement interval'

    if addr.family != OpEN::OPEN_AF_NONE
      result = OpEN.openapiBgpPeerAdvertisementIntervalSet(@client,
                                                           addr, 
                                                           scope_id, 
                                                           af,
                                                           interval,
                                                           def_value)
      print_bad_result(result, 'openapiBgpPeerAdvertisementIntervalSet')
    end

    tmp_p = OpEN.new_uint32_tp()
    if result == OpEN::OPEN_E_NONE
      result = OpEN.openapiBgpPeerAdvertisementIntervalGet(@client, 
                                              OpEN::OPEN_BGP_GET_FINAL,
                                              addr, 
                                              scope_id, 
                                              af,
                                              tmp_p)
      print_bad_result(result, 'openapiBgpPeerActivateGet')
    end

    print_sanity_results(result, interval == OpEN.uint32_tp_value(tmp_p), msg, interval)
    OpEN.delete_uint32_tp(tmp_p)
  end

  def test_peer_next_hop_self(af, address, scope_id, enable, def_value)
    #Configure BGP to use a local address when advertising routes to a given internal peer.
    #The flag is then read and compared to verify the operation.

    result, addr, type = ip_to_open_inet(address)
    msg = type + ' peer next hop'

    if addr.family != OpEN::OPEN_AF_NONE
      result = OpEN.openapiBgpPeerNextHopSelfModeSet(@client,
                                                     addr, 
                                                     scope_id, 
                                                     af,
                                                     enable,
                                                     def_value)
      print_bad_result(result, 'openapiBgpPeerNextHopSelfModeSet')
    end

    tmp_p = OpEN.new_OPEN_CONTROL_tp()
    if result == OpEN::OPEN_E_NONE
      result = OpEN.openapiBgpPeerNextHopSelfModeGet(@client, 
                                                     OpEN::OPEN_BGP_GET_FINAL,
                                                     addr, 
                                                     scope_id, 
                                                    af,
                                                    tmp_p)
      print_bad_result(result, 'openapiBgpPeerNextHopSelfModeGet')
    end

    print_sanity_results(result, enable == OpEN.OPEN_CONTROL_tp_value(tmp_p), msg, enable)
    OpEN.delete_OPEN_CONTROL_tp(tmp_p)
  end

  def test_peer_pfx_limit(af, address, scope_id, threshold, warning_only, def_value)
    #Set the prefix limit configured for a given peer.
    #In this example, the maximum prefix limit is retrieved from the switch based
    #on the IP address family. This max value and the given threshold, and
    #warningOnly flag is set and read back for comparison.

    result, addr, type = ip_to_open_inet(address)
    msg = type + ' peer prefix limit'
    limit = 0

    if addr.family != OpEN::OPEN_AF_NONE
      if addr.family == OpEN::OPEN_AF_INET
        limit_p = OpEN.new_uint32_tp()
        OpEN.openapiBgpIpv4RouteMax(@client, limit_p)
        # Demonstrate get/set the maximum number of IPv4 routes
        limit = OpEN.uint32_tp_value(limit_p)
      else
        # Demonstrate alternate method to use the maximum number routes
        limit = OpEN::OPEN_BGP_NBR_MAX_PFX_NOLIMIT
      end

      result = OpEN.openapiBgpPeerPfxLimitSet(@client,
                                              addr, 
                                              scope_id, 
                                              af,
                                              limit,
                                              threshold,
                                              warning_only,
                                              def_value)
      print_bad_result(result, 'openapiBgpPeerPfxLimitSet')
    end

    limit_p = OpEN.new_uint32_tp()
    threshold_p = OpEN.new_uint32_tp()
    warning_p = OpEN.new_bool_tp()
    if result == OpEN::OPEN_E_NONE
      result = OpEN.openapiBgpPeerPfxLimitGet(@client, 
                                              OpEN::OPEN_BGP_GET_FINAL,
                                              addr, 
                                              scope_id, 
                                              af,
                                              limit_p,
                                              threshold_p,
                                              warning_p)
      print_bad_result(result, 'openapiBgpPeerPfxLimitGet')
    end

    test = (limit == OpEN.uint32_tp_value(limit_p) and
           threshold == OpEN.uint32_tp_value(threshold_p) and
           warning_only == OpEN.bool_tp_value(warning_p))
    print_sanity_results(result, test, msg, '%d, %d, %s' % [limit, threshold, warning_only])
    OpEN.delete_uint32_tp(limit_p)
    OpEN.delete_uint32_tp(threshold_p)
    OpEN.delete_bool_tp(warning_p)
  end

  def test_peer_update_source(address, scope_id, source, def_value)
    #Set the interface whose IP address that BGP uses as the source
    #IP address in packets sent to a given peer. The interface is
    #then read and compared to verify the operation.

    result, addr, type = ip_to_open_inet(address)
    msg = type + ' peer remote source scope %d source %d' % [scope_id, source]

    if addr.family != OpEN::OPEN_AF_NONE
      result = OpEN.openapiBgpPeerUpdateSourceSet(@client,
                                                  addr, 
                                                  scope_id, 
                                                  source,
                                                  def_value)
      print_bad_result(result, 'openapiBgpPeerUpdateSourceSet')
    end

    tmp_p = OpEN.new_uint32_tp()
    if result == OpEN::OPEN_E_NONE
      result = OpEN.openapiBgpPeerUpdateSourceGet(@client, 
                                                  OpEN::OPEN_BGP_GET_FINAL,
                                                  addr, 
                                                  scope_id, 
                                                  tmp_p)
      print_bad_result(result, 'openapiBgpPeerUpdateSourceGet')
    end

    print_sanity_results(result, (source == OpEN.uint32_tp_value(tmp_p)), msg, address)
    OpEN.delete_uint32_tp(tmp_p)
  end

end

def main()
  # Demonstrate OpEN usage for BGP APIs

  ret = $open.connect("bgp_example")
  if ret == OpEN::OPEN_E_NONE
    $open.getNetworkOSVersion()
    $open.getAPIVersion()
    client = $open.client
    example = BgpExample.new(client)
    puts 'Establish BGP and global parameters...'
    example.test_local_as(1)
    example.test_local_id('10.1.1.1')
    example.test_local_pref(1234567890)
    example.test_distance(10, 20, 30)
    example.test_global_hold_time(65535)
    example.test_global_keep_alive(65535)
    example.test_log_neighbor(true)
    example.test_max_paths(OpEN::OPEN_AF_INET, false)
    example.test_max_paths(OpEN::OPEN_AF_INET, true)
    example.test_network(true, true, '20.10.0.0', 24, 'my-map1')
    example.test_network(false, true, '20.10.0.0', 24, 'my-map1')
    # Demonstrate internal, external, and NSSA external bit sets
    match_bits = 0
    match_bits |= OpEN::OPEN_OSPF_METRIC_TYPE_INTERNAL
    match_bits |= OpEN::OPEN_OSPF_METRIC_TYPE_EXT1
    match_bits |= OpEN::OPEN_OSPF_METRIC_TYPE_EXT2
    match_bits |= OpEN::OPEN_OSPF_METRIC_TYPE_NSSA_EXT1
    match_bits |= OpEN::OPEN_OSPF_METRIC_TYPE_NSSA_EXT2
    example.test_redistribution(true, true, match_bits, true, true, 4294967295, false, OpEN::OPEN_AF_INET, '')
    # Neighbor specific tests
    puts 'Create a ipv4 test peer...'
    example.test_peer_remote_as(1, "20.2.2.2", 0)
    example.test_peer_admin_status("20.2.2.2", 0, OpEN::OPEN_BGP_START, true)
    example.test_peer_keep_alive("20.2.2.2", 0, 600, true)
    example.test_peer_hold_time("20.2.2.2", 0, 700, true)
    example.test_peer_activate(OpEN::OPEN_AF_INET6, "20.2.2.2", 0, true, true)
    example.test_peer_advertisement(OpEN::OPEN_AF_INET, "20.2.2.2", 0, 200, true)
    example.test_peer_next_hop_self(OpEN::OPEN_AF_INET, "20.2.2.2", 0, OpEN::OPEN_ENABLE, true)
    example.test_peer_pfx_limit(OpEN::OPEN_AF_INET, "20.2.2.2", 0, 20, true, true)
    puts 'Create a ipv6 test peer...'
    example.test_peer_remote_as(1, "2222::", 0)
    example.test_peer_admin_status("2222::", 0, OpEN::OPEN_BGP_STOP, true)
    example.test_peer_keep_alive("2222::", 0, 600, true)
    example.test_peer_hold_time("2222::", 0, 700, true)
    example.test_peer_activate(OpEN::OPEN_AF_INET6, "2222::", 0, true, true)
    example.test_peer_advertisement(OpEN::OPEN_AF_INET6, "2222::", 0, 222, true)
    example.test_peer_next_hop_self(OpEN::OPEN_AF_INET6, "2222::", 0, OpEN::OPEN_ENABLE, true)
    example.test_peer_pfx_limit(OpEN::OPEN_AF_INET6, "2222::", 0, 60, true, true)
    # Attempt to retrieve a routing interface
    result, intf = get_first_routing_interface(client)
    if result == OpEN::OPEN_E_NONE
      # Configure remote as link local address
      example.test_peer_remote_as(1, "fe80::1", intf)
      # Configure update source using same routing interface
      example.test_peer_update_source("20.2.2.2", 0, intf, true)
    else
      puts "\nSanity tests are incomplete because no routing interfaces are available!"
      puts "Please configure a routing interface and re-run test."
      puts "\nComplete"
    end
    $open.terminate()
  else
    print "Unable to connect"
  end
end

if __FILE__ == $0 then main() end


