#!/usr/bin/ruby

require 'optparse'
require "OpEN"
require "OpENUtil"
require "ipaddr"

#
# Copyright 2016 Broadcom.
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#


#
# Ruby 1.8.7.
#

$open = OpENUtil.new()

def int_to_ip(addr)
  """Convert ipv4 integer to string"""
  return "#{IPAddr.new(addr,Socket::AF_INET)}"
end

def openIPtoStrGet(client, ipAddr)
  ipStr = ""
  max_len_p = OpEN.new_uint32_tp()
  result = OpEN.openapiIpAddressMaxStringLength(client, max_len_p)
  if result == OpEN::OPEN_E_NONE
    max_len = OpEN.uint32_tp_value(max_len_p)
    buff_string = $open.getCharBuffer(max_len)
    buff        = OpEN::Open_buffdesc.new
    buff.pstart = buff_string
    buff.size   = max_len
    result = OpEN.openapiOpenIPtoStringGet(client, ipAddr, buff)
    if (result == OpEN::OPEN_E_NONE and (buff_string.cast().length>0))
      ipStr = buff_string.cast()
    end
    OpEN.delete_uint32_tp(max_len_p)
  end
  return ipStr
end

def statusToStr(peerStatus)
  case peerStatus
    when OpEN::OPEN_BGP_PEER_STATE_ILG
      return "ILG"
    when OpEN::OPEN_BGP_PEER_STATE_IDLE
      return "IDLE"
    when OpEN::OPEN_BGP_PEER_STATE_CONNECT
      return "CONNECT"
    when OpEN::OPEN_BGP_PEER_STATE_ACTIVE
      return "ACTIVE"
    when OpEN::OPEN_BGP_PEER_STATE_OPENSENT
      return "OPENSENT"
    when OpEN::OPEN_BGP_PEER_STATE_OPENCONFIRM
      return "OPENCONFIRM"
    when OpEN::OPEN_BGP_PEER_STATE_ESTABLISHED
      return "ESTABLISHED"
    else
      return "ILG"
  end
end

def performAction(addFamily)
  # Demonstrate OpEN usage for BGP APIs

  ret = $open.connect("bgp_summary")
  if ret == OpEN::OPEN_E_NONE
    $open.getNetworkOSVersion()
    $open.getAPIVersion()
    client = $open.client

    # Fetch IPv4 Routing
    v4rtrAdminMode_p = OpEN.new_OPEN_CONTROL_tp()
    v4rtrAdminMode_rc = OpEN.openapiRtrAdminModeGet(client, OpEN::OPEN_AF_INET, v4rtrAdminMode_p)
    if v4rtrAdminMode_rc == OpEN::OPEN_E_NONE
       v4rtrAdminMode = OpEN.OPEN_CONTROL_tp_value(v4rtrAdminMode_p)
       if addFamily == OpEN::OPEN_AF_INET
          str = (v4rtrAdminMode == OpEN::OPEN_ENABLE) ? "Enabled" : "Disabled"
          print ("\nIPv4 Routing .................................. %s\n" % str)
       end
    end
    OpEN.delete_OPEN_CONTROL_tp(v4rtrAdminMode_p)

    # Fetch IPv6 Routing
    v6rtrAdminMode_p = OpEN.new_OPEN_CONTROL_tp()
    v6rtrAdminMode_rc = OpEN.openapiRtrAdminModeGet(client, OpEN::OPEN_AF_INET6, v6rtrAdminMode_p)
    if v6rtrAdminMode_rc == OpEN::OPEN_E_NONE
       v6rtrAdminMode = OpEN.OPEN_CONTROL_tp_value(v6rtrAdminMode_p)
       if addFamily == OpEN::OPEN_AF_INET6
          str = (v6rtrAdminMode == OpEN::OPEN_ENABLE) ? "Enabled" : "Disabled"
          print ("\nIPv6 Routing .................................. %s\n" % str)
       end
    end
    OpEN.delete_OPEN_CONTROL_tp(v6rtrAdminMode_p)

    # Fetch BGP Admin Mode
    bgpAdminMode_p = OpEN.new_OPEN_CONTROL_tp()
    bgpAdminMode_rc = OpEN.openapiBgpAdminModeGet(client, bgpAdminMode_p)
    if bgpAdminMode_rc == OpEN::OPEN_E_NONE
       bgpAdminMode = OpEN.OPEN_CONTROL_tp_value(bgpAdminMode_p)
       str = (bgpAdminMode == OpEN::OPEN_ENABLE) ? "Enabled" : "Disabled"
       print ("BGP Admin Mode ................................ %s\n" % str)
    end
    OpEN.delete_OPEN_CONTROL_tp(bgpAdminMode_p)

    # Fetch BGP Router ID
    bgpLocalId_p = OpEN.new_uint32_tp()
    bgpLocalId = 0
    bgpLocalId_rc = OpEN.openapiBgpLocalIdGet(client, bgpLocalId_p)
    if bgpLocalId_rc == OpEN::OPEN_E_NONE
       bgpLocalId = OpEN.uint32_tp_value(bgpLocalId_p)
       print ("BGP Router ID ................................. %s\n" % int_to_ip(bgpLocalId))
    end
    OpEN.delete_uint32_tp(bgpLocalId_p)

    # Fetch Local AS Number
    tmp_p = OpEN.new_uint32_tp()
    result = OpEN.openapiBgpLocalASGet(client, tmp_p)
    if result == OpEN::OPEN_E_NONE
       tmp = OpEN.uint32_tp_value(tmp_p)
       str = (tmp == 0) ? "N/A" : "#{tmp}"
       print ("Local AS Number ............................... %s\n" % str)
    end
    OpEN.delete_uint32_tp(tmp_p)

    # Fetch Number of Network Entries and Number of AS Paths
    numNet_p   = OpEN.new_uint32_tp()
    numPaths_p = OpEN.new_uint32_tp()
    result = OpEN.openapiBgpMapGlobalStatusInfoGet(client, addFamily,
                                                   numNet_p, numPaths_p)
    if result == OpEN::OPEN_E_NONE
       numNet = OpEN.uint32_tp_value(numNet_p)
       numPaths = OpEN.uint32_tp_value(numPaths_p)
       print ("Number of Network Entries ..................... %d\n" % numNet)
       print ("Number of AS Paths ............................ %d\n\n" % numPaths)
    end
    OpEN.delete_uint32_tp(numNet_p)
    OpEN.delete_uint32_tp(numPaths_p)

    bgp_valid_status = (v4rtrAdminMode_rc | v6rtrAdminMode_rc | bgpAdminMode_rc | bgpLocalId_rc)
    if bgp_valid_status != OpEN::OPEN_E_NONE
       printf("\r\nWarning:  Unknown error - BGP data is unavailable at this time.")
    else
      routingEnabled = OpEN::OPEN_FALSE
      if ((v4rtrAdminMode == OpEN::OPEN_ENABLE) || (v6rtrAdminMode == OpEN::OPEN_ENABLE))
        routingEnabled = OpEN::OPEN_TRUE
      end

      valid = true
      if ((bgpAdminMode == OpEN::OPEN_DISABLE) || !routingEnabled || (bgpLocalId == 0))
         valid = false
         printf("\r\nWarning:  BGP is not running. Further data is unavailable.")
      end

      i = 0
      if routingEnabled == OpEN::OPEN_FALSE
         i += 1
         printf("\r\n   %d) IP routing is not enabled.", ++i)
      end
      if bgpAdminMode == OpEN::OPEN_DISABLE
         i += 1
         printf("\r\n   %d) BGP is administratively disabled.", ++i)
      end
      if bgpLocalId == 0
         i += 1
         printf("\r\n   %d) No BGP router ID is configured.", ++i);
      end

      if valid
        print("\nNeighbor           ASN  MsgRcvd  MsgSent         State   Up/Down Time  Pfx Rcvd\n")
        print("---------------- ----- -------- -------- ------------- -------------- ---------\n")
        prevRemoteAddr = OpEN::Open_inet_addr_t.new
        prevRemoteAddr.family = addFamily
        remoteAddr = OpEN::Open_inet_addr_t.new
        remoteAddr.family = addFamily
        prevScopeId = 0
        scopeId_p = OpEN.new_uint32_tp()
        scopeId = 0
        while true
          rc = OpEN.openapiBgpMapPeerRemoteAddrGetNext(client, OpEN::OPEN_AF_NONE, prevRemoteAddr,
                                                       prevScopeId, remoteAddr, scopeId_p)
          if OpEN::OPEN_E_NONE == rc
             scopeId = OpEN.uint32_tp_value(scopeId_p)
             tmpb_p = OpEN.new_bool_tp()
             result = OpEN.openapiBgpPeerActivateGet(client, OpEN::OPEN_BGP_GET_FINAL, remoteAddr,
                                                     scopeId, addFamily, tmpb_p)
             if result == OpEN::OPEN_E_NONE
                status_p = OpEN.new_open_bgpPeerStatus_tp()
                rc1 = OpEN.openapiBgpMapPeerStatusGet(client, remoteAddr, scopeId, addFamily, status_p)
                if rc1 == OpEN::OPEN_E_NONE
                  activate = OpEN.bool_tp_value(tmpb_p)
                  if activate == true
                    remoteAS_p = OpEN.new_uint32_tp()
                    result = OpEN.openapiBgpPeerRemoteASGet(client, remoteAddr, scopeId, remoteAS_p)
                    if result == OpEN::OPEN_E_NONE
                       remoteAS = OpEN.uint32_tp_value(remoteAS_p)
                    else
                       remoteAS = 0
                    end
                    OpEN.delete_uint32_tp(remoteAS_p)
                    status = OpEN.open_bgpPeerStatus_tp_value(status_p)
                    statusStr = statusToStr(status.peerState)
                    totalTx = status.txOpen + status.txUpdate + status.txKeepalive + status.txNotif + status.txRefresh
                    totalRx = status.rxOpen + status.rxUpdate + status.rxKeepalive + status.rxNotif + status.rxRefresh
                    et = status.estTime
                    mm, ss = et.divmod(60)
                    hh, mm = mm.divmod(60)
                    dd, hh = hh.divmod(24)
                    estTime = "%3d:%02d:%02d:%02d" % [dd, hh, mm, ss]
                    pfxRx = OpEN.uint32_tArray_getitem(status.inPfxCurrent, addFamily)

                    peer_addr_max_len_p = OpEN.new_uint32_tp()
                    peer_addr_max_len = 0
                    result = OpEN.openapiBgpPeerAddressStringMaxLengthGet(client, peer_addr_max_len_p)
                    if result == OpEN::OPEN_E_NONE
                      peer_addr_max_len = OpEN.uint32_tp_value(peer_addr_max_len_p)
                      OpEN.delete_uint32_tp(peer_addr_max_len_p)
                    end

                    buff        = OpEN::Open_buffdesc.new
                    buff.size   = peer_addr_max_len

                    tmp_p = OpEN.new_OPEN_BOOL_tp()
                    ipStr = ""
                    rc = OpEN.openapiIsInetAddrZero(client, remoteAddr, tmp_p)
                    if OpEN::OPEN_E_NONE == rc
                       isZero = OpEN.OPEN_BOOL_tp_value(tmp_p)
                       if isZero != OpEN::OPEN_TRUE
                          nbrPeerAddr = OpEN::Open_inet_addr_t.new
                          nbrPeerAddr.family = addFamily
                          nbrRc = OpEN.openapiBgpPeerAutodetectedIpGet(client, scopeId, nbrPeerAddr)
                          if OpEN::OPEN_E_NONE == nbrRc
                             ipStr = openIPtoStrGet(client, nbrPeerAddr)
                             buff_string = $open.getCharBuffer(peer_addr_max_len, ipStr)
                             buff.pstart = buff_string
                             if scopeId
                                scopeCarRc = OpEN.openapiBgpPeerAddrScopeCat(client, buff, scopeId)
                                if (scopeCarRc == OpEN::OPEN_E_NONE and (buff_string.cast().length>0))
                                  ipStr = buff_string.cast()
                                end
                             end
                             ipStr = ipStr + " (autodetected)"
                          else
                             ipStr = "Unresolved"
                             buff_string = $open.getCharBuffer(peer_addr_max_len, ipStr)
                             buff.pstart = buff_string
                             if scopeId
                                scopeCarRc = OpEN.openapiBgpPeerAddrScopeCat(client, buff, scopeId)
                                if (scopeCarRc == OpEN::OPEN_E_NONE and (buff_string.cast().length>0))
                                  ipStr = buff_string.cast()
                                end
                             end
                          end
                       else
                         ipStr = openIPtoStrGet(client, remoteAddr)
                         buff_string = $open.getCharBuffer(peer_addr_max_len, ipStr)
                         buff.pstart = buff_string
                         if scopeId
                            scopeCarRc = OpEN.openapiBgpPeerAddrScopeCat(client, buff, scopeId)
                            if (scopeCarRc == OpEN::OPEN_E_NONE and (buff_string.cast().length>0))
                               ipStr = buff_string.cast()
                            end
                         end
                       end
                    end
                    OpEN.delete_OPEN_BOOL_tp(tmp_p)

                    if ipStr.length <= 16
                       printf("%-16s %5u %8u %8u %13.13s %14.14s %9d\n", ipStr, remoteAS, totalRx, totalTx,
                                                                        statusStr, estTime, pfxRx)
                    else
                       printf("%-16s \r\n%22u %8u %8u %13.13s %14.14s %9d\n", ipStr, remoteAS, totalRx, totalTx,
                                                                        statusStr, estTime, pfxRx)
                    end
                  end
                end
                OpEN.delete_open_bgpPeerStatus_tp(status_p)
             end
             OpEN.delete_bool_tp(tmpb_p)
          else
             break
          end
          prevRemoteAddr = remoteAddr
          prevScopeId = scopeId
          scopeId = OpEN.uint32_tp_value(scopeId_p)
        end #end while loop
        OpEN.delete_uint32_tp(scopeId_p)
      end
    end
    $open.terminate()
    printf("\n")
  else
    print "Unable to connect"
  end
end

options = {}
optparse = OptionParser.new do |opts|
  opts.on('-4', '--ipv4', 'display a summary of BGP IPv4 configuration and status') do
    options[:ipv4] = true
  end

  opts.on('-6', '--ipv6', 'display a summary of BGP IPv6 configuration and status') do
    options[:ipv6] = true
  end

  opts.on('-h', '--help', 'display this message') do
    puts opts
    exit
  end
end

begin
  optparse.parse!
  t = options.length
  if (t > 1) || (t == 0)
    puts optparse
    exit
  end
rescue OptionParser::InvalidOption, OptionParser::MissingArgument
  puts $!.to_s
  puts optparse
  exit
end

if options[:ipv4] != nil
  performAction(OpEN::OPEN_AF_INET)
end

if options[:ipv6] != nil
  performAction(OpEN::OPEN_AF_INET6)
end

