/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  mfdb_example.c
*
* @purpose   L2 Multicast MFDB APIs Example.
*
* @component OpEN
*
* @comments
*
* @create    06/19/2015
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_mfdb.h"

/*******************************************************************
*
* @brief  This is the main function that will demonstrates
*         L2 Multicast MFDB APIs.
*
* @returns  0: Success
* @returns  1: Failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result = OPEN_E_NONE;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100] = {0};

  open_buffdesc vidMac;
  open_buffdesc mac;
  open_buffdesc desc;
  open_buffdesc frwdIntMask;
  open_buffdesc fltIntMask;
  open_buffdesc absPortMask;
  char bufVidMac[OPEN_MFDB_VIDMAC_LENGTH] = {0};
  char bufMac[OPEN_MFDB_MAC_LENGTH] = {0};
  char bufDesc[OPENAPI_MFDB_DESCR_LEN] = {0};
  char bufFrwdIntMask[OPEN_MFDB_INTF_LIST_LENGTH] = {0};
  char bufFltIntMask[OPEN_MFDB_INTF_LIST_LENGTH] = {0};
  char bufAbsPortMask[OPEN_MFDB_INTF_LIST_LENGTH] = {0};
  OPEN_MFDB_PROTOCOL_TYPE_t nextProt = OPEN_MFDB_TOTAL_NUM_PROTOCOLS;
  OPEN_MFDB_PROTOCOL_TYPE_t compId = OPEN_MFDB_TOTAL_NUM_PROTOCOLS;
  OPEN_MFDB_PROTOCOL_TYPE_t nextCompId = OPEN_MFDB_TOTAL_NUM_PROTOCOLS;
  OPEN_MFDB_ENTRY_TYPE_t type = OPEN_MFDB_TYPE_STATIC;
  uint32_t vlanId = 0;
  uint32_t maxEntries = 0;
  uint32_t mostEntries = 0;
  uint32_t currEntries = 0;
  uint32_t matchType = OPEN_MATCH_GETNEXT;

  vidMac.pstart = bufVidMac;
  vidMac.size = sizeof(bufVidMac);
  mac.pstart = bufMac;
  mac.size = sizeof(bufMac);
  desc.pstart = bufDesc;
  desc.size = sizeof(bufDesc);
  frwdIntMask.pstart = bufFrwdIntMask;
  frwdIntMask.size = sizeof(bufFrwdIntMask);
  fltIntMask.pstart = bufFltIntMask;
  fltIntMask.size = sizeof(bufFltIntMask);
  absPortMask.pstart = bufAbsPortMask;
  absPortMask.size = sizeof(bufAbsPortMask);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("mfdb_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (Result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fails until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting MFDB API example application");

  printf ("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet (&clientHandle, &switch_os_revision) == OPEN_E_NONE)
  {
    printf ("Network OS version = %s\n", switch_os_revision_string);
  }
  else
  {
    printf ("Network OS version retrieve error\n");
  }
  printf ("\n");

  /* exercise various OPEN API Mirroring functions */
  do
  {
    /* Testing of retrieving the first entry in the MFDB table */
    strcpy((char *)vidMac.pstart, "00:00:00:00:00:00:00:00");
    vidMac.size = strlen((char *)vidMac.pstart) + 1;
    result = openapiMfdbEntryGetNext(&clientHandle, &vidMac, &nextProt);
    if (OPEN_E_NONE == result)
    {
      printf("The first entry in the MFDB table is %s\n", (char *)vidMac.pstart);

      /* Testing of retrieving the data of MFDB entry */
      result = openapiMfdbEntryDataGet(&clientHandle, &vidMac, nextProt, &mac, &vlanId, &compId, &type, &desc, &frwdIntMask, &fltIntMask, &absPortMask);
      if (OPEN_E_NONE == result)
      {
        printf("Data of entry %s is retrieved successfully.\n", (char *)vidMac.pstart);
      }
      else
      {
        printf("Failed to retrieve the data for entry in the MFDB table. (Result = %d)\n", result);
      }

      /* Testing of retrieving the next entry in the MFDB table */
      result = openapiMfdbEntryGetNext(&clientHandle, &vidMac, &nextProt);
      if (OPEN_E_NONE == result)
      {
        printf("The next entry in the MFDB table is %s\n", (char *)vidMac.pstart);
      }
      else
      {
        printf("Failed to retrieve the next entry in the MFDB table. (Result = %d)\n", result);
      }
    }
    else
    {
      printf("Failed to retrieve the first entry in the MFDB table. (Result = %d)\n", result);
    }

    /* Testinf of checking if a VLANID-MAC entry with specific protocol exists in the MFDB table */
    result = openapiMfdbEntryProtocolGetExact(&clientHandle, &vidMac, nextProt);
    if (OPEN_E_NONE == result)
    {
      printf("VLAN-MAC entry(%s) exists in the MFDB table.\n", (char *)vidMac.pstart);
    }
    else if (OPEN_E_FAIL == result)
    {
      printf("VLAN-MAC entry(%s) doesn't exist in the MFDB table.\n", (char *)vidMac.pstart);
    }
    else
    {
      printf("Failed to check f a VLANID-MAC entry exists in the MFDB table. (Result = %d)\n", result);
    }

    /* Testing of retrieving the first MFDB entry of the same user */
    strcpy((char *)vidMac.pstart, "00:00:00:00:00:00:00:00");
    vidMac.size = strlen((char *)vidMac.pstart) + 1;
    memset(desc.pstart, 0, sizeof(bufDesc));
    desc.size = sizeof(bufDesc);
    memset(frwdIntMask.pstart, 0, sizeof(bufFrwdIntMask));
    frwdIntMask.size = sizeof(bufFrwdIntMask);
    memset(fltIntMask.pstart, 0, sizeof(bufFltIntMask));
    fltIntMask.size = sizeof(bufFltIntMask);
    result = openapiMfdbComponentEntryGetNext(&clientHandle, &vidMac, compId, &nextCompId, &type, &desc, &frwdIntMask, &fltIntMask);
    if (OPEN_E_NONE == result)
    {
      printf("First entry of the same user retrieved successful. Data of entry %s is:\n\tcomponent ID - %u,\n\ttype - %u,\n\tdescription - %s,\n"
             "\tforward interface mask - %s,\n\tfilter interface mask - %s\n", (char *)vidMac.pstart, (uint32_t)nextCompId, (uint32_t)type, (char *)desc.pstart,
             (char *)frwdIntMask.pstart, (char *)fltIntMask.pstart);
    }
    else
    {
      printf("Failed to retrieve the first MFDB entry of the same user. (Result = %d)\n", result);
    }

    /* Testing of checking if a VLANID-MAC entry exists in the MFDB table */
    result = openapiMfdbEntryFwdIntfGetExact(&clientHandle, &vidMac);
    if (OPEN_E_NONE == result)
    {
      printf("VLAN-MAC entry(%s) exists in the MFDB table.\n", (char *)vidMac.pstart);
    }
    else if (OPEN_E_FAIL == result)
    {
      printf("VLAN-MAC entry(%s) doesn't exist in the MFDB table.\n", (char *)vidMac.pstart);
    }
    else
    {
      printf("Failed to check if a VLANID-MAC entry exists in the MFDB table. (Result = %d)\n", result);
    }

    /* Testing of retrieving the forwarding interfaces of the first VLANID-MAC entry */
    strcpy((char *)vidMac.pstart, "00:00:00:00:00:00:00:00");
    vidMac.size = strlen((char *)vidMac.pstart) + 1;
    memset(absPortMask.pstart, 0, sizeof(bufAbsPortMask));
    absPortMask.size = sizeof(bufAbsPortMask);
    result = openapiMfdbEntryFwdIntfGetNext(&clientHandle, &vidMac, matchType, &absPortMask);
    if (OPEN_E_NONE == result)
    {
      printf("Forwarding interfaces of the first VLANID-MAC entry retrieved successful. Absolute port mask of entry %s is: %s\n",
             (char *)vidMac.pstart, (char *)absPortMask.pstart);
    }
    else
    {
      printf("Failed to retrieve the forwarding interfaces of the first VLANID-MAC entry. (Result = %d)\n", result);
    }

    /* Testing of retrieving the forwarding interfaces of the first learned VLANID-MAC entry */
    strcpy((char *)vidMac.pstart, "00:00:00:00:00:00:00:00");
    vidMac.size = strlen((char *)vidMac.pstart) + 1;
    memset(absPortMask.pstart, 0, sizeof(bufAbsPortMask));
    absPortMask.size = sizeof(bufAbsPortMask);
    result = openapiMfdbEntryLearntIntfGetNext(&clientHandle, &vidMac, matchType, &absPortMask);
    if (OPEN_E_NONE == result)
    {
      printf("Forwarding interfaces of the first learned VLANID-MAC entry retrieved successful. Absolute port mask of entry %s is: %s\n",
             (char *)vidMac.pstart, (char *)absPortMask.pstart);
    }
    else
    {
      printf("Failed to retrieve the forwarding interfaces of the first learned VLANID-MAC entry. (Result = %d)\n", result);
    }

    /* Testing of retrieving the maximum number of entries that the MFDB table can hold */
    result = openapiMfdbMaxTableEntriesGet(&clientHandle, &maxEntries);
    if (OPEN_E_NONE == result)
    {
      printf("The maximum number of entries that the MFDB table can hold is %d\n", maxEntries);
    }
    else
    {
      printf("Failed to retrieve the maximum number of entries that the MFDB table can hold. (Result = %d)\n", result);
    }

    /* Testing of retrieving the most number of entries ever in the MFDB table */
    result = openapiMfdbMostEntriesGet(&clientHandle, &mostEntries);
    if (OPEN_E_NONE == result)
    {
      printf("The most number of entries ever in the MFDB table is %d\n", mostEntries);
    }
    else
    {
      printf("Failed to retrieve the most number of entries ever in the MFDB table. (Result = %d)\n", result);
    }

    /* Testing of retrieving the number of current entries in the MFDB table */
    result = openapiMfdbCurrEntriesGet(&clientHandle, &currEntries);
    if (OPEN_E_NONE == result)
    {
      printf("The number of current entries in the MFDB table is %d\n", currEntries);
    }
    else
    {
      printf("Failed to retrieve the number of current entries in the MFDB table. (Result = %d)\n", result);
    }

    printf("Press enter to continue...\n");
    getchar();
  } while (0);

  /* Log goodbye message with OPEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping MFDB API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
