/*********************************************************************
*
* Copyright 2017-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  pfc_example.c
*
* @purpose   Data center bridging - Priority-based Flow Control API Example.
*
* @component OPEN
*
* @comments
*
* @create    1/31/2017
*
* @end
*
**********************************************************************/

#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_pfc.h"
#include "openapi_ets.h"

static char *pfcModeToStr(OPEN_PFC_MODE_t mode)
{
  char *ret = "Unknown pfc mode";

  if (mode == OPEN_PFC_MODE_DISABLE)
  {
    ret = "disable";
  }   
  else if (mode == OPEN_PFC_MODE_ENABLE)
  {
    ret = "enable";
  }
  return ret;
}

static char *pfcDropModeToStr(OPEN_PFC_PRI_DROP_MODE_t mode)
{
  char *ret = "Unknown pfc drop mode";

  if (mode == OPEN_PFC_PRI_DROP_MODE)
  {
    ret = "drop";
  }   
  else if (mode == OPEN_PFC_PRI_NODROP_MODE)
  {
    ret = "enable";
  }
  return ret;
}

/*********************************************************************
* @purpose  Set PFC mode for a given interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   interface number
* @param    mode             @b{(input)}   desired mode
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/

void setPfcMode(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_PFC_MODE_t mode)
{
  open_error_t ret;

  ret = openapiIfPfcModeSet(client_handle, intIfNum, mode);
  if (ret == OPEN_E_UNAVAIL)
  {
    printf("unable to set mode %s for interface %d: feature not supported\n", pfcModeToStr(mode), intIfNum);
  }
  else if (ret != OPEN_E_NONE)
  {
    printf("unable to set mode %s for interface %d: error %d\n", pfcModeToStr(mode), intIfNum, ret);
  }
}

/*********************************************************************
* @purpose  Set PFC drop mode for a given interface and priority.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   interface number
* @param    priority         @b{(input)}   priority
* @param    mode             @b{(input)}   desired mode
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/

void setDropMode(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t priority, OPEN_PFC_PRI_DROP_MODE_t mode)
{
  open_error_t ret;

  ret = openapiIfPfcPriorityModeSet(client_handle, intIfNum, priority, mode);
  if (ret == OPEN_E_UNAVAIL)
  {
    printf("unable to set drop mode %s for interface %d and priority %d: feature not supported\n", pfcDropModeToStr(mode), intIfNum, priority);
  }
  else if (ret == OPEN_E_NOT_FOUND)
  {
    printf("unable to set drop mode %s for interface %d and priority %d: priority out of range\n", pfcDropModeToStr(mode), intIfNum, priority);
  }
  else if (ret != OPEN_E_NONE)
  {
    printf("unable to set mode %s for interface %d and priority %d: error %d\n", pfcDropModeToStr(mode), intIfNum, ret, priority);
  }
}

/*********************************************************************
* @purpose  Display PFC status for a given interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/

void showPfc(openapiClientHandle_t *client_handle, uint32_t intIfNum)
{
  open_error_t ret;
  uint32_t incompatibleCfgCount;
  uint32_t numPfcCap;
  uint32_t prioMin;
  uint32_t prioMax;
  uint32_t allowance;
  uint32_t count;
  OPEN_BOOL_t willing;
  OPEN_BOOL_t mbcStatus;
  OPEN_BOOL_t compat;
  OPEN_PFC_PRI_DROP_MODE_t mode;
  OPEN_PFC_STATUS_t pfcStatus;
  OPEN_PFC_MODE_t pfcMode;
  int i;

  printf("Interface %d\n", intIfNum);

  ret = openapiPfcPeerIncompatibleCfgCountGet(client_handle, intIfNum, &incompatibleCfgCount);
  if (ret != OPEN_E_NONE) 
  {
    printf("%s: openapiPfcPeerIncompatibleCfgCountGet failed (%d)\n", 
           __FUNCTION__, ret);
  }
  else
  {
    printf("Incompatible configurations received from peer: %u\n", incompatibleCfgCount);
  }
  for (i = OPEN_DOT1P_MIN_PRIORITY; i <= OPEN_DOT1P_MAX_PRIORITY; i++)
  {
    printf("Priority %d\n", i);
    printf("***********\n");
    ret = openapiPfcAdvPriorityModeGet(client_handle, intIfNum, i, &mode);
    if (ret != OPEN_E_NONE) 
    {
      if (ret == OPEN_E_NOT_FOUND)
      {
        printf("openapiPfcAdvPriorityModeGet: Interface not supported\n");
      }
      else
      {
        printf("%s: openapiPfcAdvPriorityModeGet (%d)\n", __FUNCTION__, ret);
      }
    }
    else
    {
      printf("Advertised priority mode:  %u\n", mode);
    }
    ret = openapiPfcPeerPriorityModeGet(client_handle, intIfNum, i, &mode);
    if (ret != OPEN_E_NONE) 
    {
      if (ret == OPEN_E_NOT_FOUND)
      {
        printf("openapiPfcPeerPriorityModeGet: No peer detected or priority out of range\n");
      }
      else
      {
        printf("%s: openapiPfcPeerPriorityModeGet (%d)\n", __FUNCTION__, ret);
      }
    }
    else
    {
      printf("Peer priority mode:  %u\n", mode);
    }
    ret = openapiIfPfcPriorityModeGet(client_handle, intIfNum, i, &mode);
    if (ret != OPEN_E_NONE) 
    {
      printf("%s: openapiIfPfcPriorityModeGet (%d)\n", __FUNCTION__, ret);
    }
    else
    {
      printf("IF PFC priority mode:  %u\n", mode);
    }
    ret = openapiPfcOprPriorityModeGet(client_handle, intIfNum, i, &mode);
    if (ret != OPEN_E_NONE)
    {
      printf("%s: openapiPfcOprPriorityModeGet (%d)\n", __FUNCTION__, ret);
    }
    else
    {
      printf("Operational PFC Priority participation mode:  %u\n", mode);
    }
    ret = openapiIfPfcRxPriorityStatGet(client_handle, intIfNum, i, &count);
    if (ret != OPEN_E_NONE)
    {
      if (ret == OPEN_E_UNAVAIL)
      {
        printf("%s: openapiIfPfcRxPriorityStatGet feature not supported (%d)\n",
               __FUNCTION__, ret);
      }
      else
      {
        printf("%s: openapiIfPfcRxPriorityStatGet failed (%d)\n",
               __FUNCTION__, ret);
      }
    }
    else
    {
      printf("pfc rx priority stat: %u\n", count);
    }
    ret = openapiPfcTxPriorityStatGet(client_handle, intIfNum, i, &count);
    if (ret != OPEN_E_NONE)
    {
      if (ret == OPEN_E_UNAVAIL)
      {
        printf("%s: openapiPfcTxPriorityStatGet feature not supported (%d)\n",
               __FUNCTION__, ret);
      }
      else
      {
        printf("%s: openapiPfcTxPriorityStatGet failed (%d)\n",
               __FUNCTION__, ret);
      }
    }
    else
    {
      printf("pfc tx priority stat: %u\n", count);
    }
  }
  ret = openapiPfcPeerWillingGet(client_handle, intIfNum, &willing);
  if (ret != OPEN_E_NONE) 
  {
    if (ret == OPEN_E_NOT_FOUND)
    {
      printf("Peer not detected, unable to report willingness.\n");
    }
    else
    {
      printf("%s: openapiPfcPeerWillingGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("Willingness of peer: %s\n", willing == OPEN_TRUE ? "true":"false");
  }
  ret = openapiPfcPeerMbcStatusGet(client_handle, intIfNum, &mbcStatus);
  if (ret == OPEN_E_NOT_FOUND)
  {
    printf("openapiPfcPeerMbcStatusGet: No peer detected\n");
  }
  else if (ret != OPEN_E_NONE)
  {
    printf("%s: openapiPfcPeerMbcStatusGet (%d)\n", __FUNCTION__, ret);
  }
  else
  {
    printf("MACSEC bypass capability of peer: %s\n", mbcStatus == OPEN_TRUE ? "true":"false");
  }
  ret = openapiPfcPeerCapabilityGet(client_handle, intIfNum, &numPfcCap);
  if (ret != OPEN_E_NONE) 
  {
    printf("%s: openapiPfcPeerCapabilityGet failed (%d)\n", 
           __FUNCTION__, ret);
  }
  else
  {
    printf("Capability of peer: %d\n", numPfcCap);
  }
  ret = openapiPfcMinNoDropPriorityGet(client_handle, &prioMin);
  if (ret != OPEN_E_NONE) 
  {
    printf("%s: openapiPfcMinNoDropPriorityGet failed (%d)\n", 
           __FUNCTION__, ret);
  }
  else
  {
    printf("Minimum no drop priority: %u\n", prioMin);
  }
  ret = openapiPfcMaxNoDropPriorityGet(client_handle, &prioMax);
  if (ret != OPEN_E_NONE) 
  {
    printf("%s: openapiPfcMaxNoDropPriorityGet failed (%d)\n", 
           __FUNCTION__, ret);
  }
  else
  {
    printf("Maximum no drop priority: %u\n", prioMax);
  }
  ret = openapiIfPfcStatusGet(client_handle, intIfNum, &pfcStatus);
  if (ret != OPEN_E_NONE) 
  {
    printf("%s: openapiIfPfcStatusGet failed (%d)\n", 
           __FUNCTION__, ret);
  }
  else
  {
    printf("PFC status: %s\n", pfcStatus == OPEN_PFC_STATUS_ACTIVE ? "active":"inactive");
  }
  ret = openapiIfPfcModeGet(client_handle, intIfNum, &pfcMode);
  if (ret != OPEN_E_NONE) 
  {
    printf("%s: openapiIfPfcModeGet failed (%d)\n", 
           __FUNCTION__, ret);
  }
  else
  {
    printf("PFC mode: %s\n", pfcMode == OPEN_PFC_MODE_DISABLE ? "disable":"enable");
  }
  ret = openapiPfcOperLinkDelayAllowanceGet(client_handle, intIfNum, &allowance);
  if (ret != OPEN_E_NONE) 
  {
    printf("%s: openapiPfcOperLinkDelayAllowanceGet failed (%d)\n", 
           __FUNCTION__, ret);
  }
  else
  {
    printf("Link allowance in bits: %u\n", allowance);
  }
  ret = openapiPfcPeerCfgCompatibleGet(client_handle, intIfNum, &compat);
  if (ret != OPEN_E_NONE) 
  {
    if (ret == OPEN_E_NOT_FOUND)
    {
      printf("Peer not detected, unable to determine peer compatibilty.\n");
    }
    else
    {
      printf("%s: openapiPfcPeerCfgCompatibleGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("pfc peer cfg compatible: %s\n", compat == OPEN_TRUE ? "true":"false");
  }
  ret = openapiPfcPeerCompatibleCfgCountGet(client_handle, intIfNum, &count);
  if (ret != OPEN_E_NONE) 
  {
    printf("%s: openapiPfcPeerCompatibleCfgCountGet failed (%d)\n", 
           __FUNCTION__, ret);
  }
  else
  {
    printf("pfc peer cfg compatible count: %u\n", count);
  }
  ret = openapiIfPfcRxStatGet(client_handle, intIfNum, &count);
  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("%s: openapiIfPfcRxStatGet feature not supported (%d)\n",
           __FUNCTION__, ret);
    }
    else
    {
      printf("%s: openapiIfPfcRxStatGet failed (%d)\n",
           __FUNCTION__, ret);
    }
  }
  else
  {
    printf("pfc rx stat: %u\n", count);
  }
  ret = openapiIfPfcTxStatGet(client_handle, intIfNum, &count);
  if (ret != OPEN_E_NONE)
  {
    if (ret == OPEN_E_UNAVAIL)
    {
      printf("%s: openapiIfPfcTxStatGet feature not supported (%d)\n",
           __FUNCTION__, ret);
    }
    else
    {
      printf("%s: openapiIfPfcTxStatGet failed (%d)\n",
           __FUNCTION__, ret);
    }
  }
  else
  {
    printf("pfc tx stat: %u\n", count);
  }
}

void print_usage(const char *name)
{
  printf("%s [-p priority] [-m [enable | disable]] [-d | -n]  [-c] -i interface\n", name);
  printf("-p priority -- priority\n");
  printf("-m [enable | disable] -- enable or disable pfc on an interface\n");
  printf("-d | -n -- drop (-d) or nodrop (-n) at the given priority on an interface (requires -p)\n");
  printf("-i interface -- internal interface id (required argument)\n");
  printf("-c -- clear the PFC stats on an interface\n");
  exit(0);
}

int main(int argc, char *argv[])
{
  openapiClientHandle_t clientHandle;
  int option = 0;
  uint32_t intIfNum;
  bool enable = false;
  bool disable = false;
  bool drop = false;
  bool nodrop = false;
  bool clear = false;
  bool sawif = false;
  uint32_t priority = 0;
  int rc;
  open_error_t ret;

  while ((option = getopt(argc, argv,"p:m:i:dnc")) != -1) {
    switch (option) {
      case 'p':
        priority = atoi(optarg);
        break;
      case 'm' : 
        if (!strcmp(optarg, "enable"))
        {
          enable = true;
        }
        else if (!strcmp(optarg, "disable"))
        {
          disable = true;
        }
        else
        {
          print_usage(argv[0]);
        }
        break;
      case 'd' : 
        drop = true;
        break;
      case 'n' : 
        nodrop = true;
        break;
      case 'c' : 
        clear = true;
        break;
      case 'i' : 
        intIfNum = atoi(optarg);
        sawif = true;
        break;
      default: 
        print_usage(argv[0]); 
        break;
    }
  }

  if (sawif == false)
  {
    print_usage(argv[0]); 
  }

  printf("Priority is %d\n", priority);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((rc = openapiClientRegister("routing_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", rc);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  if (clear)
  {
    ret = openapiIfPfcStatsClear(&clientHandle, intIfNum);
    if (ret != OPEN_E_NONE)
    {
      printf("%s: openapiIfPfcStatsClear failed (%d)\n", 
             __FUNCTION__, ret);
    }
  }

  if (enable || disable)
  {
    setPfcMode(&clientHandle, intIfNum, enable ? OPEN_PFC_MODE_ENABLE:OPEN_PFC_MODE_DISABLE);
  }

  if (drop || nodrop)
  {
    setDropMode(&clientHandle, intIfNum, priority, drop ? OPEN_PFC_PRI_DROP_MODE:OPEN_PFC_PRI_NODROP_MODE);
  }

  showPfc(&clientHandle, intIfNum);

  return 0;
}


