/*********************************************************************
*
*  Copyright 2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  pml_example.c
*
* @purpose   Port MAC locking API example.
*
* @component OPEN
*
* @comments
*
* @create    9/10/2018
*
* @end
*
**********************************************************************/

#include <unistd.h>
#include <stdlib.h>
#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_pml.h"

/*********************************************************************
* @purpose  Get the global admin mode.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getGlobalAdminMode(openapiClientHandle_t *clientHandle)
{
  open_error_t ret;
  OPEN_CONTROL_t mode;

  ret = openapiPmlGlobalAdminModeGet(clientHandle, &mode);
  if (ret != OPEN_E_NONE)
  {
    printf("%s: openapiPmlGlobalAdminModeGet failed (%d)\n",
           __FUNCTION__, ret);
  }
  else
  {
    printf("global admin mode %s\n", (mode == OPEN_ENABLE ? "enable" : "disable"));
  }
}

/*********************************************************************
* @purpose  Get whether an interface is a valid PML interface or not.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    ifNum           @b{(input)}   interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getPmlInterfaceIsValid(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t ret;
  OPEN_BOOL_t isValid;

  ret = openapiPmlIntfIsValid(clientHandle, ifNum, &isValid);
  if (ret != OPEN_E_NONE)
  {
    printf("%s: openapiPmlIntfIsValid failed (%d)\n",
           __FUNCTION__, ret);
  }
  else
  {
    printf("interface %d is %svalid\n", ifNum, (isValid == OPEN_TRUE ? "" : "not "));
  }
}

/*********************************************************************
* @purpose  Get mode for a valid PML interface.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    ifNum           @b{(input)}   interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getMode(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t ret;
  OPEN_CONTROL_t mode;

  ret = openapiPmlIntfModeGet(clientHandle, ifNum, &mode);
  if (ret != OPEN_E_NONE)
  {
    printf("%s: openapiPmlIntfModeGet failed (%d)\n",
           __FUNCTION__, ret);
  }
  else
  {
    printf("interface %d mode %s\n", ifNum, (mode == OPEN_ENABLE ? "enable" : "disable"));
  }
}

/*********************************************************************
* @purpose  Get dynamic limit for a valid PML interface.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    ifNum           @b{(input)}   interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getDynamicLimit(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t ret;
  uint32_t limit;

  ret = openapiPmlIntfDynamicLimitGet(clientHandle, ifNum, &limit);
  if (ret != OPEN_E_NONE)
  {
    printf("%s: openapiPmlIntfDynamicLimitGet failed (%d)\n",
           __FUNCTION__, ret);
  }
  else
  {
    printf("interface %d dynamic limit %u\n", ifNum, limit);
  }
}

/*********************************************************************
* @purpose  Get static limit for a valid PML interface.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    ifNum           @b{(input)}   interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getStaticLimit(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t ret;
  uint32_t limit;

  ret = openapiPmlIntfStaticLimitGet(clientHandle, ifNum, &limit);
  if (ret != OPEN_E_NONE)
  {
    printf("%s: openapiPmlIntfStaticLimitGet failed (%d)\n",
           __FUNCTION__, ret);
  }
  else
  {
    printf("interface %d static limit %u\n", ifNum, limit);
  }
}

/*********************************************************************
* @purpose  Get violation trap mode for a valid PML interface.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    ifNum           @b{(input)}   interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getViolationTrapMode(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t ret;
  OPEN_CONTROL_t mode;

  ret = openapiPmlIntfViolationTrapModeGet(clientHandle, ifNum, &mode);
  if (ret != OPEN_E_NONE)
  {
    printf("%s: openapiPmlIntfViolationTrapModeGet failed (%d)\n",
           __FUNCTION__, ret);
  }
  else
  {
    printf("interface %d violation trap mode %s\n", ifNum, (mode == OPEN_ENABLE ? "enable" : "disable"));
  }
}

/*********************************************************************
* @purpose  Get violation D-Disable mode for a valid PML interface.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    ifNum           @b{(input)}   interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getViolationDDisableMode(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t ret;
  OPEN_CONTROL_t mode;

  ret = openapiPmlIntfViolationDDisableModeGet(clientHandle, ifNum, &mode);
  if (ret != OPEN_E_NONE)
  {
    printf("%s: openapiPmlIntfViolationDDisableModeGet failed (%d)\n",
           __FUNCTION__, ret);
  }
  else
  {
    printf("interface %d violation D-Disable mode %s\n", ifNum, (mode == OPEN_ENABLE ? "enable" : "disable"));
  }
}

/*********************************************************************
* @purpose  Get sticky mode for a valid PML interface.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    ifNum           @b{(input)}   interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getStickyMode(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t ret;
  OPEN_CONTROL_t mode;

  ret = openapiPmlIntfStickyModeGet(clientHandle, ifNum, &mode);
  if (ret != OPEN_E_NONE)
  {
    printf("%s: openapiPmlIntfStickyModeGet failed (%d)\n",
           __FUNCTION__, ret);
  }
  else
  {
    printf("interface %d sticky mode %s\n", ifNum, (mode == OPEN_ENABLE ? "enable" : "disable"));
  }
}

/*******************************************************************
*
* @purpose  This function prints usage for the example program.
*
* @param    name    @b{(input)}   application name
*
* @returns  none
*
* @end
*********************************************************************/
void printAppMenu(char *name)
{
  printf("\nUsage: %s <test#> <arg1> <arg2> ...\n\n", name);
  printf("Test  1: get the global admin mode: %s 1\n", name);
  printf("Test  2: get whether interface is valid PML interface: %s 2 <intf>\n", name);
  printf("Test  3: get the mode for a valid PML interface: %s 3 <intf>\n", name);
  printf("Test  4: get the dynamic limit for a valid PML interface: %s 4 <intf>\n", name);
  printf("Test  5: get the static limit for a valid PML interface: %s 5 <intf>\n", name);
  printf("Test  6: get the violation trap mode for a valid PML interface: %s 6 <intf>\n", name);
  printf("Test  7: get the violation D-Disable mode for a valid PML interface: %s 7 <intf>\n", name);
  printf("Test  8: get the sticky mode for a valid PML interface: %s 8 <intf>\n", name);
  printf("Test  9: run negative tests: %s 9 <intf>\n", name);
  return;
}

/*******************************************************************
*
* @purpose  Run API negative tests.
*
* @param    clientHandle    @b{(input)}   Client handle from registration API
* @param    intf            @b{(input)}   interface
*
* @returns  none
*
* @end
*********************************************************************/
void runNegativeTests(openapiClientHandle_t *clientHandle, uint32_t intf)
{
  OPEN_CONTROL_t mode;
  OPEN_BOOL_t valid;
  uint32_t invalidIntf = 0xffffffff;
  open_error_t result;
  uint32_t value;

  /* openapiPmlGlobalAdminModeGet() */
  printf("\nTesting openapiPmlGlobalAdminModeGet(): \n");
  result = openapiPmlGlobalAdminModeGet(NULL, &mode);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiPmlGlobalAdminModeGet(clientHandle, NULL);
  printf("NULL mode arg. (result = %d)\n", result);

  /* openapiPmlIntfIsValid() */
  printf("\nTesting openapiPmlIntfIsValid(): \n");
  result = openapiPmlIntfIsValid(NULL, intf, &valid);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiPmlIntfIsValid(clientHandle, intf, NULL);
  printf("NULL valid arg. (result = %d)\n", result);
  result = openapiPmlIntfIsValid(clientHandle, invalidIntf, &valid);
  printf("Likely invalid intf arg. (result = %d)\n", result);

  /* openapiPmlIntfModeGet() */
  printf("\nTesting openapiPmlIntfModeGet(): \n");
  result = openapiPmlIntfModeGet(NULL, intf, &mode);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiPmlIntfModeGet(clientHandle, intf, NULL);
  printf("NULL mode arg. (result = %d)\n", result);
  result = openapiPmlIntfModeGet(clientHandle, invalidIntf, &mode);
  printf("Likely invalid intf arg. (result = %d)\n", result);

  /* openapiPmlIntfDynamicLimitGet() */
  printf("\nTesting openapiPmlIntfDynamicLimitGet(): \n");
  result = openapiPmlIntfDynamicLimitGet(NULL, intf, &value);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiPmlIntfDynamicLimitGet(clientHandle, intf, NULL);
  printf("NULL limit argument. (result = %d)\n", result);
  result = openapiPmlIntfDynamicLimitGet(clientHandle, invalidIntf, &value);
  printf("Likely invalid intf arg. (result = %d)\n", result);

  /* openapiPmlIntfStaticLimitGet() */
  printf("\nTesting openapiPmlIntfStaticLimitGet(): \n");
  result = openapiPmlIntfStaticLimitGet(NULL, intf, &value);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiPmlIntfStaticLimitGet(clientHandle, intf, NULL);
  printf("NULL limit argument. (result = %d)\n", result);
  result = openapiPmlIntfStaticLimitGet(clientHandle, invalidIntf, &value);
  printf("Likely invalid intf arg. (result = %d)\n", result);

  /* openapiPmlIntfViolationTrapModeGet() */
  printf("\nTesting openapiPmlIntfViolationTrapModeGet(): \n");
  result = openapiPmlIntfViolationTrapModeGet(NULL, intf, &mode);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiPmlIntfViolationTrapModeGet(clientHandle, intf, NULL);
  printf("NULL mode arg. (result = %d)\n", result);
  result = openapiPmlIntfViolationTrapModeGet(clientHandle, invalidIntf, &mode);
  printf("Likely invalid intf arg. (result = %d)\n", result);

  /* openapiPmlIntfViolationDDisableModeGet() */
  printf("\nTesting openapiPmlIntfViolationDDisableModeGet(): \n");
  result = openapiPmlIntfViolationDDisableModeGet(NULL, intf, &mode);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiPmlIntfViolationDDisableModeGet(clientHandle, intf, NULL);
  printf("NULL mode arg. (result = %d)\n", result);
  result = openapiPmlIntfViolationDDisableModeGet(clientHandle, invalidIntf, &mode);
  printf("Likely invalid intf arg. (result = %d)\n", result);

  /* openapiPmlIntfStickyModeGet() */
  printf("\nTesting openapiPmlIntfStickyModeGet(): \n");
  result = openapiPmlIntfStickyModeGet(NULL, intf, &mode);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiPmlIntfStickyModeGet(clientHandle, intf, NULL);
  printf("NULL mode arg. (result = %d)\n", result);
  result = openapiPmlIntfStickyModeGet(clientHandle, invalidIntf, &mode);
  printf("Likely invalid intf arg. (result = %d)\n", result);

  return;
}

int main(int argc, char *argv[])
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  uint32_t intf;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);
  if (argc >= 3)
  {
     intf = atoi(argv[2]);
  }

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister(argv[0], &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);

  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 2)
      {
        getGlobalAdminMode(&clientHandle);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 3)
      {
        getPmlInterfaceIsValid(&clientHandle, intf);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 3)
      {
        getMode(&clientHandle, intf);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 3)
      {
        getDynamicLimit(&clientHandle, intf);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 3)
      {
        getStaticLimit(&clientHandle, intf);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 3)
      {
        getViolationTrapMode(&clientHandle, intf);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 3)
      {
        getViolationDDisableMode(&clientHandle, intf);
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 3)
      {
        getStickyMode(&clientHandle, intf);
        show_help = 0;
      }
      break;

    case 9:
      if (argc == 3)
      {
        runNegativeTests(&clientHandle, intf);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping PML API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
