=begin
*
* Copyright 2016-2018 Broadcom.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0*
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
=end

require "OpEN"
require "OpENUtil"

Puppet::Type.type(:netdev_interface).provide(:icos) do
  confine :exists => "/etc/brl-release"
  @doc = "NOS netdev interface"

  mk_resource_methods

  def exists?
    init_resource
    @property_hash[:ensure] == :present
  end

  def create
    begin
      printf "netdev_stdlib_icos-interface create\n"

      open = OpENUtil.new()
      open.connect("netdev_stdlib_icos-interface")

      # Get the internal interface number
      intf_num_p      = OpEN.new_uint32_tp()
      buffdesc        = OpEN::Open_buffdesc.new
      buffdesc.size   = resource[:name].length + 1
      buffdesc.pstart = open.getCharBuffer(buffdesc.size, resource[:name])
      ret = OpEN::openapiIfNumGet(open.client, buffdesc, intf_num_p)

      if ret == OpEN::OPEN_E_NONE 
        intf_num = OpEN.uint32_tp_value(intf_num_p)

        # Post a info message that the Interface can't be created.
        # Only its properties can be set.
        info('Interface can not be created. Only its properties can be set.')

        # Set the interface admin state
        if resource[:admin] == :down
          admin_state = OpEN::OPEN_DISABLE
        else
          admin_state = OpEN::OPEN_ENABLE
        end
        ret = OpEN::openapiIfAdminStateSet(open.client, intf_num, admin_state)
        if ret != OpEN::OPEN_E_NONE
          printf "Failed to set interface %s admin state, return value = %d\n", resource[:name], ret
          warning('Failed to set interface admin state.')
        end

        # Set the interface description
        buffdesc        = OpEN::Open_buffdesc.new
        buffdesc.size   = resource[:description].length + 1
        buffdesc.pstart = open.getCharBuffer(buffdesc.size, resource[:description])
        ret = OpEN::openapiIfDescrSet(open.client, intf_num, buffdesc)
        if ret != OpEN::OPEN_E_NONE
          printf "Failed to set interface %s description, return value = %d\n", resource[:name], ret
          warning('Failed to set interface description.')
        end

        # Set the interface MTU size
        ret = OpEN::openapiIfMtuSizeSet(open.client, intf_num, resource[:mtu])
        if ret != OpEN::OPEN_E_NONE
          printf "Failed to set interface %s MTU size to %d, return value = %d\n", resource[:name], resource[:mtu], ret
          warning('Failed to set interface MTU size.')
        end

        # Set the interface speed and duplex
        case resource[:speed]

          when :'10m' then
            case resource[:duplex]
              when :full then
                port_speed = OpEN::OPEN_PORTCTRL_PORTSPEED_FULL_10T
                autoneg = OpEN::OPEN_PORT_AUTO_NEG_DISABLE
              else
                port_speed = OpEN::OPEN_PORTCTRL_PORTSPEED_HALF_10T
                autoneg = OpEN::OPEN_PORT_AUTO_NEG_DISABLE
            end

          when :'100m' then
            case resource[:duplex]
              when :full then
                port_speed = OpEN::OPEN_PORTCTRL_PORTSPEED_FULL_100TX
                autoneg = OpEN::OPEN_PORT_AUTO_NEG_DISABLE
              else
                port_speed = OpEN::OPEN_PORTCTRL_PORTSPEED_HALF_100TX
                autoneg = OpEN::OPEN_PORT_AUTO_NEG_DISABLE
            end

          when :'1g' then
            port_speed = OpEN::OPEN_PORTCTRL_PORTSPEED_FULL_1000SX
            autoneg = OpEN::OPEN_PORT_AUTO_NEG_DISABLE

          when :'10g' then
            port_speed = OpEN::OPEN_PORTCTRL_PORTSPEED_FULL_10GSX
            autoneg = OpEN::OPEN_PORT_AUTO_NEG_DISABLE

          else
            port_speed = OpEN::OPEN_PORTCTRL_PORTSPEED_AUTO_NEG
            autoneg = OpEN::OPEN_PORT_AUTO_NEG_ENABLE
        end

        ret1 = OpEN::openapiIfAutoNegSet(open.client, intf_num, autoneg)
        ret2 = OpEN::openapiIfSpeedSet(open.client, intf_num, port_speed)
        if ret1 != OpEN::OPEN_E_NONE || ret2 != OpEN::OPEN_E_NONE
          printf "Failed to set interface %s speed %s and duplex %s, return value = %d, %d\n", resource[:name], resource[:speed], resource[:duplex], ret1, ret2
          warning('Failed to set interface speed and duplex.')
        end

      else
        printf "Interface %s can not be created, return value = %d\n", resource[:name], ret
        fail('Failed to create interface.')
      end

      open.terminate()

      @property_hash[:ensure] = :present

    rescue Puppet::ExecutionFailure => e
      printf "Failed to create interface\n"

    ensure
      if intf_num_p
        OpEN.delete_uint32_tp(intf_num_p)
      end
    end
  end

  def destroy
    begin
      printf "netdev_stdlib_icos-interface destroy\n"

      # Post a info message that the Interface can't be deleted.
      info('Interface can not be deleted.')

      @property_hash.clear
    rescue Puppet::ExecutionFailure => e
      printf "Failed to destroy interface\n"
    end
  end

  def self.instances
    begin
      printf "netdev_stdlib_icos-interface self.instances\n"

      open = OpENUtil.new()
      open.connect("netdev_stdlib_icos-interface")

      interfaces = []

      # Get the internal interface number
      intf_num   = 0
      intf_num_p = OpEN.new_uint32_tp()
      intf_type  = OpEN::OPEN_INTF_TYPE_PHY

      buffdesc      = OpEN::Open_buffdesc.new
      admin_state_p = OpEN.new_OPEN_CONTROL_tp()
      descr_len_p   = OpEN.new_uint32_tp()
      mtu_size_p    = OpEN.new_uint32_tp()
      speed_p       = OpEN.new_OPEN_PORT_SPEEDS_tp()

      ret = OpEN::openapiIfNextGet(open.client, intf_type, intf_num, intf_num_p)

      while ret == OpEN::OPEN_E_NONE
        intf_num = OpEN.uint32_tp_value(intf_num_p)

        # Get the interface name
        buffdesc.size   = 256
        name = open.getCharBuffer(buffdesc.size, nil)
        buffdesc.pstart = name 
        OpEN::openapiIfNameGet(open.client, intf_num, buffdesc)
        intf_name = name.cast()

        # Get the interface admin state
        OpEN::openapiIfAdminStateGet(open.client, intf_num, admin_state_p)
        admin_state = OpEN.OPEN_CONTROL_tp_value(admin_state_p)

        if admin_state == OpEN::OPEN_ENABLE
          admin = :up
        else
          admin = :down
        end

        # Get the interface description
        OpEN::openapiIfDescrSizeGet(open.client, intf_num, descr_len_p)
        descr_len = OpEN.uint32_tp_value(descr_len_p)

        buffdesc.size   = descr_len
        desc = open.getCharBuffer(descr_len, nil)
        buffdesc.pstart = desc 
        OpEN::openapiIfDescrGet(open.client, intf_num, buffdesc)
        description = desc.cast()

        # Get the interface MTU size
        OpEN::openapiIfMtuSizeGet(open.client, intf_num, mtu_size_p)
        mtu = OpEN.uint32_tp_value(mtu_size_p)

        # Get the interface speed and duplex
        OpEN::openapiIfSpeedGet(open.client, intf_num, speed_p)
        speed = OpEN.OPEN_PORT_SPEEDS_tp_value(speed_p)

        case speed

          when OpEN::OPEN_PORTCTRL_PORTSPEED_FULL_10T
            port_speed = :'10m'
            port_duplex = :full

          when OpEN::OPEN_PORTCTRL_PORTSPEED_HALF_10T
            port_speed = :'10m'
            port_duplex = :half

          when OpEN::OPEN_PORTCTRL_PORTSPEED_FULL_100TX
            port_speed = :'100m'
            port_duplex = :full

          when OpEN::OPEN_PORTCTRL_PORTSPEED_HALF_100TX
            port_speed = :'100m'
            port_duplex = :half

          when OpEN::OPEN_PORTCTRL_PORTSPEED_FULL_1000SX
            port_speed = :'1g'
            port_duplex = :full

          when OpEN::OPEN_PORTCTRL_PORTSPEED_FULL_10GSX
            port_speed = :'10g'
            port_duplex = :full

          when OpEN::OPEN_PORTCTRL_PORTSPEED_AUTO_NEG
            port_speed = :auto
            port_duplex = :full

          else
            port_speed = :auto
            port_duplex = :auto
        end

        interfaces << new(:name => intf_name, :admin => admin, :description => description, :mtu => mtu, :speed => port_speed, :duplex => port_duplex, :ensure => :present)
        ret = OpEN::openapiIfNextGet(open.client, intf_type, intf_num, intf_num_p)
      end # Loop through all the available interfaces

      OpEN.delete_uint32_tp(intf_num_p)
      OpEN.delete_OPEN_CONTROL_tp(admin_state_p)
      OpEN.delete_uint32_tp(descr_len_p)
      OpEN.delete_uint32_tp(mtu_size_p)
      OpEN.delete_OPEN_PORT_SPEEDS_tp(speed_p)

      open.terminate()

      interfaces
    end
  end

  def self.prefetch(resources)
    begin
      printf "netdev_stdlib_icos-interface self.prefetch\n"

      interfaces = instances
      resources.each do |name, params|
        # Initialize with the default MTU and Description
        isMtuReset = 0
        if params[:mtu].to_i == 0
          params[:mtu] = 1500 # this is the default MTU size for the NOS interfaces
          isMtuReset = 1
        end
        params[:description] ||= "Puppet modified interface: #{params[:name]}"

        if params[:ensure] == :present # This is to support re-configuration
          if provider = interfaces.find { |interface| (interface.name == name) and (interface.admin == params[:admin]) and (interface.description == params[:description]) and (interface.mtu = params[:mtu]) and (interface.speed == params[:speed]) and (interface.duplex == params[:duplex]) }
            resources[name].provider = provider
          elsif isMtuReset == 1
            # Reset the MTU size to minimum supported value of NOS interfaces
            params[:mtu] = 1518
          end
        elsif params[:ensure] == :absent
          if provider = interfaces.find { |interface| (interface.name == name) }
            resources[name].provider = provider
          end
        end
      end
    end
  end

  def init_resource
    resource[:mtu] ||= 1518 # this is the minimum MTU size supported for the configuration of NOS interfaces
    resource[:description] ||= default_description
  end

  def default_description
    "Puppet modified interface: #{resource[:name]}"
  end

end
