#!/mnt/fastpath/usr/bin/ruby

require "OpEN"
require "OpENUtil"
require "socket"
require "ipaddr"
require "optparse"

#
# Copyright 2016 Broadcom.
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#


#
# Ruby 1.8.7.
#

$open = OpENUtil.new()
$verbose = false

# Define some constants for comparison convenience
SET                      = 'set'
MAP                      = 'map'

# Define SNMP users
SNMP_USER_LIST = [
  ["admin1", "network-admin1", OpEN::OPEN_USM_USER_AUTH_PROTOCOL_NONE, "", "", OpEN::OPEN_USM_USER_PRIV_PROTOCOL_NONE, "", "", ""],
  ["admin2", "network-admin",  OpEN::OPEN_USM_USER_AUTH_PROTOCOL_SHA, "ThisIsaShaPasswordThatIs32Chars.", "", OpEN::OPEN_USM_USER_PRIV_PROTOCOL_NONE, "", "", "0123456789abcdef"],
  ["admin3", "network-admin",  OpEN::OPEN_USM_USER_AUTH_PROTOCOL_SHA, "", "ThisIsaShaKeywordThatMustBe48CharactersInLength.", OpEN::OPEN_USM_USER_PRIV_PROTOCOL_NONE, "", "", "bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb"],
  ["admin4", "network-admin",  OpEN::OPEN_USM_USER_AUTH_PROTOCOL_SHA, "ThisIsaShaPasswordThatIs32Chars.", "", OpEN::OPEN_USM_USER_PRIV_PROTOCOL_DES, "ThisIsaDesPasswordThatIs32Chars.", "", "cccccccccccccccccccccccccccccccc"],
  ["admin5", "network-admin",  OpEN::OPEN_USM_USER_AUTH_PROTOCOL_SHA, "", "ThisIsaShaKeywordThatMustBe48CharactersInLength.", OpEN::OPEN_USM_USER_PRIV_PROTOCOL_DES, "", "ThisIsaDesKeywordThatMustBe32chr", "dddddddddddddddddddddddddddddddd"],
  ["delete-user", "network-admin", OpEN::OPEN_USM_USER_AUTH_PROTOCOL_SHA, "", "ThisIsaShaKeywordThatMustBe48CharactersInLength.", OpEN::OPEN_USM_USER_PRIV_PROTOCOL_DES, "", "ThisIsaDesKeywordThatMustBe32chr", "dddddddddddddddddddddddddddddddd"],
]

# Define SNMP groups
SNMP_GROUP_LIST = [
  ["network-admin", OpEN::OPEN_SNMP_SECURITY_MODEL_USM, OpEN::OPEN_SNMP_SECURITY_LEVEL_AUTHPRIV, "context-1", "view-1", "view-1", "view-1"],
  ["network-guest", OpEN::OPEN_SNMP_SECURITY_MODEL_SNMPV1, OpEN::OPEN_SNMP_SECURITY_LEVEL_NOAUTHNOPRIV, "context-2", "view-2", "view-2", "view-2"],
  ["delete-group",  OpEN::OPEN_SNMP_SECURITY_MODEL_SNMPV2C, OpEN::OPEN_SNMP_SECURITY_LEVEL_NOAUTHNOPRIV, "context-3", "view-3", "view-3", "view-3"],
]

# Define SNMP views
SNMP_VIEW_LIST = [
  ["view-1", "1.3.6.1.4.1.4413", OpEN::OPEN_SNMP_VIEW_TYPE_INCLUDED],
  ["view-2", "ifEntry.*.1", OpEN::OPEN_SNMP_VIEW_TYPE_EXCLUDED],
]

# Define SNMP filters
SNMP_FILTER_LIST = [
  ["filter-1", "1.3.6.1.4.1.4413", OpEN::OPEN_SNMP_VIEW_TYPE_INCLUDED],
  ["filter-2", "ifEntry.*.1", OpEN::OPEN_SNMP_VIEW_TYPE_EXCLUDED],
]

# Define SNMP hosts
SNMP_HOST_LIST = [
  ["10.10.10.1", 0, "public", OpEN::OPEN_SNMP_SECURITY_MODEL_SNMPV1, OpEN::OPEN_SNMP_SECURITY_LEVEL_NOAUTHNOPRIV, OpEN::OPEN_SNMP_NOTIFY_TYPE_TRAP, 0, 0, ""],
  ["10.10.10.2", 0, "public", OpEN::OPEN_SNMP_SECURITY_MODEL_SNMPV2C, OpEN::OPEN_SNMP_SECURITY_LEVEL_NOAUTHNOPRIV, OpEN::OPEN_SNMP_NOTIFY_TYPE_TRAP, 0, 0, ""],
  ["2001::", 65535, "admin", OpEN::OPEN_SNMP_SECURITY_MODEL_USM, OpEN::OPEN_SNMP_SECURITY_LEVEL_AUTHPRIV, OpEN::OPEN_SNMP_NOTIFY_TYPE_INFORM, 100, 200, "ipv6-filter"],
]

# Define SNMP communities and community maps
SNMP_COMMUNITY_LIST = [
  [SET, ["public-test", OpEN::OPEN_SNMP_COMMUNITY_ACCESS_NONE, "view_test", "20.20.20.1"]],
  [SET, ["private-test", OpEN::OPEN_SNMP_COMMUNITY_ACCESS_RW, "", ""]],
  [SET, ["delete-community", OpEN::OPEN_SNMP_COMMUNITY_ACCESS_RO, "", ""]],
  [MAP, ["delete-private-map", OpEN::OPENAPI_SNMP_DEFAULT_WRITE_STR, "30.30.30.1"]],
]


def print_sanity_results(result, test, msg, feat)
  #Print overall comparison results

  if result == OpEN::OPEN_E_UNAVAIL
    puts "Sanity test skipped."
  elsif result == OpEN::OPEN_E_NONE and test == true
    puts "Sanity Success - #{msg} - #{feat}."
  else
    puts "Sanity Failure - #{msg} - #{feat}."
  end
end

def print_bad_result(result, msg)
  #Print some general error messages if the result is bad
   
  if result == OpEN::OPEN_E_UNAVAIL
    puts "Feature not supported - #{msg} (err #{result})."
  elsif result != OpEN::OPEN_E_NONE
    puts "Test Failure - #{msg} (err #{result})."
  end
end

def get_type(itype)
  case itype
    when OpEN::OPEN_INTF_TYPE_PHY; return 'Port'
    when OpEN::OPEN_INTF_TYPE_VLAN; return 'VLAN'
    when OpEN::OPEN_INTF_TYPE_LOOPBACK; return 'Loop'
    when OpEN::OPEN_INTF_TYPE_TUNNEL; return 'Tnnl'
    else return ''
  end
end

def get_link(state)
  case state
    when OpEN::OPEN_LINK_UP; return 'Up'
    when OpEN::OPEN_LINK_DOWN; return 'Down'
    else return ''
  end
end

class SnmpExample
  #Simple SNMP class implementing basic CRUD examples

  def initialize(client)
    @client = client
  end

  def test_snmp_sys_name_set(name)
    #Set the SNMP system name.
    #An administratively-assigned name for this managed node.
    #By convention, this is the node's fully-qualified domain name.

    name_string = $open.getCharBuffer(name.length, name)
    name_buff = OpEN::Open_buffdesc.new
    name_buff.pstart = name_string
    name_buff.size = name.length+1

    result = OpEN.openapiSnmpSysNameSet(@client, name_buff) 
    print_bad_result(result, 'openapiSnmpSysNameSet')
    print_sanity_results(result, true, 'openapiSnmpSysNameSet', name) 
  end

  def test_snmp_sys_location_set(location)
    #Set the SNMP system location.
    #The physical location of this node, for example;
    #'telephone closet' or '3rd floor'.

    name_string = $open.getCharBuffer(location.length, location)
    name_buff = OpEN::Open_buffdesc.new
    name_buff.pstart = name_string
    name_buff.size = location.length+1

    result = OpEN.openapiSnmpSysLocationSet(@client, name_buff)
    print_bad_result(result, 'openapiSnmpSysLocationSet')
    print_sanity_results(result, true, 'openapiSnmpSysLocationSet', location)
  end

  def test_snmp_sys_contact_set(contact)
    #Set the SNMP system contact.
    #The textual identification of the contact person for this managed node,
    #together with information on how to contact this person.

    name_string = $open.getCharBuffer(contact.length, contact)
    name_buff = OpEN::Open_buffdesc.new
    name_buff.pstart = name_string
    name_buff.size = contact.length+1

    result = OpEN.openapiSnmpSysContactSet(@client, name_buff) 
    print_bad_result(result, 'openapiSnmpSysContactSet')
    print_sanity_results(result, true, 'openapiSnmpSysContactSet', contact)
  end

  def test_snmp_local_engine_id_set(engine_id)
    #Set the SNMP engine id on the local device.

    if engine_id == 'default'
      engine_id = '1'
    end

    name_string = $open.getCharBuffer(engine_id.length, engine_id)
    name_buff = OpEN::Open_buffdesc.new
    name_buff.pstart = name_string
    name_buff.size = engine_id.length+1

    result = OpEN.openapiSnmpLocalEngineIdSet(@client, name_buff)
    print_bad_result(result, 'openapiSnmpLocalEngineIdSet')
    print_sanity_results(result, true, 'openapiSnmpLocalEngineIdSet', engine_id)
  end

  def test_snmp_user_create(user_list)
    #Create a new SNMP user

    for val in user_list
      name       = val[0]
      group      = val[1]
      auth_proto = val[2]
      auth_pwd   = val[3]
      auth_key   = val[4]
      priv_proto = val[5]
      priv_pwd   = val[6]
      priv_key   = val[7]
      engine_id  = val[8]

      name_string = $open.getCharBuffer(name.length, name)
      name_buff = OpEN::Open_buffdesc.new
      name_buff.pstart = name_string
      name_buff.size = name.length+1

      group_string = $open.getCharBuffer(group.length, group)
      group_buff = OpEN::Open_buffdesc.new
      group_buff.pstart = group_string
      group_buff.size = group.length+1

      auth_pwd_string = $open.getCharBuffer(auth_pwd.length, auth_pwd)
      auth_pwd_buff = OpEN::Open_buffdesc.new
      auth_pwd_buff.pstart = auth_pwd_string
      auth_pwd_buff.size = auth_pwd.length+1

      auth_key_string = $open.getCharBuffer(auth_key.length, auth_key)
      auth_key_buff = OpEN::Open_buffdesc.new
      auth_key_buff.pstart = auth_key_string
      auth_key_buff.size = auth_key.length+1

      priv_pwd_string = $open.getCharBuffer(priv_pwd.length, priv_pwd)
      priv_pwd_buff = OpEN::Open_buffdesc.new
      priv_pwd_buff.pstart = priv_pwd_string
      priv_pwd_buff.size = priv_pwd.length+1

      priv_key_string = $open.getCharBuffer(priv_key.length, priv_key)
      priv_key_buff = OpEN::Open_buffdesc.new
      priv_key_buff.pstart = priv_key_string
      priv_key_buff.size = priv_key.length+1

      engine_id_string = $open.getCharBuffer(engine_id.length, engine_id)
      engine_id_buff = OpEN::Open_buffdesc.new
      engine_id_buff.pstart = engine_id_string
      engine_id_buff.size = engine_id.length+1

      result = OpEN.openapiSnmpUserCreate(@client,
                                          name_buff, group_buff,
                                          auth_proto, auth_pwd_buff, auth_key_buff,
                                          priv_proto, priv_pwd_buff, priv_key_buff,
                                          engine_id_buff)
      print_sanity_results(result, true, 'openapiSnmpUserCreate', name)
    end
  end

  def test_snmp_group_create(group_list)
    #Create a new SNMP group

    for val in group_list
      group          = val[0]
      security_model = val[1]
      security_level = val[2]
      context_prefix = val[3]
      read_view      = val[4]
      write_view     = val[5]
      notify_view    = val[6]

      group_string = $open.getCharBuffer(group.length, group)
      group_buff = OpEN::Open_buffdesc.new
      group_buff.pstart = group_string
      group_buff.size = group.length+1

      context_prefix_string = $open.getCharBuffer(context_prefix.length, context_prefix)
      context_prefix_buff = OpEN::Open_buffdesc.new
      context_prefix_buff.pstart = context_prefix_string
      context_prefix_buff.size = context_prefix.length+1

      read_view_string = $open.getCharBuffer(read_view.length, read_view)
      read_view_buff = OpEN::Open_buffdesc.new
      read_view_buff.pstart = read_view_string
      read_view_buff.size = read_view.length+1

      write_view_string = $open.getCharBuffer(write_view.length, write_view)
      write_view_buff = OpEN::Open_buffdesc.new
      write_view_buff.pstart = write_view_string
      write_view_buff.size = write_view.length+1

      notify_view_string = $open.getCharBuffer(notify_view.length, notify_view)
      notify_view_buff = OpEN::Open_buffdesc.new
      notify_view_buff.pstart = notify_view_string
      notify_view_buff.size = notify_view.length+1

      result = OpEN.openapiSnmpGroupCreate(@client,
                                           group_buff,
                                           security_model, security_level,
                                           context_prefix_buff, read_view_buff,
                                           write_view_buff, notify_view_buff)
      print_sanity_results(result, true, 'openapiSnmpGroupCreate', group)
    end
  end

  def test_snmp_view_create(view_list)
    #Create a new SNMP view

    for val in view_list
      view_name = val[0]
      oid_tree  = val[1]
      view_type = val[2]

      view_name_string = $open.getCharBuffer(view_name.length, view_name)
      view_name_buff = OpEN::Open_buffdesc.new
      view_name_buff.pstart = view_name_string
      view_name_buff.size = view_name.length+1

      oid_tree_string = $open.getCharBuffer(oid_tree.length, oid_tree)
      oid_tree_buff = OpEN::Open_buffdesc.new
      oid_tree_buff.pstart = oid_tree_string
      oid_tree_buff.size = oid_tree.length+1

      result = OpEN.openapiSnmpViewCreate(@client, view_name_buff, oid_tree_buff, view_type)
      print_sanity_results(result, true, 'openapiSnmpViewCreate', view_name) 
    end
  end

  def test_snmp_filter_create(filter_list)
    #Create a new SNMP filter

    for val in filter_list
      filter_name = val[0]
      oid_tree    = val[1]
      view_type   = val[2]

      filter_name_string = $open.getCharBuffer(filter_name.length, filter_name)
      filter_name_buff = OpEN::Open_buffdesc.new
      filter_name_buff.pstart = filter_name_string
      filter_name_buff.size = filter_name.length+1

      oid_tree_string = $open.getCharBuffer(oid_tree.length, oid_tree)
      oid_tree_buff = OpEN::Open_buffdesc.new
      oid_tree_buff.pstart = oid_tree_string
      oid_tree_buff.size = oid_tree.length+1

      result = OpEN.openapiSnmpFilterCreate(@client, filter_name_buff, oid_tree_buff, view_type)
      print_sanity_results(result, true, 'openapiSnmpFilterCreate', filter_name)
    end
  end

  def test_snmp_host_create(host_list)
    #Create a new SNMP host

    for val in host_list
      host           = val[0]
      server_port    = val[1]
      security_name  = val[2]
      security_model = val[3]
      security_level = val[4]
      notify_type    = val[5]
      timeout        = val[6]
      retries        = val[7]
      filter_name    = val[8]

      host_string = $open.getCharBuffer(host.length, host)
      host_buff = OpEN::Open_buffdesc.new
      host_buff.pstart = host_string
      host_buff.size = host.length+1

      security_name_string = $open.getCharBuffer(security_name.length, security_name)
      security_name_buff = OpEN::Open_buffdesc.new
      security_name_buff.pstart = security_name_string
      security_name_buff.size = security_name.length+1

      filter_name_string = $open.getCharBuffer(filter_name.length, filter_name)
      filter_name_buff = OpEN::Open_buffdesc.new
      filter_name_buff.pstart = filter_name_string
      filter_name_buff.size = filter_name.length+1

      result = OpEN.openapiSnmpHostCreate(@client,
                                          host_buff, server_port,
                                          security_name_buff, security_model, security_level,
                                          notify_type, timeout, retries, filter_name_buff)
      print_sanity_results(result, true, 'openapiSnmpHostCreate', host)
    end
  end

  def test_snmp_community_create(community_list)
    #Create a new SNMP community
    #SET key defines a basic community creation.
    #MAP key indicates a map to a new internal security name for SNMP v1 and SNMP v2 security
    #models to an existing group name.

    key = ''
 
    community_list.each do |community|
      community.collect! do |val|
        if val.class == String
          key = val
        elsif val.class == Array
          if key == SET
            name    = val[0]
            access  = val[1]
            view    = val[2]
            address = val[3]

            name_string = $open.getCharBuffer(name.length, name)
            name_buff = OpEN::Open_buffdesc.new
            name_buff.pstart = name_string
            name_buff.size = name.length+1

            view_string = $open.getCharBuffer(view.length, view)
            view_buff = OpEN::Open_buffdesc.new
            view_buff.pstart = view_string
            view_buff.size = view.length+1

            ip_addr = OpEN::Open_inet_addr_t.new
            if address.length > 0
              ip_addr.addr.ipv4 = IPAddr.new(address).to_i
              ip_addr.family = OpEN::OPEN_AF_INET
            end
            result = OpEN.openapiSnmpCommunityCreate(@client, name_buff, access, view_buff, ip_addr)
            print_sanity_results(result, true, 'openapiSnmpCommunityCreate', name)
          elsif key == MAP
            name     = val[0]
            map_name = val[1]
            address  = val[2]

            name_string = $open.getCharBuffer(name.length, name)
            name_buff = OpEN::Open_buffdesc.new
            name_buff.pstart = name_string
            name_buff.size = name.length+1

            map_name_string = $open.getCharBuffer(map_name.length, map_name)
            map_name_buff = OpEN::Open_buffdesc.new
            map_name_buff.pstart = map_name_string
            map_name_buff.size = map_name.length+1

            ip_addr = OpEN::Open_inet_addr_t.new
            if address.length > 0
              ip_addr.addr.ipv4 = IPAddr.new(address).to_i
              ip_addr.family = OpEN::OPEN_AF_INET
            end

            result = OpEN.openapiSnmpCommunityGroupCreate(@client, name_buff, map_name_buff, ip_addr)
            print_sanity_results(result, true, 'openapiSnmpCommunityGroupCreate', name)
          end
        end
      end
    end
  end

  def testSnmpTrapFlagSet(flags, mode)
    #Enable or disable the traps identified via the flags selection

    val = mode
    flag_bits = 0

    for flag in flags
      flag_bits |= flag
    end

    result = OpEN.openapiSnmpTrapFlagSet(@client, flag_bits, val)
    print_sanity_results(result, true, 'openapiSnmpTrapFlagSet', flag_bits)
  end

  def test_snmp_sys_name_get()
    #Get the SNMP system name.
    #An administratively-assigned name for this managed node.
    #By convention, this is the node's fully-qualified domain name.

    max_len = 255+1

    name_string       = $open.getCharBuffer(max_len) 
    name_buff         = OpEN::Open_buffdesc.new
    name_buff.pstart  = name_string
    name_buff.size    = max_len

    result = OpEN.openapiSnmpSysNameGet(@client, name_buff)
    print_bad_result(result, 'openapiSnmpSysNameGet')
    print_sanity_results(result, true, 'openapiSnmpSysNameGet', name_string.cast)
  end

  def test_snmp_sys_location_get()
    #Get the SNMP system location.
    #The physical location of this node, for example;
    #'telephone closet' or '3rd floor'.

    max_len = 255+1

    name_string       = $open.getCharBuffer(max_len) 
    name_buff         = OpEN::Open_buffdesc.new
    name_buff.pstart  = name_string
    name_buff.size    = max_len

    result = OpEN.openapiSnmpSysLocationGet(@client, name_buff)
    print_bad_result(result, 'openapiSnmpSysLocationGet')
    print_sanity_results(result, true, 'openapiSnmpSysLocationGet', name_string.cast()) 
  end

  def test_snmp_sys_contact_get()
    #Get the SNMP system contact.
    #The textual identification of the contact person for this managed node,
    #together with information on how to contact this person.

    max_len = 255+1

    name_string       = $open.getCharBuffer(max_len) 
    name_buff         = OpEN::Open_buffdesc.new
    name_buff.pstart  = name_string
    name_buff.size    = max_len

    result = OpEN.openapiSnmpSysContactGet(@client, name_buff)
    print_bad_result(result, 'openapiSnmpSysContactGet')
    print_sanity_results(result, true, 'openapiSnmpSysContactGet', name_string.cast()) 
  end

  def test_snmp_local_engine_id_get()
    #Get the SNMP engine id on the local device.

    max_len = 32+1

    engine_id_string       = $open.getCharBuffer(max_len) 
    engine_id_buff         = OpEN::Open_buffdesc.new
    engine_id_buff.pstart  = engine_id_string
    engine_id_buff.size    = max_len

    result = OpEN.openapiSnmpLocalEngineIdGet(@client, engine_id_buff)
    print_bad_result(result, 'openapiSnmpLocalEngineIdGet')

    if result == OpEN::OPEN_E_NONE
      engine_id = []
      for idx in 0..engine_id_buff.size-1
        engine_id += engine_id_string[idx].unpack("H*")
      end
    end

    print_sanity_results(result, true, 'openapiSnmpLocalEngineIdGet', engine_id) 
  end

  def test_snmp_user_get(user_name)
    #Get the specified SNMP user

    found = false
    max_user_len = 30+1
    max_engine_id_len = 32+1

    auth_p = OpEN.new_OPEN_USM_USER_AUTH_PROTOCOL_tp()
    priv_p = OpEN.new_OPEN_USM_USER_PRIV_PROTOCOL_tp()

    user_string       = $open.getCharBuffer(max_user_len)
    user_buff         = OpEN::Open_buffdesc.new
    user_buff.pstart  = user_string
    user_buff.size    = max_user_len

    group_string      = $open.getCharBuffer(max_user_len)
    group_buff        = OpEN::Open_buffdesc.new
    group_buff.pstart = group_string
    group_buff.size   = max_user_len

    engine_id_string      = $open.getCharBuffer(max_engine_id_len) 
    engine_id_buff        = OpEN::Open_buffdesc.new
    engine_id_buff.pstart = engine_id_string
    engine_id_buff.size   = max_engine_id_len

    next_user_string      = $open.getCharBuffer(max_user_len)
    next_user_buff        = OpEN::Open_buffdesc.new
    next_user_buff.pstart = next_user_string
    next_user_buff.size   = max_user_len

    next_engine_id_string      = $open.getCharBuffer(max_engine_id_len) 
    next_engine_id_buff        = OpEN::Open_buffdesc.new
    next_engine_id_buff.pstart = next_engine_id_string
    next_engine_id_buff.size   = max_engine_id_len

    while OpEN.openapiSnmpUserGetNext(@client, 
                                      user_buff, 
                                      engine_id_buff, 
                                      group_buff, 
                                      auth_p,
                                      priv_p, 
                                      next_user_buff, 
                                      next_engine_id_buff) == OpEN::OPEN_E_NONE

      if user_name == next_user_string.cast()
        found = true
        break
      end

      user_buff.pstart = next_user_buff.pstart
      user_buff.size   = next_user_buff.size
      engine_id_buff.pstart = next_engine_id_buff.pstart
      engine_id_buff.size   = next_engine_id_buff.size
      next_user_buff.size = max_user_len
      next_engine_id_buff.size = max_engine_id_len
    end

    print_sanity_results(OpEN::OPEN_E_NONE, found, 'openapiSnmpUserGetNext', user_name) 

    if (found and $verbose)
      print '  User name      : ', next_user_string.cast(); puts
      print '  Group name     : ', group_string.cast(); puts
      print '  Authentication : ', OpEN::OPEN_USM_USER_AUTH_PROTOCOL_tp_value(auth_p); puts
      print '  Privilege      : ', OpEN::OPEN_USM_USER_PRIV_PROTOCOL_tp_value(priv_p); puts

      engine_id = []
      for idx in 0..next_engine_id_buff.size-1
        engine_id += next_engine_id_string[idx].unpack("H*")
      end
      print '  Engine Id      : ', engine_id; puts
    end
    OpEN.delete_OPEN_USM_USER_AUTH_PROTOCOL_tp(auth_p)
    OpEN.delete_OPEN_USM_USER_PRIV_PROTOCOL_tp(priv_p)
  end

  def test_snmp_group_get(group_name, security_model, security_level, context_name)
    #Get the specified SNMP group

    found = false
    max_user_len = 30+1
    max_context_len = 30+1
    max_view_len = 30+1

    model_p = OpEN.new_OPEN_SNMP_SECURITY_MODEL_tp()
    level_p = OpEN.new_OPEN_SNMP_SECURITY_LEVEL_tp()

    group_string      = $open.getCharBuffer(max_user_len)
    group_buff        = OpEN::Open_buffdesc.new
    group_buff.pstart = group_string
    group_buff.size   = max_user_len

    read_string      = $open.getCharBuffer(max_view_len)
    read_buff        = OpEN::Open_buffdesc.new
    read_buff.pstart = read_string
    read_buff.size   = max_view_len

    write_string      = $open.getCharBuffer(max_view_len)
    write_buff        = OpEN::Open_buffdesc.new
    write_buff.pstart = write_string
    write_buff.size   = max_view_len

    notify_string      = $open.getCharBuffer(max_view_len)
    notify_buff        = OpEN::Open_buffdesc.new
    notify_buff.pstart = notify_string
    notify_buff.size   = max_view_len

    context_string      = $open.getCharBuffer(max_user_len)
    context_buff        = OpEN::Open_buffdesc.new
    context_buff.pstart = context_string
    context_buff.size   = max_context_len

    next_group_string      = $open.getCharBuffer(max_user_len)
    next_group_buff        = OpEN::Open_buffdesc.new
    next_group_buff.pstart = next_group_string
    next_group_buff.size   = max_user_len

    next_context_string      = $open.getCharBuffer(max_user_len)
    next_context_buff        = OpEN::Open_buffdesc.new
    next_context_buff.pstart = next_context_string
    next_context_buff.size   = max_user_len

    while OpEN.openapiSnmpGroupGetNext(@client, 
                                       group_buff, 
                                       model_p,
                                       level_p, 
                                       context_buff,
                                       read_buff, 
                                       write_buff, 
                                       notify_buff, 
                                       next_group_buff, 
                                       next_context_buff) == OpEN::OPEN_E_NONE

      if (group_name == next_group_string.cast() and
          context_name == next_context_string.cast()):
        found = true
        break
      end

      read_string        = $open.getCharBuffer(max_view_len)
      read_buff.pstart   = read_string
      read_buff.size     = max_view_len
      write_string       = $open.getCharBuffer(max_view_len)
      write_buff.pstart  = write_string
      write_buff.size    = max_view_len
      notify_string      = $open.getCharBuffer(max_view_len)
      notify_buff.pstart = notify_string
      notify_buff.size   = max_view_len

      group_buff.pstart = next_group_buff.pstart
      group_buff.size   = next_group_buff.size
      context_buff.pstart = next_context_buff.pstart
      context_buff.size   = next_context_buff.size
      next_group_buff.size = max_user_len
      next_context_buff.size = max_context_len
    end

    print_sanity_results(OpEN::OPEN_E_NONE, found, 'openapiSnmpGroupGetNext', group_name) 

    if (found and $verbose)
      print '  Group name     : ', next_group_string.cast(); puts
      print '  Security model : ', OpEN::OPEN_SNMP_SECURITY_MODEL_tp_value(model_p); puts
      print '  Security level : ', OpEN::OPEN_SNMP_SECURITY_LEVEL_tp_value(level_p); puts
      print '  Context name   : ', next_context_string.cast(); puts
      print '  Read view      : ', read_string.cast(); puts
      print '  Write view     : ', write_string.cast(); puts
      print '  Notify view    : ', notify_string.cast(); puts
    end

    OpEN.delete_OPEN_SNMP_SECURITY_MODEL_tp(model_p)
    OpEN.delete_OPEN_SNMP_SECURITY_LEVEL_tp(level_p)
  end

  def test_snmp_view_get(view_name, oid_tree)
    #Get the specified SNMP view name
    #Iterate through the existing SNMP views until a match for view and oid is found.
    #If found, display the associated view parameters.

    found = false
    max_oid_len = 128+1
    max_view_len = 30+1

    view_p = OpEN.new_OPEN_SNMP_VIEW_TYPE_tp()

    view_string      = $open.getCharBuffer(view_name.length,view_name)
    view_buff        = OpEN::Open_buffdesc.new
    view_buff.pstart = view_string
    view_buff.size   = view_name.length

    oid_string       = $open.getCharBuffer(oid_tree.length,oid_tree)
    oid_buff         = OpEN::Open_buffdesc.new
    oid_buff.pstart  = oid_string
    oid_buff.size    = oid_tree.length

    next_view_string      = $open.getCharBuffer(max_view_len)
    next_view_buff        = OpEN::Open_buffdesc.new
    next_view_buff.pstart = next_view_string
    next_view_buff.size   = max_view_len

    next_oid_string       = $open.getCharBuffer(max_oid_len)
    next_oid_buff         = OpEN::Open_buffdesc.new
    next_oid_buff.pstart  = next_oid_string
    next_oid_buff.size    = max_oid_len

    oid_string_string      = $open.getCharBuffer(max_oid_len)
    oid_string_buff        = OpEN::Open_buffdesc.new
    oid_string_buff.pstart = oid_string_string
    oid_string_buff.size   = max_oid_len

    oid_val_string       = $open.getCharBuffer(max_oid_len)
    oid_val_buff         = OpEN::Open_buffdesc.new
    oid_val_buff.pstart  = oid_val_string
    oid_val_buff.size    = max_oid_len

    OpEN.openapiSnmpTreeFamilyOidStringGet(@client, view_buff, oid_buff, OpEN::OPEN_SNMP_VIEW_TREE, oid_string_buff)
    OpEN.openapiSnmpTreeFamilyOidValGet(@client, view_buff, oid_buff, OpEN::OPEN_SNMP_VIEW_TREE, oid_val_buff)

    view_string      = $open.getCharBuffer(max_view_len)
    view_buff        = OpEN::Open_buffdesc.new
    view_buff.pstart = view_string
    view_buff.size   = max_view_len

    oid_string       = $open.getCharBuffer(max_oid_len)
    oid_buff         = OpEN::Open_buffdesc.new
    oid_buff.pstart  = oid_string
    oid_buff.size    = max_oid_len

    while OpEN.openapiSnmpViewGetNext(@client, 
                                      view_buff, 
                                      oid_buff, 
                                      view_p,
                                      next_view_buff, 
                                      next_oid_buff) == OpEN::OPEN_E_NONE

      if (view_name == next_view_string.cast() and
          oid_val_string.cast() == next_oid_string.cast())
        found = true
        break
      end

      view_buff.pstart = next_view_buff.pstart
      view_buff.size   = next_view_buff.size
      oid_buff.pstart = next_oid_buff.pstart
      oid_buff.size   = next_oid_buff.size
      next_view_buff.size = max_view_len
      next_oid_buff.size = max_oid_len
    end

    print_sanity_results(OpEN::OPEN_E_NONE, found, 'openapiSnmpViewGetNext', view_name) 

    if (found and $verbose)
      print '  View name  : ', next_view_string.cast(); puts
      print '  OID Tree   : ', next_oid_string.cast(); puts
      print '  OID String : ', oid_string_string.cast(); puts
      print '  View Type  : ', OpEN::OPEN_SNMP_VIEW_TYPE_tp_value(view_p); puts
    end

    OpEN.delete_OPEN_SNMP_VIEW_TYPE_tp(view_p)
  end

  def test_snmp_filter_get(filter_name, oid_tree)
    #Get the specified SNMP filter name
    #Iterate through the existing SNMP views until a match for view and oid is found.
    #If found, display the associated view parameters.

    found = false
    max_oid_len = 128+1
    max_filter_len = 30+1

    filter_p = OpEN.new_OPEN_SNMP_NOTIFY_FILTER_TYPE_tp()

    filter_string      = $open.getCharBuffer(filter_name.length,filter_name)
    filter_buff        = OpEN::Open_buffdesc.new
    filter_buff.pstart = filter_string
    filter_buff.size   = filter_name.length

    oid_string       = $open.getCharBuffer(oid_tree.length,oid_tree)
    oid_buff         = OpEN::Open_buffdesc.new
    oid_buff.pstart  = oid_string
    oid_buff.size    = oid_tree.length

    next_filter_string      = $open.getCharBuffer(max_filter_len)
    next_filter_buff        = OpEN::Open_buffdesc.new
    next_filter_buff.pstart = next_filter_string
    next_filter_buff.size   = max_filter_len

    next_oid_string       = $open.getCharBuffer(max_oid_len)
    next_oid_buff         = OpEN::Open_buffdesc.new
    next_oid_buff.pstart  = next_oid_string
    next_oid_buff.size    = max_oid_len

    oid_string_string      = $open.getCharBuffer(max_oid_len)
    oid_string_buff        = OpEN::Open_buffdesc.new
    oid_string_buff.pstart = oid_string_string
    oid_string_buff.size   = max_oid_len

    oid_val_string       = $open.getCharBuffer(max_oid_len)
    oid_val_buff         = OpEN::Open_buffdesc.new
    oid_val_buff.pstart  = oid_val_string
    oid_val_buff.size    = max_oid_len

    OpEN.openapiSnmpTreeFamilyOidStringGet(@client, filter_buff, oid_buff, OpEN::OPEN_SNMP_FILTER_TREE, oid_string_buff)
    OpEN.openapiSnmpTreeFamilyOidValGet(@client, filter_buff, oid_buff, OpEN::OPEN_SNMP_FILTER_TREE, oid_val_buff)

    filter_string      = $open.getCharBuffer(max_filter_len)
    filter_buff        = OpEN::Open_buffdesc.new
    filter_buff.pstart = filter_string
    filter_buff.size   = max_filter_len

    oid_string       = $open.getCharBuffer(max_oid_len)
    oid_buff         = OpEN::Open_buffdesc.new
    oid_buff.pstart  = oid_string
    oid_buff.size    = max_oid_len

    while OpEN.openapiSnmpFilterGetNext(@client, 
                                        filter_buff, 
                                        oid_buff, 
                                        filter_p,
                                        next_filter_buff, 
                                        next_oid_buff) == OpEN::OPEN_E_NONE

      if (filter_name == next_filter_string.cast() and
          oid_val_string.cast() == next_oid_string.cast()):
        found = true
        break
      end

      filter_buff.pstart = next_filter_buff.pstart
      filter_buff.size   = next_filter_buff.size
      oid_buff.pstart = next_oid_buff.pstart
      oid_buff.size   = next_oid_buff.size
      next_filter_buff.size = max_filter_len
      next_oid_buff.size = max_oid_len
    end

    print_sanity_results(OpEN::OPEN_E_NONE, found, 'openapiSnmpFilterGetNext', filter_name) 

    if (found and $verbose)
      print '  Filter name : ', next_filter_string.cast(); puts
      print '  OID Tree    : ', next_oid_string.cast(); puts
      print '  OID String  : ', oid_string_string.cast(); puts
      print '  View Type   : ', OpEN::OPEN_SNMP_NOTIFY_FILTER_TYPE_tp_value(filter_p); puts
    end

    OpEN.delete_OPEN_SNMP_NOTIFY_FILTER_TYPE_tp(filter_p)
  end

  def test_snmp_host_get(host)
    #Get the specified SNMP host name
    #Iterate through the existing SNMP hosts until a match for
    #host is found. If found, display the associated host parameters.

    found = false
    max_host_len = 158+1
    max_filter_len = 30+1
    max_security_len = 30+1

    port_p = OpEN.new_uint32_tp()
    model_p = OpEN.new_OPEN_SNMP_SECURITY_MODEL_tp()
    level_p = OpEN.new_OPEN_SNMP_SECURITY_LEVEL_tp()
    type_p = OpEN.new_OPEN_SNMP_NOTIFY_TYPE_tp()
    timeout_p = OpEN.new_uint32_tp()
    retries_p = OpEN.new_uint32_tp()

    host_string           = $open.getCharBuffer(max_host_len)
    host_buff             = OpEN::Open_buffdesc.new
    host_buff.pstart      = host_string
    host_buff.size        = max_host_len

    security_string       = $open.getCharBuffer(max_security_len)
    security_buff         = OpEN::Open_buffdesc.new
    security_buff.pstart  = security_string
    security_buff.size    = max_security_len

    filter_string         = $open.getCharBuffer(max_filter_len)
    filter_buff           = OpEN::Open_buffdesc.new
    filter_buff.pstart    = filter_string
    filter_buff.size      = max_filter_len

    next_host_string      = $open.getCharBuffer(max_host_len)
    next_host_buff        = OpEN::Open_buffdesc.new
    next_host_buff.pstart = next_host_string
    next_host_buff.size   = max_host_len

    while OpEN.openapiSnmpHostGetNext(@client, 
                                      host_buff, 
                                      port_p, 
                                      security_buff,
                                      model_p,
                                      level_p,
                                      type_p,
                                      timeout_p,
                                      retries_p,
                                      filter_buff, 
                                      next_host_buff) == OpEN::OPEN_E_NONE

      if (host == next_host_string.cast())
        found = true
        break
      end

      security_string       = $open.getCharBuffer(max_security_len)
      security_buff         = OpEN::Open_buffdesc.new
      security_buff.pstart  = security_string
      security_buff.size    = max_security_len

      filter_string         = $open.getCharBuffer(max_filter_len)
      filter_buff           = OpEN::Open_buffdesc.new
      filter_buff.pstart    = filter_string
      filter_buff.size      = max_filter_len

      host_buff.pstart = next_host_buff.pstart
      host_buff.size   = next_host_buff.size
      next_host_buff.size = max_host_len
    end

    print_sanity_results(OpEN::OPEN_E_NONE, found, 'openapiSnmpHostGetNext', host) 

    if (found and $verbose)
      print '  Address port   : %s %d' % [next_host_string.cast(), OpEN.uint32_tp_value(port_p)]; puts
      print '  Security name  : ', security_string.cast(); puts
      print '  Security model : ', OpEN::OPEN_SNMP_SECURITY_MODEL_tp_value(model_p); puts
      print '  Security level : ', OpEN::OPEN_SNMP_SECURITY_LEVEL_tp_value(level_p); puts
      print '  Notify type    : ', OpEN::OPEN_SNMP_NOTIFY_TYPE_tp_value(type_p); puts
      print '  Timeout        : ', OpEN.uint32_tp_value(timeout_p); puts
      print '  Retries        : ', OpEN.uint32_tp_value(retries_p); puts
      print '  Filter name    : ', filter_string.cast(); puts
    end

    OpEN.delete_uint32_tp(port_p)
    OpEN.delete_OPEN_SNMP_SECURITY_MODEL_tp(model_p)
    OpEN.delete_OPEN_SNMP_SECURITY_LEVEL_tp(level_p)
    OpEN.delete_OPEN_SNMP_NOTIFY_TYPE_tp(type_p)
    OpEN.delete_uint32_tp(timeout_p)
    OpEN.delete_uint32_tp(retries_p)
  end

  def test_snmp_community_get(community_name)
    #Get the specified SNMP community name
    #Iterate through the existing SNMP hosts until a match for
    #host is found. If found, display the associated host parameters.

    max_community_len = 20+1
    max_view_len = 30+1
    access_p = OpEN.new_OPEN_SNMP_COMMUNITY_ACCESS_TYPE_tp()
    addr = OpEN::Open_inet_addr_t.new

    community_string       = $open.getCharBuffer(max_community_len)
    community_buff         = OpEN::Open_buffdesc.new
    community_buff.pstart  = community_string
    community_buff.size    = max_community_len

    view_string            = $open.getCharBuffer(max_view_len)
    view_buff              = OpEN::Open_buffdesc.new
    view_buff.pstart       = view_string
    view_buff.size         = max_view_len

    next_community_string      = $open.getCharBuffer(max_community_len)
    next_community_buff        = OpEN::Open_buffdesc.new
    next_community_buff.pstart = next_community_string
    next_community_buff.size   = max_community_len

    while OpEN.openapiSnmpCommunityGetNext(@client,
                                           community_buff,
                                           access_p,
                                           view_buff,
                                           addr,
                                           next_community_buff) == OpEN::OPEN_E_NONE

      if (community_name == next_community_string.cast())
        found = true
        break
      end

      view_string      = $open.getCharBuffer(max_view_len)
      view_buff        = OpEN::Open_buffdesc.new
      view_buff.pstart = view_string
      view_buff.size   = max_view_len

      community_buff.pstart = next_community_buff.pstart
      community_buff.size   = next_community_buff.size
      next_community_buff.size = max_community_len
    end

    print_sanity_results(OpEN::OPEN_E_NONE, found, 'openapiSnmpCommunityGetNext', community_name) 

    if (found and $verbose)
      print '  Community   :', next_community_string.cast(); puts
      print '  Access Type :', OpEN::OPEN_SNMP_COMMUNITY_ACCESS_TYPE_tp_value(access_p); puts
      print '  View Name   :', view_string.cast(); puts
      print '  IP Address  :', IPAddr.new(addr.addr.ipv4,Socket::AF_INET); puts
    end

    OpEN.delete_OPEN_SNMP_COMMUNITY_ACCESS_TYPE_tp(access_p)
  end

  def test_snmp_trap_flag_get(flag_bits, mode)
    #Retrieve the trap mode for a single trap.

    mode_p = OpEN.new_bool_tp()

    result = OpEN.openapiSnmpTrapFlagGet(@client, flag_bits, mode_p)
    print_bad_result(result, 'openapiSnmpTrapFlagGet')
    print_sanity_results(result, (mode == OpEN.bool_tp_value(mode_p)), 'openapiSnmpTrapFlagGet', flag_bits)

    OpEN.delete_bool_tp(mode_p)
  end

  def test_snmp_local_engine_id_delete()
    #Delete the SNMP engine id on the local device.
    #The engine id is actually reset to the default id which is
    #automatically generated based on the local MAC address.

    result = OpEN.openapiSnmpLocalEngineIdDelete(@client)
    print_bad_result(result, 'openapiSnmpLocalEngineIdDelete')
    print_sanity_results(OpEN::OPEN_E_NONE, True, 'openapiSnmpLocalEngineIdDelete', '')
  end

  def test_snmp_local_engine_id_delete()
    #Delete the SNMP engine id on the local device.
    #The engine id is actually reset to the default id which is
    #automatically generated based on the local MAC address.

    result = OpEN.openapiSnmpLocalEngineIdDelete(@client)
    print_bad_result(result, 'openapiSnmpLocalEngineIdDelete')
    print_sanity_results(OpEN::OPEN_E_NONE, true, 'openapiSnmpLocalEngineIdDelete', '') 
  end

  def test_snmp_view_delete(view_name, oid_tree)
    #Delete an existing SNMP view.

    view_string      = $open.getCharBuffer(view_name.length,view_name)
    view_buff        = OpEN::Open_buffdesc.new
    view_buff.pstart = view_string
    view_buff.size   = view_name.length

    oid_string       = $open.getCharBuffer(oid_tree.length,oid_tree)
    oid_buff         = OpEN::Open_buffdesc.new
    oid_buff.pstart  = oid_string
    oid_buff.size    = oid_tree.length

    result = OpEN.openapiSnmpViewDelete(@client, view_buff, oid_buff)
    print_bad_result(result, 'openapiSnmpViewDelete')
    print_sanity_results(result, true, 'openapiSnmpViewDelete', view_name)
  end

  def test_snmp_filter_delete(filter_name, oid_tree)
    #Delete an existing SNMP filter.

    filter_string      = $open.getCharBuffer(filter_name.length,filter_name)
    filter_buff        = OpEN::Open_buffdesc.new
    filter_buff.pstart = filter_string
    filter_buff.size   = filter_name.length

    oid_string       = $open.getCharBuffer(oid_tree.length,oid_tree)
    oid_buff         = OpEN::Open_buffdesc.new
    oid_buff.pstart  = oid_string
    oid_buff.size    = oid_tree.length

    result = OpEN.openapiSnmpFilterDelete(@client, filter_buff, oid_buff)
    print_bad_result(result, 'openapiSnmpFilterDelete')
    print_sanity_results(result, true, 'openapiSnmpFilterDelete', filter_name)
  end

  def test_snmp_host_delete(host, notify_type)
    #Delete an existing SNMP host.

    host_string      = $open.getCharBuffer(host.length, host)
    host_buff        = OpEN::Open_buffdesc.new
    host_buff.pstart = host_string
    host_buff.size   = host.length

    result = OpEN.openapiSnmpHostDelete(@client, host_buff, notify_type)
    print_bad_result(result, 'openapiSnmpHostDelete')
    print_sanity_results(result, true, 'openapiSnmpHostDelete', host) 
  end

  def test_snmp_user_delete(user_name, engine_id)
    #Delete a SNMP user.

    user_string           = $open.getCharBuffer(user_name.length, user_name)
    user_buff             = OpEN::Open_buffdesc.new
    user_buff.pstart      = user_string
    user_buff.size        = user_name.length

    engine_id_string      = $open.getCharBuffer(engine_id.length, engine_id) 
    engine_id_buff        = OpEN::Open_buffdesc.new
    engine_id_buff.pstart = engine_id_string
    engine_id_buff.size   = engine_id.length

    result = OpEN.openapiSnmpUserDelete(@client, user_buff, engine_id_buff)
    print_bad_result(result, 'openapiSnmpUserDelete')
    print_sanity_results(result, true, 'openapiSnmpUserDelete', user_name)
  end

  def test_snmp_group_delete(group_name, model, level, context_name)
    #Delete a SNMP group.

    group_string           = $open.getCharBuffer(group_name.length, group_name)
    group_buff             = OpEN::Open_buffdesc.new
    group_buff.pstart      = group_string
    group_buff.size        = group_name.length

    context_string           = $open.getCharBuffer(context_name.length, context_name)
    context_buff             = OpEN::Open_buffdesc.new
    context_buff.pstart      = context_string
    context_buff.size        = context_name.length

    result = OpEN.openapiSnmpGroupDelete(@client, group_buff, model, level, context_buff)
    print_bad_result(result, 'openapiSnmpGroupDelete')
    print_sanity_results(result, true, 'openapiSnmpGroupDelete', group_name)
  end

  def test_snmp_community_delete(community_name)
    #Delete an SNMP community.

    community_string           = $open.getCharBuffer(community_name.length, community_name)
    community_buff             = OpEN::Open_buffdesc.new
    community_buff.pstart      = community_string
    community_buff.size        = community_name.length

    result = OpEN.openapiSnmpCommunityDelete(@client, community_buff)
    print_bad_result(result, 'openapiSnmpCommunityDelete')
    print_sanity_results(result, true, 'openapiSnmpCommunityDelete', community_name) 
  end

  def test_snmp_trap_source_interface(interface_type)
    #Set, then get the source interface which SNMP trap originates.
    #For demonstration purposes, this function retrieves the first
    #interface based on interface type and attempts to set it as the
    #source interface. If successful, the interface attributes are
    #retrieved and verified.

    msg = 'interface type : ' + get_type(interface_type)

    # retrieve first available interface for demonstration
    val_p = OpEN.new_uint32_tp()
    result = OpEN.openapiIfFirstGet(@client, interface_type, val_p)
    if result == OpEN::OPEN_E_FAIL
      result = OpEN::OPEN_E_NOT_FOUND
    end
    print_bad_result(result, '%s - %s' % ['openapiIfFirstGet', msg])

    if result == OpEN::OPEN_E_NONE
      intf = OpEN.uint32_tp_value(val_p)
      result = OpEN.openapiSnmpTrapSrcIntfSet(@client, intf)
      print_bad_result(result, '%s - %s' % ['openapiSnmpTrapSrcIntfSet', msg])
    end

    if result == OpEN::OPEN_E_NONE
      intf_p = OpEN.new_uint32_tp()
      type_p = OpEN.new_OPEN_INTF_TYPE_tp()
      tmp_addr = OpEN::Open_inet_addr_t.new
      result = OpEN.openapiSnmpTrapSrcIntfGet(@client, intf_p, type_p, tmp_addr)
      print_bad_result(result, '%s - %s' % ['openapiSnmpTrapSrcIntfGet', msg])
    end

    if result == OpEN::OPEN_E_NONE
      if (tmp_addr.family == OpEN::OPEN_AF_INET or tmp_addr.family == OpEN::OPEN_AF_UNIX)
        ip_addr = IPAddr.new(tmp_addr.addr.ipv4,Socket::AF_INET)
      else
        result = OpEN::OPEN_E_PARAM
      end
      print_bad_result(result, '%s - %s' % ['inet_ntoa', msg])
    end

    if result == OpEN::OPEN_E_NONE
      state_p = OpEN.new_OPEN_LINK_STATE_tp()
      result = OpEN.openapiIfLinkStateGet(@client, intf, state_p)
      print_bad_result(result, '%s - %s' % ['openapiIfLinkStateGet', msg])
    end

    if result == OpEN::OPEN_E_NONE
      print_sanity_results(result, true, 'openapiSnmpTrapSrcIntfSet', msg)
      if $verbose
        print '  Interface : ', intf; puts
        print '  Type      : ', get_type(interface_type); puts
        print '  Address   : ', ip_addr; puts
        print '  State     : ', get_link(OpEN::OPEN_LINK_STATE_tp_value(state_p)); puts
      end
    end
    OpEN.delete_uint32_tp(val_p)
  end

end

def main()
  # Demonstrate OpEN usage for SNMP APIs

  OptionParser.new do |opts|
    opts.on('-v', '--verbose') { |b| $verbose = b; }
    opts.on('-h', '--help') { puts opts; exit }
    opts.parse!
  end

  ret = $open.connect("snmp_example")
  if ret == OpEN::OPEN_E_NONE
    $open.getNetworkOSVersion()
    $open.getAPIVersion()
    client = $open.client
    example = SnmpExample.new(client)
    puts "Begin Sanity tests..."
    example.test_snmp_sys_name_set("system-name")
    example.test_snmp_sys_location_set("system-location")
    example.test_snmp_sys_contact_set("system-contact")
    example.test_snmp_local_engine_id_set("default")
    puts "\nCreate SNMP users..."
    example.test_snmp_user_create(SNMP_USER_LIST)
    puts "\nCreate SNMP groups..."
    example.test_snmp_group_create(SNMP_GROUP_LIST)
    puts "\nCreate SNMP views..."
    example.test_snmp_view_create(SNMP_VIEW_LIST)
    puts "\nCreate SNMP filters..."
    example.test_snmp_filter_create(SNMP_FILTER_LIST)
    puts "\nCreate SNMP hosts..."
    example.test_snmp_host_create(SNMP_HOST_LIST)
    puts "\nCreate SNMP communities..."
    example.test_snmp_community_create(SNMP_COMMUNITY_LIST)
    puts "\nEnable misc SNMP trap flags..."
    example.testSnmpTrapFlagSet([OpEN::OPEN_SNMP_MULTI_USERS_TRAP_FLAG, OpEN::OPEN_SNMP_USER_AUTH_TRAP_FLAG], true)
    puts "\nRetrieve some SNMP records..."
    example.test_snmp_sys_name_get()
    example.test_snmp_sys_location_get()
    example.test_snmp_sys_contact_get()
    example.test_snmp_local_engine_id_get()
    example.test_snmp_user_get('admin1')
    example.test_snmp_group_get('network-admin', OpEN::OPEN_SNMP_SECURITY_MODEL_USM, OpEN::OPEN_SNMP_SECURITY_LEVEL_AUTHPRIV, 'context-1')
    example.test_snmp_view_get('view-1', 'broadcom')
    example.test_snmp_view_get('view-2', 'ifEntry.*.1')
    example.test_snmp_filter_get('filter-1', 'broadcom')
    example.test_snmp_filter_get('filter-2', 'ifEntry.*.1')
    example.test_snmp_host_get('2001::')
    example.test_snmp_community_get('public-test')
    example.test_snmp_community_get('private-test')
    example.test_snmp_community_get('delete-private-map')
    example.test_snmp_trap_flag_get(OpEN::OPEN_SNMP_MULTI_USERS_TRAP_FLAG, true)
    example.test_snmp_trap_flag_get(OpEN::OPEN_SNMP_USER_AUTH_TRAP_FLAG, true)
    puts "\nDelete some SNMP records..."
    example.test_snmp_local_engine_id_delete()
    example.test_snmp_view_delete("view-2", "ifEntry.*.1")
    example.test_snmp_filter_delete("filter-2", "ifEntry.*.1")
    example.test_snmp_host_delete("10.10.10.2", OpEN::OPEN_SNMP_NOTIFY_TYPE_TRAP)
    example.test_snmp_user_delete("delete-user", "dddddddddddddddddddddddddddddddd")
    example.test_snmp_group_delete("delete-group", OpEN::OPEN_SNMP_SECURITY_MODEL_SNMPV2C, OpEN::OPEN_SNMP_SECURITY_LEVEL_NOAUTHNOPRIV, "context-3")
    example.test_snmp_community_delete("delete-community")
    example.test_snmp_community_delete("delete-private-map")
    puts "\nAttempt to set the trap source-interface..."
    # Routing must be enabled on first interface
    example.test_snmp_trap_source_interface(OpEN::OPEN_INTF_TYPE_PHY)

    $open.terminate()
  else
    print "Unable to connect"
  end
end

if __FILE__ == $0 then main() end


