/*********************************************************************
*
*  Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  time_example.c
*
* @purpose   System Clock Date & Time Examples
*
* @component OpEN
*
* @comments  
*
* @create    10/06/2016
*
* @end
*
**********************************************************************/

#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_time.h"

/********************************************************************* 
* 
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for System Clock Date & Time

* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
* 
*********************************************************************/
int main (int argc, char **argv)
{
  openapiClientHandle_t  clientHandle;
  open_error_t  result = OPEN_E_NONE;
  open_buffdesc  bufd;
  char  dateTime[OPEN_DATE_TIME_BUF_LEN], timezone[OPEN_TIMEZONE_BUF_LEN];
  OPEN_TIME_CLOCK_SRC_t clockSrc = OPEN_CLOCK_SRC_NONE;

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister ("time_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit (2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck (&clientHandle) != OPEN_E_NONE)
  {
    sleep (1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Date/Time API example application");

  printf("\n");

  printf("Configure System Clock...\n");
  snprintf(dateTime, sizeof(dateTime), "2016-01-02 16:20:05");
  bufd.pstart = dateTime;
  bufd.size = strlen(bufd.pstart) + 1;
  result = openapiClockSet(&clientHandle, &bufd);
  if (result == OPEN_E_NONE)
  {
    printf("System Clock set to %s.\n", (char *)bufd.pstart);
  }
  else
  {
    printf("System Clock set error (result = %d).\n", result);
  }

  printf("Display System Clock...\n");
  memset(dateTime, 0, sizeof(dateTime));
  bufd.size = sizeof(dateTime);
  result = openapiClockGet(&clientHandle, &bufd, &clockSrc);
  if (result == OPEN_E_NONE)
  {
    printf("System Clock, Source ... %s, %d.\n", (char *)bufd.pstart, clockSrc);
  }
  else
  {
    printf("System Clock retrieval error (result = %d).\n", result);
  }

  printf("Configure System Timezone...\n");
  snprintf(timezone, sizeof(timezone), "UTC-1:2 ab");
  bufd.pstart = timezone;
  bufd.size = strlen(bufd.pstart) + 1;
  result = openapiTimeZoneSet(&clientHandle, &bufd);
  if (result == OPEN_E_NONE)
  {
    printf("System Timezone set to ... %s.\n", (char *)bufd.pstart);
  }
  else
  {
    printf("System Timezone set error (result = %d).\n", result);
  }

  printf("Display System Timezone...\n");
  memset(timezone, 0, sizeof(timezone));
  bufd.size = sizeof(timezone);
  result = openapiTimeZoneGet(&clientHandle, &bufd);
  if (result == OPEN_E_NONE)
  {
    printf("System Timezone ... %s.\n", (char *)bufd.pstart);
  }
  else
  {
    printf("System Timezone retrieval error (result = %d).\n", result);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Date/Time API example application");

  return 0;
}

