/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  user_sessions_example.c
*
* @purpose   User Accounts OpEN APIs Example.
*
* @component OpEN
*
* @comments
*
* @create    13/08/2015
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_user_manager.h"

#define  MAX_STR_LEN     100
#define  LOCALHOST_IP    "127.0.0.1"
/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the User Sessions Example Application Menu.
*
* @param  none
*
* @returns  none 
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: user_sessions <test#> <arg1> <arg2> ... \n");
  printf("Test 1: Create a session: %s 1 <username> <password> \n", name);
  printf("Test 2: Validate a session: %s 2 <sid> \n", name);
  printf("Test 3: End a session: %s 3 <sid>\n", name);
  printf("Test 4: Get the login session username: %s 4 <sid>\n", name);
  printf("Test 5: Get the login session remote IP address: %s 5 <sid>\n", name);
  printf("Test 6: Get the login session idle time: %s 6 <sid>\n", name);
  printf("Test 7: Get the login session session time: %s 7 <sid>\n", name);
  printf("Test 8: Get the login session type: %s 8 <sid>\n", name);
  printf("Test 9: Run error checking tests: %s 9\n", name);

  return;
}

/******************************************************************************
* @purpose  Create user session.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    userName        @b{(input)}   User name to configure.
* @param    password        @b{(input)}   Password for the user.
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************************/
void createSession(openapiClientHandle_t *clientHandle, char *userName, char *passWord)
{
  open_error_t  result;
  open_buffdesc unameDesc;
  open_buffdesc pwdDesc;
  open_buffdesc loginType;
  open_buffdesc inetaddr;
  open_buffdesc sessionId;
  char str1[MAX_STR_LEN];
  char str2[MAX_STR_LEN];
  char str3[MAX_STR_LEN];
  char str4[MAX_STR_LEN]= LOCALHOST_IP;
  char str5[MAX_STR_LEN];
  uint32_t       accessLevel;

  snprintf(str1, MAX_STR_LEN, "%s", userName);
  unameDesc.pstart = str1;
  unameDesc.size =   strnlen(str1, MAX_STR_LEN) + 1;

  snprintf(str2, MAX_STR_LEN, "%s", passWord);
  pwdDesc.pstart = str2;
  pwdDesc.size =   strnlen(str2, MAX_STR_LEN) + 1;

  snprintf(str3, MAX_STR_LEN,"%s", "HTTP");

  loginType.pstart = str3;
  loginType.size   = strnlen(str3, MAX_STR_LEN) + 1;

  inetaddr.pstart = str4;
  inetaddr.size   = strnlen(str4, MAX_STR_LEN) + 1;

  sessionId.pstart = str5;
  sessionId.size = sizeof(str5);

  if ((result = openapiWebSessionUserLogin(clientHandle, &unameDesc, &pwdDesc,
                                 	  &loginType,&inetaddr, &accessLevel, &sessionId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to create session. (result = %d)\n", result);
  }
  else
  {
    printf("User session created successfully\n");
    printf("session Id:%s access level = %d\n", str5, accessLevel);
  }
  return;
}

/*********************************************************************
* @purpose  Validate a session. 
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    Sid             @b{(input)}   session Id.
* @param    level           @b{(output)}  User privilege level
*
* @returns  none
* 
* @notes     
* 
* @end
*********************************************************************/
void validateSession(openapiClientHandle_t *clientHandle, char *sid)
{
  open_error_t result;
  open_buffdesc session;
  open_buffdesc loginType;
  open_buffdesc inetaddr;
  char str1[MAX_STR_LEN];
  char str2[MAX_STR_LEN];
  char str3[MAX_STR_LEN] = LOCALHOST_IP;
  uint32_t   accessLevel;

  snprintf(str1, MAX_STR_LEN, "%s", sid);

  session.pstart = str1;
  session.size =   strnlen(str1, MAX_STR_LEN) + 1;

  snprintf(str2, MAX_STR_LEN, "%s", "HTTP");

  loginType.pstart = str2;
  loginType.size   = strnlen(str2, MAX_STR_LEN) + 1;

  inetaddr.pstart = str3;
  inetaddr.size   = strnlen(str3, MAX_STR_LEN) + 1;

  if ((result = openapiWebSessionValidate(clientHandle, &session, &loginType,
                                          &inetaddr, &accessLevel)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to validate session. (result = %d)\n", result);
  }
  else
  {
    printf("Session is valid, access level = %d\n", accessLevel);
  }
  return;
}

/*********************************************************************
* @purpose  End session. 
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    sid             @b{(input)}   Session id
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void endSession(openapiClientHandle_t *clientHandle, char *sid)
{
  open_error_t  result;
  open_buffdesc session;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, sid, (sizeof(str) - 1));
  session.pstart = str;
  session.size = strlen(str)+1;

  if ((result = openapiWebSessionLogout(clientHandle, &session)) != OPEN_E_NONE)
  {
    printf("Session end failed, (result = %d)\n", result);
  }
  else
  {
    printf("Session ended.\n");
  }
  return;
}

/***************************************************************************
* @purpose  Get the login session user name.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    index           @b{(input)}   session index
*
* @returns  none
* 
* @notes  
* 
* @end
****************************************************************************/
void getLoginSessionName(openapiClientHandle_t *clientHandle, uint32_t index)
{
  open_error_t  result;
  open_buffdesc name;
  char nameStr[OPEN_LOGIN_SIZE];

  name.pstart = nameStr;
  name.size = sizeof(nameStr);

  if ((result = openapiLoginSessionUserGet(clientHandle, index, &name)) != OPEN_E_NONE) 
  {
    printf("Bad return code trying to get login user name. (result = %d)\n", result);
  }
  else
  {
    printf("login user name: %s\n", nameStr);
  }
  return;
}

/*********************************************************************
* @purpose  Convert an IP address in a open_inet_addr_t structure to
*           a string for display
*
* @param    addr  @b{(input)}  address to be converted
* @param    buffer @b{(output)} storage to contain
*
* @returns  pointer to buffer if conversion successful
*           NULL if error
*
* @notes
*
* @end
*********************************************************************/
static const char *ipAddressFormat(open_inet_addr_t *addr, char *buffer)
{
  uint32_t  ip4_addr;
  uint8_t   addr8[INET6_ADDRSTRLEN];
  int af;

  switch (addr->family)
  {
  case OPEN_AF_INET:
    af = AF_INET;
    ip4_addr = addr->addr.ipv4;
    memcpy(addr8, &ip4_addr, sizeof(ip4_addr));
    break;
  case OPEN_AF_INET6:
    af = AF_INET6;
    memcpy(addr8, addr->addr.ipv6.u.addr8, sizeof(addr8));
    break;
  default:
    return(NULL);
  }

  return(inet_ntop(af, (void *)addr8, buffer, sizeof(addr8)));
}

/***************************************************************************
* @purpose  Get the login session remote IP address.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    index           @b{(input)}   session index
*
* @returns  none
* 
* @notes  
* 
* @end
****************************************************************************/
void getLoginSessionRemoteIPAddr(openapiClientHandle_t *clientHandle, uint32_t index)
{
  open_error_t  result;
  open_inet_addr_t addr;
  char ipAddrStr[80];

  if ((result = openapiLoginSessionRemoteIpAddrGet(clientHandle, index, &addr)) != OPEN_E_NONE) 
  {
    printf("Bad return code trying to get login session remote IP address. (result = %d)\n", result);
  }
  else
  {
    if (ipAddressFormat(&addr, ipAddrStr) != NULL) 
    {
      printf("remote IP address: %s\n", ipAddrStr);
    }
    else
    {
      printf("unable to convert IP address to string\n");
    }
  }
  return;
}

/***************************************************************************
* @purpose  Get the login session idle time. 
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    index           @b{(input)}   session index
*
* @returns  none
* 
* @notes  
* 
* @end
****************************************************************************/
void getLoginSessionIdleTime(openapiClientHandle_t *clientHandle, uint32_t index)
{
  open_error_t  result;
  uint32_t idleTime;

  if ((result = openapiLoginSessionIdleTimeGet(clientHandle, index, &idleTime)) != OPEN_E_NONE) 
  {
    printf("Bad return code trying to get login session idle time. (result = %d)\n", result);
  }
  else
  {
    printf("login session idle time seconds: %d\n", idleTime);
  }
  return;
}

/***************************************************************************
* @purpose  Get the login session time.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    index           @b{(input)}   session index
*
* @returns  none
* 
* @notes  
* 
* @end
****************************************************************************/
void getLoginSessionTime(openapiClientHandle_t *clientHandle, uint32_t index)
{
  open_error_t  result;
  uint32_t sessionTime;

  if ((result = openapiLoginSessionTimeGet(clientHandle, index, &sessionTime)) != OPEN_E_NONE) 
  {
    printf("Bad return code trying to get login session time. (result = %d)\n", result);
  }
  else
  {
    printf("login session time: %d seconds\n", sessionTime);
  }
  return;
}

/***************************************************************************
* @purpose  Convert an OPEN_LOGIN_TYPE_t value to a string repr.
*
* @param    type           @b{(input)}   the type to convert to string
*
* @returns  The OPEN_LOGIN_TYPE_t enum as a string or "unknown"
*
* @notes
*
* @end
****************************************************************************/

static char * typeToName(OPEN_LOGIN_TYPE_t type)
{
  char *ret = "unknown";

  if (type == OPEN_LOGIN_TYPE_UNKNWN)
  {
    ret = "OPEN_LOGIN_TYPE_UNKNWN";
  }
  else if (type == OPEN_LOGIN_TYPE_SERIAL)
  {
    ret = "OPEN_LOGIN_TYPE_SERIAL";
  }
  else if (type == OPEN_LOGIN_TYPE_TELNET)
  {
    ret = "OPEN_LOGIN_TYPE_TELNET";
  }
  else if (type == OPEN_LOGIN_TYPE_SSH )
  {
    ret = "OPEN_LOGIN_TYPE_SSH ";
  }
  else if (type == OPEN_LOGIN_TYPE_HTTP )
  {
    ret = "OPEN_LOGIN_TYPE_HTTP ";
  }
  else if (type == OPEN_LOGIN_TYPE_HTTPS)
  {
    ret = "OPEN_LOGIN_TYPE_HTTPS";
  }
  else if (type == OPEN_LOGIN_TYPE_SNMP)
  {
    ret = "OPEN_LOGIN_TYPE_SNMP";
  }
  else if (type == OPEN_LOGIN_TYPE_PING)
  {
    ret = "OPEN_LOGIN_TYPE_PING";
  }
  else if (type == OPEN_LOGIN_TYPE_UDS)
  {
    ret = "OPEN_LOGIN_TYPE_UDS";
  }
  return ret;
}

/***************************************************************************
* @purpose  Get the login session type.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
* @param    index           @b{(input)}   session index
*
* @returns  none
* 
* @notes  
* 
* @end
****************************************************************************/
void getLoginSessionType(openapiClientHandle_t *clientHandle, uint32_t index)
{
  open_error_t  result;
  OPEN_LOGIN_TYPE_t type;

  if ((result = openapiLoginSessionTypeGet(clientHandle, index, &type)) != OPEN_E_NONE) 
  {
    printf("Bad return code trying to get login session type. (result = %d)\n", result);
  }
  else
  {
    printf("login session type: %s\n", typeToName(type));
  }
  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes  
* 
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *clientHandle)
{
  char nameBuf[OPEN_LOGIN_SIZE];
  open_buffdesc buf;
  uint32_t timeVal;
  open_inet_addr_t addr;
  OPEN_LOGIN_TYPE_t typeVal;
  open_error_t result;

  printf("Testing Login Session OpEN APIs sanity:\n\n");

  /* openapiLoginSessionUserGet() */
  buf.size = sizeof(nameBuf);
  buf.pstart = nameBuf;

  printf("Testing openapiLoginSessionUserGet():\n");

  result = openapiLoginSessionUserGet(clientHandle, 999999, &buf);
  printf("Unlikely session ID:(result = %d)\n", result);
  result = openapiLoginSessionUserGet(NULL, 1, &buf);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiLoginSessionUserGet(clientHandle, 1, NULL);
  printf("NULL name:(result = %d)\n", result);
  buf.size = sizeof(nameBuf) - 1;
  result = openapiLoginSessionUserGet(clientHandle, 1, &buf);
  printf("Buffer size too smnall:(result = %d)\n", result);
  buf.size = sizeof(nameBuf);
  buf.pstart = (void *) NULL;
  result = openapiLoginSessionUserGet(clientHandle, 1, &buf);
  printf("NULL buffer pstart:(result = %d)\n", result);

  printf("Testing openapiLoginSessionRemoteIpAddrGet():\n");

  result = openapiLoginSessionRemoteIpAddrGet(clientHandle, 999999, &addr);
  printf("Unlikely session ID:(result = %d)\n", result);
  result = openapiLoginSessionRemoteIpAddrGet(NULL, 1, &addr);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiLoginSessionRemoteIpAddrGet(clientHandle, 1, NULL);
  printf("NULL addr:(result = %d)\n", result);

  printf("Testing openapiLoginSessionIdleTimeGet():\n");

  result = openapiLoginSessionIdleTimeGet(clientHandle, 999999, &timeVal);
  printf("Unlikely session ID:(result = %d)\n", result);
  result = openapiLoginSessionIdleTimeGet(NULL, 1, &timeVal);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiLoginSessionIdleTimeGet(clientHandle, 1, NULL);
  printf("NULL time:(result = %d)\n", result);

  printf("Testing openapiLoginSessionTimeGet():\n");

  result = openapiLoginSessionTimeGet(clientHandle, 999999, &timeVal);
  printf("Unlikely session ID:(result = %d)\n", result);
  result = openapiLoginSessionTimeGet(NULL, 1, &timeVal);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiLoginSessionTimeGet(clientHandle, 1, NULL);
  printf("NULL time:(result = %d)\n", result);

  printf("Testing openapiLoginSessionTypeGet():\n");

  result = openapiLoginSessionTypeGet(clientHandle, 999999, &typeVal);
  printf("Unlikely session ID:(result = %d)\n", result);
  result = openapiLoginSessionTypeGet(NULL, 1, &typeVal);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiLoginSessionTypeGet(clientHandle, 1, NULL);
  printf("NULL type:(result = %d)\n", result);
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user sessions.
*
* @returns   0: Success
* @returns  -1: Failure if the number of arguments are incorrect
* @returns   2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("user_sessions_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting User session API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);

  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 4)
      { 
        createSession(&clientHandle, argv[2], argv[3]); 
        show_help = 0;
      }
      break;

    case 2:
      if (argc == 3)
      {
        validateSession(&clientHandle, argv[2]);
        show_help = 0;
      }
      break;

    case 3:
      if (argc == 3)
      {
        endSession(&clientHandle, argv[2]);
        show_help = 0;
      }
      break;

    case 4:
      if (argc == 3)
      {
        getLoginSessionName(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;

    case 5:
      if (argc == 3)
      {
        getLoginSessionRemoteIPAddr(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;

    case 6:
      if (argc == 3)
      {
        getLoginSessionIdleTime(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;

    case 7:
      if (argc == 3)
      {
        getLoginSessionTime(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;

    case 8:
      if (argc == 3)
      {
        getLoginSessionType(&clientHandle, atoi(argv[2]));
        show_help = 0;
      }
      break;

    case 9:
      if (argc == 2)
      {
        runSanity(&clientHandle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  } 

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping User sessions API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
