/*********************************************************************
*
* (C) Copyright Broadcom 2018
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  pvlan_example.c
*
* @purpose   Private VLAN OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    09/13/2018
*
* @end
*
**********************************************************************/
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_vlan.h"
#include "openapi_if.h"

/* 
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

#define VID_LIST_BUFSIZE_MAX  (4096 * 5)          /* sized to hold 4K VLAN IDs (up to 4 digits) with comma separator */

static char *exampleName = "pvlan_example";

/*******************************************************************
* @brief    This function prints the Private VLAN Application Menu.
*
* @param    none
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(void)
{
  printf("Usage: %s <test#> <interface> <arg1> <arg2> ... \n\n", exampleName);
  printf("Test 1: Get PVLAN Interface Mode: %s 1 <interface> \n", exampleName);
  printf("Test 2: Get PVLAN Interface Host Association: %s 2 <interface> \n", exampleName);
  printf("Test 3: Get PVLAN Interface Promiscuous Association: %s 3 <interface> \n", exampleName);
  printf("Test 4: Get PVLAN Interface Operational VLANs: %s 4 <interface> \n", exampleName);
  printf("Test 5: List all PVLAN Interface Promiscuous Trunk Associations: %s 5 <interface> \n", exampleName);
  printf("Test 6: List all PVLAN Interface Isolated Trunk Associations: %s 6 <interface> \n", exampleName);
  printf("Test 7: Get PVLAN Interface Trunk Native VLAN: %s 7 <interface> \n", exampleName);
  printf("Test 8: Get PVLAN Interface Trunk Normal VLANs: %s 8 <interface> \n", exampleName);
  printf("Test 9: Test PVLAN OpEN APIs sanity: %s 9 <interface> \n", exampleName);
  printf("\nNote: For best results, please configure private VLANs in the switch prior to running this example.\n");
  return;
}

/*******************************************************************
* @brief    Helper function to check the actual vs. expected number of command args.
*
* @param    argc       @b{(input)}  Actual number of invocation args.
* @param    expected   @b{(input)}  Expected number of invocation args.
*
* @returns  OPEN_TRUE  Arg count verified
* @returns  OPEN_FAIL  Arg count mismatch
* 
* @comments 
*
* @end
*********************************************************************/
static OPEN_BOOL_t verifyArgCount(int argc, int expected)
{
  OPEN_BOOL_t result = OPEN_TRUE;

  if (argc != expected)
  {
    printAppMenu();
    result = OPEN_FALSE;
  }

  return result;
}

/*******************************************************************
* @brief    Helper function to convert Switchport mode to a display string.
*
* @param    openMode   @b{(input)}  OpEN API Switchport mode value
*
* @returns  char *     Pointer to display string (address assigned by compiler).
*
* @comments This function is NON-REENTRANT.
* 
* @end
*********************************************************************/
static char *printableSwitchportMode(OPEN_DOT1Q_SWPORT_MODE_t openMode)
{
  /* non-reentrant due to reuse of same static buffer for all invocations */
  static char buf[128];
  char *pstr;

  buf[0] = '\0';    /* init to empty string */

  switch (openMode)
  {
    case OPEN_DOT1Q_SWPORT_MODE_NONE:
      pstr = "None";
      break;
    case OPEN_DOT1Q_SWPORT_MODE_GENERAL:
      pstr = "General";
      break;
    case OPEN_DOT1Q_SWPORT_MODE_ACCESS:
      pstr = "Access";
      break;
    case OPEN_DOT1Q_SWPORT_MODE_TRUNK:
      pstr = "Trunk";
      break;
    case OPEN_DOT1Q_SWPORT_MODE_PRIVATE_HOST:
      pstr = "Private Host";
      break;
    case OPEN_DOT1Q_SWPORT_MODE_PRIVATE_PROMISC:
      pstr = "Private Promiscuous";
      break;
    case OPEN_DOT1Q_SWPORT_MODE_PRIVATE_PROMISC_TRUNK:
      pstr = "Private Promiscuous Trunk";
      break;
    case OPEN_DOT1Q_SWPORT_MODE_PRIVATE_ISOLATED_TRUNK:
      pstr = "Private Isolated Trunk";
      break;
    default:
      pstr = "Unknown";
      break;
  }
  snprintf(buf, sizeof(buf), "%s", pstr);
  return buf;
}

/*******************************************************************
* @brief    Helper function to create a displayable string for a list of VLAN IDs.
*
* @param    vidList    @b{(input)}  VLAN ID list structure
*
* @returns  char *     Pointer to display string.
* 
* @comments This function is NON-REENTRANT.
*
* @end
*********************************************************************/
static char *printableVidList(OPEN_VLAN_LIST_t *vidList)
{
  /* non-reentrant due to reuse of same static buffer for all invocations */
  static char buf[VID_LIST_BUFSIZE_MAX];
  uint32_t i, n;

  buf[0] = '\0';    /* init to empty string */

  if (vidList->numEntries == 0)
  {
    snprintf(buf, sizeof(buf), "None");
  }
  else
  {
    n = snprintf(buf, sizeof(buf), "%d", vidList->ids[0]);
    for (i = 1; i < vidList->numEntries; i++)
    {
      n += snprintf(buf + n, sizeof(buf) - n, ",%d", vidList->ids[i]);
    }
  }
  return buf;
}

/*********************************************************************
* @purpose  Get PVLAN mode on the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void pvlanIntfModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_DOT1Q_SWPORT_MODE_t mode;
  
  if ((result = openapiPvlanIntfModeGet(clientHandle, ifNum, &mode)) != OPEN_E_NONE)
  {
    if (result == OPEN_E_NOT_FOUND)
    {
      printf("Not a PVLAN interface. (result = %d)\n", result);
    }
    else
    {
      printf("Bad return code trying to get PVLAN Interface Mode. (result = %d)\n", result);
    }
  }
  else
  {
    printf("PVLAN Interface Mode for interface %u: %u (%s)\n", ifNum, mode, printableSwitchportMode(mode));
  }
}

/*********************************************************************
* @purpose  Get PVLAN host association on the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void pvlanIntfHostAssocGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  uint32_t pri, sec;
  
  if ((result = openapiPvlanIntfHostAssocGet(clientHandle, ifNum, &pri, &sec)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get PVLAN Interface Host Association. (result = %d)\n", result);
  }
  else
  {
    printf("PVLAN Interface Host Association for interface %u: primary=%u secondary=%u\n", ifNum, pri, sec);
  }
}

/*********************************************************************
* @purpose  Get PVLAN promiscuous association on the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void pvlanIntfPromiscAssocGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  uint32_t pri;
  OPEN_VLAN_LIST_t vidList;
  
  if ((result = openapiPvlanIntfPromiscAssocGet(clientHandle, ifNum, &pri, &vidList)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get PVLAN Interface Promiscuous Association. (result = %d)\n", result);
  }
  else
  {
    printf("PVLAN Interface Promiscuous Association for interface %u: primary=%u secondaries=%s\n", ifNum, pri, printableVidList(&vidList));
  }
}

/*********************************************************************
* @purpose  Get PVLAN operational VLANs association on the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void pvlanIntfOperAssocGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_VLAN_LIST_t vidList;
  
  if ((result = openapiPvlanIntfOperAssocGet(clientHandle, ifNum, &vidList)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get PVLAN Interface Operational VLANs. (result = %d)\n", result);
  }
  else
  {
    printf("PVLAN Interface Operational VLANs for interface %u: %s\n", ifNum, printableVidList(&vidList));
  }
}

/*********************************************************************
* @purpose  List all PVLAN promiscuous trunk associations on the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void pvlanIntfPromiscTrunkAssocList(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  uint32_t pri;
  OPEN_VLAN_LIST_t vidList;
  OPEN_BOOL_t needHdr = OPEN_TRUE;
  
  memset(&vidList, 0, sizeof(vidList));

  pri = 0;
  while ((result = openapiPvlanIntfPromiscTrunkAssocNextGet(clientHandle, ifNum, pri, &pri, &vidList)) == OPEN_E_NONE)
  {
    if (needHdr == OPEN_TRUE)
    {
      /* print heading first time through */
      printf("PVLAN Interface Promiscuous Trunk Association for interface %u:\n", ifNum);
      needHdr = OPEN_FALSE;
    }
    printf("\tprimary=%u secondaries=%s\n", pri, printableVidList(&vidList));
  }

  if (result != OPEN_E_NOT_FOUND)
  {
    /* abnormal termination */
    printf("Bad return code trying to get PVLAN Interface Promiscuous Trunk Association. (result = %d)\n", result);
  }
  else if (needHdr == OPEN_TRUE)
  {
    /* no associations found in preceding loop */
    printf("PVLAN Interface Promiscuous Trunk Association for interface %u: None\n", ifNum);
  }
}

/*********************************************************************
* @purpose  List all PVLAN isolated trunk associations on the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void pvlanIntfIsolatedTrunkAssocList(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  uint32_t pri, sec;
  OPEN_BOOL_t needHdr = OPEN_TRUE;
  
  pri = 0;
  while ((result = openapiPvlanIntfIsolatedTrunkAssocNextGet(clientHandle, ifNum, pri, &pri, &sec)) == OPEN_E_NONE)
  {
    if (needHdr == OPEN_TRUE)
    {
      /* print heading first time through */
      printf("PVLAN Interface Isolated Trunk Association for interface %u:\n", ifNum);
      needHdr = OPEN_FALSE;
    }
    printf("\tprimary=%u secondary=%u\n", pri, sec);
  }

  if (result != OPEN_E_NOT_FOUND)
  {
    /* abnormal termination */
    printf("Bad return code trying to get PVLAN Interface Isolated Trunk Association. (result = %d)\n", result);
  }
  else if (needHdr == OPEN_TRUE)
  {
    /* no associations found in preceding loop */
    printf("PVLAN Interface Isolated Trunk Associations for interface %u: None\n", ifNum);
  }
}

/*********************************************************************
* @purpose  Get PVLAN trunk native VLAN the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void pvlanIntfTrunkNativeVlanGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  uint32_t vid;
  
  if ((result = openapiPvlanIntfTrunkNativeVlanGet(clientHandle, ifNum, &vid)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get PVLAN Interface Trunk Native VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("PVLAN Interface Trunk Native VLAN for interface %u: %u\n", ifNum, vid);
  }
}

/*********************************************************************
* @purpose  Get PVLAN trunk normal VLANs allowed on the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void pvlanIntfTrunkNormalVlanList(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_VLAN_LIST_t vidList;
  
  if ((result = openapiPvlanIntfTrunkAllowedVlanListGet(clientHandle, ifNum, &vidList)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get PVLAN Interface Trunk Normal VLANs allowed. (result = %d)\n", result);
  }
  else
  {
    printf("PVLAN Interface Trunk Normal VLANs allowed for interface %u: %s\n", ifNum, printableVidList(&vidList));
  }
}

/*********************************************************************
* @purpose  This function sanity checks all Private VLAN OpEN APIs. All PVLAN OpEN APIs are 
*           called with possible NULL and invalid parameters to check the API robustness.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void pvlanOpENAPIsTestSanity(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  #define BAD_IFNUM   9999
  #define BAD_VLANID  4097
  open_error_t result;
  OPEN_DOT1Q_SWPORT_MODE_t swportMode;
  uint32_t pri, sec, nextPri;
  OPEN_VLAN_LIST_t vidList;
  uint32_t vid;

  printf("Testing Private VLAN OpEN APIs sanity:\n\n");

  /* openapiPvlanIntfModeGet() */
  printf("Testing openapiPvlanIntfModeGet():\n");
  result = openapiPvlanIntfModeGet(NULL, ifNum, &swportMode);
  printf("NULL Client Handle:(result = %d)\n", result);
  result = openapiPvlanIntfModeGet(clientHandle, BAD_IFNUM, &swportMode);
  printf("Invalid Interface Number:(result = %d)\n", result);
  result = openapiPvlanIntfModeGet(clientHandle, ifNum, NULL);
  printf("NULL Mode Ptr:(result = %d)\n", result);
  printf("openapiPvlanIntfModeGet() sanity successful\n\n");

  /* openapiPvlanIntfHostAssocGet() */
  printf("Testing openapiPvlanIntfHostAssocGet():\n");
  result = openapiPvlanIntfHostAssocGet(NULL, ifNum, &pri, &sec);
  printf("NULL Client Handle:(result = %d)\n", result);
  result = openapiPvlanIntfHostAssocGet(clientHandle, BAD_IFNUM, &pri, &sec);
  printf("Invalid Interface Number:(result = %d)\n", result);
  result = openapiPvlanIntfHostAssocGet(clientHandle, ifNum, NULL, &sec);
  printf("NULL Primary VLAN Ptr:(result = %d)\n", result);
  result = openapiPvlanIntfHostAssocGet(clientHandle, ifNum, &pri, NULL);
  printf("NULL Secondary VLAN Ptr:(result = %d)\n", result);
  printf("openapiPvlanIntfHostAssocGet() sanity successful\n\n");

  /* openapiPvlanIntfPromiscAssocGet() */
  printf("Testing openapiPvlanIntfPromiscAssocGet():\n");
  result = openapiPvlanIntfPromiscAssocGet(NULL, ifNum, &pri, &vidList);
  printf("NULL Client Handle:(result = %d)\n", result);
  result = openapiPvlanIntfPromiscAssocGet(clientHandle, BAD_IFNUM, &pri, &vidList);
  printf("Invalid Interface Number:(result = %d)\n", result);
  result = openapiPvlanIntfPromiscAssocGet(clientHandle, ifNum, NULL, &vidList);
  printf("NULL Primary VLAN Ptr:(result = %d)\n", result);
  result = openapiPvlanIntfPromiscAssocGet(clientHandle, ifNum, &pri, NULL);
  printf("NULL Secondary VLAN List Ptr:(result = %d)\n", result);
  printf("openapiPvlanIntfPromiscAssocGet() sanity successful\n\n");

  /* openapiPvlanIntfOperAssocGet() */
  printf("Testing openapiPvlanIntfOperAssocGet():\n");
  result = openapiPvlanIntfOperAssocGet(NULL, ifNum, &vidList);
  printf("NULL Client Handle:(result = %d)\n", result);
  result = openapiPvlanIntfOperAssocGet(clientHandle, BAD_IFNUM, &vidList);
  printf("Invalid Interface Number:(result = %d)\n", result);
  result = openapiPvlanIntfOperAssocGet(clientHandle, ifNum, NULL);
  printf("NULL Operational VLAN List Ptr:(result = %d)\n", result);
  printf("openapiPvlanIntfOperAssocGet() sanity successful\n\n");

  /* openapiPvlanIntfPromiscTrunkAssocNextGet() */
  pri = 100;
  printf("Testing openapiPvlanIntfPromiscTrunkAssocNextGet():\n");
  result = openapiPvlanIntfPromiscTrunkAssocNextGet(NULL, ifNum, pri, &nextPri, &vidList);
  printf("NULL Client Handle:(result = %d)\n", result);
  result = openapiPvlanIntfPromiscTrunkAssocNextGet(clientHandle, BAD_IFNUM, pri, &nextPri, &vidList);
  printf("Invalid Interface Number:(result = %d)\n", result);
  result = openapiPvlanIntfPromiscTrunkAssocNextGet(clientHandle, ifNum, BAD_VLANID, &nextPri, &vidList);
  printf("Invalid Primary VLAN Input Value:(result = %d)\n", result);
  result = openapiPvlanIntfPromiscTrunkAssocNextGet(clientHandle, ifNum, pri, NULL, &vidList);
  printf("NULL Primary VLAN Output Ptr:(result = %d)\n", result);
  result = openapiPvlanIntfPromiscTrunkAssocNextGet(clientHandle, ifNum, pri, &nextPri, NULL);
  printf("NULL Secondary VLAN List Ptr:(result = %d)\n", result);
  printf("openapiPvlanIntfPromiscTrunkAssocNextGet() sanity successful\n\n");

  /* openapiPvlanIntfIsolatedTrunkAssocNextGet() */
  pri = 100;
  printf("Testing openapiPvlanIntfIsolatedTrunkAssocNextGet():\n");
  result = openapiPvlanIntfIsolatedTrunkAssocNextGet(NULL, ifNum, pri, &nextPri, &sec);
  printf("NULL Client Handle:(result = %d)\n", result);
  result = openapiPvlanIntfIsolatedTrunkAssocNextGet(clientHandle, BAD_IFNUM, pri, &nextPri, &sec);
  printf("Invalid Interface Number:(result = %d)\n", result);
  result = openapiPvlanIntfIsolatedTrunkAssocNextGet(clientHandle, ifNum, BAD_VLANID, &nextPri, &sec);
  printf("Invalid Primary VLAN Input Value:(result = %d)\n", result);
  result = openapiPvlanIntfIsolatedTrunkAssocNextGet(clientHandle, ifNum, pri, NULL, &sec);
  printf("NULL Primary VLAN Output Ptr:(result = %d)\n", result);
  result = openapiPvlanIntfIsolatedTrunkAssocNextGet(clientHandle, ifNum, pri, &nextPri, NULL);
  printf("NULL Secondary VLAN Output Ptr:(result = %d)\n", result);
  printf("openapiPvlanIntfIsolatedTrunkAssocNextGet() sanity successful\n\n");

  /* openapiPvlanIntfTrunkNativeVlanGet() */
  printf("Testing openapiPvlanIntfTrunkNativeVlanGet():\n");
  result = openapiPvlanIntfTrunkNativeVlanGet(NULL, ifNum, &vid);
  printf("NULL Client Handle:(result = %d)\n", result);
  result = openapiPvlanIntfTrunkNativeVlanGet(clientHandle, BAD_IFNUM, &vid);
  printf("Invalid Interface Number:(result = %d)\n", result);
  result = openapiPvlanIntfTrunkNativeVlanGet(clientHandle, ifNum, NULL);
  printf("NULL VLAN ID Ptr:(result = %d)\n", result);
  printf("openapiPvlanIntfTrunkNativeVlanGet() sanity successful\n\n");

  /* openapiPvlanIntfTrunkAllowedVlanListGet() */
  printf("Testing openapiPvlanIntfTrunkAllowedVlanListGet():\n");
  result = openapiPvlanIntfTrunkAllowedVlanListGet(NULL, ifNum, &vidList);
  printf("NULL Client Handle:(result = %d)\n", result);
  result = openapiPvlanIntfTrunkAllowedVlanListGet(clientHandle, BAD_IFNUM, &vidList);
  printf("Invalid Interface Number:(result = %d)\n", result);
  result = openapiPvlanIntfTrunkAllowedVlanListGet(clientHandle, ifNum, NULL);
  printf("NULL VLAN List Ptr:(result = %d)\n", result);
  printf("openapiPvlanIntfTrunkAllowedVlanListGet() sanity successful\n\n");

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for switchport configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum, ifNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];

  if (argc < 3)
  {
    printAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);
  ifNum = atoi(argv[2]);
  
  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister(exampleName, &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Private VLAN API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
  {
    printf("Network OS version = %s\n", switch_os_revision_string);
  }
  else
  {
    printf("Network OS Version retrieve error\n");
  }

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (verifyArgCount(argc, 3))
      {
        pvlanIntfModeGet(&clientHandle, ifNum);
      }
      break;

    case 2:
      if (verifyArgCount(argc, 3))
      {
        pvlanIntfHostAssocGet(&clientHandle, ifNum);
      }
      break;

    case 3:
      if (verifyArgCount(argc, 3))
      {
        pvlanIntfPromiscAssocGet(&clientHandle, ifNum);
      }
      break;

    case 4:
      if (verifyArgCount(argc, 3))
      {
        pvlanIntfOperAssocGet(&clientHandle, ifNum);
      }
      break;
    
    case 5:
      if (verifyArgCount(argc, 3))
      {
        pvlanIntfPromiscTrunkAssocList(&clientHandle, ifNum);
      }
      break;

    case 6:
      if (verifyArgCount(argc, 3))
      {
        pvlanIntfIsolatedTrunkAssocList(&clientHandle, ifNum);
      }
      break;
  
    case 7:
      if (verifyArgCount(argc, 3))
      {
        pvlanIntfTrunkNativeVlanGet(&clientHandle, ifNum);
      }
      break;
  
    case 8:
      if (verifyArgCount(argc, 3))
      {
        pvlanIntfTrunkNormalVlanList(&clientHandle, ifNum);
      }
      break;
  
    case 9:
      if (verifyArgCount(argc, 3))
      {
        pvlanOpENAPIsTestSanity(&clientHandle, ifNum);
      }
      break;

    default:
      printAppMenu();
      break;
  }
                                          
  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Private VLAN API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}

