/*********************************************************************
*
*  Copyright 2016-2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.                 
*
**********************************************************************
*
* @filename  voip_example.c
*
* @purpose   Example application to Auto-VoIP information using OpEN APIs
*
* @component OpEN
*
* @comments
*
* @create    09/14/2017
*
* @end
*
**********************************************************************/

#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_voip.h"

/*******************************************************************
*
* @brief  This function prints the Auto-VoIP Application Menu.
*
* @param  none
*
* @returns  none
*
* @end 
*********************************************************************/
void printVoIPAppMenu()
{
  printf("Usage: voip_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: Set VLAN ID for voip traffic: voip_example 1 <vlanID> \n");
  printf("Test 2: Get VLAN ID for voip traffic: voip_example 2 \n");
  printf("Test 3: Reset VLAN ID for voip traffic: voip_example 3 \n");
  printf("Test 4: Get maximum number of Auto-VoIP calls supported: voip_example 4 \n");
  printf("Test 5: Get number of Auto-VoIP Protocol Mode Active Sessions: voip_example 5 \n");
  printf("Test 6: Set VoIP profile cos queue: voip_example 6 <priority-scheme> {<cos> | <priority>} \n");
  printf("Test 7: Get VoIP profile cos queue: voip_example 7 \n");
  printf("Test 8: Get VoIP profile mode on interface: voip_example 8 <interface> <profile_type> <mode> \n");
  printf("Test 9: Set VoIP profile mode on interface: voip_example 9 <interface> <profile_type> \n");
  printf("Test 10: Get VoIP Protocol mode operational status on interface: voip_example 10 <interface> \n");
  printf("Test 11: Get VoIP OUI mode operational status on interface: voip_example 11 <interface> \n");
  printf("Test 12: Set priority for voip traffic: voip_example 12 <priority> \n");
  printf("Test 13: Get priority for voip traffic: voip_example 13 \n");
  printf("Test 14: Sanity: voip_example 14 \n");

  return;
}

/*********************************************************************
* @purpose  Sanity test Auto-VoIP information  
* 
* @param    clientHandle   @b{(input)} client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void voipSanityTest(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t val = 0;
  uint32_t intf = 0;
  OPEN_VOIP_PRI_SCHEME_t ptype;
  OPEN_QOS_VOIP_TYPE_t vtype;
  OPEN_CONTROL_t mode;
  OPEN_LINK_STATE_t linkVal;

  printf("Testing Auto-VoIP OpEN APIs sanity:\n\n"); 

  /* openapiVoipGlobalVlanIdSet */
  printf("Testing openapiVoipGlobalVlanIdSet() \n");

  val = 10;
  result = openapiVoipGlobalVlanIdSet(NULL, val);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiVoipGlobalVlanIdSet(clientHandle, 0);
  printf("Invalid VLAN ID:(result = %d)\n", result); 

  printf("openapiVoipGlobalVlanIdSet() sanity successful \n");

  /* openapiVoipGlobalVlanIdGet */
  printf("Testing openapiVoipGlobalVlanIdGet() \n");

  val = 0;
  result = openapiVoipGlobalVlanIdGet(NULL, &val);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiVoipGlobalVlanIdGet(clientHandle, NULL);
  printf("NULL value:(result = %d)\n", result); 

  printf("openapiVoipGlobalVlanIdGet() sanity successful \n");

  /* openapiVoipGlobalVlanIdReset */
  printf("Testing openapiVoipGlobalVlanIdReset() \n");

  result = openapiVoipGlobalVlanIdReset(NULL);
  printf("NULL Client Handle:(result = %d)\n", result); 

  printf("openapiVoipGlobalVlanIdReset() sanity successful \n");

  /* openapiVoipMaxCallsGet */
  printf("Testing openapiVoipMaxCallsGet() \n");

  val = 0;
  result = openapiVoipMaxCallsGet(NULL, &val);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiVoipMaxCallsGet(clientHandle, NULL);
  printf("NULL value:(result = %d)\n", result); 

  printf("openapiVoipMaxCallsGet() sanity successful \n");

  /* openapiVoipNumActiveProtocolSessionsGet */
  printf("Testing openapiVoipNumActiveProtocolSessionsGet() \n");

  val = 0;
  result = openapiVoipNumActiveProtocolSessionsGet(NULL, &val);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiVoipNumActiveProtocolSessionsGet(clientHandle, NULL);
  printf("NULL value:(result = %d)\n", result); 

  printf("openapiVoipNumActiveProtocolSessionsGet() sanity successful \n");

  /* openapiVoipProfileCosQueueSet */
  printf("Testing openapiVoipProfileCosQueueSet() \n");

  val = 5;
  ptype = OPEN_VOIP_PRI_REMARK;
  result = openapiVoipProfileCosQueueSet(NULL, ptype, val);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiVoipProfileCosQueueSet(clientHandle, 3, val);
  printf("Invalid type:(result = %d)\n", result); 

  result = openapiVoipProfileCosQueueSet(clientHandle, ptype, 10);
  printf("Invalid value:(result = %d)\n", result); 

  ptype = OPEN_VOIP_PRI_TRAFFIC_CLASS;
  result = openapiVoipProfileCosQueueSet(clientHandle, ptype, 8);
  printf("Invalid value:(result = %d)\n", result); 

  printf("openapiVoipProfileCosQueueSet() sanity successful \n");

  /* openapiVoipProfileCosQueueGet */
  printf("Testing openapiVoipProfileCosQueueGet() \n");

  val = 0;
  ptype = 0;
  result = openapiVoipProfileCosQueueGet(NULL, &ptype, &val);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiVoipProfileCosQueueGet(clientHandle, NULL, &val);
  printf("NULL type:(result = %d)\n", result); 

  result = openapiVoipProfileCosQueueGet(clientHandle, &ptype, NULL);
  printf("NULL value:(result = %d)\n", result); 

  printf("openapiVoipProfileCosQueueGet() sanity successful \n");

  /* openapiVoipOUIPrioritySet */
  printf("Testing openapiVoipOUIPrioritySet() \n");

  val = 6;
  result = openapiVoipOUIPrioritySet(NULL, val);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiVoipOUIPrioritySet(clientHandle, 10);
  printf("Invalid priority:(result = %d)\n", result); 

  printf("openapiVoipOUIPrioritySet() sanity successful \n");

  /* openapiVoipOUIPriorityGet */
  printf("Testing openapiVoipOUIPriorityGet() \n");

  val = 0;
  result = openapiVoipOUIPriorityGet(NULL, &val);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiVoipOUIPriorityGet(clientHandle, NULL);
  printf("NULL value:(result = %d)\n", result); 

  printf("openapiVoipOUIPriorityGet() sanity successful \n");

  intf = 5;
  vtype = OPEN_QOS_VOIP_TYPE_PROTOCOL;
  mode = OPEN_ENABLE;

  /* openapiVoipIntfModeSet */
  printf("Testing openapiVoipIntfModeSet() \n");

  result = openapiVoipIntfModeSet(NULL, intf, vtype, mode);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiVoipIntfModeSet(clientHandle, 0, vtype, mode);
  printf("Invalid interface:(result = %d)\n", result); 

  result = openapiVoipIntfModeSet(clientHandle, intf, OPEN_QOS_VOIP_TYPE_TOTAL, mode);
  printf("Invalid VOIP type:(result = %d)\n", result); 

  result = openapiVoipIntfModeSet(clientHandle, intf, vtype, 5);
  printf("Invalid mode:(result = %d)\n", result); 

  printf("openapiVoipIntfModeSet() sanity successful \n");

  /* openapiVoipIntfModeGet */
  printf("Testing openapiVoipIntfModeGet() \n");

  result = openapiVoipIntfModeGet(NULL, intf, vtype, &mode);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiVoipIntfModeGet(clientHandle, 0, vtype, &mode);
  printf("Invalid interface:(result = %d)\n", result); 

  result = openapiVoipIntfModeGet(clientHandle, intf, OPEN_QOS_VOIP_TYPE_TOTAL, &mode);
  printf("Invalid VOIP type:(result = %d)\n", result); 

  result = openapiVoipIntfModeGet(clientHandle, intf, vtype, NULL);
  printf("NULL mode:(result = %d)\n", result); 

  printf("openapiVoipIntfModeGet() sanity successful \n");

  /* openapiVoipIntfProtocolStatusGet */
  printf("Testing openapiVoipIntfProtocolStatusGet() \n");

  result = openapiVoipIntfProtocolStatusGet(NULL, intf, &linkVal);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiVoipIntfProtocolStatusGet(clientHandle, 0, &linkVal);
  printf("Invalid interface:(result = %d)\n", result); 

  result = openapiVoipIntfProtocolStatusGet(clientHandle, intf, NULL);
  printf("NULL mode:(result = %d)\n", result); 

  printf("openapiVoipIntfProtocolStatusGet() sanity successful \n");

  /* openapiVoipIntfOUIStatusGet */
  printf("Testing openapiVoipIntfOUIStatusGet() \n");

  result = openapiVoipIntfOUIStatusGet(NULL, intf, &linkVal);
  printf("NULL Client Handle:(result = %d)\n", result); 

  result = openapiVoipIntfOUIStatusGet(clientHandle, 0, &linkVal);
  printf("Invalid interface:(result = %d)\n", result); 

  result = openapiVoipIntfOUIStatusGet(clientHandle, intf, NULL);
  printf("NULL mode:(result = %d)\n", result); 

  printf("openapiVoipIntfOUIStatusGet() sanity successful \n");

  return;      
}

/*********************************************************************
* @purpose  Configures the VLAN ID for voip traffic on switch 
*
* @param    clientHandle  @b{(input)} client handle from registration API
* @param    vlanId        @b{(input)} VLAN ID 
*
* @returns  none 
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end 
*********************************************************************/
void voipGlobalVlanIdSet(openapiClientHandle_t *clientHandle, uint32_t vlanId)
{
  open_error_t result;
      
  if ((result = openapiVoipGlobalVlanIdSet(clientHandle, vlanId)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to set VLAN ID for VoIP traffic. (result = %d)\n", result);
  }    
  else 
  {    
    printf("VLAN ID for voip traffic set successfully \n");
  }    
  return;                                         
}

/*********************************************************************
* @purpose  Gets the VLAN ID for voip traffic on switch 
*
* @param    clientHandle  @b{(input)} client handle from registration API
*
* @returns  none 
* 
* @notes
* 
* @end 
*********************************************************************/
void voipGlobalVlanIdGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t vlanId = 0;
      
  if ((result = openapiVoipGlobalVlanIdGet(clientHandle, &vlanId)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to get VLAN ID for VoIP traffic. (result = %d)\n", result);
  }    
  else 
  {    
    printf("VLAN ID for voip traffic :%u \n", vlanId);
  }    
  return;                                         
}

/*********************************************************************
* @purpose  Resets the VLAN ID for voip traffic on switch 
*
* @param    clientHandle  @b{(input)} client handle from registration API
*
* @returns  none 
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end 
*********************************************************************/
void voipGlobalVlanIdReset(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
      
  if ((result = openapiVoipGlobalVlanIdReset(clientHandle)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to reset VLAN ID for VoIP traffic. (result = %d)\n", result);
  }    
  else 
  {    
    printf("VLAN ID for voip traffic reset successfully \n");
  }    
  return;                                         
}

/*********************************************************************
* @purpose  Gets the maximum number of Auto-VoIP calls supported  
*
* @param    clientHandle  @b{(input)} client handle from registration API
*
* @returns  none 
* 
* @notes
* 
* @end 
*********************************************************************/
void voipMaxCallsGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t val = 0;
      
  if ((result = openapiVoipMaxCallsGet(clientHandle, &val)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to get maximum number of Auto-VoIP calls supported. (result = %d)\n", result);
  }    
  else 
  {    
    printf("Maximum number of Auto-VoIP calls supported :%u \n", val);
  }    
  return;                                         
}

/*********************************************************************
* @purpose  Gets the number of active Auto-VoIP protocol sessions  
*
* @param    clientHandle  @b{(input)} client handle from registration API
*
* @returns  none 
* 
* @notes
* 
* @end 
*********************************************************************/
void voipNumActiveProtocolSessionsGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t val = 0;
      
  if ((result = openapiVoipNumActiveProtocolSessionsGet(clientHandle, &val)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to get number of active Auto-VoIP protocol sessions. (result = %d)\n", result);
  }    
  else 
  {    
    printf("Number of active Auto-VoIP protocol sessions :%u \n", val);
  }    
  return;                                         
}

/*********************************************************************
* @purpose  Sets the VoIP profile cos queue  
*
* @param    clientHandle  @b{(input)} client handle from registration API
* @param    type          @b{(input)} priScheme (remark or traffic-class)
* @param    val           @b{(input)} value
*
* @returns  none 
* 
* @notes Calling this API will change the running configuration of the switch
* 
* @end 
*********************************************************************/
void voipProfileCosQueueSet(openapiClientHandle_t *clientHandle, 
                            OPEN_VOIP_PRI_SCHEME_t type,
                            uint32_t val)
{
  open_error_t result;
      
  if ((result = openapiVoipProfileCosQueueSet(clientHandle, type, val)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to set VoIP profile cos queue. (result = %d)\n", result);
  }    
  else 
  {    
    printf("VoIP profile CoS queue set successfully \n");
  }    
  return;                                         
}

/*********************************************************************
* @purpose  Gets the VoIP profile cos queue  
*
* @param    clientHandle  @b{(input)} client handle from registration API
*
* @returns  none 
* 
* @notes
* 
* @end 
*********************************************************************/
void voipProfileCosQueueGet(openapiClientHandle_t *clientHandle) 
{
  open_error_t result;
  OPEN_VOIP_PRI_SCHEME_t type;
  uint32_t val;
      
  if ((result = openapiVoipProfileCosQueueGet(clientHandle, &type, &val)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to get VoIP profile cos queue. (result = %d)\n", result);
  }    
  else 
  {    
    printf("VoIP profile CoS queue type:%u value:%u \n", type, val);
  }    
  return;                                         
}

/*********************************************************************
* @purpose  Set the VOIP profile mode on interface   
*
* @param    clientHandle  @b{(input)} client handle from registration API
* @param    ifNum         @b{(input)} interface to be configured
* @param    type          @b{(input)} Auto VoIP type (protocol/OUI based)
* @param    mode          @b{(input)} Mode value
*
* @returns  none 
* 
* @notes Calling this API will change the running configuration of the switch
* 
* @end 
*********************************************************************/
void voipIntfModeSet(openapiClientHandle_t *clientHandle, 
                     uint32_t ifNum,
                     OPEN_QOS_VOIP_TYPE_t type,
                     OPEN_CONTROL_t mode)
{
  open_error_t result;
      
  if ((result = openapiVoipIntfModeSet(clientHandle, ifNum, type, mode)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to set VoIP profile mode on interface. (result = %d)\n", result);
  }    
  else 
  {    
    printf("VoIP profile set successfully on interface : %u \n", ifNum);
  }    
  return;                                         
}

/*********************************************************************
* @purpose  Gets the VOIP profile mode on interface   
*
* @param    clientHandle  @b{(input)} client handle from registration API
* @param    ifNum         @b{(input)} interface to be configured
* @param    type          @b{(input)} Auto VoIP type (protocol/OUI based)
*
* @returns  none 
* 
* @notes    none
* 
* @end 
*********************************************************************/
void voipIntfModeGet(openapiClientHandle_t *clientHandle, 
                     uint32_t ifNum,
                     OPEN_QOS_VOIP_TYPE_t type)
{
  open_error_t result;
  OPEN_CONTROL_t mode;
      
  if ((result = openapiVoipIntfModeGet(clientHandle, ifNum, type, &mode)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to get VoIP profile mode on interface. (result = %d)\n", result);
  }    
  else 
  {    
    printf("VoIP profile mode on interface %u : %u \n", ifNum, mode);
  }    
  return;                                         
}

/*********************************************************************
* @purpose  Gets VoIP Protocol mode operational port status   
*
* @param    clientHandle  @b{(input)} client handle from registration API
* @param    ifNum         @b{(input)} interface to be configured
*
* @returns  none 
* 
* @notes    none
* 
* @end 
*********************************************************************/
void voipIntfProtocolStatusGet(openapiClientHandle_t *clientHandle, 
                               uint32_t ifNum)
{
  open_error_t result;
  OPEN_LINK_STATE_t val;
      
  if ((result = openapiVoipIntfProtocolStatusGet(clientHandle, ifNum, &val)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to get VoIP protocol operational mode on interface. (result = %d)\n", result);
  }    
  else 
  {    
    printf("VoIP profile mode on interface %u : %u \n", ifNum, val);
  }    
  return;                                         
}

/*********************************************************************
* @purpose  Gets VoIP OUI mode operational port status   
*
* @param    clientHandle  @b{(input)} client handle from registration API
* @param    ifNum         @b{(input)} interface to be configured
*
* @returns  none 
* 
* @notes    none
* 
* @end 
*********************************************************************/
void voipIntfOUIStatusGet(openapiClientHandle_t *clientHandle, 
                          uint32_t ifNum)
{
  open_error_t result;
  OPEN_LINK_STATE_t val;
      
  if ((result = openapiVoipIntfOUIStatusGet(clientHandle, ifNum, &val)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to get VoIP OUI operational mode on interface. (result = %d)\n", result);
  }    
  else 
  {    
    printf("VoIP OUI mode on interface %u : %u \n", ifNum, val);
  }    
  return;                                         
}

/*********************************************************************
* @purpose  Sets the priority for voip traffic on switch   
*
* @param    clientHandle  @b{(input)} client handle from registration API
* @param    priority      @b{(input)}  priority value
*
* @returns  none 
* 
* @notes Calling this API will change the running configuration of the switch
* 
* @end 
*********************************************************************/
void voipOUIPrioritySet(openapiClientHandle_t *clientHandle, 
                        uint32_t priority)
{
  open_error_t result;
      
  if ((result = openapiVoipOUIPrioritySet(clientHandle, priority)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to set VoIP priority on switch. (result = %d)\n", result);
  }    
  else 
  {    
    printf("Priority for VoIP traffic on switch set successfully \n");
  }    
  return;                                         
}

/*********************************************************************
* @purpose  Gets the priority for voip traffic on switch   
*
* @param    clientHandle  @b{(input)} client handle from registration API
*
* @returns  none 
* 
* @notes Calling this API will change the running configuration of the switch
* 
* @end 
*********************************************************************/
void voipOUIPriorityGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t priority;
      
  if ((result = openapiVoipOUIPriorityGet(clientHandle, &priority)) != OPEN_E_NONE)
  {    
    printf("Bad return code trying to get VoIP OUI prority. (result = %d)\n", result);
  }    
  else 
  {    
    printf("Priority for VoIP traffic on switch : %u \n", priority);
  }    
  return;                                         
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for Auot-VoIP.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t val = 0;
  uint32_t val1 = 0;
  uint32_t testNum = 0;
  uint32_t ifNum = 0;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];

  if (argc < 2)
  {   
    printVoIPAppMenu();
    exit(1);
  } 

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();
          
  /* Register with OpEN */
  if ((result = openapiClientRegister("voip_example", &clientHandle)) != OPEN_E_NONE)
  { 
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }     
              
  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {   
    sleep(1);
  }
            
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Auto-VoIP API example application");
                
  printf("\n");

  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("ICOS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS Version retrieve error\n");

  printf("\n");
  
  /* exercise various OPEN API Auto-VoIP functions */
  switch (testNum)
  {
    case 1:
      if (argc != 3)
      {
        printVoIPAppMenu();
        exit(1);
      }  
      val = atoi(argv[2]);
      voipGlobalVlanIdSet(&clientHandle, val);
      break;
    case 2:
      if (argc != 2)
      {
        printVoIPAppMenu();
        exit(1);
      }  
      voipGlobalVlanIdGet(&clientHandle);
      break;
    case 3:
      if (argc != 2)
      {
        printVoIPAppMenu();
        exit(1);
      }  
      voipGlobalVlanIdReset(&clientHandle);
      break;
    case 4:
      if (argc != 2)
      {
        printVoIPAppMenu();
        exit(1);
      }  
      voipMaxCallsGet(&clientHandle);
      break;
    case 5:
      if (argc != 2)
      {
        printVoIPAppMenu();
        exit(1);
      }  
      voipNumActiveProtocolSessionsGet(&clientHandle);
      break;
    case 6:
      if (argc != 4)
      {
        printVoIPAppMenu();
        exit(1);
      }
      val = atoi(argv[2]);
      val1 = atoi(argv[3]);
      voipProfileCosQueueSet(&clientHandle, val, val1);
      break;
    case 7:
      if (argc != 2)
      {
        printVoIPAppMenu();
        exit(1);
      }  
      voipProfileCosQueueGet(&clientHandle);
      break;
    case 8:
      if (argc != 5)
      {
        printVoIPAppMenu();
        exit(1);
      }
      ifNum = atoi(argv[2]);
      val = atoi(argv[3]);
      val1 = atoi(argv[4]);
      voipIntfModeSet(&clientHandle, ifNum, val, val1);
      break;
    case 9:
      if (argc != 4)
      {
        printVoIPAppMenu();
        exit(1);
      }  
      ifNum = atoi(argv[2]);
      val = atoi(argv[3]);
      voipIntfModeGet(&clientHandle, ifNum, val);
      break;
    case 10:
      if (argc != 3)
      {
        printVoIPAppMenu();
        exit(1);
      }  
      val = atoi(argv[2]);
      voipIntfProtocolStatusGet(&clientHandle, val);
      break;
    case 11:
      if (argc != 3)
      {
        printVoIPAppMenu();
        exit(1);
      }  
      val = atoi(argv[2]);
      voipIntfOUIStatusGet(&clientHandle, val);
      break;
    case 12:
      if (argc != 3)
      {
        printVoIPAppMenu();
        exit(1);
      }  
      val = atoi(argv[2]);
      voipOUIPrioritySet(&clientHandle, val);
      break;
    case 13:
      if (argc != 2)
      {
        printVoIPAppMenu();
        exit(1);
      }  
      voipOUIPriorityGet(&clientHandle);
      break;
    case 14:
      if (argc != 2)  
      {
        printVoIPAppMenu();
        exit(1);
      }  
      voipSanityTest(&clientHandle);
      break;
    default:
      printVoIPAppMenu();
      break;  
  }

  /* Log goodbye message with OPEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Auto-VoIP API example application");

  return 0;
}  
