/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_cos_stats.h
*
* @purpose   Switch Cosq Statistics 
*
* @component OPEN
*
* @create    09/28/2016
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_COS_STATS_H_INCLUDED
#define OPENAPI_COS_STATS_H_INCLUDED

#include <stdio.h>
#include <stddef.h>
#include <sys/un.h>

#include "openapi_common.h"
#include "openapi_instru_bst.h"

#define OPEN_COS_ALL_QUEUES 0xffffffff  /**< read all queues, not just one */


/*********************************************************************
* @purpose  Get the number of CoS queues per interface.
*
* @param    client_handle   @b{(input)}   OpEN client handle
* @param    intIfNum        @b{(input)}   Interface number
* @param    count           @b{(output)}  Number of CoS queues per unit.
*
* @returns  OPEN_E_NONE        Success.
* @returns  OPEN_E_PARAM       Invalid argument.
* @returns  OPEN_E_FAIL        Failure occurred.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueNumQueuesGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *count);

/*********************************************************************
* @purpose  Get the number of UC packets tx for an interface and CoS queue.
*
* @param    client_handle   @b{(input)}   OpEN client handle
* @param    intIfNum        @b{(input)}   Interface number
* @param    qIndex          @b{(input)}   CoS queue index 
* @param    -               Minimum value 0
* @param    -               Maximum value openapiCosQueueNumQueuesGet() - 1. Or, use @ref OPEN_COS_ALL_QUEUES to read all queues.
* @param    data            @b{(output)}  Vector containing number of UC 
*                                         packets transmitted for specified 
*                                         interface, queue. Element type is
*                                         uint64_t. Number of elements is 
*                                         either 1 or return value of 
*                                         openapiCosQueueNumQueuesGet (when
*                                         qIndex is OPEN_COS_ALL_QUEUES) for
*                                         the interface the data is being 
*                                         requested. Vector is indexed by CoS
*                                         queue index.
* @param    count           @b{(output)}  How many elements returned in data
*
* @returns  OPEN_E_NONE        Success.
* @returns  OPEN_E_PARAM       Invalid argument.
* @returns  OPEN_E_FAIL        Other failure occurred.
* @returns  OPEN_E_NOT_FOUND   Either interface or cosq not available.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueUcTxPacketsGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t qIndex, open_buffdesc *data, uint32_t *count);

/*********************************************************************
* @purpose  Get the number of MC packets tx for an interface and CoS queue.
*
* @param    client_handle   @b{(input)}   OpEN client handle
* @param    intIfNum        @b{(input)}   Interface number
* @param    qIndex          @b{(input)}   CoS queue index 
* @param    -               Minimum value 0
* @param    -               Maximum value openapiCosQueueNumQueuesGet() - 1. Or, use @ref OPEN_COS_ALL_QUEUES to read all queues.
* @param    data            @b{(output)}  Vector containing number of MC 
*                                         packets transmitted for specified 
*                                         interface, queue. Element type is
*                                         uint64_t. Number of elements is
*                                         either 1 or return value of
*                                         openapiCosQueueNumQueuesGet (when
*                                         qIndex is OPEN_COS_ALL_QUEUES) for
*                                         the interface the data is being
*                                         requested. Vector is indexed by CoS
*                                         queue index.
* @param    count           @b{(output)}  How many elements returned in data
*
* @returns  OPEN_E_NONE        Success.
* @returns  OPEN_E_FAIL        Other failure occurred.
* @returns  OPEN_E_PARAM       Invalid argument.
* @returns  OPEN_E_NOT_FOUND   Either interface or cosq not available.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueMcTxPacketsGet(openapiClientHandle_t * client_handle, uint32_t intIfNum, uint32_t qIndex, open_buffdesc *data, uint32_t *count);

/*********************************************************************
* @purpose  Get the number of CPU packets tx for a given CoS queue.
*
* @param    client_handle   @b{(input)}   OpEN client handle
* @param    qIndex          @b{(input)}   CoS queue index
* @param    -               Minimum value 0
* @param    -               Maximum value openapiCosQueueNumQueuesGet() - 1. Or, use @ref OPEN_COS_ALL_QUEUES to read all queues.
* @param    data            @b{(output)}  Vector containing number of MC
*                                         packets transmitted for specified
*                                         interface, queue. Element type is
*                                         uint64_t. Number of elements is
*                                         either 1 or return value of
*                                         openapiCosQueueNumQueuesGet (when
*                                         qIndex is OPEN_COS_ALL_QUEUES) for
*                                         the interface the data is being
*                                         requested. Vector is indexed by CoS
*                                         queue index.
* @param    count           @b{(output)}  How many elements returned in data
*
* @returns  OPEN_E_NONE        Success.
* @returns  OPEN_E_FAIL        Other failure occurred.
* @returns  OPEN_E_PARAM       Invalid argument.
* @returns  OPEN_E_NOT_FOUND   Cosq not available.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueCpuTxPacketsGet(openapiClientHandle_t * client_handle, uint32_t qIndex, open_buffdesc *data, uint32_t *count);

/*********************************************************************
* @purpose  Get the number of CPU packets dropped for a given CoS queue.
*
* @param    client_handle   @b{(input)}   OpEN client handle
* @param    qIndex          @b{(input)}   CoS queue index
* @param    -               Minimum value 0
* @param    -               Maximum value openapiCosQueueNumQueuesGet() - 1. Or, use @ref OPEN_COS_ALL_QUEUES to read all queues.
* @param    data            @b{(output)}  Vector containing number of MC
*                                         packets transmitted for specified
*                                         interface, queue. Element type is
*                                         uint64_t. Number of elements is
*                                         either 1 or return value of
*                                         openapiCosQueueNumQueuesGet (when
*                                         qIndex is OPEN_COS_ALL_QUEUES) for
*                                         the interface the data is being
*                                         requested. Vector is indexed by CoS
*                                         queue index.
* @param    count           @b{(output)}  How many elements returned in data
*
* @returns  OPEN_E_NONE        Success.
* @returns  OPEN_E_FAIL        Other failure occurred.
* @returns  OPEN_E_PARAM       Invalid argument.
* @returns  OPEN_E_NOT_FOUND   Cosq not available.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueCpuDropPacketsGet(openapiClientHandle_t * client_handle, uint32_t qIndex, open_buffdesc *data, uint32_t *count);

#endif /* OPENAPI_COS_STATS_H_INCLUDED */
