/**********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_dhcp6s.h
*
* @purpose   This code implements the OpEN DHCPv6 API
*
* @component OPEN
*
* @create    07/26/2023
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_DHCP6S_H_INCLUDED
#define OPENAPI_DHCP6S_H_INCLUDED

#include "openapi_common.h" 


#define OPEN_DHCP6S_DNS_DOMAIN_MAX      8
#define OPEN_DHCP6S_DOMAIN_NAME_MINLEN  1
#define OPEN_DHCP6S_DOMAIN_NAME_MAXLEN  253
#define OPEN_DHCP6S_DNS_SERVER_MAX      8
#define OPEN_DHCP6S_MAX_POOL_NUMBER     256

/**< Structure to store DNS-Domsins list */
typedef struct
{
  uint8_t dns_domains[OPEN_DHCP6S_DNS_DOMAIN_MAX][OPEN_DHCP6S_DOMAIN_NAME_MAXLEN];
} openDnsDomainTable_t;

/**< Structure to store DNS-Servers list */
typedef struct
{
  open_in6_addr_t dns_servers[OPEN_DHCP6S_DNS_SERVER_MAX];
} openDnsServerTable_t;

/**< DHCP6 Prefix types */
typedef enum {
  OPEN_DHCP6_PREFIX_IAPD = 0,
  OPEN_DHCP6_PREFIX_IANA, 
  OPEN_DHCP6_PREFIX_IATA 
} OPEN_DHCP6_PREFIX_TYPE_t;

/**< DHCP6 interface mode types */
typedef enum {
  OPEN_DHCP6_MODE_NONE = 0,
  OPEN_DHCP6_MODE_SERVER,  
  OPEN_DHCP6_MODE_CLIENT, 
  OPEN_DHCP6_MODE_RELAY
} OPEN_DHCP6_MODE_t;

/**< Defines for DHCP6 POOL Type */
#define OPEN_DHCP6S_INACTIVE_POOL       0
#define OPEN_DHCP6S_DYNAMIC_POOL        1
#define OPEN_DHCP6S_MANUAL_POOL         2
#define OPEN_DHCP6S_NOT_READY_POOL      3  /**< To support SNMP create operation */

#define OPEN_DHCP6S_POOL_NAME_MINLEN    1
#define OPEN_DHCP6S_POOL_NAME_MAXLEN    32
#define OPEN_DHCP6S_HOST_NAME_MAXLEN    32

#define OPEN_DHCP6S_MAX_DUID_STR_LEN    389
#define OPEN_DHCP6S_RELAYOPT_REMOTEID_MAXLEN  64
/**< wildcard for IP helper ingress interface */
#define OPEN_IH_INTF_ANY 0

/**< DHCPv6 statistics common data-structure for both interface statistics and per VRF global statistics */
typedef struct openDhcp6sStatData_s
{
                      /**< Type of packets received from client */
  uint32_t           numOfDhcpsSolicitReceived;
  uint32_t           numOfDhcpsRequestReceived;
  uint32_t           numOfDhcpsConfirmReceived;
  uint32_t           numOfDhcpsRenewReceived;
  uint32_t           numOfDhcpsRebindReceived;
  uint32_t           numOfDhcpsReleaseReceived;
  uint32_t           numOfDhcpsDeclineReceived;
  uint32_t           numOfDhcpsInformReceived;
  uint32_t           numOfDhcpsRelayReplyReceived;
  uint32_t           numOfDhcpsRelayForwReceived;
  uint32_t           totalPacketsReceived;
                      /**< Type of packets sent to client */
  uint32_t           numOfDhcpsAdvertiseSent;
  uint32_t           numOfDhcpsReplySent;
  uint32_t           numOfDhcpsReconfigureSent;
  uint32_t           numOfDhcpsRelayReplySent;
  uint32_t           numOfDhcpsRelayForwSent;

  uint32_t           numOfMalformedMessages;
  uint32_t           numOfDhcpPacketsDiscarded;
  uint32_t           totalPacketsSent;
} openDhcp6sStatData_t;

/*****************************************************************************
 * @purpose   Get the administrative mode of the IPv6 DHCP-Sever for given VRF.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     vrfName           @b{(input)}  VRF name
 * @param     mode              @b{(output)} OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sAdminModeGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the administrative mode of the IPv6 DHCP-Sever for given VRF.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     vrfName           @b{(input)}  VRF name
 * @param     mode              @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sAdminModeSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the DHCPv6 Server DUID.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     duid              @b{(output)} String representation of DUID
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sServerDuidGet(openapiClientHandle_t *client_handle, open_buffdesc *duid);

/*****************************************************************************
 * @purpose   Get the number of pool for a DHCPv6 Server.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     poolsCount        @b{(output)} Count of number of Pools
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sPoolNumGet(openapiClientHandle_t *client_handle, uint32_t *poolsCount);

/*****************************************************************************
 * @purpose   Check if a DHCPv6 pool corresponding to number is valid or not.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     poolNumber        @b{(input)}  Pool number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sPoolValidate(openapiClientHandle_t *client_handle, uint32_t poolNumber);

/*****************************************************************************
 * @purpose   Get the pool name corresponding to a valid pool number.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     poolNumber        @b{(input)}  Pool number
 * @param     poolName          @b{(output)} Name of address pool
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sNumberToPoolNameGet(openapiClientHandle_t *client_handle, uint32_t poolNumber, open_buffdesc *poolName);

/*****************************************************************************
 * @purpose   Create a DHCPv6 address pool.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     poolName          @b{(input)}  Name of address pool
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sPoolCreate(openapiClientHandle_t *client_handle, open_buffdesc *poolName);

/*****************************************************************************
 * @purpose   Delete a DHCPv6 address pool.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     poolName          @b{(input)}  Name of address pool
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sPoolDelete(openapiClientHandle_t *client_handle, open_buffdesc *poolName);

/*****************************************************************************
 * @purpose   Get the first entry in DHCPv6 address pool.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     poolName          @b{(output)} Name of address pool
 * @param     poolIndex         @b{(output)} Pool index to fetch next entry
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sPoolEntryFirst(openapiClientHandle_t *client_handle, open_buffdesc *poolName, uint32_t *poolIndex);

/*****************************************************************************
 * @purpose   Get the next entry in DHCPv6 address pool.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     poolName          @b{(output)} Name of address pool
 * @param     poolIndex         @b{(inout)}  pool index to fetch next entry
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sPoolEntryNext(openapiClientHandle_t *client_handle, open_buffdesc *poolName, uint32_t *poolIndex);

/*****************************************************************************
 * @purpose   Add a host within a DHCPv6 Server pool.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     poolName          @b{(input)}  Name of address pool
 * @param     hostName          @b{(input)}  Host name
 * @param     prefixAddr        @b{(input)}  delegated prefix address
 * @param     prefixLen         @b{(input)}  delegated prefix mask length
 * @param     validLifetime     @b{(input)}  delegated prefix valid lifetime
 * @param     preferLifetime    @b{(input)}  delegated prefix prefer lifetime
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sPoolHostAdd(openapiClientHandle_t *client_handle, open_buffdesc *poolName, open_buffdesc *hostName,
                                      open_inet_addr_t *prefixAddr, uint16_t prefixLen, uint32_t validLifetime, uint32_t preferLifetime);

/*****************************************************************************
 * @purpose   Delete the host within a DHCPv6 Server pool.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     poolName          @b{(input)}  Name of address pool
 * @param     prefixAddr        @b{(input)}  delegated prefix address
 * @param     prefixLen         @b{(input)}  delegated prefix mask length
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sPoolHostRemove(openapiClientHandle_t *client_handle, open_buffdesc *poolName, open_inet_addr_t *prefixAddr, uint16_t prefixLen);

/*****************************************************************************
 * @purpose   Get next host within a DHCPv6 Server pool.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     poolName          @b{(input)}  Name of address pool
 * @param     duidStr           @b{(inout)}  String representing host DUID
 * @param     hostName          @b{(inout)}  Host name
 * @param     iaid              @b{(inout)}  Identity Association ID
 * @param     prefixAddr        @b{(inout)}  delegated prefix address
 * @param     prefixLen         @b{(inout)}  delegated prefix mask length
 * @param     validLifetime     @b{(inout)}  delegated prefix valid lifetime
 * @param     preferLifetime    @b{(inout)}  delegated prefix prefer lifetime
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sPoolHostGetNext(openapiClientHandle_t *client_handle, open_buffdesc *poolName, open_buffdesc *duidStr, open_buffdesc *hostName, uint32_t *iaid,
                                          open_inet_addr_t *prefixAddr, uint16_t *prefixLen, uint32_t *validLifetime, uint32_t *preferLifetime);

/*****************************************************************************
 * @purpose   Get the DHCPv6 Server pool type.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     poolName          @b{(input)}  Name of address pool
 * @param     poolType          @b{(output)} Pool type
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Pool-type values [OPEN_DHCP6S_INACTIVE_POOL(0), OPEN_DHCP6S_DYNAMIC_POOL(1),
 *                              OPEN_DHCP6S_MANUAL_POOL(2),   OPEN_DHCP6S_NOT_READY_POOL(3)]
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sPoolTypeGet(openapiClientHandle_t *client_handle, open_buffdesc *poolName, uint32_t *poolType);

/*****************************************************************************
 * @purpose   Get the DHCPv6 DNS Domain Names from a pool.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     poolName          @b{(input)}  Name of address pool
 * @param     dnsDomainNames    @b{(output)} Array of DNS domain names
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sPoolDnsDomainsGet(openapiClientHandle_t *client_handle, open_buffdesc *poolName, openDnsDomainTable_t *dnsDomainNames);

/*****************************************************************************
 * @purpose   Add a DHCPv6 DNS Domain Name for a pool.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     poolName          @b{(input)}  Name of address pool
 * @param     domainName        @b{(input)}  DNS domain name
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sPoolDnsDomainAdd(openapiClientHandle_t *client_handle, open_buffdesc *poolName, open_buffdesc *domainName);

/*****************************************************************************
 * @purpose   Remove a DHCPv6 DNS Domain Name from a pool.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     poolName          @b{(input)}  Name of address pool
 * @param     domainName        @b{(input)}  DNS domain name
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sPoolDnsDomainRemove(openapiClientHandle_t *client_handle, open_buffdesc *poolName, open_buffdesc *domainName);

/*****************************************************************************
 * @purpose   Add a DHCPv6 host DNS Server for a automatic pool.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     poolName          @b{(input)}  Name of address pool
 * @param     pAddr             @b{(input)}  An IPv6 address of a DNS Server
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sPoolDnsServerAdd(openapiClientHandle_t *client_handle, open_buffdesc *poolName, open_inet_addr_t *pAddr);

/*****************************************************************************
 * @purpose   Get DHCPv6 host DNS Servers for a automatic pool.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     poolName          @b{(input)}  Name of address pool
 * @param     dnsServers        @b{(output)} Array of ipAddress of DNS Servers
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sPoolDnsServersGet(openapiClientHandle_t *client_handle, open_buffdesc *poolName, openDnsServerTable_t *dnsServers);

/*****************************************************************************
 * @purpose   Delete a DHCPv6 host DNS Servers for a automatic pool.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     poolName          @b{(input)}  Name of address pool
 * @param     pAddr             @b{(input)}  An IPv6 address of a DNS Server
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sPoolDnsServerRemove(openapiClientHandle_t *client_handle, open_buffdesc *poolName, open_inet_addr_t *pAddr);

/*****************************************************************************
 * @purpose   Delete DHCPv6 Server/Relay parameters on a given interface and VRF.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     ifNum             @b{(input)}  Internal Interface Number of the interface
 * @param     vrfName           @b{(input)}  VRF name
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sIntfModeDelete(openapiClientHandle_t *client_handle, uint32_t ifNum, open_buffdesc *vrfName);

/*****************************************************************************
 * @purpose   Set DHCPv6 Relay parameters for an interface.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     vrfName           @b{(input)}  VRF name
 * @param     ifNum             @b{(input)}  Internal Interface Number of the interface
 * @param     serverAddr        @b{(input)}  Relay Server address
 * @param     serverIfNum       @b{(input)}  Internal Interface Number of relay server interface
 * @param     remoteId          @b{(input)}  remote-id relay option
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sIntfModeRelaySet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t ifNum, open_inet_addr_t *serverAddr, uint32_t serverIfNum, open_buffdesc *remoteId);

/*****************************************************************************
 * @purpose   Delete DHCPv6 Relay entry/parameters for a given interface.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     vrfName           @b{(input)}  VRF name
 * @param     ifNum             @b{(input)}  Internal Interface Number of the interface
 * @param     serverAddr        @b{(input)}  Relay Server address
 * @param     serverIfNum       @b{(input)}  Internal Interface Number of relay server interface
 * @param     remoteId          @b{(input)}  remote-id relay option
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sIntfRelayEntryDelete(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t ifNum, open_inet_addr_t *serverAddr, uint32_t serverIfNum, open_buffdesc *remoteId);

/*****************************************************************************
 * @purpose   Set DHCPv6 Server parameters for an interface.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     ifNum             @b{(input)}  Internal Interface Number of the interface
 * @param     poolName          @b{(input)}  DHCPv6 pool name
 * @param     serverPref        @b{(input)}  Server preference
 * @param     rapidCommit       @b{(input)}  rapid commit option
 * @param     allowUnicast      @b{(input)}  allow unicast option
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sIntfModeServerSet(openapiClientHandle_t *client_handle, uint32_t ifNum, open_buffdesc *poolName, uint32_t serverPref, OPEN_BOOL_t rapidCommit, OPEN_BOOL_t allowUnicast);

/*****************************************************************************
 * @purpose   Get the DHCPv6 Interface parameters for a given interface.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     ifNum             @b{(input)}  Internal Interface Number of the interface
 * @param     mode              @b{(output)} DHCP6 mode on interface
 * @param     pPoolName         @b{(output)} DHCPv6 pool name
 * @param     serverPref        @b{(output)} Server preference
 * @param     relayAddr         @b{(output)} IPv6 address of relay server
 * @param     relayIfNum        @b{(output)} Internal Interface Number of relay server interface
 * @param     remoteId          @b{(output)} remote-id Relay option
 * @param     rapidCommit       @b{(output)} rapid commit option
 * @param     allowUnicast      @b{(output)} allow unicast option
 * @param     localVrfName      @b{(output)} VRF Name of the interface
 * @param     relayDestVrfName  @b{(output)} VRF Name of the relay destination
 * @param     pdClientIntIfNum  @b{(output)} PD client interface number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sIntfModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_DHCP6_MODE_t *mode, open_buffdesc *pPoolName, uint32_t *serverPref,
                                      open_inet_addr_t *relayAddr, uint32_t *relayIfNum, open_buffdesc *remoteId, OPEN_BOOL_t *rapidCommit, OPEN_BOOL_t *allowUnicast,
                                      open_buffdesc *localVrfName, open_buffdesc *relayDestVrfName, uint32_t *pdClientIntIfNum);

/*****************************************************************************
 * @purpose   Get the next interface with active DHCPv6 configuration.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     ifNum             @b{(inout)}  Internal Interface Number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Set ifNum to 0 to retrieve first entry.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sIntfGetNext(openapiClientHandle_t *client_handle, uint32_t *ifNum);

/*****************************************************************************
 * @purpose   Verify the interface has active DHCPv6 configuration.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     ifNum             @b{(input)}  Internal Interface Number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sIntfValidate(openapiClientHandle_t *client_handle, uint32_t ifNum);

/*****************************************************************************
 * @purpose   Determine if the interface is valid for participation in DHCPv6 components.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     ifNum             @b{(input)}  Internal Interface Number
 * @param     isValidFlag       @b{(output)} returns OPEN_TRUE or OPEN_FALSE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6MapIntfIsValid(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_BOOL_t *isValidFlag);

/*****************************************************************************
 * @purpose   Get the first valid interface for participation in DHCPv6 components.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     ifNum             @b{(output)} Internal Interface Number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6MapIntfValidFirstGet(openapiClientHandle_t *client_handle, uint32_t *ifNum);

/*****************************************************************************
 * @purpose   Get the next valid interface for participation in DHCPv6 components.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     prevIfNum         @b{(input)}  Previous Internal Interface Number
 * @param     ifNum             @b{(output)} Next Internal Interface Number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6MapIntfValidNextGet(openapiClientHandle_t *client_handle, uint32_t prevIfNum, uint32_t *ifNum);

/*****************************************************************************
 * @purpose   Get next active binding from the DHCPv6 server for a given VRF.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     vrfName           @b{(input)}  VRF name
 * @param     duid              @b{(inout)}  String representing client DUID
 * @param     iaid              @b{(inout)}  Identity Association ID
 * @param     prefixAddr        @b{(inout)}  Binding prefix address
 * @param     prefixLen         @b{(inout)}  Binding prefix mask length
 * @param     prefixType        @b{(inout)}  Binding prefix type (OPEN_DHCP6_PREFIX_TYPE_t)
 * @param     clientAddr        @b{(output)} Client address
 * @param     clientIntfNum     @b{(output)} Client interface number
 * @param     expiration        @b{(output)} Expiration (in seconds)
 * @param     validLifetime     @b{(output)} Binding prefix valid lifetime
 * @param     preferLifetime    @b{(output)} Binding prefix prefer lifetime
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Set pDuidStr to empty string to retrieve first entry.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sBindingGetNext(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, open_buffdesc *duid, uint32_t *iaid, open_inet_addr_t *prefixAddr,
                                         uint16_t *prefixLen, uint32_t *prefixType, open_inet_addr_t *clientAddr, uint32_t *clientIntfNum, uint32_t *expiration,
                                         uint32_t *validLifetime, uint32_t *preferLifetime);

/*****************************************************************************
 * @purpose   Clear the binding specified by the IPv6 address for the given VRF.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     vrfName           @b{(input)}  VRF name
 * @param     prefixType        @b{(input)}  Binding prefix type
 * @param     clientAddr        @b{(input)}  Client IPv6 address
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sBindingClear(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_DHCP6_PREFIX_TYPE_t prefixType, open_inet_addr_t *clientAddr);

/*****************************************************************************
 * @purpose   Clear all the binding entries for a given VRF.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     vrfName           @b{(input)}  VRF name
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sBindingClearAll(openapiClientHandle_t *client_handle, open_buffdesc *vrfName);

/*****************************************************************************
 * @purpose   Get the count of number of bindings in the binding database for a given VRF.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     vrfName           @b{(input)}  VRF name
 * @param     count             @b{(output)} Pointer to the count of the number of bindings
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sBindingCountGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t *count);

/*****************************************************************************
 * @purpose   Get the DHCPv6-Server statistics for a given VRF or interface.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     ifNum             @b{(input)}  Internal Interface Number
 * @param     vrfName           @b{(input)}  VRF name
 * @param     dhcp6Stats        @b{(output)} DHCPv6 statistics for the given interface or VRF
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sStatisticsGet(openapiClientHandle_t *client_handle, uint32_t ifNum, open_buffdesc *vrfName, openDhcp6sStatData_t *dhcp6Stats);

/*****************************************************************************
 * @purpose   Clear the DHCPv6-Server statistics for a given interface or VRF.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     ifNum             @b{(input)}  Internal Interface Number
 * @param     vrfName           @b{(input)}  VRF name
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcp6sStatisticsClear(openapiClientHandle_t *client_handle, uint32_t ifNum, open_buffdesc *vrfName);

/*****************************************************************************
 * @purpose   Get the next DHCPv6 interface relay parameters for an interface.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     ifNum             @b{(input)}  Internal Interface Number of ingress interface
 * @param     relayAddr         @b{(input)}  IPv6 address of relay server
 * @param     relayIfNum        @b{(input)}  Internal Interface Number of relay interface
 * @param     vrfName           @b{(input)}  VRF name of relay destination
 * @param     nextIfNum         @b{(output)} Internal Interface Number of next ingress interface
 * @param     nextRelayAddr     @b{(output)} IPv6 address of next relay server
 * @param     nextRelayIfNum    @b{(output)} Internal Interface Number of next relay interface
 * @param     nextVrfName       @b{(output)} VRF name of relay destination
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     To get the first entry set all the input parameters to zero.
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpv6RelayEntryNextGet(openapiClientHandle_t *client_handle, uint32_t ifNum, open_inet_addr_t *relayAddr, uint32_t *relayIfNum, open_buffdesc *vrfName,
                                            uint32_t *nextIfNum, open_inet_addr_t *nextRelayAddr, uint32_t *nextRelayIfNum, open_buffdesc *nextVrfName);

/*****************************************************************************
 * @purpose   Get the exact DHCPv6 interface relay entry.
 *
 * @param     client_handle     @b{(input)}  Client handle from registration API
 * @param     ifNum             @b{(input)}  Internal Interface Number of ingress interface
 * @param     relayAddr         @b{(input)}  IPv6 address of server
 * @param     relayIfNum        @b{(output)} Internal Interface Number of relay interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpv6RelayEntryGet(openapiClientHandle_t *client_handle, uint32_t ifNum, open_in6_addr_t *relayAddr, uint32_t *relayIfNum);


#endif /* OPENAPI_DHCP6S_H_INCLUDED */

