/*********************************************************************
*
* Copyright 2018-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_dhcp_server.h
*
* @purpose   DHCP Server API
*
* @component OpEN
*
* @comments  This code must not include any ICOS header files.
*
* @create    7/6/2018
*
* @end
*                                    
**********************************************************************/
#ifndef OPENAPI_DHCP_SERVER_H_INCLUDED
#define OPENAPI_DHCP_SERVER_H_INCLUDED

#include "openapi_common.h"

/**< Pool types */
typedef enum
{
  OPEN_DHCP_SERVER_POOL_TYPE_INACTIVE      = 0, /**< inactive */
  OPEN_DHCP_SERVER_POOL_TYPE_DYNAMIC       = 1, /**< dynamic */
  OPEN_DHCP_SERVER_POOL_TYPE_MANUAL        = 2  /**< manual */
} OPEN_DHCP_SERVER_POOL_TYPE_t;

/**< DHCP server lease states */
typedef enum
{
  OPEN_DHCP_SERVER_FREE_LEASE       = 0, /**< Free */
  OPEN_DHCP_SERVER_ACTIVE_LEASE     = 1, /**< Active */
  OPEN_DHCP_SERVER_OFFERED_LEASE    = 2, /**< Offered */
  OPEN_DHCP_SERVER_EXPIRED_LEASE    = 3, /**< Expired */
  OPEN_DHCP_SERVER_ABANDONED_LEASE  = 4, /**< Abandoned */
} OPEN_DHCP_SERVER_LEASE_STATE_t;

#define OPEN_DHCP_SERVER_POOL_NAME_MAXLEN     32  /**< Maximum pool name, includes terminating '\0' */
#define OPEN_DHCP_SERVER_CLIENT_ID_MAXLEN     255 /**< Maximum client ID, includes terminating '\0' */
#define OPEN_DHCP_SERVER_POOL_VRF_NAME_MAXLEN 65  /**< Maximum VRF name, includes terminating '\0' */
#define OPEN_DHCP_SERVER_CLIENT_ID_STR_MAXLEN (3 * OPEN_DHCP_SERVER_CLIENT_ID_MAXLEN) /**< Maximum client id string lenght in xx:xx:xx:xx.... fromat */
#define OPEN_DHCP_SERVER_CLIENT_NAME_MAXLEN   31  /**< Maximum client name length */ 

/**< DHCP Pool name */
typedef struct
{
  char name[OPEN_DHCP_SERVER_POOL_NAME_MAXLEN];  /**< DHCP server pool name */
} OPEN_DHCP_SERVER_POOL_NAME_t;

/**< DHCP Pool's VRF-name */
typedef struct
{
  char name[OPEN_DHCP_SERVER_POOL_VRF_NAME_MAXLEN];  /**< DHCP server pool's VRF name */
} OPEN_DHCP_SERVER_POOL_VRF_NAME_t;

/**< DHCP client ID */
typedef struct
{
  char clientid[OPEN_DHCP_SERVER_CLIENT_ID_MAXLEN];  /**< DHCP server client id  */
} OPEN_DHCP_SERVER_CLIENT_ID_t;

/**< DHCP Server Statistic IDs */
typedef enum
{
  OPEN_DHCPS_STAT_ID_RESERVED          = 0,     /**< Reserved (unused).              */
  OPEN_DHCPS_STAT_ID_ACTIVE_LEASES     = 1,     /**< Active leases.                  */
  OPEN_DHCPS_STAT_ID_EXPIRED_LEASES    = 2,     /**< Expired leases.                 */
  OPEN_DHCPS_STAT_ID_MALFORMED_MSGS    = 3,     /**< Malformed messages.             */
  OPEN_DHCPS_STAT_ID_DISCOVER_DROPPED  = 4,     /**< Discover messages dropped.      */
  OPEN_DHCPS_STAT_ID_RX_DISCOVER       = 5,     /**< Discover messages received.     */
  OPEN_DHCPS_STAT_ID_RX_REQUEST        = 6,     /**< Request messages received.      */
  OPEN_DHCPS_STAT_ID_RX_DECLINE        = 7,     /**< Decline messages received.      */
  OPEN_DHCPS_STAT_ID_RX_RELEASE        = 8,     /**< Release messages received.      */
  OPEN_DHCPS_STAT_ID_RX_INFORM         = 9,     /**< Inform messages received.       */
  OPEN_DHCPS_STAT_ID_TX_OFFER          = 10,    /**< Offer messages sent.            */
  OPEN_DHCPS_STAT_ID_TX_ACK            = 11,    /**< Ack messages sent.              */
  OPEN_DHCPS_STAT_ID_TX_NACK           = 12,    /**< Nack messages sent.             */
  OPEN_DHCPS_STAT_ID_LAST                       /**< Must be last (bounds checking). */
} OPEN_DHCPS_STAT_ID_t;

/**< NetBIOS over TCP/IP node types */
typedef enum
{
  OPEN_NETBIOS_BROADCAST_NODE = 0x1,    /**< Broadcast node */
  OPEN_NETBIOS_P2P_NODE       = 0x2,    /**< Peer to Peer node */
  OPEN_NETBIOS_MIXED_NODE     = 0x4,    /**< Mixed node */
  OPEN_NETBIOS_HYBRID_NODE    = 0x8     /**< Hybrid node */
} OPEN_NETBIOS_NODE_TYPES_t;

/**< Hardware address type of DHCP Clients */
typedef enum
{
  OPEN_DHCPS_HW_TYPE         = 1,
  OPEN_DHCPS_HW_TYPE_10B     = 1,
  OPEN_DHCPS_HW_TYPE_IEEE802 = 6
} OPEN_DHCPS_HW_TYPES_t;

#define OPEN_DHCPS_MAX_POOL_NUMBER           256
#define OPEN_DHCPS_POOL_NAME_MINLEN          1
#define OPEN_DHCPS_POOL_NAME_MAXLEN          31
#define OPEN_DHCPS_DEFAULT_ROUTER_MAX        8
#define OPEN_DHCPS_DNS_SERVER_MAX            8
#define OPEN_DHCPS_NETBIOS_NAME_SERVER_MAX   8
#define OPEN_DHCPS_NTP_SERVER_MAX            2
#define OPEN_DHCPS_DOMAIN_NAME_MAXLEN        255
#define OPEN_DHCPS_BOOT_FILE_NAME_MAXLEN     128

#define OPEN_DHCPS_LEASETIME_MIN             1     /**< 1 minute */
#define OPEN_DHCPS_LEASETIME_MAX             86400 /**< 60 days */

typedef struct
{
  uint32_t routers[OPEN_DHCPS_DEFAULT_ROUTER_MAX];
} openDhcpsDefaultRouters_t;

typedef struct
{
  uint32_t dns_servers[OPEN_DHCPS_DNS_SERVER_MAX];
} openDhcpsDnsServers_t;

typedef struct
{
  uint32_t servers [OPEN_DHCPS_NETBIOS_NAME_SERVER_MAX];
} openDhcpsNetbiosServers_t;

typedef struct
{
  uint32_t ntp_servers[OPEN_DHCPS_NTP_SERVER_MAX];
} openDhcpsNtpServers_t;

/*********************************************************************
* @purpose  Get the DHCP Server Lease Data 
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    ipaddr           @b{(input)}  IP address assigned to lease
* @param    mask             @b{(output)} IP Mask
* @param    hwaddr           @b{(output)} Hardware address
* @param    clientId         @b{(output)} Client ID
* @param    poolName         @b{(output)} Pool name
* @param    remaining        @b{(output)} Remaining time in seconds
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes Hardware address is returned as a 6 byte raw hex value, and
*        must be of size @ref OPEN_MAC_ADDR_LEN
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerLeaseDataGet(openapiClientHandle_t *client_handle,
                                           open_inet_addr_t *ipaddr,
                                           uint32_t *mask,
                                           open_buffdesc *hwaddr,
                                           OPEN_DHCP_SERVER_CLIENT_ID_t *clientId,
                                           OPEN_DHCP_SERVER_POOL_NAME_t *poolName,
                                           uint32_t *remaining);

/*********************************************************************
* @purpose  Get the first DHCP Server Pool Entry 
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    poolIndex        @b{(output)} Pool index
* @param    poolName         @b{(output)} Pool name
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_NOT_FOUND if no entry found
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes None
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerPoolEntryFirstGet(openapiClientHandle_t *client_handle,
                                                uint32_t *poolIndex,
                                                OPEN_DHCP_SERVER_POOL_NAME_t *poolName);

/*********************************************************************
* @purpose  Get the next DHCP Server Pool Entry 
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    poolIndex        @b{(inout)}  Pool index
* @param    poolName         @b{(output)} Pool name
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if failure
* @returns  OPEN_E_NOT_FOUND if no entry found
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes None
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerPoolEntryNextGet(openapiClientHandle_t *client_handle,
                                               uint32_t *poolIndex,
                                               OPEN_DHCP_SERVER_POOL_NAME_t *poolName);


/*********************************************************************
* @purpose  Get the  DHCP Server Pool Type
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    poolName         @b{(input)}  Pool name
* @param    type             @b{(output)} Pool type
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if failure
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes None
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerPoolTypeGet(openapiClientHandle_t *client_handle,
                                          OPEN_DHCP_SERVER_POOL_NAME_t *poolName,
                                          OPEN_DHCP_SERVER_POOL_TYPE_t *type);


/*********************************************************************
* @purpose  Get the first DHCP Server Lease Entry
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    poolName         @b{(input)}  Pool name
* @param    leaseIndex       @b{(output)} Lease index
* @param    ipaddr           @b{(output)} Lease IP address
* @param    state            @b{(output)} State of the lease
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if failure or no entry found found.
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes None
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerLeaseEntryFirstGet(openapiClientHandle_t *client_handle,
                                                 OPEN_DHCP_SERVER_POOL_NAME_t *poolName,
                                                 uint32_t *leaseIndex,
                                                 open_inet_addr_t *ipaddr,
                                                 OPEN_DHCP_SERVER_LEASE_STATE_t *state);

/*********************************************************************
* @purpose  Get the next DHCP Server Lease Entry
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    poolName         @b{(input)}  Pool name
* @param    leaseIndex       @b{(inout)}  Lease index
* @param    ipaddr           @b{(output)} Lease IP address
* @param    state            @b{(output)} State of the lease
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if failure
* @returns  OPEN_E_NOT_FOUND if no entry found
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes None
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerLeaseEntryNextGet(openapiClientHandle_t *client_handle,
                                                OPEN_DHCP_SERVER_POOL_NAME_t *poolName,
                                                uint32_t *leaseIndex,
                                                open_inet_addr_t *ipaddr,
                                                OPEN_DHCP_SERVER_LEASE_STATE_t *state);

/*********************************************************************
* @purpose  Get the specified DHCP Server statistic
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    id               @b{(input)}  Statistic identifier
* @param    val              @b{(output)} Statistic value
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if failure
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerStatGet(openapiClientHandle_t *client_handle, 
                                      OPEN_DHCPS_STAT_ID_t id, uint32_t *val);

/*********************************************************************
* @purpose  Get the name of the VRF instance of a given DHCP server pool
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    poolName         @b{(input)}  Pool name
* @param    vrfName          @b{(output)} Name of VRF-instance of the pool
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if failure
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes None
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerPoolVrfNameGet(openapiClientHandle_t *client_handle,
                                             OPEN_DHCP_SERVER_POOL_NAME_t *poolName,
                                             OPEN_DHCP_SERVER_POOL_VRF_NAME_t *vrfName);

/*********************************************************************
* @purpose  Set the name of the VRF instance of a given DHCP server pool
*
* @param    client_handle    @b{(input)} Client handle from
*                                        registration API
* @param    poolName         @b{(input)} Pool name
* @param    vrfName          @b{(input)} Name of VRF-instance of the pool
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if failure
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes None
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerPoolVrfNameSet(openapiClientHandle_t *client_handle,
                                             OPEN_DHCP_SERVER_POOL_NAME_t *poolName,
                                             OPEN_DHCP_SERVER_POOL_VRF_NAME_t *vrfName);

/*********************************************************************
* @purpose  Set the name of the VRF instance of a given DHCP server pool
*
* @param    client_handle    @b{(input)} Client handle from
*                                        registration API
* @param    poolName         @b{(input)} Pool name
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if failure
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes None
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerPoolVrfNameSetToDefaultVrf(openapiClientHandle_t *client_handle,
                                                         OPEN_DHCP_SERVER_POOL_NAME_t *poolName);

/*********************************************************************
* @purpose  Get the first DHCP server excluded IP address range entry
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    pRangeIndex      @b{(input)}  Index to get first entry
* @param    vrfName          @b{(output)} Name of VRF instance in which IP
                                          addresses are configured to be excluded
* @param    startIpAddr      @b{(output)} Starting IP address of range
* @param    endIpAddr        @b{(output)} Ending IP address of range
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if failure or no entry found found.
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes None
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerExcludedAddressRangeEntryFirstGet(openapiClientHandle_t *client_handle,
                                                                uint32_t *pRangeIndex,
                                                                OPEN_DHCP_SERVER_POOL_VRF_NAME_t *vrfName,
                                                                open_inet_addr_t *startIpAddr,
                                                                open_inet_addr_t *endIpAddr);

/*********************************************************************
* @purpose  Get the next DHCP server excluded IP address range entry
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    pRangeIndex      @b{(input)}  Index to get next entry
* @param    vrfName          @b{(output)} Name of VRF instance in which IP
                                          addresses are configured to be excluded
* @param    startIpAddr      @b{(output)} Starting IP address of range
* @param    endIpAddr        @b{(output)} Ending IP address of range
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if failure
* @returns  OPEN_E_NOT_FOUND if no entry found
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes None
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerExcludedAddressRangeEntryNextGet(openapiClientHandle_t *client_handle,
                                                               uint32_t *pRangeIndex,
                                                               OPEN_DHCP_SERVER_POOL_VRF_NAME_t *vrfName,
                                                               open_inet_addr_t *startIpAddr,
                                                               open_inet_addr_t *endIpAddr);

/*********************************************************************
* @purpose  Add an excluded-address range for a given VRF instance
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    vrfName          @b{(input)}  VRF instance name
* @param    lowIpAddr        @b{(input)}  Start IPv4 address of the address range
* @param    highIpAddr       @b{(input)}  End IPv4 address of the address range
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if add operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes 
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerExcludedAddressRangeAdd(openapiClientHandle_t *client_handle,
                                                      OPEN_DHCP_SERVER_POOL_VRF_NAME_t *vrfName,
                                                      open_inet_addr_t *lowIpAddr,
                                                      open_inet_addr_t *highIpAddr);

/*********************************************************************
* @purpose  Delete an excluded-address range for a given VRF instance
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    vrfName          @b{(input)}  VRF instance name
* @param    lowIpAddr        @b{(input)}  Start IPv4 address of the address range
* @param    highIpAddr       @b{(input)}  End IPv4 address of the address range
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if delete operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes 
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerExcludedAddressRangeDelete(openapiClientHandle_t *client_handle,
                                                         OPEN_DHCP_SERVER_POOL_VRF_NAME_t *vrfName,
                                                         open_inet_addr_t *lowIpAddr,
                                                         open_inet_addr_t *highIpAddr);

/*********************************************************************
* @purpose  Get the DHCP server lease data for a given VRF name and IP address 
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    vrfName          @b{(input)}  VRF instance name
* @param    ipaddr           @b{(input)}  IP address assigned to lease
* @param    mask             @b{(output)} IP Mask
* @param    hwaddr           @b{(output)} Hardware address
* @param    clientId         @b{(output)} Client ID
* @param    poolName         @b{(output)} Pool name
* @param    remaining        @b{(output)} Remaining time in seconds
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes Hardware address is returned as a 6 byte raw hex value, and
*        must be of size @ref OPEN_MAC_ADDR_LEN
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerVrfIpAddrLeaseDataGet(openapiClientHandle_t *client_handle,
                                                    OPEN_DHCP_SERVER_POOL_VRF_NAME_t *vrfName,
                                                    open_inet_addr_t *ipaddr,
                                                    uint32_t *mask,
                                                    open_buffdesc *hwaddr,
                                                    OPEN_DHCP_SERVER_CLIENT_ID_t *clientId,
                                                    OPEN_DHCP_SERVER_POOL_NAME_t *poolName,
                                                    uint32_t *remaining);

/*********************************************************************
* @purpose  Clear lease data for VRF name and IP address 
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    vrfName          @b{(input)}  VRF instance name
* @param    ipaddr           @b{(input)}  IP address assigned to lease
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes 
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerVrfIpAddrLeaseDataClear(openapiClientHandle_t *client_handle,
                                                      OPEN_DHCP_SERVER_POOL_VRF_NAME_t *vrfName,
                                                      open_inet_addr_t *ipaddr);

/*********************************************************************
* @purpose  Clear lease data for pool name and IP address 
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    poolName         @b{(input)}  Pool name
* @param    ipaddr           @b{(input)}  IP address assigned to lease
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes 
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerPoolIpAddrLeaseDataClear(openapiClientHandle_t *client_handle,
                                                       OPEN_DHCP_SERVER_POOL_NAME_t *poolName,
                                                       open_inet_addr_t *ipaddr);

/*********************************************************************
* @purpose  Clear all lease data of a VRF instance
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    vrfName          @b{(input)}  VRF instance name
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes 
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerVrfLeaseDataClear(openapiClientHandle_t *client_handle,
                                                OPEN_DHCP_SERVER_POOL_VRF_NAME_t *vrfName);

/*********************************************************************
* @purpose  Clear all lease data of a pool 
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    poolName         @b{(input)}  Pool name
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes 
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerPoolLeaseDataClear(openapiClientHandle_t *client_handle,
                                                 OPEN_DHCP_SERVER_POOL_NAME_t *poolName);

/*********************************************************************
* @purpose  Clear all lease data across all VRFs  
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
*
* @returns  OPEN_E_NONE      if successful
* @returns  OPEN_E_FAIL      if get operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied
*
* @notes 
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
**********************************************************************/
open_error_t openapiDhcpServerAllLeaseDataClear(openapiClientHandle_t *client_handle);

/*****************************************************************************
 * @purpose   Set the DHCP server admin mode.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     mode             @b{(input)}  DHCP Server mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsAdminModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the DHCP server admin mode.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     mode             @b{(output)} DHCP Server mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsAdminModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the Bootp client allocation mode.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     mode             @b{(input)}  Bootp auto allocation mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsBootpAutomaticSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the Bootp client allocation mode.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     mode             @b{(output)} Bootp auto allocation mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsBootpAutomaticGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the number of pings to be sent out by DHCP server before assigning the address.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     pingPacketCount  @b{(input)}  Number of ping packets to be sent
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsNoOfPingPktSet(openapiClientHandle_t *client_handle, uint32_t pingPacketCount);

/*****************************************************************************
 * @purpose   Get the number of pings to be sent out by DHCP server before assigning the address.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     pingPacketCount  @b{(output)} Number of ping packets to be sent
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsNoOfPingPktGet(openapiClientHandle_t *client_handle, uint32_t *pingPacketCount);

/*****************************************************************************
 * @purpose   Set the number of pings to be sent out by DHCP server before assigning the address, to default count.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsNoOfPingPktReset(openapiClientHandle_t *client_handle);

/*****************************************************************************
 * @purpose   Create a DHCP server address pool.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of DHCP server address pool
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsPoolCreate(openapiClientHandle_t *client_handle, open_buffdesc *poolName);

/*****************************************************************************
 * @purpose   Delete a DHCP server address pool.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of DHCP server address pool
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsPoolDelete(openapiClientHandle_t *client_handle, open_buffdesc *poolName);

/*****************************************************************************
 * @purpose   Set the default routers for DHCP hosts in a given pool.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 * @param     defaultRouters   @b{(input)}  Array of default router address
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsDefaultRoutersSet (openapiClientHandle_t *client_handle,
                                            open_buffdesc *poolName,
                                            openDhcpsDefaultRouters_t *defaultRouters);

/*****************************************************************************
 * @purpose   Get the default routers for DHCP hosts in a given pool.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 * @param     defaultRouters   @b{(output)} Array of default router address
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsDefaultRoutersGet (openapiClientHandle_t *client_handle,
                                            open_buffdesc *poolName,
                                            openDhcpsDefaultRouters_t *defaultRouters);

/*****************************************************************************
 * @purpose   Reset the default routers for DHCP hosts in a given pool.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsDefaultRoutersReset (openapiClientHandle_t *client_handle,
                                              open_buffdesc *poolName);

/*****************************************************************************
 * @purpose   Set the DHCP Host network for a pool.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 * @param     networkAddr      @b{(input)}  Address pool to be assigned to the hosts
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsNetworkSet(openapiClientHandle_t *client_handle, open_buffdesc *poolName, open_inet_pfx_t networkAddr);

/*****************************************************************************
 * @purpose   Get the DHCP Host network for a pool.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 * @param     networkAddr      @b{(output)} Address pool used for the hosts
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsNetworkGet(openapiClientHandle_t *client_handle, open_buffdesc *poolName, open_inet_pfx_t *networkAddr);

/*****************************************************************************
 * @purpose   Delete the DHCP Host network for a pool.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsNetworkDelete(openapiClientHandle_t *client_handle, open_buffdesc *poolName);

/*****************************************************************************
 * @purpose   Set the DNS servers for DHCP hosts in a given pool.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 * @param     dnsServers       @b{(input)}  Array of DNS server address
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsDnsServersSet (openapiClientHandle_t *client_handle,
                                        open_buffdesc *poolName,
                                        openDhcpsDnsServers_t *dnsServers);

/*****************************************************************************
 * @purpose   Get the DNS servers for DHCP hosts in a given pool.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 * @param     dnsServers       @b{(output)} Array of DNS server address
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsDnsServersGet (openapiClientHandle_t *client_handle,
                                        open_buffdesc *poolName,
                                        openDhcpsDnsServers_t *dnsServers);

/*****************************************************************************
 * @purpose   Reset the DNS servers for DHCP hosts in a given pool.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsDnsServersReset(openapiClientHandle_t *client_handle, open_buffdesc *poolName);

/*****************************************************************************
 * @purpose   Set the Netbios name server addresses in given pool.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     poolName            @b{(input)}  Name of the DHCP server address pool
 * @param     netBiosNameServers  @b{(input)}  Array of Netbios name server addresses
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsNetbiosNameServerAddressSet (openapiClientHandle_t *client_handle,
                                                      open_buffdesc *poolName,
                                                      openDhcpsNetbiosServers_t *netBiosNameServers);

/*****************************************************************************
 * @purpose   Get the Netbios name server addresses in given pool.
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     poolName            @b{(input)}  Name of the DHCP server address pool
 * @param     netBiosNameServers  @b{(output)} Array of Netbios name server addresses
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsNetbiosNameServerAddressGet (openapiClientHandle_t *client_handle,
                                                      open_buffdesc *poolName,
                                                      openDhcpsNetbiosServers_t *netBiosNameServers);

/*****************************************************************************
 * @purpose   Delete the Netbios name server addresses in given pool.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsNetbiosNameServerAddressDelete(openapiClientHandle_t *client_handle, open_buffdesc *poolName);

/*****************************************************************************
 * @purpose   Set the type of Netbios node.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 * @param     nodeType         @b{(input)}  Type of the Netbios node
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsNetbiosNodeTypeSet (openapiClientHandle_t *client_handle,
                                             open_buffdesc *poolName,
                                             OPEN_NETBIOS_NODE_TYPES_t nodeType);

/*****************************************************************************
 * @purpose   Get the type of Netbios node.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 * @param     nodeType         @b{(output)} Type of the Netbios node
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsNetbiosNodeTypeGet (openapiClientHandle_t *client_handle,
                                             open_buffdesc *poolName,
                                             OPEN_NETBIOS_NODE_TYPES_t *nodeType);

/*****************************************************************************
 * @purpose   Delete the type of Netbios node.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsNetbiosNodeTypeDelete(openapiClientHandle_t *client_handle, open_buffdesc *poolName);

/*****************************************************************************
 * @purpose   Set the DHCP lease time.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 * @param     leaseTime        @b{(input)}  Lease time in minutes
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsLeaseTimeSet(openapiClientHandle_t *client_handle, open_buffdesc *poolName, uint32_t leaseTime);

/*****************************************************************************
 * @purpose   Get the DHCP lease time.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 * @param     leaseTime        @b{(output)} Lease time in minutes
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsLeaseTimeGet(openapiClientHandle_t *client_handle, open_buffdesc *poolName, uint32_t *leaseTime);

/*****************************************************************************
 * @purpose   Reset the DHCP lease time.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsLeaseTimeReset(openapiClientHandle_t *client_handle, open_buffdesc *poolName);

/*****************************************************************************
 * @purpose   Set the domain name of a DHCP client.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 * @param     domainName       @b{(input)}  Domain name of the DHCP client
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsDomainNameSet(openapiClientHandle_t *client_handle, open_buffdesc *poolName, open_buffdesc *domainName);

/*****************************************************************************
 * @purpose   Get the domain name of a DHCP client.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 * @param     domainName       @b{(output)} Domain name of the DHCP client
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsDomainNameGet(openapiClientHandle_t *client_handle, open_buffdesc *poolName, open_buffdesc *domainName);

/*****************************************************************************
 * @purpose   Delete the domain name of a DHCP client.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsDomainNameDelete(openapiClientHandle_t *client_handle, open_buffdesc *poolName);

/*****************************************************************************
 * @purpose   Set the next server IP address.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 * @param     nextAddress      @b{(input)}  Next server address
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsNextServerSet(openapiClientHandle_t *client_handle, open_buffdesc *poolName, open_inet_addr_t nextAddress);

/*****************************************************************************
 * @purpose   Get the next server IP address.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 * @param     nextAddress      @b{(output)} Next server address
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsNextServerGet(openapiClientHandle_t *client_handle, open_buffdesc *poolName, open_inet_addr_t *nextAddress);

/*****************************************************************************
 * @purpose   Delete the next server IP address.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsNextServerDelete(openapiClientHandle_t *client_handle, open_buffdesc *poolName);

/*****************************************************************************
 * @purpose   Set the DHCP host hardware address for a manual pool.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 * @param     hwAddr           @b{(input)}  Hardware address
 * @param     hwType           @b{(input)}  Type of hardware address
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsHwAddressSet(openapiClientHandle_t *client_handle, open_buffdesc *poolName, open_buffdesc *hwAddr, OPEN_DHCPS_HW_TYPES_t hwType);

/*****************************************************************************
 * @purpose   Get the DHCP host hardware address for a manual pool.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 * @param     hwAddr           @b{(output)} Hardware address
 * @param     hwType           @b{(output)} Type of hardware address
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsHwAddressGet(openapiClientHandle_t *client_handle, open_buffdesc *poolName, open_buffdesc *hwAddr, OPEN_DHCPS_HW_TYPES_t *hwType);

/*****************************************************************************
 * @purpose   Delete the DHCP host hardware address for a manual pool.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsHwAddressDelete(openapiClientHandle_t *client_handle, open_buffdesc *poolName);

/*****************************************************************************
 * @purpose   Set the DHCP Host network for a manual pool.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 * @param     hostAddr         @b{(input)}  Address to be assigned to the host
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsHostSet(openapiClientHandle_t *client_handle, open_buffdesc *poolName, open_inet_pfx_t hostAddr);

/*****************************************************************************
 * @purpose   Get the DHCP Host network for a manual pool.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 * @param     hostAddr         @b{(output)} Address to be assigned to the host
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsHostGet(openapiClientHandle_t *client_handle, open_buffdesc *poolName, open_inet_pfx_t *hostAddr);

/*****************************************************************************
 * @purpose   Delete the DHCP Host address and mask for a pool.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsHostDelete(openapiClientHandle_t *client_handle, open_buffdesc *poolName);

/*****************************************************************************
 * @purpose   Set the boot file name.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 * @param     fileName         @b{(input)}  Boot file name for Bootp client
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsBootfileSet(openapiClientHandle_t *client_handle, open_buffdesc *poolName, open_buffdesc *fileName);

/*****************************************************************************
 * @purpose   Get the boot file name.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 * @param     fileName         @b{(output)} Boot file name for Bootp client
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsBootfileGet(openapiClientHandle_t *client_handle, open_buffdesc *poolName, open_buffdesc *fileName);

/*****************************************************************************
 * @purpose   Delete the boot file name.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsBootfileDelete(openapiClientHandle_t *client_handle, open_buffdesc *poolName);

/*****************************************************************************
 * @purpose   Set the NTP server addresses for a given pool.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 * @param     ntpServers       @b{(input)}  Array of NTP server addresses
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsNTPServerSet (openapiClientHandle_t *client_handle,
                                       open_buffdesc *poolName,
                                       openDhcpsNtpServers_t *ntpServers);

/*****************************************************************************
 * @purpose   Get the NTP server addresses for a given pool.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 * @param     ntpServers       @b{(output)} Array of NTP server addresses
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsNTPServerGet (openapiClientHandle_t *client_handle,
                                       open_buffdesc *poolName,
                                       openDhcpsNtpServers_t *ntpServers);

/*****************************************************************************
 * @purpose   Delete the NTP server addresses for a given pool.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 * @param     poolName         @b{(input)}  Name of the DHCP server address pool
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsNTPServerDelete(openapiClientHandle_t *client_handle, open_buffdesc *poolName);

/*****************************************************************************
 * @purpose   Clear the DHCP server statistics.
 *
 * @param     client_handle    @b{(input)}  client handle from registration API
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsStatisticsClear (openapiClientHandle_t *client_handle);

/*****************************************************************************
 * @purpose   Utility fucntion to convert user provided client id to
 *             string format as required for the open api.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     clientIdBuf    @b{(input)}  Buffer containing client id string in 
 *                                         xx:xx:xx:xx:xx:xx.... format
 * @param     clientId       @b{(output)} Client Identifier of the DHCP client
 *                                         in OPEN_DHCP_SERVER_CLIENT_ID_t format
 * @param     clientIdLen    @b{(output)} Length of the client identifier
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiConvertClientId (openapiClientHandle_t *client_handle,
                                     open_buffdesc *clientIdBuf,
                                     OPEN_DHCP_SERVER_CLIENT_ID_t *clientId,
                                     uint32_t *clientIdLen);

/*****************************************************************************
 * @purpose   Configure the client identifier of the DHCP client.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     poolName       @b{(input)}  Name of the DHCP server address pool
 * @param     clientId       @b{(input)}  Client Identifier of the DHCP client
 * @param     clientIdLen    @b{(input)}  Length of the client identifier
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsClientIdSet (openapiClientHandle_t *client_handle,
                                      open_buffdesc *poolName,
                                      OPEN_DHCP_SERVER_CLIENT_ID_t *clientId,
                                      uint32_t clientIdLen);

/*****************************************************************************
 * @purpose   Get the client identifier of the DHCP client.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     poolName       @b{(input)}  Name of the DHCP server address pool
 * @param     clientId       @b{(output)} Client Identifier of the DHCP client
 * @param     clientIdLen    @b{(output)} Length of the client identifier
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsClientIdGet (openapiClientHandle_t *client_handle,
                                      open_buffdesc *poolName,
                                      OPEN_DHCP_SERVER_CLIENT_ID_t *clientId,
                                      uint32_t *clientIdLen);

/*****************************************************************************
 * @purpose   Delete the client identifier of the DHCP client.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     poolName       @b{(input)}  Name of the DHCP server address pool
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsClientIdDelete (openapiClientHandle_t *client_handle,
                                         open_buffdesc *poolName);

/*****************************************************************************
 * @purpose   Configure the client name of the DHCP client.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     poolName       @b{(input)}  Name of the DHCP server address pool
 * @param     ClientName     @b{(input)}  Name of the DHCP client
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsClientNameSet (openapiClientHandle_t *client_handle,
                                        open_buffdesc *poolName,
                                        open_buffdesc *ClientName);

/*****************************************************************************
 * @purpose   Get the client name of the DHCP client.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     poolName       @b{(input)}  Name of the DHCP server address pool
 * @param     ClientName     @b{(output)} Name of the DHCP client
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsClientNameGet (openapiClientHandle_t *client_handle,
                                        open_buffdesc *poolName,
                                        open_buffdesc *ClientName);

/*****************************************************************************
 * @purpose   Delete the client name of the DHCP client.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     poolName       @b{(input)}  Name of the DHCP server address pool
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.26
 *
 * @end
 *****************************************************************************/
open_error_t openapiDhcpsClientNameDelete (openapiClientHandle_t *client_handle,
                                           open_buffdesc *poolName);
#endif /* OPENAPI_DHCP_SERVER_H_INCLUDED */
